/* uctestfc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_uconv.h>

static char *make_jis_table(int type, size_t *size);
static char *make_sjis_table(int type, size_t *size);
static char *make_euc_table(int type, size_t *size);

int test_jis()
{
	char *tab = NULL, *tmp = NULL, *out = NULL;
	int i, len, err = -1;
	size_t size;

	if ((tab = make_jis_table(0, &size)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((tmp = malloc((size >> 1) * 3)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((out = malloc(size + 4)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}

	/* test conversion to sjis */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_JIS, UC_CODE_SJIS, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_jis : convert jis to sjis -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_SJIS, UC_CODE_JIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_jis : convert sjis to jis -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_jis : convert jis <--> sjis -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_jis : convert jis <--> sjis -- ok : %d\n",
			       i);
	}
	/* test conversion to euc */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_JIS, UC_CODE_EUC, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_jis : convert jis to euc -- error!\n");
			goto done;
		}

		if ((len =
		     UC_conv(UC_CODE_EUC, UC_CODE_JIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_jis : convert euc to jis -- error!\n");
			goto done;
		}

		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_jis : convert jis <--> euc -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_jis : convert jis <--> euc -- ok : %d\n",
			       i);
	}

	/* test conversion to unicode */
	free(tab);
	if ((tab = make_jis_table(1, &size)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_JIS, UC_CODE_UNICODE, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_jis : convert jis to unicode -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UNICODE, UC_CODE_JIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_jis : convert unicode to jis -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_jis : convert jis <--> unicode -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf
			    ("test_jis : convert jis <--> unicode -- ok : %d\n",
			     i);
	}

	/* test conversion to utf8 */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_JIS, UC_CODE_UTF8, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_jis : convert jis to utf8 -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UTF8, UC_CODE_JIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_jis : convert utf8 to jis -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_jis : convert jis <--> utf8 -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_jis : convert jis <--> utf8 -- ok : %d\n",
			       i);
	}

	err = 0;
done:
	if (tab)
		free(tab);
	if (tmp)
		free(tmp);
	if (out)
		free(out);
	return err;
}

int test_sjis()
{
	char *tab = NULL, *tmp = NULL, *out = NULL;
	int i, len, err = -1;
	size_t size;

	if ((tab = make_sjis_table(0, &size)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((tmp = malloc((size >> 1) * 3)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((out = malloc(size + 4)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}

	/* test conversion to sjis */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_SJIS, UC_CODE_JIS, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_sjis : convert jis to sjis -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_JIS, UC_CODE_SJIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_sjis : convert sjis to jis -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_sjis : convert sjis <--> jis -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_sjis : convert sjis <--> jis -- ok : %d\n",
			       i);
	}
	/* test conversion to euc */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_SJIS, UC_CODE_EUC, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_sjis : convert sjis to euc -- error!\n");
			goto done;
		}

		if ((len =
		     UC_conv(UC_CODE_EUC, UC_CODE_SJIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_sjis : convert euc to sjis -- error!\n");
			goto done;
		}

		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_sjis : convert sjis <--> euc -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_sjis : convert sjis <--> euc -- ok : %d\n",
			       i);
	}

	/* test conversion to unicode */
	free(tab);
	if ((tab = make_sjis_table(1, &size)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_SJIS, UC_CODE_UNICODE, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_sjis : convert sjis to unicode -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UNICODE, UC_CODE_SJIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_sjis : convert unicode to sjis -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_sjis : convert sjis <--> unicode -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf
			    ("test_sjis : convert sjis <--> unicode -- ok : %d\n",
			     i);
	}
	/* test conversion to utf8 */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_SJIS, UC_CODE_UTF8, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_sjis : convert sjis to utf8 -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UTF8, UC_CODE_SJIS, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_sjis : convert utf8 to sjis -- error!\n");
			goto done;
		}
		/* write_buf2file("c.out",out,size); */
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_sjis : convert sjis <--> utf8 -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf
			    ("test_sjis : convert sjis <--> utf8 -- ok : %d\n",
			     i);
	}

	err = 0;
done:
	if (tab)
		free(tab);
	if (tmp)
		free(tmp);
	if (out)
		free(out);
	return err;
}

int test_euc()
{
	char *tab = NULL, *tmp = NULL, *out = NULL;
	int i, len, sz, err = -1;
	size_t size;

	if ((tab = make_euc_table(1, &size)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((tmp = malloc((size >> 1) * 3)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}
	if ((out = malloc(size + 1024)) == NULL) {
		fprintf(stderr, "cannot get test table...\n");
		goto done;
	}

	/* test conversion to unicode */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_EUC, UC_CODE_UNICODE, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_euc : convert euc to unicode -- error!\n");
			goto done;
		}
		/* write_buf2file("a.out",tab,size); */
		if ((len =
		     UC_conv(UC_CODE_UNICODE, UC_CODE_EUC, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_euc : convert unicode to euc -- error!\n");
			goto done;
		}
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_euc : convert euc <--> unicode -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf
			    ("test_euc : convert euc <--> unicode -- ok : %d\n",
			     i);
	}
	/* test conversion to utf8 */
	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_EUC, UC_CODE_UTF8, tab, size, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_euc : convert euc to utf8 -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UTF8, UC_CODE_EUC, tmp, len, out,
			     size + 4)) < 0) {
			fprintf(stderr,
				"test_euc : convert utf8 to euc -- error!\n");
			goto done;
		}
		/* write_buf2file("c.out",out,size); */
		if ((len != size) || (memcmp(tab, out, size))) {
			fprintf(stderr,
				"test_euc : convert euc <--> utf8 -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf("test_euc : convert euc <--> utf8 -- ok : %d\n",
			       i);
	}

	/* utf <--> unicode conversion test */
	if ((sz =
	     UC_conv(UC_CODE_EUC, UC_CODE_UNICODE, tab, size, tmp,
		     (size >> 1) * 3)) < 0)
		goto done;
	memcpy(tab, tmp, sz);

	for (i = 0; i < 11; i++) {
		if ((len =
		     UC_conv(UC_CODE_UNICODE, UC_CODE_UTF8, tab, sz, tmp,
			     (size >> 1) * 3)) < 0) {
			fprintf(stderr,
				"test_unicode : convert unicode to utf8 -- error!\n");
			goto done;
		}
		if ((len =
		     UC_conv(UC_CODE_UTF8, UC_CODE_UNICODE, tmp, len, out,
			     size + 1024)) < 0) {
			fprintf(stderr,
				"test_unicode : convert utf8 to unicode -- error!\n");
			goto done;
		}
		/* write_buf2file("c.out",out,size); */
		if ((len != sz) || (memcmp(tab, out, sz))) {
			fprintf(stderr,
				"test_unicode : convert unicode <--> utf8 -- error!\n");
			goto done;
		}
		if ((i % 5) == 0)
			printf
			    ("test_unicode : convert unicode <--> utf8 -- ok : %d\n",
			     i);
	}

	err = 0;
done:
	if (tab)
		free(tab);
	if (tmp)
		free(tmp);
	if (out)
		free(out);
	return err;
}

/*--------------------------------------------------*/
static char *make_jis_table(int type, size_t *size)
{
	char *ret;
	int i, j;
	size_t sz;

	if ((ret = malloc(264 + 66 + 8840 * 2)) == NULL)
		return NULL;
	memset(ret, 0, 264 + 66 + 8840 * 2);

	/* set ascii first */
	for (sz = 0, i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;	/* avoid escape sequence for JIS7 */
		else
			ret[sz] = i;
	}
	/* set JIS7 katakana */
	ret[sz] = UC_ESC;
	ret[sz + 1] = '(';
	ret[sz + 2] = 'I';
	sz += 3;
	for (i = 0x21; i <= 0x5f; sz++, i++)
		ret[sz] = i;

	/* set JIS kanji */
	ret[sz] = UC_ESC;
	ret[sz + 1] = '$';
	ret[sz + 2] = 'B';
	sz += 3;
	if (type == 0) {	/* full JIS-table */
		for (i = 0; i < 94; i++) {
			for (j = 0; j < 94; j++) {
				ret[sz] = 0x21 + i;
				ret[sz + 1] = 0x21 + j;
				sz += 2;
			}
		}
	} else if (type == 1) {	/* for UNICODE conpatible */
		for (i = 0; i < UC_JIS2UNI_MAX; i++) {
			j = jis2uni[i];
			if ((j == 0xffff) || (j == 0x5c))
				continue;

			ret[sz] = 0x21 + i / 96;
			ret[sz + 1] = 0x21 + i % 96;
			sz += 2;
		}
	}
	/* set ascii again */
	ret[sz] = UC_ESC;
	ret[sz + 1] = '(';
	ret[sz + 2] = 'B';
	sz += 3;
	for (i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;
		else
			ret[sz] = i;
	}
	*size = sz;
	return ret;
}

static char *make_sjis_table(int type, size_t *size)
{
	char *ret, buf[16];
	int i, j, k;
	size_t sz;

	if ((ret = malloc(264 + 66 + 23124)) == NULL)
		return NULL;
	memset(ret, 0, 264 + 66 + 23124);

	/* set ascii first */
	for (sz = 0, i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;	/* avoid jis escape sequence */
		else
			ret[sz] = i;
	}
	/* set JIS8 katakana */
	for (i = 0xA1; i <= 0xDF; sz++, i++)
		ret[sz] = i;

	/* set S-JIS kanji */
	for (i = 0x81; i <= 0x9F; i++) {
		for (j = 0x40; j <= 0xFC; j++) {
			if (j == 0x7f)
				continue;
			ret[sz] = i;
			ret[sz + 1] = j;

			if (type == 1) {	/* unicocde check */
				sjis2jis_c(&ret[sz], (unsigned char *)buf);
				if ((k =
				     (buf[0] - 0x21) * 96 + (buf[1] - 0x21)) >
				    UC_JIS2UNI_MAX)
					continue;
				k = jis2uni[k];
				if ((k == 0xffff) || (k == 0x5c))
					continue;
			}
			sz += 2;
		}
	}
	/* full size of S-JIS is 0xE0 <= i <= 0xFC, however,
	 * when the codes that have more 0xF0 are converted to JIS,
	 * these codes are not in JIS code difinition.
	 */
	for (i = 0xE0; i < 0xF0; i++) {
		for (j = 0x40; j <= 0xFC; j++) {
			if (j == 0x7f)
				continue;
			ret[sz] = i;
			ret[sz + 1] = j;

			if (type == 1) {	/* unicocde check */
				sjis2jis_c(&ret[sz], (unsigned char *)buf);
				if ((k =
				     (buf[0] - 0x21) * 96 + (buf[1] - 0x21)) >=
				    UC_JIS2UNI_MAX)
					continue;
				k = jis2uni[k];
				if ((k == 0xffff) || (k == 0x5c))
					continue;
			}
			sz += 2;
		}
	}

	/* set ascii again */
	for (i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;
		else
			ret[sz] = i;
	}
	*size = sz;
	return ret;
}

static char *make_euc_table(int type, size_t *size)
{
	char *ret;
	int i, j;
	size_t sz;

	if ((ret = malloc(1024 + 8840 * 2)) == NULL)
		return NULL;
	memset(ret, 0, 1024 + 8840 * 2);

	/* set ascii first */
	for (sz = 0, i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;	/* avoid escape sequence for JIS7 */
		else
			ret[sz] = i;
	}
	/* set EUC katakana */
	for (i = 0xa1; i <= 0xdf; sz += 2, i++) {
		ret[sz] = 0x8e;
		ret[sz + 1] = i;
	}

	/* set EUC kanji */
	if (type == 0) {	/* full JIS-table */
		for (i = 0; i < 94; i++) {
			for (j = 0; j < 94; j++) {
				ret[sz] = 0xa1 + i;
				ret[sz + 1] = 0xa1 + j;
				sz += 2;
			}
		}
	} else if (type == 1) {	/* for UNICODE conpatible */
		for (i = 0; i < UC_JIS2UNI_MAX; i++) {
			j = jis2uni[i];
			if ((j == 0xffff) || (j == 0x5c))
				continue;

			ret[sz] = 0xa1 + i / 96;
			ret[sz + 1] = 0xa1 + i % 96;
			sz += 2;
		}
	}
	/* set ascii again */
	for (i = 1; i < 0x80; sz++, i++) {
		if ((i == 0x0e) || (i == 0x0f))
			ret[sz] = 0x41;
		else
			ret[sz] = i;
	}
	*size = sz;
	return ret;
}

int write_buf2file(char *fname, unsigned char *buf, int len)
{
	FILE *fp;

	if ((fp = fopen(fname, "w")) == NULL) {
		fprintf(stderr, "fopen error!\n");
		return -1;
	}
	fwrite(buf, 1, len, fp);

	fclose(fp);
	return 0;
}
