/* signature.c */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pkcs11.h>
#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_tool.h>

int set_digalgo_from_sigalgo(int algo);

/*-------------------------------------------------
  get signature from data
  (only called by test programs)
-------------------------------------------------*/
int OK_do_signature(Key *prv, unsigned char *data, int data_len, unsigned char **signature,int *sig_len, int sig_algo){
	return NRG_do_signature(prv, data, data_len, signature, sig_len,
				sig_algo, NULL);
}

int NRG_do_signature(Key *prv, unsigned char *data, int data_len,
		     unsigned char **signature,int *sig_len, int sig_algo,
		     void *params)
{
	unsigned char	digest[AC_MAX_DIGESTSIZE];
	int dlen;

	if(set_digalgo_from_sigalgo(sig_algo)<0) goto error;

	if(OK_do_digest(sign_digest_algo,data,data_len,digest,&dlen)==NULL) goto error;
	
	switch(prv->key_type){
	case KEY_RSA_PRV:
	case KEY_P11RSA_PRV:
		switch (RSA_get_encoding_method(sig_algo)) {
		case RSA_EMSA_PSS:
			if (params == NULL && sign_digest_algo != OBJ_HASH_SHA1) {
				OK_set_error(ERR_ST_UNSUPPORTED_ALGO,
					     ERR_LC_TOOL, ERR_PT_SIG, NULL);
				goto error;
			}
			*signature = RSA_PSS_sign_digest((Prvkey_RSA *) prv,
							 digest, dlen,
							 params);
			break;
		case RSA_EMSA_PKCS1:
			*signature = P1_sign_digest(prv, digest, dlen,
						    sign_digest_algo);
			break;
		default:
			OK_set_error(ERR_ST_UNSUPPORTED_ALGO,
				     ERR_LC_TOOL, ERR_PT_SIG, NULL);
			goto error;
		}
		if (*signature == NULL)
			goto error;
		*sig_len = prv->size;
		break;
	case KEY_DSA_PRV:
		if((*signature=DSA_get_signature((Prvkey_DSA*)prv,digest,dlen,sig_len))==NULL)
			goto error;
		break;
	case KEY_ECDSA_PRV:
		if((*signature=ECDSA_get_signature((Prvkey_ECDSA*)prv,digest,dlen,sig_len))==NULL)
			goto error;
		break;
	case KEY_P11DSA_PRV:
	case KEY_P11ECDSA_PRV:
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_TOOL,ERR_PT_SIG,NULL);
		goto error;
	}

	return 0;
error:
	return -1;
}

/*-------------------------------------------------
  Verify signature
  0 ... signature OK
  1 ... signature error;
  -1... system error
  -2... PKCS#1 padding error or else
-------------------------------------------------*/
/* data should be smaller than key size (RSA) */
int OK_do_verify(Key *pub, unsigned char *digest, unsigned char *sig, int sig_algo){
	return NRG_do_verify(pub, digest, sig, sig_algo, NULL);
}

int NRG_do_verify(Key *pub, unsigned char *digest, unsigned char *sig,
		  int sig_algo, void *params)
{
	unsigned char *org=NULL,*dec=NULL;
	int i,halgo,slen,dlen,ret=-1;

	if((halgo= obj_sig2hash(sig_algo))<0) goto done;
	if((dlen = hash_size(halgo))<0) goto done;

	switch(pub->key_type){
	case KEY_RSA_PUB:
	case KEY_P11RSA_PUB:
		/* decode certificate signature */
		slen= pub->size;
		if((dec=OK_do_sign(pub,sig,slen,NULL))==NULL)
			goto done;

		switch (RSA_get_encoding_method(sig_algo)) {
		case RSA_EMSA_PSS:
			if (params == NULL &&
			    sign_digest_algo != OBJ_HASH_SHA1) {
				OK_set_error(ERR_ST_UNSUPPORTED_ALGO,
					     ERR_LC_TOOL, ERR_PT_SIG+1, NULL);
				goto done;
			}
			LNm *mod = ((Pubkey_RSA *) pub)->n;
			int modbyte = LN_now_byte(mod);
			int modbit = LN_now_bit(mod);
			int consistent = 0;
			consistent = RSA_PSS_verify(digest, dlen,
						    dec, modbyte,
						    modbit - 1,
						    params);
			if (consistent == 0) {
				ret = 0;
			} else {
				ret = 1;
				goto done;
			}
			break;
		case RSA_EMSA_PKCS1:
			/* PKCS#1 Signature Check
			 * check padding and OBJECT IDENTIFIER */
			if((org = P1_pad2digest(dec,&i,slen))==NULL) goto done;

			if(i != halgo){
				OK_set_error(ERR_ST_BADPARAM,ERR_LC_TOOL,ERR_PT_SIG+1,NULL);
				ret = -2;
				goto done;
			}

			ret = 0;
			if(memcmp(digest,org,dlen)) ret = 1;
			break;
		default:
			OK_set_error(ERR_ST_UNSUPPORTED_ALGO,
				     ERR_LC_TOOL, ERR_PT_SIG+1, NULL);
			ret = -1;
			goto done;
		}
		break;

	case KEY_DSA_PUB:
		ret = DSA_vfy_signature((Pubkey_DSA*)pub,digest,dlen,sig);
		break;

	case KEY_ECDSA_PUB:
		ret = ECDSA_vfy_signature((Pubkey_ECDSA*)pub,digest,dlen,sig);
		break;

	case KEY_P11DSA_PUB:
	case KEY_P11ECDSA_PUB:
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_TOOL,ERR_PT_SIG+1,NULL);
		goto done;
	}
done:
	if(org) free(org);
	if(dec) free(dec);
	return ret;
}

/*-----------------------------------------------------
  Low level signing function (not generate signature)
-----------------------------------------------------*/
/* data_len must be smaller than key size */
unsigned char *OK_do_sign(Key *key,unsigned char *data,int data_len,unsigned char *ret){
	unsigned char *tmp,*cp=ret;
	int i;

	if(key->size<data_len){
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_TOOL,ERR_PT_SIG+2,NULL);
		return NULL;
	}

	if(ret==NULL){
		if((cp=ret=(unsigned char*)malloc(key->size))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_TOOL,ERR_PT_SIG+2,NULL);
			return NULL;
		}
	}
	switch(key->key_type){
	case KEY_RSA_PRV:
		if(RSAprv_doCrypt(data_len,data,cp,(Prvkey_RSA*)key)) goto error;
		break;
	case KEY_RSA_PUB:
		if(RSApub_doCrypt(data_len,data,cp,(Pubkey_RSA*)key)) goto error;
		break;
	case KEY_P11RSA_PRV:
		if(P11RSAprv_doCrypt(data_len,data,cp,(Prvkey_P11RSA*)key)) goto error;
		break;
	case KEY_P11RSA_PUB:
		if(P11RSApub_doCrypt(data_len,data,cp,(Pubkey_P11RSA*)key)) goto error;
		break;
	case KEY_DSA_PRV:
		if((tmp=DSA_get_signature((Prvkey_DSA*)key,data,data_len,&i))==NULL) goto error;
		if(cp!=ret){ memcpy(ret,tmp,i); free(tmp); }
		else { free(cp); cp=tmp; }
		break;
	case KEY_ECDSA_PRV:
		if((tmp=ECDSA_get_signature((Prvkey_ECDSA*)key,data,data_len,&i))==NULL) goto error;
		if(cp!=ret){ memcpy(ret,tmp,i); free(tmp); }
		else { free(cp); cp=tmp; }
		break;
	case KEY_DSA_PUB:
	case KEY_ECDSA_PUB:
	case KEY_P11DSA_PUB:
	case KEY_P11ECDSA_PUB:
		/* DSA public key cannot be used for key encryption */
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_TOOL,ERR_PT_SIG+2,NULL);
		goto error;
	case KEY_P11DSA_PRV:
	case KEY_P11ECDSA_PRV:
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_TOOL,ERR_PT_SIG+2,NULL);
		goto error;
	}
	return cp;
error:
	if(ret!=cp) free(cp);
	return NULL;
}

/*-------------------------------------------------
  Get signature from digest with PKCS1 padding
-------------------------------------------------*/
unsigned char *P1_do_sign(Key *prv,unsigned char *data,int *ret_len){
	unsigned char	digest[AC_MAX_DIGESTSIZE],*ret;

	if(ASN1_do_digest(sign_digest_algo,data,digest,ret_len))
		return NULL;

	ret=P1_sign_digest(prv,digest,*ret_len,sign_digest_algo);
	*ret_len=prv->size;

	return ret;
}

/*-------------------------------------------------
  Get digest with PKCS1 padding
-------------------------------------------------*/
int P1_DER_digestinfo(unsigned char *dig,int dig_size,int dig_type,unsigned char *ret,int *ret_len){
	unsigned char *cp;
	int	i,j;

	if(ASN1_int_2object(dig_type,ret,&i)) return -1;
	cp = ret+i;
	ASN1_set_null(cp);
	ASN1_set_sequence(i+2,ret,&i);
	cp = ret+i;
	ASN1_set_octetstring(dig_size,dig,cp,&j);
	i+=j;
	ASN1_set_sequence(i,ret,ret_len);
	return 0;
}

unsigned char *P1_sign_digest(Key *key,unsigned char *digest,int dig_size,int dig_type){
	unsigned char *ret,*sign,dinfo[128];
	int	ks,dis;
	unsigned int sz_sign;

	if((dig_size>AC_MAX_DIGESTSIZE)||(dig_size<0)){
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_TOOL,ERR_PT_SIG+5,NULL);
		return NULL;
	}
	if(P1_DER_digestinfo(digest,dig_size,dig_type,dinfo,&dis))
		return NULL;
#ifdef PKCS11_DEBUG
	/* print the first 8 characters */
	for (ks = 0; ks < 8; ks++)
		fprintf(stderr, "%.2x ", dinfo[ks]);
	fprintf(stderr, "\n");
	fprintf(stderr, "strlen(dinfo) = %zd\n", strlen(dinfo));
	fprintf(stderr, "          dis = %d\n", dis);
	fflush(stderr);
	ks = 0;
#endif
	/*
	 * RFC3447 9.2 EMSA-PKCS1-v1_5
	 *
	 * 3. If emLen < tLen + 11, output "intended encoded message
	 *    length too short" and stop.
	 */
	ks=key->size;
	if(ks<dis+11)
		return NULL;

	if((sign=(unsigned char*)malloc(ks))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_TOOL,ERR_PT_SIG+5,NULL);
		return NULL;
	}
 
	/*
	 * RFC3447 9.2 EMSA-PKCS1-v1_5 (Padding)
	 *
	 * 4. Generate an octet string PS consisting of emLen - tLen - 3 octets
	 *    with hexadecimal value 0xff.  The length of PS will be at least 8
	 *    octets.
	 * 5. Concatenate PS, the DER encoding T, and other padding to form the
	 *    encoded message EM as
	 *       EM = 0x00 || 0x01 || PS || 0x00 || T.
	 *
	 * Note for HSM: if you use an HSM with CKM_RSA_PKCS, do nothing here.
	 */
#ifdef PKCS11_DEBUG
	fprintf(stderr, "signature.c: NRG_CKM_SIGNING = 0x%.4x\n",
		NRG_CKM_SIGNING);
#endif
	if (KEY_P11RSA_PRV == key->key_type &&
	    CKM_RSA_PKCS == NRG_CKM_SIGNING) {
		memset(sign, 0, ks);
		memcpy(sign, dinfo, dis);
		sz_sign = dis;
#ifdef PKCS11_DEBUG
		fprintf(stderr, "signature.c: No padding\n");
		fflush(stderr);
#endif
	} else {
		memset(sign, 0xff, ks);
		sign[0] = 0; sign[1] = 1;
		memcpy(&sign[ks - dis], dinfo, dis);
		sign[ks - dis - 1] = 0;
		sz_sign = ks;
#ifdef PKCS11_DEBUG
		fprintf(stderr, "signature.c: PKCS1-v1_5 padding\n");
		fflush(stderr);
#endif
	}

	/* if ret is NULL, something's wrong with key X( */
	ret = OK_do_sign(key, sign, sz_sign, NULL);

	free(sign);
	return ret;
}

unsigned char *P1_pad2digest(unsigned char *dec,int *dig_algo,int keylen){
	unsigned char *dtop,*ret,*cp;
	int i,j,k;

	/* check PKCS#1 Padding */
	if((dec[0]!=0)||(dec[1]!=1)){ /* bad Padding or decryption error */
		OK_set_error(ERR_ST_P1_BADPADDING,ERR_LC_TOOL,ERR_PT_SIG+6,NULL);
		return NULL;
	}

	for(i=2;dec[i];i++){
		if(dec[i]!=0xff){
			OK_set_error(ERR_ST_P1_BADPADDING,ERR_LC_TOOL,ERR_PT_SIG+6,NULL);
			return NULL;
		}
	}

	dtop = &dec[i+1];
	j    = ASN1_length(dtop+1,&k);
	if(keylen != (i+2+j+k)){
		OK_set_error(ERR_ST_P1_BADPADDING,ERR_LC_TOOL,ERR_PT_SIG+6,NULL);
		return NULL;
	}

	dtop =ASN1_next(dtop);
	cp   =ASN1_next(dtop);

	if((*dig_algo=ASN1_object_2int(cp))<=0){
		OK_set_errorlocation(ERR_LC_TOOL,ERR_PT_SIG+6);
		return NULL;
	}

	/* Algorithm is OK, so check byte strings */
	if((cp=ASN1_skip(dtop))==NULL) return NULL;

	if(ASN1_octetstring(cp,&i,&ret,&j))
		OK_set_errorlocation(ERR_LC_TOOL,ERR_PT_SIG+6);

	return ret;
}
