/* sigalgo.c */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <string.h>
#include <aicrypto/key_type.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_tool.h>

/*-----------------------------------------
  return signature algorithm OID.

  return -1 when invalid combination.
-----------------------------------------*/
int
select_sigalgo(int key_type, int hash_algo)
{
	int sig = -1;

	switch(key_type){
	case KEY_DSA_PUB:
		switch(hash_algo){
		case OBJ_HASH_SHA1: sig = OBJ_SIG_SHA1DSA; break;
		case OBJ_HASH_SHA224: sig = OBJ_SIG_SHA224DSA; break;
		case OBJ_HASH_SHA256: sig = OBJ_SIG_SHA256DSA; break;
		case OBJ_HASH_SHA3_224: sig = OBJ_SIG_SHA3_224DSA; break;
		case OBJ_HASH_SHA3_256: sig = OBJ_SIG_SHA3_256DSA; break;
		}
		break;
	case KEY_ECDSA_PUB:
		switch(hash_algo){
		case OBJ_HASH_SHA1: sig = OBJ_SIG_SHA1ECDSA; break;
		case OBJ_HASH_SHA224: sig = OBJ_SIG_SHA224ECDSA; break;
		case OBJ_HASH_SHA256: sig = OBJ_SIG_SHA256ECDSA; break;
		case OBJ_HASH_SHA384: sig = OBJ_SIG_SHA384ECDSA; break;
		case OBJ_HASH_SHA512: sig = OBJ_SIG_SHA512ECDSA; break;
		case OBJ_HASH_SHA3_224: sig = OBJ_SIG_SHA3_224ECDSA; break;
		case OBJ_HASH_SHA3_256: sig = OBJ_SIG_SHA3_256ECDSA; break;
		case OBJ_HASH_SHA3_384: sig = OBJ_SIG_SHA3_384ECDSA; break;
		case OBJ_HASH_SHA3_512: sig = OBJ_SIG_SHA3_512ECDSA; break;
		}
		break;
	case KEY_RSA_PUB:
		switch(hash_algo){
		case OBJ_HASH_SHA1: sig = OBJ_SIG_SHA1RSA; break;
		case OBJ_HASH_SHA224: sig = OBJ_SIG_SHA224RSA; break;
		case OBJ_HASH_SHA256: sig = OBJ_SIG_SHA256RSA; break;
		case OBJ_HASH_SHA384: sig = OBJ_SIG_SHA384RSA; break;
		case OBJ_HASH_SHA512: sig = OBJ_SIG_SHA512RSA; break;
		case OBJ_HASH_SHA512224: sig = OBJ_SIG_SHA512224RSA; break;
		case OBJ_HASH_SHA512256: sig = OBJ_SIG_SHA512256RSA; break;
		case OBJ_HASH_SHA3_224: sig = OBJ_SIG_SHA3_224RSA; break;
		case OBJ_HASH_SHA3_256: sig = OBJ_SIG_SHA3_256RSA; break;
		case OBJ_HASH_SHA3_384: sig = OBJ_SIG_SHA3_384RSA; break;
		case OBJ_HASH_SHA3_512: sig = OBJ_SIG_SHA3_512RSA; break;
		case OBJ_HASH_MD2: sig = OBJ_SIG_MD2RSA; break;
		case OBJ_HASH_MD5: sig = OBJ_SIG_MD5RSA; break;
		}
		break;
	}

	return sig;
}

/*-----------------------------------------
  return OBJ_HASH_XXX from OBJ_SIG_XXX.
  return -1 when sig_oid was unknown.
-----------------------------------------*/
int obj_sig2hash(int sig_oid){
	int ret;
	switch(sig_oid){
	case OBJ_SIG_MD2RSA:
	case OBJ_SIGOIW_MD2RSA:
	case OBJ_HASH_MD2:
		ret = OBJ_HASH_MD2; break;
	case OBJ_SIG_MD5RSA:
	case OBJ_SIGOIW_MD5RSA:
	case OBJ_HASH_MD5:
		ret = OBJ_HASH_MD5; break;
	case OBJ_SIG_SHA1RSA:
	case OBJ_SIG_SHA1RSA_PSS:
	case OBJ_SIG_SHA1DSA:
	case OBJ_SIG_SHA1ECDSA:
	case OBJ_SIGOIW_SHA1RSA:
	case OBJ_HASH_SHA1:
		ret = OBJ_HASH_SHA1; break;
	case OBJ_SIG_SHA224RSA:
	case OBJ_SIG_SHA224RSA_PSS:
	case OBJ_SIG_SHA224DSA:
	case OBJ_SIG_SHA224ECDSA:
	case OBJ_HASH_SHA224:
		ret = OBJ_HASH_SHA224; break;
	case OBJ_SIG_SHA256RSA:
	case OBJ_SIG_SHA256RSA_PSS:
	case OBJ_SIG_SHA256DSA:
	case OBJ_SIG_SHA256ECDSA:
	case OBJ_HASH_SHA256:
		ret = OBJ_HASH_SHA256; break;
	case OBJ_SIG_SHA384RSA:
	case OBJ_SIG_SHA384RSA_PSS:
	case OBJ_SIG_SHA384ECDSA:
	case OBJ_HASH_SHA384:
		ret = OBJ_HASH_SHA384; break;
	case OBJ_SIG_SHA512RSA:
	case OBJ_SIG_SHA512RSA_PSS:
	case OBJ_SIG_SHA512ECDSA:
	case OBJ_HASH_SHA512:
		ret = OBJ_HASH_SHA512; break;
	case OBJ_SIG_SHA512224RSA:
	case OBJ_SIG_SHA512224RSA_PSS:
	case OBJ_HASH_SHA512224:
		ret = OBJ_HASH_SHA512224;
		break;
	case OBJ_SIG_SHA512256RSA:
	case OBJ_SIG_SHA512256RSA_PSS:
	case OBJ_HASH_SHA512256:
		ret = OBJ_HASH_SHA512256;
		break;
	case OBJ_SIG_SHA3_224RSA:
	case OBJ_SIG_SHA3_224DSA:
	case OBJ_SIG_SHA3_224ECDSA:
	case OBJ_HASH_SHA3_224:
		ret = OBJ_HASH_SHA3_224;
		break;
	case OBJ_SIG_SHA3_256RSA:
	case OBJ_SIG_SHA3_256DSA:
	case OBJ_SIG_SHA3_256ECDSA:
	case OBJ_HASH_SHA3_256:
		ret = OBJ_HASH_SHA3_256;
		break;
	case OBJ_SIG_SHA3_384RSA:
	case OBJ_SIG_SHA3_384ECDSA:
	case OBJ_HASH_SHA3_384:
		ret = OBJ_HASH_SHA3_384;
		break;
	case OBJ_SIG_SHA3_512RSA:
	case OBJ_SIG_SHA3_512ECDSA:
	case OBJ_HASH_SHA3_512:
		ret = OBJ_HASH_SHA3_512;
		break;
	default:
		ret = -1;
	}
	return ret;
}

/*-----------------------------------------
  Looks up the key type corresponding to arg.
  
  key type is defined in "key_type.h".
  If specified arg is unknown, returns -1.
-----------------------------------------*/
int get_keytype(char *arg)
{
	if(!strcmp("rsa",arg)) return KEY_RSA_PUB;
	else if(!strcmp("dsa",arg)) return KEY_DSA_PUB;
	else if(!strcmp("ecdsa",arg)) return KEY_ECDSA_PUB;
	else return -1;
}

/**
 * Return the name of an algorithm for public-key cryptography.
 *
 * This function transforms the name of an algorithm for public-key
 * cryptography from the internal integer into the character strings.
 *
 * @param[in] keytype an internal integer for an algorithm for public-key cryptography. Each integer of algorithm is defined in key_type.h.
 * @param[out] FIXME: need a short description.
 * @ingroup unclassified FIXME: more appropriate group.
 */
char *get_algoname(int keytype)
{
  switch (keytype) {
  case KEY_RSA_PUB: return "RSA"; break;
  case KEY_DSA_PUB: return "DSA"; break;
  case KEY_ECDSA_PUB: return "ECDSA"; break;
  default: return "UNKNOWN"; break;
  }
}
