/* lock.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#ifdef HAVE_MUTEX_INIT /* solaris, linux, etc */
# include <sys/mman.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_tool.h>

/*----------------------------------------
	initialize MUTEX lock
----------------------------------------*/
AILock OK_init_lock(char *name){
	AILock ret=NULL;
#ifdef HAVE_MUTEX_INIT
	int ipc_fd,i=sizeof(MTBuf)<<1;
	unsigned char buf[sizeof(MTBuf)<<1];
#else
	int ipc_fd,i=32;
	unsigned char buf[32];
#endif
	if((ret = (AILock)malloc(sizeof(struct ai_lock)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		goto error;
	}
	memset(ret,0,sizeof(struct ai_lock));

	if((ret->fd = open(name, O_RDWR))<0){
		/* create new file (first time) */
		if((ipc_fd = creat(name,O_CREAT|O_RDWR))<0){
		    OK_set_error(ERR_ST_FILEOPEN,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		    goto error;
		}
		memset(buf,0,i);
		if(write(ipc_fd,buf,i)<i){
		    OK_set_error(ERR_ST_FILEWRITE,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		    goto error;
		}
		close(ipc_fd);
		chmod(name, S_IRWXU|S_IRWXG|S_IRWXO);

		if((ret->fd = open(name, O_RDWR))<0){
		    OK_set_error(ERR_ST_FILEOPEN,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		    goto error;
		}
	}
/*
# ifdef HAVE_MUTEX_INIT
	if((ret->mbuf=(MTBuf*)mmap(NULL,sizeof(MTBuf),PROT_READ|PROT_WRITE,
				    MAP_SHARED,ret->fd,0))==MAP_FAILED){
		OK_set_error(ERR_ST_MEMMAP,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		goto error;
	}
	if(mutex_init(&ret->mbuf->mt,USYNC_PROCESS_ROBUST,0)){
		OK_set_error(ERR_ST_MUTEXINIT,ERR_LC_TOOL,ERR_PT_LOCK,NULL);
		goto error;
	}
# endif
*/
	return ret;
error:
	if (ret != NULL) {
		free(ret);
		ret = NULL;
	}
	return NULL;
}

int OK_release_lock(AILock al){
	if(al==NULL) return 0;
# ifdef HAVE_MUTEX_INIT
	if(al->locked) mutex_unlock(&al->mbuf->mt);
	if(al->mbuf) mutex_destroy(&al->mbuf->mt);
	if(al->mbuf) munmap((void*)al->mbuf,sizeof(MTBuf));
# endif
	if(al->fd) close(al->fd);
	free(al);

	return 0;
error:
	return -1;
}

int OK_lock(AILock al, int msec){
#if 0 /* defined(HAVE_MUTEX_INIT) */
	if((al==NULL)||(al->mbuf==NULL)){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_TOOL,ERR_PT_LOCK+2,NULL);
		goto error;
	}
	if(mutex_lock(&al->mbuf->mt)){
		OK_set_error(ERR_ST_MUTEXLOCK,ERR_LC_TOOL,ERR_PT_LOCK+2,NULL);
		goto error;
	}
#elif defined(HAVE_FCNTL_H)
	struct flock lock;
	struct timeval tv;

	if(al==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_TOOL,ERR_PT_LOCK+2,NULL);
		goto error;
	}

	tv.tv_sec  = 0;
	tv.tv_usec = 20*1000; /* ms */

	do{
		lock.l_type   = F_WRLCK;
		lock.l_whence = SEEK_SET;
		lock.l_start  = (off_t) 0;
		lock.l_len    = (off_t) 1;

		if(fcntl(al->fd,F_SETLK,&lock) < 0) {
			select(0,NULL,NULL,NULL,&tv); /* wait 20 ms */
			msec-=20;
# ifdef linux
			tv.tv_sec  = 0;
			tv.tv_usec = 20*1000; /* reset ms */
# endif
		}else{
			msec =-1000; 
		}
	}while(msec>0);

	if(msec != -1000){
		OK_set_error(ERR_ST_MUTEXTMOUT,ERR_LC_TOOL,ERR_PT_LOCK+2,NULL);
		goto error;
	}
#else
#endif
	al->locked = 1;

	return 0;
error:
	return -1;
}


int OK_unlock(AILock al){
#if 0 /* def HAVE_MUTEX_INIT */
	if(mutex_unlock(&al->mbuf->mt)){
		OK_set_error(ERR_ST_MUTEXUNLOCK,ERR_LC_TOOL,ERR_PT_LOCK+3,NULL);
		goto error;
	}
#elif defined(HAVE_FCNTL_H)
	struct flock lock;

	lock.l_type   = F_UNLCK;
	lock.l_whence = SEEK_SET;
	lock.l_start  = (off_t) 0;
	lock.l_len    = (off_t) 1;
	if(fcntl(al->fd, F_SETLKW, &lock) < 0) {
		OK_set_error(ERR_ST_MUTEXUNLOCK,ERR_LC_TOOL,ERR_PT_LOCK+3,NULL);
		goto error;
	}
#endif
	al->locked = 0;

	return 0;
error:
	return -1;
}
