/* hash.c */
/*
 * Copyright (c) 2014-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "aiconfig.h"

#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_tool.h>

HASHAlgorithm hash_algorithms[] = {
#ifdef HAVE_MD2
	{ "MD2",	OBJ_HASH_MD2,
	  16,
	  (HASHCompute)OK_MD2,
	  (HASHInit)MD2Init,
	  (HASHUpdate)MD2Update,
	  (HASHFinal)MD2Final
	},
#endif
#ifdef HAVE_MD4
	{ "MD4",	OBJ_HASH_MD4,
	  16,
	  (HASHCompute)OK_MD4,
	  (HASHInit)MD4Init,
	  (HASHUpdate)MD4Update,
	  (HASHFinal)MD4Final
	},
#endif
	{ "MD5",	OBJ_HASH_MD5,
	  16,
	  (HASHCompute)OK_MD5,
	  (HASHInit)MD5Init,
	  (HASHUpdate)MD5Update,
	  (HASHFinal)MD5Final
	},
	{ "SHA1",	OBJ_HASH_SHA1,
	  20,
	  (HASHCompute)OK_SHA1,
	  (HASHInit)SHA1init,
	  (HASHUpdate)SHA1update,
	  (HASHFinal)SHA1final
	},
	{ "SHA224",	OBJ_HASH_SHA224,
	  SHA224_DIGESTSIZE,
	  (HASHCompute)OK_SHA224,
	  (HASHInit)SHA224init,
	  (HASHUpdate)SHA224update,
	  (HASHFinal)SHA224final
	},
	{ "SHA256",	OBJ_HASH_SHA256,
	  SHA256_DIGESTSIZE,
	  (HASHCompute)OK_SHA256,
	  (HASHInit)SHA256init,
	  (HASHUpdate)SHA256update,
	  (HASHFinal)SHA256final
	},
	{ "SHA384",	OBJ_HASH_SHA384,
	  SHA384_DIGESTSIZE,
	  (HASHCompute)OK_SHA384,
	  (HASHInit)SHA384init,
	  (HASHUpdate)SHA384update,
	  (HASHFinal)SHA384final
	},
	{ "SHA512",	OBJ_HASH_SHA512,
	  SHA512_DIGESTSIZE,
	  (HASHCompute)OK_SHA512,
	  (HASHInit)SHA512init,
	  (HASHUpdate)SHA512update,
	  (HASHFinal)SHA512final
	},
	{ "SHA512-224",	OBJ_HASH_SHA512224,
	  SHA512224_DIGESTSIZE,
	  (HASHCompute)OK_SHA512224,
	  (HASHInit)SHA512224init,
	  (HASHUpdate)SHA512224update,
	  (HASHFinal)SHA512224final
	},
	{ "SHA512-256",	OBJ_HASH_SHA512256,
	  SHA512256_DIGESTSIZE,
	  (HASHCompute)OK_SHA512256,
	  (HASHInit)SHA512256init,
	  (HASHUpdate)SHA512256update,
	  (HASHFinal)SHA512256final
	},
	{ "SHA3-224",	OBJ_HASH_SHA3_224,
	  SHA3_224_DIGESTSIZE,
	  (HASHCompute)NRG_SHA3_224,
	  (HASHInit)SHA3_224init,
	  (HASHUpdate)SHA3_224update,
	  (HASHFinal)SHA3_224final
	},
	{ "SHA3-256",	OBJ_HASH_SHA3_256,
	  SHA3_256_DIGESTSIZE,
	  (HASHCompute)NRG_SHA3_256,
	  (HASHInit)SHA3_256init,
	  (HASHUpdate)SHA3_256update,
	  (HASHFinal)SHA3_256final
	},
	{ "SHA3-384",	OBJ_HASH_SHA3_384,
	  SHA3_384_DIGESTSIZE,
	  (HASHCompute)NRG_SHA3_384,
	  (HASHInit)SHA3_384init,
	  (HASHUpdate)SHA3_384update,
	  (HASHFinal)SHA3_384final
	},
	{ "SHA3-512",	OBJ_HASH_SHA3_512,
	  SHA3_512_DIGESTSIZE,
	  (HASHCompute)NRG_SHA3_512,
	  (HASHInit)SHA3_512init,
	  (HASHUpdate)SHA3_512update,
	  (HASHFinal)SHA3_512final
	},
	{ "SHAKE128",	OBJ_HASH_SHAKE128,
	  SHAKE128_DIGESTSIZE,
	  (HASHCompute)NRG_SHAKE128,
	  (HASHInit)SHAKE128init,
	  (HASHUpdate)SHAKE128update,
	  (HASHFinal)SHAKE128final,
	  (HASHFinal2)SHAKE128final2
	},
	{ "SHAKE256",	OBJ_HASH_SHAKE256,
	  SHAKE256_DIGESTSIZE,
	  (HASHCompute)NRG_SHAKE256,
	  (HASHInit)SHAKE256init,
	  (HASHUpdate)SHAKE256update,
	  (HASHFinal)SHAKE256final,
	  (HASHFinal2)SHAKE256final2
	},
	{ NULL, 0 }
};

/**
 * Get the Hash algorithm object by hash algorithm name.
 *
 */
HASHAlgorithm *gethashalgobyname(const char *hash_name)
{
	HASHAlgorithm *ha;

	for (ha = hash_algorithms; ha->name; ha++) {
		if (strcasecmp(ha->name, hash_name) == 0) {
			break;
		}
	}
	return ha->name ? ha : NULL;
}

/**
 * Get the hash algorithm object by hash algorithm aioid.
 *
 */
HASHAlgorithm *gethashalgobyaioid(const int hash_algo)
{
	HASHAlgorithm *ha;

	for (ha = hash_algorithms; ha->aioid; ha++) {
		if (ha->aioid == hash_algo) {
			break;
		}
	}
	return ha->aioid ? ha : NULL;
}

/**
 * Get the size of a hash value
 *
 */
int hash_size(const int hash_algo)
{
	HASHAlgorithm *ha;

	ha = gethashalgobyaioid(hash_algo);
	return ha != NULL ? ha->hash_size : -1;
}

/**
 * Get the aioid of the hash algorithm
 *
 */
int get_hash_algo(const char *hash_name)
{
	HASHAlgorithm *ha;

	ha = gethashalgobyname(hash_name);
	return ha != NULL ? ha->aioid : -1;
}
