/* error.c */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_err.h>

static uint32_t aicrypto_error=0;
static CK_RV *error_info = NULL;

/* set error */
void OK_set_error(int error,int location,int point,CK_RV *info){
	aicrypto_error = (uint32_t)(error|(location<<16)|(point<<24));
	error_info = info;
	OK_print_error();
}

void OK_set_errorlocation(int location,int point){
	aicrypto_error = (uint32_t)((aicrypto_error&0xffff)|(location<<16)|(point<<24));
}

/* clear error */
void OK_clear_error(){
	aicrypto_error = 0;
	error_info = NULL;
}

/* get error */
uint32_t OK_get_error(){
	return aicrypto_error;
}

CK_RV *OK_get_errorinfo(){
	return error_info;
}

/*---------------------------------------
  AiCrypto error string
---------------------------------------*/
char *get_err_location(int err){
	static char buf[64],tmp[16];
	int point = err>>8;

	err &= 0xff;
	*buf = 0;

	switch(err){
	case ERR_LC_NON:
		strcat(buf,"NO ERROR");
		break;
	case ERR_LC_LNM:
		strcat(buf,"LNM:");
		switch(point&0xf0){
		case ERR_PT_LNMADD: strcat(buf,"LNMADD"); break;
		case ERR_PT_LNMSUB: strcat(buf,"LNMSUB"); break;
		case ERR_PT_LNMMUL: strcat(buf,"LNMMULTI"); break;
		case ERR_PT_LNMDIV: strcat(buf,"LNMDIV"); break;
		case ERR_PT_LNMSQR: strcat(buf,"LNMSQR"); break;
		case ERR_PT_LNMSHF: strcat(buf,"LNMSHIFT"); break;
		case ERR_PT_LNMSET: strcat(buf,"LNMSET"); break;
		case ERR_PT_LNMLONG: strcat(buf,"LNMLONG"); break;
		case ERR_PT_LNMSYS: strcat(buf,"LNMSYS"); break;
		case ERR_PT_LNMRAND: strcat(buf,"LNMRAND"); break;
		case ERR_PT_LNMPRIME: strcat(buf,"LNMPRIME"); break;
		case ERR_PT_LNMSQRT: strcat(buf,"LNMSQRT"); break;
		case ERR_PT_LNMMONT: strcat(buf,"LNMMONT"); break;
		}
		break;
	case ERR_LC_ECC:
		strcat(buf,"ECC:");
		switch(point&0xf0){
		case ERR_PT_ECC: strcat(buf,"ECC"); break;
		case ERR_PT_ECCADD: strcat(buf,"ECCADD"); break;
		case ERR_PT_ECCMUL: strcat(buf,"ECCMUL"); break;
		case ERR_PT_ECCPADD: strcat(buf,"ECCPADD"); break;
		case ERR_PT_ECCPMUL: strcat(buf,"ECCPMUL"); break;
		case ERR_PT_ECCTOOL: strcat(buf,"ECCTOOL"); break;
		case ERR_PT_ECCCONV: strcat(buf,"ECCCONV"); break;
		case ERR_PT_ECCSTD: strcat(buf,"ECCSTD"); break;
		case ERR_PT_ECCGEN: strcat(buf,"ECCGEN"); break;
		case ERR_PT_ECCVFY: strcat(buf,"ECCVFY"); break;
		case ERR_PT_ECCASN1: strcat(buf,"ECCASN1"); break;
		}
		break;
	case ERR_LC_RAND:
		strcat(buf,"RAND:");
		switch(point&0xf0){
		case ERR_PT_RAND: strcat(buf,"RAND"); break;
		case ERR_PT_LUTZRAND: strcat(buf,"LUTZRAND"); break;
		case ERR_PT_LUTZSEED: strcat(buf,"LUTZSEED"); break;
		}
		break;
	case ERR_LC_EDC:
		strcat(buf,"EDC:");
		switch(point&0xf0){
		case ERR_PT_EDC: strcat(buf,"EDC"); break;
		case ERR_PT_EDCKEY: strcat(buf,"EDCKEY"); break;
		}
		break;
	case ERR_LC_DES:
		strcat(buf,"DES:");
		switch(point&0xf0){
		case ERR_PT_DES: strcat(buf,"DES"); break;
		case ERR_PT_3DES: strcat(buf,"3DES"); break;
		case ERR_PT_DESKEY: strcat(buf,"DESKEY"); break;
		case ERR_PT_DESMODE: strcat(buf,"DESMODE"); break;
		}
		break;
	case ERR_LC_RC2:
		strcat(buf,"RC2:");
		switch(point&0xf0){
		case ERR_PT_RC2: strcat(buf,"RC2"); break;
		case ERR_PT_RC2KEY: strcat(buf,"RC2KEY"); break;
		case ERR_PT_RC2MODE: strcat(buf,"RC2MODE"); break;
		}
		break;
	case ERR_LC_RC4:
		strcat(buf,"RC4:");
		switch(point&0xf0){
		case ERR_PT_RC4: strcat(buf,"RC4"); break;
		case ERR_PT_RC4KEY: strcat(buf,"RC4KEY"); break;
		}
		break;
	case ERR_LC_AES:
		strcat(buf,"AES:");
		switch(point&0xf0){
		case ERR_PT_AES: strcat(buf,"AES"); break;
		case ERR_PT_AESKEY: strcat(buf,"AESKEY"); break;
		case ERR_PT_AESMODE: strcat(buf,"AESMODE"); break;
		}
		break;
	case ERR_LC_MODES:
		strcat(buf,"MODES:");
		switch(point&0xf0){
		case ERR_PT_GCMENC: strcat(buf,"GCMENC"); break;
		case ERR_PT_GCMDEC: strcat(buf,"GCMDEC"); break;
		}
		break;
/*
	case ERR_LC_MD2:
		strcat(buf,"MD2:");
		break;
	case ERR_LC_MD5:
		strcat(buf,"MD5:");
		break;
	case ERR_LC_SHA1:
		strcat(buf,"SHA1:");
		break;
	case ERR_LC_HMAC:
		strcat(buf,"HMAC:");
		break;
*/
	case ERR_LC_RSA:
		strcat(buf,"RSA:");
		switch(point&0xf0){
		case ERR_PT_RSA: strcat(buf,"RSA"); break;
		case ERR_PT_RSAKEY: strcat(buf,"RSAKEY"); break;
		case ERR_PT_RSAASN: strcat(buf,"RSAASN"); break;
		case ERR_PT_RSAPSS: strcat(buf,"RSAPSS"); break;
		}
		break;
	case ERR_LC_DSA:
		strcat(buf,"DSA:");
		switch(point&0xf0){
		case ERR_PT_DSA: strcat(buf,"DSA"); break;
		case ERR_PT_DSAKEY: strcat(buf,"DSAKEY"); break;
		case ERR_PT_DSAGEN: strcat(buf,"DSAGEN"); break;
		case ERR_PT_DSAASN: strcat(buf,"DSAASN"); break;
		case ERR_PT_DSASIG: strcat(buf,"DSASIG"); break;
		}
		break;
	case ERR_LC_ECDSA:
		strcat(buf,"ECDSA:");
		switch(point&0xf0){
		case ERR_PT_ECDSA: strcat(buf,"ECDSA"); break;
		case ERR_PT_ECDSAKEY: strcat(buf,"ECDSAKEY"); break;
		case ERR_PT_ECDSAASN: strcat(buf,"ECDSAASN"); break;
		}
		break;
	case ERR_LC_DH:
		strcat(buf,"DH:");
		switch(point&0xf0){
		case ERR_PT_DH: strcat(buf,"DH"); break;
		case ERR_PT_DHKEY: strcat(buf,"DHKEY"); break;
		case ERR_PT_DHASN: strcat(buf,"DHASN"); break;
		}
		break;
	case ERR_LC_ASN1:	/* asn1 .. 1 */
		strcat(buf,"ASN1:");
		switch(point&0xf0){
		case ERR_PT_ASN1: strcat(buf,"ASN1"); break;
		case ERR_PT_ASN1CERT: strcat(buf,"ASN1CERT"); break;
		case ERR_PT_ASN1CRL: strcat(buf,"ASN1CRL"); break;
		case ERR_PT_ASN1FILE: strcat(buf,"ASN1FILE"); break;
		case ERR_PT_ASN1OBJ: strcat(buf,"ASN1OBJ"); break;
		case ERR_PT_ASN1P7E: strcat(buf,"ASN1P7E"); break;
		case ERR_PT_ASN1P7S: strcat(buf,"ASN1P7S"); break;
		case ERR_PT_ASN1REQ: strcat(buf,"ASN1REQ"); break;
		case ERR_PT_ASN1RSA: strcat(buf,"ASN1RSA"); break;
		case ERR_PT_ASN1SET: strcat(buf,"ASN1SET"); break;
		case ERR_PT_ASN1PRT: strcat(buf,"ASN1PRT"); break;
		case ERR_PT_ASN1P12: strcat(buf,"ASN1P12"); break;
		case ERR_PT_ASN1EXT: strcat(buf,"ASN1EXT"); break;
		case ERR_PT_ASN1ECC: strcat(buf,"ASN1ECC"); break;
		case ERR_PT_ASN1DSA: strcat(buf,"ASN1DSA"); break;
		}
		break;
	case ERR_LC_ASN1_:	/* asn1 .. 2 */
		strcat(buf,"ASN1:");
		switch(point&0xf0){
		case ERR_PT_ASN1DH: strcat(buf,"ASN1DH"); break;
		case ERR_PT_ASN1EXTDEF: strcat(buf,"ASN1EXTDEF"); break;
		case ERR_PT_ASN1EXTMOJ: strcat(buf,"ASN1EXTMOJ"); break;
		case ERR_PT_ASN1CRTP: strcat(buf,"ASN1CRTP"); break;
		case ERR_PT_ASN1ECDSA: strcat(buf,"ASN1ECDSA"); break;
		case ERR_PT_ASN1ATTR: strcat(buf,"ASN1ATTR"); break;
		case ERR_PT_ASN1EXTIP: strcat(buf,"ASN1EXTIP"); break;
		case ERR_PT_ASN1SPKAC: strcat(buf,"ASN1SPKAC"); break;
		}
		break;
	case ERR_LC_X509:
		strcat(buf,"X509:");
		switch(point&0xf0){
		case ERR_PT_X509FILE: strcat(buf,"X509FILE"); break;
		case ERR_PT_X509TIME: strcat(buf,"X509TIME"); break;
		}
		break;
	case ERR_LC_X509CERT:
		strcat(buf,"X509CERT:");
		switch(point&0xf0){
		case ERR_PT_CERT: strcat(buf,"CERT"); break;
		case ERR_PT_CERTASN1: strcat(buf,"CERTASN1"); break;
		case ERR_PT_CERTEXT: strcat(buf,"CERTEXT"); break;
		case ERR_PT_CERTEXTNS: strcat(buf,"CERTEXTNS"); break;
		case ERR_PT_CERTEXTSTR: strcat(buf,"CERTEXTSTR"); break;
		case ERR_PT_CERTPRINT: strcat(buf,"CERTPRINT"); break;
		case ERR_PT_CERTTOOL: strcat(buf,"CERTTOOL"); break;
		case ERR_PT_CERTVFY: strcat(buf,"CERTVFY"); break;
		case ERR_PT_CLIST: strcat(buf,"CERTLIST"); break;
		case ERR_PT_CLFILE: strcat(buf,"CERTLISTFILE"); break;
		case ERR_PT_CLTOOL: strcat(buf,"CERTLISTTOOL"); break;
		case ERR_PT_CRTP: strcat(buf,"CERTPAIR"); break;
		case ERR_PT_CRTPASN1: strcat(buf,"CERTPAIR_ASN1"); break;
		}
		break;
	case ERR_LC_X509CRL:
		strcat(buf,"X509CRL:");
		switch(point&0xf0){
		case ERR_PT_CRL: strcat(buf,"CRL"); break;
		case ERR_PT_CRLASN1: strcat(buf,"CRLASN1"); break;
		case ERR_PT_CRLEXT: strcat(buf,"CRLEXT"); break;
		case ERR_PT_CRLEXTSTR: strcat(buf,"CRLEXTSTR"); break;
		case ERR_PT_CRLPRINT: strcat(buf,"CRLPRINT"); break;
		case ERR_PT_CRLVFY: strcat(buf,"CRLVFY"); break;
		}
		break;
	case ERR_LC_X509KEY:
		strcat(buf,"X509KEY:");
		switch(point&0xf0){
		case ERR_PT_KEY: strcat(buf,"KEY"); break;
		case ERR_PT_KEYTOOL: strcat(buf,"KEYTOOL"); break;
		}
		break;
	case ERR_LC_X509EXT:
		strcat(buf,"X509EXT:");
		switch(point&0xf0){
		case ERR_PT_EXTGN: strcat(buf,"EXTGN"); break;
		case ERR_PT_EXTPOL: strcat(buf,"EXTPOL"); break;
		case ERR_PT_EXTCERT: strcat(buf,"EXTCERT"); break;
		case ERR_PT_EXTCRL: strcat(buf,"EXTCRL"); break;
		case ERR_PT_EXTMS: strcat(buf,"EXTMS"); break;
		case ERR_PT_EXTMOJ: strcat(buf,"EXTMOJ"); break;
		case ERR_PT_ATTRHC: strcat(buf,"ATTRHC"); break;
		case ERR_PT_EXTIP: strcat(buf,"EXTIP"); break;
		}
		break;
	case ERR_LC_X509REQ:
		strcat(buf,"X509REQ:");
		switch(point&0xf0){
		case ERR_PT_REQASN1: strcat(buf,"REQASN1"); break;
		case ERR_PT_REQVFY: strcat(buf,"REQVFY"); break;
		}
		break;
	case ERR_LC_PKCS:
		strcat(buf,"PKCS:");
		switch(point&0xf0){
		case ERR_PT_PKCS8: strcat(buf,"PKCS8"); break;
		case ERR_PT_P8FILE: strcat(buf,"P8FILE"); break;
		case ERR_PT_PBE: strcat(buf,"PBE"); break;
		case ERR_PT_PBECRY: strcat(buf,"PBECRY"); break;
		case ERR_PT_PBEKEY: strcat(buf,"PBEKEY"); break;
		case ERR_PT_DECINFO: strcat(buf,"DECINFO"); break;
		}
		break;
	case ERR_LC_PKCS7:
		strcat(buf,"PKCS7:");
		switch(point&0xf0){
		case ERR_PT_PKCS7: strcat(buf,"PKCS8"); break;
		case ERR_PT_P7DATA: strcat(buf,"P7DATA"); break;
		case ERR_PT_P7ENC: strcat(buf,"P7ENC"); break;
		case ERR_PT_P7ENV: strcat(buf,"P7ENV"); break;
		case ERR_PT_P7FILE: strcat(buf,"P7FILE"); break;
		case ERR_PT_P7SIGN: strcat(buf,"P7SIGN"); break;
		case ERR_PT_P7MASN1: strcat(buf,"P7MASN1"); break;
		case ERR_PT_P7SASN1: strcat(buf,"P7SASN1"); break;
		case ERR_PT_P7SATTR: strcat(buf,"P7SATTR"); break;
		}
		break;
	case ERR_LC_PKCS11:
		strcat(buf,"PKCS11:");
		switch(point&0xf0){
		case ERR_PT_PKCS11: strcat(buf,"PKCS11"); break;
		case ERR_PT_P11SESS: strcat(buf,"P11SESS"); break;
		case ERR_PT_P11TOOL: strcat(buf,"P11TOOL"); break;
		case ERR_PT_P11KEY: strcat(buf,"P11KEY"); break;
		}
		break;
	case ERR_LC_PKCS12:
		strcat(buf,"PKCS12:");
		switch(point&0xf0){
		case ERR_PT_PKCS12: strcat(buf,"PKCS12"); break;
		case ERR_PT_P12ASN1: strcat(buf,"P12ASN1"); break;
		case ERR_PT_P12FILE: strcat(buf,"P12FILE"); break;
		case ERR_PT_P12KEY: strcat(buf,"P12KEY"); break;
		case ERR_PT_P12MAC: strcat(buf,"P12MAC"); break;
		case ERR_PT_P12TOOL: strcat(buf,"P12TOOL"); break;
		}
		break;
	case ERR_LC_PEM:
		strcat(buf,"PEM:");
		switch(point&0xf0){
		case ERR_PT_BASE64: strcat(buf,"BASE64"); break;
		case ERR_PT_PEM: strcat(buf,"PEM"); break;
		case ERR_PT_PEMCRY: strcat(buf,"PEMCRY"); break;
		case ERR_PT_PEMMSG: strcat(buf,"PEMMSG"); break;
		case ERR_PT_PEMWRITE: strcat(buf,"PEMWRITE"); break;
		case ERR_PT_PEMPKCS: strcat(buf,"PEMPKCS"); break;
		}
		break;
	case ERR_LC_SMIME:
		strcat(buf,"SMIME:");
		switch(point&0xf0){
		case ERR_PT_SMIME_DEC: strcat(buf,"SMIMEDEC"); break;
		case ERR_PT_SMIME_ENC: strcat(buf,"SMIMEENC"); break;
		case ERR_PT_MIME_HEAD: strcat(buf,"MIMEHEAD"); break;
		}
		break;
	case ERR_LC_SSL:
		strcat(buf,"SSL:");
		switch(point&0xf0){
		case ERR_PT_SSL: strcat(buf,"SSL"); break;
		case ERR_PT_SSL_BIND: strcat(buf,"BIND"); break;
		case ERR_PT_SSL_CB: strcat(buf,"CALL BACK"); break;
		case ERR_PT_SSL_CS: strcat(buf,"CIPHER SPEC"); break;
		case ERR_PT_SSL_HELLO: strcat(buf,"HELLO"); break;
		case ERR_PT_SSL_LIST: strcat(buf,"LIST"); break;
		case ERR_PT_SSL_NAME: strcat(buf,"NAME"); break;
		case ERR_PT_SSL_RAND: strcat(buf,"RAND"); break;
		case ERR_PT_SSL_READ: strcat(buf,"READ"); break;
		case ERR_PT_SSL_SOCK: strcat(buf,"SOCK"); break;
		case ERR_PT_SSL_TOOL: strcat(buf,"TOOL"); break;
		case ERR_PT_SSL_VFY: strcat(buf,"VERIFY"); break;
		case ERR_PT_SSL_WRITE: strcat(buf,"WRITE"); break;
		}
		break;
	case ERR_LC_SSLHS:
		strcat(buf,"SSLHS:");
		switch(point&0xf0){
		case ERR_PT_SSLHS: strcat(buf,"HAND SHAKE"); break;
		case ERR_PT_SSLHS_CLNT: strcat(buf,"HAND SHAKE CLIENT"); break;
		case ERR_PT_SSLHS_KEY: strcat(buf,"HAND SHAKE KEY"); break;
		case ERR_PT_SSLHS_SERV: strcat(buf,"HAND SHAKE SERV"); break;
		}
		break;
	case ERR_LC_SSLREC:
		strcat(buf,"SSLREC:");
		switch(point&0xf0){
		case ERR_PT_SSLREC: strcat(buf,"RECORD"); break;
		case ERR_PT_SSLREC_PROC: strcat(buf,"RECORD PROC"); break;
		}
		break;
	case ERR_LC_SSLALERT:
		strcat(buf,"SSLALERT:");
		switch(point&0xf0){
		case ERR_PT_SSLALERT: strcat(buf,"ALERT"); break;
		}
		break;
	case ERR_LC_TOOL:
		strcat(buf,"TOOL:");
		switch(point&0xf0){
		case ERR_PT_DIGEST: strcat(buf,"DIGEST"); break;
		case ERR_PT_SIG: strcat(buf,"SIGNATURE"); break;
		case ERR_PT_PASS: strcat(buf,"PASSWORD"); break;
		case ERR_PT_LOCK: strcat(buf,"LOCK"); break;
		}
		break;
	case ERR_LC_STORE:
		strcat(buf,"STORE:");
		switch(point&0xf0){
		case ERR_PT_STORE: strcat(buf,"STORE"); break;
		case ERR_PT_STADD: strcat(buf,"STORE ADD"); break;
		case ERR_PT_STDEL: strcat(buf,"STORE DEL"); break;
		case ERR_PT_STSEARCH: strcat(buf,"STORE SEARCH"); break;
		case ERR_PT_STTOOL: strcat(buf,"STORE TOOL"); break;
		case ERR_PT_MANAGER: strcat(buf,"MANAGER"); break;
		case ERR_PT_MANADD: strcat(buf,"MANAGER ADD"); break;
		case ERR_PT_MANDEL: strcat(buf,"MANAGER DEL"); break;
		case ERR_PT_MANSEARCH: strcat(buf,"MANAGER SEARCE"); break;
		case ERR_PT_MANASN1: strcat(buf,"MANAGER ASN1"); break;
		case ERR_PT_MANTOOL: strcat(buf,"MANAGER TOOL"); break;
		}
		break;
	case ERR_LC_STOREDEV:
		strcat(buf,"STORE DEV:");
		switch(point&0xf0){
		case ERR_PT_STFILE: strcat(buf,"STFILE"); break;
		case ERR_PT_STFILEMETH: strcat(buf,"STFILEMETH"); break;
		}
		break;
	case ERR_LC_WINCRY:
		strcat(buf,"WINCRY:");
		switch(point&0xf0){
		case ERR_PT_WINCRY_CERT: strcat(buf,"CERT"); break;
		case ERR_PT_WINCRY_CLIST: strcat(buf,"CLIST"); break;
		case ERR_PT_WINCRY_CRL: strcat(buf,"CRL"); break;
		case ERR_PT_WINCRY_KEY: strcat(buf,"KEY"); break;
		}
		break;
	case ERR_LC_UCONV:
		strcat(buf,"UCONV:");
		switch(point&0xf0){
		case ERR_PT_UCONV: strcat(buf,"UCONV"); break;
		case ERR_PT_UC_JIS: strcat(buf,"UC_JIS"); break;
		case ERR_PT_UC_SJIS: strcat(buf,"UC_SJIS"); break;
		case ERR_PT_UC_EUC: strcat(buf,"UC_EUC"); break;
		case ERR_PT_UC_UNI: strcat(buf,"UC_UNI"); break;
		case ERR_PT_UC_UTF8: strcat(buf,"UC_UTF8"); break;
		}
		break;
	case ERR_LC_SPKAC:
		strcat(buf,"SPKAC:");
		switch(point&0xf0){
		case ERR_PT_SPKAC: strcat(buf,"SPKAC"); break;
		case ERR_PT_SPKACASN1: strcat(buf,"SPKACASN1"); break;
		case ERR_PT_SPKACPRINT: strcat(buf,"SPKACPRINT"); break;
		}
		break;

	/* tls implementation */
	case ERR_LC_TLS1:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS:                   strcat(buf, "TLS");           break;
		case ERR_PT_TLS_ALERT:             strcat(buf, "ALERT");         break;
		case ERR_PT_TLS_CCS:               strcat(buf, "CCS");           break;
		case ERR_PT_TLS_CERT:              strcat(buf, "CERT");          break;
		case ERR_PT_TLS_CERT2:             strcat(buf, "CERT");          break;
		case ERR_PT_TLS_CIPHER:            strcat(buf, "CIPHER");        break;
		case ERR_PT_TLS_CIPHER_AEAD:       strcat(buf, "CIPHER/AEAD");   break;
		case ERR_PT_TLS_CIPHER_BLOCK:      strcat(buf, "CIPHER/BLOCK");  break;
		case ERR_PT_TLS_CIPHER_STREAM:     strcat(buf, "CIPHER/STREAM"); break;
		case ERR_PT_TLS_COMPAT:            strcat(buf, "COMPAT");        break;
		case ERR_PT_TLS_COMPRESS:          strcat(buf, "COMPRESS");      break;
		}
		break;

	case ERR_LC_TLS2:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS_HS:                strcat(buf, "HS");              break;
		case ERR_PT_TLS_HS_CS_CLIENT:      strcat(buf, "HS/CS/CLIENT");    break;
		case ERR_PT_TLS_HS_CS_SERVER:      strcat(buf, "HS/CS/SERVER");    break;
		case ERR_PT_TLS_HS_CS_FINALE:      strcat(buf, "HS/CS/FINALE");    break;
		case ERR_PT_TLS_HS_MSG_CERT:       strcat(buf, "HS/MSG/CERT");     break;
		case ERR_PT_TLS_HS_MSG_CERTREQ:    strcat(buf, "HS/MSG/CERTREQ");  break;
		case ERR_PT_TLS_HS_MSG_CERTVFY:    strcat(buf, "HS/MSG/CERTVFY");  break;
		case ERR_PT_TLS_HS_MSG_CHELLO:     strcat(buf, "HS/MSG/CHELLO");   break;
		case ERR_PT_TLS_HS_MSG_CKEYEXC:    strcat(buf, "HS/MSG/CKEYEXC");  break;
		case ERR_PT_TLS_HS_MSG_FINISHED:   strcat(buf, "HS/MSG/FINISHED"); break;
		case ERR_PT_TLS_HS_MSG_HELLOREQ:   strcat(buf, "HS/MSG/HELLOREQ"); break;
		}
		break;

	case ERR_LC_TLS3:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS_HS_MSG_SHELLO:     strcat(buf, "HS/MSG/SHELLO");     break;
		case ERR_PT_TLS_HS_MSG_SHELLODONE: strcat(buf, "HS/MSG/SHELLODONE"); break;
		case ERR_PT_TLS_HS_MSG_SKEYEXC:    strcat(buf, "HS/MSG/SKEYEXC");    break;
		case ERR_PT_TLS_HS_EXT_SIGHASH:    strcat(buf, "HS/EXT/SIGHASH");    break;
		case ERR_PT_TLS_HS_UTIL_HASH:      strcat(buf, "HS/UTIL/HASH");      break;
		case ERR_PT_TLS_HS_UTIL_MSG:       strcat(buf, "HS/UTIL/MSG");       break;
		case ERR_PT_TLS_HS_UTIL_SIGNATURE: strcat(buf, "HS/UTIL/SIGNATURE"); break;
		case ERR_PT_TLS_HS_EXT_ECC:        strcat(buf, "HS/EXT/ECC");        break;
		case ERR_PT_TLS_HS_ECDH_ECDH:      strcat(buf, "HS/ECDH/ECDH");      break;
		case ERR_PT_TLS_HS_ECDH_ECDHKEY:   strcat(buf, "HS/ECDH/ECDHKEY");   break;
		}
		break;

	case ERR_LC_TLS4:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS_KEY:               strcat(buf, "KEY");     break;
		case ERR_PT_TLS_MAC:               strcat(buf, "MAC");     break;
		case ERR_PT_TLS_OPT:               strcat(buf, "OPT");     break;
		case ERR_PT_TLS_PRF:               strcat(buf, "PRF");     break;
		case ERR_PT_TLS_RECORD:            strcat(buf, "RECORD");  break;
		case ERR_PT_TLS_SESSION:           strcat(buf, "SESSION"); break;
		case ERR_PT_TLS_SOCKET:            strcat(buf, "SOCKET");  break;
		case ERR_PT_TLS_STM:               strcat(buf, "STM");     break;
		case ERR_PT_TLS_UTIL:              strcat(buf, "UTIL");    break;
		}
		break;

	case ERR_LC_TLS5:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS_HS_EXT_COOKIE:             strcat(buf, "HS/EXT/COOKIE");             break;
		case ERR_PT_TLS_HS_EXT_KEYSHARE:           strcat(buf, "HS/EXT/KEYSHARE");           break;
		case ERR_PT_TLS_HS_EXT_PARSE:              strcat(buf, "HS/EXT/PARSE");              break;
		case ERR_PT_TLS_HS_EXT_SERVERNAME:         strcat(buf, "HS/EXT/SERVERNAME");         break;
		case ERR_PT_TLS_HS_EXT_SUPPORTED_VERSIONS: strcat(buf, "HS/EXT/SUPPORTED_VERSIONS"); break;
		case ERR_PT_TLS_HS_EXT_SIGHASH2:           strcat(buf, "HS/EXT/SIGHASH");            break;
		case ERR_PT_TLS_HS_CS_CLIENT2:             strcat(buf, "HS/CS/CLIENT");              break;
		case ERR_PT_TLS_HS_CS_SERVER2:             strcat(buf, "HS/CS/SERVER");              break;
		}
		break;

	case ERR_LC_TLS6:
		strcat(buf, "TLS:");
		switch(point & 0xf0) {
		case ERR_PT_TLS_HS_MSG_CERT2:          strcat(buf, "HS/MSG/CERT");      break;
		case ERR_PT_TLS_HS_MSG_CERTREQ2:       strcat(buf, "HS/MSG/CERTREQ");   break;
		case ERR_PT_TLS_HS_MSG_ENCEXT:         strcat(buf, "HS/MSG/ENCEXT");    break;
		case ERR_PT_TLS_HS_MSG_KEYUPDATE:      strcat(buf, "HS/MSG/KEYUPDATE"); break;
		case ERR_PT_TLS_HS_MSG_MSGHASH:        strcat(buf, "HS/MSG/MSGHASH");   break;
		case ERR_PT_TLS_HS_MSG_SHELLO2:        strcat(buf, "HS/MSG/SHELLO");    break;
		case ERR_PT_TLS_RECORD2:               strcat(buf, "RECORD");           break;
		case ERR_PT_TLS_KEY2:                  strcat(buf, "KEY");              break;
		case ERR_PT_TLS_CIPHER2:               strcat(buf, "CIPHER");           break;

		}
		break;

	default:
		strcat(buf,"UNKNOWN");
		break;
	}
	sprintf(tmp,"(%d)",point&0xf);
	strcat(buf,tmp);
	return buf;
}

char *get_err_type(int err){
	static char tmp[32];
	char *ret;
	switch(err){
	case ERR_ST_NON: ret="no error"; break;
	case ERR_ST_MEMALLOC: ret="memory alloc"; break;
	case ERR_ST_NULLPOINTER: ret="null pointer"; break;
	case ERR_ST_BADPARAM: ret="bad parameter"; break;
	case ERR_ST_BADFORMAT: ret="bad format"; break;
	case ERR_ST_BADVER: ret="bad version"; break;
	case ERR_ST_BADPADDING: ret="bad padding"; break;
	case ERR_ST_UNMATCHEDPARAM: ret="parameter unmatched"; break;
	case ERR_ST_STRDUP: ret="strdup error"; break;
	case ERR_ST_BADSTATE: ret="bad state"; break;
	case ERR_ST_UNSUPPORTED_ALGO: ret="unsupported algorithm"; break;
	case ERR_ST_UNSUPPORTED_VER: ret="unsupported version"; break;
	case ERR_ST_UNSUPPORTED_PARAM: ret="unsupported parameter"; break;
	case ERR_ST_UNSUPPORTED_CODE: ret="unsupported code"; break;
	case ERR_ST_BADNAME: ret="bad name"; break;
	case ERR_ST_NULLKEY: ret="null key"; break;
	case ERR_ST_BADKEY: ret="bad key"; break;
	case ERR_ST_FILEOPEN: ret="cannot open file"; break;
	case ERR_ST_FILEREAD: ret="cannot read file"; break;
	case ERR_ST_FILEWRITE: ret="cannot write file"; break;
	case ERR_ST_MEMMAP: ret="cannot map memory"; break;
	case ERR_ST_LOADLIB: ret="cannot load dynamic library"; break;
	case ERR_ST_GETPROCADDR: ret="cannot get function address"; break;
	case ERR_ST_FILERENAME: ret="cannot rename file"; break;
	/* RFC8017 output */
	case ERR_ST_RSA_MSGTOOLONG: ret="message too long"; break;
	case ERR_ST_RSA_MSKTOOLONG: ret="mask too long"; break;
	case ERR_ST_RSA_ENCODING: ret="encoding error"; break;
	case ERR_ST_RSA_INCONSISTENT: ret="inconsistent"; break;

	case ERR_ST_LNM_BUFOVERFLOW: ret="buffer over flow"; break;
	case ERR_ST_LNM_DIVBYZERO: ret="divide by zero"; break;
	case ERR_ST_LNM_NOSQRT: ret="square root doesn't exist"; break;

	case ERR_ST_ASN_NOTINTEGER: ret="asn1 not integer"; break;
	case ERR_ST_ASN_NOTENUMERATED: ret="asn1 not enumerated"; break;
	case ERR_ST_ASN_NOTBITSTR: ret="asn1 not bit-string"; break;
	case ERR_ST_ASN_NOTOCTETSTR: ret="asn1 not octet string"; break;
	case ERR_ST_ASN_NOTOID: ret="asn1 not object-id"; break;
	case ERR_ST_ASN_NOTNUMERICSTR: ret="asn1 not numeric-string"; break;
	case ERR_ST_ASN_NOTPRINTABLESTR: ret="asn1 not printable-string"; break;
	case ERR_ST_ASN_NOTUTF8STR: ret="asn1 not utf8-string"; break;
	case ERR_ST_ASN_NOTT61STR: ret="asn1 not t61-string"; break;
	case ERR_ST_ASN_NOTIA5STR: ret="asn1 not ia5-string"; break;
	case ERR_ST_ASN_NOTBMPSTR: ret="asn1 not bmp-string"; break;
	case ERR_ST_ASN_NOTISO64STR: ret="asn1 not iso64-string"; break;
	case ERR_ST_ASN_NOTUTCTIME: ret="asn1 not utc-time"; break;
	case ERR_ST_ASN_NOTGENTIME: ret="asn1 not generalized-time"; break;
	case ERR_ST_ASN_UNKNOWNOID: ret="asn1 unknown object-id"; break;
	case ERR_ST_ASN_BADOID: ret="asn1 bad object-id"; break;
	case ERR_ST_ASN_NOTASN1: ret="not asn1 format"; break;
	case ERR_ST_ASN_NOTBOOLEAN: ret="asn1 not boolean"; break;
	case ERR_ST_ASN_INVALIDLENGTH: ret="asn1 invalid length"; break;

	case ERR_ST_P12_BADDEPTH: ret="pkcs12 bad depth"; break;
	case ERR_ST_P12_NOBAG: ret="pkcs12 no bag"; break;
	case ERR_ST_P12_NOCERT: ret="pkcs12 no cert"; break;
	case ERR_ST_P12_NOCRL: ret="pkcs12 no crl"; break;
	case ERR_ST_P12_NOKEY: ret="pkcs12 no key"; break;
	case ERR_ST_P12_BADMAC: ret="pkcs12 bad mac"; break;

	case ERR_ST_GETFUNCLIST: ret="pkcs11 no function list"; break;
	case ERR_ST_P11_INIT: ret="pkcs11 not initialized"; break;
	case ERR_ST_P11_GETINFO: ret="pkcs11 get info error"; break;
	case ERR_ST_P11_SLOTLIST: ret="pkcs11 slot list error"; break;
	case ERR_ST_P11_SLOTINFO: ret="pkcs11 slot info error"; break;
	case ERR_ST_P11_TKNINFO: ret="pkcs11 token info error"; break;
	case ERR_ST_P11_OPENSS: ret="pkcs11 not open session"; break;
	case ERR_ST_P11_CLOSESS: ret="pkcs11 not close session"; break;
	case ERR_ST_P11_LOGIN: ret="pkcs11 login error"; break;
	case ERR_ST_P11_LOGOUT: ret="pkcs11 logout error"; break;
	case ERR_ST_P11_GENKEY: ret="pkcs11 key generation error"; break;
	case ERR_ST_P11_GETATTR: ret="pkcs11 get attribute error"; break;
	case ERR_ST_P11_SETATTR: ret="pkcs11 set attribute error"; break;
	case ERR_ST_P11_CREOBJ: ret="pkcs11 create object error"; break;
	case ERR_ST_P11_FOBJINIT: ret="pkcs11 find object init error"; break;
	case ERR_ST_P11_FOBJCT: ret="pkcs11 no object found"; break;
	case ERR_ST_P11_FOBJFIN: ret="pkcs11 find object final error"; break;
	case ERR_ST_P11_SIGINIT: ret="pkcs11 sign init error"; break;
	case ERR_ST_P11_SIGN: ret="pkcs11 sign error"; break;
	case ERR_ST_P11_SIGFIN: ret="pkcs11 sign final error"; break;
	case ERR_ST_P11_VFYINIT: ret="pkcs11 verify init error"; break;
	case ERR_ST_P11_VERIFY: ret="pkcs11 verify error"; break;
	case ERR_ST_P11_VFYFIN: ret="pkcs11 verify final error"; break;
	case ERR_ST_P11_ENCINIT: ret="pkcs11 encrypt init error"; break;
	case ERR_ST_P11_ENCRYPT: ret="pkcs11 encrypt error"; break;
	case ERR_ST_P11_ENCFIN: ret="pkcs11 encrypt final error"; break;
	case ERR_ST_P11_DECINIT: ret="pkcs11 decrypt init error"; break;
	case ERR_ST_P11_DECRYPT: ret="pkcs11 decrypt error"; break;
	case ERR_ST_P11_DECFIN: ret="pkcs11 decrypt final error"; break;
	case ERR_ST_P11_WRAP: ret="pkcs11 wrap key error"; break;
	case ERR_ST_P11_UNWRAP: ret="pkcs11 unwrap key error"; break;
	case ERR_ST_P11_ACTMOFN: ret="pkcs11 activate m of n error"; break;
	case ERR_ST_P11_GETSINFO: ret="pkcs11 get session info error"; break;

	case ERR_ST_PEM_BADHEADER: ret="pem bad header"; break;
	case ERR_ST_PEM_BADFOOTER: ret="pem bad footer"; break;
	case ERR_ST_PEM_BADPASSWD: ret="pem bad password"; break;
	case ERR_ST_P1_BADPADDING: ret="pkcs1 bad padding"; break;
	case ERR_ST_MUTEXINIT: ret="mutex not initialized"; break;
	case ERR_ST_MUTEXOPEN: ret="mutex not open"; break;
	case ERR_ST_MUTEXCLOSE: ret="mutex not closed"; break;
	case ERR_ST_MUTEXLOCK: ret="mutex bad lock"; break;
	case ERR_ST_MUTEXUNLOCK: ret="mutex bad unlock"; break;
	case ERR_ST_MUTEXTMOUT: ret="mutex lock timeout"; break;

	case ERR_ST_STO_MANAGNOTFOUND: ret="store manager not found"; break;
	case ERR_ST_STO_STORENOTFOUND: ret="store not found"; break;
	case ERR_ST_STO_BAGNOTFOUND: ret="bag not found"; break;
	case ERR_ST_STO_BADMANAG: ret="invalid store manager"; break;
	case ERR_ST_STO_BADSTORE: ret="invalid store"; break;
	case ERR_ST_STO_BADBAG: ret="invalid store bag"; break;
	case ERR_ST_STO_BADID: ret="invalid unique id"; break;

	case ERR_ST_WINAPI: ret="windows CryptAPI"; break;

	case ERR_ST_UC_BADJISCODE: ret="bad JIS code"; break;
	case ERR_ST_UC_BADUTF8CODE: ret="bad UTF8 code"; break;
	case ERR_ST_UC_UNKNOWNCODE: ret="unknown code"; break;

	case ERR_ST_MIME_BADHEADER: ret="mime bad header"; break;
	case ERR_ST_MIME_BADFOOTER: ret="mime bad footer"; break;

	case ERR_ST_RAND_NOPOOL: ret="random pool is empty"; break;
	case ERR_ST_RAND_NOTSEEDED: ret="random pool is not seeded"; break;

	case ERR_ST_SSL_CLOSE_NOTIFY: ret="ssl close notify"; break;
	case ERR_ST_SSL_UNEXPECTED_MESSAGE: ret="ssl unexpected message"; break;
	case ERR_ST_SSL_BAD_RECORD_MAC: ret="ssl bad record mac"; break;
	case ERR_ST_SSL_DECOMPRESSION_FAILURE: ret="ssl decompression failure"; break;
	case ERR_ST_SSL_HAND_SHAKE_FAILURE: ret="ssl hand shake failure"; break;
	case ERR_ST_SSL_NO_CERT: ret="ssl no cert"; break;
	case ERR_ST_SSL_BAD_CERT: ret="ssl bad cert"; break;
	case ERR_ST_SSL_UNSUPPORTED_CERT: ret="ssl unsupported cert"; break;
	case ERR_ST_SSL_CERT_REVOKED: ret="ssl cert revoked"; break;
	case ERR_ST_SSL_CERT_EXPIRED: ret="ssl cert expired"; break;
	case ERR_ST_SSL_CERT_UNKNOWN: ret="ssl cert unknown"; break;
	case ERR_ST_SSL_ILLEGAL_PARAMETER: ret="ssl illegal parameter"; break;
	case ERR_ST_SSL_WRITE: ret="ssl write"; break;
	case ERR_ST_SSL_READ: ret="ssl read"; break;
	case ERR_ST_SSL_BADHEADER: ret="ssl bad header"; break;
	case ERR_ST_SSL_BADSIGNATURE: ret="ssl bad signature"; break;
	case ERR_ST_SSL_BADFINISHED: ret="ssl bad finished"; break;
	case ERR_ST_SSL_CERT_BADSIG: ret="ssl cert bad signature"; break;
	case ERR_ST_SSL_CERT_BADCHAIN: ret="ssl cert bad chain"; break;
	case ERR_ST_SSL_CRL_EXPIRED: ret="ssl crl expired"; break;
	case ERR_ST_SSL_CRL_BADSIG: ret="ssl crl bad signature"; break;
	case ERR_ST_SSL_CRL_NOTFOUND: ret="ssl crl not found"; break;
	case ERR_ST_SSL_CERT_UNKNOWNCA: ret="ssl cert untrusted"; break;

	case ERR_ST_ACCEPT: ret="sock accept"; break;
	case ERR_ST_CONNECT: ret="sock connect"; break;
	case ERR_ST_SOCKOPEN: ret="sock open"; break;
	case ERR_ST_SOCKWRITE: ret="sock write"; break;
	case ERR_ST_SOCKREAD: ret="sock read"; break;
	case ERR_ST_SOCKBIND: ret="sock bind"; break;
	case ERR_ST_SOCKLISTEN: ret="sock listen"; break;
	case ERR_ST_SELECT: ret="select error"; break;
	case ERR_ST_SELTIMEOUT: ret="select timeout"; break;

	case ERR_ST_BADCERTID: ret="bad cert id"; break;
	case ERR_ST_BADKEYID: ret="bad key id"; break;

	case ERR_ST_EDC_DERIVATION_FAILURE: ret="edc derivation failure"; break;
	case ERR_ST_EDC_ALL_ZERO_VALUE: ret="edc all zero value"; break;

	/* tls */
	case ERR_ST_TLS_CALLOC:                         ret = "calloc";                          break;
	case ERR_ST_TLS_MALLOC:                         ret = "malloc";                          break;
	case ERR_ST_TLS_REALLOC:                        ret = "realloc";                         break;
	case ERR_ST_TLS_SNPRINTF:                       ret = "snprintf";                        break;
	case ERR_ST_TLS_STRDUP:                         ret = "strdup";                          break;
	case ERR_ST_TLS_SOCKET:                         ret = "socket";                          break;
	case ERR_ST_TLS_CONNECT:                        ret = "connect";                         break;
	case ERR_ST_TLS_TLS_BIND:                       ret = "bind";                            break;
	case ERR_ST_TLS_LISTEN:                         ret = "listen";                          break;
	case ERR_ST_TLS_ACCEPT:                         ret = "accept";                          break;
	case ERR_ST_TLS_PSELECT:                        ret = "pselect";                         break;
	case ERR_ST_TLS_READ:                           ret = "read";                            break;
	case ERR_ST_TLS_WRITE:                          ret = "write";                           break;
	case ERR_ST_TLS_PSELECT_TIMEOUT:                ret = "read/write timeout";              break;
	case ERR_ST_TLS_DISCONNECT:                     ret = "read/write disconnect";           break;
	case ERR_ST_TLS_TLS_RECORD_INIT:                ret = "tls_record_init";                 break;
	case ERR_ST_TLS_INIT_RECORD_Q:                  ret = "init_record_queue";               break;
	case ERR_ST_TLS_INIT_HS:                        ret = "init_handshake";                  break;
	case ERR_ST_TLS_TLS_NEW:                        ret = "TLS_new";                         break;
	case ERR_ST_TLS_TLS_DUP:                        ret = "TLS_dup";                         break;
	case ERR_ST_TLS_COPY_CERTS:                     ret = "copy_certs";                      break;
	case ERR_ST_TLS_TLS_NULL:                       ret = "tls (null)";                      break;
	case ERR_ST_TLS_UNKNOWN_ALERT_LEVEL:            ret = "unknown alert level";             break;
	case ERR_ST_TLS_INVALID_ALERT_LEVEL:            ret = "invalid alert level";             break;
	case ERR_ST_TLS_INVALID_ALERT_DESC:             ret = "invalid alert desc";              break;
	case ERR_ST_TLS_TLS_RECORD_WRITE:               ret = "tls_record_write";                break;
	case ERR_ST_TLS_INVALID_RECORD_LENGTH:          ret = "invalid record length";           break;
	case ERR_ST_TLS_INVALID_STATUS:                 ret = "invalid status";                  break;
	case ERR_ST_TLS_TLS_CIPHER_PARAM_SET:           ret = "failed to set cipher param";      break;
	case ERR_ST_TLS_INVALID_CCS_BODY:               ret = "invalid ccs body";                break;
	case ERR_ST_TLS_P12_CHECK_CHAIN:                ret = "P12_check_chain";                 break;
	case ERR_ST_TLS_X509_REVOKED:                   ret = "cert: revoked";                   break;
	case ERR_ST_TLS_X509_NOTBEFORE:                 ret = "cert: not before";                break;
	case ERR_ST_TLS_X509_NOTAFTER:                  ret = "cert: not after";                 break;
	case ERR_ST_TLS_X509_LASTUPDATE:                ret = "cert: lastupdate";                break;
	case ERR_ST_TLS_X509_NEXTUPDATE:                ret = "cert: nextupdate";                break;
	case ERR_ST_TLS_X509_SIGNATURE:                 ret = "cert: signature";                 break;
	case ERR_ST_TLS_X509_SIGNATURE_CRL:             ret = "cert: signature/crl";             break;
	case ERR_ST_TLS_X509_ISSUER_CRL:                ret = "cert: issuer/crl";                break;
	case ERR_ST_TLS_X509_NOT_IN_CERTLIST:           ret = "cert: not in certlist";           break;
	case ERR_ST_TLS_X509_SELF_SIGN:                 ret = "cert: self sign";                 break;
	case ERR_ST_TLS_X509_NOT_CACERT:                ret = "cert: not ca cert";               break;
	case ERR_ST_TLS_X509_SYSTEMERR:                 ret = "cert: system error";              break;
	case ERR_ST_TLS_CHECK_PKCS12:                   ret = "check_pkcs12";                    break;
	case ERR_ST_TLS_P12_READ_FILE:                  ret = "P12_read_file";                   break;
	case ERR_ST_TLS_CERT_SET_BY_PKCS12:             ret = "TLS_cert_set_by_pkcs12";          break;
	case ERR_ST_TLS_STM_FIND:                       ret = "TLS_stm_find";                    break;
	case ERR_ST_TLS_P12_GET_USERCERT:               ret = "P12_usercert";                    break;
	case ERR_ST_TLS_STM_VERIFY:                     ret = "TLS_stm_verify";                  break;
	case ERR_ST_TLS_CHECK_VERIFY_STATUS:            ret = "check_verify_status";             break;
	case ERR_ST_TLS_PKCS12_CLIENT_NULL:             ret = "pkcs12 client = NULL";            break;
	case ERR_ST_TLS_PKCS12_SERVER_NULL:             ret = "pkcs12 server = NULL";            break;
	case ERR_ST_TLS_CIPHER_UNSELECT:                ret = "cipher list unselected";          break;
	case ERR_ST_TLS_UNKNOWN_CIPHER_TYPE:            ret = "unknown cipher type";             break;
	case ERR_ST_TLS_ENCRYPT_STREAM_FAILED:          ret = "enrypt: stream: failed";          break;
	case ERR_ST_TLS_ENCRYPT_BLOCK_FAILED:           ret = "enrypt: block: failed";           break;
	case ERR_ST_TLS_ENCRYPT_AEAD_FAILED:            ret = "enrypt: aead: failed";            break;
	case ERR_ST_TLS_DECRYPT_STREAM_FAILED:          ret = "decrypt: stream: failed";         break;
	case ERR_ST_TLS_DECRYPT_BLOCK_FAILED:           ret = "decrypt: block: failed";          break;
	case ERR_ST_TLS_DECRYPT_AEAD_FAILED:            ret = "decrypt: aead: failed";           break;
	case ERR_ST_TLS_UNSUPPORT_SSL_OPT_HS_SEPARATE:  ret = "unsupport: SSL_OPT_HS_SEPARATE";  break;
	case ERR_ST_TLS_UNSUPPORT_SSL_OPT_CERTREQOPT_F: ret = "unsupport: SSL_OPT_CERTREQOPT_F"; break;
	case ERR_ST_TLS_UNSUPPORT_SSL_OPT_KEEPWBUF:     ret = "unsupport: SSL_OPT_KEEPWBUF";     break;
	case ERR_ST_TLS_UNKNOWN_COMPRESSION_ALGO:       ret = "unknown: compress algo";          break;
	case ERR_ST_TLS_HS_SINGLE:                      ret = "other handshake progress";        break;
	case ERR_ST_TLS_HS_QUEUE_NULL:                  ret = "handshake already free-ed";       break;
	case ERR_ST_TLS_TLS_RECORD_READ:                ret = "tls_record_read";                 break;
	case ERR_ST_TLS_UNKNOWN_HS_TYPE:                ret = "unknown handshake type";          break;
	case ERR_ST_TLS_UNEXPECTED_MSG:                 ret = "unexpected hs msg";               break;
	case ERR_ST_TLS_ASN1_READ_CERT:                 ret = "asn1_read_cert failed";           break;
	case ERR_ST_TLS_AICRYPTO_SIG_TYPE:              ret = "failed: get aicrypto sig type";   break;
	case ERR_ST_TLS_BAD_CERT:                       ret = "bad cert";                        break;
	case ERR_ST_TLS_NOT_RSA_PUB:                    ret = "keymethod: not rsapub";           break;
	case ERR_ST_TLS_CERT_KEYENCIPHERMENT:           ret = "x509: keyEncipherment bit";       break;
	case ERR_ST_TLS_UNSUPPORTED_CERT:               ret = "unsupported cert";                break;
	case ERR_ST_TLS_CERT_FLAG_DIGITALSIGNATURE:     ret = "x509: digitalSignature bit";      break;
	case ERR_ST_TLS_P12_MAX_DEPTH:                  ret = "p12_max_depth";                   break;
	case ERR_ST_TLS_P12_FIND_BAG:                   ret = "p12_find_bag";                    break;
	case ERR_ST_TLS_P12_NEW:                        ret = "P12_new (null)";                  break;
	case ERR_ST_TLS_X509_VERSION:                   ret = "X509 version != 3";               break;
	case ERR_ST_TLS_P12_ADD_CERT:                   ret = "P12_add_cert";                    break;
	case ERR_ST_TLS_UNMATCH_SIGHASH:                ret = "no suitable sighash";             break;
	case ERR_ST_TLS_UNSUPPORTED_KEYALGO:            ret = "unsupport: key algo";             break;
	case ERR_ST_TLS_P12_GET_PRIVATEKEY:             ret = "P12_get_privatekey";              break;
	case ERR_ST_TLS_P12_SIGN_DIGEST:                ret = "P1_sign_digest";                  break;
	case ERR_ST_TLS_INVALID_SIGHASH:                ret = "invalid sighash algo";            break;
	case ERR_ST_TLS_OK_DO_VERIFY:                   ret = "verification failed";             break;
	case ERR_ST_TLS_GET_EPOCHTIME:                  ret = "failed to get epochtime";         break;
	case ERR_ST_TLS_GET_RANDOM:                     ret = "failed to get random";            break;
	case ERR_ST_TLS_GET_SESSION:                    ret = "failed to create session";        break;
	case ERR_ST_TLS_PROTOCOL_VERSION:               ret = "unsupport: protocol version";     break;
	case ERR_ST_TLS_CET_NO_PUBKEY:                  ret = "cert no pubkey";                  break;
	case ERR_ST_TLS_PKCS1_ENCRYPT:                  ret = "PKCS#1 encryption failed";        break;
	case ERR_ST_TLS_NOT_RSA_PRIV:                   ret = "keytype: not rsapriv";            break;
	case ERR_ST_TLS_PKCS1_DECRYPT:                  ret = "PKCS#1 decryption failed";        break;
	case ERR_ST_TLS_INVALID_PREMASTER_SECRET:       ret = "invalid premaster secret";        break;
	case ERR_ST_TLS_INVALID_DIGEST:                 ret = "invalid handshake digest";        break;
	case ERR_ST_TLS_READ_BLOCK:                     ret = "invalid block (padding or mac)";  break;
	case ERR_ST_TLS_INVALID_MAC:                    ret = "invalid mac";                     break;
	case ERR_ST_TLS_INVALID_BLOCK_LENGTH:           ret = "invalid block length";            break;
	case ERR_ST_TLS_AESKEY_NEW:                     ret = "AESKey_new";                      break;
	case ERR_ST_TLS_DES3KEY_NEW:                    ret = "DES3key_new_c";                   break;
	case ERR_ST_TLS_RC4KEY_NEW:                     ret = "RC4Key_new";                      break;
	case ERR_ST_TLS_INVALID_MAC_LENGTH:             ret = "invalid mac length";              break;
	case ERR_ST_TLS_INVALID_OPT_KEY:                ret = "invalid opt key";                 break;
	case ERR_ST_TLS_SESSION_NULL:                   ret = "session (null)";                  break;
	case ERR_ST_TLS_SESSION_USED:                   ret = "session used by other handshake"; break;
	case ERR_ST_TLS_SESSION_ID_LENGTH:              ret = "invalid session id length";       break;
	case ERR_ST_TLS_RANDOM_LENGTH:                  ret = "invalid random length";           break;
	case ERR_ST_TLS_RAND_BYTES:                     ret = "RAND_bytes failed";               break;
	case ERR_ST_TLS_RSAPUB_DOCRYPT:                 ret = "RSApub_doCrypt";                  break;
	case ERR_ST_TLS_PKCS1_PADDING:                  ret = "invalid pkcs#1 padding";          break;
	case ERR_ST_TLS_PKCS1_SIZE:                     ret = "invalid pkcs#1 content size";     break;
	case ERR_ST_TLS_STM_FIND_BYNAME:                ret = "STM_find_byName";                 break;
	case ERR_ST_TLS_STM_CS_GET_KEYHASH:             ret = "cs_get_keyhash";                  break;
	case ERR_ST_TLS_STM_CSTORE_FIND_BYKEYHASH:      ret = "CStore_find_byKeyHash";           break;
	case ERR_ST_TLS_STM_CSTORE_GET_KEY:             ret = "CStore_get_key";                  break;
	case ERR_ST_TLS_STM_GET_PATHCERT:               ret = "STM_get_pathcert";                break;
	case ERR_ST_TLS_STM_FIND_ID:                    ret = "STM_find_byID";                   break;
	case ERR_ST_TLS_STM_VERIFY_CERT:                ret = "STM_verify_cert";                 break;
	case ERR_ST_TLS_STM_OPEN:                       ret = "STM_open";                        break;
	case ERR_ST_TLS_INVALID_FRAGMENT_LENGTH:        ret = "invalid fragment length";         break;
	case ERR_ST_TLS_INVALID_RECORD_TYPE:            ret = "invalid record type";             break;
	case ERR_ST_TLS_NO_RENEGOTIATION:               ret = "unsupport: renegotiation";        break;
	case ERR_ST_TLS_NOT_ECDH_PUB:                   ret = "keytype: not ecdhpub";            break;
	case ERR_ST_TLS_UNSUPPORTED_EXTENSION:          ret = "unsupport: hello extension";      break;
	case ERR_ST_TLS_ECP_NEW:                        ret = "ECp_new";                         break;
	case ERR_ST_TLS_ECP_MULTI:                      ret = "ECp_multi";                       break;
	case ERR_ST_TLS_FE2OSP:                         ret = "FE2OSP";                          break;
	case ERR_ST_TLS_ECPM_GET_STD_PARAMETER:         ret = "ECPm_get_std_parameter";          break;
	case ERR_ST_TLS_ECP_P2OS:                       ret = "ECp_P2OS";                        break;
	case ERR_ST_TLS_ECDSAPUBKEY_NEW:                ret = "ECDSApubkey_new";                 break;
	case ERR_ST_TLS_ECP_OS2EP:                      ret = "ECp_OS2P";                        break;
	case ERR_ST_TLS_NOT_ECDH_PRIV:                  ret = "keytype: not ecdhpriv";           break;
	case ERR_ST_TLS_UNSUPPORTED_CURVE:              ret = "NamedCurve: unsupported";         break;
	case ERR_ST_TLS_UNSUPPORTED_CURVE_TYPE:         ret = "unsupport: curve type";           break;
	case ERR_ST_TLS_ILLEGAL_PARAMETER:              ret = "illegal parameter";               break;
	case ERR_ST_TLS_HANDSHAKE_FAILURE:              ret = "handshake failure";               break;
	case ERR_ST_TLS_UNRECOGNIZED_NAME:              ret = "unrecognized name";               break;
	case ERR_ST_TLS_MISSING_EXTENSION:              ret = "missing extension";               break;
	case ERR_ST_TLS_INTERNAL_ERROR:                 ret = "internal error";                  break;
	case ERR_ST_TLS_PROTOCOL_VERSION_DOWNGRADE:     ret = "protocol version downgrade";      break;
	case ERR_ST_TLS_CIPHER_LIST_FULL:               ret = "cipher list full";                break;
	case ERR_ST_TLS_EMPTY_CERT:                     ret = "x509: certificate empty";         break;
	case ERR_ST_TLS_UNSUPPORTED_KEY_EXCHANGE:       ret = "unsupport: key exchange";         break;
	case ERR_ST_TLS_UNSUPPORTED_SIGSCHEME:          ret = "unsupport: signature scheme";     break;
	case ERR_ST_TLS_UNSUPPORTED_HASH:               ret = "unsupport: hash";                 break;
	case ERR_ST_TLS_UNSUPPORTED_SIGNATURE:          ret = "unsupport: signature";            break;
	case ERR_ST_TLS_FORBIDDEN_HS_TYPE:              ret = "forbidden handshake type";        break;
	case ERR_ST_TLS_FORBIDDEN_HASH:                 ret = "forbidden hash";                  break;
	case ERR_ST_TLS_FORBIDDEN_SIGSCHEME:            ret = "forbiddedn signature scheme";     break;
	case ERR_ST_TLS_SAME_TYPE_EXTENSION:            ret = "same type extension";             break;
	case ERR_ST_TLS_CHACHAKEY_NEW:                  ret = "ChaChakey_new";                   break;
	case ERR_ST_TLS_X25519_SHARED_SECRET:           ret = "X25519 shared secret";            break;
	case ERR_ST_TLS_X448_SHARED_SECRET:             ret = "X448 shared secret";              break;
	case ERR_ST_TLS_INVALID_NIST_PUB:               ret = "invalid nist curve public";       break;
	case ERR_ST_TLS_INVALID_HKDF_LABEL_LENGTH:      ret = "invalid hkdf label length";       break;
	case ERR_ST_TLS_INVALID_HKDF_CONTEXT_LENGTH:    ret = "invalid hkdf context length";     break;
	case ERR_ST_TLS_RECORD_OVERFLOW:                ret = "record size overflow";            break;
	case ERR_ST_TLS_CONNECTION_CLOSED:              ret = "connection closed";               break;
	case ERR_ST_TLS_GETADDRINFO:                    ret = "getaddrinfo";                     break;
	case ERR_ST_TLS_SERVER_NAME_NUMHOST:            ret = "sni: numeric hostname";           break;
	case ERR_ST_TLS_SERVER_NAME_EMPTY:              ret = "sni: empty hostname";             break;
	case ERR_ST_TLS_SERVER_NAME_INVALID:            ret = "sni: invalid hostname";           break;
	case ERR_ST_TLS_SERVER_NAME_TOOLONG:            ret = "sni: hostname too long";          break;

	default: ret=tmp; sprintf(tmp,"%d",err); break;
	}
	return ret;
}

char *OK_get_errstr(){
	static char buf[128];

	snprintf(buf,126,"[aicrypto] error:0x%x:%s:%s",
		aicrypto_error,
		get_err_location(aicrypto_error>>16),
		get_err_type(aicrypto_error&0xffff));
	return buf;
}

/*---------------------------------------
  print error string
---------------------------------------*/
void OK_print_error(){
	if(okerr) fprintf(okerr,"%s\n",OK_get_errstr());
}
