/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdlib.h>
#include <check.h>

#include "tls_util.c"

static int ut_error;
static int ut_location;
static int ut_point;

static void ut_error_setup()
{
	ut_error = 0;
	ut_location = 0;
	ut_point = 0;
}

/*
 * fixture
 */
void setup(void)
{
	ut_error_setup();
}
void teardown(void)
{
	/* code */
}

/*
 * unit testing for tls_util_write_2()
 */
START_TEST (test_tls_util_write_2_normal)
{
	uint8_t buf[4] = { 0x00, 0x00, 0x00, 0x00 };
	uint8_t expect[4] = { 0xab, 0xcd, 0x00, 0x00 };
	int32_t val = 0xabcd;

	tls_util_write_2(buf, val);
	ck_assert_mem_eq(buf, expect, 4);
}
END_TEST

Suite *tls_util_write_2_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_write_2()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_write_2_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for tls_util_write_3()
 */
START_TEST (test_tls_util_write_3_normal)
{
	uint8_t buf[4] = { 0x00, 0x00, 0x00, 0x00 };
	uint8_t expect[4] = { 0xab, 0xcd, 0xef, 0x00 };
	int32_t val = 0xabcdef;

	tls_util_write_3(buf, val);
	ck_assert_mem_eq(buf, expect, 4);
}
END_TEST

Suite *tls_util_write_3_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_write_3()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_write_3_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for tls_util_read_2()
 */
START_TEST (test_tls_util_read_2_normal)
{
	uint8_t buf[4] = { 0xab, 0xcd, 0x00, 0x00 };
	uint16_t val;
	uint16_t expect = 0xabcd;

	val = tls_util_read_2(buf);
	ck_assert_int_eq(val, expect);
}
END_TEST

Suite *tls_util_read_2_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_read_2()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_read_2_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for tls_util_read_3()
 */
START_TEST (test_tls_util_read_3_normal)
{
	uint8_t buf[4] = { 0xab, 0xcd, 0xef, 0x00 };
	uint32_t val;
	uint32_t expect = 0xabcdef;

	val = tls_util_read_3(buf);
	ck_assert_int_eq(val, expect);
}
END_TEST

Suite *tls_util_read_3_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_read_3()");

	/* tls_util_read test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_read_3_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for tls_util_convert_ver_to_protover()
 */
START_TEST (test_tls_util_convert_ver_to_protover_normal)
{
	uint16_t tls13 = TLS_VER_TLS13;
	uint16_t tls12 = TLS_VER_TLS12;
	struct tls_protocol_version tls13_st;
	struct tls_protocol_version tls12_st;

	tls_util_convert_ver_to_protover(tls13, &tls13_st);
	ck_assert_int_eq(tls13_st.major, TLS_MAJOR_TLS);
	ck_assert_int_eq(tls13_st.minor, TLS_MINOR_TLS13);

	tls_util_convert_ver_to_protover(tls12, &tls12_st);
	ck_assert_int_eq(tls12_st.major, TLS_MAJOR_TLS);
	ck_assert_int_eq(tls12_st.minor, TLS_MINOR_TLS12);
}
END_TEST

Suite *tls_util_convert_ver_to_protover_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_convert_ver_to_protover()");

	/* tls_util_convert_ver_to_protover test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_convert_ver_to_protover_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for tls_util_convert_protover_to_ver()
 */
START_TEST (test_tls_util_convert_protover_to_ver_normal)
{
	uint16_t tls13;
	uint16_t tls12;
	struct tls_protocol_version tls13_st;
	struct tls_protocol_version tls12_st;
	tls13_st.major = TLS_MAJOR_TLS;
	tls13_st.minor = TLS_MINOR_TLS13;
	tls12_st.major = TLS_MAJOR_TLS;
	tls12_st.minor = TLS_MINOR_TLS12;

	tls13 = tls_util_convert_protover_to_ver(&tls13_st);
	ck_assert_uint_eq(tls13, TLS_VER_TLS13);

	tls12 = tls_util_convert_protover_to_ver(&tls12_st);
	ck_assert_uint_eq(tls12, TLS_VER_TLS12);
}
END_TEST

Suite *tls_util_convert_protover_to_ver_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_util_convert_protover_to_ver()");

	/* tls_util_convert_ver_to_protover test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_tls_util_convert_protover_to_ver_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/**
 * test name (src file) suite (dummy).
 */
Suite *tls_util_suite(void)
{
	Suite *s;

	return s = suite_create("tls_util_suites");
}

int main(void)
{
	int number_failed;
	SRunner *sr;

	sr = srunner_create(tls_util_suite());
	srunner_add_suite(sr, tls_util_write_2_suite());
	srunner_add_suite(sr, tls_util_write_3_suite());
	srunner_add_suite(sr, tls_util_read_2_suite());
	srunner_add_suite(sr, tls_util_read_3_suite());
	srunner_add_suite(sr, tls_util_convert_ver_to_protover_suite());
	srunner_add_suite(sr, tls_util_convert_protover_to_ver_suite());

	srunner_run_all(sr, CK_NORMAL);
	number_failed = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

/* ------------------------------------------------------------------------- */
#pragma GCC diagnostic ignored "-Wunused-parameter"
/* puts stub functions below this line. */

void OK_set_error(int error, int location, int point, CK_RV *info)
{
	printf("%d, %s\n", __LINE__, __func__);
	ut_error = error;
	ut_location = location;
	ut_point = point;

	fprintf(stderr, "error:%0x location:%0x point:%0x\n",
		error, location, point);
	if (info != NULL) {
		fprintf(stderr, "*info=%zu\n", (size_t)(*info));
	} else {
		fprintf(stderr, "info=NULL\n");
	}
}

int RAND_bytes(unsigned char *buf,int num)
{
	return 0;
}

int RSApub_doCrypt(int len, unsigned char *from,
			  unsigned char *to, Pubkey_RSA *key)
{
	return 0;
}

unsigned char *OK_do_sign(Key *key,unsigned char *data,int data_len,unsigned char *ret)
{
	return NULL;
}

int ASN1_length(unsigned char *in,int *mv)
{
	return 0;
}
