/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdlib.h>
#include <check.h>

#include "tls_cipher.c"

static int ut_error;
static int ut_location;
static int ut_point;

static void ut_error_setup()
{
	ut_error = 0;
	ut_location = 0;
	ut_point = 0;
}

static enum tls_alert_desc ut_alert_desc;

static void ut_alert_setup()
{
	ut_alert_desc = 0;
}

/*
 * fixture
 */
void setup(void)
{
	ut_error_setup();
	ut_alert_setup();
}
void teardown(void)
{
	/* code */
}

/*
 * unit testing for merge_cipher_suite_list()
 */
START_TEST (test_merge_cipher_suite_list_normal)
{
	uint16_t buf_len = TLS_CIPHER_SUITES_SIZE_MAX / sizeof(uint16_t);
	enum tls_cipher_suite buf[buf_len];
	uint16_t clist_len = 0;
	uint16_t rc;
	uint16_t clist_len_tls13 = sizeof(tls_cipher_list_tls13)
		/ sizeof(enum tls_cipher_suite);
	uint16_t clist_len_tls12 = sizeof(tls_cipher_list_tls12)
		/ sizeof(enum tls_cipher_suite);
	uint16_t clist_len_tls11 = sizeof(tls_cipher_list_tls11)
		/ sizeof(enum tls_cipher_suite);

	rc = merge_cipher_suite_list(buf,
				     buf_len,
				     clist_len,
				     tls_cipher_list_tls13,
				     clist_len_tls13);

	ck_assert_uint_eq(rc, clist_len_tls13);
	for (uint16_t i = 0; i < rc; i++) {
		ck_assert_uint_eq(buf[i], tls_cipher_list_tls13[i]);
	}

	clist_len = rc;

	rc = merge_cipher_suite_list(buf,
				     buf_len,
				     clist_len,
				     tls_cipher_list_tls12,
				     clist_len_tls12);

	ck_assert_uint_eq(rc, clist_len_tls13 + clist_len_tls12);
	for (uint16_t i = 0; i < clist_len_tls13; i++) {
		ck_assert_uint_eq(buf[i], tls_cipher_list_tls13[i]);
	}

	for (uint16_t i = 0; i < clist_len_tls12; i++) {
		ck_assert_uint_eq(buf[i + clist_len_tls13],
			  tls_cipher_list_tls12[i]);
	}

	clist_len = rc;

	rc = merge_cipher_suite_list(buf,
				     buf_len,
				     clist_len,
				     tls_cipher_list_tls11,
				     clist_len_tls11);

	ck_assert_uint_eq(rc, clist_len_tls13 + clist_len_tls12 + 1);
	for (uint16_t i = 0; i < clist_len_tls13; i++) {
		ck_assert_uint_eq(buf[i], tls_cipher_list_tls13[i]);
	}

	for (uint16_t i = 0; i < clist_len_tls12; i++) {
		ck_assert_uint_eq(buf[i + clist_len_tls13],
			  tls_cipher_list_tls12[i]);
	}

	ck_assert_uint_eq(buf[clist_len_tls13 + clist_len_tls12],
			  TLS_RSA_WITH_DES_CBC_SHA);
}
END_TEST

Suite *merge_cipher_suite_list_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("merge_cipher_suite_list()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup, teardown);
	tcase_add_test(tc_core, test_merge_cipher_suite_list_normal);
	suite_add_tcase(s, tc_core);
	return s;
}

/**
 * test name (src file) suite (dummy).
 */
Suite *tls_cipher_suite(void)
{
	Suite *s;

	return s = suite_create("tls_cipher_suites");
}

int main(void)
{
	int number_failed;
	SRunner *sr;

	sr = srunner_create(tls_cipher_suite());
	srunner_add_suite(sr, merge_cipher_suite_list_suite());

	srunner_run_all(sr, CK_NORMAL);
	number_failed = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

/* ------------------------------------------------------------------------- */
#pragma GCC diagnostic ignored "-Wunused-parameter"
/* puts stub functions below this line. */

void OK_set_error(int error, int location, int point, CK_RV *info)
{
	printf("%d, %s\n", __LINE__, __func__);
	ut_error = error;
	ut_location = location;
	ut_point = point;

	fprintf(stderr, "error:%0x location:%0x point:%0x\n",
		error, location, point);
	if (info != NULL) {
		fprintf(stderr, "*info=%zu\n", (size_t)(*info));
	} else {
		fprintf(stderr, "info=NULL\n");
	}
}

/* -----------------tls_cert.c---------------------------------------------- */

bool tls_cert_info_get(PKCS12 *p12, struct tls_cert_info *info)
{
	return true;
}

bool tls_cert_info_can_use_ecc_cipher_suite(struct tls_cert_info *cinfo,
					    struct tls_hs_ecc_eclist *eclist,
					    struct tls_hs_ecc_pflist *pflist)
{
	return true;
}

bool tls_cert_info_available(enum tls_cipher_suite suite,
			     struct tls_cert_info *info,  bool can_use_ecc,
			     bool set_error)
{
	return true;
}

/* -----------------tls_alert.c--------------------------------------------- */

bool tls_alert_send(TLS *tls,
		    const enum tls_alert_level level,
		    const enum tls_alert_desc desc)
{
	ut_alert_desc = desc;
	return true;
}

/* -----------------tls_util.c---------------------------------------------- */

uint16_t tls_util_convert_protover_to_ver(struct tls_protocol_version *version_st)
{
	return (version_st->major << 8) + ((version_st->minor) & 0x00ff);
}

/* -----------------cipher/aead.c------------------------------------------- */

int32_t tls_cipher_aead(TLS *tls,
			uint8_t *dest,
			const enum tls_record_ctype type,
			const uint8_t *src,
			const int32_t len)
{
	return 0;
}

int32_t tls_decipher_aead(TLS *tls,
			  uint8_t *dest,
			  const enum tls_record_ctype type,
			  const uint8_t *src,
			  const int32_t len)
{
	return 0;
}

/* -----------------cipher/block.c------------------------------------------ */
int32_t tls_cipher_block(TLS *tls,
			 uint8_t *dest,
			 const enum tls_record_ctype type,
			 const uint8_t *src,
			 const int32_t len)
{
	return 0;
}

int32_t tls_decipher_block(TLS *tls,
			   uint8_t *dest,
			   const enum tls_record_ctype type,
			   const uint8_t *src,
			   const int32_t len)
{
	return 0;
}

/* -----------------cipher/stream.c----------------------------------------- */
int32_t tls_cipher_stream(TLS *tls,
			  uint8_t *dest,
			  const enum tls_record_ctype type,
			  const uint8_t *src,
			  const int32_t len)
{
	return 0;
}

int32_t tls_decipher_stream(TLS *tls,
			    uint8_t *dest,
			    const enum tls_record_ctype type,
			    const uint8_t *src,
			    const int32_t len)
{
	return 0;
}

/* -----------------handshake/ecdh/ecdh.c----------------------------------- */

void tls_hs_ecdh_set_curve(struct tls_hs_ecdh *ecdh)
{
	;
}

void tls_hs_ecdh_set_curve_by_cert(struct tls_hs_ecdh *ecdh,
				   struct tls_cert_info *cinfo)
{
	;
}
