/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls.h"
#include "tls_util.h"

#include <string.h>
#include <time.h>
#ifdef __MACH__
#include <mach/clock.h>
#include <mach/mach.h>
#endif

/* for RAND_bytes */
#include <aicrypto/ok_rand.h>

/* for Key */
#include <aicrypto/ok_x509.h>

/* for RSApub_doCrypt */
#include <aicrypto/ok_rsa.h>

/* for OK_do_sign */
#include <aicrypto/ok_tool.h>

/* for ASN1_length */
#include <aicrypto/ok_asn1.h>

uint64_t tls_util_get_epochtime(void) {
	struct timespec tms;

	/* do not use gettimeofday(2) but clock_gettime(2) since Linux
	 * gettimeofday(2) man page says "POSIX.1-2008 marks
	 * gettimeofday() as obsolete, recommending the use of
	 * clock_gettime(2) instead.". */
#ifdef __MACH__ /* OS X does not have clock_gettime(), use clock_get_time() */
	clock_serv_t cclock;
	mach_timespec_t m_tms;

	host_get_clock_service(mach_host_self(), REALTIME_CLOCK, &cclock);
	clock_get_time(cclock, &m_tms); /* XXX: how about an error handler? */
	mach_port_deallocate(mach_task_self(), cclock);
	tms.tv_sec = m_tms.tv_sec;
	tms.tv_nsec = m_tms.tv_sec;
#else
	if (clock_gettime(CLOCK_REALTIME, &tms)) {
		TLS_DPRINTF("clock_realtime");
		return 0;
	}
#endif

	return tms.tv_sec;
}

bool tls_util_get_random(uint8_t *buf, uint32_t len) {
	if (len == 0) {
		TLS_DPRINTF("len");
		OK_set_error(ERR_ST_TLS_RANDOM_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 0, NULL);
		return true;
	}

	/* do not call RAND_init function since SSL_set_rand function
	 * (aicrypto/ssl/ssl_rand.c) also do not call RAND_init. And,
	 * it seems RAND_bytes calls RAND_init internally. */
	if (RAND_bytes(buf, len) < 0) {
		TLS_DPRINTF("RAND_bytes");
		OK_set_error(ERR_ST_TLS_RAND_BYTES,
			     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 1, NULL);
		return false;
	}
	/* RAND_cleanup is called by TLS_cleanup. */
	return true;
}

void tls_util_write_2(uint8_t *buf, int32_t val) {
	buf[0] = ((val) >> 8) & 0xff;
	buf[1] = ((val)     ) & 0xff;
}

void tls_util_write_3(uint8_t *buf, int32_t val) {
	buf[0] = ((val) >> 16) & 0xff;
	buf[1] = ((val) >>  8) & 0xff;
	buf[2] = ((val)      ) & 0xff;
}

uint16_t tls_util_read_2(uint8_t *buf) {
	return (((buf[0] << 8) & 0xff00) |
		((buf[1]     ) & 0x00ff));
}

uint32_t tls_util_read_3(uint8_t *buf) {
	return (((buf[0] << 16) & 0xff0000) |
		((buf[1] <<  8) & 0x00ff00) |
		((buf[2]      ) & 0x0000ff));
}

void tls_util_convert_ver_to_protover(uint16_t version,
				      struct tls_protocol_version *version_st)
{
	version_st->major = (version >> 8) & 0xff;
	version_st->minor = (version     ) & 0xff;
}

uint16_t tls_util_convert_protover_to_ver(struct tls_protocol_version *version_st)
{
	return (version_st->major << 8) + ((version_st->minor) & 0x00ff);
}

bool tls_util_check_version_in_supported_version(
	struct tls_protocol_version_list *vlist,
	uint16_t version)
{
	for (int i = 0; i < vlist->len; i++) {
		if (vlist->list[i] == version) {
			return true;
		}
	}

	return false;
}

int32_t tls_util_pkcs1_encrypt(Key *pubkey,
			       uint8_t *dest, uint8_t *src, int32_t src_len) {
	/* TODO: this code should be implemented in other
	 * module (e.g. aicrypto/pkcs).
	 *
	 * do PKCS1 encryption.
	 *
	 *   M    = message.
	 *   mLen = length of M.
	 *
	 *   a. Generate an octet string PS of length k - mLen - 3
	 *      consisting of pseudo-randomly generated nonzero octets.
	 *
	 *   b. Concatenate PS, the message M, and other padding to form
	 *      an encoded message EM of length k octets as
	 *
	 *      EM = 0x00 || 0x02 || PS || 0x00 || M.
	 *
	 * for more information, see
	 * aicrypto/ssl/ssl_hsclnt.c@SSL_get_clikeyexchange and RFC
	 * 3447.
	 */

	int32_t size = pubkey->size;
	uint8_t buf[size];

	buf[0]   = 0;
	buf[1]   = 2;

	/* PS */
	int len = size - (src_len + 1);
	{
		if (! tls_util_get_random(&(buf[2]), size - 2)) {
			OK_set_error(ERR_ST_TLS_GET_RANDOM,
				     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 2, NULL);
			return -1;
		}

		for (int i = 2; i < len; ++i) {
			buf[i] |= 0x10;
		}
	}

	buf[len] = 0;

	memcpy(&(buf[len + 1]), &(src[0]), src_len);

	/* RFC 5246 describes public-key-encrypted is variant length
	 * value. but RSApub_doCrypt do not returns length.  ssl
	 * implementation of aicrypto uses fixed length (pubkey->size).
	 * So, in this code, use fixed length once. */
	if (RSApub_doCrypt(size,
			   &(buf[0]), &(dest[0]), (Pubkey_RSA *)pubkey) < 0) {
		TLS_DPRINTF("RSApub_doCrypt");
		OK_set_error(ERR_ST_TLS_RSAPUB_DOCRYPT,
			     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 3, NULL);
		return -1;
	}

	return size;
}

int32_t tls_util_pkcs1_decrypt(Key *privkey, uint8_t *dest,
			       const uint8_t *src, const int32_t src_len) {
	uint8_t buff[privkey->size];

	if (OK_do_sign(privkey, (uint8_t *)&(src[0]), src_len,
		       &(buff[0])) == NULL) {
		return -1;
	}

	/* check PKCS#1 padding */
	if ((buff[0] != 0) || (buff[1] != 2)) {
		OK_set_error(ERR_ST_TLS_PKCS1_PADDING,
			     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 4, NULL);
		return -1;
	}

	/* skip padding */
	int32_t offset = 2;
	for (; offset < privkey->size; ++offset) {
		if (buff[offset] == 0) {
			/* skip this (buff[offset] == 0) element. */
			offset += 1;
			break;
		}
	}

	if (offset == privkey->size) {
		OK_set_error(ERR_ST_TLS_PKCS1_SIZE,
			     ERR_LC_TLS4, ERR_PT_TLS_UTIL + 5, NULL);
		return -1;
	}

	int32_t len = privkey->size - offset;

	memcpy(&(dest[0]), &(buff[offset]), len);

	return len;
}

int32_t tls_util_asn1_length(uint8_t *in) {
	/* wrap aicrypto's ASN1_length function to hide 2nd argument of
	 * ASN1_length function. */
	int mv = 0;
	int32_t len = ASN1_length(in, &mv);

	/* NOTE: why len + mv + 1 ? */
	return len + mv + 1;
}
