/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_record.h"
#include "tls_alert.h"
#include "tls_cipher.h"
#include "tls_compress.h"
#include "tls_ccs.h"
#include "tls_handshake.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/select.h>

/**
 * direction of record (write or read).
 *
 * this enum is used to determine the sequence number that is
 * incremented (sequence number for writing or reading).
 */
enum tls_record_io_type {
	/** reading direction */
	TLS_RECORD_IO_READ,

	/** writing direction */
	TLS_RECORD_IO_WRITE
};

/**
 * output the name of record type as the debug purpose information.
 *
 * this is done when TLS_DEBUG macro is set in the building
 * phase. otherwise, this function do nothing.
 */
static void dump_record_type(const enum tls_record_ctype type);

/**
 * get timeout second for read(2)/write(2) from option of tls structure.
 */
static struct timespec get_timeout(TLS *tls);

/**
 * increment sequence number of tls record.
 */
static bool incr_seqnum(TLS *tls, const enum tls_record_io_type type);

/**
 * check the record layer version of received record data.
 */
static bool check_record_version(TLS* tls,
				 const struct tls_protocol_version ver);

/**
 * check if handshake stays in a state in which tls record is plain text.
 */
static bool check_unencrypted_state(TLS *tls);

/**
 * wrapper of read(2).
 *
 * this function is wrapper of read(2). if the once argument is false,
 * this function waits until all data that is specified in the size
 * argument is read.
 */
static int32_t read_record(TLS *tls, uint8_t *buf, const uint32_t size,
			   const bool once);

/**
 * wrapper of write(2).
 *
 * this function is wrapper of write(2). if the once argument is false,
 * this function waits until all data that is specified in the size
 * argument is written.
 */
static int32_t write_record(TLS *tls, const uint8_t *buf, const uint32_t size,
			    const bool once);

/**
 * get record type from read record data.
 */
static enum tls_record_ctype get_record_type(const uint8_t *buf);

/**
 * get TLS/SSL version of record layer from read record data.
 */
static struct tls_protocol_version get_tls_version(const uint8_t *buf);

/**
 * get one record layer data.
 *
 * return value is data that is allocated memory. if that data become
 * unnecessary, you must free the data data by free_record function.
 */
static struct tls_record *get_record_up_to_tls12(TLS *tls);

static struct tls_record *get_record_tls13_encrypted(TLS *tls);

static struct tls_record * get_record(TLS *tls);

/**
 * free the memory that is allocated by get_record function.
 */
static void free_record(struct tls_record *record);

/**
 * remove speicified entry (record) from queue.
 *
 * when remove entry of queue, memory that was allocated to that entry
 * is freed (entry is record. and, call free_record function
 * internally).
 */
static void remove_queue(struct tls_record_queue *queue,
			 struct tls_record *record);

/**
 * read record layer data that was stored in queue.
 *
 * this function do not fail. if do not be read anything from queue,
 * return zero.
 */
static ssize_t read_queue(struct tls_record_queue *queue,
			  uint8_t *buf, size_t count);

/**
 * do the process when the handshake data was read in the return value
 * of get_record function.
 *
 * when handshake data was read in the case which the user intended to
 * read the application data, this function will fail because
 * re-negotiation does not be supported. in TLS 1.3, as re-negotiation
 * is forbidden, terminate a connection with an unexpected_message alert.
 * otherwise, enqueue the read record data.
 */
static bool read_fragments_handshake(const enum tls_record_ctype upper_type,
				     TLS *tls, struct tls_record *record);

/**
 * do the process when the application data was read in the return value
 * of get_record function.
 *
 * if first handshake was over, enqueue the read record data. otherwise,
 * this function will fail because application received the unexpected
 * message.
 */
static bool read_fragments_app(TLS *tls, struct tls_record *record);

/**
 * read record layer data from network and queue.
 *
 * this function do following process.
 *
 * 1. read from queue.
 * 2. if queue is not empty and queue is enough, go to finish.
 * 3. read from network
 * 4. handle read record.
 *    - ccs and alert immediately is processed.
 *    - handshake see read_fragments_handshake function.
 *    - application see read_fragments_app funciton.
 * 5. read from queue.
 * 6. repeat 2-5 until enough data is read.
 */
static ssize_t read_fragments(const enum tls_record_ctype upper_type,
			      TLS *tls, uint8_t *buf, const size_t count);

/**
 * initialize record version that is used in the record layer.
 */
static void write_record_version(const TLS* tls, uint8_t *buf);

/**
 * compose tls record with encrypting plain text.
 */
static int32_t compose_record_up_to_tls12(const enum tls_record_ctype type,
					  TLS *tls,
					  uint8_t *buf,
					  const uint8_t *pbuf,
					  const size_t plen);

static int32_t compose_record_tls13(const enum tls_record_ctype type,
				    TLS *tls,
				    uint8_t *buf,
				    const uint8_t *pbuf,
				    const size_t plen,
				    const size_t padlen);

static int32_t compose_record(const enum tls_record_ctype type,
			      TLS *tls,
			      uint8_t *buf,
			      const uint8_t *pbuf,
			      const size_t plen,
			      const size_t padlen UNUSED);

/**
 * the length of record layer header.
 *
 * record type (1 byte) + protocol version (2 byte) + fragment length (2
 * byte)
 */
static const uint32_t TLS_RECORD_HEADER_LENGTH = 5;

/* if TLS_DEBUG is off, type is not used. so, add attribute UNUSED. */
static void dump_record_type(const enum tls_record_ctype type UNUSED) {
#ifdef TLS_DEBUG
	char name[30];
	switch (type) {
	case TLS_CTYPE_CHANGE_CIPHER_SPEC:
		strcpy(name, "CCS");
		break;

	case TLS_CTYPE_ALERT:
		strcpy(name, "ALERT");
		break;

	case TLS_CTYPE_HANDSHAKE:
		strcpy(name, "HANDSHAKE");
		break;

	case TLS_CTYPE_APPLICATION_DATA:
		strcpy(name, "APPLICATION_DATA");
		break;

	default:
		strcpy(name, "unknown record type");
		break;
	}

	TLS_DPRINTF("record: type = %s", name);
#endif
}

static struct timespec get_timeout(TLS *tls) {
	struct timespec timeout;

	timeout.tv_sec  = TLS_opt_get(tls, TLS_OPT_TIMEOUT);
	timeout.tv_nsec = 0;

	return timeout;
}

static bool incr_seqnum(TLS *tls, const enum tls_record_io_type type) {
	uint64_t num;

	switch(type) {
	case TLS_RECORD_IO_READ:
		TLS_DPRINTF("record: type: read");
		tls->active_read.seqnum += 1;
		num = tls->active_read.seqnum;
		break;

	case TLS_RECORD_IO_WRITE:
	default:
		TLS_DPRINTF("record: type: write");
		tls->active_write.seqnum += 1;
		num = tls->active_write.seqnum;
		break;
	}

	TLS_DPRINTF("record: seqnum: %"PRIu64"", num);

	/* TODO: re-negotiation if necessary (seqnum == 2^64 -1). */
	if (num == (0xFFFFFFFFFFFFFFFF - 1)) {
		/* not implemented yet. if support secure
		 * re-negotiation, implement this code. */
		;
	}

	return true;
}

static bool check_record_version(TLS* tls,
				 const struct tls_protocol_version ver) {
	/* first read of handshake data do not check version of record
	 * (do not negotiate yet). */
	if (tls->state == TLS_STATE_HS_BEFORE_RECV_CHELLO) {
		return true;
	}

	if (tls->state == TLS_STATE_HS_BEFORE_RECV_SHELLO) {
		/* handle a record received from AiSSL as protocol version
		 * mismatch at here until NRGTLS support SSL 3.0. */
		if ((ver.major == TLS_MAJOR_TLS) &&
		    (ver.minor == TLS_MINOR_SSL30)) {
			tls->errnum = TLS_ERR_PROTOCOL_VERSION;
		}

		return true;
	}

	if ((ver.major != tls->record_version.major) ||
	    (ver.minor != tls->record_version.minor)) {
		return false;
	}

	return true;
}

static bool check_unencrypted_state(TLS *tls) {
	switch (tls->state) {
	/* client side state */
	case TLS_STATE_HS_AFTER_SEND_CHELLO:
	case TLS_STATE_HS_BEFORE_SEND_2NDCHELLO:
	case TLS_STATE_HS_AFTER_SEND_2NDCHELLO:
	case TLS_STATE_HS_BEFORE_RECV_SHELLO:
	case TLS_STATE_HS_BEFORE_RECV_2NDSHELLO:
	case TLS_STATE_HS_AFTER_RECV_HRREQ:

	/* server side state */
	case TLS_STATE_HS_BEFORE_RECV_CHELLO:
	case TLS_STATE_HS_AFTER_RECV_CHELLO:
	case TLS_STATE_HS_BEFORE_RECV_2NDCHELLO:
	case TLS_STATE_HS_AFTER_RECV_2NDCHELLO:
	case TLS_STATE_HS_BEFORE_SEND_HRREQ:
		return true;

	default:
		return false;
	}
}

static int32_t read_record(TLS *tls, uint8_t *buf, const uint32_t size,
			   const bool once) {
	int32_t sum = 0;

	fd_set readfds;

	FD_ZERO(&readfds);
	FD_SET(tls->sockfd, &readfds);

	int maxfd = tls->sockfd;
	do {
		fd_set fds;
		struct timespec timeout = get_timeout(tls);

		memcpy(&fds, &readfds, sizeof(fd_set));

		/* use pselect instead of select since I want not to use
		 * timeval but timespec (timespec newer than
		 * timeval). */
		int ready;
		if ((ready = pselect(maxfd + 1,
				     &fds, NULL, NULL, &timeout, NULL)) < 0) {
			/* TODO: check ENOMEM and EINTER. */
			TLS_DPRINTF("pselect: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_PSELECT,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 0, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		if (ready == 0) {
			/* consider timeout same as error. */
			TLS_DPRINTF("pselect: timeout");
			OK_set_error(ERR_ST_TLS_PSELECT_TIMEOUT,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 1, NULL);
			return -1;
		}

		if (FD_ISSET(tls->sockfd, &fds)) {
			int res;

			if ((res = read(tls->sockfd,
					&(buf[sum]), size - sum)) < 0) {
				TLS_DPRINTF("read: %s", strerror(errno));
				OK_set_error(ERR_ST_TLS_READ, ERR_LC_TLS4,
					     ERR_PT_TLS_RECORD + 2, NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_INTERNAL_ERROR);
				return -1;
			}

			if (res == 0) {
				TLS_DPRINTF("read: connection end");
				OK_set_error(ERR_ST_TLS_DISCONNECT, ERR_LC_TLS4,
					     ERR_PT_TLS_RECORD + 3, NULL);
				/* if return value of read(2) is zero,
				 * it is not necessary to send alert to
				 * peer since connection is end. */
				return -1;
			}

			sum += res;
		}

		if (once == true) {
			break;
		}
	} while (sum != (int32_t)size);

	return sum;
}

static int32_t write_record(TLS *tls, const uint8_t *buf, const uint32_t size,
			    const bool once) {
	int32_t sum = 0;

	fd_set writefds;

	FD_ZERO(&writefds);
	FD_SET(tls->sockfd, &writefds);

	int maxfd = tls->sockfd;
	do {
		fd_set fds;
		struct timespec timeout = get_timeout(tls);

		memcpy(&fds, &writefds, sizeof(fd_set));

		/* use pselect for a reason same as read_record. */
		int ready;
		if ((ready = pselect(maxfd + 1,
				     NULL, &fds, NULL, &timeout, NULL)) < 0) {
			/* TODO: check ENOMEM and EINTER. */
			TLS_DPRINTF("pselect: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_PSELECT,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 4, NULL);
			return -1;
		}

		if (ready == 0) {
			/* consider timeout same as error. */
			TLS_DPRINTF("pselect: timeout");
			OK_set_error(ERR_ST_TLS_PSELECT_TIMEOUT,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 5, NULL);
			return -1;
		}

		if (FD_ISSET(tls->sockfd, &fds)) {
			int res;

			if ((res = write(tls->sockfd,
					 &(buf[sum]), size - sum)) < 0) {
				TLS_DPRINTF("write: %s", strerror(errno));
				OK_set_error(ERR_ST_TLS_WRITE, ERR_LC_TLS4,
					     ERR_PT_TLS_RECORD + 6, NULL);
				return -1;
			}

			sum += res;
		}

		if (once == true) {
			break;
		}
	} while (sum != (int32_t)size);

	if (! incr_seqnum(tls, TLS_RECORD_IO_WRITE)) {
		return -1;
	}

	return sum;
}

static enum tls_record_ctype get_record_type(const uint8_t *buf) {
	return buf[0];
}

static struct tls_protocol_version get_tls_version(const uint8_t *buf) {
	struct tls_protocol_version ver;

	ver.major = buf[1];
	ver.minor = buf[2];

	return ver;
}

static struct tls_record *get_record_up_to_tls12(TLS *tls) {
	/* read header part of record layer. */
	uint8_t head[TLS_RECORD_HEADER_LENGTH];
	if (read_record(tls, head, TLS_RECORD_HEADER_LENGTH, false) < 0) {
		TLS_DPRINTF("retrieve_record");
		/* alert for read_record is done by internal of
		 * read_record. */
		return NULL;
	}

	/* do not check record type here. record type is checked later
	 * (internal of read_fragments function). */
	enum tls_record_ctype type = get_record_type(head);

	struct tls_protocol_version ver = get_tls_version(head);
	if (check_record_version(tls, ver) == false) {
		/* do not send protocol version alert because this
		 * routine is not at handshake layer. send unexpected
		 * message instead. */
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return NULL;
	}

	int32_t elen = tls_util_read_2(&(head[3]));
	if (elen > TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_UP_TO_TLS12) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_RECORD_OVERFLOW);
		return NULL;
	}

	uint8_t ebuf[elen];
	if (read_record(tls, ebuf, elen, false) < 0) {
		TLS_DPRINTF("read_record");
		/* alert for read_record is done by internal of
		 * read_record. */
		return NULL;
	}

	int32_t clen;
	uint8_t cbuf[TLS_RECORD_COMPRESSED_FRAGMENT_SIZE_MAX];
	if ((clen = tls_decipher(tls, cbuf, type, ebuf, elen)) < 0) {
		/* alert for tls_decipher is done by that internal. */
		return NULL;
	}

	if (clen > TLS_RECORD_COMPRESSED_FRAGMENT_SIZE_MAX) {
		/* can it happen? */
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_RECORD_OVERFLOW);
		return NULL;
	}

	uint8_t *pbuf;
	size_t   size = TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX * sizeof(uint8_t);
	if ((pbuf = malloc(size)) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 10, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	int32_t plen;
	if ((plen = tls_decompress(tls, pbuf, cbuf, clen)) < 0) {
		/* alert for tls_decompress is done by internal of
		 * tls_decompress. */
		goto err;
	}

	/*
	 * RFC8446 5.1.  Record Layer
	 *
	 *    length:  The length (in bytes) of the following
	 *       TLSPlaintext.fragment.  The length MUST NOT exceed 2^14 bytes.  An
	 *       endpoint that receives a record that exceeds this length MUST
	 *       terminate the connection with a "record_overflow" alert.
	 */
	if (plen > TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX) {
		/* can it happen? */
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 11, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_RECORD_OVERFLOW);
		goto err;
	}

	/* RFC 5246 section 6.2.1:
	 *
	 *   Implementations MUST NOT send zero-length fragments of
	 *   Handshake, Alert, or ChangeCipherSpec content types.
	 *   Zero-length fragments of Application data MAY be sent as
	 *   they are potentially useful as a traffic analysis
	 *   countermeasure. */
	if ((plen == 0) && (type != TLS_CTYPE_APPLICATION_DATA)) {
		OK_set_error(ERR_ST_TLS_INVALID_FRAGMENT_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 12, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		goto err;
	}

	struct tls_record *record;
	if ((record = malloc (1 * sizeof (struct tls_record))) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 13, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	record->type = type;
	record->ver  = ver;
	record->len  = plen;
	record->frag = pbuf;

	dump_record_type(record->type);
	TLS_DPRINTF("record: v.major = %d v.minor = %d",
		    record->ver.major, record->ver.minor);
	TLS_DPRINTF("record: fragment length = %d", record->len);

	if (! incr_seqnum(tls, TLS_RECORD_IO_READ)) {
		/* don't free pbuf. it is freed in free_record() */
		free_record(record);
		/* this should not happen. */
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	return record;

err:
	free(pbuf);

	return NULL;
}

static struct tls_record *get_record_tls13_encrypted(TLS *tls) {
	/* read header part of record layer. */
	uint8_t head[TLS_RECORD_HEADER_LENGTH];
	if (read_record(tls, head, TLS_RECORD_HEADER_LENGTH, false) < 0) {
		TLS_DPRINTF("retrieve_record");
		/* alert for read_record is done by internal of
		 * read_record. */
		return NULL;
	}

	/* do not check record type here. record type is checked later
	 * (internal of read_fragments function). */
	enum tls_record_ctype type = get_record_type(head);

	struct tls_protocol_version ver = get_tls_version(head);
	if (check_record_version(tls, ver) == false) {
		/* do not send protocol version alert because this
		 * routine is not at handshake layer. send unexpected
		 * message instead. */
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return NULL;
	}

	/*
	 * RFC8446 5.2.  Record Payload Protection
	 *
	 *    length:  The length (in bytes) of the following
	 *       TLSCiphertext.encrypted_record, which is the sum of the lengths of
	 *       the content and the padding, plus one for the inner content type,
	 *       plus any expansion added by the AEAD algorithm.  The length
	 *       MUST NOT exceed 2^14 + 256 bytes.  An endpoint that receives a
	 *       record that exceeds this length MUST terminate the connection with
	 *       a "record_overflow" alert.
	 */
	int32_t elen = tls_util_read_2(&(head[3]));
	if (elen > TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_TLS13) {
		OK_set_error(ERR_ST_TLS_RECORD_OVERFLOW,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_RECORD_OVERFLOW);
		return NULL;
	}

	uint8_t ebuf[elen];
	if (read_record(tls, ebuf, elen, false) < 0) {
		TLS_DPRINTF("read_record");
		/* alert for read_record is done by internal of
		 * read_record. */
		return NULL;
	}

	int32_t plen;
	int32_t dlen;
	uint8_t dbuf[TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_TLS13];
	uint8_t *srcbuf;

	/*
	 * RFC8446 5.  Record Protocol
	 *
	 *    An implementation may receive an unencrypted record of type
	 *    change_cipher_spec consisting of the single byte value 0x01 at any
	 *    time after the first ClientHello message has been sent or received
	 *    and before the peer's Finished message has been received and MUST
	 *    simply drop it without further processing.  Note that this record may
	 *    appear at a point at the handshake where the implementation is
	 *    expecting protected records, and so it is necessary to detect this
	 *    condition prior to attempting to deprotect the record.  An
	 *    implementation which receives any other change_cipher_spec value or
	 *    which receives a protected change_cipher_spec record MUST abort the
	 *    handshake with an "unexpected_message" alert.  If an implementation
	 *    detects a change_cipher_spec record received before the first
	 *    ClientHello message or after the peer's Finished message, it MUST be
	 *    treated as an unexpected record type (though stateless servers may
	 *    not be able to distinguish these cases from allowed cases).
	 *
	 *    Implementations MUST NOT send record types not defined in this
	 *    document unless negotiated by some extension.  If a TLS
	 *    implementation receives an unexpected record type, it MUST terminate
	 *    the connection with an "unexpected_message" alert.  New record
	 *    content type values are assigned by IANA in the TLS ContentType
	 *    registry as described in Section 11.
	 *
	 * NOTE: additional check about message length and body content is
	 * performed later because Change Cipher Spec record have to be ignored
	 * if it has no error.
	 */
	if (check_unencrypted_state(tls) ||
	    type == TLS_CTYPE_CHANGE_CIPHER_SPEC) {
		plen = elen;
		srcbuf = ebuf;
	} else {
		if (type != TLS_CTYPE_APPLICATION_DATA) {
			TLS_DPRINTF("invalid encrypted record");
			OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
				     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 2, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			return NULL;
		}

		if ((dlen = tls_decipher(tls, dbuf, type, ebuf, elen)) < 0) {
			/* alert for tls_decipher is done by that internal. */
			return NULL;
		}
		TLS_DPRINTF("record: read: total plain len = %u", dlen);

		/*
		 * RFC8446 5.4.  Record Padding
		 *
		 *    The presence of padding does not change the overall record size
		 *    limitations: the full encoded TLSInnerPlaintext MUST NOT exceed 2^14
		 *    + 1 octets.  If the maximum fragment length is reduced -- as, for
		 *    example, by the record_size_limit extension from [RFC8449] -- then
		 *    the reduced limit applies to the full plaintext, including the
		 *    content type and padding.
		 */
		uint16_t itype_len = sizeof(uint8_t);
		if (dlen > TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX + itype_len) {
			/* can it happen? */
			OK_set_error(ERR_ST_TLS_RECORD_OVERFLOW,
				     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 3, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_RECORD_OVERFLOW);
			return NULL;
		}

		/*
		 * RFC8446 5.4.  Record Padding
		 *
		 *    Implementations MUST limit their scanning to the cleartext returned
		 *    from the AEAD decryption.  If a receiving implementation does not
		 *    find a non-zero octet in the cleartext, it MUST terminate the
		 *    connection with an "unexpected_message" alert.
		 */
		/* check padding size. */
		uint16_t padlen;
		for (padlen = 0; padlen < dlen; padlen++) {
			if (dbuf[dlen - (itype_len + padlen)] != TLS_PADDING_VALUE_TLS13) {
				break;
			}
		}

		if (dlen == padlen) {
			TLS_DPRINTF("content type not found");
			OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
				     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 4, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			return NULL;
		}
		TLS_DPRINTF("record: read: padding len = %u", padlen);

		plen = dlen - (itype_len + padlen);
		type = get_record_type(&(dbuf[plen]));

		/* check whether the record is encrypted change cipher spec. */
		if (type == TLS_CTYPE_CHANGE_CIPHER_SPEC) {
			TLS_DPRINTF("received encrypted change cipher spec");
			OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
				     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 5, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			return NULL;
		}

		srcbuf = dbuf;
	}

	/*
	 * RFC8446 5.1.  Record Layer
	 *
	 *    Implementations MUST NOT send zero-length fragments of Handshake
	 *    types, even if those fragments contain padding.
	 *
	 *    Alert messages (Section 6) MUST NOT be fragmented across records, and
	 *    multiple alert messages MUST NOT be coalesced into a single
	 *    TLSPlaintext record.  In other words, a record with an Alert type
	 *    MUST contain exactly one message.
	 *
	 *    Application Data messages contain data that is opaque to TLS.
	 *    Application Data messages are always protected.  Zero-length
	 *    fragments of Application Data MAY be sent, as they are potentially
	 *    useful as a traffic analysis countermeasure.  Application Data
	 *    fragments MAY be split across multiple records or coalesced into a
	 *    single record.
	 */
	/*
	 * RFC8446 5.4.  Record Padding
	 *
	 *                                                       Implementations
	 *    MUST NOT send Handshake and Alert records that have a zero-length
	 *    TLSInnerPlaintext.content; if such a message is received, the
	 *    receiving implementation MUST terminate the connection with an
	 *    "unexpected_message" alert.
	 */
	if ((plen == 0) && (type != TLS_CTYPE_APPLICATION_DATA)) {
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 6, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return NULL;
	}

	uint8_t *pbuf;
	size_t   size = TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX * sizeof(uint8_t);
	if ((pbuf = malloc(size)) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}
	memcpy(pbuf, srcbuf, plen);

	struct tls_record *record;
	if ((record = malloc (1 * sizeof (struct tls_record))) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		free(pbuf);
		return NULL;
	}

	record->type = type;
	record->ver  = ver;
	record->len  = plen;
	record->frag = pbuf;

	dump_record_type(record->type);
	TLS_DPRINTF("record: v.major = %d v.minor = %d",
		    record->ver.major, record->ver.minor);
	TLS_DPRINTF("record: fragment length = %d", record->len);

	if (type != TLS_CTYPE_CHANGE_CIPHER_SPEC &&
	    ! incr_seqnum(tls, TLS_RECORD_IO_READ)) {
		/* don't free pbuf. it is freed in free_record() */
		free_record(record);
		/* this should not happen. */
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	return record;
}

static struct tls_record * get_record(TLS *tls) {
	/*
	 * use get_record_up_to_tls12() until ServerHello is sent even if
	 * TLS 1.3 is negotiated.
	 */
	if (tls->state < TLS_STATE_HS_AFTER_RECV_SHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_SEND_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_SEND_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_2NDSHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_HRREQ ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_CHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_CHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_SEND_HRREQ) {
		return get_record_up_to_tls12(tls);
	}

	/* version was negotiated. */
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		return get_record_tls13_encrypted(tls);

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		return get_record_up_to_tls12(tls);
	}
}

static void free_record(struct tls_record *record) {
/*	free(record->frag); */
	record->frag = NULL;

/* 	free(record); */
	record = NULL;
}

static void remove_queue(struct tls_record_queue *queue,
			 struct tls_record *record) {
	TAILQ_REMOVE(queue->head, record, link);

	free_record(record);

	queue->pos = 0;
}

static ssize_t read_queue(struct tls_record_queue *queue,
			  uint8_t *buf, size_t count) {
	ssize_t read_byte = 0;

	/* TODO: do not operate count variable. count should be const. */
	struct tls_record *record;
	TAILQ_FOREACH(record, queue->head, link) {
		TLS_DPRINTF("record: read: Q: len = %d, pos = %zd, count = %zd",
			    record->len, queue->pos, count);
		size_t len = record->len - queue->pos;
		if (len <= count) {
			memcpy(&(buf[read_byte]),
			       &(record->frag[queue->pos]), len);

			read_byte += len;
			count     -= len;

			remove_queue(queue, record);
		} else {
			len = count;
			memcpy(&(buf[read_byte]),
			       &(record->frag[queue->pos]), len);

			read_byte  += len;
			queue->pos += len;
			count -= len;
		}

		if (count == 0) {
			break;
		}
	}
	TLS_DPRINTF("record: Q: pos = %zd, read = %zd", queue->pos, read_byte);

	return read_byte;
}

static bool read_fragments_handshake(const enum tls_record_ctype upper_type,
				     TLS *tls, struct tls_record *record) {
	/* if upper_type is application and type is handshake, do
	 * re-handshake. this is a re-negotiation. */
	if (upper_type == TLS_CTYPE_APPLICATION_DATA) {
		uint16_t version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_TLS13:
			/* handle post-handshake message. */
			TAILQ_INSERT_TAIL(tls->queue_control->head,
					  record, link);
			if (tls_hs_posthandshake_message(tls) == false) {
				TLS_DPRINTF("tls_hs_posthandshake_message");
				return false;
			}
			break;

		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
		default:
#if 1
			/* currently, re-negotiation do not be
			 * supported. because secure re-negotiation extension is
			 * not implemented. send no renegotiation alert. since
			 * no renegotiation alert is always a warning, ignore
			 * this handshake, and, progress. */
			OK_set_error(ERR_ST_TLS_NO_RENEGOTIATION,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 14, NULL);
			TLS_ALERT_WARN(tls, TLS_ALERT_DESC_NO_RENEGOTIATION);
#else
			tls->state = TLS_STATE_HS_REINIT;
#endif /* 0 */
			break;
		}
	} else {
		TAILQ_INSERT_TAIL(tls->queue_control->head, record, link);
	}

	return true;
}

static bool read_fragments_app(TLS *tls, struct tls_record *record) {
	if (tls->handshake_over == true) {
		TAILQ_INSERT_TAIL(tls->queue_data->head, record, link);
		return true;
	}

	/* first handshake is not over. this message is not
	 * acceptable. */
	OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG, ERR_LC_TLS4,
		     ERR_PT_TLS_RECORD + 15, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
	return false;
}

static ssize_t read_fragments(const enum tls_record_ctype upper_type,
			      TLS *tls, uint8_t *buf, const size_t count) {
	/* read only the count size data from the fragment of one or
	 * more record. */
	ssize_t pos = 0;

	struct tls_record_queue *queue;
	switch (upper_type) {
	case TLS_CTYPE_HANDSHAKE:
		queue = tls->queue_control;
		break;

	case TLS_CTYPE_APPLICATION_DATA:
		queue = tls->queue_data;
		break;

	default:
		/* this is specification mistake of upper program
		 * (application that use this tls library). */
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_TYPE,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 16, NULL);
		return -1;
	}

	if (! TAILQ_EMPTY(queue->head)) {
		if ((size_t) (pos = read_queue(queue, buf, count)) == count) {
			goto fin;
		}
	}

	/* count that application data was read. */
	uint32_t count_app = 0;

	/* queue->head is empty now. */
	while ((size_t)pos != count) {
		/* retrieve the whole 1 record. */
		struct tls_record *record = get_record(tls);
		if (record == NULL) {
			/* in application, if pos > 0, return that
			 * value. */
			TLS_DPRINTF("get_record: NULL");
			if ((upper_type == TLS_CTYPE_APPLICATION_DATA) &&
			    (pos >= 0)) {
				return pos;
			}
			/* in handshake or pos == 0, return error.
			 * alert for this is done by internal of
			 * get_record. */
			return -1;
		}

		switch (record->type) {
		case TLS_CTYPE_HANDSHAKE:
			if (read_fragments_handshake(upper_type,
						     tls, record) == false) {
				free_record(record);
				return -1;
			}
			break;

		case TLS_CTYPE_APPLICATION_DATA:
			if (read_fragments_app(tls, record) == false) {
				free_record(record);
				return -1;
			}
			count_app += 1;
			break;

		case TLS_CTYPE_CHANGE_CIPHER_SPEC:
			/* check for executability of tls_ccs_recv do in
			 * the tls_ccs_recv. */
			if (! tls_ccs_recv(tls, record)) {
				free_record(record);
				return -1;
			}
			free_record(record);
			break;

		case TLS_CTYPE_ALERT:
			/* alert for tls_alert_recv is occurred in the
			 * internal of tls_alert_recv. */
			if (! tls_alert_recv(tls, record)) {
				free_record(record);
				if (tls->errnum == TLS_ERR_CLOSE_NOTIFY) {
					goto fin;
				}
				return -1;
			}
			free_record(record);
			break;

		default:
			/* received unknown record type. */
			/*
			 * RFC8446 5.  Record Protocol
			 *
			 *                                                   If a TLS
			 *    implementation receives an unexpected record type, it MUST terminate
			 *    the connection with an "unexpected_message" alert.
			 */
			TLS_DPRINTF("E: type %d is invalid.", record->type);
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_TYPE,
				     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 17, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			return -1;
		}

		/* if record->type is ccs or alert, queue is still
		 * empty. so read_queue ends by doing nothing (return
		 * 0). */
		ssize_t inc;
		if ((inc = read_queue(queue, buf, count)) < 0) {
			/* this should not happen. if happend,
			 * implementation error. */
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		pos += inc;

		/* in application, read(2) is called only once in the
		 * maximum. return read value (do not wait to be pos ==
		 * count). but, confirm that data is read once. (avoid
		 * the returning zero without reading application
		 * data) */
		if ((upper_type == TLS_CTYPE_APPLICATION_DATA) &&
		    (count_app > 0) &&
		    (pos > 0)) {
			break;
		}
	}

fin:
	return pos;
}

static void write_record_version(const TLS* tls, uint8_t *buf) {
	/*
	 * RFC8446 D.5.  Security Restrictions Related to Backward Compatibility
	 *
	 *    Implementations MUST NOT send any records with a version less than
	 *    0x0300.
	 */
	buf[0] = tls->record_version.major;
	buf[1] = tls->record_version.minor;
}

static int32_t compose_record_up_to_tls12(const enum tls_record_ctype type,
					  TLS *tls,
					  uint8_t *buf,
					  const uint8_t *pbuf,
					  const size_t plen) {
	/* generate compressed fragment. */
	int32_t clen;
	uint8_t cbuf[TLS_RECORD_COMPRESSED_FRAGMENT_SIZE_MAX];
	if ((clen = tls_compress(tls, cbuf, pbuf, plen)) < 0) {
		TLS_DPRINTF("tls_compress");
		return -1;
	}
	TLS_DPRINTF("record: write: compressed len = %d", clen);

	/* generate encrypted fragment. */
	int32_t elen;
	uint8_t ebuf[TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_UP_TO_TLS12];
	if ((elen = tls_cipher(tls, ebuf, type, cbuf, clen)) < 0) {
		TLS_DPRINTF("tls_cipher");
		return -1;
	}
	TLS_DPRINTF("record: write: encrypted len = %d", elen);

	uint32_t len = TLS_RECORD_HEADER_LENGTH + elen;

	buf[0] = type;

	write_record_version(tls, &(buf[1]));

	tls_util_write_2(&(buf[3]), elen);

	memcpy(&(buf[TLS_RECORD_HEADER_LENGTH]), ebuf, elen);

	return len;
}

static int32_t compose_record_tls13(const enum tls_record_ctype type,
				    TLS *tls,
				    uint8_t *buf,
				    const uint8_t *pbuf,
				    const size_t plen,
				    const size_t padlen) {
	if (type == TLS_CTYPE_CHANGE_CIPHER_SPEC) {
		buf[0] = TLS_CTYPE_CHANGE_CIPHER_SPEC;
		write_record_version(tls, &(buf[1]));
		tls_util_write_2(&(buf[3]), plen);
		memcpy(&(buf[TLS_RECORD_HEADER_LENGTH]), pbuf, plen);
		return TLS_RECORD_HEADER_LENGTH + plen;
	}

	/* format TLSInnerPlaintext. */
	int32_t iplen;
	int32_t itype_len = sizeof(uint8_t);
	uint8_t ipbuf[TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX + itype_len];

	/*
	 * RFC8446 5.4.  Record Padding
	 *
	 *    When generating a TLSCiphertext record, implementations MAY choose to
	 *    pad.  An unpadded record is just a record with a padding length of
	 *    zero.  Padding is a string of zero-valued bytes appended to the
	 *    ContentType field before encryption.  Implementations MUST set the
	 *    padding octets to all zeros before encrypting.
	 */
	memcpy(ipbuf, pbuf, plen);
	ipbuf[plen] = type;
	memset(&(ipbuf[plen + itype_len]), 0, padlen);
	iplen = plen + itype_len + padlen;
	TLS_DPRINTF("record: write: padding len = %lu", padlen);
	TLS_DPRINTF("record: write: total plain len = %d", iplen);

	/* generate encrypted fragment. */
	int32_t elen;
	uint8_t ebuf[TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_TLS13];
	if ((elen = tls_cipher(tls, ebuf, type, ipbuf, iplen)) < 0) {
		TLS_DPRINTF("tls_cipher");
		return -1;
	}
	TLS_DPRINTF("record: write: encrypted len = %d", elen);

	uint32_t len = TLS_RECORD_HEADER_LENGTH + elen;
	buf[0] = TLS_CTYPE_APPLICATION_DATA;

	write_record_version(tls, &(buf[1]));

	tls_util_write_2(&(buf[3]), elen);

	memcpy(&(buf[TLS_RECORD_HEADER_LENGTH]), ebuf, elen);

	return len;
}

static int32_t compose_record(const enum tls_record_ctype type,
			      TLS *tls,
			      uint8_t *buf,
			      const uint8_t *pbuf,
			      const size_t plen,
			      const size_t padlen UNUSED) {
	if (tls->state < TLS_STATE_HS_AFTER_RECV_SHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_SEND_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_SEND_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_2NDSHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_HRREQ ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_CHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_CHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_RECV_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_AFTER_RECV_2NDCHELLO ||
	    tls->state == TLS_STATE_HS_BEFORE_SEND_HRREQ) {
		return compose_record_up_to_tls12(type, tls, buf, pbuf, plen);
	}

	/* version was negotiated. */
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		return compose_record_tls13(type, tls, buf, pbuf, plen, padlen);

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		return compose_record_up_to_tls12(type, tls, buf, pbuf, plen);
	}
}

struct tls_record_queue * tls_record_init(void) {
	struct tls_record_queue *que;

	/* tls_record_init function is called before handshake. thefore,
	 * it is not necessary to care alert. */

	if ((que = malloc (1 * sizeof(struct tls_record_queue))) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 18, NULL);
		return NULL;
	}
	que->pos = 0;

	if ((que->head = malloc(1 * sizeof(struct tls_record_head))) == NULL) {
		TLS_DPRINTF("record: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 19, NULL);
		free(que);
		return NULL;
	}

	TAILQ_INIT(que->head);

	return que;
}

void tls_record_free(struct tls_record_queue *queue) {
	if ((queue == NULL) || (queue->head == NULL)) {
		OK_set_error(ERR_ST_NULLPOINTER,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 9, NULL);
		return ;
	}

	while(!(TAILQ_EMPTY(queue->head))) {
		struct tls_record *record = TAILQ_FIRST(queue->head);
		remove_queue(queue, record);
		record = NULL;
	}

	free(queue->head);
	queue->head = NULL;

	free(queue);
	queue = NULL;
}

ssize_t tls_record_read(const enum tls_record_ctype type,
			TLS *tls, uint8_t *buf, const size_t count) {
	if (tls->state == TLS_STATE_CLOSED ||
	    tls->active_read.closed == true) {
		OK_set_error(ERR_ST_TLS_CONNECTION_CLOSED,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 10, NULL);
		return -1;
	}

	/* CCS and ALERT type do not read directly. So, if specified
	 * these types, consider invalid. */
	switch (type) {
	case TLS_CTYPE_HANDSHAKE:
	case TLS_CTYPE_APPLICATION_DATA:
		return read_fragments(type, tls, buf, count);

	case TLS_CTYPE_CHANGE_CIPHER_SPEC:
	case TLS_CTYPE_ALERT:
		TLS_DPRINTF("E: type %d is invlaid.", type);
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_TYPE,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 20, NULL);
		return -1;

	default:
		TLS_DPRINTF("E: type %d is unknown.", type);
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_TYPE,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 21, NULL);
		return -1;
	}
}

ssize_t tls_record_write(const enum tls_record_ctype type,
			 TLS *tls, const uint8_t *buf, const size_t count) {
	if (tls->state == TLS_STATE_CLOSED ||
	    tls->active_write.closed == true) {
		OK_set_error(ERR_ST_TLS_CONNECTION_CLOSED,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 11, NULL);
		return -1;
	}

	switch (type) {
	case TLS_CTYPE_HANDSHAKE:
		TLS_DPRINTF("record: type = HANDSHAKE");
		break;

	case TLS_CTYPE_CHANGE_CIPHER_SPEC:
		TLS_DPRINTF("record: type = CCS");
		break;

	case TLS_CTYPE_ALERT:
		TLS_DPRINTF("record: type = ALERT");
		break;

	case TLS_CTYPE_APPLICATION_DATA:
		TLS_DPRINTF("record: type = APPLICATION");
		break;

	default:
		TLS_DPRINTF("E: type %d is unknown type", type);
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_TYPE,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 22, NULL);
		return -1;
	}

	/* zero-length fragment do not be allowed except application
	 * type. */
	if ((type != TLS_CTYPE_APPLICATION_DATA) &&
	    (count == 0)) {
		TLS_DPRINTF("record: count = 0");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS4, ERR_PT_TLS_RECORD + 23, NULL);
		return -1;
	}

	size_t woffset = 0;
	do {
		size_t plen = count - woffset;
		size_t pad_len = 0;

		/*
		 * TODO: set a value that is lower than or equal to 2^14 to
		 * pad_len when TLS 1.3 is negotiated and record padding is in
		 * use. pad_len will be determined by implementor but there is
		 * no guideline in RFC8446. it can be difficult to implement
		 * padding policy that is actually useful.
		 */
		if (plen > TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX - pad_len) {
			plen = TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX - pad_len;
		}
		TLS_DPRINTF("record: write: plain len = %zu / %zu", plen, count);

		int32_t len;
		uint8_t dat[TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_UP_TO_TLS12];
		if ((len = compose_record(type, tls, dat,
				  &(buf[woffset]), plen, pad_len)) < 0) {
			/* return the ever written size if compose
			 * process failed. */
			TLS_DPRINTF("compose_record");
			return woffset;
		}

		TLS_DPRINTF("record: write: len = %d", len);
		if (write_record(tls, dat, len, false) < 0) {
			/* if failed to write ciphered text, return the
			 * written failed information since it does not
			 * know the written size in the plainText. */
			return -1;
		}

		woffset += plen;
		TLS_DPRINTF("record: offset = %zu, count = %zu",
			    woffset, count);
	} while (woffset < count);

	return count;
}


ssize_t TLS_read(TLS *tls, void *buf, const size_t count) {
	/* FIXME: this is a compatible code for AiSSL. this code means
	 * call the raw read(2) when TLS_handshake have not been still
	 * executed. this code should be fixed since i think this is a
	 * layer violation. */
	if ((tls->state == TLS_STATE_CLOSED) &&
	    (tls->handshake_over == false)) {
		return (ssize_t) read_record(tls, buf, count, true);
	}

	/* if connection of tls was closed, return zero; */
	if (tls->state == TLS_STATE_CLOSED) {
		return 0;
	}

	/* if read side of connection is closed, return zero; */
	if (tls->active_read.closed == true) {
		return 0;
	}

	/* do not permit count as zero. */
	if (count == 0) {
		TLS_DPRINTF("count == 0");
		return 0;
	}

	ssize_t read_bytes;
	read_bytes = tls_record_read(TLS_CTYPE_APPLICATION_DATA,
				     tls, buf, count);

	/*
	 * RFC8446 5.5.  Limits on Key Usage
	 *
	 *    There are cryptographic limits on the amount of plaintext which can
	 *    be safely encrypted under a given set of keys.  [AEAD-LIMITS]
	 *    provides an analysis of these limits under the assumption that the
	 *    underlying primitive (AES or ChaCha20) has no weaknesses.
	 *    Implementations SHOULD do a key update as described in Section 4.6.3
	 *    prior to reaching these limits.
	 *
	 *    For AES-GCM, up to 2^24.5 full-size records (about 24 million) may be
	 *    encrypted on a given connection while keeping a safety margin of
	 *    approximately 2^-57 for Authenticated Encryption (AE) security.  For
	 *    ChaCha20/Poly1305, the record sequence number would wrap before the
	 *    safety limit is reached.
	 */
	/*
	 * AES-GCM is available regardless of protocol version, so maybe
	 * version check doesn't be needed here.
	 */
	if (tls->active_read.cipher.cipher_algorithm == TLS_BULK_CIPHER_AES &&
	    tls->active_read.cipher.cipher_type == TLS_CIPHER_TYPE_AEAD &&
	    tls->active_read.seqnum >= TLS_AEAD_RECORD_LIMIT_AES_GCM) {
		TLS_ALERT_WARN(tls, TLS_ALERT_DESC_CLOSE_NOTIFY);
	}

	/*
	 * RFC8446 5.3.  Per-Record Nonce
	 *
	 *    Because the size of sequence numbers is 64-bit, they should not wrap.
	 *    If a TLS implementation would need to wrap a sequence number, it MUST
	 *    either rekey (Section 4.6.3) or terminate the connection.
	 */
	if (tls->active_read.seqnum >= TLS_RECORD_SEQNUM_SIZE_MAX_TLS13) {
		TLS_ALERT_WARN(tls, TLS_ALERT_DESC_CLOSE_NOTIFY);
	}

	return read_bytes;
}

ssize_t TLS_gets(TLS *tls, char *buf, const size_t size)
{
	/* this function is deprecated. prepare the TLS version of getline(3). */
	size_t i;

	for (i = 0; i < size; ++i) {
		char tmp[4];

		int n UNUSED;
		if ((n = TLS_read(tls, tmp, 1)) < 1) {
			TLS_DPRINTF("TLS_read: %d", n);
			return -1;
		}

		buf[i] = tmp[0];

		if (buf[i] == '\n') {
			break;
		}
	}

	if (i == size) {
		i--;
	}

	return i + 1;
}

ssize_t TLS_write(TLS *tls, const void *buf, const size_t count) {
	/* FIXME: this is a compatible code for AiSSL. this code means
	 * call the raw write(2) when TLS_handshake have not been still
	 * executed. this code should be fixed since i think this is a
	 * layer violation. */
	if ((tls->state == TLS_STATE_CLOSED) &&
	    (tls->handshake_over == false)) {
		return (ssize_t)write_record(tls, buf, count, true);
	}

	if (tls->active_write.closed == true) {
		OK_set_error(ERR_ST_TLS_CONNECTION_CLOSED,
			     ERR_LC_TLS6, ERR_PT_TLS_RECORD2 + 12, NULL);
		return -1;
	}

	ssize_t write_bytes;
	write_bytes = tls_record_write(TLS_CTYPE_APPLICATION_DATA,
				       tls, buf, count);

	/*
	 * RFC8446 5.5.  Limits on Key Usage
	 *
	 *    There are cryptographic limits on the amount of plaintext which can
	 *    be safely encrypted under a given set of keys.  [AEAD-LIMITS]
	 *    provides an analysis of these limits under the assumption that the
	 *    underlying primitive (AES or ChaCha20) has no weaknesses.
	 *    Implementations SHOULD do a key update as described in Section 4.6.3
	 *    prior to reaching these limits.
	 *
	 *    For AES-GCM, up to 2^24.5 full-size records (about 24 million) may be
	 *    encrypted on a given connection while keeping a safety margin of
	 *    approximately 2^-57 for Authenticated Encryption (AE) security.  For
	 *    ChaCha20/Poly1305, the record sequence number would wrap before the
	 *    safety limit is reached.
	 */
	/*
	 * AES-GCM is available regardless of protocol version, so maybe
	 * version check doesn't be needed here.
	 */
	if (tls->active_write.cipher.cipher_algorithm == TLS_BULK_CIPHER_AES &&
	    tls->active_write.cipher.cipher_type == TLS_CIPHER_TYPE_AEAD &&
	    tls->active_write.seqnum >= TLS_AEAD_RECORD_LIMIT_AES_GCM) {
		TLS_ALERT_WARN(tls, TLS_ALERT_DESC_CLOSE_NOTIFY);
	}

	/*
	 * RFC8446 5.3.  Per-Record Nonce
	 *
	 *    Because the size of sequence numbers is 64-bit, they should not wrap.
	 *    If a TLS implementation would need to wrap a sequence number, it MUST
	 *    either rekey (Section 4.6.3) or terminate the connection.
	 */
	if (tls->active_write.seqnum >= TLS_RECORD_SEQNUM_SIZE_MAX_TLS13) {
		TLS_ALERT_WARN(tls, TLS_ALERT_DESC_CLOSE_NOTIFY);
	}

	return write_bytes;
}
