/*
 * Copyright (c) 2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls.h"

/**
 * get current timeout setting of read(2)/write(2).
 *
 * if sel_timeout variable is positive value (> 0), this function
 * returns that value for compatibility of AiSSL.
 */
static int get_timeout(TLS *tls);

/**
 * get current setting of TLS_OPT_IMMEDIATE_HANDSHAKE flag.
 */
static int get_immeidate_handshake(TLS *tls);

/**
 * get current setting of TLS_OPT_USE_CLIENT_CERT_AUTH flag.
 */
static int get_opt_of_client_cert_auth(TLS *tls);

/**
 * get current setting of TLS_OPT_MAY_USE_CLIENT_CERT_AUTH flag.
 */
static int get_opt_of_client_cert_auth_with_no_check(TLS *tls);

/**
 * set timeout of read(2)/write(2).
 */
static void set_timeout(TLS *tls, void *val);

/**
 * set TLS_OPT_IMMEDIATE_HANDSHAKE flag.
 */
static void set_immediate_handshake(TLS *tls, void *val);

/**
 * set TLS_OPT_USE_CLIENT_CERT_AUTH flag.
 *
 * if set this option, TLS_OPT_MAY_USE_CLIENT_CERT_AUTH become off.
 * internally, set_client_certificate_check function is called with
 * "off" argument.
 */
static void set_certificate_request(TLS *tls, void *val);

/**
 * switch null check of certificate in the certificate request handshake
 * protocol message.
 */
static void set_client_certificate_check(TLS *tls, void *val);

/**
 * set TLS_OPT_MAY_USE_CLIENT_CERT_AUTH flag.
 *
 * if set this option, TLS_OPT_USE_CLIENT_CERT_AUTH become off.
 * internally, set_client_certificate_check function is called with "on"
 * argument.
 */
static void set_certificate_request_with_no_check(TLS *tls, void *val);

/**
 * set verification type of certificate.
 */
static void set_cert_verify_type(TLS *tls, void *val);

/**
 * set verification depth of certificate.
 */
static void set_cert_verify_depth(TLS *tls, void *val);

/**
 * timeout second for read(2)/write(2)
 *
 * this global variable is prepared for compatibility of AiSSL. this is
 * a deprecated variable. this variable should be removed from TLS
 * module in the future.
 */
int sel_timeout = 0;

static int get_timeout(TLS *tls) {
	if (sel_timeout > 0) {
		return sel_timeout;
	}

	return (tls->opt.timeout.tv_sec * 1000) +
		(tls->opt.timeout.tv_nsec * (1000 * 1000));
}

static int get_immeidate_handshake(TLS *tls) {
	if (tls->opt.immediate_handshake) {
		return 0;
	}
	return -1;
}

static int get_opt_of_client_cert_auth(TLS *tls) {
	if (tls->opt.use_certreq == true) {
		return 0;
	}
	return -1;
}

static int get_opt_of_client_cert_auth_with_no_check(TLS *tls) {
	if (tls->opt.skip_ccertificate_check == true) {
		return 0;
	}
	return -1;
}

static void set_timeout(TLS *tls, void *val) {
	struct timespec *timeout = (struct timespec *) val;

	tls->opt.timeout.tv_sec  = timeout->tv_sec;
	tls->opt.timeout.tv_nsec = timeout->tv_nsec;
}

static void set_immediate_handshake(TLS *tls, void *val) {
	bool *on = (bool *) val;

	tls->opt.immediate_handshake = *on;
}

static void set_certificate_request(TLS *tls, void *val) {
	bool *on = (bool *) val;
	bool off = false;

	tls->opt.use_certreq = *on;

	set_client_certificate_check(tls, &off);
}

static void set_client_certificate_check(TLS *tls, void *val) {
	bool *off = (bool *) val;

	tls->opt.skip_ccertificate_check = *off;
}

static void set_certificate_request_with_no_check(TLS *tls, void *val) {
	bool *on = (bool *) val;

	set_certificate_request(tls, on);
	set_client_certificate_check(tls, on);
}


static void set_cert_verify_type(TLS *tls, void *val) {
	enum tls_cert_verify_type *type = (enum tls_cert_verify_type *) val;

	tls->opt.verify_type = *type;
}

static void set_cert_verify_depth(TLS *tls, void *val) {
	uint32_t *depth = (uint32_t *) val;

	tls->opt.verify_depth = *depth;
}

int TLS_opt_get(TLS *tls, enum tls_opt_key key) {
	switch (key) {
	case TLS_OPT_TIMEOUT:
		return get_timeout(tls);

	case TLS_OPT_IMMEDIATE_HANDSHAKE:
		return get_immeidate_handshake(tls);


	case TLS_OPT_USE_CLIENT_CERT_AUTH:
		return get_opt_of_client_cert_auth(tls);

	case TLS_OPT_MAY_USE_CLIENT_CERT_AUTH:
		return get_opt_of_client_cert_auth_with_no_check(tls);

	default:
		OK_set_error(ERR_ST_TLS_INVALID_OPT_KEY,
			     ERR_LC_TLS4, ERR_PT_TLS_OPT + 0, NULL);
		break;
	}

	return -1;
}

void TLS_opt_set(TLS *tls, enum tls_opt_key key, void *val) {
	switch (key) {
	case TLS_OPT_TIMEOUT:
		set_timeout(tls, val);
		break;

	case TLS_OPT_IMMEDIATE_HANDSHAKE:
		set_immediate_handshake(tls, val);
		break;

	case TLS_OPT_USE_CLIENT_CERT_AUTH:
		set_certificate_request(tls, val);
		break;

	case TLS_OPT_MAY_USE_CLIENT_CERT_AUTH:
		set_certificate_request_with_no_check(tls, val);
		break;

	case TLS_OPT_CERT_VERIFY_TYPE:
		set_cert_verify_type(tls, val);
		break;

	case TLS_OPT_CERT_VERIFY_DEPTH:
		set_cert_verify_depth(tls, val);
		break;

	default:
		OK_set_error(ERR_ST_TLS_INVALID_OPT_KEY,
			     ERR_LC_TLS4, ERR_PT_TLS_OPT + 1, NULL);
		break;
	}
}
