/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_session.h"
#include "tls_record.h"
#include "tls_handshake.h"
#include "tls_ccs.h"
#include "tls_alert.h"
#include "tls_cipher.h"

#include <stdlib.h>
#include <string.h>

/** @see tls_key.c */
extern bool tls_key_derive_application_traffic_secret_n(
	TLS *tls, struct tls_connection *connection);

/** @see tls_key.c */
bool tls_key_make_traffic_key(TLS *tls, struct tls_connection *connection);

/**
 * output the name of handshake type as the debug purpose information.
 *
 * debug purpose output of this function is done when TLS_DEBUG macro is
 * defined on building phase.
 */
static void dump_handshake_type(const enum tls_handshake_type type);

/**
 * initialize queue of handshake.
 *
 * this queue is used by calculating digest of handshakes that is used
 * by certificate request and finished handshake protocol message.
 *
 * in the initializing queue, memory allocation is done. if that memory
 * become unnecessary, you must free by handshake_queue_free function.
 */
static bool handshake_queue_init(TLS *tls);

/**
 * free the memory that is allocated by handshake_queue_init function.
 */
static void handshake_queue_free(TLS *tls);

/**
 * check whether the handshake type is known.
 */
static bool check_handshake_type_up_to_tls12(const enum tls_handshake_type type);
static bool check_handshake_type_tls13(const enum tls_handshake_type type);
static bool check_handshake_type(TLS *tls, const enum tls_handshake_type);

/**
 * update hash when tls_handshake_read function was done.
 *
 * by handshake type, determine whether function updates a hash in fact.
 * for example, if read handshake type was hello request, do not update
 * hash because hello request do not include to the hash.
 */
static void hash_update_for_read(TLS *tls, struct tls_hs_msg *msg);

/**
 * update hash when tls_handshake_write function was done.
 *
 * by handshake type, determine whether function updates a hash in fact.
 * for example, if read handshake type was hello request, do not update
 * hash because hello request do not include to the hash.
 */
static void hash_update_for_write(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake type from received TLS/SSL record data.
 *
 * TODO: i think this function is enough in inline.
 */
static enum tls_handshake_type read_handshake_type(const uint8_t *buf);

/**
 * make handshake message.
 *
 * in this function, make one char pointer buffer data from tls_hs_msg
 * structure data. this function is used by tls_handshake_write function
 * and calculating digest of handshake.
 */
static void make_handshake_message(uint8_t *buf,
				   const struct tls_hs_msg *msg);

/**
 * clean up session information as the thing that is not referred to.
 *
 * this function is called at the last of handshake process.
 */
static void cleanup_session(TLS *tls);

/**
 * start handshake process.
 */
static bool start_handshake(TLS *tls);

/**
 * process post-handshake messages.
 */
static bool posthandshake_message_tls13(TLS *tls);

/* if TLS_DEBUG is off, type is not used. so, add attribute UNUSED. */
static void dump_handshake_type(const enum tls_handshake_type type UNUSED) {
#ifdef TLS_DEBUG
	char name[30];
	switch(type) {
	case TLS_HANDSHAKE_HELLO_REQUEST:
		strcpy(name, "HELLO REQUEST");
		break;

	case TLS_HANDSHAKE_CLIENT_HELLO:
		strcpy(name, "CLIENT HELLO");
		break;

	case TLS_HANDSHAKE_SERVER_HELLO:
		strcpy(name, "SERVER HELLO");
		break;

	case TLS_HANDSHAKE_NEW_SESSION_TICKET:
		strcpy(name, "NEW SESSION TICKET");
		break;

	case TLS_HANDSHAKE_END_OF_EARLY_DATA:
		strcpy(name, "END OF EARLY DATA");
		break;

	case TLS_HANDSHAKE_ENCRYPTED_EXTENSIONS:
		strcpy(name, "ENCRYPTED EXTENSIONS");
		break;

	case TLS_HANDSHAKE_CERTIFICATE:
		strcpy(name, "CERTIFICATE");
		break;

	case TLS_HANDSHAKE_SERVER_KEY_EXCHANGE:
		strcpy(name, "SERVER KEY EXCHANGE");
		break;

	case TLS_HANDSHAKE_CERTIFICATE_REQUEST:
		strcpy(name, "CERTIFICATE_REQUEST");
		break;

	case TLS_HANDSHAKE_SERVER_HELLO_DONE:
		strcpy(name, "SERVER_HELLO_DONE");
		break;

	case TLS_HANDSHAKE_CERTIFICATE_VERIFY:
		strcpy(name, "CERTIFICATE_VERIFY");
		break;

	case TLS_HANDSHAKE_CLIENT_KEY_EXCHANGE:
		strcpy(name, "CLIENT_KEY_EXCHANGE");
		break;

	case TLS_HANDSHAKE_FINISHED:
		strcpy(name, "FINISHED");
		break;

	case TLS_HANDSHAKE_KEY_UPDATE:
		strcpy(name, "KEY UPDATE");
		break;

	case TLS_HANDSHAKE_MESSAGE_HASH:
		strcpy(name, "MESSAGE HASH:");
		break;

	default:
		strcpy(name, "unknown type");
		break;
	}

	TLS_DPRINTF("handshake: type = %s", name);
#endif
}

static bool handshake_queue_init(TLS *tls) {
	if (tls->queue_handshake != NULL) {
		TLS_DPRINTF("handshake: other handshake progress.");
		OK_set_error(ERR_ST_TLS_HS_SINGLE,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 0, NULL);
		return false;
	}

	if((tls->queue_handshake =
	    malloc(1 * sizeof (struct tls_handshake_queue))) == NULL) {
		TLS_DPRINTF("handshake: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 1, NULL);
		return false;
	}

	TAILQ_INIT(tls->queue_handshake);

	return true;
}

static void handshake_queue_free(TLS *tls) {
	struct tls_hs_msg *msg;

	if (tls->queue_handshake == NULL) {
		TLS_DPRINTF("handshake: queue_handshake NULL");
		OK_set_error(ERR_ST_TLS_HS_QUEUE_NULL,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 2, NULL);
		return ;
	}

	while(!(TAILQ_EMPTY(tls->queue_handshake))) {
		msg = TAILQ_FIRST(tls->queue_handshake);

		TAILQ_REMOVE(tls->queue_handshake, msg, link);

		tls_hs_msg_free(msg);
		msg = NULL;
	}

	free(tls->queue_handshake);
	tls->queue_handshake = NULL;
}

static bool check_handshake_type_up_to_tls12(const enum tls_handshake_type type) {
	switch (type) {
	case TLS_HANDSHAKE_HELLO_REQUEST:
	case TLS_HANDSHAKE_CLIENT_HELLO:
	case TLS_HANDSHAKE_SERVER_HELLO:
	case TLS_HANDSHAKE_CERTIFICATE:
	case TLS_HANDSHAKE_SERVER_KEY_EXCHANGE:
	case TLS_HANDSHAKE_CERTIFICATE_REQUEST:
	case TLS_HANDSHAKE_SERVER_HELLO_DONE:
	case TLS_HANDSHAKE_CERTIFICATE_VERIFY:
	case TLS_HANDSHAKE_CLIENT_KEY_EXCHANGE:
	case TLS_HANDSHAKE_FINISHED:
		return true;

	default:
		return false;
	}
}

static bool check_handshake_type_tls13(const enum tls_handshake_type type) {
	/*
	 * RFC8446 4.  Handshake Protocol
	 *
	 *      enum {
	 *          client_hello(1),
	 *          server_hello(2),
	 *          new_session_ticket(4),
	 *          end_of_early_data(5),
	 *          encrypted_extensions(8),
	 *          certificate(11),
	 *          certificate_request(13),
	 *          certificate_verify(15),
	 *          finished(20),
	 *          key_update(24),
	 *          message_hash(254),
	 *          (255)
	 *      } HandshakeType;
	 */
	switch (type) {
	case TLS_HANDSHAKE_CLIENT_HELLO:
	case TLS_HANDSHAKE_SERVER_HELLO:
	case TLS_HANDSHAKE_NEW_SESSION_TICKET:
	case TLS_HANDSHAKE_END_OF_EARLY_DATA:
	case TLS_HANDSHAKE_ENCRYPTED_EXTENSIONS:
	case TLS_HANDSHAKE_CERTIFICATE:
	case TLS_HANDSHAKE_CERTIFICATE_REQUEST:
	case TLS_HANDSHAKE_CERTIFICATE_VERIFY:
	case TLS_HANDSHAKE_FINISHED:
	case TLS_HANDSHAKE_KEY_UPDATE:
	case TLS_HANDSHAKE_MESSAGE_HASH:
		return true;

	default:
		return false;
	}
}

static bool check_handshake_type(TLS *tls, const enum tls_handshake_type type) {
	/*
	 * check a handshake type before version negotiation. at this point,
	 * client hello, server hello or version specific handshake type
	 * sent before version negotiation can come.
	 */
	switch (tls->state) {
	case TLS_STATE_HS_REINIT:
		if (type == TLS_HANDSHAKE_HELLO_REQUEST) return true;
		break;

	case TLS_STATE_HS_BEFORE_RECV_CHELLO:
		if (type == TLS_HANDSHAKE_CLIENT_HELLO) return true;
		break;

	case TLS_STATE_HS_BEFORE_RECV_SHELLO:
		if (type == TLS_HANDSHAKE_SERVER_HELLO) return true;
		break;

	default:
		break;
	}

	/*
	 * check a handshake type according to tls protocol version after
	 * version negotiation.
	 */
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		return check_handshake_type_up_to_tls12(type);

	case TLS_VER_TLS13:
		return check_handshake_type_tls13(type);

	default:
		/* Unknown version */
		break;
	}

	return false;
}

static void hash_update_for_read(TLS *tls, struct tls_hs_msg *msg) {
	switch(msg->type) {
	case TLS_HANDSHAKE_HELLO_REQUEST:
		break;

	case TLS_HANDSHAKE_CLIENT_HELLO:
	case TLS_HANDSHAKE_SERVER_HELLO:
		/* client hello and server hello delay the hash
		 * calculation since hash algorithm of finished message
		 * is determined after the server hello message. */
		TAILQ_INSERT_TAIL(tls->queue_handshake, msg, link);
		break;

	default:
		/* do hash update by handshake that stored in the
		 * queue. */
		tls_hs_update_hash(tls);
		TAILQ_INSERT_TAIL(tls->queue_handshake, msg, link);
		break;
	}
}

static void hash_update_for_write(TLS *tls, struct tls_hs_msg *msg) {
	switch(msg->type) {
	case TLS_HANDSHAKE_HELLO_REQUEST:
		break;

	default:
		TAILQ_INSERT_TAIL(tls->queue_handshake, msg, link);

		/* hold the hash update if current handshake message is
		 * client hello. otherwise, do hash update. */
		if (msg->type != TLS_HANDSHAKE_CLIENT_HELLO) {
			tls_hs_update_hash(tls);
		}
		break;
	}
}

static enum tls_handshake_type read_handshake_type(const uint8_t *buf) {
	return buf[0];
}

static void make_handshake_message(uint8_t *buf, const struct tls_hs_msg *msg) {
	buf[0] = msg->type;

	tls_util_write_3(&(buf[1]), msg->len);

	memcpy(&(buf[TLS_HANDSHAKE_HEADER_LENGTH]), &(msg->msg[0]), msg->len);
}

static void cleanup_session(TLS *tls) {
	tls_session_unrefer(tls->pending);
	tls->pending = NULL;
}

static bool start_handshake(TLS *tls) {
	bool ret = false;

	tls->certreq_used = false;
	tls->ccert_null   = false;

	if (! handshake_queue_init(tls)) {
		goto fin;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_REINIT) == false) {
		tls_hs_change_state(tls, TLS_STATE_HS_INIT);
	}

	switch (tls->entity) {
	case TLS_CONNECT_SERVER:
		if (! tls_hs_cs_server_handshake(tls)) {
			goto fin;
		}
		break;

	case TLS_CONNECT_CLIENT:
	default:
		if (! tls_hs_cs_client_handshake(tls)) {
			goto fin;
		}
		break;
	}
	tls_hs_change_state(tls, TLS_STATE_ESTABLISHED);

	/* enable estabilished session for re-session. */
	tls_session_enable(tls->pending);

	ret = true;

fin:
	;
	uint16_t version;
	version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		handshake_queue_free(tls);
		break;

	case TLS_VER_TLS13:
		/*
		 * return here to avoid cleaning up session information used for
		 * key update message.
		 */
		return ret;

	default:
		ret = false;
		break;
	}

	cleanup_session(tls);
	return ret;
}

static bool posthandshake_message_tls13(TLS *tls) {
	struct tls_hs_msg *msg;
	if ((msg = tls_handshake_read(tls)) == NULL) {
		TLS_DPRINTF("tls_handshake_read");
		return false;
	}

	/*
	 * RFC8446 4.  Handshake Protocol
	 *
	 *    Protocol messages MUST be sent in the order defined in Section 4.4.1
	 *    and shown in the diagrams in Section 2.  A peer which receives a
	 *    handshake message in an unexpected order MUST abort the handshake
	 *    with an "unexpected_message" alert.
	 *
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    Because TLS 1.3 forbids renegotiation, if a server has negotiated
	 *    TLS 1.3 and receives a ClientHello at any other time, it MUST
	 *    terminate the connection with an "unexpected_message" alert.
	 */
	switch (msg->type) {
	case TLS_HANDSHAKE_NEW_SESSION_TICKET:
		/*
		 * TODO: implement NetSessionTicket message. ignore these
		 * messages currently.
		 */
		break;

	case TLS_HANDSHAKE_KEY_UPDATE:
		/*
		 * RFC8446 4.6.3.  Key and Initialization Vector Update
		 *
		 *    The KeyUpdate handshake message is used to indicate that the sender
		 *    is updating its sending cryptographic keys.  This message can be sent
		 *    by either peer after it has sent a Finished message.  Implementations
		 *    that receive a KeyUpdate message prior to receiving a Finished
		 *    message MUST terminate the connection with an "unexpected_message"
		 *    alert.  After sending a KeyUpdate message, the sender SHALL send all
		 *    its traffic using the next generation of keys, computed as described
		 *    in Section 7.2.  Upon receiving a KeyUpdate, the receiver MUST update
		 *    its receiving keys.
		 */
		if (tls_hs_keyupdate_parse(tls, msg) == false) {
			return false;
		}

		/*
		 * 4.6.3.  Key and Initialization Vector Update
		 *
		 *    If the request_update field is set to "update_requested", then the
		 *    receiver MUST send a KeyUpdate of its own with request_update set to
		 *    "update_not_requested" prior to sending its next Application Data
		 *    record.  This mechanism allows either side to force an update to the
		 *    entire connection, but causes an implementation which receives
		 *    multiple KeyUpdates while it is silent to respond with a single
		 *    update.  Note that implementations may receive an arbitrary number of
		 *    messages between sending a KeyUpdate with request_update set to
		 *    "update_requested" and receiving the peer's KeyUpdate, because those
		 *    messages may already be in flight.  However, because send and receive
		 *    keys are derived from independent traffic secrets, retaining the
		 *    receive traffic secret does not threaten the forward secrecy of data
		 *    sent before the sender changed keys.
		 *
		 *    If implementations independently send their own KeyUpdates with
		 *    request_update set to "update_requested" and they cross in flight,
		 *    then each side will also send a response, with the result that each
		 *    side increments by two generations.
		 *
		 *    Both sender and receiver MUST encrypt their KeyUpdate messages with
		 *    the old keys.  Additionally, both sides MUST enforce that a KeyUpdate
		 *    with the old key is received before accepting any messages encrypted
		 *    with the new key.  Failure to do so may allow message truncation
		 *    attacks.
		 */
		if (tls->need_sending_keyupdate == true) {
			if (tls_hs_send_key_update(
				tls, TLS_KEYUPDATE_NOT_REQUESTED) == false) {
				return false;
			}

			tls->need_sending_keyupdate = false;
		}
		break;

	default:
		TLS_DPRINTF("receive forbidden handshake message");
		OK_set_error(ERR_ST_TLS_FORBIDDEN_HS_TYPE,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 11, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return false;
	}

	return true;
}

struct tls_extension *tls_extension_init(void) {
	struct tls_extension *ext;

	if ((ext = malloc(1 * sizeof(struct tls_extension))) == NULL) {
		TLS_DPRINTF("extension: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 12, NULL);
		return NULL;
	}
	ext->opaque = NULL;

	return ext;
}

void tls_extension_free(struct tls_extension *ext) {
	if (ext == NULL) {
		return;
	}

	free(ext->opaque);
	ext->opaque = NULL;

	free(ext);
}

struct tls_hs_interim_params *tls_hs_interim_params_init(void) {
	struct tls_hs_interim_params *params;

	if ((params = calloc(1, sizeof(struct tls_hs_interim_params))) == NULL) {
		TLS_DPRINTF("interim_params: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_CALLOC,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 13, NULL);
		return NULL;
	}
	TAILQ_INIT(&(params->head));
	TAILQ_INIT(&(params->share_head));

	return params;
}

void tls_hs_interim_params_free(struct tls_hs_interim_params *params) {
	struct tls_extension *ext;
	struct tls_hs_key_share *share;

	if (params == NULL) {
		return;
	}

	free(params->session);
	params->session = NULL;

	free(params->cmplist);
	params->cmplist = NULL;

	tls_cipher_list_free(params->cipher_suites);

	while (!TAILQ_EMPTY(&(params->head))) {
		ext = TAILQ_FIRST(&(params->head));
		TAILQ_REMOVE(&(params->head), ext, link);
		tls_extension_free(ext);
	}

	while (!TAILQ_EMPTY(&(params->share_head))) {
		share = TAILQ_FIRST(&(params->share_head));
		TAILQ_REMOVE(&(params->share_head), share, link);
		tls_hs_keyshare_free(share);
	}

	free(params);
}

void tls_hs_change_state(TLS *tls, enum tls_state state) {
	tls->state = state;

#ifdef TLS_DEBUG
	char name[30];
	switch(state) {
	case TLS_STATE_HS_INIT:
		strcpy(name, "INIT");
		break;

	case TLS_STATE_HS_SEND_HELLOREQ:
		strcpy(name, "SEND_HELLOREQ");
		break;

	case TLS_STATE_HS_RECV_HELLOREQ:
		strcpy(name, "RECV_HELLOREQ");
		break;

	case TLS_STATE_HS_BEFORE_SEND_CHELLO:
		strcpy(name, "SEND_CHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_SEND_CHELLO:
		strcpy(name, "SEND_CHELLO (AFTER)");
		break;

	case TLS_STATE_HS_BEFORE_RECV_SHELLO:
		strcpy(name, "RECV_SHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_SHELLO:
		strcpy(name, "RECV_SHELLO (AFTER)");
		break;

	case TLS_STATE_HS_BEFORE_SEND_2NDCHELLO:
		strcpy(name, "SEND_2NDCHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_SEND_2NDCHELLO:
		strcpy(name, "SEND_2NDCHELLO (AFTER)");
		break;

	case TLS_STATE_HS_BEFORE_RECV_2NDSHELLO:
		strcpy(name, "RECV_2NDSHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_2NDSHELLO:
		strcpy(name, "RECV_2NDSHELLO (AFTER)");
		break;

	case TLS_STATE_HS_AFTER_RECV_HRREQ:
		strcpy(name, "RECV_HRREQ (AFTER)");
		break;

	case TLS_STATE_HS_RECV_ENCEXT:
		strcpy(name, "RECV_ENCEXT");
		break;

	case TLS_STATE_HS_RECV_SCERT:
		strcpy(name, "RECV_SCERT");
		break;

	case TLS_STATE_HS_RECV_SKEYEXC:
		strcpy(name, "RECV_SKEYEXC");
		break;

	case TLS_STATE_HS_RECV_SCERTVFY:
		strcpy(name, "RECV_SCERTVFY");
		break;

	case TLS_STATE_HS_RECV_CERTREQ:
		strcpy(name, "RECV_CERTREQ");
		break;

	case TLS_STATE_HS_RECV_SHELLODONE:
		strcpy(name, "RECV_SHELLODONE");
		break;

	case TLS_STATE_HS_SEND_CCERT:
		strcpy(name, "SEND_CCERT");
		break;

	case TLS_STATE_HS_SEND_CKEYEXC:
		strcpy(name, "SEND_CKEYEXC");
		break;

	case TLS_STATE_HS_SEND_CCERTVFY:
		strcpy(name, "SEND_CCERTVFY");
		break;

	case TLS_STATE_HS_BEFORE_RECV_CHELLO:
		strcpy(name, "RECV_CHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_CHELLO:
		strcpy(name, "RECV_CHELLO (AFTER)");
		break;

	case TLS_STATE_HS_BEFORE_RECV_2NDCHELLO:
		strcpy(name, "RECV_2NDCHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_2NDCHELLO:
		strcpy(name, "RECV_2NDCHELLO (AFTER)");
		break;

	case TLS_STATE_HS_SEND_SHELLO:
		strcpy(name, "SEND_SHELLO");
		break;

	case TLS_STATE_HS_BEFORE_SEND_HRREQ:
		strcpy(name, "SEND_HRREQ (BEFORE)");
		break;

	case TLS_STATE_HS_SEND_ENCEXT:
		strcpy(name, "SEND_ENCEXT");
		break;

	case TLS_STATE_HS_SEND_SCERT:
		strcpy(name, "SEND_SCERT");
		break;

	case TLS_STATE_HS_SEND_SCERTVFY:
		strcpy(name, "SEND_SCERTVFY");
		break;

	case TLS_STATE_HS_SEND_SKEYEXC:
		strcpy(name, "SEND_SKEYEXC");
		break;

	case TLS_STATE_HS_SEND_CERTREQ:
		strcpy(name, "SEND_CERTREQ");
		break;

	case TLS_STATE_HS_SEND_SHELLODONE:
		strcpy(name, "SEND_SHELLODONE");
		break;

	case TLS_STATE_HS_RECV_CCERT:
		strcpy(name, "RECV_CCERT");
		break;

	case TLS_STATE_HS_RECV_CKEYEXC:
		strcpy(name, "RECV_CKEYEXC");
		break;

	case TLS_STATE_HS_RECV_CCERTVFY:
		strcpy(name, "RECV_CCERTVFY");
		break;

	case TLS_STATE_HS_RECV_FINISH:
		strcpy(name, "RECV_FINISH");
		break;

	case TLS_STATE_HS_SEND_FINISH:
		strcpy(name, "SEND_FINISH");
		break;

	case TLS_STATE_HS_BEFORE_FINISH:
		strcpy(name, "BEFORE_FINISH");
		break;

	case TLS_STATE_CCS_SEND:
		strcpy(name, "CCS_SEND");
		break;

	case TLS_STATE_CCS_RECV:
		strcpy(name, "CCS_RECV");
		break;

	case TLS_STATE_ESTABLISHED:
		strcpy(name, "ESTABLISHED");
		break;

	case TLS_STATE_CLOSED:
		strcpy(name, "CLOSED");
		break;

	default:
		strcpy(name, "unknown handshake type");
		break;
	}

	TLS_DPRINTF("handshake: state => %s", name);
#endif
}

bool tls_hs_check_state(TLS *tls, enum tls_state state) {
	if (tls->state == state) {
		return true;
	}
	return false;
}

void tls_hs_update_hash(TLS *tls) {
	struct tls_hs_msg *msg;

	while(!(TAILQ_EMPTY(tls->queue_handshake))) {
		msg = TAILQ_FIRST(tls->queue_handshake);

		uint32_t len = msg->len + TLS_HANDSHAKE_HEADER_LENGTH;
		uint8_t  buf[len];

		make_handshake_message(buf, msg);

		/* for finished message and certificate verify message,
		 * calcucate the candidates that may be used
		 * beforehand. */
		tls_hs_hash_update(tls, buf, len);

		TAILQ_REMOVE(tls->queue_handshake, msg, link);
		tls_hs_msg_free(msg);
		msg = NULL;
	}
}

struct tls_hs_msg * tls_handshake_read(TLS *tls) {
	struct tls_hs_msg *msg;

	while (true) {
		uint8_t head[TLS_HANDSHAKE_HEADER_LENGTH];
		if (tls_record_read(TLS_CTYPE_HANDSHAKE, tls, head,
				    TLS_HANDSHAKE_HEADER_LENGTH) < 0) {
			TLS_DPRINTF("handshake: tls_record_read");
			OK_set_error(ERR_ST_TLS_TLS_RECORD_READ,
				     ERR_LC_TLS2, ERR_PT_TLS_HS + 3, NULL);
			/* alert for tls_record_read is done by that
			 * internal. */
			return NULL;
		}

		enum tls_handshake_type type = read_handshake_type(head);
		if (check_handshake_type(tls, type) == false) {
			TLS_DPRINTF("handshake: handshake type");
			OK_set_error(ERR_ST_TLS_UNKNOWN_HS_TYPE,
				     ERR_LC_TLS2, ERR_PT_TLS_HS + 4, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
			return NULL;
		}

		/* it is not possible to check length of handshake here
		 * because there is not criterion. */
		uint32_t len = tls_util_read_3(&(head[1]));

		if ((msg = malloc (1 * sizeof (struct tls_hs_msg))) == NULL) {
			TLS_DPRINTF("handshake: malloc: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_MALLOC,
				     ERR_LC_TLS2, ERR_PT_TLS_HS + 5, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return NULL;
		}

		msg->type = type;
		msg->len  = len;

		dump_handshake_type(msg->type);
		TLS_DPRINTF("handshake: read: len = %d", msg->len);

		if ((msg->msg = malloc(1 * msg->len)) == NULL) {
			TLS_DPRINTF("handshake: malloc: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_MALLOC,
				     ERR_LC_TLS2, ERR_PT_TLS_HS + 6, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			tls_hs_msg_free(msg);
			return NULL;
		}
		msg->max = msg->len;

		if (tls_record_read(TLS_CTYPE_HANDSHAKE,
				    tls, &(msg->msg[0]), msg->len) < 0) {
			/* alert for tls_record_read is done by that
			 * internal. */
			TLS_DPRINTF("handshake: tls_record_read");
			OK_set_error(ERR_ST_TLS_TLS_RECORD_READ,
				     ERR_LC_TLS2, ERR_PT_TLS_HS + 7, NULL);
			tls_hs_msg_free(msg);
			return NULL;
		}

		if ((tls->entity == TLS_CONNECT_CLIENT) &&
		    (tls->state != TLS_STATE_HS_REINIT) &&
		    (type == TLS_HANDSHAKE_HELLO_REQUEST)) {
			/* RFC 5246 says.
			 *
			 *   The one message that is not bound by these
			 *   ordering rules is the HelloRequest message,
			 *   which can be sent at any time, but which
			 *   SHOULD be ignored by the client if it
			 *   arrives in the middle of a handshake. */
			tls_hs_msg_free(msg);
			msg = NULL;
		} else {
			break;
		}
	}

	hash_update_for_read(tls, msg);

	return msg;
}

bool tls_handshake_write(TLS *tls, struct tls_hs_msg * msg) {
	uint32_t len = TLS_HANDSHAKE_HEADER_LENGTH + msg->len;
	uint8_t  buf[len];

	make_handshake_message(buf, msg);

	ssize_t n UNUSED;
	if ((n = tls_record_write(TLS_CTYPE_HANDSHAKE, tls, buf, len)) < 0) {
		TLS_DPRINTF("handshake: tls_record_write (%zd)", n);
		OK_set_error(ERR_ST_TLS_TLS_RECORD_WRITE,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 8, NULL);
		return false;
	}

	dump_handshake_type(msg->type);
	TLS_DPRINTF("handshake: write: len = %d, whole = %d", msg->len, len);

	hash_update_for_write(tls, msg);

	return true;
}

bool tls_hs_posthandshake_message(TLS *tls) {
	uint16_t version;
	version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 14, NULL);
		return false;

	case TLS_VER_TLS13:
		return posthandshake_message_tls13(tls);

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 15, NULL);
		return false;
	}
}

bool tls_hs_send_key_update(TLS *tls, enum tls_keyupdate_request req) {
	struct tls_hs_msg *keyupdate_msg;
	if ((keyupdate_msg = tls_hs_keyupdate_compose(tls, req))
	    == NULL) {
		return false;
	}

	if (tls_handshake_write(tls, keyupdate_msg) == false) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		tls_hs_msg_free(keyupdate_msg);
		return false;
	}

	if (tls_key_derive_application_traffic_secret_n(
		tls, &(tls->active_write)) == false) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	if (tls_key_make_traffic_key(tls, &(tls->active_write)) == false) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	return true;
}

void tls_handshake_free(TLS *tls) {
	if (tls == NULL) {
		return;
	}

	if (tls->queue_handshake != NULL) {
		handshake_queue_free(tls);
	}
}

int TLS_handshake(TLS *tls) {
	TLS_DPRINTF("handshake: start");

	if (tls == NULL) {
		TLS_DPRINTF("handshake: tls: null");
		OK_set_error(ERR_ST_TLS_TLS_NULL,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 9, NULL);
		goto err;
	}

	/* check re-negotiation or not. */
	if (tls->handshake_over == true) {
		TLS_DPRINTF("handshake: re-negotiation is not supported");
		OK_set_error(ERR_ST_TLS_NO_RENEGOTIATION,
			     ERR_LC_TLS2, ERR_PT_TLS_HS + 10, NULL);
		/* do not jump in `err' label since do not intend to
		 * close connection. */
		return -1;
	}

	if (start_handshake(tls) == false) {
		goto err;
	}
	tls->handshake_over = true;

	TLS_DPRINTF("handshake: finish (ok)");
	return 0;

err:
	tls_hs_change_state(tls, TLS_STATE_CLOSED);
	TLS_DPRINTF("handshake: finish (ng)");
	return -1;
}
