/*
 * Copyright (c) 2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <aicrypto/ok_ssl.h>

/* for Cert */
#include <aicrypto/ok_x509.h>

#include "tls.h"

/** @see tls_opt.c */
extern int sel_timeout __attribute__ ((deprecated));


SSL *SSL_new(void)  {
	return (SSL *) TLS_new ();
}

SSL *SSL_dup(SSL *ssl) {
	return (SSL *) TLS_dup((TLS *) ssl);
}

void SSL_free(SSL *ssl) {
	TLS_free((TLS *) ssl);
}

int SSL_get_error(SSL *ssl) {
	return TLS_get_error((TLS *) ssl);
}

int SSL_handshake(SSL *ssl) {
	return TLS_handshake(((TLS *)ssl));
}

SSL * SSL_socket(int af,int type,int protocol) {
	return (SSL *) TLS_socket(af, type, protocol);
}

int SSL_shutdown(SSL *ssl, int how) {
	return TLS_shutdown((TLS *) ssl, how);
}

int SSL_close(SSL *ssl) {
	return TLS_close((TLS *) ssl);
}

int SSL_getsockopt(SSL *ssl, int level, int optname, void  *optval, int *optlen) {
	return TLS_getsockopt((TLS *) ssl, level, optname,
			      optval, (socklen_t *) optlen);
}

int SSL_setsockopt(SSL *ssl, int level, int optname, const void * optval, int optlen) {
	return TLS_setsockopt((TLS *) ssl, level, optname,
			      optval, (socklen_t)optlen);
}

void SSL_set_timeout(int msec) {
	sel_timeout = msec;
}

int SSL_get_timeout(void) {
	return sel_timeout;
}

int SSL_bind(SSL *ssl, const struct sockaddr *name, int namelen) {
	return TLS_bind((TLS *) ssl, name, namelen);
}

int SSL_listen(SSL *ssl, int backlog) {
	return TLS_listen((TLS *) ssl, backlog);
}

SSL *SSL_accept(SSL *ssl, struct sockaddr *addr, int *addrlen) {
	return (SSL *) TLS_accept((TLS *) ssl, addr, (socklen_t *) addrlen);
}

int SSL_connect(SSL *ssl, const struct sockaddr *name, int namelen) {
	return TLS_connect((TLS *) ssl, name, (socklen_t) namelen);
}

void SSL_setopt(SSL *ssl, int flag) {
	bool on = true;

	switch (flag) {
	case SSL_OPT_IMMEDIATE:
		TLS_opt_set((TLS *) ssl, TLS_OPT_IMMEDIATE_HANDSHAKE, &on);
		break;

	case SSL_OPT_HS_SEPARATE:
		TLS_DPRINTF("compat: unsupport: SSL_OPT_HS_SEPARATE");
		OK_set_error(ERR_ST_TLS_UNSUPPORT_SSL_OPT_HS_SEPARATE,
			     ERR_LC_TLS1, ERR_PT_TLS_COMPAT + 0, NULL);
		break;

	case SSL_OPT_CERTREQ:
		TLS_opt_set((TLS *) ssl, TLS_OPT_USE_CLIENT_CERT_AUTH, &on);
		break;

	case SSL_OPT_CERTREQOPT_F:
		TLS_DPRINTF("compat: unsupport: SSL_OPT_CERTREQOPT_F");
		OK_set_error(ERR_ST_TLS_UNSUPPORT_SSL_OPT_CERTREQOPT_F,
			     ERR_LC_TLS1, ERR_PT_TLS_COMPAT + 1, NULL);
		break;

	case SSL_OPT_CERTREQOPT:
		TLS_opt_set((TLS *) ssl, TLS_OPT_MAY_USE_CLIENT_CERT_AUTH, &on);
		break;

	case SSL_OPT_KEEPWBUF:
		TLS_DPRINTF("compat: unsupport: SSL_OPT_KEEPWBUF");
		OK_set_error(ERR_ST_TLS_UNSUPPORT_SSL_OPT_KEEPWBUF,
			     ERR_LC_TLS1, ERR_PT_TLS_COMPAT + 2, NULL);
		break;

	default:
		break;
	}
}

int SSL_getopt(SSL *ssl) {
	int opt = 0;

	if (TLS_opt_get((TLS *) ssl, TLS_OPT_IMMEDIATE_HANDSHAKE) > 0) {
		opt |= SSL_OPT_IMMEDIATE;
	}

	/* SSL_OPT_HS_SEPRATE do not be supported in the tls implementation. */

	if (TLS_opt_get((TLS *) ssl, TLS_OPT_USE_CLIENT_CERT_AUTH) > 0) {
		opt |= SSL_OPT_CERTREQ;
	}

	if (TLS_opt_get((TLS *) ssl, TLS_OPT_MAY_USE_CLIENT_CERT_AUTH) > 0) {
		opt |= SSL_OPT_CERTREQOPT_F;
	}

	/* SSL_OPT_KEEPWBUF do not be supported in the tls implementation. */

	return opt;
}

int SSL_getpeername(SSL *ssl, struct sockaddr *name, int *namelen) {
	return TLS_getpeername((TLS *) ssl, name, (socklen_t *) namelen);
}

int SSL_getsockname(SSL *ssl, struct sockaddr *name, int *namelen) {
	return TLS_getsockname((TLS *) ssl, name, (socklen_t *) namelen);
}

int SSL_write(SSL *ssl, void *buf, size_t nbyte) {
	return TLS_write((TLS *) ssl, buf, nbyte);
}

int SSL_read(SSL *ssl, void *buf, size_t nbyte) {
	return TLS_read((TLS *) ssl, buf, nbyte);
}

int SSL_gets(SSL *ssl, char *buf, size_t nbyte) {
	return TLS_gets((TLS *) ssl, buf, nbyte);
}

int SSL_set_server_p12(SSL *ssl,char *fname,char *passwd) {
	return TLS_set_serverkey_file((TLS *) ssl, fname, passwd);
}

int SSL_set_serverkey_file(SSL *ssl,char *fname,char *passwd) {
	return TLS_set_serverkey_file((TLS *) ssl, fname, passwd);
}

int SSL_set_serverkey_p12(SSL *ssl,PKCS12 *p12) {
	return TLS_set_serverkey_p12((TLS *) ssl, p12);
}

int SSL_set_serverkey_id(SSL *ssl,char *id) {
	return TLS_set_serverkey_id((TLS *) ssl, id);
}

int SSL_set_client_p12(SSL *ssl,char *fname,char *passwd) {
	return TLS_set_clientkey_file((TLS *) ssl, fname, passwd);
}

int SSL_set_clientkey_file(SSL *ssl,char *fname,char *passwd) {
	return TLS_set_clientkey_file((TLS *) ssl, fname, passwd);
}

int SSL_set_clientkey_p12(SSL *ssl,PKCS12 *p12) {
	return TLS_set_clientkey_p12((TLS *) ssl, p12);
}

int SSL_set_clientkey_id(SSL *ssl,char *id) {
	return TLS_set_clientkey_id((TLS *) ssl, id);
}

Cert *SSL_get_server_cert(SSL *ssl) {
	return TLS_get_server_cert((TLS *)ssl);
}

Cert *SSL_get_client_cert(SSL *ssl) {
	return TLS_get_client_cert((TLS *)ssl);
}

int SSL_got_certreq(SSL *ssl) {
	if (((TLS *) ssl)->is_ccert_auth == true) {
		return 1;
	}
	return 0;
}

void SSL_set_list_max(SSL *ssl UNUSED, int num) {
	/* in tls implementation, number of session do not bind the tls
	 * (ssl) structure. */
	TLS_session_set_list_size(num);
}

int SSL_set_vfytype(SSL *ssl, int type) {
	if (ssl == NULL) {
		TLS_DPRINTF("compat: ssl: null");
		OK_set_error(ERR_ST_TLS_TLS_NULL,
			     ERR_LC_TLS1, ERR_PT_TLS_COMPAT + 3, NULL);
		return -1;
	}

	TLS_opt_set((TLS *)ssl, TLS_OPT_CERT_VERIFY_TYPE, &type);

	return 0;
}

int SSL_set_vfydepth(SSL *ssl, int depth) {
	if (ssl == NULL) {
		TLS_DPRINTF("compat: ssl: null");
		OK_set_error(ERR_ST_TLS_TLS_NULL,
			     ERR_LC_TLS1, ERR_PT_TLS_COMPAT + 4, NULL);
		return -1;
	}

	TLS_opt_set((TLS *)ssl, TLS_OPT_CERT_VERIFY_DEPTH, &depth);

	return 0;
}

int SSL_set_store(SSL *ssl,char *path) {
	if (TLS_stm_set((TLS *) ssl, path) == true) {
		return 0;
	}

	return -1;
}

int SSL_get_fd(SSL *ssl) {
	return TLS_get_fd((TLS *) ssl);
}

int SSL_set_fd(SSL *ssl, int sockfd) {
	return (((TLS *) ssl)->sockfd = sockfd);
}

Cert *SSL_get_peer_certificate(SSL *ssl) {
	return TLS_get_peer_certificate((TLS *) ssl);
}
