/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_session.h"
#include "tls_cipher.h"
#include "tls_handshake.h"
#include "tls_alert.h"
#include "tls_cert.h"
#include <string.h>

/**
 * check whether the specified cipher suite is available in the
 * negotiated TLS version (in this case, it is 1.2).
 *
 * TODO: i think this function name should be renamed because this
 * function name is hard to understand (why return value is bool though
 * function name has a set).
 */
static bool tls_cipher_set_tls12(TLS *tls, const uint16_t suite);

static bool tls_cipher_set_tls13(TLS *tls, const uint16_t suite);

/**
 * merge cipher suites specific to a version into cipher suites list
 * without duplication. cipher suites in the list are unique.
 */
static uint16_t merge_cipher_suite_list(enum tls_cipher_suite *suites,
				 uint16_t suites_len,
				 uint16_t offset,
				 enum tls_cipher_suite *version_suites,
				 uint16_t version_suites_len);

/**
 * Set cipher suite already checked to TLS structure.
 */
static bool cipher_suite_decided(TLS *tls, enum tls_cipher_suite cipher,
				 struct tls_cert_info *cinfo);

/**
 * default cipher suite that is used by TLS 1.1. this is used only for test
 * currently.
 */
UNUSED
static enum tls_cipher_suite tls_cipher_list_tls11[] = {
	TLS_RSA_WITH_AES_256_CBC_SHA,
	TLS_RSA_WITH_AES_128_CBC_SHA,
	TLS_RSA_WITH_3DES_EDE_CBC_SHA,
	TLS_RSA_WITH_DES_CBC_SHA,
#ifdef HAVE_ARC4
	TLS_RSA_WITH_RC4_128_SHA,
	TLS_RSA_WITH_RC4_128_MD5,
#endif
#ifdef TLS_DEBUG
	TLS_RSA_WITH_NULL_SHA,
	TLS_RSA_WITH_NULL_MD5,
#endif
};

/**
 * default cipher suite that is used by TLS 1.2.
 */
static enum tls_cipher_suite tls_cipher_list_tls12[] = {
	TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
	TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
	TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
	TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
	TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
	TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
	TLS_RSA_WITH_AES_256_GCM_SHA384,
	TLS_RSA_WITH_AES_128_GCM_SHA256,
	TLS_RSA_WITH_AES_256_CBC_SHA256,
	TLS_RSA_WITH_AES_128_CBC_SHA256,
	TLS_RSA_WITH_AES_256_CBC_SHA,
	TLS_RSA_WITH_AES_128_CBC_SHA,
	TLS_RSA_WITH_3DES_EDE_CBC_SHA,
#if TLS_OBSOLETE_ALGO
#ifdef HAVE_ARC4
	TLS_RSA_WITH_RC4_128_SHA,
	TLS_RSA_WITH_RC4_128_MD5,
#endif
#endif
#ifdef TLS_DEBUG
	TLS_RSA_WITH_NULL_SHA256,
	TLS_RSA_WITH_NULL_SHA,
	TLS_RSA_WITH_NULL_MD5,
#endif
};

/**
 * default cipher suite that is used by TLS 1.3.
 */
static enum tls_cipher_suite tls_cipher_list_tls13[] = {
#ifdef TLS_DEBUG_WITHOUT_CIPHER
	TLS_NULL_WITH_NULL_NULL,
#endif
	TLS_AES_256_GCM_SHA384,
	TLS_AES_128_GCM_SHA256,
	TLS_CHACHA20_POLY1305_SHA256,
};

/* the definition of ciperh suite (RFC 5246 Appendix C.)
 *
 *                           Key      IV   Block
 *   Cipher        Type    Material  Size  Size
 *   ------------  ------  --------  ----  -----
 *   NULL          Stream      0       0    N/A
 *   RC4_128       Stream     16       0    N/A
 *   3DES_EDE_CBC  Block      24       8      8
 *   AES_128_CBC   Block      16      16     16
 *   AES_256_CBC   Block      32      16     16
 *
 *
 *   MAC       Algorithm    mac_length  mac_key_length
 *   --------  -----------  ----------  --------------
 *   NULL      N/A              0             0
 *   MD5       HMAC-MD5        16            16
 *   SHA       HMAC-SHA1       20            20
 *   SHA256    HMAC-SHA256     32            32
 *
 *   Type
 *      Indicates whether this is a stream cipher or a block cipher
 *      running in CBC mode.
 *
 *   Key Material
 *      The number of bytes from the key_block that are used for
 *      generating the write keys.
 *
 *   IV Size
 *      The amount of data needed to be generated for the initialization
 *      vector.  Zero for stream ciphers; equal to the block size for
 *      block ciphers (this is equal to
 *      SecurityParameters.record_iv_length).
 *
 *   Block Size
 *      The amount of data a block cipher enciphers in one chunk; a
 *      block cipher running in CBC mode can only encrypt an even
 *      multiple of its block size.
 */

/* following is list of stream cipher. */

/** stream cipher: NULL cipher with MD5. */
static const struct tls_cipher_param tls_cipher_null_md5 = {
	.cipher_algorithm = TLS_BULK_CIPHER_NULL,
	.cipher_type      = TLS_CIPHER_TYPE_STREAM,
	.mac_algorithm    = TLS_MAC_HMAC_MD5,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 0,
	.block_length     = 0,
	.fixed_iv_length  = 0,
	.record_iv_length = 0,
	.mac_length       = 16,
	.mac_key_length   = 16
};

/** stream cipher: NULL cipher with SHA. */
static const struct tls_cipher_param tls_cipher_null_sha = {
	.cipher_algorithm = TLS_BULK_CIPHER_NULL,
	.cipher_type      = TLS_CIPHER_TYPE_STREAM,
	.mac_algorithm    = TLS_MAC_HMAC_SHA1,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 0,
	.block_length     = 0,
	.fixed_iv_length  = 0,
	.record_iv_length = 0,
	.mac_length       = 20,
	.mac_key_length   = 20
};

/** stream cipher: NULL cipher with SHA256. */
static const struct tls_cipher_param tls_cipher_null_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_NULL,
	.cipher_type      = TLS_CIPHER_TYPE_STREAM,
	.mac_algorithm    = TLS_MAC_HMAC_SHA256,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 0,
	.block_length     = 0,
	.fixed_iv_length  = 0,
	.record_iv_length = 0,
	.mac_length       = 32,
	.mac_key_length   = 32
};

#ifdef HAVE_ARC4
/** stream cipher: RC4 128 cipher with MD5. */
static const struct tls_cipher_param tls_cipher_rc4_128_md5 = {
	.cipher_algorithm = TLS_BULK_CIPHER_RC4,
	.cipher_type      = TLS_CIPHER_TYPE_STREAM,
	.mac_algorithm    = TLS_MAC_HMAC_MD5,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 16,
	.block_length     = 0,
	.fixed_iv_length  = 0,
	.record_iv_length = 0,
	.mac_length       = 16,
	.mac_key_length   = 16
};

/** stream cipher: RC4 128 cipher with SHA. */
static const struct tls_cipher_param tls_cipher_rc4_128_sha = {
	.cipher_algorithm = TLS_BULK_CIPHER_RC4,
	.cipher_type      = TLS_CIPHER_TYPE_STREAM,
	.mac_algorithm    = TLS_MAC_HMAC_SHA1,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 16,
	.block_length     = 0,
	.fixed_iv_length  = 0,
	.record_iv_length = 0,
	.mac_length       = 20,
	.mac_key_length   = 20
};
#endif	/* HAVE_ARC4 */

/* following is list of CBC block cipher.
 *
 * in the CBC block cipher, record_iv_length must be same as
 * block_length. */

/** block cipher: 3DES EDE CBC cipher with SHA. */
static const struct tls_cipher_param tls_cipher_3des_ede_cbc_sha = {
	.cipher_algorithm = TLS_BULK_CIPHER_3DES,
	.cipher_type      = TLS_CIPHER_TYPE_BLOCK,
	.mac_algorithm    = TLS_MAC_HMAC_SHA1,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 24,
	.block_length     = 8,
	.fixed_iv_length  = 0,
	.record_iv_length = 8,
	.mac_length       = 20,
	.mac_key_length   = 20
};

/** block cipher: AES 128 CBC cipher with SHA. */
static const struct tls_cipher_param tls_cipher_aes_128_cbc_sha = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_BLOCK,
	.mac_algorithm    = TLS_MAC_HMAC_SHA1,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 16,
	.block_length     = 16,
	.fixed_iv_length  = 0,
	.record_iv_length = 16,
	.mac_length       = 20,
	.mac_key_length   = 20
};

/** block cipher: AES 256 CBC cipher with SHA. */
static const struct tls_cipher_param tls_cipher_aes_256_cbc_sha = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_BLOCK,
	.mac_algorithm    = TLS_MAC_HMAC_SHA1,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 32,
	.block_length     = 16,
	.fixed_iv_length  = 0,
	.record_iv_length = 16,
	.mac_length       = 20,
	.mac_key_length   = 20
};

/** block cipher: AES 128 CBC cipher with SHA256. */
static const struct tls_cipher_param tls_cipher_aes_128_cbc_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_BLOCK,
	.mac_algorithm    = TLS_MAC_HMAC_SHA256,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 16,
	.block_length     = 16,
	.fixed_iv_length  = 0,
	.record_iv_length = 16,
	.mac_length       = 32,
	.mac_key_length   = 32,
};

/** block cipher: AES 256 CBC cipher with SHA256. */
static const struct tls_cipher_param tls_cipher_aes_256_cbc_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_BLOCK,
	.mac_algorithm    = TLS_MAC_HMAC_SHA256,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 32,
	.block_length     = 16,
	.fixed_iv_length  = 0,
	.record_iv_length = 16,
	.mac_length       = 32,
	.mac_key_length   = 32,
};

/* following is list of GCM block cipher.
 */

/** AEAD cipher: AES 128 GCM cipher with SHA256. */
static const struct tls_cipher_param tls_cipher_aes_128_gcm_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_AEAD,
	.mac_algorithm    = TLS_MAC_NULL,
	.prf_algorithm    = TLS_PRF_SHA256,
	.verify_length    = 12,
	.enc_key_length   = 16,
	.block_length     = 16,
	.fixed_iv_length  = 4,
	.record_iv_length = 8,
	.mac_length       = 0,
	.mac_key_length   = 0,
};

/** AEAD cipher: AES 256 GCM cipher with SHA384. */
static const struct tls_cipher_param tls_cipher_aes_256_gcm_sha384 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_AEAD,
	.mac_algorithm    = TLS_MAC_NULL,
	.prf_algorithm    = TLS_PRF_SHA384,
	.verify_length    = 12,
	.enc_key_length   = 32,
	.block_length     = 16,
	.fixed_iv_length  = 4,
	.record_iv_length = 8,
	.mac_length       = 0,
	.mac_key_length   = 0,
};

/** AEAD cipher: AES 128 GCM cipher with HKDF SHA256. */
static const struct tls_cipher_param tls_cipher_aes_128_gcm_hkdf_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_AEAD,
	.mac_algorithm    = TLS_MAC_NULL,
	.prf_algorithm    = TLS_HKDF_SHA256,
	.verify_length    = 32,
	.enc_key_length   = 16,
	.block_length     = 16,
	.fixed_iv_length  = 4,
	.record_iv_length = 8,
	.mac_length       = 0,
	.mac_key_length   = 0,
};

/** AEAD cipher: AES 256 GCM cipher with HKDF SHA384. */
static const struct tls_cipher_param tls_cipher_aes_256_gcm_hkdf_sha384 = {
	.cipher_algorithm = TLS_BULK_CIPHER_AES,
	.cipher_type      = TLS_CIPHER_TYPE_AEAD,
	.mac_algorithm    = TLS_MAC_NULL,
	.prf_algorithm    = TLS_HKDF_SHA384,
	.verify_length    = 48,
	.enc_key_length   = 32,
	.block_length     = 16,
	.fixed_iv_length  = 4,
	.record_iv_length = 8,
	.mac_length       = 0,
	.mac_key_length   = 0,
};

/** AEAD cipher: CHACHA20 POLY1305 cipher with HKDF SHA256 */
static const struct tls_cipher_param tls_cipher_chacha20_poly1305_hkdf_sha256 = {
	.cipher_algorithm = TLS_BULK_CIPHER_CHACHA20,
	.cipher_type      = TLS_CIPHER_TYPE_AEAD,
	.mac_algorithm    = TLS_MAC_NULL,
	.prf_algorithm    = TLS_HKDF_SHA256,
	.verify_length    = 32,
	.enc_key_length   = 32,
	.block_length     = 0,
	.fixed_iv_length  = 4,
	.record_iv_length = 8,
	.mac_length       = 0,
	.mac_key_length   = 0,
};

/* if TLS_DEBUG is off, suite is not used. so, add attribute UNUSED. */
void tls_cipher_suite_dump(enum tls_cipher_suite suite UNUSED) {
#ifdef TLS_DEBUG
	char name[50];

	switch(suite) {
	case TLS_NULL_WITH_NULL_NULL:
		strcpy(name, "TLS_NULL_WITH_NULL_NULL");
		break;

	case TLS_RSA_WITH_NULL_MD5:
		strcpy(name, "RSA_WITH_NULL_MD5");
		break;

	case TLS_RSA_WITH_NULL_SHA:
		strcpy(name, "RSA_WITH_NULL_SHA");
		break;

	case TLS_RSA_WITH_NULL_SHA256:
		strcpy(name, "RSA_WITH_NULL_SHA256");
		break;

	case TLS_RSA_WITH_RC4_128_MD5:
		strcpy(name, "RSA_WITH_RC4_128_MD5");
		break;

	case TLS_RSA_WITH_RC4_128_SHA:
		strcpy(name, "RSA_WITH_RC4_128_SHA");
		break;

	case TLS_RSA_WITH_3DES_EDE_CBC_SHA:
		strcpy(name, "RSA_WITH_3DES_EDE_CBC_SHA");
		break;

	case TLS_RSA_WITH_AES_128_CBC_SHA:
		strcpy(name, "RSA_WITH_AES_128_CBC_SHA");
		break;

	case TLS_RSA_WITH_AES_256_CBC_SHA:
		strcpy(name, "RSA_WITH_AES_256_CBC_SHA");
		break;

	case TLS_RSA_WITH_AES_128_CBC_SHA256:
		strcpy(name, "RSA_WITH_AES_128_CBC_SHA256");
		break;

	case TLS_RSA_WITH_AES_256_CBC_SHA256:
		strcpy(name, "RSA_WITH_AES_256_CBC_SHA256");
		break;

	case TLS_RSA_WITH_AES_128_GCM_SHA256:
		strcpy(name, "RSA_WITH_AES_128_GCM_SHA256");
		break;

	case TLS_RSA_WITH_AES_256_GCM_SHA384:
		strcpy(name, "RSA_WITH_AES_256_GCM_SHA384");
		break;

	case TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA:
		strcpy(name, "ECDHE_RSA_WITH_AES_128_CBC_SHA");
		break;

	case TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA:
		strcpy(name, "ECDHE_RSA_WITH_AES_256_CBC_SHA");
		break;

	case TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256:
		strcpy(name, "ECDHE_RSA_WITH_AES_128_GCM_SHA256");
		break;

	case TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384:
		strcpy(name, "ECDHE_RSA_WITH_AES_256_GCM_SHA384");
		break;

	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA:
		strcpy(name, "ECDH_RSA_WITH_AES_128_CBC_SHA");
		break;

	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256:
		strcpy(name, "ECDH_RSA_WITH_AES_128_CBC_SHA256");
		break;

	case TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256:
		strcpy(name, "ECDH_RSA_WITH_AES_128_GCM_SHA256");
		break;

	case TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384:
		strcpy(name, "ECDH_RSA_WITH_AES_256_GCM_SHA384");
		break;

	case TLS_AES_128_GCM_SHA256:
		strcpy(name, "AES_128_GCM_SHA256");
		break;

	case TLS_AES_256_GCM_SHA384:
		strcpy(name, "AES_256_GCM_SHA384");
		break;

	case TLS_CHACHA20_POLY1305_SHA256:
		strcpy(name, "CHACHA20_POLY1305_SHA256");
		break;

	case TLS_AES_128_CCM_SHA256:
		strcpy(name, "AES_128_CCM_SHA256");
		break;

	case TLS_AES_128_CCM_8_SHA256:
		strcpy(name, "AES_128_CCM_8_SHA256");
		break;

	default:
		strcpy(name, "unknown type");
		break;
	}

	TLS_DPRINTF("suite: %04x %s", suite, name);
#endif
}

#ifdef TLS_DEBUG
void tls_cipher_keyexc_method_dump(enum tls_keyexchange_method method)
{
	char name[50];

	switch(method) {
	case TLS_KXC_NULL:
		strcpy(name, "(NULL)");
		break;

	case TLS_KXC_DHE_DSS:
		strcpy(name, "DHE_DSS");
		break;

	case TLS_KXC_DHE_RSA:
		strcpy(name, "DHE_RSA");
		break;

	case TLS_KXC_DH_anon:
		strcpy(name, "DH_anon");
		break;

	case TLS_KXC_RSA:
		strcpy(name, "RSA");
		break;

	case TLS_KXC_DH_DSS:
		strcpy(name, "DH_DSS");
		break;

	case TLS_KXC_DH_RSA:
		strcpy(name, "DH_RSA");
		break;

	case TLS_KXC_ECDH_ECDSA:
		strcpy(name, "ECDH_ECDSA");
		break;

	case TLS_KXC_ECDHE_ECDSA:
		strcpy(name, "ECDHE_ECDSA");
		break;

	case TLS_KXC_ECDH_RSA:
		strcpy(name, "ECDH_RSA");
		break;

	case TLS_KXC_ECDHE_RSA:
		strcpy(name, "ECDHE_RSA");
		break;

	case TLS_KXC_ECDH_anon:
		strcpy(name, "ECDH_anon");
		break;

	case TLS_KXC_DHE:
		strcpy(name, "DHE");
		break;

	case TLS_KXC_ECDHE:
		strcpy(name, "ECDHE");
		break;

	case TLS_KXC_PSK:
		strcpy(name, "PSK");
		break;

	case TLS_KXC_PSK_DHE:
		strcpy(name, "PSK_DHE");
		break;

	case TLS_KXC_PSK_ECDHE:
		strcpy(name, "PSK_ECDHE");
		break;

	default:
		strcpy(name, "unknown type");
		break;
	}

	TLS_DPRINTF("key exchange method: %s", name);
}
#endif

static bool cipher_suite_decided(TLS *tls, enum tls_cipher_suite cipher,
				 struct tls_cert_info *cinfo)
{
	TLS_DPRINTF("cipher: Decided: %04x", cipher);

	tls->pending->cipher_suite = cipher;
	tls->keymethod = tls_cipher_keymethod(cipher);

	switch (tls->keymethod) {
	case TLS_KXC_ECDHE_RSA:
		tls_hs_ecdh_set_curve(tls->ecdh);
		break;

	case TLS_KXC_ECDH_RSA:
		tls_hs_ecdh_set_curve_by_cert(tls->ecdh, cinfo);
		if (tls->ecdh->namedcurve == 0) {
			OK_set_error(ERR_ST_TLS_UNSUPPORTED_CURVE,
				     ERR_LC_TLS1,
				     ERR_PT_TLS_CIPHER + 11, NULL);
			return false;
		}
		break;

	default:
		/* Nothing to do */
		break;
	}

	return true;
}

static uint16_t merge_cipher_suite_list(enum tls_cipher_suite *suites,
				 uint16_t suites_len,
				 uint16_t offset,
				 enum tls_cipher_suite *version_suites,
				 uint16_t version_suites_len)
{
	uint16_t actlen = 0;

	for (int i = 0; i < version_suites_len && offset + actlen < suites_len; i++) {

		for (int j = 0; j < offset + actlen; j++) {
			if (suites[j] == version_suites[i]) {
				goto duplicated;
			}
		}

		tls_cipher_suite_dump(version_suites[i]);
		suites[offset+actlen] = version_suites[i];
		actlen++;

duplicated:
		;
	}

	return offset + actlen;
}

bool tls_cipher_select(TLS *tls, const struct tls_cipher_list *cipher)
{
	if (cipher == NULL || cipher->len == 0) {
		OK_set_error(ERR_ST_NULLPOINTER,
			     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 0, NULL);
		return false;
	}

	struct tls_cipher_list *supported_suites =
		tls_cipher_supported_list(tls, cipher);
	if (supported_suites == NULL) {
		return false;
	}

	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	enum tls_cipher_suite candidate;
	struct tls_cert_info cert_info;

	/*
	 * To check whether cipher suites proposed by client can use.
	 * This check execute in order of the preference defined by server.
	 */
	switch (version) {
	case TLS_VER_TLS13:
		TLS_DPRINTF("cipher: Decided: %04x", supported_suites->list[0]);
		tls->pending->cipher_suite = supported_suites->list[0];
		break;

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		if (tls_cert_info_get(tls->pkcs12_server, &cert_info) != true) {
			tls_cipher_list_free(supported_suites);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return false;
		}

		/* Whether the server can use the ECC cipher.
		 * Compare with CliehtHelloExtension.
		 */
		bool can_use_ecc;
		can_use_ecc = tls_cert_info_can_use_ecc_cipher_suite(&cert_info,
								     tls->ecdh->eclist,
								     tls->ecdh->pflist);

		if (tls->resession == true) {
			/* if resession case, check whether the cipher suite
			 * that used by former session is included. */
			if (tls_cert_info_available(tls->pending->cipher_suite,
						    &cert_info,
						    can_use_ecc, false) == true) {
				tls_cipher_list_free(supported_suites);
				return true;
			}

			/* the cipher suite that used by former session was not
			 * included. make new session and do full negotiation. */
			tls_session_unrefer(tls->pending);
			if ((tls->pending = tls_session_new()) == NULL) {
				OK_set_error(ERR_ST_TLS_GET_SESSION, ERR_LC_TLS1,
					     ERR_PT_TLS_CIPHER + 12, NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
				tls_cipher_list_free(supported_suites);
				return false;
			}
			tls_session_refer(tls->pending);

			tls->resession = false;
		}

		for (int32_t i = 0; i < supported_suites->len; i++) {
			candidate = supported_suites->list[i];
			if (tls_cert_info_available(candidate, &cert_info, can_use_ecc,
						    false) == true) {
				if (cipher_suite_decided(tls, candidate, &cert_info)
				    == true) {
					break;
				}
				TLS_DPRINTF("cipher: cannot to be used: %04x",
					    candidate);
			} else {
				TLS_DPRINTF("cipher: cannot to be used: %04x",
					    candidate);
			}
		}
		break;
	}

	tls_cipher_list_free(supported_suites);

	if (tls->pending->cipher_suite != TLS_NULL_WITH_NULL_NULL) {
		return true;
	}

	TLS_DPRINTF("cipher: cipher list unselected.");
	OK_set_error(ERR_ST_TLS_CIPHER_UNSELECT,
		     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 0, NULL);
	return false;
}

struct tls_cipher_list *tls_cipher_supported_list(TLS* tls,
				const struct tls_cipher_list *preference)
{
	struct tls_cipher_list *clist;

	uint32_t len = 0;

	if (tls->state == TLS_STATE_HS_BEFORE_SEND_CHELLO) {
		uint16_t buf_len = TLS_CIPHER_SUITES_SIZE_MAX / sizeof(uint16_t);
		enum tls_cipher_suite buf[buf_len];
		uint16_t clist_len = 0;
		uint16_t ver_clist_len;
		enum tls_cipher_suite *ver_clist;

		memset(buf, 0, sizeof(buf));

		for (int i = 0; i < tls->supported_versions.len; i++) {
			ver_clist = NULL;

			switch (tls->supported_versions.list[i]) {
			case TLS_VER_SSL30:
			case TLS_VER_TLS10:
			case TLS_VER_TLS11:
				/* Not implemented */
				break;

			case TLS_VER_TLS12:
				ver_clist_len = sizeof(tls_cipher_list_tls12)
						 / sizeof(enum tls_cipher_suite);
				ver_clist = tls_cipher_list_tls12;
				break;

			case TLS_VER_TLS13:
				ver_clist_len = sizeof(tls_cipher_list_tls13)
						 / sizeof(enum tls_cipher_suite);
				ver_clist = tls_cipher_list_tls13;
				break;

			default:
				/* Ignore unknown version */
				break;
			}

			if (ver_clist != NULL) {
				clist_len = merge_cipher_suite_list(buf,
						    buf_len,
						    clist_len,
						    ver_clist,
						    ver_clist_len);
			}

			/* check buffer is full */
			if (buf_len < clist_len) {
				OK_set_error(ERR_ST_TLS_CIPHER_LIST_FULL,
					     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 1,
					     NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
				return NULL;
			} else if (buf_len == clist_len) {
				break;
			}
		}

		if ((clist = tls_cipher_list_alloc(clist_len)) == NULL) {
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return NULL;
		}

		memcpy(clist->list, buf, clist->len * sizeof(enum tls_cipher_suite));

		return clist;
	}

	/* return list of cipher list that the client can support. */
	enum tls_cipher_suite *suite;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
		/* TODO: not implementation. */
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 2, NULL);
		return NULL;

	case TLS_VER_TLS12:
		suite = tls_cipher_list_tls12;
		len   = (sizeof (tls_cipher_list_tls12) /
			 sizeof (enum tls_cipher_suite));
		break;

	case TLS_VER_TLS13:
		suite = tls_cipher_list_tls13;
		len   = (sizeof (tls_cipher_list_tls13) /
			 sizeof (enum tls_cipher_suite));
		break;

	default:
		/* Unknown version */
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 3, NULL);
		return NULL;
	}

	if ((clist = tls_cipher_list_alloc(len)) == NULL) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	uint32_t actlen = 0;
	for (uint16_t i = 0; i < len; ++i) {
		bool use = false;
		if (preference == NULL) {
			use = true;
		} else {
			for (int32_t j = 0; j < preference->len; ++j) {
				if (preference->list[j] == suite[i]) {
					use = true;
					break;
				}
			}
		}

		if (use == true) {
			clist->list[actlen] = suite[i];
			actlen++;
		} else {
			TLS_DPRINTF("cipher: cannot to be used: %04x",
				    suite[i]);
		}
	}

	clist->len = actlen;

	return clist;
}

struct tls_cipher_list *tls_cipher_list_alloc(const uint16_t len)
{
	struct tls_cipher_list *clist;

	clist = (struct tls_cipher_list *)
		malloc(sizeof(struct tls_cipher_list));

	if (clist == NULL) {
		TLS_DPRINTF("cipher: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 1, NULL);
		return NULL;
	}

	clist->len = len;
	if (len > 0) {
		clist->list = (enum tls_cipher_suite *)
			calloc(len, sizeof(enum tls_cipher_suite));
		if (clist->list == NULL) {
			TLS_DPRINTF("cipher: calloc: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_CALLOC,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 2, NULL);
			free(clist);
			return NULL;
		}
	} else {
		clist->list = NULL;
	}

	return clist;
}

void tls_cipher_list_free(struct tls_cipher_list *list) {
	if (list == NULL) {
		return;
	}

	free(list->list);
	list->list = NULL;
	list->len  = 0;

	free(list);
	list = NULL;
}

enum tls_hs_sighash_hash_algo tls_cipher_hashalgo(const uint16_t suite) {
	switch(suite) {
	case TLS_AES_128_GCM_SHA256:
	case TLS_CHACHA20_POLY1305_SHA256:
	case TLS_AES_128_CCM_SHA256:
	case TLS_AES_128_CCM_8_SHA256:
		return TLS_HASH_ALGO_SHA256;

	case TLS_AES_256_GCM_SHA384:
		return TLS_HASH_ALGO_SHA384;

	default:
		/* Unknown type */
		TLS_DPRINTF("cipher: unknown type %d", suite);
		OK_set_error(ERR_ST_TLS_UNKNOWN_CIPHER_TYPE,
			     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 4, NULL);
		return TLS_HASH_ALGO_NONE;
	}
}

enum tls_keyexchange_method tls_cipher_keymethod(const uint16_t suite) {
	switch(suite) {
	case TLS_RSA_WITH_NULL_MD5:
	case TLS_RSA_WITH_NULL_SHA:
	case TLS_RSA_WITH_NULL_SHA256:
#ifdef HAVE_ARC4
	case TLS_RSA_WITH_RC4_128_MD5:
	case TLS_RSA_WITH_RC4_128_SHA:
#endif
	case TLS_RSA_WITH_3DES_EDE_CBC_SHA:
	case TLS_RSA_WITH_AES_128_CBC_SHA:
	case TLS_RSA_WITH_AES_256_CBC_SHA:
	case TLS_RSA_WITH_AES_128_CBC_SHA256:
	case TLS_RSA_WITH_AES_256_CBC_SHA256:

	case TLS_RSA_WITH_AES_128_GCM_SHA256:
	case TLS_RSA_WITH_AES_256_GCM_SHA384:
		return TLS_KXC_RSA;

	case TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA:
	case TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA:
	case TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256:
	case TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384:
		return TLS_KXC_ECDHE_RSA;

	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA:
	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256:
	case TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256:
	case TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384:
		return TLS_KXC_ECDH_RSA;

	default:
		/* Unknown type was set by server. */
		TLS_DPRINTF("cipher: unknown type %d", suite);
		OK_set_error(ERR_ST_TLS_UNKNOWN_CIPHER_TYPE,
			     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 13, NULL);
		return TLS_KXC_NULL;
	}
}

static bool tls_cipher_set_tls12(TLS *tls, const uint16_t suite) {
	enum tls_cipher_suite *suites = tls_cipher_list_tls12;
	uint32_t len = (sizeof (tls_cipher_list_tls12) /
			 sizeof (enum tls_cipher_suite));
	for (uint32_t i = 0; i < len; ++i) {
		if (suites[i] == suite) {
			tls->pending->cipher_suite = suite;
			tls->keymethod = tls_cipher_keymethod(suite);
			if (tls->keymethod == TLS_KXC_NULL) {
				break;
			}
			return true;
		}
	}

	/* Unknown type was set by server. */
	TLS_DPRINTF("cipher: unknown type %d", suite);
	OK_set_error(ERR_ST_TLS_UNKNOWN_CIPHER_TYPE,
		     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 3, NULL);
	return false;
}

static bool tls_cipher_set_tls13(TLS *tls, const uint16_t suite) {
	enum tls_cipher_suite *suites = tls_cipher_list_tls13;
	uint32_t len = (sizeof (tls_cipher_list_tls13) /
			 sizeof (enum tls_cipher_suite));
	for (uint32_t i = 0; i < len; ++i) {
		if (suites[i] == suite) {
			tls->pending->cipher_suite = suite;
			TLS_DPRINTF("cipher: Decided: %04x", suite);
			return true;
		}
	}

	/* Unknown type was set by server. */
	TLS_DPRINTF("cipher: unknown type %d", suite);
	OK_set_error(ERR_ST_TLS_UNKNOWN_CIPHER_TYPE,
		     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 14, NULL);
	return false;
}

bool tls_cipher_set(TLS *tls, const uint16_t suite) {
	uint16_t version = tls_util_convert_protover_to_ver(
	    &(tls->negotiated_version));

	switch(version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
		/* TODO: not implementation. */
		break;

	case TLS_VER_TLS12:
		return tls_cipher_set_tls12(tls, suite);

	case TLS_VER_TLS13:
		return tls_cipher_set_tls13(tls, suite);

	default:
		break;
	}

	OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
		     ERR_LC_TLS6, ERR_PT_TLS_CIPHER2 + 5, NULL);
	return false;
}

bool tls_cipher_param_set(const enum tls_cipher_suite suite,
			  struct tls_cipher_param *param) {
	switch(suite) {
	case TLS_RSA_WITH_NULL_MD5:
		*param = tls_cipher_null_md5;
		break;

	case TLS_RSA_WITH_NULL_SHA:
		*param = tls_cipher_null_sha;
		break;

	case TLS_RSA_WITH_NULL_SHA256:
		*param = tls_cipher_null_sha256;
		break;

#ifdef HAVE_ARC4
	case TLS_RSA_WITH_RC4_128_MD5:
		*param = tls_cipher_rc4_128_md5;
		break;

	case TLS_RSA_WITH_RC4_128_SHA:
		*param = tls_cipher_rc4_128_sha;
		break;

#endif
	case TLS_RSA_WITH_3DES_EDE_CBC_SHA:
		*param = tls_cipher_3des_ede_cbc_sha;
		break;

	case TLS_RSA_WITH_AES_128_CBC_SHA:
	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA:
	case TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA:
		*param = tls_cipher_aes_128_cbc_sha;
		break;

	case TLS_RSA_WITH_AES_256_CBC_SHA:
	case TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA:
		*param = tls_cipher_aes_256_cbc_sha;
		break;

	case TLS_RSA_WITH_AES_128_CBC_SHA256:
	case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256:
		*param = tls_cipher_aes_128_cbc_sha256;
		break;

	case TLS_RSA_WITH_AES_256_CBC_SHA256:
		*param = tls_cipher_aes_256_cbc_sha256;
		break;

	case TLS_RSA_WITH_AES_128_GCM_SHA256:
	case TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256:
	case TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256:
		*param = tls_cipher_aes_128_gcm_sha256;
		break;

	case TLS_RSA_WITH_AES_256_GCM_SHA384:
	case TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384:
	case TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384:
		*param = tls_cipher_aes_256_gcm_sha384;
		break;

	case TLS_AES_128_GCM_SHA256:
		*param = tls_cipher_aes_128_gcm_hkdf_sha256;
		break;

	case TLS_AES_256_GCM_SHA384:
		*param = tls_cipher_aes_256_gcm_hkdf_sha384;
		break;

	case TLS_CHACHA20_POLY1305_SHA256:
		*param = tls_cipher_chacha20_poly1305_hkdf_sha256;
		break;

	default:
		TLS_DPRINTF("cipher: unknown cipher param %d", suite);
		OK_set_error(ERR_ST_TLS_UNKNOWN_CIPHER_TYPE,
			     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 4, NULL);
		return false;
	}

	return true;
}

int32_t tls_cipher(TLS *tls,
		   uint8_t *dest,
		   const enum tls_record_ctype type,
		   const uint8_t *src,
		   const int32_t len) {
	int32_t cipher_length = -1;
	switch (tls->active_write.cipher.cipher_type) {
	case TLS_CIPHER_TYPE_STREAM:
		if ((cipher_length = tls_cipher_stream(
			     tls, dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: encrypt (stream) failed");
			OK_set_error(ERR_ST_TLS_ENCRYPT_STREAM_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 5, NULL);
			return -1;
		}
		break;

	case TLS_CIPHER_TYPE_BLOCK:
		if ((cipher_length = tls_cipher_block(
			     tls, dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: encrypt (block) failed");
			OK_set_error(ERR_ST_TLS_ENCRYPT_BLOCK_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 6, NULL);
			return -1;
		}
		break;

	case TLS_CIPHER_TYPE_AEAD:
		if ((cipher_length = tls_cipher_aead(tls,
						     dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: encrypt (block) failed");
			OK_set_error(ERR_ST_TLS_ENCRYPT_AEAD_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 7, NULL);
			return -1;
		}
		break;

	default:
		assert(!"selected cipher type is unknown.");
	}

	return cipher_length;
}

int32_t tls_decipher(TLS *tls,
		     uint8_t *dest,
		     const enum tls_record_ctype type,
		     const uint8_t *src,
		     const int32_t len) {
	int32_t cipher_length = -1;
	switch (tls->active_read.cipher.cipher_type) {
	case TLS_CIPHER_TYPE_STREAM:
		if ((cipher_length = tls_decipher_stream(
			     tls, dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: decrypt (stream) failed");
			OK_set_error(ERR_ST_TLS_DECRYPT_STREAM_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 8, NULL);
			return -1;
		}
		break;

	case TLS_CIPHER_TYPE_BLOCK:
		if ((cipher_length = tls_decipher_block(
			     tls, dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: decrypt (block) failed");
			OK_set_error(ERR_ST_TLS_DECRYPT_BLOCK_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 9, NULL);
			return -1;
		}
		break;

	case TLS_CIPHER_TYPE_AEAD:
		if ((cipher_length = tls_decipher_aead(tls,
						       dest, type, src, len)) < 0) {
			TLS_DPRINTF("cipher: decrypt (block) failed");
			OK_set_error(ERR_ST_TLS_DECRYPT_AEAD_FAILED,
				     ERR_LC_TLS1, ERR_PT_TLS_CIPHER + 10, NULL);
			return -1;
		}
		break;

	default:
		assert(!"selected cipher type is unknown.");
	}

	return cipher_length;
}
