/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_alert.h"
#include "tls_session.h"

/** @see tls.c */
extern void tls_free_connection(TLS *tls);

/**
 * output the debug purpose information related with ALERT protocol.
 *
 * output the alert level and alert description. and, this function do
 * nothing when TLS_DEBUG macro is not defined on building state.
 */
static inline void dump_alert(const enum tls_alert_level level,
			      const enum tls_alert_desc desc);

/**
 * check whether the ALERT level is valid.
 *
 * alert level on TLS is only fatal or warning. when the alert level is
 * not which, this function fail.
 */
static bool check_alert_level(const enum tls_alert_level level);

/**
 * check whether the ALERT description is valid.
 *
 * this function check the following item.
 *
 * - is specified description defined in the RFC 5246?
 * - is specified description and level pair valid combination?
 *
 * a latter case means, for example, check whether the level is fatal
 * when specified alert was unexpected_message (unexpected_message alert
 * accepts fatal level only).
 */
static bool check_alert_desc(const enum tls_alert_level level,
			     const enum tls_alert_desc desc);

/**
 * get ALERT level based on ALERT description.
 *
 * in TLS 1.3, ALERT level is determined by each ALERT description.
 * this function returns appropriate ALERT level associated with ALERT
 * description.
 */
static enum tls_alert_level get_alert_level_tls13(enum tls_alert_desc desc);

/**
 * send alert with level and description.
 */
static bool send_alert_record(TLS *tls, const enum tls_alert_level level,
			      const enum tls_alert_desc desc);

/**
 * send alert in a manner of TLS 1.2.
 */
static bool send_alert_up_to_tls12(TLS *tls,
				   const enum tls_alert_level level,
				   const enum tls_alert_desc desc);

/**
 * send alert in a manner of TLS 1.3.
 */
static bool send_alert_tls13(TLS *tls,
			     const enum tls_alert_level level UNUSED,
			     const enum tls_alert_desc desc);

/**
 * receive alert in a manner of TLS 1.2.
 */
static bool recv_alert_up_to_tls12(TLS *tls, const struct tls_record *record);

/**
 * receive alert in a manner of TLS 1.3.
 */
static bool recv_alert_tls13(TLS *tls, const struct tls_record *record);

/**
 * message (body) length of alert protocol.
 *
 * in this implementation, fragmentation of alert protocol does not be
 * considered. that is handled as invalid alert (send fatal alert in
 * that cse).
 */
static const uint32_t TLS_ALERT_BODY_LENGTH = 2;

/* if TLS_DEBUG is off, level and desc is not used. so, add attribute
 * UNUSED.  */
static inline void dump_alert(const enum tls_alert_level level UNUSED,
			      const enum tls_alert_desc desc UNUSED) {
#ifdef TLS_DEBUG
	struct {
		char name[20];
	} name_of_level[] = {
		[TLS_ALERT_LEVEL_WARNING] = { "WARNING" },
		[TLS_ALERT_LEVEL_FATAL]   = { "FATAL" }
	};

	char desc_name[30];
	switch(desc) {
	case TLS_ALERT_DESC_CLOSE_NOTIFY:
		strcpy(desc_name, "CLOSE_NOTIFY");
		break;
	case TLS_ALERT_DESC_UNEXPECTED_MESSAGE:
		strcpy(desc_name, "UNEXPECTED_MESSAGE");
		break;

	case TLS_ALERT_DESC_BAD_RECORD_MAC:
		strcpy(desc_name, "BAD_RECORD_MAC");
		break;

	case TLS_ALERT_DESC_DECRYPTION_FAILED_RESERVED:
		strcpy(desc_name, "DECRYPTION_FAILED_RESERVED");
		break;

	case TLS_ALERT_DESC_RECORD_OVERFLOW:
		strcpy(desc_name, "RECORD_OVERFLOW");
		break;

	case TLS_ALERT_DESC_DECOMPRESSION_FAILURE:
		strcpy(desc_name, "DECOMPRESSION_FAILURE");
		break;

	case TLS_ALERT_DESC_HANDSHAKE_FAILURE:
		strcpy(desc_name, "HANDSHAKE_FAILURE");
		break;

	case TLS_ALERT_DESC_NO_CERTIFICATE_RESERVED:
		strcpy(desc_name, "NO_CERTIFICATE_RESERVED");
		break;

	case TLS_ALERT_DESC_BAD_CERTIFICATE:
		strcpy(desc_name, "BAD_CERTIFICATE");
		break;

	case TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE:
		strcpy(desc_name, "UNSUPPORTED_CERTIFICATE");
		break;

	case TLS_ALERT_DESC_CERTIFICATE_REVOKED:
		strcpy(desc_name, "CERTIFICATE_REVOKED");
		break;

	case TLS_ALERT_DESC_CERTIFICATE_EXPIRED:
		strcpy(desc_name, "CERTIFICATE_EXPIRED");
		break;

	case TLS_ALERT_DESC_CERTIFICATE_UNKNOWN:
		strcpy(desc_name, "CERTIFICATE_UNKNOWN");
		break;

	case TLS_ALERT_DESC_ILLEGAL_PARAMETER:
		strcpy(desc_name, "ILLEGAL_PARAMETER");
		break;

	case TLS_ALERT_DESC_UNKNOWN_CA:
		strcpy(desc_name, "UNKNOWN_CA");
		break;

	case TLS_ALERT_DESC_ACCESS_DENIED:
		strcpy(desc_name, "ACCESS_DENIED");
		break;

	case TLS_ALERT_DESC_DECODE_ERROR:
		strcpy(desc_name, "DECODE_ERROR");
		break;

	case TLS_ALERT_DESC_DECRYPT_ERROR:
		strcpy(desc_name, "DECRYPT_ERROR");
		break;

	case TLS_ALERT_DESC_EXPORT_RESTRICTION_RESERVED:
		strcpy(desc_name, "EXPORT_RESTRICTION_RESERVED");
		break;

	case TLS_ALERT_DESC_PROTOCOL_VERSION:
		strcpy(desc_name, "PROTOCOL_VERSION");
		break;

	case TLS_ALERT_DESC_INSUFFICIENT_SECURITY:
		strcpy(desc_name, "INSUFFICIENT_SECURITY");
		break;

	case TLS_ALERT_DESC_INTERNAL_ERROR:
		strcpy(desc_name, "INTERNAL_ERROR");
		break;

	case TLS_ALERT_DESC_INAPPROPRIATE_FALLBACK:
		strcpy(desc_name, "INAPPROPRIATE_FALLBACK");
		break;

	case TLS_ALERT_DESC_USER_CANCELED:
		strcpy(desc_name, "USER_CANCELED");
		break;

	case TLS_ALERT_DESC_NO_RENEGOTIATION:
		strcpy(desc_name, "NO_RENEGOTIATION");
		break;

	case TLS_ALERT_DESC_MISSING_EXTENSION:
		strcpy(desc_name, "MISSING_EXTENSION");
		break;

	case TLS_ALERT_DESC_UNSUPPORTED_EXTENSION:
		strcpy(desc_name, "UNSUPPORTED_EXTENSION");
		break;

	case TLS_ALERT_DESC_CERTIFICATE_UNOBTAINABLE:
		strcpy(desc_name, "CERTIFICATE_UNOBTAINABLE");
		break;

	case TLS_ALERT_DESC_UNRECOGNIZED_NAME:
		strcpy(desc_name, "UNRECOGNIZED_NAME");
		break;

	case TLS_ALERT_DESC_BAD_CERTIFICATE_STATUS_RESPONSE:
		strcpy(desc_name, "BAD_CERTIFICATE_STATUS_RESPONSE");
		break;

	case TLS_ALERT_DESC_BAD_CERTIFICATE_HASH_VALUE:
		strcpy(desc_name, "BAD_CERTIFICATE_HASH_VALUE");
		break;

	case TLS_ALERT_DESC_UNKNOWN_PSK_IDENTITY:
		strcpy(desc_name, "UNKNOWN_PSK_IDENTITY");
		break;

	case TLS_ALERT_DESC_CERTIFICATE_REQUIRED:
		strcpy(desc_name, "CERTIFICATE_REQUIRED");
		break;

	case TLS_ALERT_DESC_NO_APPLICATION_PROTOCOL:
		strcpy(desc_name, "NO_APPLICATION_PROTOCOL");
		break;

	default:
		strcpy(desc_name, "unknown desc type");
		break;
	}

	TLS_DPRINTF("alert: level = %s, desc = %s",
		    name_of_level[level].name, desc_name);
#endif
}

static bool check_alert_level(const enum tls_alert_level level) {
	switch(level) {
	case TLS_ALERT_LEVEL_WARNING:
	case TLS_ALERT_LEVEL_FATAL:
		break;

	default:
		/* unknown alert level received. */
		TLS_DPRINTF("alert: unknown level.");
		OK_set_error(ERR_ST_TLS_UNKNOWN_ALERT_LEVEL,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 0, NULL);
		return false;
	}

	return true;
}

static bool check_alert_desc(const enum tls_alert_level level,
			     const enum tls_alert_desc desc) {
	switch(desc) {
	case TLS_ALERT_DESC_CLOSE_NOTIFY:
		/* it seems the close notify alert is sent by the
		 * warning level generally. but, RFC 5246 do not
		 * describe close notify should be sent by warning
		 * level. therefore, accept both fatal and warning. */

	case TLS_ALERT_DESC_BAD_CERTIFICATE:
	case TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE:
	case TLS_ALERT_DESC_CERTIFICATE_REVOKED:
	case TLS_ALERT_DESC_CERTIFICATE_EXPIRED:
	case TLS_ALERT_DESC_CERTIFICATE_UNKNOWN:
	case TLS_ALERT_DESC_USER_CANCELED:
		/* user canceled is generally warning. */
	case TLS_ALERT_DESC_DECRYPTION_FAILED_RESERVED:
	case TLS_ALERT_DESC_NO_CERTIFICATE_RESERVED:
	case TLS_ALERT_DESC_EXPORT_RESTRICTION_RESERVED:
	case TLS_ALERT_DESC_CERTIFICATE_UNOBTAINABLE:
		/* see RFC6066 for details of certificate_unobtainable */
	case TLS_ALERT_DESC_UNRECOGNIZED_NAME:
		/*
		 * warning level of unrecognized_name is not recommended.
		 * see RFC6066 for details.
		 */
		break;

	case TLS_ALERT_DESC_UNEXPECTED_MESSAGE:
	case TLS_ALERT_DESC_BAD_RECORD_MAC:
	case TLS_ALERT_DESC_RECORD_OVERFLOW:
	case TLS_ALERT_DESC_DECOMPRESSION_FAILURE:
	case TLS_ALERT_DESC_HANDSHAKE_FAILURE:
	case TLS_ALERT_DESC_ILLEGAL_PARAMETER:
	case TLS_ALERT_DESC_UNKNOWN_CA:
	case TLS_ALERT_DESC_ACCESS_DENIED:
	case TLS_ALERT_DESC_DECODE_ERROR:
	case TLS_ALERT_DESC_DECRYPT_ERROR:
	case TLS_ALERT_DESC_PROTOCOL_VERSION:
	case TLS_ALERT_DESC_INSUFFICIENT_SECURITY:
	case TLS_ALERT_DESC_INTERNAL_ERROR:
	case TLS_ALERT_DESC_INAPPROPRIATE_FALLBACK:
		/* see RFC7507 for details of inappropriate_fallback */
	case TLS_ALERT_DESC_MISSING_EXTENSION:
	case TLS_ALERT_DESC_UNSUPPORTED_EXTENSION:
	case TLS_ALERT_DESC_BAD_CERTIFICATE_STATUS_RESPONSE:
		/* see RFC6066 for details of bad_certificate_status_response */
	case TLS_ALERT_DESC_BAD_CERTIFICATE_HASH_VALUE:
		/* see RFC6066 for details of bad_certificate_hash_value */
	case TLS_ALERT_DESC_UNKNOWN_PSK_IDENTITY:
	case TLS_ALERT_DESC_CERTIFICATE_REQUIRED:
	case TLS_ALERT_DESC_NO_APPLICATION_PROTOCOL:
		/* see RFC6066 for details of no_application_protocol */
		if (level == TLS_ALERT_LEVEL_WARNING) {
			TLS_DPRINTF("alert: invalid alert level.");
			OK_set_error(ERR_ST_TLS_INVALID_ALERT_LEVEL,
				     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 1, NULL);
			return false;
		}
		break;

	case TLS_ALERT_DESC_NO_RENEGOTIATION:
		if (level == TLS_ALERT_LEVEL_FATAL) {
			TLS_DPRINTF("alert: invalid alert level.");
			OK_set_error(ERR_ST_TLS_INVALID_ALERT_LEVEL,
				     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 2, NULL);
			return false;
		}
		break;

	default:
		/* unknown alert description received. */
		return false;
	}

	return true;
}

static enum tls_alert_level get_alert_level_tls13(enum tls_alert_desc desc) {
	switch (desc) {
	case TLS_ALERT_DESC_CLOSE_NOTIFY:
	case TLS_ALERT_DESC_USER_CANCELED:
		return TLS_ALERT_LEVEL_WARNING;

	case TLS_ALERT_DESC_UNEXPECTED_MESSAGE:
	case TLS_ALERT_DESC_BAD_RECORD_MAC:
	case TLS_ALERT_DESC_RECORD_OVERFLOW:
	case TLS_ALERT_DESC_HANDSHAKE_FAILURE:
	case TLS_ALERT_DESC_BAD_CERTIFICATE:
	case TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE:
	case TLS_ALERT_DESC_CERTIFICATE_REVOKED:
	case TLS_ALERT_DESC_CERTIFICATE_EXPIRED:
	case TLS_ALERT_DESC_CERTIFICATE_UNKNOWN:
	case TLS_ALERT_DESC_ILLEGAL_PARAMETER:
	case TLS_ALERT_DESC_UNKNOWN_CA:
	case TLS_ALERT_DESC_ACCESS_DENIED:
	case TLS_ALERT_DESC_DECODE_ERROR:
	case TLS_ALERT_DESC_DECRYPT_ERROR:
	case TLS_ALERT_DESC_PROTOCOL_VERSION:
	case TLS_ALERT_DESC_INSUFFICIENT_SECURITY:
	case TLS_ALERT_DESC_INTERNAL_ERROR:
	case TLS_ALERT_DESC_INAPPROPRIATE_FALLBACK:
	case TLS_ALERT_DESC_MISSING_EXTENSION:
	case TLS_ALERT_DESC_UNSUPPORTED_EXTENSION:
	case TLS_ALERT_DESC_UNRECOGNIZED_NAME:
	case TLS_ALERT_DESC_BAD_CERTIFICATE_STATUS_RESPONSE:
	case TLS_ALERT_DESC_UNKNOWN_PSK_IDENTITY:
	case TLS_ALERT_DESC_CERTIFICATE_REQUIRED:
	case TLS_ALERT_DESC_NO_APPLICATION_PROTOCOL:
	default:
		return TLS_ALERT_LEVEL_FATAL;
	}
}

static bool send_alert_record(TLS *tls, const enum tls_alert_level level,
			      const enum tls_alert_desc desc) {
	uint32_t len = TLS_ALERT_BODY_LENGTH;
	uint8_t buff[len];

	dump_alert(level, desc);

	buff[0] = level;
	buff[1] = desc;

	ssize_t n UNUSED;
	if ((n = tls_record_write(TLS_CTYPE_ALERT, tls, buff, len)) < 0) {
		TLS_DPRINTF("alert: tls_record_write (%zd)", n);
		OK_set_error(ERR_ST_TLS_TLS_RECORD_WRITE,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 3, NULL);
		return false;
	}

	return true;
}

static bool send_alert_up_to_tls12(TLS *tls,
				   const enum tls_alert_level level,
				   const enum tls_alert_desc desc) {
	if (send_alert_record(tls, level, desc) == false) {
		/* if failed to send alert message, try to send fatal
		 * level internal error alert. */
		send_alert_record(tls, TLS_ALERT_LEVEL_FATAL,
				  TLS_ALERT_DESC_INTERNAL_ERROR);
		goto fatal;
	}

	if (desc == TLS_ALERT_DESC_CLOSE_NOTIFY) {
		/* do not wait close_notify of peer. close connection
		 * immediately. */
		goto close;
	}

	if (level == TLS_ALERT_LEVEL_FATAL) {
		goto fatal;
	}

	/* sent warning level. continue connection. */
	return true;

fatal:
	/* sent fatal level. close connection. */
	tls_session_disable(tls->pending);

close:
	tls_free_connection(tls);
	return false;
}

static bool send_alert_tls13(TLS *tls,
			     const enum tls_alert_level level UNUSED,
			     const enum tls_alert_desc desc) {
	/*
	 * RFC8446 6.  Alert Protocol
	 *
	 *    Alert messages convey a description of the alert and a legacy field
	 *    that conveyed the severity level of the message in previous versions
	 *    of TLS.  Alerts are divided into two classes: closure alerts and
	 *    error alerts.  In TLS 1.3, the severity is implicit in the type of
	 *    alert being sent, and the "level" field can safely be ignored.  The
	 *    "close_notify" alert is used to indicate orderly closure of one
	 *    direction of the connection.  Upon receiving such an alert, the TLS
	 *    implementation SHOULD indicate end-of-data to the application.
	 *
	 *    Error alerts indicate abortive closure of the connection (see
	 *    Section 6.2).  Upon receiving an error alert, the TLS implementation
	 *    SHOULD indicate an error to the application and MUST NOT allow any
	 *    further data to be sent or received on the connection.  Servers and
	 *    clients MUST forget the secret values and keys established in failed
	 *    connections, with the exception of the PSKs associated with session
	 *    tickets, which SHOULD be discarded if possible.
	 *
	 *    All the alerts listed in Section 6.2 MUST be sent with
	 *    AlertLevel=fatal and MUST be treated as error alerts when received
	 *    regardless of the AlertLevel in the message.  Unknown Alert types
	 *    MUST be treated as error alerts.
	 */
	enum tls_alert_desc actdesc = desc;
	enum tls_alert_level actlevel = get_alert_level_tls13(actdesc);

	/*
	 * RFC8446 6.2.  Error Alerts
	 *
	 *    Error handling in TLS is very simple.  When an error is detected, the
	 *    detecting party sends a message to its peer.  Upon transmission or
	 *    receipt of a fatal alert message, both parties MUST immediately close
	 *    the connection.
	 */
	if (send_alert_record(tls, actlevel, actdesc) == false) {
		/* if failed to send alert message, try to send fatal
		 * level internal error alert. */
		if (actlevel == TLS_ALERT_LEVEL_WARNING) {
			send_alert_record(tls, TLS_ALERT_LEVEL_FATAL,
					  TLS_ALERT_DESC_INTERNAL_ERROR);
		}
		goto fatal;
	}

	if (actdesc == TLS_ALERT_DESC_CLOSE_NOTIFY) {
		/*
		 * RFC8446 6.1.  Closure Alerts
		 *
		 *                                        Any data received after a closure
		 *    alert has been received MUST be ignored.
		 */
		tls->active_write.closed = true;
		TLS_DPRINTF("alert: close write side");

		return false;
	}

	/*
	 * RFC8446 6.2.  Error Alerts
	 *
	 *                                                  Upon transmission or
	 *    receipt of a fatal alert message, both parties MUST immediately close
	 *    the connection.
	 */
	if (actlevel == TLS_ALERT_LEVEL_FATAL) {
		goto fatal;
	}

	return true;

fatal:
	tls_free_connection(tls);
	return false;
}

static bool recv_alert_up_to_tls12(TLS *tls, const struct tls_record *record) {
	if (record->len != TLS_ALERT_BODY_LENGTH) {
		TLS_DPRINTF("invalid record length.");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 4, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		goto fatal;
	}

	enum tls_alert_level level = record->frag[0];
	if (check_alert_level(level) == false) {
		TLS_DPRINTF("alert: invalid alert level.");
		OK_set_error(ERR_ST_TLS_INVALID_ALERT_LEVEL,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 5, NULL);
		/* if received alert has unknown level, send fatal error
		 * and exit immediately. */
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		goto fatal;
	}

	enum tls_alert_desc desc = record->frag[1];
	if (check_alert_desc(level, desc) == false) {
		TLS_DPRINTF("alert: invalid alert desc.");
		OK_set_error(ERR_ST_TLS_INVALID_ALERT_DESC,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 6, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		goto fatal;
	}

	dump_alert(level, desc);

	if (desc == TLS_ALERT_DESC_PROTOCOL_VERSION) {
		tls->errnum = TLS_ERR_PROTOCOL_VERSION;
	}

	if (level == TLS_ALERT_LEVEL_FATAL) {
		/* case of fatal level alerts */
		goto fatal;
	}

	/* case of warning level alerts */

	/* check close notify alert (RFC 5246 seciton 7.2.1).
	 *
	 *   Unless some other fatal alert has been transmitted, each
	 *   party is required to send a close_notify alert before
	 *   closing the write side of the connection.  The other party
	 *   MUST respond with a close_notify alert of its own and close
	 *   down the connection immediately, discarding any pending
	 *   writes.  It is not required for the initiator of the close
	 *   to wait for the responding close_notify alert before
	 *   closing the read side of the connection. */
	if (desc == TLS_ALERT_DESC_CLOSE_NOTIFY) {
		TLS_ALERT_WARN(tls, TLS_ALERT_DESC_CLOSE_NOTIFY);
		/* set TLS_ERR_CLOSE_NOTIFY to errnum so that server
		 * distinguish it when finished by close notify.  if did
		 * not do that, server can not distinguish whether the
		 * server finished by error. */
		tls->errnum = TLS_ERR_CLOSE_NOTIFY;
		/* close connection immediately. */
		goto close;
	}

	/* received no renegotiation alert (by client probably). since
	 * current implemetation do not support re-negotiation,
	 * therefore, disconnect immediately by fatal alert. */
	if (desc == TLS_ALERT_DESC_NO_RENEGOTIATION) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		goto fatal;
	}

	return true;

fatal:
	tls_session_disable(tls->pending);

close:
	tls_free_connection(tls);
	return false;
}

static bool recv_alert_tls13(TLS *tls, const struct tls_record *record) {
	if (record->len != TLS_ALERT_BODY_LENGTH) {
		TLS_DPRINTF("invalid record length.");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		goto fatal;
	}

	/*
	 * RFC8446 6.  Alert Protocol
	 *
	 *    Alert messages convey a description of the alert and a legacy field
	 *    that conveyed the severity level of the message in previous versions
	 *    of TLS.  Alerts are divided into two classes: closure alerts and
	 *    error alerts.  In TLS 1.3, the severity is implicit in the type of
	 *    alert being sent, and the "level" field can safely be ignored.  The
	 *    "close_notify" alert is used to indicate orderly closure of one
	 *    direction of the connection.  Upon receiving such an alert, the TLS
	 *    implementation SHOULD indicate end-of-data to the application.
	 *
	 *    Error alerts indicate abortive closure of the connection (see
	 *    Section 6.2).  Upon receiving an error alert, the TLS implementation
	 *    SHOULD indicate an error to the application and MUST NOT allow any
	 *    further data to be sent or received on the connection.  Servers and
	 *    clients MUST forget the secret values and keys established in failed
	 *    connections, with the exception of the PSKs associated with session
	 *    tickets, which SHOULD be discarded if possible.
	 *
	 *    All the alerts listed in Section 6.2 MUST be sent with
	 *    AlertLevel=fatal and MUST be treated as error alerts when received
	 *    regardless of the AlertLevel in the message.  Unknown Alert types
	 *    MUST be treated as error alerts.
	 */
	enum tls_alert_desc desc = record->frag[1];
	enum tls_alert_level level = get_alert_level_tls13(desc);

	dump_alert(level, desc);

	if (desc == TLS_ALERT_DESC_PROTOCOL_VERSION) {
		tls->errnum = TLS_ERR_PROTOCOL_VERSION;
	}

	/*
	 * RFC8446 6.2.  Error Alerts
	 *
	 *                                                  Upon transmission or
	 *    receipt of a fatal alert message, both parties MUST immediately close
	 *    the connection.
	 */
	if (level == TLS_ALERT_LEVEL_FATAL) {
		/* case of fatal level alerts */
		goto fatal;
	}

	/* case of warning level alerts */

	/*
	 * RFC8446 6.1.  Closure Alerts
	 *
	 *    Each party MUST send a "close_notify" alert before closing its write
	 *    side of the connection, unless it has already sent some error alert.
	 *    This does not have any effect on its read side of the connection.
	 *    Note that this is a change from versions of TLS prior to TLS 1.3 in
	 *    which implementations were required to react to a "close_notify" by
	 *    discarding pending writes and sending an immediate "close_notify"
	 *    alert of their own.  That previous requirement could cause truncation
	 *    in the read side.  Both parties need not wait to receive a
	 *    "close_notify" alert before closing their read side of the
	 *    connection, though doing so would introduce the possibility of
	 *    truncation.
	 */
	if (desc == TLS_ALERT_DESC_CLOSE_NOTIFY) {
		/* set TLS_ERR_CLOSE_NOTIFY to errnum so that server
		 * distinguish it when finished by close notify.  if did
		 * not do that, server can not distinguish whether the
		 * server finished by error. */
		tls->errnum = TLS_ERR_CLOSE_NOTIFY;

		/*
		 * RFC8446 6.1.  Closure Alerts
		 *
		 *                                        Any data received after a closure
		 *    alert has been received MUST be ignored.
		 */
		tls->active_read.closed = true;
		TLS_DPRINTF("alert: close read side");

		return false;
	}

	return true;

fatal:
	/*
	 * RFC8446 6.  Alert Protocol
	 *
	 *                   Upon receiving an error alert, the TLS implementation
	 *    SHOULD indicate an error to the application and MUST NOT allow any
	 *    further data to be sent or received on the connection.  Servers and
	 *    clients MUST forget the secret values and keys established in failed
	 *    connections, with the exception of the PSKs associated with session
	 *    tickets, which SHOULD be discarded if possible.
	 */
	tls_free_connection(tls);
	return false;
}

bool tls_alert_send(TLS *tls,
		    const enum tls_alert_level level,
		    const enum tls_alert_desc desc) {
	TLS_DPRINTF("alert: send");

	if (tls->active_write.closed == true) {
		TLS_DPRINTF("alert: already closed");
		OK_set_error(ERR_ST_TLS_CONNECTION_CLOSED,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 8, NULL);
		return false;
	}

	if (tls->state < TLS_STATE_HS_AFTER_RECV_SHELLO ||
		tls->state == TLS_STATE_HS_BEFORE_RECV_CHELLO) {
		return send_alert_up_to_tls12(tls, level, desc);
	} else {
		/* version must be negotiated */
		uint16_t version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
			return send_alert_up_to_tls12(tls, level, desc);

		case TLS_VER_TLS13:
			return send_alert_tls13(tls, level, desc);

		default:
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 9, NULL);
			return false;
		}
	}
}

bool tls_alert_recv(TLS *tls, const struct tls_record *record) {
	TLS_DPRINTF("alert: recv");

	if (tls->active_read.closed == true) {
		TLS_DPRINTF("alert: already closed");
		OK_set_error(ERR_ST_TLS_CONNECTION_CLOSED,
			     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 10, NULL);
		return false;
	}

	if (tls->state < TLS_STATE_HS_AFTER_RECV_SHELLO ||
		tls->state == TLS_STATE_HS_BEFORE_RECV_CHELLO) {
		return recv_alert_up_to_tls12(tls, record);
	} else {
		/* version must be negotiated */
		uint16_t version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
			return recv_alert_up_to_tls12(tls, record);

		case TLS_VER_TLS13:
			return recv_alert_tls13(tls, record);

		default:
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS1, ERR_PT_TLS_ALERT + 11, NULL);
			return false;
		}
	}
}
