/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_27B04159_DC20_888C_141E_ED533DC8A1A7
#define INCLUSION_GUARD_UUID_27B04159_DC20_888C_141E_ED533DC8A1A7

#include "tls.h"

/**
 * get current epoch time
 *
 * @return epoch time (UTC base)
 */
uint64_t tls_util_get_epochtime(void);

/**
 * fill buffer with a random value.
 *
 * @param buf buffer to fill ramdom value.
 * @param len length of buffer
 */
bool tls_util_get_random(uint8_t *buf, uint32_t len);

/**
 * write 2 byte to the array.
 *
 * @param buf array of the writing destination.
 * @param val value to write.
 */
void tls_util_write_2(uint8_t *buf, int32_t val);

/**
 * write 2 byte to the array.
 *
 * @param buf array of the writing destination.
 * @param val value to write.
 */
void tls_util_write_3(uint8_t *buf, int32_t val);

/**
 * read 2 byte from the array.
 *
 * @param buf array of the reading source.
 */
uint16_t tls_util_read_2(uint8_t *buf);

/**
 * read 3 byte from the array.
 *
 * @param buf array of the reading source.
 */
uint32_t tls_util_read_3(uint8_t *buf);

/**
 * convert version format from uint16_t to struct tls_protocol_version.
 *
 * @param version version of uint16_t format.
 * @param version_st destination version of struct tls_protocol_version format.
 */
void tls_util_convert_ver_to_protover(uint16_t version,
				      struct tls_protocol_version *version_st);

/**
 * convert version format from struct tls_protocol_version to uint16_t.
 *
 * @param version_st version of struct tls_protocol_version format.
 * @return version of uint16_t format.
 */
uint16_t tls_util_convert_protover_to_ver(struct tls_protocol_version *version_st);

/**
 * Check whether given version is included in struct tls_protocol_version_list.
 */
bool tls_util_check_version_in_supported_version(
	struct tls_protocol_version_list *vlist,
	uint16_t version);

/**
 * do PKCS1 encryption.
 *
 * @param pubkey  public key structure.
 * @param dest    destination buffer that stores result of rsa encryption.
 * @param src     source buffer that do rsa encryption.
 * @param src_len length of src.
 */
int32_t tls_util_pkcs1_encrypt(Key *pubkey,
			       uint8_t *dest, uint8_t *src, int32_t src_len);

/**
 * do PKCS1 decryption.
 *
 * @param privkey private key structure.
 * @param dest    destination buffer that stores result of rsa decryption.
 * @param src     source buffer that was encrypted by RSA.
 * @param src_len length of src.
 */
int32_t tls_util_pkcs1_decrypt(Key *privkey, uint8_t *dest,
			       const uint8_t *src, const int32_t src_len);

/**
 * get ASN.1 length. this is a wrapper of aicrypto's ASN1_length
 * function.
 *
 * @param  in 1st argument of ASN_length.
 * @return length of ASN.1.
 */
int32_t tls_util_asn1_length(uint8_t *in);

/* wrapper for gcc/clang extension. */
#define UNUSED	__attribute__((unused))

/* debug output macro.
 *
 * if you want to output debug purpose message, enable following
 * defines.
 *
 *  #define TLS_DEBUG	1
 *
 * debug purpose messages are outputed to stderr. */

#ifdef TLS_DEBUG
#include <stdio.h>
#include <libgen.h>

#define TLS_DPRINTF(format, ...)					\
	do {								\
		fprintf(stderr, "%-15s (%4u) @%-30s: ",			\
			basename(__FILE__), __LINE__, __func__);	\
		fprintf(stderr, format, ## __VA_ARGS__);		\
		fprintf(stderr, "\n");					\
	} while (0)
#else
#define TLS_DPRINTF(format, ...)
#endif

#endif /* INCLUSION_GUARD_UUID_27B04159_DC20_888C_141E_ED533DC8A1A7 */
