#ifndef INCLUSION_GUARD_UUID_C8CF23DF_D66E_3A68_BA46_A5183F271F99
#define INCLUSION_GUARD_UUID_C8CF23DF_D66E_3A68_BA46_A5183F271F99

#include "tls.h"

/* for TAILQ_ENTRY */
#include <aicrypto/nrg_queue.h>

/**
 * TLS session parameter
 *
 * this structure stores session information for re-session use. this
 * structure is based on security parameter in RFC 5246 section 6.1. in
 * here, in comparison with security parameter, following point are
 * diffrenet.
 *
 * 1. following parameter moved right under the TLS structure. it is not
 *    necessary to remember these parameters in the re-session.
 *
 *    - connection_end
 *    - client_random
 *    - server_random
 *
 * 2. following parameter moved to tls_cipher_param structure. these
 *    parameter can be derived by enum cipher_suite member in this
 *    structure.
 *
 *    - prf_algorithm
 *    - cipher_algorithm
 *    - cipher_type
 *    - enc_key_length
 *    - block_length
 *    - record_iv_length
 *    - mac_algorithm
 *    - mac_length
 *    - mac_key_length
 *
 * 3. following parameter was added. it is necessary to remember these
 *    parameter for re-session.
 *
 *    - session_id
 *    - session_id_length
 *    - last_used_epochtime
 *    - references
 *    - link
 *
 * by this structure, tls_session_param may be free at any time if not
 * used by handshake. i.e, may free if references == 0.
 */
struct tls_session_param {
	/** link for next queue */
	TAILQ_ENTRY(tls_session_param) link;

	/** cipher suite to be used for data encryption.  */
	enum tls_cipher_suite cipher_suite;

	/** algorithm to be used for data compression. */
	enum compression_algorithm compression_algorithm;

	/** 48-byte secret shared between the two peers in the connection. */
	uint8_t master_secret[48];

	/** the length of session_id (0 .. 32) */
	uint8_t session_id_length;

	/** data of session_id. allocate max session id length bytes
	 * (32) by fixation. the real length (used length) is controlled
	 * by session_id_length. */
	uint8_t session_id[32];

	/** the epochtime that session was created.
	 *
	 * store this variable because RFC 5246 section F 1.4 says "An
	 * upper limit of 24 hours is suggested for session ID
	 * lifetimes, since an attacker who obtains a master_secret may
	 * be able to impersonate the compromised party until the
	 * corresponding session ID is retired.".
	 */
	uint64_t created_epochtime;

	/** the number that refer to this session in the handshake (it
	 * must not free session during the handshake). */
	int32_t references;

	/** whether session has been disabled.  if session is disabled,
	 * that session can not be re-use. */
	bool disabled;
};

/**
 * allocate new session.
 *
 * this function is called in handshake process. after the call of this
 * function, usually, tls_session_refer function is called to refer
 * allocated session. in the last of handshake, when tls_session_unrefer
 * is called, if state of session is "enable", allocated session is
 * added to the list session_list (@see tls_sessin.c). otherwise (state
 * of session is disabled), session is free-ed by calling
 * tls_session_free function in the internal of tls_session_unrefer
 * function.
 */
struct tls_session_param * tls_session_new(void);

/**
 * free memory that was allocated by tls_session_new function.
 */
bool tls_session_free(struct tls_session_param *session);

/**
 * find session by session id.
 *
 * this function intend to be used in server side (parsing server
 * hello). if session that matches to specified session id was found,
 * return that session in the sessin_list queue (of tls_session.c).
 */
struct tls_session_param * tls_session_find_by_id(const uint8_t *session_id,
						  const uint32_t len);

/**
 * mark the session as referred state.
 *
 * this means that the session must not be free because session is used
 * by any handshake. */
void tls_session_refer(struct tls_session_param *session);

/**
 * mark the session as unreferred state.
 *
 * this means that the session can be free because session is not used
 * by any handshake.
 */
void tls_session_unrefer(struct tls_session_param *session);

/**
 * mark the session as effective.
 *
 * if the handshake is successfully finished, this function is called.
 */
void tls_session_enable(struct tls_session_param *session);

/**
 * mark the session as uneffective.
 *
 * if the handshake is abnormally finished (e.g. by receiving fatal
 * alert), this function is called. this session is destroyed in the
 * near future.
 */
void tls_session_disable(struct tls_session_param *session);

#endif /* INCLUSION_GUARD_UUID_C8CF23DF_D66E_3A68_BA46_A5183F271F99 */
