/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_8149D790_D74E_4CB7_3F08_07B787F5A0EB
#define INCLUSION_GUARD_UUID_8149D790_D74E_4CB7_3F08_07B787F5A0EB

#include "tls.h"

/* for TAILQ */
#include <aicrypto/nrg_queue.h>

/**
 * available record protocols.
 *
 * this enum is defined in RFC 5246 section 6.2.1.
 */
enum tls_record_ctype {
	/** change cipher spec */
	TLS_CTYPE_CHANGE_CIPHER_SPEC = 20U,

	/** alert */
	TLS_CTYPE_ALERT = 21U,

	/** handshake */
	TLS_CTYPE_HANDSHAKE = 22U,

	/** application data */
	TLS_CTYPE_APPLICATION_DATA   = 23U
};

/**
 * structure that handle tls record data.
 */
struct tls_record {
	TAILQ_ENTRY(tls_record)  link;

	/** type of record protocol */
	enum tls_record_ctype type;

	/** length of fragment (frag member) */
	uint32_t len;

	/** TLS/SSL version  */
	struct tls_protocol_version  ver;

	/** fragment data */
	uint8_t *frag;
};

/**
 * TAILQ of tls_record structure.
 */
struct tls_record_queue {
	/**
	 * indicates point of one record (in queue) where it handled the
	 * record data by tls_record_read function.
	 *
	 * for example, this is as follows.
	 *
	 * 1. read 1 record. and store that record to queue.
	 * 2. parse top 4 byte of that record (queue-ed record).
	 * 3. pos become 4.
	 *
	 * if one record (in queue) is parsed fully, pos will become
	 * zero.
	 *
	 * image is as follows.
	 *
	 * @verbatim
	 *  <Q>
	 *   |       pos
	 *   |_      v
	 *   | | record: 1 (length : 10, handled: 4) |
	 *   | 0     4                              10
	 *   |_
	 *   | | record: 2 (length : 40, handled: 0) |
	 *   | 0                                    40
	 *   \_
	 *     | record: 3 (length : 30, handled: 0) |
	 *     0                                    30
	 * @endverbatim
	 */
	ssize_t pos;

	TAILQ_HEAD(tls_record_head, tls_record) *head;
};

/**
 * maximum size of plain fragment.
 */
#define TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX	(2 << (14 - 1))

/**
 * maximum size of compressed fragment.
 */
#define TLS_RECORD_COMPRESSED_FRAGMENT_SIZE_MAX	((2 << (14 - 1)) + 1024)

/**
 * maximum size of ciphered fragment.
 */
#define TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_UP_TO_TLS12	((2 << (14 - 1)) + 2048)

#define TLS_RECORD_CIPHERED_FRAGMENT_SIZE_MAX_TLS13		((2 << (14 - 1)) + 256)

#define TLS_PADDING_VALUE_TLS13		(0)

/**
 * full-size records can be safely encrypted up to 2^24.5 for AES-GSM.
 * 2^24.5, ie, 2^24 * sqrt(2) is 23726566.
 */
#define TLS_AEAD_RECORD_LIMIT_AES_GCM		(23726566)

/**
 * max sequence number of tls record including safety margin in TLS 1.3.
 *
 * sequence number of tls record in TLS 1.3 is 64 bit, so it ranges from 0
 * to 2^64 - 1(i.e. 18446744073709551615). to take a margin againt the max
 * sequence number, subtract 2^20(i.e. 1048576) from the number here.
 */
#define TLS_RECORD_SEQNUM_SIZE_MAX_TLS13	(18446744073709551615U - 1048576U)

/**
 * initialize the structure that manage record data.
 *
 * this function returns memory allocated structure. so, if that
 * structure became unnecessary, you must free by tls_record_free
 * function.
 */
struct tls_record_queue * tls_record_init(void);

void tls_record_free(struct tls_record_queue *queue);

/**
 * read tls record data.
 */
ssize_t tls_record_read(enum tls_record_ctype type,
			TLS *tls, uint8_t *buf, size_t count);

/**
 * write tls record data.
 */
ssize_t tls_record_write(enum tls_record_ctype type,
			 TLS *tls, const uint8_t *buf, size_t count);

#endif /* INCLUSION_GUARD_UUID_8149D790_D74E_4CB7_3F08_07B787F5A0EB */
