/*
 * Copyright (c) 2016-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_99A9297C_1E5E_4799_BC69_EF85B383D1A7
#define INCLUSION_GUARD_UUID_99A9297C_1E5E_4799_BC69_EF85B383D1A7

#include "tls.h"
#include "extension/tls_ecc.h"

/* for Pubkey_ECDSA, Prvkey_ECDSA. */
#include <aicrypto/ok_ecdsa.h>

/* for Pubkey_X25519, Prvkey_X25519, Pubkey_X448, Prvkey_X448. */
#include <aicrypto/nrg_edc.h>

/**
 * Ephemeral ECDH key.
 */
struct tls_hs_ecdh_key {

	/**
	 * ECPoint length.
	 */
	uint8_t ecpoint_len;

	/**
	 * ECPoint.point buffer.
	 */
	uint8_t ecpoint[256];

	/**
	 * ephemeral ECDH public key.
	 */
	Key *pub;

	/**
	 * ephemeral ECDH private key.
	 */
	Key *prv;
};

/**
 * Generate ephemeral ECDH key on server.
 */
bool tls_hs_ecdhkey_gen_for_server(enum tls_hs_named_curve curve,
				   struct tls_hs_ecdh_key *key);

/**
 * Set ephemeral ECDH public key to ECPoint.point.
 *
 * The public key stored in the structure to convert to ECPoint format.
 */
bool tls_hs_ecdhkey_set_to_ecpoint(struct tls_hs_ecdh_key *key);

/**
 * Get ephemeral ECDH public key from ECPoint.point.
 *
 * Restoring the public key from the ECPoint data stored in the 'key',
 * and stores to the 'ctx->peer_pubkey'.
 *
 * @param[out] ctx  To store the public key.
 * @param[in]  key  Public key read from ServerECDHParams.public.
 */
bool tls_hs_ecdhkey_set_peer_pubkey(struct tls_hs_ecdh *ctx,
				    struct tls_hs_ecdh_key *key);

/**
 * Generate ephemeral ECDH key on client.
 *
 * RFC4492 section 5.7.
 *
 * Actions of the sender:
 *
 * The client selects an ephemeral ECDH public key corresponding to the
 * parameters it received from the server according to the ECKAS-DH1
 * scheme from IEEE 1363 [6].  It conveys this information to the client
 * in the ClientKeyExchange message using the format defined above.
 */
bool tls_hs_ecdhkey_gen_for_client(Key *peer_pubkey,
				   struct tls_hs_ecdh_key *key);

/**
 * Get a private key of the PKCS12 and save to the structure.
 */
bool tls_hs_ecdhkey_set_my_privkey_from_pkcs12(struct tls_hs_ecdh *ctx,
					       PKCS12 *p12);

/**
 * Get an ECDH-capable public key of the PKCS12 and save to the structure.
 */
bool tls_hs_ecdhkey_set_peer_pubkey_from_pkcs12(struct tls_hs_ecdh *ctx,
						PKCS12 *p12);

#endif /* INCLUSION_GUARD_UUID_99A9297C_1E5E_4799_BC69_EF85B383D1A7 */
