/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_95C347E7_6082_FDC1_89B7_A49DB5FBB138
#define INCLUSION_GUARD_UUID_95C347E7_6082_FDC1_89B7_A49DB5FBB138

#include "tls.h"
#include "tls_session.h"
#include "extension/tls_sighash.h"
#include "extension/tls_ecc.h"
#include "extension/tls_supported_versions.h"
#include "extension/tls_keyshare.h"
#include "extension/tls_servername.h"
#include "extension/tls_cookie.h"

/* for PKCS12. */
#include <aicrypto/ok_pkcs12.h>

/* for Pubkey_ECDSA, Prvkey_ECDSA. */
#include <aicrypto/ok_ecdsa.h>

#include <stdint.h>

/* for TAILQ */
#include <aicrypto/nrg_queue.h>

/**
 * @see tls_cert.h
 */
struct tls_cert_info;

/**
 * available handshake protocol messages.
 *
 * this enum is defined in RFC 5246 section 7.4.
 *
 * RFC4347 4.3.2. Handshake Protocol
 *
 *       enum {
 *         hello_request(0), client_hello(1), server_hello(2),
 *         hello_verify_request(3),                          // New field
 *         certificate(11), server_key_exchange (12),
 *         certificate_request(13), server_hello_done(14),
 *         certificate_verify(15), client_key_exchange(16),
 *         finished(20), (255)
 *       } HandshakeType;
 *
 * RFC4680 2.  Supplemental Data Handshake Message
 *
 *       enum {
 *              supplemental_data(23), (255)
 *          } HandshakeType;
 *
 * RFC6066 2.  Extensions to the Handshake Protocol
 *
 *    enum {
 *        hello_request(0), client_hello(1), server_hello(2),
 *        certificate(11), server_key_exchange (12),
 *        certificate_request(13), server_hello_done(14),
 *        certificate_verify(15), client_key_exchange(16),
 *        finished(20), certificate_url(21), certificate_status(22),
 *        (255)
 *    } HandshakeType;
 *
 * RFC8446 B.3.  Handshake Protocol
 *
 *       enum {
 *           hello_request_RESERVED(0),
 *           client_hello(1),
 *           server_hello(2),
 *           hello_verify_request_RESERVED(3),
 *           new_session_ticket(4),
 *           end_of_early_data(5),
 *           hello_retry_request_RESERVED(6),
 *           encrypted_extensions(8),
 *           certificate(11),
 *           server_key_exchange_RESERVED(12),
 *           certificate_request(13),
 *           server_hello_done_RESERVED(14),
 *           certificate_verify(15),
 *           client_key_exchange_RESERVED(16),
 *           finished(20),
 *           certificate_url_RESERVED(21),
 *           certificate_status_RESERVED(22),
 *           supplemental_data_RESERVED(23),
 *           key_update(24),
 *           message_hash(254),
 *           (255)
 *       } HandshakeType;
 */
enum tls_handshake_type {
	/** hello request */
	TLS_HANDSHAKE_HELLO_REQUEST =  0U,

	/** client hello */
	TLS_HANDSHAKE_CLIENT_HELLO =  1U,

	/** server hello  */
	TLS_HANDSHAKE_SERVER_HELLO =  2U,

	/** hello verify request defined by RFC4347 */
	TLS_HANDSHAKE_HELLO_VERIFY_REQUEST = 3U,

	/** new session ticket defined by RFC8446 */
	TLS_HANDSHAKE_NEW_SESSION_TICKET = 4U,

	/** end of early data defined by RFC8446 */
	TLS_HANDSHAKE_END_OF_EARLY_DATA = 5U,

	/** hello retry request reserved by RFC8446 */
	TLS_HANDSHAKE_HELLO_RETRY_REQUEST = 6U,

	/** encrypted extensions defined by RFC8446 */
	TLS_HANDSHAKE_ENCRYPTED_EXTENSIONS = 8U,

	/** certificate  */
	TLS_HANDSHAKE_CERTIFICATE = 11U,

	/** server key exchange  */
	TLS_HANDSHAKE_SERVER_KEY_EXCHANGE = 12U,

	/** certificate request */
	TLS_HANDSHAKE_CERTIFICATE_REQUEST = 13U,

	/** server hello done */
	TLS_HANDSHAKE_SERVER_HELLO_DONE = 14U,

	/** certificate verify  */
	TLS_HANDSHAKE_CERTIFICATE_VERIFY = 15U,

	/* client key exchange */
	TLS_HANDSHAKE_CLIENT_KEY_EXCHANGE = 16U,

	/* finished */
	TLS_HANDSHAKE_FINISHED = 20U,

	/** certificate url defined by RFC6066 */
	TLS_HANDSHAKE_CERTIFICATE_URL = 21U,

	/** certificate status defined by RFC6066 */
	TLS_HANDSHAKE_CERTIFICATE_STATUS = 22U,

	/** supplemental data defined by RFC4680 */
	TLS_HANDSHAKE_SUPPLEMENTAL_DATA = 23U,

	/** key update defined by RFC8446 */
	TLS_HANDSHAKE_KEY_UPDATE = 24U,

	/** message hash defined by RFC8446 */
	TLS_HANDSHAKE_MESSAGE_HASH = 254U
};

/**
 * available client hello handshake protocol message extensions.
 *
 * this enum is defined in
 * RFC 5246 7.4.1.4. Hello Extensions
 *    enum {
 *        signature_algorithms(13), (65535)
 *    } ExtensionType;
 *
 * RFC 4492 5.1.  Client Hello Extensions
 *     enum { elliptic_curves(10), ec_point_formats(11) } ExtensionType;
 */
enum tls_extension_type {
	/** server name indication extension. */
	TLS_EXT_SERVER_NAME = 0U,

	/** maximum fragment length negotiation extention. */
	TLS_EXT_MAX_FRAGMENT_LENGTH = 1U,

	/** certificate status request extension. */
	TLS_EXT_STATUS_REQUEST = 5U,

	/** Supported Elliptic Curves Extension. */
	TLS_EXT_ELLIPTIC_CURVES = 10U,

	/** supported groups extension(For TLS 1.3). */
	TLS_EXT_SUPPORTED_GROUPS = 10U,

	/** Supported Point Formats Extension. */
	TLS_EXT_EC_POINT_FORMATS = 11U,

	/** signature algorithm extension. */
	TLS_EXT_SIGNATURE_ALGO = 13U,

	/** use srtp extension. */
	TLS_EXT_USE_SRTP = 14U,

	/** heartbeat extension. */
	TLS_EXT_HEARTBEAT = 15U,

	/** application-layer protocol negotiation extension. */
	TLS_EXT_APP_LAYER_PROTO_NEGOTIATION = 16U,

	/** signed certificate timestamp extension. */
	TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP = 18U,

	/** client certificate type extension. */
	TLS_EXT_CLIENT_CERTIFICATE_TYPE = 19U,

	/** server certificate type extension. */
	TLS_EXT_SERVER_CERTIFICATE_TYPE = 20U,

	/** padding extension. */
	TLS_EXT_PADDING = 21U,

	/** pre-shared key extension. */
	TLS_EXT_PRE_SHARED_KEY = 41U,

	/** early data indication extension. */
	TLS_EXT_EARLY_DATA = 42U,

	/** supported versions extension. */
	TLS_EXT_SUPPORTED_VERSIONS = 43U,

	/** cokkie extension. */
	TLS_EXT_COOKIE = 44U,

	/** pre-shared key exchange modes extension. */
	TLS_EXT_PSK_KEY_EXCHANGE_MODES = 45U,

	/** certificate authorities extension. */
	TLS_EXT_CERTIFICATE_AUTHORITIES = 47U,

	/** oid filters extension. */
	TLS_EXT_OID_FILTERS = 48U,

	/** post-handshake client authentication extension. */
	TLS_EXT_POST_HANDSHAKE_AUTH = 49U,

	/** signature algorithsm certificate extension. */
	TLS_EXT_SIGNATURE_ALGO_CERT = 50U,

	/** key share extension. */
	TLS_EXT_KEY_SHARE = 51U,

	/** this is not a extension, just the end of definition */
	TLS_EXT_MAX,
};

/**
 * RFC8446 4.6.3.  Key and Initialization Vector Update
 *
 *       enum {
 *           update_not_requested(0), update_requested(1), (255)
 *       } KeyUpdateRequest;
 */
enum tls_keyupdate_request {
	/* not requested to send KeyUpdate message. */
	TLS_KEYUPDATE_NOT_REQUESTED = 0U,

	/* request to send KeyUpdate message. */
	TLS_KEYUPDATE_REQUESTED = 1U,
};

/**
 * structure that stores extension data.
 */
struct tls_extension {
	TAILQ_ENTRY(tls_extension) link;

	/* type of extension */
	enum tls_extension_type type;

	/* length of extension data */
	uint16_t len;

	/* extension data */
	uint8_t *opaque;
};

/**
 * structure that stores interim parameters.
 */
struct tls_hs_interim_params {
	/* protocol version */
	struct tls_protocol_version version;

	/* raw random value in client hello handshake message */
	uint8_t client_random[32];

	/* raw session value length */
	uint8_t seslen;

	/* raw session value in handshake message */
	uint8_t *session;

	/* raw cipher_suite values in client hello handshake message */
	struct tls_cipher_list *cipher_suites;

	/* raw cipher_suite value in server hello handshake message */
	uint16_t cipher_suite;

	/* raw compression method length */
	uint8_t cmplen;

	/* raw compression method values in handshake message */
	uint8_t *cmplist;

	/* extension list */
	TAILQ_HEAD(tls_extension_head, tls_extension) head;

	/* extensions sent to peer */
	bool sent_ext_flags[TLS_EXT_MAX];

	/* extensions received from peer */
	bool recv_ext_flags[TLS_EXT_MAX];

	/* key share list */
	TAILQ_HEAD(tls_hs_key_share_head, tls_hs_key_share) share_head;
};

/**
 * structure that handle handshake data.
 */
struct tls_hs_msg {
	TAILQ_ENTRY(tls_hs_msg) link;

	/* type of handshake message */
	enum tls_handshake_type type;

	/* length of *msg */
	uint32_t  len;

	/* allocated size of *msg */
	uint32_t  max;

	/* message body */
	uint8_t  *msg;
};

/**
 * TAILQ of tls_hs_msg structure.
 */
TAILQ_HEAD(tls_handshake_queue, tls_hs_msg);

/**
 * phase that is used to manage state of handshake.
 */
enum hs_phase {
	/** phase for client hello and server hello. */
	TLS_HS_PHASE_HELLO,

	/**
	 * phase for receiving certificate related message.
	 *
	 * TODO: probably, this name is not accurate because there may
	 * not be treated certificate handshake protocol message in the
	 * handshake.
	 */
	TLS_HS_PHASE_CERTIFICATE_RECVING,

	/**
	 * phase for sending certificate related message.
	 *
	 * TODO: probably, this name is not accurate because there may
	 * not be treated certificate handshake protocol message in the
	 * handshake.
	 */
	TLS_HS_PHASE_CERTIFICATE_SENDING,

	/** phase for ccs and finished. */
	TLS_HS_PHASE_FINAL,

	/** phase that the handshake completed. */
	TLS_HS_PHASE_DONE,

	/** phase that the handshake failed. */
	TLS_HS_PHASE_FAILD,
};

/**
 * structure of perfect forward secrecy.
 */
struct tls_hs_ecdh {
	/**
	 * list of peer Elliptic Curve sent by peer.
	 */
	struct tls_hs_ecc_eclist *peer_eclist;

	/**
	 * list of available Elliptic Curve.
	 */
	struct tls_hs_ecc_eclist *eclist;

	/**
	 * list of available Point Format.
	 */
	struct tls_hs_ecc_pflist *pflist;

	/**
	 * The type of the elliptic curve domain parameters.
	 */
	enum tls_hs_ecc_ec_curve_type curvetype;

	/**
	 * Determined value when received Client Hello Extension.
	 *
	 * 'curve_type' MUST be set TLS_ECC_CTYPE_NAMED_CURVE.
	 */
	enum tls_hs_named_curve namedcurve;

	/**
	 * ECDH public key genereated on peer.
	 */
	Key *peer_pubkey;

	/**
	 * True if the public key is ephemeral.
	 */
	bool pubkey_ephemeral;

	/**
	 * ECDH private key genereated on local.
	 */
	Key *my_prvkey;

	/**
	 * True if the private key is ephemeral.
	 */
	bool prvkey_ephemeral;
};

/**
 * length of handshake header (msg_type : 1byte and length : 3byte)
 */
#define TLS_HANDSHAKE_HEADER_LENGTH	4

/**
 * maximum length of extension in each messages.
 */
#define TLS_EXT_SIZE_MAX	((2 << (16 - 1)) - 1)

/**
 * initialize structure that stores extension data.
 *
 * this function returns memory allocated structure. so, if that
 * structure became unnecessary, you must free by tls_extension_free
 * function.
 */
struct tls_extension *tls_extension_init(void);

void tls_extension_free(struct tls_extension *ext);

/**
 * initialize structure that stores interim parameters.
 *
 * this function returns memory allocated structure. so, if that
 * structure became unnecessary, you must free by
 * tls_hs_interim_params_free function.
 */
struct tls_hs_interim_params *tls_hs_interim_params_init(void);

void tls_hs_interim_params_free(struct tls_hs_interim_params *params);

/**
 * change state of tls communication.
 */
void tls_hs_change_state(TLS *tls, enum tls_state state);

/**
 * check whether the tls communication state is supposed state.
 */
bool tls_hs_check_state(TLS *tls, enum tls_state state);

/**
 * select appropriate protocol version based on a client hello message.
 */
bool tls_hs_version_select(TLS *tls);

/**
 * validate protocol version based on a server shello message.
 */
bool tls_hs_version_validate(TLS *tls);

/**
 * update hash by data in tls->queue_handshake.
 *
 * hash is used by certificate verify and finished handshake protocol
 * message.
 */
void tls_hs_update_hash(TLS *tls);

/**
 * write handshake data to the peer.
 *
 * this is a wrapper of tls_record_write. this function split the
 * handshake data into maximum record size and write it.
 *
 * if this function fails, you must free tls_hs_msg structure
 * by tls_hs_msg_free().
 */
bool tls_handshake_write(TLS *tls, struct tls_hs_msg * msg);

/**
 * read handshake data from the peer.
 *
 * this is a wrapper of tls_record_read. this function combine the
 * record data into the one handshake data and return the that data.
 */
struct tls_hs_msg * tls_handshake_read(TLS *tls);

/**
 * free allocated structures that are mainly used in handshake module.
 */
void tls_handshake_free(TLS *tls);

/**
 * start client side handshake.
 */
bool tls_hs_cs_client_handshake(TLS *tls);

/**
 * start server side handshake.
 */
bool tls_hs_cs_server_handshake(TLS *tls);

/**
 * initialize the structure that manage handshake data.
 *
 * length of message is allocated the max record size by default. if
 * message that got by this function became unnecessary, you must free
 * allocated memory by tls_hs_msg_free function.
 */
struct tls_hs_msg * tls_hs_msg_init(void);

/**
 * free memory that is allocated by tls_hs_msg_init.
 */
void tls_hs_msg_free(struct tls_hs_msg *msg);

/**
 * write 1 byte to the data that is allocated by tls_hs_init.
 *
 * if message length of data was fill, realloc the message internally.
 * realloced size is multiple of max record size.
 */
bool tls_hs_msg_write_1(struct tls_hs_msg *msg, const uint8_t dat);

/**
 * write 2 byte to the data that is allocated by tls_hs_init.
 *
 * if message length of data was fill, realloc the message internally.
 * realloced size is multiple of max record size.
 */
bool tls_hs_msg_write_2(struct tls_hs_msg *msg, const uint16_t dat);

/**
 * write 3 byte to the data that is allocated by tls_hs_init.
 *
 * if message length of data was fill, realloc the message internally.
 * realloced size is multiple of max record size.
 */
bool tls_hs_msg_write_3(struct tls_hs_msg *msg, const uint32_t dat);

/**
 * write n byte to the data that is allocated by tls_hs_init.
 *
 * if message length of data was fill, realloc the message internally.
 * realloced size is multiple of max record size.
 */
bool tls_hs_msg_write_n(struct tls_hs_msg *msg,
			const uint8_t *dat, const uint32_t len);

/**
 * initialize hash that is used by certificate verify and finished
 * handshake protocol message.
 *
 * "initalize" means that this function call XXXInit function
 * (e.g. MD5Init, SHA1init and so on ...).
 */
void tls_hs_hash_init(TLS *tls);

/**
 * update hash that is used by certificate verify and finished
 * handshake protocol message.
 *
 * the difference of tls_hs_update_hash is not to scan the
 * tls->queue_handshake. this function treats one data.
 */
void tls_hs_hash_update(TLS *tls, uint8_t *buf, uint32_t len);

/**
 * get current handshake digest of specified hash algorithm.
 *
 * handshake digest is a finalized hash of tls handshake that is used by
 * certificate request and finished handshake protocol message.
 */
void tls_hs_hash_get_digest(const enum tls_hs_sighash_hash_algo hash,
			    const TLS *tls, uint8_t *seed);

/**
 * Get a signed hash of the ServerKeyExchange.params.
 *
 * RFC4492 section 5.4.
 *
 *  ServerKeyExchange.signed_params.sha_hash
 *      SHA(ClientHello.random + ServerHello.random +
 *                                        ServerKeyExchange.params);
 *
 * Refernce the following struct's members:
 *
 * Data                      | Member of TLS
 * ------------------------- | -------------------------
 * ClientHello.random        | client_random
 * ServerHello.random        | server_random
 * ServerKeyExchange.params  | skeyexc_params,
 *                           | skeyexc_params_len
 */
void tls_hs_signature_get_digest(const enum tls_hs_sighash_hash_algo hash,
				 TLS *tls, uint8_t *seed);

/**
 * Convert an ECParam curve type to enum NamedCurve.
 */
enum tls_hs_named_curve tls_hs_ecdh_get_named_curve(int curve_type);

/**
 * Convert an ECParam curve type to enum ECPointFormat.
 */
enum tls_hs_ecc_ec_point_format tls_hs_ecdh_get_point_format(int curve_type);

/**
 * Convert an ECParam curve type to enum ECCurveType.
 */
enum tls_hs_ecc_ec_curve_type tls_hs_ecdh_get_curve_type(int curve_type);

/**
 * Calculate the premaster secret.
 */
int32_t tls_hs_ecdh_calc_shared_secret(TLS *tls, struct tls_hs_ecdh *ctx);

/**
 * Allocate memeory for TLS.ecdh.
 */
bool tls_hs_ecdh_alloc(TLS *tls);

/**
 * Free memory of struct tls_hs_ecdh.
 */
void tls_hs_ecdh_free(struct tls_hs_ecdh *ecdh);

/**
 * Generate ephemeral ECDH public key.
 */
bool tls_hs_ecdhkey_generate(struct tls_hs_ecdh *ctx);

/**
 * Write octet string representation of the ephemeral ECDH public key.
 */
int32_t tls_hs_ecdhkey_write(struct tls_hs_ecdh *ctx,
			     struct tls_hs_msg *msg);

/**
 * Read an ephemeral ECDH public key from octet string.
 */
bool tls_hs_ecdhkey_read(struct tls_hs_ecdh *ctx, uint8_t *oct,
			 const uint8_t len);

/**
 * Write client ECDH public to the send data.
 *
 * RFC4492 section 5.7.
 *  struct {
 *       select (PublicValueEncoding) {
 *           case implicit: struct { };
 *           case explicit: ECPoint ecdh_Yc;
 *       } ecdh_public;
 *  } ClientECDiffieHellmanPublic;
 */
int32_t tls_hs_ecdh_ckeyexc_write_exchange_keys(TLS *tls,
						struct tls_hs_msg *msg);

/**
 * Read ClientKeyExchange.exchange_keys from message buffer.
 */
int32_t tls_hs_ecdh_ckeyexc_read_exchange_keys(TLS *tls,
					       const struct tls_hs_msg *msg,
					       const uint32_t offset);

/**
 * Write ServerECDHParams to ServrKeyExchange message buffer.
 *
 * RFC4492 section 5.4.
 *
 *  struct {
 *      ECParameters    curve_params;
 *      ECPoint         public;
 *  } ServerECDHParams;
 */
int32_t tls_hs_ecdh_skeyexc_write_server_params(TLS *tls,
						struct tls_hs_msg *msg);

/**
 * Read ServerECDHParams from ServrKeyExchange message buffer.
 */
int32_t tls_hs_ecdh_skeyexc_read_server_params(TLS *tls, struct tls_hs_msg *msg,
					       const uint32_t offset);

/**
 * Get a private key of the PKCS12 and save to the structure.
 */
bool tls_hs_ecdh_set_privkey_from_pkcs12(TLS *tls, PKCS12 *p12);

/**
 * Get an ECDH-capable public key of the PKCS12 and save to the structure.
 */
bool tls_hs_ecdh_set_pubkey_from_pkcs12(TLS *tls, PKCS12 *p12);

/**
 * Set a elliptic curve that to be used to struct tls_hs_ecdh.
 */
void tls_hs_ecdh_set_curve(struct tls_hs_ecdh *ecdh);

/**
 * Set a elliptic curve that to be used to struct tls_hs_ecdh.
 */
void tls_hs_ecdh_set_curve_by_cert(struct tls_hs_ecdh *ecdh,
				   struct tls_cert_info *cinfo);

/**
 * Return true if there is a need to send the ServerKeyExchange message.
 *
 * See the following documents:
 * - RFC5246 7.4.3.  Server Key Exchange Message
 * - RFC4429 5.4.  Server Key Exchange
 */
bool tls_hs_skeyexc_need_to_send(TLS *tls);

/**
 * compose hello request handshake protocol message.
 */
struct tls_hs_msg * tls_hs_helloreq_compose(TLS *tls);

/**
 * compose client hello handshake protocol message.
 */
struct tls_hs_msg * tls_hs_chello_compose(TLS *tls);

/**
 * compose server hello handshake protocol message.
 */
struct tls_hs_msg * tls_hs_shello_compose(TLS *tls);

/**
 * compose ecrypted extensions handshake protocol message.
 */
struct tls_hs_msg *tls_hs_encext_compose(TLS *tls UNUSED);

/**
 * compose (server) certificate handshake protocol message.
 */
struct tls_hs_msg * tls_hs_scert_compose(TLS *tls);

/**
 * compose server key exchange handshake protocol message.
 */
struct tls_hs_msg * tls_hs_skeyexc_compose(TLS *tls);

/**
 * compose certificate request handshake protocol message.
 */
struct tls_hs_msg * tls_hs_certreq_compose(TLS *tls);

/**
 * compose server hello done handshake protocol message.
 */
struct tls_hs_msg * tls_hs_shellodone_compose(TLS *tls);

/**
 * compose (client) certificate handshake protocol message.
 */
struct tls_hs_msg * tls_hs_ccert_compose(TLS *tls);

/**
 * compose client key exchange handshake protocol message.
 */
struct tls_hs_msg * tls_hs_ckeyexc_compose(TLS *tls);

/**
 * compose certificate verify handshake protocol message.
 */
struct tls_hs_msg * tls_hs_certvfy_compose(TLS *tls);

/**
 * compose finished handshake protocol message.
 */
struct tls_hs_msg * tls_hs_finished_compose(TLS *tls);

/**
 * make message hash handshake protocol message.
 */
struct tls_hs_msg *tls_hs_msghash_make(TLS *tls);

/**
 * compose message key update handshake protocol message.
 */
struct tls_hs_msg *tls_hs_keyupdate_compose(TLS *tls,
					    enum tls_keyupdate_request req);

/**
 * parse received data by considering it that hello request handshake
 * protocol message.
 */
bool tls_hs_helloreq_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that client hello handshake
 * protocol message.
 */
bool tls_hs_chello_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that server hello handshake
 * protocol message.
 */
bool tls_hs_shello_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that encrypted extensions
 * handshake protocol message.
 */
bool tls_hs_encext_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that (server) certificate
 * handshake protocol message.
 */
bool tls_hs_scert_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that server key exchange
 * handshake protocol message.
 */
bool tls_hs_skeyexc_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that certificate request
 * handshake protocol message.
 */
bool tls_hs_certreq_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that server hello done
 * handshake protocol message.
 */
bool tls_hs_shellodone_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that client certificate
 * handshake protocol message.
 */
bool tls_hs_ccert_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that client key exchange
 * handshake protocol message.
 */
bool tls_hs_ckeyexc_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that certificate verify
 * handshake protocol message.
 */
bool tls_hs_certvfy_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that finished handshake
 * protocol message.
 */
bool tls_hs_finished_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse received data by considering it that key update handshake
 * protocol message.
 */
bool tls_hs_keyupdate_parse(TLS *tls, struct tls_hs_msg *msg);

/**
 * parse extensions in a message and store results to interim_params
 * structure in tls structure.
 */
int32_t tls_hs_extension_parse(TLS *tls,
			      const struct tls_hs_msg *msg,
			      const uint32_t offset);

/**
 * validate protocol version in client hello message.
 */
bool tls_hs_chello_version_check(TLS *tls);


/**
 * validate protocol version in server hello message.
 */
bool tls_hs_shello_version_check(TLS *tls);

/**
 * interpret remaining data in client hello message after protocol
 * version is negotiated.
 */
bool tls_hs_chello_interpret(TLS *tls);

/**
 * interpret extensions in client hello for hello retry request.
 */
bool tls_hs_chello_interpret_extensions(TLS *tls);

/**
 * interpret remaining data in server hello message after protocol
 * version is negotiated.
 */
bool tls_hs_shello_interpret(TLS *tls);

/**
 * interpret encrypted extensions message.
 */
bool tls_hs_encext_interpret(TLS *tls);

/**
 * process post-handshake message.
 */
bool tls_hs_posthandshake_message(TLS *tls);

/**
 * send key update message set to update_requested.
 */
bool tls_hs_send_key_update(TLS *tls, enum tls_keyupdate_request req);

#endif /* INCLUSION_GUARD_UUID_95C347E7_6082_FDC1_89B7_A49DB5FBB138 */
