/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_1CF0BFD4_91A9_0C38_46D5_9C97C276F791
#define INCLUSION_GUARD_UUID_1CF0BFD4_91A9_0C38_46D5_9C97C276F791

#include "tls.h"
#include "tls_record.h"

/* for debug */
#include "tls_util.h"

/**
 * list of cipher suite (enum tls_cipher_suite).
 */
struct tls_cipher_list {
	/** length of list */
	int32_t len;

	/** list of cipehr suite */
	enum tls_cipher_suite *list;
};

/**
 * maximum size of cipher_suites.
 */
#define TLS_CIPHER_SUITES_SIZE_MAX	((2 << (16 - 1)) -2)

/**
 * size of AES GCM nonce.
 */
#define TLS_AES_GCM_NONCE_SIZE (12)

/**
 * return hash algorithm corresponding to the cipher suite.
 */
enum tls_hs_sighash_hash_algo tls_cipher_hashalgo(const uint16_t suite);

/**
 * return the key exchange method corresponding to the cipher suite.
 */
enum tls_keyexchange_method tls_cipher_keymethod(const uint16_t suite);

/**
 * output debug purpose information of cipher suite.
 *
 * this function will output debug purpose information when the
 * TLS_DEBUG macro was available on build.
 */
void tls_cipher_suite_dump(enum tls_cipher_suite suite);

#ifdef TLS_DEBUG
/**
 * Output debug information of key exchange method.
 */
void tls_cipher_keyexc_method_dump(enum tls_keyexchange_method method);

#define TLS_KXC_METHOD_DUMP(method) tls_cipher_keyexc_method_dump((method))
#else
#define TLS_KXC_METHOD_DUMP(method)
#endif

/**
 * select cipher suite to use by negotiated TLS/SSL version.
 *
 * this function is called in server side to determine cipher suite that
 * use in TLS/SSL connection.
 */
bool tls_cipher_select(TLS *tls, const struct tls_cipher_list *cipher);

/**
 * Get list of cipher suite that is used by defaut.
 *
 * Get default suites determines by negotiated TLS/SSL version.
 *
 * If the argument PREFERENCE is not NULL, a suite that is not in
 * the preference list will be excluded.
 *
 * Return value of this function is memory allocated value.
 * If that value became unnecessary, must free by tls_cipher_list_free
 * function.
 */
struct tls_cipher_list *tls_cipher_supported_list(TLS* tls,
				const struct tls_cipher_list *prefernce);

/**
 * Allocate memory for storing list of cihper suites.
 */
struct tls_cipher_list *tls_cipher_list_alloc(const uint16_t len);

/**
 * free memory that is allocated by tls_cipher_list function.
 */
void tls_cipher_list_free(struct tls_cipher_list *list);

/**
 * set specified cipher suite to the instance of tls structure.
 *
 * this function is used by client in the receiving server hello.
 * (tls_cipher_select function is used by server in the receiving client
 * hello).
 */
bool tls_cipher_set(TLS *tls, const uint16_t suite);

/**
 * set parameter of cipher suite to the instance of tls structure.
 *
 * this function is used by receiving/sending ccs. by initialized
 * parameter of cipher suite, communicate by encryption from next
 * communication.
 */
bool tls_cipher_param_set(const enum tls_cipher_suite suite,
			  struct tls_cipher_param *param);

/**
 * encrypt by stream cipher.
 */
int32_t tls_cipher_stream(TLS *tls,
			  uint8_t *dest,
			  const enum tls_record_ctype type,
			  const uint8_t *src,
			  const int32_t len);

/**
 * encrypt by block cipher.
 */
int32_t tls_cipher_block(TLS *tls,
			 uint8_t *dest,
			 const enum tls_record_ctype type,
			 const uint8_t *src,
			 const int32_t len);

/**
 * encrypt by aead cipher.
 *
 * @param[in] tls TLS
 * @param[out] dest must be allocated the size of the following:
 * <nonce_explicit size> + <ciphertext size> + <authentication tag size>.
 * @param[in] type content type.
 * @param[in] src plaintext.
 * @param[in] len byte length of plaintext.
 */
int32_t tls_cipher_aead(TLS *tls,
			uint8_t *dest,
			const enum tls_record_ctype type,
			const uint8_t *src,
			const int32_t len);

/**
 * encrypt specified data.
 *
 * actual encryption is done by tls_cipher_{stream,block,aead} function.
 */
int32_t tls_cipher(TLS *tls,
		   uint8_t *dest,
		   const enum tls_record_ctype type,
		   const uint8_t *src,
		   const int32_t len);

/**
 * decrypt received data by considering it the stream cipher.
 */
int32_t tls_decipher_stream(TLS *tls,
			    uint8_t *dest,
			    const enum tls_record_ctype type,
			    const uint8_t *src,
			    const int32_t len);

/**
 * decrypt received data by considering it the block cipher.
 */
int32_t tls_decipher_block(TLS *tls,
			   uint8_t *dest,
			   const enum tls_record_ctype type,
			   const uint8_t *src,
			   const int32_t len);

/**
 * decrypt received data by considering it the aead cipher.
 *
 * @param[in] tls TLS
 * @param[out] dest
 * @param[in] type content type.
 * @param[in] src AEAD encrypted.
 * <nonce_explicit> + <ciphertext> + <authentication tag>.
 * @param[in] len byte length of plaintext.
 * <nonce_explicit size> + <ciphertext size> + <authentication tag size>.
 */
int32_t tls_decipher_aead(TLS *tls,
			  uint8_t *dest,
			  const enum tls_record_ctype type,
			  const uint8_t *src,
			  const int32_t len);

/**
 * decrypt specified data.
 *
 * actual decryption is done by tls_decipher_{stream,block,aead}
 * function.
 */
int32_t tls_decipher(TLS *tls,
		     uint8_t *dest,
		     const enum tls_record_ctype type,
		     const uint8_t *src,
		     const int32_t len);

#endif /* INCLUSION_GUARD_UUID_1CF0BFD4_91A9_0C38_46D5_9C97C276F791 */
