/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_3C676470_15B3_EDC1_AFFB_99EBD6025051
#define INCLUSION_GUARD_UUID_3C676470_15B3_EDC1_AFFB_99EBD6025051

#include "tls.h"
#include "tls_handshake.h"

/* for PKCS12 */
#include <aicrypto/ok_pkcs12.h>

/**
 * available certificate type that is used by certificate request
 * handshake protocol message. this enum is described in RFC 5246
 * section 7.4.4. */
enum tls_cert_type {
	TLS_CERT_RSA_SIGN                  =  1U,
	TLS_CERT_DSS_SIGN                  =  2U,
	TLS_CERT_RSA_FIXED_DH              =  3U,
	TLS_CERT_DSS_FIXED_DH              =  4U,
	TLS_CERT_RSA_EPHEMERAL_DH_RESERVED =  5U,
	TLS_CERT_DSS_EPHEMERAL_DH_RESERVED =  6U,
	TLS_CERT_FORTEZZA_DMS_RESERVED     = 20U
};

/**
 * list of certificate type (enum tls_cert_type).
 */
struct tls_cert_type_list {
	/** length of list member of this structure. */
	int32_t len;

	/** list of certificate type. */
	enum tls_cert_type *list;
};

/**
 * Server Certificate information.
 *
 * These information will check when choosing a cipher suite to use.
 */
struct tls_cert_info {
	/**
	 * Extension Extended Key Usage.
	 *
	 * If the extension is not present, set 0xff.
	 */
	uint8_t keyusage;

	/**
	 * Signature algorithm from aicrypto/ok_asn1.h. ex. OBJ_SIG_SHA1RSA
	 */
	int signature_algo;

	/**
	 * Public key type from aicrypto/key_type.h. ex. KEY_RSA_PUB
	 */
	int pubkey_algo;

	/**
	 * Type from aicrypto/ok_ecc.h. ex. ECP_X962_prime256v1
	 */
	int curve_type;
};

/**
 * Key usage extension flag.
 *
 * @see aicrypto/ok_x509ext.h
 * @see x509/ext_crtstr.c Ext_keyusage_str
 */
enum ext_ku {
	digitalSignature = 0x80,
	keyEncipherment = 0x20,
	keyAgreement = 0x08,
};

/**
 * get list of certificate type that is used in default.
 *
 * default determines by negotiated TLS/SSL version. return value of
 * this function is memory allocated value. if that value became
 * unnecessary, must free by tls_cert_type_free function.
 */
struct tls_cert_type_list * tls_cert_type_list(const TLS *tls);

/**
 * free memory that is allocated by tls_cert_type_list function.
 */
void tls_cert_type_free(struct tls_cert_type_list *list);

/**
 * check availability of specified certificate type.
 */
bool tls_cert_type_availablep(enum tls_cert_type type);

/**
 * Structure initialize.
 */
void tls_cert_info_init(struct tls_cert_info *info);

/**
 * Get server certificate infomation.
 */
bool tls_cert_info_get(PKCS12 *p12, struct tls_cert_info *info);

/**
 * Return true if the certificate can use with ECC cipher suite.
 */
bool tls_cert_info_can_use_ecc_cipher_suite(struct tls_cert_info *info,
					    struct tls_hs_ecc_eclist *eclist,
					    struct tls_hs_ecc_pflist *pflist);

/**
 * Return true if the certificate can use with ECC cipher suite.
 */
enum tls_hs_ecc_ec_curve_type tls_cert_info_ecc_get_type(struct tls_cert_info
							 *cinfo);

/**
 * Return true if the certificate can use with ECC cipher suite.
 */
enum tls_hs_named_curve tls_cert_info_ecc_get_curve(struct tls_cert_info
							*cinfo);

/**
 * To check whether the cipher suite is available with the certificate.
 *
 * @param suite cipher suite.
 * @param info certificate infomation.
 * @param can_use_ecc If true, this certificate can use with ECC cipher suites.
 * @param set_error If true, set error by OK_set_error() when the certificate
 * is not available.
 *
 * @return true or false
 */
bool tls_cert_info_available(enum tls_cipher_suite suite,
			     struct tls_cert_info *info, bool can_use_ecc,
			     bool set_error);

/**
 * verify specified certificate (expire, revoke ...).
 *
 * this function internally use store manager to verify certificate. so,
 * it is necessary to initialize store manager to use verfication of
 * this function.
 */
bool TLS_cert_verify(TLS *tls, PKCS12 *p12);

#endif /* INCLUSION_GUARD_UUID_3C676470_15B3_EDC1_AFFB_99EBD6025051 */
