/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_29C3378B_6CC3_497F_6E8D_173E5FE790AF
#define INCLUSION_GUARD_UUID_29C3378B_6CC3_497F_6E8D_173E5FE790AF

#include "tls_record.h"

enum tls_alert_level {
	TLS_ALERT_LEVEL_WARNING = 1U,
	TLS_ALERT_LEVEL_FATAL   = 2U,
};

/**
 * RFC5246 A.3.  Alert Messages
 *
 *    enum {
 *        close_notify(0),
 *        unexpected_message(10),
 *        bad_record_mac(20),
 *        decryption_failed_RESERVED(21),
 *        record_overflow(22),
 *        decompression_failure(30),
 *        handshake_failure(40),
 *        no_certificate_RESERVED(41),
 *        bad_certificate(42),
 *        unsupported_certificate(43),
 *        certificate_revoked(44),
 *        certificate_expired(45),
 *        certificate_unknown(46),
 *        illegal_parameter(47),
 *        unknown_ca(48),
 *        access_denied(49),
 *        decode_error(50),
 *        decrypt_error(51),
 *        export_restriction_RESERVED(60),
 *        protocol_version(70),
 *        insufficient_security(71),
 *        internal_error(80),
 *        user_canceled(90),
 *        no_renegotiation(100),
 *        unsupported_extension(110),           / new /
 *        (255)
 *    } AlertDescription;
 *
 * RFC6066 9.  Error Alerts
 *
 *       enum {
 *           close_notify(0),
 *           unexpected_message(10),
 *           bad_record_mac(20),
 *           decryption_failed(21),
 *           record_overflow(22),
 *           decompression_failure(30),
 *           handshake_failure(40),
 *           / 41 is not defined, for historical reasons /
 *           bad_certificate(42),
 *           unsupported_certificate(43),
 *           certificate_revoked(44),
 *           certificate_expired(45),
 *           certificate_unknown(46),
 *           illegal_parameter(47),
 *           unknown_ca(48),
 *           access_denied(49),
 *           decode_error(50),
 *           decrypt_error(51),
 *           export_restriction(60),
 *           protocol_version(70),
 *           insufficient_security(71),
 *           internal_error(80),
 *           user_canceled(90),
 *           no_renegotiation(100),
 *           unsupported_extension(110),
 *           certificate_unobtainable(111),        / new /
 *           unrecognized_name(112),               / new /
 *           bad_certificate_status_response(113), / new /
 *           bad_certificate_hash_value(114),      / new /
 *           (255)
 *       } AlertDescription;
 *
 *
 * RFC7301 3.2.  Protocol Selection
 *
 *    enum {
 *        no_application_protocol(120),
 *        (255)
 *    } AlertDescription;
 *
 * RFC7507 2.  Protocol Values
 *
 *         enum {
 *           / ... /
 *           inappropriate_fallback(86),
 *           / ... /
 *           (255)
 *         } AlertDescription;
 *
 * RFC8446 B.2.  Alert Messages
 *
 *       enum {
 *           close_notify(0),
 *           unexpected_message(10),
 *           bad_record_mac(20),
 *           decryption_failed_RESERVED(21),
 *           record_overflow(22),
 *           decompression_failure_RESERVED(30),
 *           handshake_failure(40),
 *           no_certificate_RESERVED(41),
 *           bad_certificate(42),
 *           unsupported_certificate(43),
 *           certificate_revoked(44),
 *           certificate_expired(45),
 *           certificate_unknown(46),
 *           illegal_parameter(47),
 *           unknown_ca(48),
 *           access_denied(49),
 *           decode_error(50),
 *           decrypt_error(51),
 *           export_restriction_RESERVED(60),
 *           protocol_version(70),
 *           insufficient_security(71),
 *           internal_error(80),
 *           inappropriate_fallback(86),
 *           user_canceled(90),
 *           no_renegotiation_RESERVED(100),
 *           missing_extension(109),
 *           unsupported_extension(110),
 *           certificate_unobtainable_RESERVED(111),
 *           unrecognized_name(112),
 *           bad_certificate_status_response(113),
 *           bad_certificate_hash_value_RESERVED(114),
 *           unknown_psk_identity(115),
 *           certificate_required(116),
 *           no_application_protocol(120),
 *           (255)
 *       } AlertDescription;
 */
enum tls_alert_desc {
	/* alerts below are defined by RFC5246 and RFC8446 */
	TLS_ALERT_DESC_CLOSE_NOTIFY                =   0U,
	TLS_ALERT_DESC_UNEXPECTED_MESSAGE          =  10U,
	TLS_ALERT_DESC_BAD_RECORD_MAC              =  20U,
	TLS_ALERT_DESC_DECRYPTION_FAILED_RESERVED  =  21U,
	TLS_ALERT_DESC_RECORD_OVERFLOW             =  22U,
	TLS_ALERT_DESC_DECOMPRESSION_FAILURE       =  30U,
	TLS_ALERT_DESC_HANDSHAKE_FAILURE           =  40U,
	TLS_ALERT_DESC_NO_CERTIFICATE_RESERVED     =  41U,
	TLS_ALERT_DESC_BAD_CERTIFICATE             =  42U,
	TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE     =  43U,
	TLS_ALERT_DESC_CERTIFICATE_REVOKED         =  44U,
	TLS_ALERT_DESC_CERTIFICATE_EXPIRED         =  45U,
	TLS_ALERT_DESC_CERTIFICATE_UNKNOWN         =  46U,
	TLS_ALERT_DESC_ILLEGAL_PARAMETER           =  47U,
	TLS_ALERT_DESC_UNKNOWN_CA                  =  48U,
	TLS_ALERT_DESC_ACCESS_DENIED               =  49U,
	TLS_ALERT_DESC_DECODE_ERROR                =  50U,
	TLS_ALERT_DESC_DECRYPT_ERROR               =  51U,
	TLS_ALERT_DESC_EXPORT_RESTRICTION_RESERVED =  60U,
	TLS_ALERT_DESC_PROTOCOL_VERSION            =  70U,
	TLS_ALERT_DESC_INSUFFICIENT_SECURITY       =  71U,
	TLS_ALERT_DESC_INTERNAL_ERROR              =  80U,

	/* alert below is defined by RFC7507 and RFC8446 */
	TLS_ALERT_DESC_INAPPROPRIATE_FALLBACK      =  86U,

	/* alerts below are defined by RFC5246 and RFC8446 */
	TLS_ALERT_DESC_USER_CANCELED               =  90U,
	TLS_ALERT_DESC_NO_RENEGOTIATION            = 100U,

	/* alert below is defined by RFC8446 */
	TLS_ALERT_DESC_MISSING_EXTENSION           = 109U,

	/* alert below is defined by RFC5246 and RFC8446 */
	TLS_ALERT_DESC_UNSUPPORTED_EXTENSION       = 110U,

	/* alerts below are defined by RFC6066 and RFC8446 */
	TLS_ALERT_DESC_CERTIFICATE_UNOBTAINABLE    = 111U,
	TLS_ALERT_DESC_UNRECOGNIZED_NAME           = 112U,
	TLS_ALERT_DESC_BAD_CERTIFICATE_STATUS_RESPONSE = 113U,
	TLS_ALERT_DESC_BAD_CERTIFICATE_HASH_VALUE  = 114U,

	/* alerts below are defined by RFC8446 */
	TLS_ALERT_DESC_UNKNOWN_PSK_IDENTITY        = 115U,
	TLS_ALERT_DESC_CERTIFICATE_REQUIRED        = 116U,

	/* alert below is defined by RFC7301 and RFC8446 */
	TLS_ALERT_DESC_NO_APPLICATION_PROTOCOL     = 120U
};

/**
 * send tls alert protocol.
 *
 * return value of this function changes by sent alert level.
 *
 * - sent alert level is warning: true
 * - sent alert level is fatal:   false
 *
 * in the case of warning, connection continues. in that meaning, return
 * true. in the case of fata, connection do not continue. thefore,
 * return false. even as the case of passing warning level to this
 * function input, there is a case to send fatal level by internal
 * error.
 */
bool tls_alert_send(TLS *tls,
		    const enum tls_alert_level level,
		    const enum tls_alert_desc desc);

/**
 * parse tls alert protocol.
 *
 * if alert protocol data meets the following conditions,
 *
 * - received data has fatal level
 * - received data length is invalid
 * - received data has invalid level
 * - received data has invalid description
 *
 * send unexpected alert to the peer. and close connection.
 *
 * return value of this function changes by received data level.
 * it is as follows.
 *
 * - received data level is warning: true
 * - received data level is fatal:   false
 *
 * in the case of warning, connection continues. in that meaning, return
 * true. in the case of fatal, connection do not continue. therefore,
 * return false. consider to be the same meaning as error.
 */
bool tls_alert_recv(TLS *tls, const struct tls_record *record);

#define TLS_ALERT_FATAL(tls, desc)					\
	do {								\
		tls_alert_send((tls), TLS_ALERT_LEVEL_FATAL, (desc));	\
	} while(0)							\

#define TLS_ALERT_WARN(tls, desc)					\
	do {								\
		tls_alert_send((tls), TLS_ALERT_LEVEL_WARNING, (desc));	\
	} while(0)							\

#endif /* INCLUSION_GUARD_UUID_29C3378B_6CC3_497F_6E8D_173E5FE790AF */
