/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_6C553695_8E3E_23FA_E42D_485AA53E43E1
#define INCLUSION_GUARD_UUID_6C553695_8E3E_23FA_E42D_485AA53E43E1

#include <aicrypto/nrg_tls.h>

#include <stdint.h>
#include <time.h>
#include <assert.h>

#define _STDC_FORMAT_MACROS
#include <inttypes.h>

/* for strerror and errno */
#include <string.h>
#include <errno.h>

/* for OK_set_error */
#include <aicrypto/ok_err.h>

/* for Key */
#include <aicrypto/ok_x509.h>

/* for PKCS12 */
#include <aicrypto/ok_pkcs12.h>

/* for STManager */
#include <aicrypto/ok_store.h>

/* for MD5_CTX */
#include <aicrypto/ok_md5.h>

/* for SHA1_CTX */
#include <aicrypto/ok_sha1.h>

/* for SHA256_CTX and SHA512_CTX. */
#include <aicrypto/ok_sha2.h>

/* maximum length of vector type in each length byte */
#define TLS_VECTOR_1_BYTE_SIZE_MAX	((2 << (8 - 1)) - 1)
#define TLS_VECTOR_2_BYTE_SIZE_MAX	((2 << (16 - 1)) - 1)
#define TLS_VECTOR_3_BYTE_SIZE_MAX	((2 << (24 - 1)) - 1)

/* premaster secret length in case of RSA. */
#define TLS_PREMASTER_SECRET_RSA_LEN	48

/*
 * maximum length of context that is used for deriving application secret.
 */
#define TLS_APPLICATION_SECRET_CONTEXT_SIZE_MAX 64

/** available prf algorithm that used in security parameter. */
enum prf_algorithm {
	/** PRF_SHA256 algorithm. */
	TLS_PRF_SHA256,
	TLS_PRF_SHA384,
	/** HKDF algorithm. */
	TLS_HKDF_SHA256,
	TLS_HKDF_SHA384
};

/** available bulk cipher algorithm that used in security parameter. */
enum bulk_cipher_algorithm {
	/** null cipher algorithm. this means no cipher. */
	TLS_BULK_CIPHER_NULL,

	/** RC4 algorithm. */
	TLS_BULK_CIPHER_RC4,

	/** triple DES algorithm. */
	TLS_BULK_CIPHER_3DES,

	/** AES algorithm. */
	TLS_BULK_CIPHER_AES,

	/** CHACHA20 algorithm. */
	TLS_BULK_CIPHER_CHACHA20
};

/** available types of cipher that used in security parameter. */
enum cipher_type {
	/* stream cipher. */
	TLS_CIPHER_TYPE_STREAM,

	/* block cipher. */
	TLS_CIPHER_TYPE_BLOCK,

	/* aead cipher. */
	TLS_CIPHER_TYPE_AEAD
};

/** available mac algorithm that used in security parameter. */
enum mac_algorithm {
	/** null mac algorithm. this means no mac. */
	TLS_MAC_NULL,

	/** hmac_md5 algorithm. */
	TLS_MAC_HMAC_MD5,

	/** hmac_sha1 algorithm. */
	TLS_MAC_HMAC_SHA1,

	/** hmac_sha256 algorithm. */
	TLS_MAC_HMAC_SHA256,

	/** hmac_sha384 algorithm. */
	TLS_MAC_HMAC_SHA384,

	/** hmac_sha512 algorithm. */
	TLS_MAC_HMAC_SHA512
};

/** available cipher suite that used in security parameter.  */
enum tls_cipher_suite {
	TLS_NULL_WITH_NULL_NULL               = 0x0000,

	TLS_RSA_WITH_NULL_MD5		      = 0x0001,
	TLS_RSA_WITH_NULL_SHA		      = 0x0002,
	TLS_RSA_WITH_NULL_SHA256	      = 0x003B,
	TLS_RSA_WITH_RC4_128_MD5	      = 0x0004,
	TLS_RSA_WITH_RC4_128_SHA	      = 0x0005,
	TLS_RSA_WITH_IDEA_CBC_SHA             = 0x0007,
	TLS_RSA_WITH_DES_CBC_SHA              = 0x0009,
	TLS_RSA_WITH_3DES_EDE_CBC_SHA	      = 0x000A,
	TLS_RSA_WITH_AES_128_CBC_SHA	      = 0x002F,
	TLS_RSA_WITH_AES_256_CBC_SHA	      = 0x0035,
	TLS_RSA_WITH_AES_128_CBC_SHA256       = 0x003C,
	TLS_RSA_WITH_AES_256_CBC_SHA256       = 0x003D,

	TLS_DH_DSS_WITH_DES_CBC_SHA           = 0x000C,
	TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA      = 0x000D,
	TLS_DH_RSA_WITH_DES_CBC_SHA           = 0x000F,
	TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA      = 0x0010,
	TLS_DHE_DSS_WITH_DES_CBC_SHA          = 0x0012,
	TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA     = 0x0013,
	TLS_DHE_RSA_WITH_DES_CBC_SHA          = 0x0015,
	TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA     = 0x0016,
	TLS_DH_DSS_WITH_AES_128_CBC_SHA       = 0x0030,
	TLS_DH_RSA_WITH_AES_128_CBC_SHA       = 0x0031,
	TLS_DHE_DSS_WITH_AES_128_CBC_SHA      = 0x0032,
	TLS_DHE_RSA_WITH_AES_128_CBC_SHA      = 0x0033,
	TLS_DH_DSS_WITH_AES_256_CBC_SHA       = 0x0036,
	TLS_DH_RSA_WITH_AES_256_CBC_SHA       = 0x0037,
	TLS_DHE_DSS_WITH_AES_256_CBC_SHA      = 0x0038,
	TLS_DHE_RSA_WITH_AES_256_CBC_SHA      = 0x0039,
	TLS_DH_DSS_WITH_AES_128_CBC_SHA256    = 0x003E,
	TLS_DH_RSA_WITH_AES_128_CBC_SHA256    = 0x003F,
	TLS_DHE_DSS_WITH_AES_128_CBC_SHA256   = 0x0040,
	TLS_DHE_RSA_WITH_AES_128_CBC_SHA256   = 0x0067,
	TLS_DH_DSS_WITH_AES_256_CBC_SHA256    = 0x0068,
	TLS_DH_RSA_WITH_AES_256_CBC_SHA256    = 0x0069,
	TLS_DHE_DSS_WITH_AES_256_CBC_SHA256   = 0x006A,
	TLS_DHE_RSA_WITH_AES_256_CBC_SHA256   = 0x006B,

	TLS_DH_anon_WITH_RC4_128_MD5          = 0x0018,
	TLS_DH_anon_WITH_DES_CBC_SHA          = 0x001A,
	TLS_DH_anon_WITH_3DES_EDE_CBC_SHA     = 0x001B,
	TLS_DH_anon_WITH_AES_128_CBC_SHA      = 0x0034,
	TLS_DH_anon_WITH_AES_256_CBC_SHA      = 0x003A,
	TLS_DH_anon_WITH_AES_128_CBC_SHA256   = 0x006C,
	TLS_DH_anon_WITH_AES_256_CBC_SHA256   = 0x006D,

	TLS_RSA_EXPORT_WITH_RC4_40_MD5        = 0x0003,
	TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5    = 0x0006,
	TLS_RSA_EXPORT_WITH_DES40_CBC_SHA     = 0x0008,
	TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SH   = 0x000B,
	TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA  = 0x000E,
	TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x0011,
	TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0014,
	TLS_DH_anon_EXPORT_WITH_RC4_40_MD5    = 0x0017,
	TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA = 0x0019,

	TLS_KRB5_WITH_DES_CBC_SHA             = 0x001E,
	TLS_KRB5_WITH_3DES_EDE_CBC_SHA        = 0x001F,
	TLS_KRB5_WITH_RC4_128_SHA             = 0x0020,
	TLS_KRB5_WITH_IDEA_CBC_SHA            = 0x0021,
	TLS_KRB5_WITH_DES_CBC_MD5             = 0x0022,
	TLS_KRB5_WITH_3DES_EDE_CBC_MD5        = 0x0023,
	TLS_KRB5_WITH_RC4_128_MD5             = 0x0024,
	TLS_KRB5_WITH_IDEA_CBC_MD5            = 0x0025,

	TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA   = 0x0026,
	TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA   = 0x0027,
	TLS_KRB5_EXPORT_WITH_RC4_40_SHA       = 0x0028,
	TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5   = 0x0029,
	TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5   = 0x002A,
	TLS_KRB5_EXPORT_WITH_RC4_40_MD5       = 0x002B,

	/* rfc4492 : 6.  Cipher Suites */
	TLS_ECDH_ECDSA_WITH_NULL_SHA          = 0xC001,
	TLS_ECDH_ECDSA_WITH_RC4_128_SHA       = 0xC002,
	TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA  = 0xC003,
	TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA   = 0xC004,
	TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA   = 0xC005,

	TLS_ECDHE_ECDSA_WITH_NULL_SHA         = 0xC006,
	TLS_ECDHE_ECDSA_WITH_RC4_128_SHA      = 0xC007,
	TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC008,
	TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA  = 0xC009,
	TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA  = 0xC00A,

	TLS_ECDH_RSA_WITH_NULL_SHA            = 0xC00B,
	TLS_ECDH_RSA_WITH_RC4_128_SHA         = 0xC00C,
	TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA    = 0xC00D,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA     = 0xC00E,
	TLS_ECDH_RSA_WITH_AES_256_CBC_SHA     = 0xC00F,

	TLS_ECDHE_RSA_WITH_NULL_SHA           = 0xC010,
	TLS_ECDHE_RSA_WITH_RC4_128_SHA        = 0xC011,
	TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA   = 0xC012,
	TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA    = 0xC013,
	TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA    = 0xC014,

	TLS_ECDH_anon_WITH_NULL_SHA           = 0xC015,
	TLS_ECDH_anon_WITH_RC4_128_SHA        = 0xC016,
	TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA   = 0xC017,
	TLS_ECDH_anon_WITH_AES_128_CBC_SHA    = 0xC018,
	TLS_ECDH_anon_WITH_AES_256_CBC_SHA    = 0xC019,

	/* rfc5288 : 3.  AES-GCM Cipher Suites */
	TLS_RSA_WITH_AES_128_GCM_SHA256       = 0x009C,
	TLS_RSA_WITH_AES_256_GCM_SHA384       = 0x009D,
	TLS_DHE_RSA_WITH_AES_128_GCM_SHA256   = 0x009E,
	TLS_DHE_RSA_WITH_AES_256_GCM_SHA384   = 0x009F,
	TLS_DH_RSA_WITH_AES_128_GCM_SHA256    = 0x00A0,
	TLS_DH_RSA_WITH_AES_256_GCM_SHA384    = 0x00A1,
	TLS_DHE_DSS_WITH_AES_128_GCM_SHA256   = 0x00A2,
	TLS_DHE_DSS_WITH_AES_256_GCM_SHA384   = 0x00A3,
	TLS_DH_DSS_WITH_AES_128_GCM_SHA256    = 0x00A4,
	TLS_DH_DSS_WITH_AES_256_GCM_SHA384    = 0x00A5,
	TLS_DH_anon_WITH_AES_128_GCM_SHA256   = 0x00A6,
	TLS_DH_anon_WITH_AES_256_GCM_SHA384   = 0x00A7,

	/* rfc5289 : 3.1.  HMAC-Based Cipher Suites */
	TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256   = 0xC023,
	TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384   = 0xC024,
	TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256    = 0xC025,
	TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384    = 0xC026,
	TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256     = 0xC027,
	TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384     = 0xC028,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256      = 0xC029,
	TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384      = 0xC02A,

	/* rfc5289 : 3.2.  Galois Counter Mode-Based Cipher Suites */
	TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256   = 0xC02B,
	TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384   = 0xC02C,
	TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256    = 0xC02D,
	TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384    = 0xC02E,
	TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256     = 0xC02F,
	TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384     = 0xC030,
	TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256      = 0xC031,
	TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384      = 0xC032,

	/* rfc8446 : B.4. Cipher Suites */
	TLS_AES_128_GCM_SHA256                    = 0x1301,
	TLS_AES_256_GCM_SHA384                    = 0x1302,
	TLS_CHACHA20_POLY1305_SHA256              = 0x1303,
	TLS_AES_128_CCM_SHA256                    = 0x1304,
	TLS_AES_128_CCM_8_SHA256                  = 0x1305,
};

/** available compression algorithm that used in security parameter.  */
enum compression_algorithm {
	/** null compression algorithm. this means no compression. */
	TLS_COMPRESSION_NULL = 0x0U
};

/**
 * TLS cipehr suite parameter.
 *
 * in here, tls_cipher_param structure stores parameters that are
 * derived by cipher_suite of security parameter.  in comparision with
 * security parameter of RFC 5246 section 6.1, verify_length parameter
 * is added. this parameter is used by finishd handshake protocol, and
 * that value is determined by prf_algorithm.
 */
struct tls_cipher_param {
	/* this value used by FINISHED handshake protocol message. in
	 * the scope of rfc 5246, this value is 12. */
	uint32_t verify_length;

	/** algorithm used to generate keys from the master secret */
	enum prf_algorithm prf_algorithm;

	/** algorithm to be used for bulk encryption. */
	enum bulk_cipher_algorithm  cipher_algorithm;

	/** type to cipher. */
	enum cipher_type  cipher_type;

	/** encryption key length of cipher algorithm. */
	uint8_t enc_key_length;

	/** block length of cipher algorithm. */
	uint8_t block_length;

	/** fixed initial vector length of cipher algorithm. */
	uint8_t fixed_iv_length;

	/** record initial vector length of cipher algorithm.  */
	uint8_t record_iv_length;

	/** algorithm to be used for message authentication. */
	enum mac_algorithm mac_algorithm;

	/** length of message authentication. */
	uint8_t mac_length;

	/** key length of message authentication. */
	uint8_t mac_key_length;
};

/**
 * TLS connection structure.
 *
 * this structure is used by compression, decompression, decryption and
 * encryption.
 */
struct tls_connection {
	/* cipher suite parameter of this connection */
	struct tls_cipher_param cipher;

	/* compression algorithm of this connection */
	enum compression_algorithm compression_algorithm;

	/* message authentication key. */
	uint8_t *mac_key;

	/* write key for cipher. this is representation of wirte key and
	 * write iv (RFC 5246 section 6.3). */
	Key *key;

	/* sequnce number */
	uint64_t seqnum;

	/* traffic secret used in TLS 1.3 */
	uint8_t *secret;

	/* state that indicates if connection is closed or not  */
	bool closed;
};

/**
 * available TLS major version(Deprecated).
 *
 * for example, when TLS 2.0 comes with a version number like "0x0400",
 * we cannot distinguish it from SSL 3.0(0x0300) and branch code by separated
 * major version and minor version. we will need to handle major version and
 * minor version together to identify correct version.
 */
enum tls_major_version {
	/**
	 * major version number for SSL3.0, TLS1.0, TLS1.1 and
	 * TLS 1.2. */
	TLS_MAJOR_TLS = 3U,

	/**
	 * maximum number of tls_major_version enum.  this is a
	 * value of convenience for writing program. */
	TLS_MAJOR_MAX
};

/**
 * available TLS minor version(Deprecated).
 *
 * for example, when TLS 2.0 comes with a version number like "0x0400",
 * we cannot distinguish it from SSL 3.0(0x0300) and branch code by separated
 * major version and minor version. we will need to handle major version and
 * minor version together to identify correct version.
 */
enum tls_minor_version {
	/** minor version number for SSL 3.0 */
	TLS_MINOR_SSL30 = 0U,
	/** minor version number for TLS 1.0 */
	TLS_MINOR_TLS10 = 1U,
	/** minor version number for TLS 1.1 */
	TLS_MINOR_TLS11 = 2U,
	/** minor version number for TLS 1.2 */
	TLS_MINOR_TLS12 = 3U,
	/** minor version number for TLS 1.3 */
	TLS_MINOR_TLS13 = 4U,

	/**
	 * maximum number of tls_minor_version enum.  this is a
	 * value of convenience for writing program. */
	TLS_MINOR_MAX
};
/**
 * TLS protocol version structure.
 */
struct tls_protocol_version {
	/**
	 * tls major version enum that are used by client hello, server
	 * hello and so on.
	 */
	enum tls_major_version  major;

	/**
	 * tls minor version enum that are used by client hello, server hello
	 * and so on.
	 */
	enum tls_minor_version  minor;
};

/**
 * available TLS version.
 */
enum tls_version {
	/* SSL 3.0 */
	TLS_VER_SSL30 = 0x0300,
	/* TLS 1.0 */
	TLS_VER_TLS10 = 0x0301,
	/* TLS 1.1 */
	TLS_VER_TLS11 = 0x0302,
	/* TLS 1.2 */
	TLS_VER_TLS12 = 0x0303,
	/* TLS 1.3 */
	TLS_VER_TLS13 = 0x0304
};

/**
 * list of TLS protocol version (uint16_t).
 */
struct tls_protocol_version_list {
	/** length of list */
	int32_t len;

	/** list of protocol version */
	uint16_t *list;
};

/**
 * TLS connection status structure.
 *
 * this enum indicates connection status of handshake mainly.
 */
enum tls_state {
	/** state that is initial handshake. this is a state of both
	 * client and server. */
	TLS_STATE_HS_INIT,

	/** state that means a start of handshake by
	 * renegotiation. client become this state by receiving the
	 * hello request. this is a state of client side. */
	TLS_STATE_HS_REINIT,

	/** state that is after sending the hello request handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_SEND_HELLOREQ,

	/** state that is after receiving the hello request handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_HELLOREQ,

	/** state that is before sending the client hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_BEFORE_SEND_CHELLO,

	/** state that is after sending the client hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_AFTER_SEND_CHELLO,

	/** state that is before receiving the server hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_BEFORE_RECV_SHELLO,

	/** state that is after receiving the server hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_AFTER_RECV_SHELLO,

	/** state that is before sending the second client hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_BEFORE_SEND_2NDCHELLO,

	/** state that is after sending the second client hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_AFTER_SEND_2NDCHELLO,

	/** state that is before receiving the second server hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_BEFORE_RECV_2NDSHELLO,

	/** state that is after receiving the second server hello handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_AFTER_RECV_2NDSHELLO,

	/** state that is after receiving the hello retry request handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_AFTER_RECV_HRREQ,

	/** state that is after receiving the encrypted extensions
	 * handshake protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_ENCEXT,

	/** state that is after receiving the certificate handshake
	 * protocol message in the server side. this is a client side
	 * state. */
	TLS_STATE_HS_RECV_SCERT,

	/** state that is after receiving the server key exchange
	 * handshake protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_SKEYEXC,

	/** state that is after receiving (server) certificate request
	 * handshake protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_SCERTVFY,

	/** state that is after receiving the certificate request
	 * handshake protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_CERTREQ,

	/** state that is after receiving server hello done handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_RECV_SHELLODONE,

	/* state that is after sending certificate handshake protocol
	 * message in the client side. this is a client side state. */
	TLS_STATE_HS_SEND_CCERT,

	/* state that is after sending client key exchange handshake
	 * protocol message. this is a client side state. */
	TLS_STATE_HS_SEND_CKEYEXC,

	/** state that is after sending (client) certificate request
	 * handshake protocol message. this is a client side state. */
	TLS_STATE_HS_SEND_CCERTVFY,

	/** state that is before receiving client hello handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_BEFORE_RECV_CHELLO,

	/** state that is after receiving client hello handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_AFTER_RECV_CHELLO,

	/** state that is before receiving second client hello handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_BEFORE_RECV_2NDCHELLO,

	/** state that is after receiving second client hello handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_AFTER_RECV_2NDCHELLO,

	/** state that is after sending server hello handshake protocol
	 * message. this is a server side state. */
	TLS_STATE_HS_SEND_SHELLO,

	/** state that is before sending hello retry request handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_BEFORE_SEND_HRREQ,

	/** state that is after sending encrypted extensions handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_SEND_ENCEXT,

	/** state that is after sending certificate handshake protocol
	 * message in the server side. this is a server state. */
	TLS_STATE_HS_SEND_SCERT,

	/** state that is after sending certificate verify handshake
	 * protocol message in the server side. this is a server state. */
	TLS_STATE_HS_SEND_SCERTVFY,

	/** state that is after sending server key exchange handshake
	 * protocol mesasge. this is a server side state. */
	TLS_STATE_HS_SEND_SKEYEXC,

	/** state that is after sending certificate request handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_SEND_CERTREQ,

	/** state that is after sending server hello done handhsake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_SEND_SHELLODONE,

	/** state that is after receiving certificate handshake protocol
	 * message. this is a server side state. */
	TLS_STATE_HS_RECV_CCERT,

	/** state that is after receiving client key exchange handshake
	 * protocol message. this is a server side state. */
	TLS_STATE_HS_RECV_CKEYEXC,

	/** state that is after receiving (client) certificate verify
	 * handshake protocol message. this is a server side state. */
	TLS_STATE_HS_RECV_CCERTVFY,

	/** state that is after receiving finish handshake protocol
	 * message. this is a state of both client and server. */
	TLS_STATE_HS_RECV_FINISH,

	/** state that is after sending finish handshake protocol
	 * message. this is a state of both client and server. */
	TLS_STATE_HS_SEND_FINISH,

	/** state that is before sending finish handshake protocol
	 * message. this is a state of both client and server. */
	TLS_STATE_HS_BEFORE_FINISH,

	/** state that is after sending change cipher spec
	 * protocol. this is a state of both client and server. */
	TLS_STATE_CCS_SEND,

	/** state that is after receiving change cipher spec
	 * protocol. this is a state of both client and server. */
	TLS_STATE_CCS_RECV,

	/** state that is after the handshake. this is a state of both
	 * client and server. */
	TLS_STATE_ESTABLISHED,

	/** state that is connection closed. this is a state of both
	 * client and server. */
	TLS_STATE_CLOSED
};

/**
 * end entity of tls connection.
 */
enum connection_end {
	/** entity is client. */
	TLS_CONNECT_CLIENT,

	/** entity is server. */
	TLS_CONNECT_SERVER
};

/**
 * TLS option structure.
 *
 * this struct stores value of optional parameter.
 */
struct tls_option {
	/** read(2)/write(2) timeout seconds. */
	struct timespec timeout;

	/* certificate verification type */
	enum tls_cert_verify_type verify_type;

	/* certificate verification depth */
	uint32_t verify_depth;

	/* start handshake after accept(2) immediately. */
	bool immediate_handshake;

	/* server sends certificate request. */
	bool use_certreq;

	/* server skip the check of the client certificate is
	 * non-null. */
	bool skip_ccertificate_check;
};

/**
 * Enumrateds of TLS key exchange algorthm.
 *
 * RFC5246 7.4.3.  Server Key Exchange Message
 *
 * enum { dhe_dss, dhe_rsa, dh_anon, rsa,dh_dss, dh_rsa
 *      } KeyExchangeAlgorithm;
 *
 * RFC4492 5.4.  Server Key Exchange
 *
 * The ServerKeyExchange message is extended as follows.
 *
 *      enum { ec_diffie_hellman } KeyExchangeAlgorithm;
 *
 * NOTE: In TLS 1.1 (RFC4346), KeyExchangeAlgorithm 'diffie_hellman'
 * was defined. It was changed to {dhe_dss, dhe_rsa, dh_anon, dh_dss,
 * dh_rsa} in TLS1.2.  'ec_diffie_hellman' has also changed to
 * {ecdh_ecdsa, ecdhe_ecdsa, ecdh_rsa, ecdhe_rsa, ecdh_anon} according
 * the precedent.
 *
 * RFC8446 2.  Protocol Overview
 *
 *   TLS supports three basic key exchange modes:
 *
 *   -  (EC)DHE (Diffie-Hellman over either finite fields or elliptic
 *      curves)
 *
 *   -  PSK-only
 *
 *   -  PSK with (EC)DHE
 *
 * NOTE: In TLS 1.3, KeyExchangeAlgorithm is not defined. However, for
 * backward compatibility of implementation, new key exchange algorithms
 * are added to this type.
 */
enum tls_keyexchange_method {
	/** Undefined by RFC.
	 * This is used as initial value and to handle error.
	 */
	TLS_KXC_NULL,

	/** Ephemeral DH with DSS signatures */
	TLS_KXC_DHE_DSS,

	/** Ephemeral DH with RSA signatures */
	TLS_KXC_DHE_RSA,

	/** Anonymous DH, no signatures */
	TLS_KXC_DH_anon,

	/** RSA key exchange */
	TLS_KXC_RSA,

	/** DH with DSS-based certificates */
	TLS_KXC_DH_DSS,

	/** DH with RSA-based certificates */
	TLS_KXC_DH_RSA,

	/** Fixed ECDH with ECDSA-signed certificates.
	 * RFC 4492 section 2.
	 */
	TLS_KXC_ECDH_ECDSA,

	/** Ephemeral ECDH with ECDSA signatures.
	 * RFC 4492 section 2.
	 */
	TLS_KXC_ECDHE_ECDSA,

	/** Fixed ECDH with RSA-signed certificates.
	 * RFC 4492 section 2.
	 */
	TLS_KXC_ECDH_RSA,

	/** Ephemeral ECDH with RSA signatures.
	 * RFC 4492 section 2.
	 */
	TLS_KXC_ECDHE_RSA,

	/** Anonymous ECDH, no signatures.
	 * RFC 4492 section 2.
	 */
	TLS_KXC_ECDH_anon,

	/** Ephemeral DH
	 * RFC 8446 section 2.
	 */
	TLS_KXC_DHE,

	/** Ephemeral ECDH for TLS 1.3.
	 * RFC 8446 section 2.
	 */
	TLS_KXC_ECDHE,

	/** Pre-Shared Key for TLS 1.3.
	 * RFC 8446 section 2.
	 */
	TLS_KXC_PSK,

	/** Pre-Shared Key with Ephemeral DH for TLS 1.3.
	 * RFC 8446 section 2.
	 */
	TLS_KXC_PSK_DHE,

	/** Pre-Shared Key with Ephemeral ECDH for TLS 1.3.
	 * RFC 8446 section 2.
	 */
	TLS_KXC_PSK_ECDHE,
};

/**
 * @see tls_session.h
 */
struct tls_session_param;

/**
 * @see tls_record.h
 */
struct tls_record_queue;

/**
 * @see tls_handshake.h
 */
struct tls_hs_interim_params;

/**
 * @see tls_handshake.h
 */
struct tls_handshake_queue;

/**
 * @see tls_cert.h
 */
struct tls_cert_type_list;

/**
 * @see ext/tls_sighash.h
 */
struct tls_hs_sighash_list;

/**
 * @see tls_handshake.h
 */
struct tls_hs_ecdh;

/**
 * @see extension/tls_cookie.h
 */
struct tls_hs_cookie;

/**
 * tls structure that has tls connection, option, state, and so on.
 */
struct tls {
	/**
	 * socket descriptor for this structure.
	 */
	int sockfd;

	/**
	 * tls connection status (save processing status)
	 */
	enum tls_state state;

	/**
	 * first handshake is over or not.
	 *
	 * if first handshake is over, this variable is true. otherwise
	 * false. this variable is used by judgment of alert
	 * (application data is not acceptable under the first
	 * handshake) and so on.
	 */
	bool handshake_over;

	/**
	 * resession or not.
	 *
	 * if true, current handshake is re-session.
	 */
	bool resession;

	/**
	 * nrg_tls option values.
	 */
	struct tls_option opt;

	/**
	 * a error status about the tls connection.
	 *
	 * if no error has occurred, this variable is 0.
	 */
	enum tls_errno errnum;

	/**
	 * PKCS#12 structure.
	 *
	 * save certificate chain which was gotten by client certificate
	 * handshake protocol message.
	 */
	PKCS12 *pkcs12_client;

	/**
	 * PKCS#12 structure.
	 *
	 * save certificate chain which was gotten by server certificate
	 * handshake protocol message.
	 */
	PKCS12 *pkcs12_server;

	/**
	 * aicrypto's Store Manager structure.
	 */
	STManager *store_manager;

	/**
	 * path of aicrypto's Store Manager.
	 */
	char *store_manager_path;

	/**
	 * connection end.
	 */
	enum connection_end entity;

	/**
	 * the version that negotiated by clientHello and serverHello
	 * handshake protocol message. */
	struct tls_protocol_version negotiated_version;

	/**
	 * the version that was sent by clientHello handshake protocol
	 * message.
	 *
	 * this variable is used by client key exchange handshake
	 * protocol message to generate premaster secret (premaster
	 * secret uses the version that included in the
	 * clientHello.). */
	struct tls_protocol_version client_version;

	/**
	 * the version that is set to tls record.
	 *
	 * this variable is used to control protocol version in tls record according
	 * to specification of each protocol.
	 *
	 * RFC8446 5.1 says:
	 *
	 * This document describes TLS 1.3, which uses the version 0x0304.  This
	 * version value is historical, deriving from the use of 0x0301 for
	 * TLS 1.0 and 0x0300 for SSL 3.0.  In order to maximize backward
	 * compatibility, a record containing an initial ClientHello SHOULD have
	 * version 0x0301 (reflecting TLS 1.0) and a record containing a second
	 * ClientHello or a ServerHello MUST have version 0x0303 (reflecting
	 * TLS 1.2).  When negotiating prior versions of TLS, endpoints follow
	 * the procedure and requirements provided in Appendix D.
	 *
	 * RFC8446 Appendix D. says:
	 *
	 * Prior versions of TLS used the record layer version number
	 * (TLSPlaintext.legacy_record_version and
	 * TLSCiphertext.legacy_record_version) for various purposes.  As of
	 * TLS 1.3, this field is deprecated.  The value of
	 * TLSPlaintext.legacy_record_version MUST be ignored by all
	 * implementations.  The value of TLSCiphertext.legacy_record_version is
	 * included in the additional data for deprotection but MAY otherwise be
	 * ignored or MAY be validated to match the fixed constant value.
	 * Version negotiation is performed using only the handshake versions
	 * (ClientHello.legacy_version and ServerHello.legacy_version, as well
	 * as the ClientHello, HelloRetryRequest, and ServerHello
	 * "supported_versions" extensions).  In order to maximize
	 * interoperability with older endpoints, implementations that negotiate
	 * the use of TLS 1.0-1.2 SHOULD set the record layer version number to
	 * the negotiated version for the ServerHello and all records
	 * thereafter.
	 *
	 * RFC5246 Appendix E.1. says:
	 *
	 * TLS clients that wish to negotiate with older servers MAY send any
	 * value {03,XX} as the record layer version number.  Typical values
	 * would be {03,00}, the lowest version number supported by the client,
	 * and the value of ClientHello.client_version.  No single value will
	 * guarantee interoperability with all old servers, but this is a
	 * complex topic beyond the scope of this document.
	 */
	struct tls_protocol_version record_version;

	/**
	 * list of protocol versions used to request and accept protocol version
	 * at version negotiation.
	 */
	struct tls_protocol_version_list supported_versions;

	/**
	 * list of protocol versions sent by peer at version negotiation.
	 */
	struct tls_protocol_version_list peer_supported_versions;

	/**
	 * premaster secret
	 *
	 * this is a used by handshake process temporarilly.
	 *
	 * pre_master_secret length: RSA is 48 bytes, SECP256R1 is 32 bytes
	 * SECP384R1 is 48 bytes, SECP521R1 is 66 bytes, X25519 is 32 bytes,
	 * X448 is 56 bytes.
	 */
	uint8_t premaster_secret[66];

	/**
	 * premaster secret length.
	 *
	 * pre_master_secret length varies by key exchange method, so this
	 * variable holds the length.
	 */
	uint32_t premaster_secret_len;

	/**
	 * early_secret.
	 *
	 * this is used by handshake process temporarily.
	 */
	uint8_t *early_secret;

	/**
	 * handshake_secret.
	 *
	 * this is used by handshake process temporarily.
	 */
	uint8_t *handshake_secret;

	/**
	 * application_secret_context.
	 *
	 * this is used by handshake process temporarily.
	 */
	uint8_t application_secret_context[TLS_APPLICATION_SECRET_CONTEXT_SIZE_MAX];

	/**
	 * selected key exchange method (enum KeyExchangeAlgorithm).
	 *
	 * Values defined at:
	 * RFC5246 7.4.3.  Server Key Exchange Message
	 * RFC4492 5.4.  Server Key Exchange
	 */
	enum tls_keyexchange_method keymethod;

	/**
	 * client random.
	 *
	 * moved from security parameter because client random does not
	 * be reused (old client random value is not used when
	 * re-session or re-negotiation occurred). */
	uint8_t client_random[32];

	/**
	 * server random.
	 *
	 * moved from security parameter because server random does not
	 * be reused (old server random value is not used when
	 * re-session or re-negotiation occurred). */
	uint8_t server_random[32];

	/**
	 * list of available certificate type in the server.
	 *
	 * this member set by certificate request handshake protocol
	 * message. and, used by certificate verify handshake protocol
	 * message to make digitally-signed structure.
	 */
	struct tls_cert_type_list *certtype_list_server;

	/**
	 * list of available signature/hash algorithm pair
	 *
	 * this variable is used by both client hello extesion
	 * `signature algorithm` and certificate request commonly. in
	 * client hello extension `signature algorithm`, server save
	 * that value to this variable. in certificate request handshake
	 * protocol message, client save this that value to this
	 * variable.
	 */
	struct tls_hs_sighash_list *sighash_list;

	/**
	 * list of available signature/hash algorithm pair from
	 * "signature_algorithms_cert" extension.
	 *
	 * this variable is used by both client and server. a list of
	 * signature/hash algorithm pairs is stored to the variable when
	 * receiving "signature_algorithms_cert" extension. server may receive
	 * "signature_algorithms_cert" extension in client hello. client may
	 * receive "signature_algorithms_cert" extension in certificate request.
	 */
	struct tls_hs_sighash_list *sighash_list_cert;

	/**
	 * ECDH data.
	 */
	struct tls_hs_ecdh *ecdh;

	/**
	 * Temporary storage area of cipher suites list from client.
	 *
	 * ClientHello.cipher_suites.
	 */
	struct tls_cipher_list *chello_cipher_suites;

	/**
	 * Head of ServerKeyExchange.params (for digitally-signing).
	 */
	uint8_t *skeyexc_params;

	/**
	 * Length of ServerKeyExchange.params (for digitally-signing).
	 */
	uint32_t skeyexc_params_len;

#if 0
	/**
	 * list of available certificate authorities in the server.
	 *
	 * if this member set by certifcate request handshake protocol
	 * message in the client. and, used by certifcate verify
	 * handshake protocol message to make digitally-signed structure
	 * (choose certificate). if this value is NULL, all certificate
	 * are available.
	 */
	char** ca_list_server;
#endif /* 0 */

	/**
	 * send client certificate or not.
	 *
	 * if client send client certificate as null data, this variable
	 * true. in this case, client do not sends certificate verify
	 * handshake protocol message.
	 */
	bool ccert_null;

	/**
	 * client received certificate request handshake protocol or
	 * not.
	 *
	 * if client received the certificate request handshake protocol
	 * message, or server sends the certificate request handshake
	 * protocol message, this variable true. in this case, client
	 * sends client certificate handshake protocol message.
	 */
	bool certreq_used;


	/**
	 * client certificate authentication used or not.
	 *
	 * this variable is falsed by default. this varialbe is prepared
	 * for compatible with aissl.
	 *
	 * server side:
	 *
	 * whether server could verify certificate verify handshake
	 * protocol message. if verified, this variable true. otherwise
	 * false.
	 *
	 * client side:
	 *
	 * whether client could send certificate verify handshake
	 * protocol message. if sent, this variable true. otherwise
	 * false.
	 */
	bool is_ccert_auth;

	/**
	 * dns hostname of server used in server name indication.
	 *
	 * if server_name is set, client tells server a hostname that client
	 * wants to access and requests server to use a certificate of the
	 * hostname as server name indication.
	 */
	char *server_name;

	/**
	 * cookie content sent by server in hello retry request.
	 */
	struct tls_hs_cookie *cookie;

	/**
	 * send key update message or not.
	 *
	 * if peer send key update message set to update_requested, this
	 * variable is set to true until send key update message to the peer.
	 * otherwise, this variable is set to false.
	 */
	bool need_sending_keyupdate;

	/**
	 * queue for saving handshake message
	 *
	 * this queue is used by finished and certificate verify for
	 * calculating digest.
	 */
	struct tls_handshake_queue *queue_handshake;

	/**
	 * md5 hash context of current handshake.
	 */
	MD5_CTX handshake_md5_ctx;

	/**
	 * sha1 hash context of current handshake.
	 */
	SHA1_CTX handshake_sha1_ctx;

	/**
	 * sha224 hash context of current handshake.
	 */
	SHA256_CTX handshake_sha224_ctx;

	/**
	 * sha256 hash context of current handshake.
	 */
	SHA256_CTX handshake_sha256_ctx;

	/**
	 * sha384 hash context of current handshake.
	 */
	SHA512_CTX handshake_sha384_ctx;

	/**
	 * sha512 hash context of current handshake.
	 */
	SHA512_CTX handshake_sha512_ctx;

	/**
	 * pending status.
	 *
	 * pending status does not have connection. there is session
	 * parameter (security parameter in term of RFC 5246) only.
	 */
	struct tls_session_param *pending;

	/**
	 * current read connection status
	 */
	struct tls_connection active_read;

	/**
	 * current write connection status
	 */
	struct tls_connection active_write;

	/* prepare the queue for applicaton and handshake protocol
	 * data. ccs protocol and alert protocol does not be stored in
	 * the queue. these protocol data is processed immediately. */

	/**
	 * the queue that stores tls record layer application protocol
	 * data.
	 */
	struct tls_record_queue *queue_data;

	/**
	 * the queue that stores tls record layer handshake protocol
	 * data.
	 */
	struct tls_record_queue *queue_control;

	/**
	 * structure that stores parameters to be interpreted with protocol
	 * version context after version negotiation.
	 */
	struct tls_hs_interim_params *interim_params;

	/**
	 * structure that stores parameters interpreted in first client or
	 * server hello. this is used to verify values in second client hello.
	 */
	struct tls_hs_interim_params *first_hello_params;
};

#include "tls_util.h"

#endif /* INCLUSION_GUARD_UUID_6C553695_8E3E_23FA_E42D_485AA53E43E1 */
