/*
 * Copyright (c) 2016-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#ifndef INCLUSION_GUARD_UUID_22187E75_470C_7FCB_385D_7509075166A1
#define INCLUSION_GUARD_UUID_22187E75_470C_7FCB_385D_7509075166A1

#include "tls.h"

/** @see tls_handshake.h */
struct tls_hs_msg;

/**
 * available hash algorithm.
 * this enum is defined in RFC 5246 section 7.4.1.4.1.
 */
enum tls_hs_sighash_hash_algo {
	/** NONE */
	TLS_HASH_ALGO_NONE = 0x00U,

	/* MD5 */
	TLS_HASH_ALGO_MD5 = 0x01U,

	/** SHA1 */
	TLS_HASH_ALGO_SHA1 = 0x02U,

	/** SHA224 */
	TLS_HASH_ALGO_SHA224 = 0x03U,

	/** SHA256 */
	TLS_HASH_ALGO_SHA256 = 0x04U,

	/** SHA384 */
	TLS_HASH_ALGO_SHA384 = 0x05U,

	/** SHA512 */
	TLS_HASH_ALGO_SHA512 = 0x06U
};

/**
 * available signature algorithm.
 * this enum is defined in RFC 5246 section 7.4.1.4.1.
 */
enum tls_hs_sighash_sig_algo {
	/** Anonymouse */
	TLS_SIG_ALGO_ANON = 0x00U,

	/** RSA */
	TLS_SIG_ALGO_RSA = 0x01U,

	/** DSA */
	TLS_SIG_ALGO_DSA = 0x02U,

	/** ECDSA */
	TLS_SIG_ALGO_ECDSA = 0x03U
};

/**
 * available signature scheme.
 *
 * RFC5246 7.4.1.4.1.  Signature Algorithms
 *
 *
 *       enum {
 *           none(0), md5(1), sha1(2), sha224(3), sha256(4), sha384(5),
 *           sha512(6), (255)
 *       } HashAlgorithm;
 *
 *       enum { anonymous(0), rsa(1), dsa(2), ecdsa(3), (255) }
 *         SignatureAlgorithm;
 *
 *       struct {
 *             HashAlgorithm hash;
 *             SignatureAlgorithm signature;
 *       } SignatureAndHashAlgorithm;
 *
 * RFC8446 4.2.3.  Signature Algorithms
 *
 *       enum {
 *           / RSASSA-PKCS1-v1_5 algorithms /
 *           rsa_pkcs1_sha256(0x0401),
 *           rsa_pkcs1_sha384(0x0501),
 *           rsa_pkcs1_sha512(0x0601),
 *
 *           / ECDSA algorithms /
 *           ecdsa_secp256r1_sha256(0x0403),
 *           ecdsa_secp384r1_sha384(0x0503),
 *           ecdsa_secp521r1_sha512(0x0603),
 *
 *           / RSASSA-PSS algorithms with public key OID rsaEncryption /
 *           rsa_pss_rsae_sha256(0x0804),
 *           rsa_pss_rsae_sha384(0x0805),
 *           rsa_pss_rsae_sha512(0x0806),
 *
 *           / EdDSA algorithms /
 *           ed25519(0x0807),
 *           ed448(0x0808),
 *
 *           / RSASSA-PSS algorithms with public key OID RSASSA-PSS /
 *           rsa_pss_pss_sha256(0x0809),
 *           rsa_pss_pss_sha384(0x080a),
 *           rsa_pss_pss_sha512(0x080b),
 *
 *           / Legacy algorithms /
 *           rsa_pkcs1_sha1(0x0201),
 *           ecdsa_sha1(0x0203),
 *
 *           / Reserved Code Points /
 *           private_use(0xFE00..0xFFFF),
 *           (0xFFFF)
 *       } SignatureScheme;
 */
enum tls_hs_sighash_sig_scheme {
	/* not defined by RFC8446 and generated by RFC5246 */
	TLS_SS_ANON_NONE              = 0x0000,
	TLS_SS_ANON_MD5               = 0x0100,
	TLS_SS_ANON_SHA1              = 0x0200,
	TLS_SS_ANON_SHA224            = 0x0300,
	TLS_SS_ANON_SHA256            = 0x0400,
	TLS_SS_ANON_SHA384            = 0x0500,
	TLS_SS_ANON_SHA512            = 0x0600,

	TLS_SS_RSA_PKCS1_NONE         = 0x0001,
	TLS_SS_RSA_PKCS1_MD5          = 0x0101,
	TLS_SS_RSA_PKCS1_SHA224       = 0x0301,

	TLS_SS_DSA_NONE               = 0x0002,
	TLS_SS_DSA_MD5                = 0x0102,
	TLS_SS_DSA_SHA1               = 0x0202,
	TLS_SS_DSA_SHA224             = 0x0302,
	TLS_SS_DSA_SHA256             = 0x0402,
	TLS_SS_DSA_SHA384             = 0x0502,
	TLS_SS_DSA_SHA512             = 0x0602,

	TLS_SS_ECDSA_NONE             = 0x0003,
	TLS_SS_ECDSA_MD5              = 0x0103,
	TLS_SS_ECDSA_SHA224           = 0x0303,

	/* Legacy algorithms defined by RFC5246 and RFC8446 */
	TLS_SS_RSA_PCKS1_SHA1         = 0x0201,
	TLS_SS_ECDSA_SHA1             = 0x0203,

	/* RSASSA-PKCS1-v1_5 algorithms defined by RFC5246 and RFC8446 */
	TLS_SS_RSA_PKCS1_SHA256       = 0x0401,
	TLS_SS_RSA_PKCS1_SHA384       = 0x0501,
	TLS_SS_RSA_PKCS1_SHA512       = 0x0601,

	/* ECDSA algorithms defined by RFC5246 and RFC8446 */
	TLS_SS_ECDSA_SECP256R1_SHA256 = 0x0403,
	TLS_SS_ECDSA_SECP384R1_SHA384 = 0x0503,
	TLS_SS_ECDSA_SECP521R1_SHA512 = 0x0603,

	/*
	 * RSASSA-PSS algorithms with public key OID rsaEncryption defined by
	 * RFC8446
	 */
	TLS_SS_RSA_PSS_RSAE_SHA256    = 0x0804,
	TLS_SS_RSA_PSS_RSAE_SHA384    = 0x0805,
	TLS_SS_RSA_PSS_RSAE_SHA512    = 0x0806,

	/* EdDSA algorithms defined by RFC5246 and RFC8446 */
	TLS_SS_ED25519                = 0x0807,
	TLS_SS_ED448                  = 0x0808,

	/*
	 * RSASSA-PSS algorithms with public key OID RSASSA-PSS defined by
	 * RFC5246 and RFC8446
	 */
	TLS_SS_RSA_PSS_PSS_SHA256     = 0x0809,
	TLS_SS_RSA_PSS_PSS_SHA384     = 0x080A,
	TLS_SS_RSA_PSS_PSS_SHA512     = 0x080B,

	/*
	 * reserved (0xFE00..0xFFFF)
	 */
};

/**
 * maximum size of signature_algorithms extension.
 */
#define TLS_EXT_SIG_ALGO_SIZE_MAX	((2 << (16 - 1)) -2)

/**
 * signature/hash algorithm pair structure.
 */
struct tls_hs_sighash_algo {
	/** hash algorithm */
	enum tls_hs_sighash_hash_algo hash;

	/** signature algorithm  */
	enum tls_hs_sighash_sig_algo sig;
};

/* list of signature/hash algorithm pair. */
struct tls_hs_sighash_list {
	/** length of list */
	int32_t len;

	/** list of signature/hash algorithm pair */
	struct tls_hs_sighash_algo *list;
};

/**
 * get signature scheme value from tls_hs_sighash_algo structure.
 */
uint16_t tls_hs_sighash_convert_sighash_to_sigscheme(
    struct tls_hs_sighash_algo *sigalgo);

/**
 * get list of signature/hash algorithm that is used by default.
 *
 * default determines by TLS/SSL version. retrun value of this functin
 * is memory allocated value. if that value became unnecessary, you must
 * free by tls_hs_sighash_free function.
 */
struct tls_hs_sighash_list * tls_hs_sighash_list(TLS *tls);

/**
 * get list of signature/hash algorithm that is used for
 * signature_algorithms_cert extension.
 *
 * default determines by TLS/SSL version. retrun value of this functin
 * is memory allocated value. if that value became unnecessary, you must
 * free by tls_hs_sighash_free function.
 */
struct tls_hs_sighash_list * tls_hs_sighash_list_cert(TLS *tls);

/**
 * free memory that is allocated by tls_hs_sighash_list function.
 */
void tls_hs_sighash_free(struct tls_hs_sighash_list *sighash);

/**
 * check availability of specified signature/hash algorithm pair.
 */
bool tls_hs_sighash_availablep(const TLS *tls,
			       const struct tls_hs_sighash_algo sighash);

/**
 * check availability of specified signature/hash algorithm pair in
 * signature algorithm cert extension.
 */
bool tls_hs_sighash_cert_availablep(const TLS *tls,
				    const struct tls_hs_sighash_algo sighash);

/**
 * get size of specified hash algorithm.
 */
int32_t tls_hs_sighash_get_hash_size_by_ss(const uint16_t sigscheme);

int32_t tls_hs_sighash_get_hash_size(const enum tls_hs_sighash_hash_algo hash);

/**
 * get constant definition of specified hash type in the aicrypto.
 */
int32_t tls_hs_sighash_get_ai_hash_type_by_ss(uint16_t sigscheme);

int32_t tls_hs_sighash_get_ai_hash_type(enum tls_hs_sighash_hash_algo hash);

/**
 * get constant definition of specified signature type in the aicrypto.
 */
int32_t tls_hs_sighash_get_ai_sig_type_by_ss(uint16_t sigscheme);

int32_t tls_hs_sighash_get_ai_sig_type(struct tls_hs_sighash_algo sighash);

/**
 * get hash type value from signature scheme.
 */
int32_t tls_hs_sighash_get_hash_type(uint16_t sigscheme,
				    enum tls_hs_sighash_hash_algo *hash_algo);

/**
 * write signature algorithm extension to the handshake data.
 *
 * this function is used in not only composing of client hello but also
 * composing certificate request.
 */
int32_t tls_hs_sighash_write(TLS *tls, struct tls_hs_msg *msg);

/**
 * write signature algorithm cert extension to the handshake data.
 *
 * this function is used in not only composing of client hello but also
 * composing certificate request.
 */
int32_t tls_hs_sighash_cert_write(TLS *tls, struct tls_hs_msg *msg);

/**
 * read received data by considering it the signature algorithm extension.
 *
 * this function is used in not only parsing of client hello but also
 * parsing certificate request.
 */
int32_t tls_hs_sighash_read(TLS *tls,
			    const struct tls_hs_msg *msg,
			    const uint32_t offset);

/**
 * read received data by considering it the signature algorithm cert extension.
 *
 * this function is used in not only parsing of client hello but also
 * parsing certificate request.
 */
int32_t tls_hs_sighash_cert_read(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset);

/**
 * get default list of signature/hash algorithm pair when signature/hash
 * algorithm extension is not provided by client.
 *
 * this function is called when client do not provide the signature
 * algorithm extension. in that case, determine default list of
 * signature/hash algorithm pair by negotiated key exchange method.
 */
bool tls_hs_sighash_save_default(TLS *tls);

#endif /* INCLUSION_GUARD_UUID_22187E75_470C_7FCB_385D_7509075166A1 */
