/*
 * Copyright (c) 2016-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file tls_ecc.h
 *
 * This file defines function and structures to Client Hello Extensions.
 */

/**
 * @defgroup tls_ecc Client Hello Extensions
 *
 * This module provides ECC API for TLS connection.
 *
 * For specifications of the extensions, see RFC 4492.
 */

#ifndef INCLUSION_GUARD_UUID_E109B1CF_88C1_4DD1_A430_25B914AC2B69
#define INCLUSION_GUARD_UUID_E109B1CF_88C1_4DD1_A430_25B914AC2B69

#include "tls.h"

/** @see tls_handshake.h */
struct tls_hs_msg;

/*
 * RFC4492 5.1.1.  Supported Elliptic Curves Extension
 *
 *      enum {
 *          sect163k1 (1), sect163r1 (2), sect163r2 (3),
 *          sect193r1 (4), sect193r2 (5), sect233k1 (6),
 *          sect233r1 (7), sect239k1 (8), sect283k1 (9),
 *          sect283r1 (10), sect409k1 (11), sect409r1 (12),
 *          sect571k1 (13), sect571r1 (14), secp160k1 (15),
 *          secp160r1 (16), secp160r2 (17), secp192k1 (18),
 *          secp192r1 (19), secp224k1 (20), secp224r1 (21),
 *          secp256k1 (22), secp256r1 (23), secp384r1 (24),
 *          secp521r1 (25),
 *          reserved (0xFE00..0xFEFF),
 *          arbitrary_explicit_prime_curves(0xFF01),
 *          arbitrary_explicit_char2_curves(0xFF02),
 *          (0xFFFF)
 *      } NamedCurve;
 *
 * RFC4492 Appendix A.  Equivalent Curves (Informative)
 *
 *           ------------------------------------------
 *                     Curve names chosen by
 *                different standards organizations
 *           ------------+---------------+-------------
 *           SECG        |  ANSI X9.62   |  NIST
 *           ------------+---------------+-------------
 *           sect163k1   |               |   NIST K-163
 *           sect163r1   |               |
 *           sect163r2   |               |   NIST B-163
 *           sect193r1   |               |
 *           sect193r2   |               |
 *           sect233k1   |               |   NIST K-233
 *           sect233r1   |               |   NIST B-233
 *           sect239k1   |               |
 *           sect283k1   |               |   NIST K-283
 *           sect283r1   |               |   NIST B-283
 *           sect409k1   |               |   NIST K-409
 *           sect409r1   |               |   NIST B-409
 *           sect571k1   |               |   NIST K-571
 *           sect571r1   |               |   NIST B-571
 *           secp160k1   |               |
 *           secp160r1   |               |
 *           secp160r2   |               |
 *           secp192k1   |               |
 *           secp192r1   |  prime192v1   |   NIST P-192
 *           secp224k1   |               |
 *           secp224r1   |               |   NIST P-224
 *           secp256k1   |               |
 *           secp256r1   |  prime256v1   |   NIST P-256
 *           secp384r1   |               |   NIST P-384
 *           secp521r1   |               |   NIST P-521
 *           ------------+---------------+-------------
 *
 * RFC8422 5.1.1.  Supported Elliptic Curves Extension
 *
 *          enum {
 *              deprecated(1..22),
 *              secp256r1 (23), secp384r1 (24), secp521r1 (25),
 *              x25519(29), x448(30),
 *              reserved (0xFE00..0xFEFF),
 *              deprecated(0xFF01..0xFF02),
 *              (0xFFFF)
 *          } NamedCurve;
 *
 * RFC8446 4.2.7.  Supported Groups
 *
 *      enum {
 *
 *          / Elliptic Curve Groups (ECDHE) /
 *          secp256r1(0x0017), secp384r1(0x0018), secp521r1(0x0019),
 *          x25519(0x001D), x448(0x001E),
 *
 *          / Finite Field Groups (DHE) /
 *          ffdhe2048(0x0100), ffdhe3072(0x0101), ffdhe4096(0x0102),
 *          ffdhe6144(0x0103), ffdhe8192(0x0104),
 *
 *          / Reserved Code Points /
 *          ffdhe_private_use(0x01FC..0x01FF),
 *          ecdhe_private_use(0xFE00..0xFEFF),
 *          (0xFFFF)
 *      } NamedGroup;
 */
/**
 * RFC4492 5.1.1.  Supported Elliptic Curves Extension
 *
 * @ingroup tls_ecc
 */
enum tls_hs_named_curve {
	/*
	 * Curves below are defined by RFC4492 and deprecated by RFC8422.
	 */
	TLS_ECC_CURVE_SECT163K1 = 1U,
	TLS_ECC_CURVE_SECT163R1 = 2U,
	TLS_ECC_CURVE_SECT163R2 = 3U,
	TLS_ECC_CURVE_SECT193R1 = 4U,
	TLS_ECC_CURVE_SECT193R2 = 5U,
	TLS_ECC_CURVE_SECT233K1 = 6U,
	TLS_ECC_CURVE_SECT233R1 = 7U,
	TLS_ECC_CURVE_SECT239K1 = 8U,
	TLS_ECC_CURVE_SECT283K1 = 9U,
	TLS_ECC_CURVE_SECT283R1 = 10U,
	TLS_ECC_CURVE_SECT409K1 = 11U,
	TLS_ECC_CURVE_SECT409R1 = 12U,
	TLS_ECC_CURVE_SECT571K1 = 13U,
	TLS_ECC_CURVE_SECT571R1 = 14U,
	TLS_ECC_CURVE_SECP160K1 = 15U,
	TLS_ECC_CURVE_SECP160R1 = 16U,
	TLS_ECC_CURVE_SECP160R2 = 17U,
	TLS_ECC_CURVE_SECP192K1 = 18U,
	TLS_ECC_CURVE_SECP192R1 = 19U,
	TLS_ECC_CURVE_SECP224K1 = 20U,
	TLS_ECC_CURVE_SECP224R1 = 21U,
	TLS_ECC_CURVE_SECP256K1 = 22U,

	/*
	 * Curves below are defined by RFC4492, RFC8422 and RFC 8446.
	 */
	TLS_ECC_CURVE_SECP256R1 = 23U,      /* 0x0017 */
	TLS_NAMED_GROUP_SECP256R1 = 0x0017, /*    23U */
	TLS_ECC_CURVE_SECP384R1 = 24U,      /* 0x0018 */
	TLS_NAMED_GROUP_SECP384R1 = 0x0018, /*    24U */
	TLS_ECC_CURVE_SECP521R1 = 25U,      /* 0x0019 */
	TLS_NAMED_GROUP_SECP521R1 = 0x0019, /*    25U */

	/*
	 * Curves below are define by RFC8422 and RFC8446.
	 */
	TLS_ECC_CURVE_X25519    = 29U,      /* 0x001D */
	TLS_NAMED_GROUP_X25519  = 0x001D,   /*    29U */
	TLS_ECC_CURVE_X448      = 30U,      /* 0x001E */
	TLS_NAMED_GROUP_X448    = 0x001E,   /*    30U */

	/*
	 * reserved (0x01FC..0x01FF), (0xFE00..0xFEFF)
	 */

	/*
	 * Curves below are defined by RFC4492 and deprecated by RFC8422.
	 */
	TLS_ECC_CURVE_ARBITRARY_EXPLICIT_PRIME_CURVES = 0XFF01U,
	TLS_ECC_CURVE_ARBITRARY_EXPLICIT_CHAR2_CURVES = 0XFF02U,
};

/**
 * RFC4492 5.1.2.  Supported Point Formats Extension
 *
 *      enum { uncompressed (0), ansiX962_compressed_prime (1),
 *             ansiX962_compressed_char2 (2), reserved (248..255)
 *      } ECPointFormat;
 *
 * RFC8422 5.1.2.  Supported Point Formats Extension
 *
 *           enum {
 *               uncompressed (0),
 *               deprecated (1..2),
 *               reserved (248..255)
 *           } ECPointFormat;
 *
 * @ingroup tls_ecc
 */
enum tls_hs_ecc_ec_point_format {
	TLS_ECC_PF_UNCOMPRESSED = 0,

	/*
	 * Formats below are defined by RFC4492 and deprecated by RFC8422.
	 */
	TLS_ECC_PF_ANSIX962_COMPRESSED_PRIME = 1,
	TLS_ECC_PF_ANSIX962_COMPRESSED_CHAR2 = 2,
	/*
	 * reserved (248..255)
	 */
};

/**
 * RFC4492 5.4.  Server Key Exchange
 *
 *      enum { explicit_prime (1), explicit_char2 (2),
 *             named_curve (3), reserved(248..255) } ECCurveType;
 *
 * RFC8422 5.4.  Server Key Exchange
 *
 *           enum {
 *               deprecated (1..2),
 *               named_curve (3),
 *               reserved(248..255)
 *           } ECCurveType;
 */
enum tls_hs_ecc_ec_curve_type {
	/*
	 * Curve types below are defined by RFC4492 and deprecated by RFC8422.
	 */
	TLS_ECC_CTYPE_EXPLICIT_PRIME = 1,
	TLS_ECC_CTYPE_EXPLICIT_CHAR2 = 2,

	/*
	 * Curve types below are defined by RFC4492 and RFC8422.
	 */
	TLS_ECC_CTYPE_NAMED_CURVE = 3,
	/*
	 * reserved(248..255)
	 */
};

/** List of received ec supported */
struct tls_hs_ecc_eclist {
	/** length of list */
	int32_t len;

	/** list of NamedCurve */
	enum tls_hs_named_curve *list;
};

/** List of received ec supported */
struct tls_hs_ecc_pflist {
	/** length of list */
	int32_t len;

	/** list of NamedCurve */
	enum tls_hs_ecc_ec_point_format *list;
};

/**
 * Write Supported Elliptic Curves Extension as the extension data.
 *
 * @param[in] tls TLS structure
 * @param[in] msg
 *
 * @returns
 * return length of ...
 *
 * @ingroup tls_ecc
 */
int32_t tls_hs_ecc_write_elliptic_curves(TLS *tls, struct tls_hs_msg *msg);

/**
 * Write Supported Point Formats Extension as the extension data.
 *
 * @param[in] tls TLS structure
 * @param[in] msg
 *
 * @returns
 * return length of ...
 *
 * @ingroup tls_ecc
 */
int32_t tls_hs_ecc_write_ec_point_formats(TLS *tls, struct tls_hs_msg *msg);

/**
 * Read Supported Elliptic Curves Extension from received data.
 *
 * @param[in] tls TLS structure
 * @param[in] msg
 * @param[in] offset
 *
 * @returns
 * return length of ...
 *
 * @ingroup tls_ecc
 */
int32_t tls_hs_ecc_read_elliptic_curves(TLS *tls,
					const struct tls_hs_msg *msg,
					const uint32_t offset);

/**
 * Read Supported Point Formats Extension from received data.
 *
 * @param[in] tls TLS structure
 * @param[in] msg
 * @param[in] offset
 *
 * @returns
 * return length of ...
 *
 * @ingroup tls_ecc
 */
int32_t tls_hs_ecc_read_point_format(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset);

/**
 * Set implemented named curve list to the structure.
 *
 * List is read-only. There is no need to free the memory.
 */
void tls_hs_ecc_get_supported_eclist(struct tls_hs_ecc_eclist *eclist);

/**
 * Set implemented point format list to the structure.
 *
 * List is read-only. There is no need to free the memory.
 */
void tls_hs_ecc_get_supported_pflist(struct tls_hs_ecc_pflist *pflist);

/**
 * free memory that is allocated to tls_hs_ecc_eclist structure.
 */
void tls_hs_ecc_eclist_free(struct tls_hs_ecc_eclist *eclist);

/**
 * free memory that is allocated to tls_hs_ecc_pflist structure.
 */
void tls_hs_ecc_pflist_free(struct tls_hs_ecc_pflist *pflist);

#endif /* INCLUSION_GUARD_UUID_E109B1CF_88C1_4DD1_A430_25B914AC2B69 */
