/*
 * Copyright (c) 2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"

/* TLS_DFPRINT */
#include "tls_util.h"

/* MD5 functions */
#include <aicrypto/ok_md5.h>

/* SHA1 functions */
#include <aicrypto/ok_sha1.h>

/* SHA2 functions */
#include <aicrypto/ok_sha2.h>

void tls_hs_signature_get_digest(const enum tls_hs_sighash_hash_algo hash,
				TLS *tls, uint8_t *seed)
{
	/*
	 * RFC4492 section 5.4.
	 *
	 *  select (SignatureAlgorithm) {
	 *      case ecdsa:
	 *          digitally-signed struct {
	 *              opaque sha_hash[sha_size];
	 *          };
	 *  } Signature;
	 *
	 *  ServerKeyExchange.signed_params.sha_hash
	 *      SHA(ClientHello.random + ServerHello.random +
	 *                                       ServerKeyExchange.params);
	 */
	const uint32_t crandlen = sizeof (tls->client_random);
	const uint32_t srandlen = sizeof (tls->server_random);
	MD5_CTX md5ctx;
	SHA1_CTX sha1ctx;
	SHA256_CTX sha224ctx;
	SHA256_CTX sha256ctx;
	SHA512_CTX sha384ctx;
	SHA512_CTX sha512ctx;

	switch (hash) {
	case TLS_HASH_ALGO_NONE:
		/* TODO */
		break;

	case TLS_HASH_ALGO_MD5:
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, tls->client_random, crandlen);
		MD5Update(&md5ctx, tls->server_random, srandlen);
		MD5Update(&md5ctx, tls->skeyexc_params,
				   tls->skeyexc_params_len);
		MD5Final(seed, &md5ctx);
		break;

	case TLS_HASH_ALGO_SHA1:
		SHA1init(&sha1ctx);
		SHA1update(&sha1ctx, tls->client_random, crandlen);
		SHA1update(&sha1ctx, tls->server_random, srandlen);
		SHA1update(&sha1ctx, tls->skeyexc_params,
				     tls->skeyexc_params_len);
		SHA1final(seed, &sha1ctx);
		break;
	case TLS_HASH_ALGO_SHA224:
		SHA224init(&sha224ctx);
		SHA224update(&sha224ctx, tls->client_random, crandlen);
		SHA224update(&sha224ctx, tls->server_random, srandlen);
		SHA224update(&sha224ctx, tls->skeyexc_params,
					 tls->skeyexc_params_len);
		SHA224final(seed, &sha224ctx);
		break;
	case TLS_HASH_ALGO_SHA256:
		SHA256init(&sha256ctx);
		SHA256update(&sha256ctx, tls->client_random, crandlen);
		SHA256update(&sha256ctx, tls->server_random, srandlen);
		SHA256update(&sha256ctx, tls->skeyexc_params,
					 tls->skeyexc_params_len);
		SHA256final(seed, &sha256ctx);
		break;
	case TLS_HASH_ALGO_SHA384:
		SHA384init(&sha384ctx);
		SHA384update(&sha384ctx, tls->client_random, crandlen);
		SHA384update(&sha384ctx, tls->server_random, srandlen);
		SHA384update(&sha384ctx, tls->skeyexc_params,
					 tls->skeyexc_params_len);
		SHA384final(seed, &sha384ctx);
		break;
	case TLS_HASH_ALGO_SHA512:
		SHA512init(&sha512ctx);
		SHA512update(&sha512ctx, tls->client_random, crandlen);
		SHA512update(&sha512ctx, tls->server_random, srandlen);
		SHA512update(&sha512ctx, tls->skeyexc_params,
					 tls->skeyexc_params_len);
		SHA512final(seed, &sha512ctx);
		break;
	default:
		assert(!"unknown hash algorithm");
		break;
	}
}
