/*
 * Copyright (c) 2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "extension/tls_sighash.h"

#include <string.h>

/* for MD5_CTX and MD5Final. */
#include <aicrypto/ok_md5.h>

/* for SHA256_CTX, SHA512_CTX, SHA224final, SHA256final, SHA384final and
 * SHA512final. */
#include <aicrypto/ok_sha2.h>

/**
 * make md5 digest.
 */
static inline void make_data_md5(const TLS *tls, uint8_t *seed);

/**
 * make sha1 digest.
 */
static inline void make_data_sha1(const TLS *tls, uint8_t *seed);

/**
 * make sha224 digest.
 */
static inline void make_data_sha224(const TLS *tls, uint8_t *seed);

/**
 * make sha256 digest.
 */
static inline void make_data_sha256(const TLS *tls, uint8_t *seed);

/**
 * make sha384 digest.
 */
static inline void make_data_sha384(const TLS *tls, uint8_t *seed);

/**
 * make sha512 digest.
 */
static inline void make_data_sha512(const TLS *tls, uint8_t *seed);

static inline void make_data_md5(const TLS *tls, uint8_t *seed) {
	MD5_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_md5_ctx), sizeof (ctx));
	MD5Final(seed, &ctx);
}

static inline void make_data_sha1(const TLS *tls, uint8_t *seed) {
	SHA1_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_sha1_ctx), sizeof (ctx));
	SHA1final(seed, &ctx);
}

static inline void make_data_sha224(const TLS *tls, uint8_t *seed) {
	SHA256_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_sha224_ctx), sizeof (ctx));
	SHA256final(seed, &ctx);
}

static inline void make_data_sha256(const TLS *tls, uint8_t *seed) {
	SHA256_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_sha256_ctx), sizeof (ctx));
	SHA256final(seed, &ctx);
}

static inline void make_data_sha384(const TLS *tls, uint8_t *seed) {
	SHA512_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_sha384_ctx), sizeof (ctx));
	SHA384final(seed, &ctx);
}

static inline void make_data_sha512(const TLS *tls, uint8_t *seed) {
	SHA512_CTX ctx;
	memcpy(&(ctx), &(tls->handshake_sha512_ctx), sizeof (ctx));
	SHA512final(seed, &ctx);
}

void tls_hs_hash_init(TLS *tls) {
	MD5Init   (&(tls->handshake_md5_ctx));
	SHA1init  (&(tls->handshake_sha1_ctx));
	SHA224init(&(tls->handshake_sha224_ctx));
	SHA256init(&(tls->handshake_sha256_ctx));
	SHA384init(&(tls->handshake_sha384_ctx));
	SHA512init(&(tls->handshake_sha512_ctx));
}

void tls_hs_hash_update(TLS *tls, uint8_t *buf, uint32_t len) {
	MD5Update    (&(tls->handshake_md5_ctx),    buf, len);
	SHA1update   (&(tls->handshake_sha1_ctx),   buf, len);
	SHA224update (&(tls->handshake_sha224_ctx), buf, len);
	SHA256update (&(tls->handshake_sha256_ctx), buf, len);
	SHA384update (&(tls->handshake_sha384_ctx), buf, len);
	SHA512update (&(tls->handshake_sha512_ctx), buf, len);
}

void tls_hs_hash_get_digest(const enum tls_hs_sighash_hash_algo hash,
			    const TLS *tls, uint8_t *seed) {
	switch(hash) {
#if 0
	case TLS_HASH_ALGO_NONE:
		/* XXX: should consider? */
		break;
#endif /* 0 */

	case TLS_HASH_ALGO_MD5:
		make_data_md5(tls, seed);
		break;

	case TLS_HASH_ALGO_SHA1:
		make_data_sha1(tls, seed);
		break;

	case TLS_HASH_ALGO_SHA224:
		make_data_sha224(tls, seed);
		break;

	case TLS_HASH_ALGO_SHA256:
		make_data_sha256(tls, seed);
		break;

	case TLS_HASH_ALGO_SHA384:
		make_data_sha384(tls, seed);
		break;

	case TLS_HASH_ALGO_SHA512:
		make_data_sha512(tls, seed);
		break;

	default:
		assert(!"unknown hash algorithm");
	}
}
