/*
 * Copyright (c) 2017-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/* template for unit testing with Check <https://libcheck.github.io/check/> */
#include <stdlib.h>
#include <check.h>

#include <aicrypto/nrg_tls.h>
#include <stdbool.h>

#include "shello.c"

#define SH_SERVER_VERSION_TLS11	SH_VERSION_TLS11
#define SH_SERVER_VERSION_TLS12	SH_VERSION_TLS12
#define SH_VERSION_TLS11	0x03, 0x02
#define SH_VERSION_TLS12	0x03, 0x03
#define SH_VERSION_TLS13	0x03, 0x04
#define SH_RANDOM_TLS13_AFTER_HRR	0xbb, 0x34, 0x1d, 0x84, 0x7f, 0xd7, 0x89, 0xc4, \
			0x7c, 0x38, 0x71, 0x72, 0xdc, 0x0c, 0x9b, 0xf1, \
			0x47, 0xfc, 0xca, 0xcb, 0x50, 0x43, 0xd8, 0x6c, \
			0xa4, 0xc5, 0x98, 0xd3, 0xff, 0x57, 0x1b, 0x98

#define SH_RANDOM_TLS13_HRR	0xcf, 0x21, 0xad, 0x74, 0xe5, 0x9a, 0x61, 0x11, \
			0xbe, 0x1d, 0x8c, 0x02, 0x1e, 0x65, 0xb8, 0x91, \
			0xc2, 0xa2, 0x11, 0x16, 0x7a, 0xbb, 0x8c, 0x5e, \
			0x07, 0x9e, 0x09, 0xe2, 0xc8, 0xa8, 0x33, 0x9c
#define SH_RANDOM_TLS12	0xc2, 0xd0, 0xa6, 0x87, 0x8b, 0x29, 0xf6, 0xb3, \
			0x9c, 0x6f, 0xf4, 0x19, 0x9b, 0x27, 0x72, 0xc1, \
			0xa8, 0xa8, 0xf6, 0xaf, 0xab, 0x95, 0xd5, 0xa7, \
			0x74, 0xee, 0x01, 0xce, 0xef, 0x0b, 0xdd, 0x67
#define SH_RANDOM_TLS13		0xa6, 0xaf, 0x06, 0xa4, 0x12, 0x18, 0x60, 0xdc, \
			0x5e, 0x6e, 0x60, 0x24, 0x9c, 0xd3, 0x4c, 0x95, \
			0x93, 0x0c, 0x8a, 0xc5, 0xcb, 0x14, 0x34, 0xda, \
			0xc1, 0x55, 0x77, 0x2e, 0xd3, 0xe2, 0x69, 0x28
#define SH_RANDOM_DOWNGRADE_TLS11	0xa6, 0xaf, 0x06, 0xa4, 0x12, 0x18, 0x60, 0xdc, \
			0x5e, 0x6e, 0x60, 0x24, 0x9c, 0xd3, 0x4c, 0x95, \
			0x93, 0x0c, 0x8a, 0xc5, 0xcb, 0x14, 0x34, 0xda, \
			0x44, 0x4f, 0x57, 0x4e, 0x47, 0x52, 0x44, 0x00
#define SH_RANDOM_DOWNGRADE_TLS12	0xa6, 0xaf, 0x06, 0xa4, 0x12, 0x18, 0x60, 0xdc, \
			0x5e, 0x6e, 0x60, 0x24, 0x9c, 0xd3, 0x4c, 0x95, \
			0x93, 0x0c, 0x8a, 0xc5, 0xcb, 0x14, 0x34, 0xda, \
			0x44, 0x4f, 0x57, 0x4e, 0x47, 0x52, 0x44, 0x01

#define SH_SESSIONID_LENGTH_32	0x20
#define SH_SESSIONID_LENGTH_0	0x00
#define SH_SESSIONID	0x9f, 0x59, 0xe7, 0x26, 0x03, 0x2e, 0xb3, 0xd9, \
			0xed, 0x80, 0x4f, 0x9c, 0xb8, 0x97, 0x68, 0x7e, \
			0xf9, 0x31, 0x3e, 0x10, 0xfa, 0x83, 0x0c, 0x8e, \
			0xdd, 0x3a, 0x9b, 0x61, 0xbf, 0x3f, 0x34, 0x50

/* Cipher Suite: TLS_RSA_WITH_AES_128_CBC_SHA */
#define SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA		0x00, 0x2f
#define SH_CS_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384	0xc0, 0x30
/* Cipher Suite: TLS_AES_128_GCM_SHA256 */
#define SH_CS_TLS_AES_128_GCM_SHA256		0x13, 0x01

/* Compression Method: null(0) */
#define	SH_CM_NULL				0x00

/* ExtensionType: elliptic_curves(0) */
#define SH_EXTTYPE_SERVER_NAME		0x00, 0x00
/* ExtensionType: elliptic_curves(10) */
#define SH_EXTTYPE_ELLIPTIC_CURVES	0x00, 0x0a
/* ExtensionType: ec_point_formats(11) */
#define SH_EXTTYPE_EC_POINT_FORMATS	0x00, 0x0b
/* ExtensionType: supported_versions(43) */
#define SH_EXTTYPE_SUPPORTED_VERSIONS	0x00, 0x2b
/* ExtensionType: cookie(44) */
#define SH_EXTTYPE_COOKIE		0x00, 0x2c
/* ExtensionType: key_share(51) */
#define SH_EXTTYPE_KEYSHARE		0x00, 0x33

/* elliptic curve: secp256r1 (23) */
#define SH_CURVE_SECP256R1		0x00, 0x17
/* elliptic curve: sect193r1 (4) */
#define SH_CURVE_SECT193R1		0x00, 0x04
/* elliptic curve: secp192r1 (19) */
#define SH_CURVE_SECP192R1		0x00, 0x13
/* elliptic curve: secp224r1 (21) */
#define SH_CURVE_SECP224R1		0x00, 0x15
/* elliptic curve: X25519 (0x1D) */
#define SH_CURVE_X25519			0x00, 0x1D

/* server x25519 public key */
#define SH_SERVER_X25519_PUB	0xc9, 0x82, 0x88, 0x76, 0x11, 0x20, 0x95, 0xfe, \
			0x66, 0x76, 0x2b, 0xdb, 0xf7, 0xc6, 0x72, 0xe1, \
			0x56, 0xd6, 0xcc, 0x25, 0x3b, 0x83, 0x3d, 0xf1, \
			0xdd, 0x69, 0xb1, 0xb0, 0x4e, 0x75, 0x1f, 0x0f

/* server SECP256R1 public key */
#define SH_SERVER_SECP256R1_PUB	0x04, 0x58, 0x3e, 0x05, 0x4b, 0x7a, 0x66, 0x67, \
			0x2a, 0xe0, 0x20, 0xad, 0x9d, 0x26, 0x86, 0xfc, \
			0xc8, 0x5b, 0x5a, 0xd4, 0x1a, 0x13, 0x4a, 0x0f, \
			0x03, 0xee, 0x72, 0xb8, 0x93, 0x05, 0x2b, 0xd8, \
			0x5b, 0x4c, 0x8d, 0xe6, 0x77, 0x6f, 0x5b, 0x04, \
			0xac, 0x07, 0xd8, 0x35, 0x40, 0xea, 0xb3, 0xe3, \
			0xd9, 0xc5, 0x47, 0xbc, 0x65, 0x28, 0xc4, 0x31, \
			0x7d, 0x29, 0x46, 0x86, 0x09, 0x3a, 0x6c, 0xad, \
			0x7d

#define SH_COOKIE_CONTENT	0x71, 0xdc, 0xd0, 0x4b, 0xb8, 0x8b, 0xc3, 0x18, \
			0x91, 0x19, 0x39, 0x8a, 0x00, 0x00, 0x00, 0x00, \
			0xee, 0xfa, 0xfc, 0x76, 0xc1, 0x46, 0xb8, 0x23, \
			0xb0, 0x96, 0xf8, 0xaa, 0xca, 0xd3, 0x65, 0xdd, \
			0x00, 0x30, 0x95, 0x3f, 0x4e, 0xdf, 0x62, 0x56, \
			0x36, 0xe5, 0xf2, 0x1b, 0xb2, 0xe2, 0x3f, 0xcc, \
			0x65, 0x4b, 0x1b, 0x5b, 0x40, 0x31, 0x8d, 0x10, \
			0xd1, 0x37, 0xab, 0xcb, 0xb8, 0x75, 0x74, 0xe3, \
			0x6e, 0x8a, 0x1f, 0x02, 0x5f, 0x7d, 0xfa, 0x5d, \
			0x6e, 0x50, 0x78, 0x1b, 0x5e, 0xda, 0x4a, 0xa1, \
			0x5b, 0x0c, 0x8b, 0xe7, 0x78, 0x25, 0x7d, 0x16, \
			0xaa, 0x30, 0x30, 0xe9, 0xe7, 0x84, 0x1d, 0xd9, \
			0xe4, 0xc0, 0x34, 0x22, 0x67, 0xe8, 0xca, 0x0c, \
			0xaf, 0x57, 0x1f, 0xb2, 0xb7, 0xcf, 0xf0, 0xf9, \
			0x34, 0xb0

// related to server
static bool tls_hs_msg_write_1_failure = false;
static unsigned int tls_hs_msg_write_1_through_count = 0;
static bool tls_hs_msg_write_2_failure = false;
static unsigned int tls_hs_msg_write_2_through_count = 0;
static bool tls_hs_msg_write_n_failure = false;
static bool tls_hs_msg_init_failure = false;
static bool tls_util_get_random_failure = false;
static unsigned int tls_util_get_random_through_count = 0;
static bool tls_hs_ecc_write_ec_point_formats_failure = false;
static bool tls_hs_ecc_write_ec_point_formats_0 = false;
static bool tls_hs_servername_write_failure = false;
static bool tls_hs_servername_write_0 = false;
static bool tls_hs_supported_versions_write_failure = false;
static bool tls_hs_supported_versions_write_0 = false;
static bool tls_hs_keyshare_write_failure = false;
static bool tls_hs_keyshare_write_0 = false;

// related to client
static bool tls_session_new_failure = false;
static bool tls_cipher_set_failure = false;
static bool tls_compress_set_failure = false;
static bool tls_hs_ecc_read_point_format_failure = false;
static bool tls_hs_servername_read_failure = false;
static bool tls_hs_keyshare_read_failure = false;

static TLS global_tls;

static uint8_t static_msg_body[1024];
static struct tls_hs_msg static_msg = {
	.len = 0,
	.max = TLS_RECORD_PLAIN_FRAGMENT_SIZE_MAX,
	.msg = static_msg_body,
};

static struct tls_session_param static_param = {
	.cipher_suite = TLS_NULL_WITH_NULL_NULL,
};

static uint8_t static_random_hrr[] = {
	SH_RANDOM_TLS13_HRR
};

static uint8_t static_session_id[33] = "abcdefghijABCDEFGHIJabcdefghijAB";

static uint8_t static_extensions_tls12_length[2] = { 0x00, 0x0a, };
static uint8_t static_extensions_tls12[10] = {
	SH_EXTTYPE_SERVER_NAME,
	0x00, 0x00, /* actual length (0 byte) */
	SH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02, /* actual length (2 byte) */
	0x01,  /* number of bytes (1) */
	0x00,  /* uncompressed(0) */
};

static uint8_t static_extensions_tls13_length[2] = { 0x00, 0x2e, };
static uint8_t static_extensions_tls13[48] = {
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02,
	SH_VERSION_TLS13,
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x24,
	SH_CURVE_X25519,
	0x00, 0x20,
	SH_SERVER_X25519_PUB,
};

static uint8_t static_extensions_tls13_hrr_length[2] = { 0x00, 0x0c, };
static uint8_t static_extensions_tls13_hrr[12] = {
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02,
	SH_VERSION_TLS13,
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x02,
	SH_CURVE_X25519
};

static uint8_t static_servername_extension[] = {
	SH_EXTTYPE_SERVER_NAME,
	0x00, 0x00,
};

static uint8_t static_ec_point_formats_extension[] = {
	SH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02,
	0x01,
	0x00,
};

static uint8_t static_supported_versions_extension[] = {
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02,
	SH_VERSION_TLS13,
};

static uint8_t static_keyshare_extension[] = {
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x24,
	SH_CURVE_X25519,
	0x00, 0x20,
	SH_SERVER_X25519_PUB,
};

static uint8_t static_keyshare_extension_hrr[] = {
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x02,
	SH_CURVE_X25519,
};

static uint8_t *static_server_hello_common;
static uint32_t static_server_hello_common_length;

static uint8_t static_server_hello_tls12[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS12,
	SH_SESSIONID_LENGTH_32,
	SH_SESSIONID,
	SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA, /* offset = 67 */
	SH_CM_NULL, /* offset = 69 */
	0x00, 0x0a, /* extensions length (10 byte) : offset=70 */
	SH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02, /* actual length (2 byte) : offset=74 */
	0x01,  /* number of bytes (1) : offset=76 */
	0x00,  /* uncompressed(0) */
	SH_EXTTYPE_SERVER_NAME,
	0x00, 0x00, /* actual length (0 byte) : offset=80 */
};

/* defined by RFC8448 3. Simple 1-RTT Handshake */
static uint8_t static_server_hello_tls13[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS13,
	SH_SESSIONID_LENGTH_0,
	SH_CS_TLS_AES_128_GCM_SHA256,
	SH_CM_NULL,
	0x00, 0x2e, /* extensions: 46 byte */
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x24, /* extension length 36 byte */
	SH_CURVE_X25519,
	0x00, 0x20, /* key_exchange length 32 byte */
	SH_SERVER_X25519_PUB,
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02, /* extension length 2 byte */
	SH_VERSION_TLS13, // supported_versions
};

static uint8_t static_server_hello_tls13_sesslen32[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS13,
	SH_SESSIONID_LENGTH_32,
	SH_SESSIONID,
	SH_CS_TLS_AES_128_GCM_SHA256,
	SH_CM_NULL,
	0x00, 0x2e, /* extensions: 46 byte */
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x24, /* extension length 36 byte */
	SH_CURVE_X25519,
	0x00, 0x20, /* key_exchange length 32 byte */
	SH_SERVER_X25519_PUB,
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02, /* extension length 2 byte */
	SH_VERSION_TLS13,
};

/* defined by RFC8448 5. HelloRetryRequest */
static uint8_t static_server_hello_tls13_hrr[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS13_HRR,
	SH_SESSIONID_LENGTH_0,
	SH_CS_TLS_AES_128_GCM_SHA256,
	SH_CM_NULL,
	0x00, 0x84, /* extensions: 132 byte */
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x02, /* extension length 2 byte */
	SH_CURVE_SECP256R1,
	SH_EXTTYPE_COOKIE,
	0x00, 0x74, /* extension length 116 byte */
	0x00, 0x72,
	SH_COOKIE_CONTENT,
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02, /* extension length 2 byte */
	SH_VERSION_TLS13,
};

/* defined by RFC8448 5. HelloRetryRequest */
static uint8_t static_server_hello_tls13_after_hrr[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS13_AFTER_HRR,
	SH_SESSIONID_LENGTH_0,
	SH_CS_TLS_AES_128_GCM_SHA256,
	SH_CM_NULL,
	0x00, 0x4f, /* extensions: 79 byte */
	SH_EXTTYPE_KEYSHARE,
	0x00, 0x45, /* extension length 69 byte */
	SH_CURVE_SECP256R1,
	0x00, 0x41, /* key_exchange length 65 byte */
	SH_SERVER_SECP256R1_PUB,
	SH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x02, /* extension length 2 byte */
	SH_VERSION_TLS13
};

static uint8_t static_server_hello_tls12_downgrade[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_DOWNGRADE_TLS12,
	SH_SESSIONID_LENGTH_32,
	SH_SESSIONID,
	SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	SH_CM_NULL,
};

static uint8_t static_server_hello_tls11_downgrade[] = {
	SH_SERVER_VERSION_TLS11,
	SH_RANDOM_DOWNGRADE_TLS11,
	SH_SESSIONID_LENGTH_32,
	SH_SESSIONID,
	SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	SH_CM_NULL,
};

static uint8_t static_server_hello_elliptic[] = {
	SH_SERVER_VERSION_TLS12,
	SH_RANDOM_TLS12,
	SH_SESSIONID_LENGTH_32,
	SH_SESSIONID,
	SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA, /* offset = 67 */
	SH_CM_NULL, /* offset = 69 */
	0x00, 0x0a, /* extensions length (10 byte) */
	SH_EXTTYPE_ELLIPTIC_CURVES, /* AiCrypto server unsupported */
	0x00, 0x06, /* actual length (6 byte) */
	SH_CURVE_SECT193R1,
	SH_CURVE_SECP192R1,
	SH_CURVE_SECP224R1,
};

static uint8_t static_downgrade_value_tls12[] = {
	0x44, 0x4f, 0x57, 0x4e, 0x47, 0x52, 0x44, 0x01
};

static uint8_t static_downgrade_value_tls11[] = {
	0x44, 0x4f, 0x57, 0x4e, 0x47, 0x52, 0x44, 0x00
};

static uint16_t static_supported_versions_list_tls13[] = {
	TLS_VER_TLS13,
	TLS_VER_TLS12,
};

static uint16_t static_supported_versions_list_tls12[] = {
	TLS_VER_TLS12,
};

static int ut_error;
static int ut_location;
static int ut_point;

static void ut_error_setup()
{
	ut_error = 0;
	ut_location = 0;
	ut_point = 0;
}

static enum tls_alert_desc ut_alert_desc;

static void ut_alert_setup()
{
	ut_alert_desc = 0;
}

/*
 * fixture
 */
void setup(void)
{
	global_tls.interim_params = tls_hs_interim_params_init();

	global_tls.resession = false;
	global_tls.pending = &static_param;

	static_server_hello_common = static_server_hello_tls12;
	static_server_hello_common_length = sizeof(static_server_hello_tls12);

	ut_error_setup();
	ut_alert_setup();
}

void setup_client(void)
{
	setup();
	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.client_version);
	global_tls.state = TLS_STATE_HS_BEFORE_RECV_SHELLO;

	tls_session_new_failure = false;
	tls_cipher_set_failure = false;
	tls_compress_set_failure = false;
	tls_hs_ecc_read_point_format_failure = false;
	tls_hs_servername_read_failure = false;
	tls_hs_keyshare_read_failure = false;
}

void setup_server(void)
{
	setup();
	static_msg.len = 0;
	memset(static_msg.msg, 0, 1024);
	global_tls.state = TLS_STATE_HS_BEFORE_RECV_CHELLO;

	tls_hs_msg_write_1_failure = false;
	tls_hs_msg_write_1_through_count = 0;
	tls_hs_msg_write_2_failure = false;
	tls_hs_msg_write_2_through_count = 0;
	tls_hs_msg_write_n_failure = false;
	tls_hs_msg_init_failure = false;
	tls_util_get_random_failure = false;
	tls_util_get_random_through_count = 0;
	tls_hs_ecc_write_ec_point_formats_failure = false;
	tls_hs_ecc_write_ec_point_formats_0 = false;
	tls_hs_servername_write_failure = false;
	tls_hs_servername_write_0 = false;
	tls_hs_supported_versions_write_failure = false;
	tls_hs_supported_versions_write_0 = false;
	tls_hs_keyshare_write_failure = false;
	tls_hs_keyshare_write_0 = false;
}

void teardown(void)
{
	if (global_tls.interim_params != NULL) {
		tls_hs_interim_params_free(global_tls.interim_params);
	}
}

/*
 * unit testing for write_version().
 */
/* normal case */
START_TEST (test_write_version_normal_tls12)
{
	int32_t version_length;

	/* tls.c: init_version */
	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	version_length = write_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, 2);
	ck_assert_mem_eq(&static_msg.msg[0],
			 &global_tls.negotiated_version.major, 1);
	ck_assert_mem_eq(&static_msg.msg[1],
			 &global_tls.negotiated_version.minor, 1);
}
END_TEST

START_TEST (test_write_version_normal_tls13)
{
	int32_t version_length;
	uint16_t msg_version;

	/* tls.c: init_version */
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	version_length = write_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, 2);
	msg_version = tls_util_read_2(&(static_msg.msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, tls->negotiated_version.major)) {... */
START_TEST (test_write_version_failure_tls12_msg_write_major)
{
	int32_t version_length;

	/* tls.c: init_version */
	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;
	version_length = write_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, -1);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, tls->negotiated_version.minor)) {... */
START_TEST (test_write_version_failure_tls12_msg_write_minor)
{
	int32_t version_length;

	/* tls.c: init_version */
	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 1;
	version_length = write_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, -1);
}
END_TEST

/* if (! tls_hs_msg_write_2(msg, TLS_VER_TLS12)) {... */
START_TEST (test_write_version_failure_tls13_msg_write_2)
{
	int32_t version_length;

	/* tls.c: init_version */
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	tls_hs_msg_write_2_failure = true;
	version_length = write_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, -1);
}
END_TEST

Suite *write_version_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_version()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_version_normal_tls12);
	tcase_add_test(tc_core, test_write_version_normal_tls13);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_version_failure_tls12_msg_write_major);
	tcase_add_test(tc_limits, test_write_version_failure_tls12_msg_write_minor);
	tcase_add_test(tc_limits, test_write_version_failure_tls13_msg_write_2);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_server_random().
 */
/* normal case */
START_TEST (test_write_server_random_normal_tls11)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS11,
					 &global_tls.negotiated_version);

	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&global_tls.server_random,
			 static_session_id, 24);
	ck_assert_mem_eq(&global_tls.server_random[24],
			 static_downgrade_value_tls11, 8);
}
END_TEST

START_TEST (test_write_server_random_normal_tls12)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&global_tls.server_random,
			 static_session_id, 24);
	ck_assert_mem_eq(&global_tls.server_random[24],
			 static_downgrade_value_tls12, 8);
}
END_TEST

START_TEST (test_write_server_random_normal_tls13)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&global_tls.server_random,
			 static_session_id, len);
}
END_TEST

START_TEST (test_write_server_random_normal_tls13_hrr)
{
	int32_t len;

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_HRREQ;
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&global_tls.server_random,
			static_random_hrr, len);
}
END_TEST

/* if (! tls_util_get_random(&(tls->server_random[0]), len)) {... */
START_TEST (test_write_server_random_failure_get_random)
{
	int32_t len;

	tls_util_get_random_failure = true;
	tls_util_get_random_through_count = 0;
	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (! tls_hs_msg_write_n(msg, tls->server_random, len)) {... */
START_TEST (test_write_server_random_failure_msg_write)
{
	int32_t len;

	tls_hs_msg_write_n_failure = true;
	len = write_server_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_server_random_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_server_random()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_server_random_normal_tls11);
	tcase_add_test(tc_core, test_write_server_random_normal_tls12);
	tcase_add_test(tc_core, test_write_server_random_normal_tls13);
	tcase_add_test(tc_core, test_write_server_random_normal_tls13_hrr);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_server_random_failure_get_random);
	tcase_add_test(tc_limits, test_write_server_random_failure_msg_write);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_session_id().
 */
/* normal case */
START_TEST (test_write_session_id_normal_tls12)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, global_tls.pending->session_id_length + 1);
	ck_assert_mem_eq(global_tls.pending->session_id,
			 static_session_id,
			 global_tls.pending->session_id_length);
	ck_assert_mem_eq(&static_msg.msg[0],
			 &(global_tls.pending->session_id_length), 1);
	ck_assert_mem_eq(&static_msg.msg[1],
			 global_tls.pending->session_id,
			 global_tls.pending->session_id_length);
}
END_TEST

START_TEST (test_write_session_id_normal_tls13)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 1);
	ck_assert_int_eq(static_msg.msg[0], global_tls.interim_params->seslen);
}
END_TEST

/* if (tls->resession == false) {... */
START_TEST (test_write_session_id_normal_tls12_ressesion_32)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	global_tls.pending->session_id_length = 32;
	memcpy(global_tls.pending->session_id, static_session_id, 32);
	global_tls.resession = true;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32+1);
	ck_assert_mem_eq(&static_msg.msg[0],
			 &(global_tls.pending->session_id_length), 1);
	ck_assert_mem_eq(&static_msg.msg[1],
			 global_tls.pending->session_id,
			 global_tls.pending->session_id_length);
}
END_TEST

/* if (tls->resession == false) {... */
START_TEST (test_write_session_id_normal_tls12_ressesion_0)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	global_tls.pending->session_id_length = 0;
	memset(global_tls.pending->session_id, 0, 32);
	global_tls.resession = true;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 1);
	ck_assert_mem_eq(&static_msg.msg[0],
			 &(global_tls.pending->session_id_length), 1);
}
END_TEST

START_TEST (test_write_session_id_normal_tls13_compat)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.interim_params->seslen = 32;
	global_tls.interim_params->session = static_session_id;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 33);
	ck_assert_int_eq(global_tls.interim_params->seslen, 32);
	ck_assert_int_eq(static_msg.msg[0], 32);
	ck_assert_ptr_eq(global_tls.interim_params->session, static_session_id);
	ck_assert_mem_eq(&(static_msg.msg[1]), static_session_id, 32);

	global_tls.interim_params->session = NULL;
}
END_TEST

/* if (! tls_util_get_random(&(session[0]), seslen)) {...*/
START_TEST (test_write_session_id_failure_tls12_get_random)
{
	int32_t len;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_util_get_random_failure = true;
	tls_util_get_random_through_count = 0;
	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_RANDOM);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 0);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, tls->pending->session_id_length)) {... */
START_TEST (test_write_session_id_failure_write_1)
{
	int32_t len;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;
	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if (! tls_hs_msg_write_n(msg,... */
START_TEST (test_write_session_id_failure_write_n)
{
	int32_t len;

	tls_hs_msg_write_n_failure = true;
	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

Suite *write_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_session_id()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_session_id_normal_tls12);
	tcase_add_test(tc_core, test_write_session_id_normal_tls13);
	tcase_add_test(tc_core, test_write_session_id_normal_tls12_ressesion_32);
	tcase_add_test(tc_core, test_write_session_id_normal_tls12_ressesion_0);
	tcase_add_test(tc_core, test_write_session_id_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_session_id_failure_tls12_get_random);
	tcase_add_test(tc_limits, test_write_session_id_failure_write_1);
	tcase_add_test(tc_limits, test_write_session_id_failure_write_n);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_cipher_suite().
 */
/* normal case */
START_TEST (test_write_cipher_suite_normal)
{
	int32_t len;
	uint8_t cipher_suite[] = { 0x00, 0x3c, };

	global_tls.pending->cipher_suite = TLS_RSA_WITH_AES_128_CBC_SHA256;

	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, 2);
	ck_assert_mem_eq(&static_msg.msg[0], cipher_suite, 2);
}
END_TEST

/* if (! tls_hs_msg_write_2(msg, tls->pending->cipher_suite)) {... */
START_TEST (test_write_cipher_suite_failure_write_2)
{
	int32_t len;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;
	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_cipher_suite_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_cipher_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_cipher_suite_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_cipher_suite_failure_write_2);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_cmp_method().
 */
/* normal case */
START_TEST (test_write_cmp_method_normal)
{
	int32_t len;
	uint8_t compression_algorithm[] = { 0x00 };

	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;

	len = write_cmp_method(&global_tls, &static_msg);

	ck_assert_int_eq(len, 1);
	ck_assert_mem_eq(&static_msg.msg[0], compression_algorithm, 1);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, tls->pending->compression_algorithm)) {... */
START_TEST (test_write_cmp_method_failure_write_1)
{
	int32_t len;

	global_tls.pending = &static_param;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;
	len = write_cmp_method(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_cmp_method()");

	/* Core_cmp_method() test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_cmp_method_failure_write_1);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_extension_shello().
 */
/* normal case */
START_TEST (test_write_extension_shello_normal_tls12)
{
	int32_t len;
	int32_t msg_len = static_msg.len;

	tls_hs_supported_versions_write_0 = true;
	tls_hs_keyshare_write_0 = true;

	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, 12);
	ck_assert_mem_eq(&static_msg.msg[0], static_extensions_tls12_length, 2);
	ck_assert_mem_eq(&static_msg.msg[2], static_extensions_tls12, 10);
	ck_assert_int_eq(static_msg.len, msg_len + 12);
}
END_TEST

START_TEST (test_write_extension_shello_normal_tls13)
{
	int32_t len;
	int32_t msg_len = static_msg.len;

	tls_hs_ecc_write_ec_point_formats_0 = true;
	tls_hs_servername_write_0 = true;

	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, 48);
	ck_assert_mem_eq(&static_msg.msg[0], static_extensions_tls13_length, 2);
	ck_assert_mem_eq(&static_msg.msg[2], static_extensions_tls13, 46);
	ck_assert_int_eq(static_msg.len, msg_len + 48);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, 0) == false) {... */
START_TEST (test_write_extension_shello_failure_write_2)
{
	int32_t len;
	int32_t msg_len = static_msg.len;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;
	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, msg_len);
}
END_TEST

/* if (server_name_len < 0) {... */
START_TEST (test_write_extension_shello_failure_tls12_servername_write)
{
	int32_t len;

	tls_hs_servername_write_failure = true;
	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (ec_point_formats_len < 0) {... */
START_TEST (test_write_extension_shello_failure_tls12_write_ec_point_formats)
{
	int32_t len;

	tls_hs_ecc_write_ec_point_formats_failure = true;
	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (supported_versions_len < 0) {...  */
START_TEST (test_write_extension_shello_failure_tls13_supported_versions_write)
{
	int32_t len;

	tls_hs_supported_versions_write_failure = true;
	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (key_share_len < 0) {... */
START_TEST (test_write_extension_shello_failure_tls13_keyshare_write)
{
	int32_t len;

	tls_hs_keyshare_write_failure = true;
	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (off == length_bytes) {... */
START_TEST (test_write_extension_shello_normal_write_length_0)
{
	int32_t len;
	int32_t msg_len = static_msg.len;

	tls_hs_ecc_write_ec_point_formats_0 = true;
	tls_hs_servername_write_0 = true;
	tls_hs_supported_versions_write_0 = true;
	tls_hs_keyshare_write_0 = true;

	len = write_extension_shello(&global_tls, &static_msg);

	ck_assert_int_eq(len, 0);
	ck_assert_int_eq(static_msg.len, msg_len);
}
END_TEST

Suite *write_extension_shello_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_extension_shello()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_extension_shello_normal_tls12);
	tcase_add_test(tc_core, test_write_extension_shello_normal_tls13);
	tcase_add_test(tc_core, test_write_extension_shello_normal_write_length_0);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_extension_shello_failure_write_2);
	tcase_add_test(tc_limits, test_write_extension_shello_failure_tls12_servername_write);
	tcase_add_test(tc_limits, test_write_extension_shello_failure_tls12_write_ec_point_formats);
	tcase_add_test(tc_limits, test_write_extension_shello_failure_tls13_supported_versions_write);
	tcase_add_test(tc_limits, test_write_extension_shello_failure_tls13_keyshare_write);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_extension_hrr().
 */
/* normal case */
START_TEST (test_write_extension_hrr_normal)
{
	int32_t len;

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_HRREQ;
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	len = write_extension_hrr(&global_tls, &static_msg);

	ck_assert_int_eq(len, 14);
	ck_assert_mem_eq(&static_msg.msg[0], static_extensions_tls13_hrr_length, 2);
	ck_assert_mem_eq(&static_msg.msg[2], static_extensions_tls13_hrr, 12);
	ck_assert_int_eq(static_msg.len, 14);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, 0) == false) {... */
START_TEST (test_write_extension_hrr_failure_write_2)
{
	int32_t len;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;
	len = write_extension_hrr(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 0);
}
END_TEST

/* if (supported_versions_len < 0) {...  */
START_TEST (test_write_extension_hrr_failure_supported_versions_write)
{
	int32_t len;

	tls_hs_supported_versions_write_failure = true;
	len = write_extension_hrr(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 2);
}
END_TEST

/* if (key_share_len < 0) {... */
START_TEST (test_write_extension_hrr_failure_keyshare_write)
{
	int32_t len;

	tls_hs_keyshare_write_failure = true;
	len = write_extension_hrr(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 8);
}
END_TEST

Suite *write_extension_hrr_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_extension_hrr()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_write_extension_hrr_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_write_extension_hrr_failure_write_2);
	tcase_add_test(tc_limits, test_write_extension_hrr_failure_supported_versions_write);
	tcase_add_test(tc_limits, test_write_extension_hrr_failure_keyshare_write);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_shello_compose().
 */
/* normal case */
START_TEST (test_tls_hs_shello_compose_normal_tls12)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint8_t cipher[2] = {
		SH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	};

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	global_tls.pending->cipher_suite = TLS_RSA_WITH_AES_128_CBC_SHA;
	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;
	tls_hs_supported_versions_write_0 = true;
	tls_hs_keyshare_write_0 = true;

	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_SERVER_HELLO);
	ck_assert_int_eq(msg->len, 82);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&msg->msg[2], static_session_id, 24);
	ck_assert_mem_eq(&msg->msg[26], static_downgrade_value_tls12, 8);
	ck_assert_uint_eq(msg->msg[34], 32);
	ck_assert_mem_eq(&msg->msg[35], static_session_id, 32);
	ck_assert_mem_eq(&msg->msg[67], cipher, 2);
	ck_assert_uint_eq(msg->msg[69], 0);
	ck_assert_mem_eq(&msg->msg[70], static_extensions_tls12_length, 2);
	ck_assert_mem_eq(&msg->msg[72], static_extensions_tls12, 10);
}
END_TEST

START_TEST (test_tls_hs_shello_compose_normal_tls13)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint8_t cipher[2] = {
		SH_CS_TLS_AES_128_GCM_SHA256,
	};

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;
	global_tls.pending->cipher_suite = TLS_AES_128_GCM_SHA256;
	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;
	tls_hs_ecc_write_ec_point_formats_0 = true;
	tls_hs_servername_write_0 = true;

	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_SERVER_HELLO);
	ck_assert_int_eq(msg->len, 86);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&msg->msg[2], static_session_id, 32);
	ck_assert_uint_eq(msg->msg[34], SH_SESSIONID_LENGTH_0);
	ck_assert_mem_eq(&msg->msg[35], cipher, 2);
	ck_assert_uint_eq(msg->msg[37], SH_CM_NULL);
	ck_assert_mem_eq(&msg->msg[38], static_extensions_tls13_length, 2);
	ck_assert_mem_eq(&msg->msg[40], static_extensions_tls13, 46);
}
END_TEST

START_TEST (test_tls_hs_shello_compose_normal_tls13_hrr)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint8_t cipher[2] = {
		SH_CS_TLS_AES_128_GCM_SHA256,
	};

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.state = TLS_STATE_HS_BEFORE_SEND_HRREQ;
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;
	global_tls.pending->cipher_suite = TLS_AES_128_GCM_SHA256;
	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;

	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_SERVER_HELLO);
	ck_assert_int_eq(msg->len, 52);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&msg->msg[2], static_random_hrr, 32);
	ck_assert_uint_eq(msg->msg[34], SH_SESSIONID_LENGTH_0);
	ck_assert_mem_eq(&msg->msg[35], cipher, 2);
	ck_assert_uint_eq(msg->msg[37], SH_CM_NULL);
	ck_assert_mem_eq(&msg->msg[38], static_extensions_tls13_hrr_length, 2);
	ck_assert_mem_eq(&msg->msg[40], static_extensions_tls13_hrr, 12);
}
END_TEST

START_TEST (test_tls_hs_shello_compose_normal_tls13_after_hrr)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint8_t cipher[2] = {
		SH_CS_TLS_AES_128_GCM_SHA256,
	};

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.state = TLS_STATE_HS_AFTER_RECV_2NDCHELLO;
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;
	global_tls.pending->cipher_suite = TLS_AES_128_GCM_SHA256;
	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;
	tls_hs_ecc_write_ec_point_formats_0 = true;
	tls_hs_servername_write_0 = true;

	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_SERVER_HELLO);
	ck_assert_int_eq(msg->len, 86);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&msg->msg[2], static_session_id, 32);
	ck_assert_uint_eq(msg->msg[34], SH_SESSIONID_LENGTH_0);
	ck_assert_mem_eq(&msg->msg[35], cipher, 2);
	ck_assert_uint_eq(msg->msg[37], SH_CM_NULL);
	ck_assert_mem_eq(&msg->msg[38], static_extensions_tls13_length, 2);
	ck_assert_mem_eq(&msg->msg[40], static_extensions_tls13, 46);
}
END_TEST

START_TEST (test_tls_hs_shello_compose_normal_tls13_compat)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint8_t cipher[2] = {
		SH_CS_TLS_AES_128_GCM_SHA256,
	};

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.interim_params->seslen = 32;
	global_tls.interim_params->session = static_session_id;
	global_tls.pending->cipher_suite = TLS_AES_128_GCM_SHA256;
	global_tls.pending->compression_algorithm = TLS_COMPRESSION_NULL;
	tls_hs_ecc_write_ec_point_formats_0 = true;
	tls_hs_servername_write_0 = true;

	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_SERVER_HELLO);
	ck_assert_int_eq(msg->len, 118);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&msg->msg[2], static_session_id, 32);
	ck_assert_uint_eq(msg->msg[34], SH_SESSIONID_LENGTH_32);
	ck_assert_mem_eq(&msg->msg[35], static_session_id, 32);
	ck_assert_mem_eq(&msg->msg[67], cipher, 2);
	ck_assert_uint_eq(msg->msg[69], SH_CM_NULL);
	ck_assert_mem_eq(&msg->msg[70], static_extensions_tls13_length, 2);
	ck_assert_mem_eq(&msg->msg[72], static_extensions_tls13, 46);

	global_tls.interim_params->session = NULL;
}
END_TEST

/* if ((msg = tls_hs_msg_init()) == NULL) {... */
START_TEST (test_tls_hs_shello_compose_failure_msg_init)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_init_failure = true;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if ((verlen = write_version(tls, msg)) < 0) { */
START_TEST (test_tls_hs_shello_compose_failure_write_version)
{
	struct tls_hs_msg *msg;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if ((randlen = write_server_random(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_shello_compose_failure_write_server_random)
{
	struct tls_hs_msg *msg;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_util_get_random_failure = true;
	tls_util_get_random_through_count = 0;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if ((seslen = write_session_id(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_shello_compose_failure_write_session_id)
{
	struct tls_hs_msg *msg;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_util_get_random_failure = true;
	tls_util_get_random_through_count = 1;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_RANDOM);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 0);
}
END_TEST

/* if ((cslen = write_cipher_suite(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_shello_compose_failure_write_cipher_suite)
{
	struct tls_hs_msg *msg;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if ((cmplen = write_cmp_method(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_shello_compose_failure_write_cmp_method)
{
	struct tls_hs_msg *msg = NULL;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 3;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

/* if ((extlen = write_extension(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_shello_compose_failure_write_extension)
{
	struct tls_hs_msg *msg = NULL;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;
	msg = tls_hs_shello_compose(&global_tls);

	ck_assert_ptr_null(msg);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
}
END_TEST

Suite *tls_hs_shello_compose_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_shello_compose()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown);
	tcase_add_test(tc_core, test_tls_hs_shello_compose_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_shello_compose_normal_tls13);
	tcase_add_test(tc_core, test_tls_hs_shello_compose_normal_tls13_hrr);
	tcase_add_test(tc_core, test_tls_hs_shello_compose_normal_tls13_after_hrr);
	tcase_add_test(tc_core, test_tls_hs_shello_compose_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_msg_init);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_version);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_server_random);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_session_id);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_cipher_suite);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_cmp_method);
	tcase_add_test(tc_limits, test_tls_hs_shello_compose_failure_write_extension);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_version().
 */
/* normal case */
START_TEST (test_read_version_normal_tls12)
{
	int32_t read_bytes;
	const uint32_t offset = 0;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	read_bytes = read_version(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 2);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(&(global_tls.interim_params->version.major),
			 &static_msg.msg[0], 1);
	ck_assert_mem_eq(&(global_tls.interim_params->version.minor),
			 &static_msg.msg[1], 1);
}
END_TEST

START_TEST (test_read_version_normal_tls13)
{
	int32_t read_bytes;
	const uint32_t offset = 0;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	read_bytes = read_version(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 2);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(&(global_tls.interim_params->version.major),
			 &static_msg.msg[0], 1);
	ck_assert_mem_eq(&(global_tls.interim_params->version.minor),
			 &static_msg.msg[1], 1);
}
END_TEST

/* if (msg->len < (offset + verlen)) {... */
START_TEST (test_read_version_failure_invalid_record_length)
{
	int32_t read_bytes;
	const uint32_t offset = 0;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 1;

	read_bytes = read_version(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 1);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_version_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_version()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_version_normal_tls12);
	tcase_add_test(tc_core, test_read_version_normal_tls13);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_version_failure_invalid_record_length);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_server_random().
 */
/* normal case */
START_TEST (test_read_server_random_normal_tls12)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 32);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(&global_tls.server_random[0], &static_msg.msg[2], 32);
}
END_TEST

START_TEST (test_read_server_random_normal_tls13)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 32);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(&global_tls.server_random[0], &static_msg.msg[2], 32);
}
END_TEST

START_TEST (test_read_server_random_normal_tls13_hrr)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 32);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(&global_tls.server_random[0], &static_msg.msg[2], 32);
	ck_assert_mem_eq(&global_tls.server_random[0], static_random_hrr, 32);
}
END_TEST

/* if (msg->len < (offset + randlen)) {... */
START_TEST (test_read_server_random_failure_invalid_record_length)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 33;

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 3);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (memcmp(protrand, buf, protrand_len) == 0) {... */
START_TEST (test_read_server_random_failure_downgrade_tls12)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12_downgrade;
	static_msg.len = sizeof(static_server_hello_tls12_downgrade);

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list_tls13;

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_PROTOCOL_VERSION_DOWNGRADE);
	ck_assert_int_eq(ut_location, ERR_LC_TLS6);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO2 + 0);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
}
END_TEST

/* if (memcmp(protrand, buf, protrand_len) == 0) {... */
START_TEST (test_read_server_random_failure_downgrade_tls11)
{
	int32_t read_bytes;
	const uint32_t offset = 2;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls11_downgrade;
	static_msg.len = sizeof(static_server_hello_tls11_downgrade);

	global_tls.supported_versions.len = 1;
	global_tls.supported_versions.list = static_supported_versions_list_tls12;

	read_bytes = read_server_random(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_PROTOCOL_VERSION_DOWNGRADE);
	ck_assert_int_eq(ut_location, ERR_LC_TLS6);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO2 + 1);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
}
END_TEST

Suite *read_server_random_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_server_random()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_server_random_normal_tls12);
	tcase_add_test(tc_core, test_read_server_random_normal_tls13);
	tcase_add_test(tc_core, test_read_server_random_normal_tls13_hrr);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_server_random_failure_invalid_record_length);
	tcase_add_test(tc_limits, test_read_server_random_failure_downgrade_tls12);
	tcase_add_test(tc_limits, test_read_server_random_failure_downgrade_tls11);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_session_id().
 */
/* normal case */
START_TEST (test_read_session_id_normal_tls12)
{
	int32_t read_bytes;
	const uint32_t offset = 34;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	global_tls.pending->session_id_length = 0;

	read_bytes = read_session_id(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 33);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(global_tls.interim_params->session,
			 &static_msg.msg[offset+1], 32);
	ck_assert_int_eq(global_tls.interim_params->seslen, 32);
}
END_TEST

START_TEST (test_read_session_id_normal_tls13_len0)
{
	int32_t read_bytes;
	const uint32_t offset = 34;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	global_tls.pending->session_id_length = 0;

	read_bytes = read_session_id(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_ptr_null(global_tls.interim_params->session);
	ck_assert_int_eq(global_tls.interim_params->seslen, 0);
}
END_TEST

START_TEST (test_read_session_id_normal_tls13_len32)
{
	int32_t read_bytes;
	const uint32_t offset = 34;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_sesslen32;
	static_msg.len = sizeof(static_server_hello_tls13_sesslen32);

	global_tls.pending->session_id_length = 32;

	read_bytes = read_session_id(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 33);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_mem_eq(global_tls.interim_params->session,
			 &static_msg.msg[offset+1], 32);
	ck_assert_int_eq(global_tls.interim_params->seslen, 32);
}
END_TEST

/* if (msg->len < (offset + read_bytes + length_bytes)) {... */
START_TEST (test_read_session_id_failure_invalid_record_length_1)
{
	int32_t read_bytes;
	const uint32_t offset = 34;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 34;

	read_bytes = read_session_id(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 4);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (msg->len < (offset + read_bytes + len)) {... */
START_TEST (test_read_session_id_failure_invalid_record_length_2)
{
	int32_t read_bytes;
	const uint32_t offset = 34;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 66;

	read_bytes = read_session_id(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 5);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_session_id()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_session_id_normal_tls12);
	tcase_add_test(tc_core, test_read_session_id_normal_tls13_len0);
	tcase_add_test(tc_core, test_read_session_id_normal_tls13_len32);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_session_id_failure_invalid_record_length_1);
	tcase_add_test(tc_limits, test_read_session_id_failure_invalid_record_length_2);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_cipher_suite().
 */
/* normal case */
START_TEST (test_read_cipher_suite_normal)
{
	int32_t read_bytes;
	const uint32_t offset = 67;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	read_bytes = read_cipher_suite(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 2);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(global_tls.interim_params->cipher_suite,
			 tls_util_read_2(&(static_server_hello_tls12[offset])));
}
END_TEST

/* if (msg->len < (offset + cslen)) {... */
START_TEST (test_read_cipher_suite_failure_invalid_record_length)
{
	int32_t read_bytes;
	const uint32_t offset = 67;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 68;

	read_bytes = read_cipher_suite(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 7);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_cipher_suite_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_cipher_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_cipher_suite_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_cipher_suite_failure_invalid_record_length);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_cmp_method().
 */
/* normal case */
START_TEST (test_read_cmp_method_normal)
{
	int32_t read_bytes;
	const uint32_t offset = 69;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	read_bytes = read_cmp_method(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(global_tls.interim_params->cmplen, 1);
	ck_assert_mem_eq(global_tls.interim_params->cmplist,
			 &(static_msg.msg[offset]), 1);
}
END_TEST

/* if (msg->len < (offset + cmplen)) {... */
START_TEST (test_read_cmp_method_failure_invalid_record_length)
{
	int32_t read_bytes;
	const uint32_t offset = 69;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 69;

	read_bytes = read_cmp_method(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 8);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_cmp_method()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_cmp_method_failure_invalid_record_length);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for check_ext_availability_tls12().
 */
/* normal case */
START_TEST (test_check_ext_availability_tls12_normal)
{
	bool rc;

	rc = check_ext_availability_tls12(TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls12(TLS_EXT_EC_POINT_FORMATS);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_check_ext_availability_tls12_failure)
{
	bool rc;

	rc = check_ext_availability_tls12(TLS_EXT_ELLIPTIC_CURVES);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_SIGNATURE_ALGO);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *check_ext_availability_tls12_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("check_ext_availability_tls12()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_test(tc_core, test_check_ext_availability_tls12_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_test(tc_limits, test_check_ext_availability_tls12_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for check_ext_availability_tls13().
 */
/* normal case */
START_TEST (test_check_ext_availability_tls13_shello_normal)
{
	bool rc;

	rc = check_ext_availability_tls13_shello(TLS_EXT_PRE_SHARED_KEY);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13_shello(TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13_shello(TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13_shello(TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(rc, false);
}
END_TEST

START_TEST (test_check_ext_availability_tls13_shello_failure)
{
	bool rc;

	rc = check_ext_availability_tls13_shello(TLS_EXT_SUPPORTED_GROUPS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_shello(TLS_EXT_SIGNATURE_ALGO);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_shello(TLS_EXT_COOKIE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_shello(TLS_EXT_SIGNATURE_ALGO_CERT);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *check_ext_availability_tls13_shello_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("check_ext_availability_tls13_shello()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_test(tc_core, test_check_ext_availability_tls13_shello_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_test(tc_limits, test_check_ext_availability_tls13_shello_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for check_ext_availability_tls13_hrr().
 */
/* normal case */
START_TEST (test_check_ext_availability_tls13_hrr_normal)
{
	bool rc;

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_COOKIE);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_check_ext_availability_tls13_hrr_failure)
{
	bool rc;

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_MAX_FRAGMENT_LENGTH);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_STATUS_REQUEST);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SUPPORTED_GROUPS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SIGNATURE_ALGO);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_USE_SRTP);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_HEARTBEAT);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_APP_LAYER_PROTO_NEGOTIATION);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_CLIENT_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SERVER_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_PADDING);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_PRE_SHARED_KEY);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_EARLY_DATA);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_PSK_KEY_EXCHANGE_MODES);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_CERTIFICATE_AUTHORITIES);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_OID_FILTERS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_POST_HANDSHAKE_AUTH);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13_hrr(TLS_EXT_SIGNATURE_ALGO_CERT);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *check_ext_availability_tls13_hrr_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("check_ext_availability_tls13_hrr()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_test(tc_core, test_check_ext_availability_tls13_hrr_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_test(tc_limits, test_check_ext_availability_tls13_hrr_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_ext_list().
 */
/* normal case */
START_TEST (test_read_ext_list_normal_server_name)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SERVER_NAME;
	const uint32_t offset = 80;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

START_TEST (test_read_ext_list_normal_ec_point_formats)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_EC_POINT_FORMATS;
	const uint32_t offset = 76;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

START_TEST (test_read_ext_list_normal_supported_versions)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SUPPORTED_VERSIONS;
	const uint32_t offset = 72;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

START_TEST (test_read_ext_list_normal_key_share)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_KEY_SHARE;
	const uint32_t offset = 44;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

/* if (recv_exts[type] == true) {... */
START_TEST (test_read_ext_list_failure_multiple_server_name)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SERVER_NAME;
	const uint32_t offset = 80;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (recv_exts[type] == true) {... */
START_TEST (test_read_ext_list_failure_multiple_ec_point_formats)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_EC_POINT_FORMATS;
	const uint32_t offset = 76;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (recv_exts[type] == true) {... */
START_TEST (test_read_ext_list_failure_multiple_supported_versions)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SUPPORTED_VERSIONS;
	const uint32_t offset = 72;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (recv_exts[type] == true) {... */
START_TEST (test_read_ext_list_failure_multiple_key_share)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_KEY_SHARE;
	const uint32_t offset = 44;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (tls_hs_servername_read(tls, msg, offset) < 0) {... */
START_TEST (test_read_ext_list_failure_servername_read)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SERVER_NAME;
	const uint32_t offset = 80;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	tls_hs_servername_read_failure = true;
	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

/* if (tls_hs_ecc_read_point_format(tls, msg, offset) < 0) {... */
START_TEST (test_read_ext_list_failure_ecc_read_point_format)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_EC_POINT_FORMATS;
	const uint32_t offset = 76;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	tls_hs_ecc_read_point_format_failure = true;
	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

/* if (tls_hs_keyshare_read(tls, msg, offset) < 0) {... */
START_TEST (test_read_ext_list_failure_keyshare_read)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_KEY_SHARE;
	const uint32_t offset = 44;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	tls_hs_keyshare_read_failure = true;
	rc = read_ext_list(&global_tls, type, &static_msg, offset);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.interim_params->recv_ext_flags[type], true);
}
END_TEST

START_TEST (test_read_ext_list_failure_elliptic_curves)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_ELLIPTIC_CURVES;
	const uint32_t offset = 76;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);
	/* Not reach the following steps because the above function aborts. */

	ck_assert_int_eq(rc, false);
}
END_TEST

START_TEST (test_read_ext_list_failure_signature_algo)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SIGNATURE_ALGO;
	const uint32_t offset = 76;
	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = read_ext_list(&global_tls, type, &static_msg, offset);
	/* Not reach the following steps because the above function aborts. */

	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *read_ext_list_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_ext_list()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_read_ext_list_normal_server_name);
	tcase_add_test(tc_core, test_read_ext_list_normal_ec_point_formats);
	tcase_add_test(tc_core, test_read_ext_list_normal_supported_versions);
	tcase_add_test(tc_core, test_read_ext_list_normal_key_share);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_read_ext_list_failure_multiple_server_name);
	tcase_add_test(tc_limits, test_read_ext_list_failure_multiple_ec_point_formats);
	tcase_add_test(tc_limits, test_read_ext_list_failure_multiple_supported_versions);
	tcase_add_test(tc_limits, test_read_ext_list_failure_multiple_key_share);
	tcase_add_test(tc_limits, test_read_ext_list_failure_servername_read);
	tcase_add_test(tc_limits, test_read_ext_list_failure_ecc_read_point_format);
	tcase_add_test(tc_limits, test_read_ext_list_failure_keyshare_read);
	tcase_add_test_raise_signal(tc_limits,
				    test_read_ext_list_failure_elliptic_curves, 6);
	tcase_add_test_raise_signal(tc_limits,
				    test_read_ext_list_failure_signature_algo, 6);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_extension_parse().
 */
/* normal case */
START_TEST (test_tls_hs_extension_parse_normal_tls12)
{
	int32_t read_bytes;
	const uint32_t offset = 70;
	struct tls_extension *ext_cur, *ext_prev;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 12);
	ck_assert_int_eq(global_tls.errnum, 0);

	ext_cur = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_EC_POINT_FORMATS);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(ext_cur->len, 0);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_null(ext_cur);
}
END_TEST

START_TEST (test_tls_hs_extension_parse_normal_tls13)
{
	int32_t read_bytes;
	const uint32_t offset = 38;
	struct tls_extension *ext_cur, *ext_prev;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 48);
	ck_assert_int_eq(global_tls.errnum, 0);

	ext_cur = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(ext_cur->len, 36);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_null(ext_cur);
}
END_TEST

START_TEST (test_tls_hs_extension_parse_normal_tls13_hrr)
{
	int32_t read_bytes;
	const uint32_t offset = 38;
	struct tls_extension *ext_cur, *ext_prev;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 134);
	ck_assert_int_eq(global_tls.errnum, 0);

	ext_cur = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_COOKIE);
	ck_assert_int_eq(ext_cur->len, 116);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_null(ext_cur);
}
END_TEST

/* if (msg->len == offset) { */
START_TEST (test_tls_hs_extension_parse_normal_no_extension)
{
	int32_t read_bytes;
	const uint32_t offset = 70;
	struct tls_extension *ext;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = 70;

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, 0);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
	ck_assert_int_eq(ut_alert_desc, 0);

	ext = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_null(ext);
}
END_TEST

/* if (msg->len < (offset + length_bytes)) {... */
START_TEST (test_tls_hs_extension_parse_failure_invalid_record_length_1)
{
	int32_t read_bytes;
	const uint32_t offset = 70;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = 71;

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (msg->len < (offset + read_bytes + list_length)) {... */
START_TEST (test_tls_hs_extension_parse_failure_invalid_record_length_2)
{
	int32_t read_bytes;
	const uint32_t offset = 70;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = 77;

	read_bytes = tls_hs_extension_parse(&global_tls, &static_msg, offset);

	ck_assert_int_eq(read_bytes, -1);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *tls_hs_extension_parse_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_extension_parse()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal_tls13);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal_tls13_hrr);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal_no_extension);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_tls_hs_extension_parse_failure_invalid_record_length_1);
	tcase_add_test(tc_limits, test_tls_hs_extension_parse_failure_invalid_record_length_2);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_shello_parse().
 */
/* normal case */
START_TEST (test_tls_hs_shello_parse_normal_tls12)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_tls_hs_shello_parse_normal_tls13)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);
}
END_TEST

/* if (msg->type != TLS_HANDSHAKE_SERVER_HELLO) {... */
START_TEST (test_tls_hs_shello_parse_failure_unexpected_message)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_CLIENT_HELLO;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 10);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

/* if ((verlen = read_version(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_version)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 1;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((randlen = read_server_random(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_server_random)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 32;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((seslen = read_session_id(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_session_id)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 34;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((cslen = read_cipher_suite(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_cipher_suite)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 67;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((cmplen = read_cmp_method(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_cmp_method)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 69;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((extlen = tls_hs_extension_parse(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_shello_parse_failure_tls_hs_extension_parse)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = 71;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (msg->len != offset) {... */
START_TEST (test_tls_hs_shello_parse_failure_read_whole_message)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length + 1;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 11);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *tls_hs_shello_parse_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_shello_parse()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_tls_hs_shello_parse_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_shello_parse_normal_tls13);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_unexpected_message);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_version);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_server_random);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_session_id);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_cipher_suite);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_cmp_method);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_tls_hs_extension_parse);
	tcase_add_test(tc_limits, test_tls_hs_shello_parse_failure_read_whole_message);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_server_random().
 */
/* normal case */
START_TEST (test_interpret_server_random_normal_tls12)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_server_random(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_BEFORE_RECV_SHELLO);
}
END_TEST

START_TEST (test_interpret_server_random_normal_tls13)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_server_random(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_BEFORE_RECV_SHELLO);
}
END_TEST

START_TEST (test_interpret_server_random_normal_tls13_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_server_random(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_AFTER_RECV_HRREQ);
}
END_TEST

START_TEST (test_interpret_server_random_normal_tls13_after_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_after_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_after_hrr);

	global_tls.state = TLS_STATE_HS_BEFORE_RECV_2NDSHELLO;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_server_random(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_BEFORE_RECV_2NDSHELLO);
}
END_TEST

START_TEST (test_interpret_server_random_failure_tls13_after_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);

	global_tls.state = TLS_STATE_HS_AFTER_SEND_2NDCHELLO;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_server_random(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS6);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO2 + 4);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

Suite *interpret_server_random_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_server_random_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_interpret_server_random_normal_tls12);
	tcase_add_test(tc_core, test_interpret_server_random_normal_tls13);
	tcase_add_test(tc_core, test_interpret_server_random_normal_tls13_hrr);
	tcase_add_test(tc_core, test_interpret_server_random_normal_tls13_after_hrr);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_interpret_server_random_failure_tls13_after_hrr);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_session_id_suite().
 */
/* normal case */
START_TEST (test_interpret_session_id_normal_tls12)
{
	bool rc;
	uint8_t session_id[] = {
		SH_SESSIONID
	};

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	global_tls.pending->session_id_length = 0;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 32);
	ck_assert_mem_eq(&(global_tls.pending->session_id[0]), session_id, 32);
}
END_TEST

START_TEST (test_interpret_session_id_normal_tls13)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	global_tls.pending->session_id_length = 0;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);
}
END_TEST

/* if (tls->pending->session_id_length > 0) {... */
START_TEST (test_interpret_session_id_normal_tls12_same_session_id)
{
	bool rc;
	uint8_t session_id[] = {
		SH_SESSIONID
	};

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	global_tls.pending->session_id_length = 32;
	memcpy(global_tls.pending->session_id, session_id, 32);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 32);
	ck_assert_mem_eq(&(global_tls.pending->session_id[0]), session_id, 32);
	ck_assert_int_eq(global_tls.resession, true);
}
END_TEST

/* if ((len == tls->pending->session_id_length) && */
START_TEST (test_interpret_session_id_normal_tls12_different_session_id_len)
{
	bool rc;
	uint8_t session_id[] = {
		SH_SESSIONID
	};

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	/* if ((len == tls->pending->session_id_length) && */
	global_tls.pending->session_id_length = 16;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 32);
	ck_assert_mem_eq(&(global_tls.pending->session_id[0]), session_id, 32);
}
END_TEST

/* (memcmp(&(tls->pending->session_id[0]), &(session[0]), len) == 0)) {... */
START_TEST (test_interpret_session_id_normal_tls12_different_session_id_value)
{
	bool rc;
	uint8_t session_id[] = {
		SH_SESSIONID
	};

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	/* (memcmp(&(tls->pending->session_id[0]), &(session[0]), len) == 0)) {... */
	global_tls.pending->session_id_length = 32;
	memcpy(global_tls.pending->session_id, static_session_id, 32);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 32);
	ck_assert_mem_eq(&(global_tls.pending->session_id[0]), session_id, 32);
}
END_TEST

START_TEST (test_interpret_session_id_normal_tls13_compat)
{
	bool rc;
	uint8_t session_id[] = {
		SH_SESSIONID
	};

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_sesslen32;
	static_msg.len = sizeof(static_server_hello_tls13_sesslen32);

	global_tls.pending->session_id_length = 32;
	memcpy(global_tls.pending->session_id, session_id, 32);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->session_id_length, 32);
	ck_assert_mem_eq(&(global_tls.pending->session_id[0]), session_id, 32);
}
END_TEST

/* if ((tls->pending = tls_session_new()) == NULL) {... */
START_TEST (test_interpret_session_id_failure_tls12_get_session)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	global_tls.pending->session_id_length = 16;
	tls_session_new_failure = true;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_session_id(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_SESSION);
	ck_assert_int_eq(ut_location, ERR_LC_TLS3);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_SHELLO + 6);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_INTERNAL_ERROR);
}
END_TEST

Suite *interpret_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_session_id_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls13);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12_same_session_id);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12_different_session_id_len);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12_different_session_id_value);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_interpret_session_id_failure_tls12_get_session);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_cipher_suite().
 */
/* normal case */
START_TEST (test_interpret_cipher_suite_normal)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	rc = interpret_cipher_suite(&global_tls);

	ck_assert_int_eq(rc, true);
}
END_TEST

/* if (! tls_cipher_set(tls, suite)) {... */
START_TEST (test_interpret_cipher_suite_failure_cipher_set)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_cipher_set_failure = true;
	rc = interpret_cipher_suite(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
}
END_TEST

Suite *interpret_cipher_suite_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_cipher_suite_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_interpret_cipher_suite_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_interpret_cipher_suite_failure_cipher_set);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_cmp_method().
 */
/* normal case */
START_TEST (test_interpret_cmp_method_normal)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	rc = interpret_cmp_method(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);
}
END_TEST

/* if (! tls_compress_set(tls, method)) {... */
START_TEST (test_interpret_cmp_method_failure_compress_set)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_common;
	static_msg.len = static_server_hello_common_length;

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_compress_set_failure = true;
	rc = interpret_cmp_method(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
}
END_TEST

Suite *interpret_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_cmp_method_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_interpret_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_interpret_cmp_method_failure_compress_set);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_ext_list().
 */
/* normal case */
START_TEST (test_interpret_ext_list_normal_tls12)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, true);
	/* TODO */
	/* tls->ecdh->pflist[0] == TLS_ECC_PF_UNCOMPRESSED */
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13_hrr)
{
	bool rc;
	struct tls_extension *ext_cur, *ext_prev;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	global_tls.state = TLS_STATE_HS_AFTER_RECV_HRREQ;

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, true);

	ext_cur = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_COOKIE);
	ck_assert_int_eq(ext_cur->len, 116);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_nonnull(ext_cur);

	ck_assert_int_eq(ext_cur->type, TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(ext_cur->len, 2);

	ext_prev = ext_cur;
	ext_cur = TAILQ_NEXT(ext_prev, link);
	ck_assert_ptr_null(ext_cur);
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13_after_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_after_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_after_hrr);

	global_tls.state = TLS_STATE_HS_BEFORE_RECV_2NDSHELLO;
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_BEFORE_RECV_2NDSHELLO);
}
END_TEST

/* if (check_ext_availability(tls, type) == false) {... */
START_TEST (test_interpret_ext_list_failure_tls12_unsupported_extension)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_elliptic;
	static_msg.len = sizeof(static_server_hello_elliptic);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNSUPPORTED_EXTENSION);
}
END_TEST

/* if (check_ext_availability(tls, type) == false) {... */
START_TEST (test_interpret_ext_list_failure_tls13_unsupported_extension)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_elliptic;
	static_msg.len = sizeof(static_server_hello_elliptic);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNSUPPORTED_EXTENSION);
}
END_TEST

/* if (read_ext_list(tls, type, msg, base + off) == false) {... */
START_TEST (test_interpret_ext_list_failure_tls12_read_ext_list)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_ecc_read_point_format_failure = true;
	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (read_ext_list(tls, type, msg, base + off) == false) {... */
START_TEST (test_interpret_ext_list_failure_tls13_read_ext_list)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);
	tls_hs_keyshare_read_failure = true;
	rc = interpret_ext_list(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *interpret_ext_list_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_ext_list()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls12);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13_hrr);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13_after_hrr);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls12_unsupported_extension);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_unsupported_extension);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls12_read_ext_list);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_read_ext_list);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_shello_interpret().
 */
/* normal case */
START_TEST (test_tls_hs_shello_interpret_normal_tls12)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);

	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_tls_hs_shello_interpret_normal_tls13)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13;
	static_msg.len = sizeof(static_server_hello_tls13);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_tls_hs_shello_interpret_normal_tls13_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_hrr);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_tls_hs_shello_interpret_normal_tls13_after_hrr)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls13_after_hrr;
	static_msg.len = sizeof(static_server_hello_tls13_after_hrr);

	global_tls.state = TLS_STATE_HS_BEFORE_RECV_2NDSHELLO;
	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &global_tls.negotiated_version);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.state, TLS_STATE_HS_BEFORE_RECV_2NDSHELLO);
}
END_TEST

/* if (! interpret_session_id(tls)) {... */
START_TEST (test_tls_hs_shello_interpret_failure_session_id)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	global_tls.pending->session_id_length = 16;
	tls_session_new_failure = true;
	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! interpret_cipher_suite(tls)) {... */
START_TEST (test_tls_hs_shello_interpret_failure_failure_cipher_set)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_cipher_set_failure = true;
	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! interpret_cmp_method(tls)) {... */
START_TEST (test_tls_hs_shello_interpret_failure_compress_set)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_compress_set_failure = true;
	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! interpret_ext_list(tls)) {... */
START_TEST (test_tls_hs_shello_interpret_failure_read_ext_list)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	static_msg.msg = static_server_hello_tls12;
	static_msg.len = sizeof(static_server_hello_tls12);

	rc = tls_hs_shello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &global_tls.negotiated_version);
	tls_hs_ecc_read_point_format_failure = true;
	rc = tls_hs_shello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *tls_hs_shello_interpret_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_shello_interpret()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_tls_hs_shello_interpret_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_shello_interpret_normal_tls13);
	tcase_add_test(tc_core, test_tls_hs_shello_interpret_normal_tls13_hrr);
	tcase_add_test(tc_core, test_tls_hs_shello_interpret_normal_tls13_after_hrr);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_tls_hs_shello_interpret_failure_session_id);
	tcase_add_test(tc_limits, test_tls_hs_shello_interpret_failure_failure_cipher_set);
	tcase_add_test(tc_limits, test_tls_hs_shello_interpret_failure_compress_set);
	tcase_add_test(tc_limits, test_tls_hs_shello_interpret_failure_read_ext_list);
	suite_add_tcase(s, tc_limits);

	return s;
}

/**
 * test name (src file) suite (dummy).
 */
Suite *tls_hs_shello_suite(void)
{
	Suite *s;

	return s = suite_create("tls_hs_shello_suites");
}

int main(void)
{
	int number_failed;
	SRunner *sr;

	sr = srunner_create(tls_hs_shello_suite());

	srunner_add_suite(sr, write_version_suite());
	srunner_add_suite(sr, write_server_random_suite());
	srunner_add_suite(sr, write_session_id_suite());
	srunner_add_suite(sr, write_cipher_suite_suite());
	srunner_add_suite(sr, write_cmp_method_suite());
	srunner_add_suite(sr, write_extension_shello_suite());
	srunner_add_suite(sr, write_extension_hrr_suite());
	srunner_add_suite(sr, tls_hs_shello_compose_suite());

	srunner_add_suite(sr, read_version_suite());
	srunner_add_suite(sr, read_server_random_suite());
	srunner_add_suite(sr, read_session_id_suite());
	srunner_add_suite(sr, read_cipher_suite_suite());
	srunner_add_suite(sr, read_cmp_method_suite());
	srunner_add_suite(sr, check_ext_availability_tls12_suite());
	srunner_add_suite(sr, check_ext_availability_tls13_shello_suite());
	srunner_add_suite(sr, check_ext_availability_tls13_hrr_suite());
	srunner_add_suite(sr, read_ext_list_suite());
	srunner_add_suite(sr, tls_hs_extension_parse_suite());
	srunner_add_suite(sr, tls_hs_shello_parse_suite());

	srunner_add_suite(sr, interpret_server_random_suite());
	srunner_add_suite(sr, interpret_session_id_suite());
	srunner_add_suite(sr, interpret_cipher_suite_suite());
	srunner_add_suite(sr, interpret_cmp_method_suite());
	srunner_add_suite(sr, interpret_ext_list_suite());
	srunner_add_suite(sr, tls_hs_shello_interpret_suite());

	srunner_run_all(sr, CK_NORMAL);
	number_failed = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

/* ------------------------------------------------------------------------- */
#pragma GCC diagnostic ignored "-Wunused-parameter"
/* puts stub functions below this line. */

void OK_set_error(int error, int location, int point, CK_RV *info)
{
	ut_error = error;
	ut_location = location;
	ut_point = point;

	fprintf(stderr, "error:%0x location:%0x point:%0x\n",
		error, location, point);
	if (info != NULL) {
		fprintf(stderr, "*info=%zu\n", (size_t)(*info));
	} else {
		fprintf(stderr, "info=NULL\n");
	}
}

bool tls_alert_send(TLS *tls,
		    const enum tls_alert_level level,
		    const enum tls_alert_desc desc)
{
	ut_alert_desc = desc;
	return true;
}

bool tls_hs_msg_write_1(struct tls_hs_msg *msg, const uint8_t dat)
{
	if (tls_hs_msg_write_1_failure) {
		if (tls_hs_msg_write_1_through_count == 0) {
			return false;
		}
		tls_hs_msg_write_1_through_count--;
	}

	msg->msg[msg->len] = dat;
	msg->len++;
	return true;
}

bool tls_hs_msg_write_2(struct tls_hs_msg *msg, const uint16_t dat)
{
	if (tls_hs_msg_write_2_failure) {
		if (tls_hs_msg_write_2_through_count == 0) {
			return false;
		}
		tls_hs_msg_write_2_through_count--;
	}

	tls_util_write_2(&msg->msg[msg->len], dat);
	msg->len +=2;

	return true;
}

bool tls_hs_msg_write_n(struct tls_hs_msg *msg, const uint8_t *dat,
			const uint32_t len)
{
	if (tls_hs_msg_write_n_failure) {
		return false;
	}

	memcpy(&msg->msg[msg->len], dat, len);
	msg->len += len;
	return true;
}
bool tls_util_get_random(uint8_t *buf, uint32_t len)
{
	if (tls_util_get_random_failure) {
		if (tls_util_get_random_through_count == 0) {
			return false;
		}
		tls_util_get_random_through_count--;
	}

	memcpy(buf, static_session_id, 32);
	return true;
}
struct tls_session_param * tls_session_new(void)
{
	if (tls_session_new_failure) {
		return NULL;
	}
	return &static_param;
}
void tls_session_refer(struct tls_session_param *session)
{
	;
}
void tls_session_unrefer(struct tls_session_param *session)
{
	;
}
void tls_util_write_2(uint8_t *buf, int32_t val)
{
	buf[0] = ((val) >> 8) & 0xff;
	buf[1] = ((val)     ) & 0xff;
}
uint16_t tls_util_read_2(uint8_t *buf)
{
	return (((buf[0] << 8) & 0xff00) |
		((buf[1]     ) & 0x00ff));
}
void tls_util_convert_ver_to_protover(uint16_t version,
				      struct tls_protocol_version *version_st)
{
	version_st->major = (version >> 8) & 0xff;
	version_st->minor = (version     ) & 0xff;
}
uint16_t tls_util_convert_protover_to_ver(struct tls_protocol_version *version_st)
{
	return (version_st->major << 8) + ((version_st->minor) & 0x00ff);
}
bool tls_util_check_version_in_supported_version(
	struct tls_protocol_version_list *vlist,
	uint16_t version)
{
	for (int i = 0; i < vlist->len; i++) {
		if (vlist->list[i] == version) {
			return true;
		}
	}

	return false;
}
bool tls_cipher_set(TLS *tls, const uint16_t suite)
{
	if (tls_cipher_set_failure) {
		return false;
	}

	return true;
}
bool tls_compress_set(TLS *tls, enum compression_algorithm algo)
{
	if (tls_compress_set_failure) {
		return false;
	}

	tls->pending->compression_algorithm = algo;
	return true;
}
int32_t tls_hs_ecc_write_ec_point_formats(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_ecc_write_ec_point_formats_failure) {
		return -1;
	}

	if (tls_hs_ecc_write_ec_point_formats_0) {
		return 0;
	}

	memcpy(&msg->msg[msg->len], static_ec_point_formats_extension,
	       sizeof(static_ec_point_formats_extension));
	msg->len += sizeof(static_ec_point_formats_extension);
	return sizeof(static_ec_point_formats_extension);
}
int32_t tls_hs_ecc_read_point_format(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset)
{
	if (tls_hs_ecc_read_point_format_failure) {
		return -1;
	}
	// tls->ecdh->pflist = pflist;

	return 4;
}
int32_t tls_hs_supported_versions_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_supported_versions_write_failure) {
		return -1;
	}

	if (tls_hs_supported_versions_write_0) {
		return 0;
	}

	memcpy(&msg->msg[msg->len], static_supported_versions_extension,
	       sizeof(static_supported_versions_extension));
	msg->len += sizeof(static_supported_versions_extension);
	return sizeof(static_supported_versions_extension);
}
int32_t tls_hs_keyshare_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_keyshare_write_failure) {
		return -1;
	}

	if (tls_hs_keyshare_write_0) {
		return 0;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_HRREQ) == true) {
		memcpy(&msg->msg[msg->len], static_keyshare_extension_hrr,
		       sizeof(static_keyshare_extension_hrr));
		msg->len += sizeof(static_keyshare_extension_hrr);
		return sizeof(static_keyshare_extension_hrr);
	}

	memcpy(&msg->msg[msg->len], static_keyshare_extension,
	       sizeof(static_keyshare_extension));
	msg->len += sizeof(static_keyshare_extension);
	return sizeof(static_keyshare_extension);
}
int32_t tls_hs_keyshare_read(TLS *tls, const struct tls_hs_msg *msg,
			      const uint32_t offset)
{
	if (tls_hs_keyshare_read_failure) {
		return -1;
	}

	return 0;
}
int32_t tls_hs_servername_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_servername_write_failure) {
		return -1;
	}

	if (tls_hs_servername_write_0) {
		return 0;
	}

	memcpy(&msg->msg[msg->len], static_servername_extension,
	       sizeof(static_servername_extension));
	msg->len += sizeof(static_servername_extension);
	return sizeof(static_servername_extension);
}
int32_t tls_hs_servername_read(TLS *tls, const struct tls_hs_msg *msg,
			       const uint32_t offset)
{
	if (tls_hs_servername_read_failure) {
		return -1;
	}

	return 0;
}
int32_t tls_hs_cookie_read(TLS *tls, const struct tls_hs_msg *msg,
			   const uint32_t offset)
{
	return 0;
}

struct tls_hs_msg * tls_hs_msg_init(void)
{
	if (tls_hs_msg_init_failure) {
		return NULL;
	}
	return &static_msg;
}
void tls_hs_msg_free(struct tls_hs_msg *msg)
{
	if (msg != NULL) {
		msg->len = 0;
	}
}

/* -----------------tls_handshake.c----------------------------------------- */
void tls_hs_change_state(TLS *tls, enum tls_state state) {
	tls->state = state;

#ifdef TLS_DEBUG
	char name[30];
	switch(state) {
	case TLS_STATE_HS_INIT:
		strcpy(name, "INIT");
		break;

	case TLS_STATE_HS_SEND_HELLOREQ:
		strcpy(name, "SEND_HELLOREQ");
		break;

	case TLS_STATE_HS_RECV_HELLOREQ:
		strcpy(name, "RECV_HELLOREQ");
		break;

	case TLS_STATE_HS_BEFORE_SEND_CHELLO:
		strcpy(name, "SEND_CHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_SEND_CHELLO:
		strcpy(name, "SEND_CHELLO (AFTER)");
		break;

	case TLS_STATE_HS_BEFORE_RECV_SHELLO:
		strcpy(name, "RECV_SHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_SHELLO:
		strcpy(name, "RECV_SHELLO (AFTER)");
		break;

	case TLS_STATE_HS_RECV_ENCEXT:
		strcpy(name, "RECV_ENCEXT");
		break;

	case TLS_STATE_HS_RECV_SCERT:
		strcpy(name, "RECV_SCERT");
		break;

	case TLS_STATE_HS_RECV_SKEYEXC:
		strcpy(name, "RECV_SKEYEXC");
		break;

	case TLS_STATE_HS_RECV_SCERTVFY:
		strcpy(name, "RECV_SCERTVFY");
		break;

	case TLS_STATE_HS_RECV_CERTREQ:
		strcpy(name, "RECV_CERTREQ");
		break;

	case TLS_STATE_HS_RECV_SHELLODONE:
		strcpy(name, "RECV_SHELLODONE");
		break;

	case TLS_STATE_HS_SEND_CCERT:
		strcpy(name, "SEND_CCERT");
		break;

	case TLS_STATE_HS_SEND_CKEYEXC:
		strcpy(name, "SEND_CKEYEXC");
		break;

	case TLS_STATE_HS_SEND_CCERTVFY:
		strcpy(name, "SEND_CCERTVFY");
		break;

	case TLS_STATE_HS_BEFORE_RECV_CHELLO:
		strcpy(name, "RECV_CHELLO (BEFORE)");
		break;

	case TLS_STATE_HS_AFTER_RECV_CHELLO:
		strcpy(name, "RECV_CHELLO (AFTER)");
		break;

	case TLS_STATE_HS_SEND_SHELLO:
		strcpy(name, "SEND_SHELLO");
		break;

	case TLS_STATE_HS_SEND_ENCEXT:
		strcpy(name, "SEND_ENCEXT");
		break;

	case TLS_STATE_HS_SEND_SCERT:
		strcpy(name, "SEND_SCERT");
		break;

	case TLS_STATE_HS_SEND_SCERTVFY:
		strcpy(name, "SEND_SCERTVFY");
		break;

	case TLS_STATE_HS_SEND_SKEYEXC:
		strcpy(name, "SEND_SKEYEXC");
		break;

	case TLS_STATE_HS_SEND_CERTREQ:
		strcpy(name, "SEND_CERTREQ");
		break;

	case TLS_STATE_HS_SEND_SHELLODONE:
		strcpy(name, "SEND_SHELLODONE");
		break;

	case TLS_STATE_HS_RECV_CCERT:
		strcpy(name, "RECV_CCERT");
		break;

	case TLS_STATE_HS_RECV_CKEYEXC:
		strcpy(name, "RECV_CKEYEXC");
		break;

	case TLS_STATE_HS_RECV_CCERTVFY:
		strcpy(name, "RECV_CCERTVFY");
		break;

	case TLS_STATE_HS_RECV_FINISH:
		strcpy(name, "RECV_FINISH");
		break;

	case TLS_STATE_HS_SEND_FINISH:
		strcpy(name, "SEND_FINISH");
		break;

	case TLS_STATE_HS_BEFORE_FINISH:
		strcpy(name, "BEFORE_FINISH");
		break;

	case TLS_STATE_CCS_SEND:
		strcpy(name, "CCS_SEND");
		break;

	case TLS_STATE_CCS_RECV:
		strcpy(name, "CCS_RECV");
		break;

	case TLS_STATE_ESTABLISHED:
		strcpy(name, "ESTABLISHED");
		break;

	case TLS_STATE_CLOSED:
		strcpy(name, "CLOSED");
		break;

	default:
		strcpy(name, "unknown handshake type");
		break;
	}

	TLS_DPRINTF("handshake: state => %s", name);
#endif
}

bool tls_hs_check_state(TLS *tls, enum tls_state state) {
	if (tls->state == state) {
		return true;
	}
	return false;
}

struct tls_extension *tls_extension_init(void)
{
	struct tls_extension *ext;

	if ((ext = malloc(1 * sizeof(struct tls_extension))) == NULL) {
		TLS_DPRINTF("extension: malloc: %s", strerror(errno));
		return NULL;
	}
	ext->opaque = NULL;

	return ext;
}

void tls_extension_free(struct tls_extension *ext)
{
	if (ext == NULL) {
		return;
	}

	free(ext->opaque);
	ext->opaque = NULL;

	free(ext);
}

struct tls_hs_interim_params *tls_hs_interim_params_init(void) {
	struct tls_hs_interim_params *params;

	if ((params = calloc(1, sizeof(struct tls_hs_interim_params))) == NULL) {
		TLS_DPRINTF("interim_params: malloc: %s", strerror(errno));
		return NULL;
	}
	TAILQ_INIT(&(params->head));

	return params;
}

void tls_hs_interim_params_free(struct tls_hs_interim_params *params) {
	struct tls_extension *ext;

	if (params == NULL) {
		return;
	}

	free(params->session);
	params->session = NULL;

	free(params->cmplist);
	params->cmplist = NULL;

	while (!TAILQ_EMPTY(&(params->head))) {
		ext = TAILQ_FIRST(&(params->head));
		TAILQ_REMOVE(&(params->head), ext, link);
		tls_extension_free(ext);
	}

	free(params);
}

int32_t tls_hs_extension_parse(TLS *tls,
			      const struct tls_hs_msg *msg,
			      const uint32_t offset) {
	uint32_t read_bytes = 0;

	/* no extesion provided. */
	if (msg->len == offset) {
		return 0;
	}

	/* extension provided. */
	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	const uint32_t list_length = tls_util_read_2(&(msg->msg[offset]));
	if (list_length > TLS_EXT_SIZE_MAX) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	if (msg->len < (offset + read_bytes + list_length)) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* NOTE: in the case of tls->resession == true, do full check of
	 * extension. */
	for (uint32_t off = 0; off < list_length;) {
		uint32_t base = offset + read_bytes;

		struct tls_extension *ext;
		if ((ext = tls_extension_init()) == NULL) {
			TLS_DPRINTF("tls_extension_init");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		ext->type = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		ext->len = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		TLS_DPRINTF("extension: type = %02d, len = %d", ext->type, ext->len);

		if ((ext->opaque = malloc(1 * ext->len)) == NULL) {
			tls_extension_free(ext);
			TLS_DPRINTF("malloc: %s", strerror(errno));
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		memcpy(ext->opaque, &(msg->msg[base+off]), ext->len);
		TAILQ_INSERT_TAIL(&(tls->interim_params->head), ext, link);
		off += ext->len;
	}
	read_bytes += list_length;

	return read_bytes;
}
