/*
 * Copyright (c) 2017-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/* template for unit testing with Check <https://libcheck.github.io/check/> */
#include <stdlib.h>
#include <check.h>

#include <aicrypto/nrg_tls.h>
#include <stdbool.h>
#include "tls_handshake.h"

#include "chello.c"

// 2019-03-01 00:00:00
#define CH_EPOCH_TIME	1551366000

#define CH_VERSION_TLS12	0x03, 0x03
#define CH_VERSION_TLS13	0x03, 0x04
#define CH_CLIENT_VERSION	0x03, 0x03
#define CH_RANDOM_TLS13_2NDCHELLO	0xb0, 0xb1, 0xc5, 0xa5, 0xaa, 0x37, 0xc5, 0x91, \
			0x9f, 0x2e, 0xd1, 0xd5, 0xc6, 0xff, 0xf7, 0xfc, \
			0xb7, 0x84, 0x97, 0x16, 0x94, 0x5a, 0x2b, 0x8c, \
			0xee, 0x92, 0x58, 0xa3, 0x46, 0x67, 0x7b, 0x6f
#define CH_RANDOM_TLS13_COMPAT	0x4e, 0x64, 0x0a, 0x3f, 0x2c, 0x27, 0x38, 0xf0, \
			0x9c, 0x94, 0x18, 0xbd, 0x78, 0xed, 0xcc, 0xd7, \
			0x55, 0x9d, 0x05, 0x31, 0x19, 0x92, 0x76, 0xd4, \
			0xd9, 0x2a, 0x0e, 0x9e, 0xe9, 0xd7, 0x7d, 0x09
#define CH_RANDOM_TLS13	0xcb, 0x34, 0xec, 0xb1, 0xe7, 0x81, 0x63, 0xba, \
			0x1c, 0x38, 0xc6, 0xda, 0xcb, 0x19, 0x6a, 0x6d, \
			0xff, 0xa2, 0x1a, 0x8d, 0x99, 0x12, 0xec, 0x18, \
			0xa2, 0xef, 0x62, 0x83, 0x02, 0x4d, 0xec, 0xe7
#define CH_RANDOM	0xc9, 0xeb, 0xdb, 0x57, 0x46, 0x05, 0xb6, 0xec,	\
			0x2f, 0x98, 0xdc, 0x2f, 0x2f, 0xa6, 0x19, 0x8b, \
			0x75, 0x76, 0xa5, 0x0e, 0x7d, 0xca, 0x8d, 0x52,	\
			0x23, 0xaa, 0xd3, 0x01, 0xbe, 0xfb, 0xa7, 0x54
#define CH_SESSIONID_LENGTH_32	0x20
#define CH_SESSIONID_LENGTH_0	0x00
#define CH_SESSIONID_COMPAT	0xa8, 0x0c, 0x16, 0x55, 0x81, 0xa8, 0xe0, 0xd0, \
			0x6c, 0x00, 0x18, 0xd5, 0x4d, 0x3a, 0x06, 0xdd, \
			0x32, 0xcf, 0xd4, 0x05, 0x1e, 0xb0, 0x26, 0xfa, \
			0xd3, 0xfd, 0x0b, 0xa9, 0x92, 0x69, 0xe6, 0xef

/* Cipher Suite: TLS_RSA_WITH_3DES_EDE_CBC_SHA */
#define CH_CS_TLS_RSA_WITH_3DES_EDE_CBC_SHA				0x00, 0x0a
/* Cipher Suite: TLS_RSA_WITH_AES_128_CBC_SHA */
#define CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA				0x00, 0x2f
/* Cipher Suite: TLS_RSA_WITH_AES_256_CBC_SHA */
#define CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA				0x00, 0x35
/* Cipher Suite: TLS_RSA_WITH_AES_128_CBC_SHA256 */
#define CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA256			0x00, 0x3c
/* Cipher Suite: TLS_RSA_WITH_AES_256_CBC_SHA256 */
#define CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA256			0x00, 0x3d
/* Cipher Suite: TLS_RSA_WITH_AES_128_GCM_SHA256 */
#define CH_CS_TLS_RSA_WITH_AES_128_GCM_SHA256			0x00, 0x9c
/* Cipher Suite: TLS_RSA_WITH_AES_256_GCM_SHA384 */
#define CH_CS_TLS_RSA_WITH_AES_256_GCM_SHA384			0x00, 0x9d
/* Cipher Suite: TLS_ECDH_RSA_WITH_AES_128_CBC_SHA */
#define CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA			0xc0, 0x0e
/* Cipher Suite: TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA */
#define CH_CS_TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA		0xc0, 0x13
/* Cipher Suite: TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA */
#define CH_CS_TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA		0xc0, 0x14
/* Cipher Suite: TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 */
#define CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256		0xc0, 0x29
/* Cipher Suite: TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 */
#define CH_CS_TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256		0xc0, 0x2f
/* Cipher Suite: TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 */
#define CH_CS_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384		0xc0, 0x30
/* Cipher Suite: TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 */
#define CH_CS_TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256		0xc0, 0x31
/* Cipher Suite: TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 */
#define CH_CS_TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384		0xc0, 0x32
/* Cipher Suite: TLS_AES_128_GCM_SHA256 */
#define CH_CS_TLS_AES_128_GCM_SHA256					0x13, 0x01
/* Cipher Suite: TLS_AES_256_GCM_SHA384 */
#define CH_CS_TLS_AES_256_GCM_SHA384					0x13, 0x02
/* Cipher Suite: TLS_CHACHA20_POLY1305_SHA256 */
#define CH_CS_TLS_CHACHA20_POLY1305_SHA256				0x13, 0x03

/* Compression Method: null(0) */
#define	CH_CM_NULL				0x00

/* sha512(6) rsa(1) */
#define CH_SIGALGO_SHA512_RSA	0x06, 0x01
/* sha384(5) rsa(1) */
#define CH_SIGALGO_SHA384_RSA	0x05, 0x01
/* sha256(4) rsa(1) */
#define CH_SIGALGO_SHA256_RSA	0x04, 0x01
/* sha224(3) rsa(1) */
#define CH_SIGALGO_SHA224_RSA	0x03, 0x01
/* sha1(2)   rsa(1) */
#define CH_SIGALGO_SHA1_RSA	0x02, 0x01
/* md5(1)    rsa(1) */
#define CH_SIGALGO_MD5_RSA	0x01, 0x01
/* sha512(6) ecdsa(3) */
#define CH_SIGALGO_SHA512_ECDSA	0x06, 0x03
/* sha384(5) ecdsa(3) */
#define CH_SIGALGO_SHA384_ECDSA	0x05, 0x03
/* sha256(4) ecdsa(3) */
#define CH_SIGALGO_SHA256_ECDSA	0x04, 0x03
/* sha224(3) ecdsa(3) */
#define CH_SIGALGO_SHA224_ECDSA	0x03, 0x03
/* sha1(2)   ecdsa(3) */
#define CH_SIGALGO_SHA1_ECDSA	0x02, 0x03
/* sha512(6) dsa(2) */
#define CH_SIGALGO_SHA512_DSA	0x06, 0x02
/* sha384(5) dsa(2) */
#define CH_SIGALGO_SHA384_DSA	0x05, 0x02
/* sha256(4) dsa(2) */
#define CH_SIGALGO_SHA256_DSA	0x04, 0x02
/* sha224(3) dsa(2) */
#define CH_SIGALGO_SHA224_DSA	0x03, 0x02
/* sha1(2)   dsa(2) */
#define CH_SIGALGO_SHA1_DSA	0x02, 0x02
/* rsa_pss_rsae_sha256(0x0804) */
#define CH_SIGALGO_RSA_PSS_RSAE_SHA256	0x08, 0x04
/* rsa_pss_rsae_sha384(0x0805) */
#define CH_SIGALGO_RSA_PSS_RSAE_SHA384	0x08, 0x05
/* rsa_pss_rsae_sha512(0x0806) */
#define CH_SIGALGO_RSA_PSS_RSAE_SHA512	0x08, 0x06

#define CH_SIGALGO_RSASSA_PSS	0x08

/* ExtensionType: server_name(0) */
#define CH_EXTTYPE_SERVER_NAME	0x00, 0x0
/* ExtensionType: elliptic_curves(10) */
#define CH_EXTTYPE_ELLIPTIC_CURVES	0x00, 0x0a
/* ExtensionType: ec_point_formats(11) */
#define CH_EXTTYPE_EC_POINT_FORMATS	0x00, 0x0b
/* ExtensionType: signature_algorithms(13) */
#define CH_EXTTYPE_SIGNATURE_ALGORITHMS	0x00, 0x0d
/* ExtensionType: padding(21) */
#define CH_EXTTYPE_PADDING	0x00, 0x15
/* ExtensionType: record_size_limit(28) */
#define CH_EXTTYPE_RECORD_SIZE_LIMIT	0x00, 0x1c
/* ExtensionType: session_ticket(35) */
#define CH_EXTTYPE_SESSION_TICKET	0x00, 0x23
/* ExtensionType: supported_versions(43) */
#define CH_EXTTYPE_SUPPORTED_VERSIONS	0x00, 0x2b
/* ExtensionType: cookie(44) */
#define CH_EXTTYPE_COOKIE	0x00, 0x2c
/* ExtensionType: psk_key_exchange_modes(45) */
#define CH_EXTTYPE_PSK_KEY_EXCHANGE_MODES	0x00, 0x2d
/* ExtensionType: signature_algorithms(50) */
#define CH_EXTTYPE_SIGNATURE_ALGORITHMS_CERT	0x00, 0x32
/* ExtensionType: key_share(51) */
#define CH_EXTTYPE_KEY_SHARE	0x00, 0x33
/* ExtensionType: renegotiation_info */
#define CH_EXTTYPE_RENEGOTIATION_INFO	0xff, 0x01

/* server_name NameType: host_name(0) */
#define CH_SERVER_NAME_NAMETYPE_HOST_NAME	0x00

/* elliptic curve: secp256r1 (23) */
#define CH_CURVE_SECP256R1		0x00, 0x17
/* elliptic curve: secp384r1 (24) */
#define CH_CURVE_SECP384R1		0x00, 0x18
/* elliptic curve: secp521r1 (25) */
#define CH_CURVE_SECP521R1		0x00, 0x19
/* elliptic curve: X25519    (0x001D) */
#define CH_CURVE_X25519			0x00, 0x1d
/* elliptic curve: X448      (0x001E) */
#define CH_CURVE_X448			0x00, 0x1e

/* finite field group: ffdhe2048 (0x0100) */
#define CH_FFG_FFDHE2048		0x10, 0x00
/* finite field group: ffdhe3072 (0x0101) */
#define CH_FFG_FFDHE3072		0x10, 0x01
/* finite field group: ffdhe4096 (0x0102) */
#define CH_FFG_FFDHE4096		0x10, 0x02
/* finite field group: ffdhe6144 (0x0103) */
#define CH_FFG_FFDHE6144		0x10, 0x03
/* finite field group: ffdhe8192 (0x0104) */
#define CH_FFG_FFDHE8192		0x10, 0x04

#define CH_PADDING_CONTENT	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

#define CH_COOKIE_CONTENT	0x71, 0xdc, 0xd0, 0x4b, 0xb8, 0x8b, 0xc3, 0x18, \
			0x91, 0x19, 0x39, 0x8a, 0x00, 0x00, 0x00, 0x00, \
			0xee, 0xfa, 0xfc, 0x76, 0xc1, 0x46, 0xb8, 0x23, \
			0xb0, 0x96, 0xf8, 0xaa, 0xca, 0xd3, 0x65, 0xdd, \
			0x00, 0x30, 0x95, 0x3f, 0x4e, 0xdf, 0x62, 0x56, \
			0x36, 0xe5, 0xf2, 0x1b, 0xb2, 0xe2, 0x3f, 0xcc, \
			0x65, 0x4b, 0x1b, 0x5b, 0x40, 0x31, 0x8d, 0x10, \
			0xd1, 0x37, 0xab, 0xcb, 0xb8, 0x75, 0x74, 0xe3, \
			0x6e, 0x8a, 0x1f, 0x02, 0x5f, 0x7d, 0xfa, 0x5d, \
			0x6e, 0x50, 0x78, 0x1b, 0x5e, 0xda, 0x4a, 0xa1, \
			0x5b, 0x0c, 0x8b, 0xe7, 0x78, 0x25, 0x7d, 0x16, \
			0xaa, 0x30, 0x30, 0xe9, 0xe7, 0x84, 0x1d, 0xd9, \
			0xe4, 0xc0, 0x34, 0x22, 0x67, 0xe8, 0xca, 0x0c, \
			0xaf, 0x57, 0x1f, 0xb2, 0xb7, 0xcf, 0xf0, 0xf9, \
			0x34, 0xb0

/* PSK Key Exchange Modes: psk_dhe_ke */
#define CH_PSK_KEX_MODES_PSK_DHE_KE	0x01

/* client x25519 public key */
#define CH_CLIENT_X25519_PUB_COMPAT	0x8e, 0x72, 0x92, 0xcf, 0x30, 0x56, 0xdb, 0xb0, \
			0xd2, 0x5f, 0xcb, 0xe5, 0x5c, 0x10, 0x7d, 0xc9, \
			0xbb, 0xf8, 0x3d, 0xd9, 0x70, 0x8f, 0x39, 0x20, \
			0x3b, 0xa3, 0x41, 0x24, 0x9a, 0x7d, 0x9b, 0x63

#define CH_CLIENT_X25519_PUB	0x99, 0x38, 0x1d, 0xe5, 0x60, 0xe4, 0xbd, 0x43, \
			0xd2, 0x3d, 0x8e, 0x43, 0x5a, 0x7d, 0xba, 0xfe, \
			0xb3, 0xc0, 0x6e, 0x51, 0xc1, 0x3c, 0xae, 0x4d, \
			0x54, 0x13, 0x69, 0x1e, 0x52, 0x9a, 0xaf, 0x2c

/* client SECP256R1 public key */
#define CH_CLIENT_SECP256R1_PUB	0x04, 0xa6, 0xda, 0x73, 0x92, 0xec, 0x59, 0x1e, \
			0x17, 0xab, 0xfd, 0x53, 0x59, 0x64, 0xb9, 0x98, \
			0x94, 0xd1, 0x3b, 0xef, 0xb2, 0x21, 0xb3, 0xde, \
			0xf2, 0xeb, 0xe3, 0x83, 0x0e, 0xac, 0x8f, 0x01, \
			0x51, 0x81, 0x26, 0x77, 0xc4, 0xd6, 0xd2, 0x23, \
			0x7e, 0x85, 0xcf, 0x01, 0xd6, 0x91, 0x0c, 0xfb, \
			0x83, 0x95, 0x4e, 0x76, 0xba, 0x73, 0x52, 0x83, \
			0x05, 0x34, 0x15, 0x98, 0x97, 0xe8, 0x06, 0x57, \
			0x80

static uint8_t static_msg_body[1024];
static struct tls_hs_msg static_msg = {
	.len = 0,
	.max = 1023,
	.msg = static_msg_body,
};

static enum tls_cipher_suite static_cipher_list_body[] = {
	TLS_AES_256_GCM_SHA384,
	TLS_AES_128_GCM_SHA256,
	TLS_CHACHA20_POLY1305_SHA256,
	TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
	TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
	TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
	TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
	TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
	TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
	TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
	TLS_RSA_WITH_AES_256_GCM_SHA384,
	TLS_RSA_WITH_AES_128_GCM_SHA256,
	TLS_RSA_WITH_AES_256_CBC_SHA256,
	TLS_RSA_WITH_AES_128_CBC_SHA256,
	TLS_RSA_WITH_AES_256_CBC_SHA,
	TLS_RSA_WITH_AES_128_CBC_SHA,
	TLS_RSA_WITH_3DES_EDE_CBC_SHA,
};

static struct tls_cipher_list static_cipher_list = {
	.len = 18,
	.list = static_cipher_list_body,
};

static uint8_t static_session_id[33] = "abcdefghijABCDEFGHIJabcdefghijAB";
static uint8_t static_compat_session_id[] = {
	CH_SESSIONID_COMPAT,
};

static uint8_t *static_client_hello_common;
static uint32_t static_client_hello_common_length;

/* defined by RFC8448 3. Simple 1-RTT Handshake */
static uint8_t static_client_hello_tls13[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM_TLS13,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x06, /* cipher_suites: 6 byte */
	CH_CS_TLS_AES_128_GCM_SHA256,
	CH_CS_TLS_CHACHA20_POLY1305_SHA256,
	CH_CS_TLS_AES_256_GCM_SHA384,
	0x01, /* legacy_compression_methods: 1 byte */
	CH_CM_NULL,
	0x00, 0x91, /* extensions: 139 byte */
	CH_EXTTYPE_SERVER_NAME,
	0x00, 0x0b, /* extension length 11 byte */
	0x00, 0x09, /* server_name_list length 9 byte */
	CH_SERVER_NAME_NAMETYPE_HOST_NAME,
	0x00, 0x06, /* HostName length 6 byte */
	0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	CH_EXTTYPE_RENEGOTIATION_INFO,
	0x00, 0x01, /* extension length 1 byte */
	0x00,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x14, /* extension length 20 byte */
	0x00, 0x12, /* named_group_list length 18 byte */
	CH_CURVE_X25519,
	CH_CURVE_SECP256R1,
	CH_CURVE_SECP384R1,
	CH_CURVE_SECP521R1,
	CH_FFG_FFDHE2048,
	CH_FFG_FFDHE3072,
	CH_FFG_FFDHE4096,
	CH_FFG_FFDHE6144,
	CH_FFG_FFDHE8192,
	CH_EXTTYPE_SESSION_TICKET,
	0x00, 0x00, /* extension length 0 byte */
	CH_EXTTYPE_KEY_SHARE,
	0x00, 0x26, /* extension length 38 byte */
	0x00, 0x24, /* client_shares length 36 byte */
	CH_CURVE_X25519,
	0x00, 0x20, /* key_exchange length 32 byte */
	CH_CLIENT_X25519_PUB,
	CH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x03, /* extension length 3 byte */
	0x02, /* versions 2 byte */
	CH_VERSION_TLS13,
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x20, /* extension length 32 byte */
	0x00, 0x1e, /* supported_signature_algorithms length */
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA384_ECDSA,
	CH_SIGALGO_SHA512_ECDSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_SIGALGO_RSA_PSS_RSAE_SHA256,
	CH_SIGALGO_RSA_PSS_RSAE_SHA384,
	CH_SIGALGO_RSA_PSS_RSAE_SHA512,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA256_DSA,
	CH_SIGALGO_SHA384_DSA,
	CH_SIGALGO_SHA512_DSA,
	CH_SIGALGO_SHA1_DSA,
	CH_EXTTYPE_PSK_KEY_EXCHANGE_MODES,
	0x00, 0x02, /* extension length 2 byte */
	0x01, /* ke_modes length 1 byte */
	CH_PSK_KEX_MODES_PSK_DHE_KE,
	CH_EXTTYPE_RECORD_SIZE_LIMIT,
	0x00, 0x02, /* extension length 2 byte */
	0x40, 0x01,
};

/* defined by RFC8448 5. HelloRetryRequest */
static uint8_t static_2nd_client_hello_tls13[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM_TLS13_2NDCHELLO,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x06, /* cipher_suites: 6 byte */
	CH_CS_TLS_AES_128_GCM_SHA256,
	CH_CS_TLS_CHACHA20_POLY1305_SHA256,
	CH_CS_TLS_AES_256_GCM_SHA384,
	0x01, /* legacy_compression_methods: 1 byte */
	CH_CM_NULL,
	0x01, 0xcd, /* extensions: 461 byte */
	CH_EXTTYPE_SERVER_NAME,
	0x00, 0x0b, /* extension length 11 byte */
	0x00, 0x09, /* server_name_list length 9 byte */
	CH_SERVER_NAME_NAMETYPE_HOST_NAME,
	0x00, 0x06, /* HostName length 6 byte */
	0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	CH_EXTTYPE_RENEGOTIATION_INFO,
	0x00, 0x01, /* extension length 1 byte */
	0x00,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x08, /* extension length 8 byte */
	0x00, 0x06, /* named_group_list length 6 byte */
	CH_CURVE_X25519,
	CH_CURVE_SECP256R1,
	CH_CURVE_SECP384R1,
	CH_EXTTYPE_KEY_SHARE,
	0x00, 0x47, /* extension length 38 byte */
	0x00, 0x45, /* client_shares length 69 byte */
	CH_CURVE_SECP256R1,
	0x00, 0x41, /* key_exchange length 65 byte */
	CH_CLIENT_SECP256R1_PUB,
	CH_EXTTYPE_SUPPORTED_VERSIONS,
	00, 03, /* extension length 3 byte */
	02, /* versions length 2 byte */
	CH_VERSION_TLS13,
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x20, /* extension length 32 byte */
	0x00, 0x1e, /* supported_signature_algorithms length */
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA384_ECDSA,
	CH_SIGALGO_SHA512_ECDSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_SIGALGO_RSA_PSS_RSAE_SHA256,
	CH_SIGALGO_RSA_PSS_RSAE_SHA384,
	CH_SIGALGO_RSA_PSS_RSAE_SHA512,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA256_DSA,
	CH_SIGALGO_SHA384_DSA,
	CH_SIGALGO_SHA512_DSA,
	CH_SIGALGO_SHA1_DSA,
	CH_EXTTYPE_COOKIE,
	0x00, 0x74, /* extension length 116 byte */
	0x00, 0x72,
	CH_COOKIE_CONTENT,
	CH_EXTTYPE_PSK_KEY_EXCHANGE_MODES,
	0x00, 0x02, /* extension length 2 byte */
	0x01, /* ke_modes length 1 byte */
	CH_PSK_KEX_MODES_PSK_DHE_KE,
	CH_EXTTYPE_RECORD_SIZE_LIMIT,
	0x00, 0x02, /* extension length 2 byte */
	0x40, 0x01,
	CH_EXTTYPE_PADDING,
	0x00, 0xaf, /* extension length 175 byte */
	CH_PADDING_CONTENT
};

/* defined by RFC8448 7. Compatibility Mode */
static uint8_t static_client_hello_tls13_compat[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM_TLS13_COMPAT,
	CH_SESSIONID_LENGTH_32,
	CH_SESSIONID_COMPAT,
	0x00, 0x06, /* cipher_suites: 6 byte */
	CH_CS_TLS_AES_128_GCM_SHA256,
	CH_CS_TLS_CHACHA20_POLY1305_SHA256,
	CH_CS_TLS_AES_256_GCM_SHA384,
	0x01, /* legacy_compression_methods: 1 byte */
	CH_CM_NULL,
	0x00, 0x8d, /* extensions: 141 byte */
	CH_EXTTYPE_SERVER_NAME,
	0x00, 0x0b, /* extension length 11 byte */
	0x00, 0x09, /* server_name_list length 9 byte */
	CH_SERVER_NAME_NAMETYPE_HOST_NAME,
	0x00, 0x06, /* HostName length 6 byte */
	0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	CH_EXTTYPE_RENEGOTIATION_INFO,
	0x00, 0x01, /* extension length 1 byte */
	0x00,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x14, /* extension length 20 byte */
	0x00, 0x12, /* named_group_list length 18 byte */
	CH_CURVE_X25519,
	CH_CURVE_SECP256R1,
	CH_CURVE_SECP384R1,
	CH_CURVE_SECP521R1,
	CH_FFG_FFDHE2048,
	CH_FFG_FFDHE3072,
	CH_FFG_FFDHE4096,
	CH_FFG_FFDHE6144,
	CH_FFG_FFDHE8192,
	CH_EXTTYPE_KEY_SHARE,
	0x00, 0x26, /* extension length 38 byte */
	0x00, 0x24, /* client_shares length 36 byte */
	CH_CURVE_X25519,
	0x00, 0x20, /* key_exchange length 32 byte */
	CH_CLIENT_X25519_PUB_COMPAT,
	CH_EXTTYPE_SUPPORTED_VERSIONS,
	0x00, 0x03, /* extension length 3 byte */
	0x02, /* versions 2 byte */
	CH_VERSION_TLS13,
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x20, /* extension length 32 byte */
	0x00, 0x1e, /* supported_signature_algorithms length */
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA384_ECDSA,
	CH_SIGALGO_SHA512_ECDSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_SIGALGO_RSA_PSS_RSAE_SHA256,
	CH_SIGALGO_RSA_PSS_RSAE_SHA384,
	CH_SIGALGO_RSA_PSS_RSAE_SHA512,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA256_DSA,
	CH_SIGALGO_SHA384_DSA,
	CH_SIGALGO_SHA512_DSA,
	CH_SIGALGO_SHA1_DSA,
	CH_EXTTYPE_PSK_KEY_EXCHANGE_MODES,
	0x00, 0x02, /* extension length 2 byte */
	0x01, /* ke_modes length 1 byte */
	CH_PSK_KEX_MODES_PSK_DHE_KE,
	CH_EXTTYPE_RECORD_SIZE_LIMIT,
	0x00, 0x02, /* extension length 2 byte */
	0x40, 0x01,
};

static uint8_t static_client_hello_tls12[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
	0x00, 0x2e, /* extensions: 46 byte */
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x1c, /* opaque extension_data: 28 byte */
	0x00, 0x1a, /* 26 byte */
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA224_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA512_ECDSA,
	CH_SIGALGO_SHA384_ECDSA,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA224_ECDSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_SIGALGO_SHA256_DSA,
	CH_SIGALGO_SHA224_DSA,
	CH_SIGALGO_SHA1_DSA,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x04, /* 4 byte */
	0x00, 0x02, /* elliptic_curve_list: 2 byte */
	CH_CURVE_SECP256R1,
	CH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02, /* 2 byte */
	0x01, /* ec_point_format_list: 1 byte */
	0x00, /* uncompressed(0) */
};

__attribute__((unused))
static uint8_t static_client_hello_tls11[] = {
	0x03, 0x02, /* client_version: TLS_MAJOR_TLS, TLS_MINOR_TLS11 */
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
	0x00, 0x16, /* extensions: 22 byte */
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x04, /* opaque extension_data: 4 byte */
	0x00, 0x02, /* 2 byte */
	CH_SIGALGO_SHA512_RSA,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x04, /* opaque extension_data: 4 byte */
	0x00, 0x02, /* elliptic_curve_list: 2 byte */
	CH_CURVE_SECP256R1,
	CH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02, /* opaque extension_data: 2 byte */
	0x01, /* ec_point_format_list: 1 byte */
	0x00, /* uncompressed(0) */
};


static uint8_t static_client_hello_tls12_no_extension[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
};

static uint8_t static_client_hello_too_short[] = {
	CH_CLIENT_VERSION,
	0xc9, 0xeb, 0xdb, 0x57, /* Random.gmt_unix_time */
};

static uint8_t static_client_hello_tls12_session_error[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	0x10, /* SessionID.length : <= wrong value */
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
};

static uint8_t static_client_hello_tls12_cipher_suite_error[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x08, /* cipher_suites : <= wrong value */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
};

static uint8_t static_client_hello_tls12_cmp_method_error[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x04, /* compression_methods: <= wrong value */
	0x00, /* null(0) */
};

static uint8_t static_client_hello_tls12_ext_error[] = {
	CH_CLIENT_VERSION,
	CH_RANDOM,
	CH_SESSIONID_LENGTH_0,
	0x00, 0x02, /* cipher_suites: 2 byte */
	CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
	0x01, /* compression_methods: 1 byte */
	CH_CM_NULL,
	0x00, 0x08, /* extensions: <= wrong value */
};

static uint8_t static_client_hello_extension[] = {
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x1e, /* opaque extension_data: 30 byte */
	0x00, 0x1c, /* 28 byte */
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA224_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_MD5_RSA,
	CH_SIGALGO_SHA512_ECDSA,
	CH_SIGALGO_SHA384_ECDSA,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA224_ECDSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_SIGALGO_SHA256_DSA,
	CH_SIGALGO_SHA224_DSA,
	CH_SIGALGO_SHA1_DSA,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x04, /* opaque extension_data: 4 byte */
	0x00, 0x02, /* elliptic_curve_list: 2 byte */
	CH_CURVE_SECP256R1,
	CH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x05, /* opaque extension_data: 5 byte */
	0x04, /* ec_point_format_list: 4 byte */
	0x00, /* uncompressed(0) */
	0x65, /*  */
	0x61, /*  */
	0x2e, /*  */
};

static uint8_t static_extensions[] = {
	CH_EXTTYPE_SERVER_NAME,
	0x00, 0x0b,
	0x00, 0x09,
	CH_SERVER_NAME_NAMETYPE_HOST_NAME,
	0x00, 0x06,
	0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
	CH_EXTTYPE_SIGNATURE_ALGORITHMS,
	0x00, 0x14,
	0x00, 0x12,
	CH_SIGALGO_RSA_PSS_RSAE_SHA512,
	CH_SIGALGO_RSA_PSS_RSAE_SHA384,
	CH_SIGALGO_RSA_PSS_RSAE_SHA256,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_EXTTYPE_SIGNATURE_ALGORITHMS_CERT,
	0x00, 0x0e,
	0x00, 0x0c,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA1_ECDSA,
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x08,
	0x00, 0x06,
	CH_CURVE_X448,
	CH_CURVE_X25519,
	CH_CURVE_SECP256R1,
	CH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02,
	0x01,
	0x00,
	CH_EXTTYPE_SUPPORTED_VERSIONS,
	00, 05,
	04,
	CH_VERSION_TLS13,
	CH_VERSION_TLS12,
	CH_EXTTYPE_KEY_SHARE,
	0x00, 0x26,
	0x00, 0x24,
	CH_CURVE_X25519,
	00, 20,
	CH_CLIENT_X25519_PUB,
};

static uint8_t static_servername_extension[] = {
	CH_EXTTYPE_SERVER_NAME,
	0x00, 0x0b,
	0x00, 0x09,
	CH_SERVER_NAME_NAMETYPE_HOST_NAME,
	0x00, 0x06,
	0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
};

static uint8_t static_elliptic_curves_extension[] = {
	CH_EXTTYPE_ELLIPTIC_CURVES,
	0x00, 0x08,
	0x00, 0x06,
	CH_CURVE_X448,
	CH_CURVE_X25519,
	CH_CURVE_SECP256R1,
};

static uint8_t static_ec_point_formats_extension[] = {
	CH_EXTTYPE_EC_POINT_FORMATS,
	0x00, 0x02,
	0x01,
	0x00,
};

static uint8_t static_sigalgo_extension[] = {
	0x00, 0x12,
	CH_SIGALGO_RSA_PSS_RSAE_SHA512,
	CH_SIGALGO_RSA_PSS_RSAE_SHA384,
	CH_SIGALGO_RSA_PSS_RSAE_SHA256,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA1_ECDSA,
};

static uint8_t static_supported_versions_extension[] = {
	CH_EXTTYPE_SUPPORTED_VERSIONS,
	00, 05,
	04,
	CH_VERSION_TLS13,
	CH_VERSION_TLS12,
};

static uint8_t static_sigalgo_cert_extension[] = {
	0x00, 0x0c,
	CH_SIGALGO_SHA512_RSA,
	CH_SIGALGO_SHA384_RSA,
	CH_SIGALGO_SHA256_RSA,
	CH_SIGALGO_SHA256_ECDSA,
	CH_SIGALGO_SHA1_RSA,
	CH_SIGALGO_SHA1_ECDSA,
};

static uint8_t static_keyshare_extension[] = {
	CH_EXTTYPE_KEY_SHARE,
	0x00, 0x26,
	0x00, 0x24,
	CH_CURVE_X25519,
	00, 20,
	CH_CLIENT_X25519_PUB,
};

static struct tls_hs_sighash_algo tls_hs_sighash_algo_tls13[] = {
	{ .sig = TLS_SIG_ALGO_ECDSA,    .hash = TLS_HASH_ALGO_SHA256 },
	{ .sig = CH_SIGALGO_RSASSA_PSS, .hash = TLS_HASH_ALGO_SHA256 },
	{ .sig = CH_SIGALGO_RSASSA_PSS, .hash = TLS_HASH_ALGO_SHA384 },
	{ .sig = CH_SIGALGO_RSASSA_PSS, .hash = TLS_HASH_ALGO_SHA512 },
};

static struct tls_hs_sighash_list tls_hs_sighash_list_tls13 = {
	.len = 4,
	.list = tls_hs_sighash_algo_tls13,
};

static struct tls_hs_sighash_algo tls_hs_sighash_algo_cert_tls13[] = {
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA256 },
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA1   },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA256 },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA384 },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA512 },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA1   },
};

static struct tls_hs_sighash_list tls_hs_sighash_list_cert_tls13 = {
	.len = 6,
	.list = tls_hs_sighash_algo_cert_tls13,
};

static uint16_t static_supported_versions_list[] = {
	TLS_VER_TLS13,
	TLS_VER_TLS12,
};

static bool tls_hs_msg_write_1_failure = false;
static unsigned int tls_hs_msg_write_1_through_count = 0;
static bool tls_hs_msg_write_2_failure = false;
static unsigned int tls_hs_msg_write_2_through_count = 0;
static bool tls_hs_msg_write_n_failure = false;
static bool tls_hs_msg_init_failure = false;
static bool tls_util_get_epochtime_failure = false;
static unsigned int tls_util_get_epochtime_increment = 0;
static bool tls_util_get_random_failure = false;
static bool tls_util_get_random_need_randomness = false;
static unsigned int tls_util_get_random_through_count = 0;
static bool tls_cipher_supported_list_failure = false;

static bool tls_session_new_failure = false;
static bool tls_cipher_list_alloc_failure = false;
static unsigned int tls_cipher_list_alloc_count = 0;
static bool tls_compress_set_failure = false;

static bool tls_hs_servername_write_failure = false;
static bool tls_hs_ecc_write_elliptic_curves_failure = false;
static bool tls_hs_ecc_write_ec_point_formats_failure = false;
static bool tls_hs_sighash_write_failure = false;
static bool tls_hs_cookie_write_failure = false;
static bool tls_hs_supported_versions_write_failure = false;
static bool tls_hs_sighash_cert_write_failure = false;
static bool tls_hs_keyshare_write_failure = false;

static int tls_hs_ecc_read_elliptic_curves_false = false;
static int tls_hs_ecc_read_point_format_false = false;
static int tls_hs_sighash_read_false = false;
static int tls_hs_keyshare_read_false = false;
static int tls_hs_servername_read_false = false;
static int tls_hs_sighash_save_default_failure = false;

static int tls_hs_sighash_read_set_sighash_list = false;
static int tls_hs_sighash_read_set_sighash_list_cert = false;

static int ut_error;
static int ut_location;
static int ut_point;

static void ut_error_setup()
{
	ut_error = 0;
	ut_location = 0;
	ut_point = 0;
}

static enum tls_alert_desc ut_alert_desc;

static void ut_alert_setup()
{
	ut_alert_desc = 0;
}

static void _util_write_2(uint8_t *buf, int32_t val);

/*
 * fixture
 */
void setup(void)
{
	/* code */
}
void teardown(void)
{
	/* code */
}

static TLS global_tls;
void setup_tls(void)
{
	static_msg.len = 0;
	memset(static_msg.msg, 0, 1024);

	global_tls.client_version.major = TLS_MAJOR_MAX;
	global_tls.client_version.minor = TLS_MINOR_MAX;

	global_tls.negotiated_version.major = TLS_MAJOR_MAX;
	global_tls.negotiated_version.minor = TLS_MINOR_MAX;

	global_tls.record_version.major = TLS_MAJOR_MAX;
	global_tls.record_version.minor = TLS_MINOR_MAX;

	global_tls.errnum = 0;

	global_tls.interim_params = tls_hs_interim_params_init();

	// for read_session_id() test
	global_tls.resession = false;
	global_tls.pending = NULL;

	// for read_ext_list() test
	global_tls.sighash_list = NULL;

	static_client_hello_common = static_client_hello_tls12;
	static_client_hello_common_length = sizeof(static_client_hello_tls12);

	ut_error_setup();
	ut_alert_setup();

	tls_hs_msg_write_1_failure = false;
	tls_hs_msg_write_1_through_count = 0;
	tls_hs_msg_write_2_failure = false;
	tls_hs_msg_write_2_through_count = 0;
	tls_hs_msg_write_n_failure = false;
	tls_hs_msg_init_failure = false;
	tls_util_get_epochtime_failure = false;
	tls_util_get_epochtime_increment = 0;
	tls_util_get_random_failure = false;
	tls_util_get_random_need_randomness = false;
	tls_util_get_random_through_count = 0;
	tls_cipher_supported_list_failure = false;

	tls_session_new_failure = false;
	tls_cipher_list_alloc_failure = false;
	tls_cipher_list_alloc_count = 0;
	tls_compress_set_failure = false;
	tls_hs_servername_write_failure = false;
	tls_hs_ecc_write_elliptic_curves_failure = false;
	tls_hs_ecc_write_ec_point_formats_failure = false;
	tls_hs_sighash_write_failure = false;
	tls_hs_cookie_write_failure = false;
	tls_hs_supported_versions_write_failure = false;
	tls_hs_sighash_cert_write_failure = false;
	tls_hs_keyshare_write_failure = false;
	tls_hs_ecc_read_elliptic_curves_false = false;
	tls_hs_ecc_read_point_format_false = false;
	tls_hs_sighash_read_false = false;
	tls_hs_keyshare_read_false = false;
	tls_hs_servername_read_false = false;
	tls_hs_sighash_save_default_failure = false;

	tls_hs_sighash_read_set_sighash_list = false;
	tls_hs_sighash_read_set_sighash_list_cert = false;
}

void setup_client(void)
{
	setup_tls();

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_CHELLO;
}

void setup_server(void)
{
	setup_tls();

	global_tls.state = TLS_STATE_HS_BEFORE_RECV_CHELLO;
}

void teardown_tls(void)
{
	if (global_tls.interim_params != NULL) {
		tls_hs_interim_params_free(global_tls.interim_params);
	}

	if (global_tls.pending != NULL) {
		free(global_tls.pending);
	}
}

/*
 * unit testing for write_client_version().
 */
/* normal case */
START_TEST (test_write_client_version_normal)
{
	int32_t len;
	uint16_t msg_version;
	uint16_t client_version;
	uint16_t record_version;

	len = write_client_version(&global_tls, &static_msg);

	msg_version = tls_util_read_2(&(static_msg.msg[0]));
	client_version = tls_util_convert_protover_to_ver(
		&(global_tls.client_version));
	record_version = tls_util_convert_protover_to_ver(
		&(global_tls.record_version));

	ck_assert_int_eq(len, 2);
	ck_assert_int_eq(msg_version, TLS_VER_TLS12);
	ck_assert_int_eq(client_version, TLS_VER_TLS12);
	ck_assert_int_eq(record_version, TLS_VER_TLS10);
}
END_TEST

START_TEST (test_write_client_version_normal_2ndchello)
{
	int32_t len;
	uint16_t msg_version;
	uint16_t client_version;
	uint16_t record_version;

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_2NDCHELLO;
	tls_util_convert_ver_to_protover(0x0000, &(global_tls.client_version));
	tls_util_convert_ver_to_protover(0x0000, &(global_tls.record_version));

	len = write_client_version(&global_tls, &static_msg);

	msg_version = tls_util_read_2(&(static_msg.msg[0]));
	client_version = tls_util_convert_protover_to_ver(
		&(global_tls.client_version));
	record_version = tls_util_convert_protover_to_ver(
		&(global_tls.record_version));

	ck_assert_int_eq(len, 2);
	ck_assert_int_eq(msg_version, TLS_VER_TLS12);
	ck_assert_int_eq(client_version, 0x0000);
	ck_assert_int_eq(record_version, 0x0000);
}
END_TEST

/* if (! tls_hs_msg_write_2(msg, client_version)) {... */
START_TEST (test_write_client_version_failure_msg_write_2)
{
	int32_t version_length;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;

	version_length = write_client_version(&global_tls, &static_msg);

	ck_assert_int_eq(version_length, -1);
}
END_TEST

Suite *write_client_version_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_client_version()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_client_version_normal);
	tcase_add_test(tc_core, test_write_client_version_normal_2ndchello);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_client_version_failure_msg_write_2);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_client_random().
 */
/* normal case */
START_TEST (test_write_client_random_normal)
{
	int32_t len;
	uint64_t sec = CH_EPOCH_TIME;

	len = write_client_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&(global_tls.client_random[0]), &sec, 4);
	ck_assert_mem_eq(&(global_tls.client_random[4]), static_session_id, 28);
	ck_assert_mem_eq(&(static_msg.msg[0]), &sec, 4);
	ck_assert_mem_eq(&(static_msg.msg[4]), static_session_id, 28);
}
END_TEST

START_TEST (test_write_client_random_normal_2ndchello)
{
	int32_t len;

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_2NDCHELLO;
	memcpy(global_tls.client_random, static_session_id, 32);

	len = write_client_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, 32);
	ck_assert_mem_eq(&(global_tls.client_random[0]), static_session_id, 32);
	ck_assert_mem_eq(&(static_msg.msg[0]), static_session_id, 32);
}
END_TEST

/* if ((sec = tls_util_get_epochtime()) == 0) {... */
START_TEST (test_write_client_random_failure_get_epochtime)
{
	int32_t len;

	tls_util_get_epochtime_failure = true;

	len = write_client_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_EPOCHTIME);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 0);
}
END_TEST

/* if (! tls_util_get_random(&(crandom[0]), 28)) {... */
START_TEST (test_write_client_random_failure_get_random)
{
	int32_t len;

	tls_util_get_random_failure = true;

	len = write_client_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_RANDOM);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 1);
}
END_TEST

/* if(! tls_hs_msg_write_n(msg, tls->client_random, randlen)) {... */
START_TEST (test_write_client_random_failure_msg_write_n)
{
	int32_t len;

	tls_hs_msg_write_n_failure = true;

	len = write_client_random(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_client_random_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_client_random()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_client_random_normal);
	tcase_add_test(tc_core, test_write_client_random_normal_2ndchello);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_client_random_failure_get_epochtime);
	tcase_add_test(tc_limits, test_write_client_random_failure_get_random);
	tcase_add_test(tc_limits, test_write_client_random_failure_msg_write_n);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_session_id().
 */
/* normal case */
START_TEST (test_write_session_id_normal)
{
	int32_t len;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 1);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);
	ck_assert_int_eq(static_msg.msg[0], 0);
}
END_TEST

START_TEST (test_write_session_id_normal_2ndchello)
{
	int32_t len;
	struct tls_session_param *param;

	global_tls.state = TLS_STATE_HS_BEFORE_SEND_2NDCHELLO;
	param = tls_session_new();
	global_tls.pending = param;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, 1);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);
	ck_assert_int_eq(static_msg.msg[0], 0);
	ck_assert_ptr_eq(global_tls.pending, param);
}
END_TEST

/* if ((tls->pending = tls_session_new()) == NULL) {... */
START_TEST (test_write_session_id_failure_get_session)
{
	int32_t len;

	tls_session_new_failure = true;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_SESSION);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 2);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, session_id_length)) {... */
START_TEST (test_write_session_id_failure_msg_write_1)
{
	int32_t len;

	tls_hs_msg_write_1_failure = true;

	len = write_session_id(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_session_id()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_session_id_normal);
	tcase_add_test(tc_core, test_write_session_id_normal_2ndchello);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_session_id_failure_get_session);
	tcase_add_test(tc_limits, test_write_session_id_failure_msg_write_1);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_cipher_suite().
 */
/* normal case */
START_TEST (test_write_cipher_suite_normal)
{
	int32_t len;
	uint16_t list_len;
	uint16_t cipher;

	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, 38);
	list_len = tls_util_read_2(&(static_msg.msg[0]));
	ck_assert_uint_eq(list_len, 36);

	cipher = tls_util_read_2(&(static_msg.msg[2]));
	ck_assert_int_eq(cipher, TLS_AES_256_GCM_SHA384);

	cipher = tls_util_read_2(&(static_msg.msg[4]));
	ck_assert_int_eq(cipher, TLS_AES_128_GCM_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[6]));
	ck_assert_int_eq(cipher, TLS_CHACHA20_POLY1305_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[8]));
	ck_assert_int_eq(cipher, TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384);

	cipher = tls_util_read_2(&(static_msg.msg[10]));
	ck_assert_int_eq(cipher, TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[12]));
	ck_assert_int_eq(cipher, TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA);

	cipher = tls_util_read_2(&(static_msg.msg[14]));
	ck_assert_int_eq(cipher, TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA);

	cipher = tls_util_read_2(&(static_msg.msg[16]));
	ck_assert_int_eq(cipher, TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384);

	cipher = tls_util_read_2(&(static_msg.msg[18]));
	ck_assert_int_eq(cipher, TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[20]));
	ck_assert_int_eq(cipher, TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[22]));
	ck_assert_int_eq(cipher, TLS_ECDH_RSA_WITH_AES_128_CBC_SHA);

	cipher = tls_util_read_2(&(static_msg.msg[24]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_256_GCM_SHA384);

	cipher = tls_util_read_2(&(static_msg.msg[26]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_128_GCM_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[28]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_256_CBC_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[30]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_128_CBC_SHA256);

	cipher = tls_util_read_2(&(static_msg.msg[32]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_256_CBC_SHA);

	cipher = tls_util_read_2(&(static_msg.msg[34]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_AES_128_CBC_SHA);

	cipher = tls_util_read_2(&(static_msg.msg[36]));
	ck_assert_int_eq(cipher, TLS_RSA_WITH_3DES_EDE_CBC_SHA);
}
END_TEST

/* if (list == NULL) {... */
START_TEST (test_write_cipher_suite_failure_supported_list)
{
	int32_t len;

	tls_cipher_supported_list_failure = true;

	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (! tls_hs_msg_write_2(msg, list->len * 2)) {... */
START_TEST (test_write_cipher_suite_failure_write_length)
{
	int32_t len;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;

	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

/* if (! tls_hs_msg_write_2(msg, list->list[i])) {... */
START_TEST (test_write_cipher_suite_failure_write_list)
{
	int32_t len;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;

	len = write_cipher_suite(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
}
END_TEST

Suite *write_cipher_suite_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_cipher_suite()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_cipher_suite_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_cipher_suite_failure_supported_list);
	tcase_add_test(tc_limits, test_write_cipher_suite_failure_write_length);
	tcase_add_test(tc_limits, test_write_cipher_suite_failure_write_list);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_cmp_method().
 */
/* normal case */
START_TEST (test_write_cmp_method_normal)
{
	int32_t len;

	len = write_cmp_method(&static_msg);

	ck_assert_int_eq(len, 2);
	ck_assert_uint_eq(static_msg.msg[0], 1);
	ck_assert_uint_eq(static_msg.msg[1], TLS_COMPRESSION_NULL);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, 1)) {... */
START_TEST (test_write_cmp_method_failure_write_length)
{
	int32_t len;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;

	len = write_cmp_method(&static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 0);
}
END_TEST

/* if (! tls_hs_msg_write_1(msg, TLS_COMPRESSION_NULL)) {... */
START_TEST (test_write_cmp_method_failure_write_list)
{
	int32_t len;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 1;

	len = write_cmp_method(&static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 1);
}
END_TEST

Suite *write_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_cmp_method()");

	/* Core_cmp_method() test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_cmp_method_failure_write_length);
	tcase_add_test(tc_limits, test_write_cmp_method_failure_write_list);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_ext_sigalgo().
 */
/* normal case */
START_TEST (test_write_ext_sigalgo_normal)
{
	int32_t len;
	uint16_t ext_type;
	uint16_t ext_length;
	uint16_t list_length;
	uint16_t sighash;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext_sigalgo(&global_tls, &static_msg);

	ck_assert_int_eq(len, 24);

	ext_type = tls_util_read_2(&(static_msg.msg[0]));
	ck_assert_int_eq(ext_type, TLS_EXT_SIGNATURE_ALGO);

	ext_length = tls_util_read_2(&(static_msg.msg[2]));
	ck_assert_int_eq(ext_length, 20);

	list_length = tls_util_read_2(&(static_msg.msg[4]));
	ck_assert_int_eq(list_length, 18);

	sighash = tls_util_read_2(&(static_msg.msg[6]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PSS_RSAE_SHA512);

	sighash = tls_util_read_2(&(static_msg.msg[8]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PSS_RSAE_SHA384);

	sighash = tls_util_read_2(&(static_msg.msg[10]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PSS_RSAE_SHA256);

	sighash = tls_util_read_2(&(static_msg.msg[12]));
	ck_assert_int_eq(sighash, TLS_SS_ECDSA_SECP256R1_SHA256);

	sighash = tls_util_read_2(&(static_msg.msg[14]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA512);

	sighash = tls_util_read_2(&(static_msg.msg[16]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA384);

	sighash = tls_util_read_2(&(static_msg.msg[18]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA256);

	sighash = tls_util_read_2(&(static_msg.msg[20]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PCKS1_SHA1);

	sighash = tls_util_read_2(&(static_msg.msg[22]));
	ck_assert_int_eq(sighash, TLS_SS_ECDSA_SHA1);
}
END_TEST

START_TEST (test_write_ext_sigalgo_normal_empty)
{
	int32_t len;

	len = write_ext_sigalgo(&global_tls, &static_msg);

	ck_assert_int_eq(len, 0);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, TLS_EXT_SIGNATURE_ALGO) == false) {... */
START_TEST (test_write_ext_sigalgo_failure_write_type)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;

	len = write_ext_sigalgo(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 0);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, 0) == false) { */
START_TEST (test_write_ext_sigalgo_failure_write_length)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;

	len = write_ext_sigalgo(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 2);
}
END_TEST

/* if ((sighash_len = tls_hs_sighash_write(tls, msg)) < 0) {... */
START_TEST (test_write_ext_sigalgo_failure_sighash_write)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_sighash_write_failure = true;

	len = write_ext_sigalgo(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 4);
}
END_TEST

Suite *write_ext_sigalgo_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_ext_sigalgo()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_ext_sigalgo_normal);
	tcase_add_test(tc_core, test_write_ext_sigalgo_normal_empty);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_failure_write_type);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_failure_write_length);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_failure_sighash_write);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_ext_sigalgo_cert().
 */
/* normal case */
START_TEST (test_write_ext_sigalgo_cert_normal)
{
	int32_t len;
	uint16_t ext_type;
	uint16_t ext_length;
	uint16_t list_length;
	uint16_t sighash;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext_sigalgo_cert(&global_tls, &static_msg);

	ck_assert_int_eq(len, 18);

	ext_type = tls_util_read_2(&(static_msg.msg[0]));
	ck_assert_int_eq(ext_type, TLS_EXT_SIGNATURE_ALGO_CERT);

	ext_length = tls_util_read_2(&(static_msg.msg[2]));
	ck_assert_int_eq(ext_length, 14);

	list_length = tls_util_read_2(&(static_msg.msg[4]));
	ck_assert_int_eq(list_length, 12);

	sighash = tls_util_read_2(&(static_msg.msg[6]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA512);

	sighash = tls_util_read_2(&(static_msg.msg[8]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA384);

	sighash = tls_util_read_2(&(static_msg.msg[10]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PKCS1_SHA256);

	sighash = tls_util_read_2(&(static_msg.msg[12]));
	ck_assert_int_eq(sighash, TLS_SS_ECDSA_SECP256R1_SHA256);

	sighash = tls_util_read_2(&(static_msg.msg[14]));
	ck_assert_int_eq(sighash, TLS_SS_RSA_PCKS1_SHA1);

	sighash = tls_util_read_2(&(static_msg.msg[16]));
	ck_assert_int_eq(sighash, TLS_SS_ECDSA_SHA1);
}
END_TEST

START_TEST (test_write_ext_sigalgo_cert_normal_empty)
{
	int32_t len;

	len = write_ext_sigalgo_cert(&global_tls, &static_msg);

	ck_assert_int_eq(len, 0);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, TLS_EXT_SIGNATURE_ALGO_CERT) == false) {...  */
START_TEST (test_write_ext_sigalgo_cert_failure_write_type)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;

	len = write_ext_sigalgo_cert(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 0);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, 0) == false) {... */
START_TEST (test_write_ext_sigalgo_cert_failure_write_length)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;

	len = write_ext_sigalgo_cert(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 2);
}
END_TEST

/* if ((sighash_len = tls_hs_sighash_cert_write(tls, msg)) < 0) {...  */
START_TEST (test_write_ext_sigalgo_cert_failure_sighash_cert_write)
{
	int32_t len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;
	tls_hs_sighash_cert_write_failure = true;

	len = write_ext_sigalgo_cert(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 4);
}
END_TEST

Suite *write_ext_sigalgo_cert_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_ext_sigalgo_cert()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_ext_sigalgo_cert_normal);
	tcase_add_test(tc_core, test_write_ext_sigalgo_cert_normal_empty);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_cert_failure_write_type);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_cert_failure_write_length);
	tcase_add_test(tc_limits, test_write_ext_sigalgo_cert_failure_sighash_cert_write);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for write_ext().
 */
/* normal case */
START_TEST (test_write_ext_normal)
{
	int32_t len;
	uint8_t whole_len;
	uint8_t ext_type;
	uint8_t ext_len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, 128);

	whole_len = tls_util_read_2(&(static_msg.msg[0]));
	ck_assert_uint_eq(whole_len, 126);

	ext_type = tls_util_read_2(&(static_msg.msg[2]));
	ck_assert_uint_eq(ext_type, TLS_EXT_SERVER_NAME);
	ext_len = tls_util_read_2(&(static_msg.msg[4]));
	ck_assert_uint_eq(ext_len, 11);
	ck_assert_mem_eq(&(static_msg.msg[6]), &static_servername_extension[4],
			 ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[17]));
	ck_assert_uint_eq(ext_type, TLS_EXT_SIGNATURE_ALGO);
	ext_len = tls_util_read_2(&(static_msg.msg[19]));
	ck_assert_uint_eq(ext_len, 20);
	ck_assert_mem_eq(&(static_msg.msg[21]), static_sigalgo_extension,
			 ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[41]));
	ck_assert_uint_eq(ext_type, TLS_EXT_SIGNATURE_ALGO_CERT);
	ext_len = tls_util_read_2(&(static_msg.msg[43]));
	ck_assert_uint_eq(ext_len, 14);
	ck_assert_mem_eq(&(static_msg.msg[45]), static_sigalgo_cert_extension,
			 ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[59]));
	ck_assert_uint_eq(ext_type, TLS_EXT_ELLIPTIC_CURVES);
	ext_len = tls_util_read_2(&(static_msg.msg[61]));
	ck_assert_uint_eq(ext_len, 8);
	ck_assert_mem_eq(&(static_msg.msg[63]),
			 &static_elliptic_curves_extension[4], ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[71]));
	ck_assert_uint_eq(ext_type, TLS_EXT_EC_POINT_FORMATS);
	ext_len = tls_util_read_2(&(static_msg.msg[73]));
	ck_assert_uint_eq(ext_len, 2);
	ck_assert_mem_eq(&(static_msg.msg[75]),
			 &static_ec_point_formats_extension[4], ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[77]));
	ck_assert_uint_eq(ext_type, TLS_EXT_SUPPORTED_VERSIONS);
	ext_len = tls_util_read_2(&(static_msg.msg[79]));
	ck_assert_uint_eq(ext_len, 5);
	ck_assert_mem_eq(&(static_msg.msg[81]),
			 &static_supported_versions_extension[4], ext_len);

	ext_type = tls_util_read_2(&(static_msg.msg[86]));
	ck_assert_uint_eq(ext_type, TLS_EXT_KEY_SHARE);
	ext_len = tls_util_read_2(&(static_msg.msg[88]));
	ck_assert_uint_eq(ext_len, 38);
	ck_assert_mem_eq(&(static_msg.msg[90]), &static_keyshare_extension[4],
			 ext_len);

	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if (tls_hs_msg_write_2(msg, 0) == false) {... */
START_TEST (test_write_ext_failure_write_length)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 0);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if ((server_name_len = tls_hs_servername_write(tls, msg)) < 0) {... */
START_TEST (test_write_ext_failure_servername)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_servername_write_failure = true;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 2);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if ((sigalgo_len = write_ext_sigalgo(tls, msg)) < 0) {... */
START_TEST (test_write_ext_failure_sigalgo)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 17);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if ((sigalgo_cert_len = write_ext_sigalgo_cert(tls, msg)) < 0) {... */
START_TEST (test_write_ext_failure_sigalgo_cert)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 3;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 41);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if (elliptic_curves_len < 0) {... */
START_TEST (test_write_ext_failure_elliptic_curves)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_ecc_write_elliptic_curves_failure = true;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 59);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if (ec_point_formats_len < 0) {... */
START_TEST (test_write_ext_failure_ec_point_formats)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_ecc_write_ec_point_formats_failure = true;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 71);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if (supported_versions_len < 0) {... */
START_TEST (test_write_ext_failure_supported_versions)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_supported_versions_write_failure = true;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 77);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

/* if (key_share_len < 0) {... */
START_TEST (test_write_ext_failure_keyshare)
{
	int32_t len;
	bool *sent_flags = global_tls.interim_params->sent_ext_flags;

	tls_hs_keyshare_write_failure = true;
	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	len = write_ext(&global_tls, &static_msg);

	ck_assert_int_eq(len, -1);
	ck_assert_int_eq(static_msg.len, 86);
	ck_assert_int_eq(sent_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SIGNATURE_ALGO_CERT], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_EC_POINT_FORMATS], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(sent_flags[TLS_EXT_KEY_SHARE], false);
	ck_assert_int_eq(sent_flags[TLS_EXT_COOKIE], false);
}
END_TEST

Suite *write_ext_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("write_ext()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_write_ext_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_write_ext_failure_write_length);
	tcase_add_test(tc_limits, test_write_ext_failure_servername);
	tcase_add_test(tc_limits, test_write_ext_failure_sigalgo);
	tcase_add_test(tc_limits, test_write_ext_failure_sigalgo_cert);
	tcase_add_test(tc_limits, test_write_ext_failure_elliptic_curves);
	tcase_add_test(tc_limits, test_write_ext_failure_ec_point_formats);
	tcase_add_test(tc_limits, test_write_ext_failure_supported_versions);
	tcase_add_test(tc_limits, test_write_ext_failure_keyshare);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_chello_compose()
 */
/* normal case */
START_TEST (test_tls_hs_chello_compose_normal)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint16_t client_version;
	uint16_t record_version;
	uint64_t sec = CH_EPOCH_TIME;
	uint8_t cipher[] = {
		CH_CS_TLS_AES_256_GCM_SHA384,
		CH_CS_TLS_AES_128_GCM_SHA256,
		CH_CS_TLS_CHACHA20_POLY1305_SHA256,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA256,
		CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA256,
		CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA,
		CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_RSA_WITH_3DES_EDE_CBC_SHA,
	};
	uint16_t cipher_len;
	uint16_t ext_len;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_CLIENT_HELLO);
	ck_assert_int_eq(msg->len, 203);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&(msg->msg[2]), &sec, 4);
	ck_assert_mem_eq(&(msg->msg[6]), static_session_id, 28);
	ck_assert_uint_eq(msg->msg[34], CH_SESSIONID_LENGTH_0);
	cipher_len = tls_util_read_2(&(static_msg.msg[35]));
	ck_assert_uint_eq(cipher_len, 36);
	ck_assert_mem_eq(&(msg->msg[37]), cipher, 36);
	ck_assert_uint_eq(msg->msg[73], 1);
	ck_assert_uint_eq(msg->msg[74], CH_CM_NULL);
	ext_len = tls_util_read_2(&(static_msg.msg[75]));
	ck_assert_uint_eq(ext_len, 126);
	ck_assert_mem_eq(&(msg->msg[77]), static_extensions, 126);

	client_version = tls_util_convert_protover_to_ver(
		&(global_tls.client_version));
	ck_assert_int_eq(client_version, TLS_VER_TLS12);
	record_version = tls_util_convert_protover_to_ver(
		&(global_tls.record_version));
	ck_assert_int_eq(record_version, TLS_VER_TLS10);
	ck_assert_mem_eq(&(global_tls.client_random[0]), &sec, 4);
	ck_assert_mem_eq(&(global_tls.client_random[4]), static_session_id, 28);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);
}
END_TEST

START_TEST (test_tls_hs_chello_compose_normal_2ndchello)
{
	struct tls_hs_msg *msg;
	uint16_t msg_version;
	uint16_t client_version;
	uint16_t record_version;
	uint64_t sec = CH_EPOCH_TIME;
	uint8_t cipher[] = {
		CH_CS_TLS_AES_256_GCM_SHA384,
		CH_CS_TLS_AES_128_GCM_SHA256,
		CH_CS_TLS_CHACHA20_POLY1305_SHA256,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
		CH_CS_TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256,
		CH_CS_TLS_ECDH_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_RSA_WITH_AES_256_GCM_SHA384,
		CH_CS_TLS_RSA_WITH_AES_128_GCM_SHA256,
		CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA256,
		CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA256,
		CH_CS_TLS_RSA_WITH_AES_256_CBC_SHA,
		CH_CS_TLS_RSA_WITH_AES_128_CBC_SHA,
		CH_CS_TLS_RSA_WITH_3DES_EDE_CBC_SHA,
	};
	uint16_t cipher_len;
	uint16_t ext_len;
	uint8_t msg_buf[1024];
	uint32_t msg_len;
	struct tls_session_param *param;

	global_tls.supported_versions.len = 2;
	global_tls.supported_versions.list = static_supported_versions_list;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_CLIENT_HELLO);
	ck_assert_int_eq(msg->len, 203);
	msg_version = tls_util_read_2(&(msg->msg[0]));
	ck_assert_uint_eq(msg_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&(msg->msg[2]), &sec, 4);
	ck_assert_mem_eq(&(msg->msg[6]), static_session_id, 28);
	ck_assert_uint_eq(msg->msg[34], CH_SESSIONID_LENGTH_0);
	cipher_len = tls_util_read_2(&(static_msg.msg[35]));
	ck_assert_uint_eq(cipher_len, 36);
	ck_assert_mem_eq(&(msg->msg[37]), cipher, 36);
	ck_assert_uint_eq(msg->msg[73], 1);
	ck_assert_uint_eq(msg->msg[74], CH_CM_NULL);
	ext_len = tls_util_read_2(&(static_msg.msg[75]));
	ck_assert_uint_eq(ext_len, 126);
	ck_assert_mem_eq(&(msg->msg[77]), static_extensions, 126);

	client_version = tls_util_convert_protover_to_ver(
		&(global_tls.client_version));
	ck_assert_int_eq(client_version, TLS_VER_TLS12);
	record_version = tls_util_convert_protover_to_ver(
		&(global_tls.record_version));
	ck_assert_int_eq(record_version, TLS_VER_TLS10);
	ck_assert_mem_eq(&(global_tls.client_random[0]), &sec, 4);
	ck_assert_mem_eq(&(global_tls.client_random[4]), static_session_id, 28);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);

	msg_len = msg->len;
	memcpy(msg_buf, msg->msg, msg->len);
	memset(msg->msg, 0, msg->len);
	msg->len = 0;
	msg->type = 0;
	param = global_tls.pending;
	tls_util_get_epochtime_increment = 10;
	tls_util_get_random_need_randomness = true;
	global_tls.state = TLS_STATE_HS_BEFORE_SEND_2NDCHELLO;
	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.record_version));

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_nonnull(msg);
	ck_assert_int_eq(msg->type, TLS_HANDSHAKE_CLIENT_HELLO);
	ck_assert_int_eq(msg->len, msg_len);
	ck_assert_mem_eq(msg->msg, msg_buf, msg_len);

	client_version = tls_util_convert_protover_to_ver(
		&(global_tls.client_version));
	ck_assert_int_eq(client_version, TLS_VER_TLS12);
	record_version = tls_util_convert_protover_to_ver(
		&(global_tls.record_version));
	ck_assert_int_eq(record_version, TLS_VER_TLS12);
	ck_assert_mem_eq(&(global_tls.client_random[0]), &sec, 4);
	ck_assert_mem_eq(&(global_tls.client_random[4]), static_session_id, 28);
	ck_assert_ptr_eq(global_tls.pending, param);
	ck_assert_int_eq(global_tls.pending->session_id_length, 0);
}
END_TEST

/* if ((msg = tls_hs_msg_init()) == NULL) {... */
START_TEST (test_tls_hs_chello_compose_failure_msg_init)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_init_failure = true;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((verlen = write_client_version(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_chello_compose_failure_write_client_version)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 0;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((randlen = write_client_random(tls, msg)) < 0) {...  */
START_TEST (test_tls_hs_chello_compose_failure_write_client_random)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_write_n_failure = true;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((seslen = write_session_id(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_chello_compose_failure_write_session_id)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 0;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((cslen = write_cipher_suite(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_chello_compose_failure_write_cipher_suite)
{
	struct tls_hs_msg *msg;

	tls_cipher_supported_list_failure = true;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((cmplen = write_cmp_method(msg)) < 0) {... */
START_TEST (test_tls_hs_chello_compose_failure_write_cmp_method)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_write_1_failure = true;
	tls_hs_msg_write_1_through_count = 1;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

/* if ((extlen = write_ext(tls, msg)) < 0) {... */
START_TEST (test_tls_hs_chello_compose_failure_write_ext)
{
	struct tls_hs_msg *msg;

	tls_hs_msg_write_2_failure = true;
	tls_hs_msg_write_2_through_count = 1;

	msg = tls_hs_chello_compose(&global_tls);

	ck_assert_ptr_null(msg);
}
END_TEST

Suite *tls_hs_chello_compose_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_chello_compose()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_client, teardown);
	tcase_add_test(tc_core, test_tls_hs_chello_compose_normal);
	tcase_add_test(tc_core, test_tls_hs_chello_compose_normal_2ndchello);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_client, teardown);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_msg_init);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_client_version);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_client_random);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_session_id);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_cipher_suite);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_cmp_method);
	tcase_add_test(tc_limits, test_tls_hs_chello_compose_failure_write_ext);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_client_version()
 */
/* normal case */
START_TEST (test_read_client_version_normal_tls12)
{
	int32_t rc;
	uint16_t version;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];

	msg.len = 2;
	msg.msg = msg_buffer;
	msg.msg[0] = TLS_MAJOR_TLS;
	msg.msg[1] = TLS_MINOR_TLS12;

	rc = read_client_version(&global_tls, &msg, 0);

	ck_assert_int_eq(rc, 2);
	version = tls_util_convert_protover_to_ver(
		&(global_tls.interim_params->version));
	ck_assert_int_eq(version, TLS_VER_TLS12);
	ck_assert_int_eq(global_tls.errnum, 0);
}
END_TEST

/* if (msg->len < (offset + verlen)) {... */
START_TEST (test_read_client_version_failure_record_length)
{
	int32_t rc;
	struct tls_hs_msg msg;

	msg.len = 2;

	rc = read_client_version(&global_tls, &msg, 2);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 3);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* NULL check of pointer args */
/* note: exclude this test case from test suite because need to fix
 *       the target function. */
__attribute__((unused))
START_TEST (test_read_client_version_failure_argument_null)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];

	msg.len = 2;
	msg.msg = msg_buffer;
	msg.msg[0] = TLS_MAJOR_TLS;
	msg.msg[1] = TLS_MINOR_TLS12;

	rc = read_client_version(NULL, &msg, 0);
	ck_assert_int_eq(rc, -1);

	rc = read_client_version(&global_tls, NULL, 0);
	ck_assert_int_eq(rc, -1);
}
END_TEST

Suite *read_client_version_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_client_version()");

	/* read_client_version test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_client_version_normal_tls12);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_client_version_failure_record_length);
	//tcase_add_test(tc_limits, test_read_client_version_failure_argument_null);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_client_random()
 */
/* normal case */
START_TEST (test_read_client_random_normal)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const uint8_t data[] = "abcdefghijklmnopqrstuvwxyz012345";
	const int offset = 2;
	const int randomlen = 32;

	msg.msg = msg_buffer;
	memcpy(&msg.msg[offset], data, randomlen);
	msg.len = offset + randomlen;

	rc = read_client_random(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, randomlen);
	ck_assert_mem_eq(global_tls.interim_params->client_random,
			  data, randomlen);
}
END_TEST

/* if (msg->len < (offset + randlen)) {... */
START_TEST (test_read_client_random_failure_record_length)
{
	int32_t rc;
	struct tls_hs_msg msg;
	const int offset = 2;
	const int randomlen = 32;

	msg.len = randomlen;
	rc = read_client_random(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 5);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* NULL check of pointer args */
/* note: exclude this test case from test suite because need to fix
 *       the target function. */
__attribute__((unused))
START_TEST (test_read_client_random_failure_argument_null)
{
	int32_t rc;
	struct tls_hs_msg msg;
	const int offset = 2;
	const int randomlen = 32;

	msg.len = randomlen;

	rc = read_client_random(NULL, &msg, offset);
	ck_assert_int_eq(rc, -1);

	rc = read_client_random(&global_tls, NULL, offset);
	ck_assert_int_eq(rc, -1);
}
END_TEST

Suite *read_client_random_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_client_random()");

	/* read_client_random test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_client_random_normal);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_client_random_failure_record_length);
	//tcase_add_test(tc_limits, test_read_client_random_failure_argument_null);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_session_id()
 */
/* normal case */
START_TEST (test_read_session_id_normal_new_session)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 34;
	const int seslen = 0;

	msg.msg = msg_buffer;
	msg.msg[offset] = seslen;
	msg.len = offset + 1 + seslen;

	rc = read_session_id(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, 1);
	ck_assert_int_ne(global_tls.resession, true);
	ck_assert_ptr_null(global_tls.pending);

	ck_assert_int_eq(global_tls.interim_params->seslen, seslen);
	ck_assert_ptr_null(global_tls.interim_params->session);
}
END_TEST

/* if (seslen > 0) {...} */
START_TEST (test_read_session_id_normal_resession)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const uint8_t session_id[] = "12345678123456781234567812345678";
	const int seslen = 32;
	const int offset = 34;

	msg.msg = msg_buffer;
	msg.msg[offset] = seslen; //seslen
	memcpy(&msg.msg[offset + 1], session_id, seslen); // session id
	msg.len = offset + 1 + seslen;

	rc = read_session_id(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, 1 + seslen);
	ck_assert_int_ne(global_tls.resession, true);
	ck_assert_ptr_null(global_tls.pending);

	ck_assert_int_eq(global_tls.interim_params->seslen, seslen);
	ck_assert_mem_eq(global_tls.interim_params->session, session_id, seslen);
}
END_TEST

/* if (msg->len < (offset + length_bytes)) {... */
START_TEST (test_read_session_id_failure_record_length_1)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 34;

	msg.msg = msg_buffer;
	msg.len = offset;

	rc = read_session_id(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 6);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (msg->len < (offset + read_bytes + seslen)) {... */
START_TEST (test_read_session_id_failure_record_length_2)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int seslen = 32;
	const int offset = 34;

	msg.msg = msg_buffer;
	msg.msg[offset] = seslen; //seslen
	msg.len = offset + 1;

	rc = read_session_id(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 7);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_session_id()");

	/* read_session_id test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_session_id_normal_new_session);
	tcase_add_test(tc_core, test_read_session_id_normal_resession);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_session_id_failure_record_length_1);
	tcase_add_test(tc_limits, test_read_session_id_failure_record_length_2);
	suite_add_tcase(s, tc_limits);
	return s;
}

/*
 * unit testing for read_cipher_suite()
 */
/* normal case */
START_TEST (test_read_cipher_suite_normal)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 35;
	const uint16_t cslist_bytes = 2; // at least 2

	msg.msg = msg_buffer;
	// set to short
	_util_write_2(&msg.msg[offset], cslist_bytes);
	_util_write_2(&msg.msg[offset + cslist_bytes], TLS_RSA_WITH_AES_128_CBC_SHA256);

	msg.len = offset + 2 + cslist_bytes;

	ck_assert_ptr_nonnull(global_tls.interim_params);
	rc = read_cipher_suite(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, 2 + cslist_bytes);
	ck_assert_int_eq(global_tls.interim_params->cipher_suites->len, 1);
	ck_assert_int_eq(global_tls.interim_params->cipher_suites->list[0],
			 TLS_RSA_WITH_AES_128_CBC_SHA256);
}
END_TEST

/* if (msg->len < (offset + length_bytes)) {... */
START_TEST (test_read_cipher_suite_failure_record_length_1)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 35;
	const uint16_t cslist_bytes = 2; // at least 2

	msg.msg = msg_buffer;
	_util_write_2(&msg.msg[offset], cslist_bytes);

	msg.len = offset + 2;
	msg.len -= 2; // < offset + length_bytes

	rc = read_cipher_suite(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 9);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (msg->len < (offset + read_bytes + cslist_bytes)) {... */
START_TEST (test_read_cipher_suite_failure_record_length_2)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 35;
	const uint16_t cslist_bytes = 2; // at least 2

	msg.msg = msg_buffer;
	_util_write_2(&msg.msg[offset], cslist_bytes);
	_util_write_2(&msg.msg[offset + cslist_bytes], TLS_RSA_WITH_AES_128_CBC_SHA256);

	msg.len = offset + 2 + cslist_bytes;
	msg.len -= 2; // < offset + read_bytes + cslist_bytes

	rc = read_cipher_suite(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 10);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((cipher = tls_cipher_list_alloc(cslist_length)) == NULL) { ... */
START_TEST (test_read_cipher_suite_failure_list_alloc)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 35;
	const uint16_t cslist_bytes = 2; // at least 2

	msg.msg = msg_buffer;
	// set to short
	_util_write_2(&msg.msg[offset], cslist_bytes);
	_util_write_2(&msg.msg[offset + cslist_bytes], TLS_RSA_WITH_AES_128_CBC_SHA256);

	msg.len = offset + 2 + cslist_bytes;

	tls_cipher_list_alloc_failure = true;
	rc = read_cipher_suite(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, 0);
	ck_assert_int_eq(ut_location, 0);
	ck_assert_int_eq(ut_point, 0);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_INTERNAL_ERROR);
}
END_TEST

Suite *read_cipher_suite_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_cipher_suite()");

	/* read_cipher_suite test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_cipher_suite_normal);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_cipher_suite_failure_record_length_1);
	tcase_add_test(tc_limits, test_read_cipher_suite_failure_record_length_2);
	tcase_add_test(tc_limits, test_read_cipher_suite_failure_list_alloc);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for read_cmp_method()
 */
/* normal case */
START_TEST (test_read_cmp_method_normal)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 39;
	const uint8_t list_length = 1;

	msg.msg = msg_buffer;
	msg.msg[offset] = list_length;
	msg.msg[offset + 1] = TLS_COMPRESSION_NULL;
	msg.len = offset + 1 + list_length;
	global_tls.pending = malloc(sizeof(struct tls_session_param));

	rc = read_cmp_method(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, 1 + list_length);
	ck_assert_int_eq(global_tls.interim_params->cmplen, list_length);
	ck_assert_mem_eq(global_tls.interim_params->cmplist,
			 &(msg.msg[offset + 1]), list_length);
}
END_TEST

/* if (msg->len < (offset + length_bytes)) {... */
START_TEST (test_read_cmp_method_failure_record_length_1)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 39;

	msg.msg = msg_buffer;
	msg.len = offset + 1;
	msg.len -= 1;

	rc = read_cmp_method(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 12);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (msg->len < (offset + read_bytes + list_length)) {... */
START_TEST (test_read_cmp_method_failure_record_length_2)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 39;
	const uint8_t list_length = 1;

	msg.msg = msg_buffer;
	msg.msg[offset] = list_length;
	msg.msg[offset + 1] = TLS_COMPRESSION_NULL;
	msg.len = offset + 1 + list_length;
	msg.len -= 1;

	rc = read_cmp_method(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, -1);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 13);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

Suite *read_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_cmp_method()");

	/* read_cmp_method test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_cmp_method_failure_record_length_1);
	tcase_add_test(tc_limits, test_read_cmp_method_failure_record_length_2);
	suite_add_tcase(s, tc_limits);
	return s;
}

/*
 * unit testing for check_ext_availability_tls12().
 */
/* normal case */
START_TEST (test_check_ext_availability_tls12_normal)
{
	bool rc;

	rc = check_ext_availability_tls12(TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls12(TLS_EXT_ELLIPTIC_CURVES);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls12(TLS_EXT_EC_POINT_FORMATS);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls12(TLS_EXT_SIGNATURE_ALGO);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_check_ext_availability_tls12_failure)
{
	bool rc;

	rc = check_ext_availability_tls12(TLS_EXT_MAX_FRAGMENT_LENGTH);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_STATUS_REQUEST);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_USE_SRTP);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_HEARTBEAT);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_APP_LAYER_PROTO_NEGOTIATION);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_CLIENT_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_SERVER_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_PADDING);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_PRE_SHARED_KEY);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_EARLY_DATA);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_COOKIE);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_PSK_KEY_EXCHANGE_MODES);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_CERTIFICATE_AUTHORITIES);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_OID_FILTERS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_POST_HANDSHAKE_AUTH);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_SIGNATURE_ALGO_CERT);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls12(TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *check_ext_availability_tls12_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("check_ext_availability_tls12()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_test(tc_core, test_check_ext_availability_tls12_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_test(tc_limits, test_check_ext_availability_tls12_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for check_ext_availability_tls13().
 */
/* normal case */
START_TEST (test_check_ext_availability_tls13_normal)
{
	bool rc;

	rc = check_ext_availability_tls13(TLS_EXT_SERVER_NAME);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_MAX_FRAGMENT_LENGTH);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_STATUS_REQUEST);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SUPPORTED_GROUPS);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SIGNATURE_ALGO);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_USE_SRTP);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_HEARTBEAT);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_APP_LAYER_PROTO_NEGOTIATION);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_CLIENT_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SERVER_CERTIFICATE_TYPE);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_PADDING);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_PRE_SHARED_KEY);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_EARLY_DATA);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_COOKIE);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_PSK_KEY_EXCHANGE_MODES);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_CERTIFICATE_AUTHORITIES);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_POST_HANDSHAKE_AUTH);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_SIGNATURE_ALGO_CERT);
	ck_assert_int_eq(rc, true);

	rc = check_ext_availability_tls13(TLS_EXT_KEY_SHARE);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_check_ext_availability_tls13_failure)
{
	bool rc;

	rc = check_ext_availability_tls13(TLS_EXT_EC_POINT_FORMATS);
	ck_assert_int_eq(rc, false);

	rc = check_ext_availability_tls13(TLS_EXT_OID_FILTERS);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *check_ext_availability_tls13_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("check_ext_availability_tls13()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_test(tc_core, test_check_ext_availability_tls13_normal);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits = tcase_create("Limits");
	tcase_add_test(tc_limits, test_check_ext_availability_tls13_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for (tls_hs_extension_parse)
 */
/* normal case */
START_TEST (test_tls_hs_extension_parse_normal)
{
	int32_t rc;
	uint8_t msg_buffer[512];
	struct tls_hs_msg msg;
	const int offset = 0;
	struct tls_extension *ext, *ext_next;

	msg.msg = msg_buffer;
	msg.len = 2 + sizeof(static_client_hello_extension); // total extension length
	tls_util_write_2(msg.msg, sizeof(static_client_hello_extension));
	memcpy(&(msg.msg[2]), static_client_hello_extension, msg.len);

	rc = tls_hs_extension_parse(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, msg.len);

	ext = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext);
	ck_assert_uint_eq(ext->type, TLS_EXT_SIGNATURE_ALGO);
	ck_assert_uint_eq(ext->len, 30);

	ext_next = TAILQ_NEXT(ext, link);
	ck_assert_ptr_nonnull(ext_next);
	ext = ext_next;
	ck_assert_uint_eq(ext->type, TLS_EXT_ELLIPTIC_CURVES);
	ck_assert_uint_eq(ext->len, 4);

	ext_next = TAILQ_NEXT(ext, link);
	ck_assert_ptr_nonnull(ext_next);
	ext = ext_next;
	ck_assert_uint_eq(ext->type, TLS_EXT_EC_POINT_FORMATS);
	ck_assert_uint_eq(ext->len, 5);

	ext_next = TAILQ_NEXT(ext, link);
	ck_assert_ptr_null(ext_next);
}
END_TEST

/* if (msg->len == offset) {... */
START_TEST (test_tls_hs_extension_parse_normal_no_extesion_provided)
{
	int32_t rc;
	struct tls_hs_msg msg;
	uint8_t msg_buffer[512];
	const int offset = 41;

	msg.msg = msg_buffer;
	msg.len = offset;

	rc = tls_hs_extension_parse(&global_tls, &msg, offset);
	ck_assert_int_eq(rc, 0);
}
END_TEST

Suite *tls_hs_extension_parse_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("tls_hs_extension_parse()");

	/* tls_hs_extension_parse test case */
	tc_core = tcase_create("tls_hs_extension_parse");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal);
	tcase_add_test(tc_core, test_tls_hs_extension_parse_normal_no_extesion_provided);
	suite_add_tcase(s, tc_core);
	return s;
}

/*
 * unit testing for (read_ext_list)
 */
/* normal case */
START_TEST (test_read_ext_list_normal_elliptic_curves)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_ELLIPTIC_CURVES;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_read_ext_list_failure_elliptic_curves)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_ELLIPTIC_CURVES;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	tls_hs_ecc_read_elliptic_curves_false = true;
	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, false);
}
END_TEST

START_TEST (test_read_ext_list_normal_ec_point_formats)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_EC_POINT_FORMATS;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_read_ext_list_failure_ec_point_formats)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_EC_POINT_FORMATS;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	tls_hs_ecc_read_point_format_false = true;
	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, false);
}
END_TEST

START_TEST (test_read_ext_list_normal_ext_signature_algo)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SIGNATURE_ALGO;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_read_ext_list_failure_ext_signature_algo)
{
	bool rc;
	const enum tls_extension_type type = TLS_EXT_SIGNATURE_ALGO;
	struct tls_hs_msg msg;
	msg.msg = static_client_hello_extension;
	msg.len = sizeof(static_client_hello_extension);
	const uint32_t offset = 0;

	tls_hs_sighash_read_false = true;
	rc = read_ext_list(&global_tls, type, &msg, offset);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *read_ext_list_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("read_ext_list()");

	/* read_ext_list test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_read_ext_list_normal_elliptic_curves);
	tcase_add_test(tc_core, test_read_ext_list_normal_ec_point_formats);
	tcase_add_test(tc_core, test_read_ext_list_normal_ext_signature_algo);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_read_ext_list_failure_elliptic_curves);
	tcase_add_test(tc_limits, test_read_ext_list_failure_ec_point_formats);
	tcase_add_test(tc_limits, test_read_ext_list_failure_ext_signature_algo);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_chello_parse().
 */
/* normal case */
START_TEST (test_tls_hs_chello_parse_normal_tls12)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	ck_assert_int_eq(global_tls.client_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.negotiated_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.record_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.interim_params->version.major, TLS_MAJOR_TLS);

	ck_assert_int_eq(global_tls.client_version.minor, TLS_MINOR_MAX);
	ck_assert_int_eq(global_tls.negotiated_version.minor, TLS_MINOR_MAX);
	ck_assert_int_eq(global_tls.record_version.minor, TLS_MINOR_MAX);
	ck_assert_int_eq(global_tls.interim_params->version.minor, TLS_MINOR_TLS12);

	ck_assert_mem_eq(global_tls.interim_params->client_random,
			 &static_client_hello_tls12[2], 32);

	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_null(global_tls.pending);
	ck_assert_int_eq(global_tls.interim_params->seslen, 0);
	ck_assert_ptr_null(global_tls.interim_params->session);

	ck_assert_int_eq(global_tls.interim_params->cipher_suites->len, 1);
	ck_assert_int_eq(global_tls.interim_params->cipher_suites->list[0],
			 TLS_RSA_WITH_AES_128_CBC_SHA);

	ck_assert_int_eq(global_tls.interim_params->cmplen, 1);
	ck_assert_ptr_nonnull(global_tls.interim_params->cmplist);
	ck_assert_int_eq(global_tls.interim_params->cmplist[0], TLS_COMPRESSION_NULL);

	ck_assert_ptr_null(global_tls.sighash_list);

	struct tls_extension *ext1, *ext2;
	ext1 = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SIGNATURE_ALGO);
	ck_assert_uint_eq(ext1->len, 28);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_ELLIPTIC_CURVES);
	ck_assert_uint_eq(ext1->len, 4);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_EC_POINT_FORMATS);
	ck_assert_uint_eq(ext1->len, 2);

	ck_assert_int_eq(global_tls.errnum, 0);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_normal_tls13)
{
	struct tls_hs_msg msg;
	uint16_t version;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	ck_assert_int_eq(global_tls.client_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.client_version.minor, TLS_MINOR_MAX);

	ck_assert_int_eq(global_tls.negotiated_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.negotiated_version.minor, TLS_MINOR_MAX);

	ck_assert_int_eq(global_tls.record_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.record_version.minor, TLS_MINOR_MAX);

	version = tls_util_convert_protover_to_ver(
		&(global_tls.interim_params->version));
	ck_assert_uint_eq(version, TLS_VER_TLS12);

	ck_assert_mem_eq(global_tls.interim_params->client_random,
			 &static_client_hello_tls13[2], 32);

	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_null(global_tls.pending);
	ck_assert_int_eq(global_tls.interim_params->seslen, 0);
	ck_assert_ptr_null(global_tls.interim_params->session);

	ck_assert_int_eq(global_tls.interim_params->cipher_suites->len, 3);
	ck_assert_int_eq(global_tls.interim_params->cipher_suites->list[0],
			 TLS_AES_128_GCM_SHA256);

	ck_assert_int_eq(global_tls.interim_params->cmplen, 1);
	ck_assert_ptr_nonnull(global_tls.interim_params->cmplist);
	ck_assert_int_eq(global_tls.interim_params->cmplist[0], TLS_COMPRESSION_NULL);

	ck_assert_ptr_null(global_tls.sighash_list);

	struct tls_extension *ext1, *ext2;
	ext1 = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SERVER_NAME);
	ck_assert_uint_eq(ext1->len, 11);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, 0xff01); // renegotiation_info
	ck_assert_uint_eq(ext1->len, 1);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SUPPORTED_GROUPS);
	ck_assert_uint_eq(ext1->len, 20);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, 0x23); // session_ticket
	ck_assert_uint_eq(ext1->len, 0);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_KEY_SHARE);
	ck_assert_uint_eq(ext1->len, 38);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_uint_eq(ext1->len, 3);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SIGNATURE_ALGO);
	ck_assert_uint_eq(ext1->len, 32);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_PSK_KEY_EXCHANGE_MODES);
	ck_assert_uint_eq(ext1->len, 2);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, 0x1c); // record_size_limit
	ck_assert_uint_eq(ext1->len, 2);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_null(ext1);

	ck_assert_int_eq(global_tls.errnum, 0);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_normal_tls13_2ndchello)
{
	struct tls_hs_msg msg;
	uint16_t version;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_2nd_client_hello_tls13;
	msg.len = sizeof(static_2nd_client_hello_tls13);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	ck_assert_int_eq(global_tls.client_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.client_version.minor, TLS_MINOR_MAX);

	ck_assert_int_eq(global_tls.negotiated_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.negotiated_version.minor, TLS_MINOR_MAX);

	ck_assert_int_eq(global_tls.record_version.major, TLS_MAJOR_MAX);
	ck_assert_int_eq(global_tls.record_version.minor, TLS_MINOR_MAX);

	version = tls_util_convert_protover_to_ver(
		&(global_tls.interim_params->version));
	ck_assert_uint_eq(version, TLS_VER_TLS12);

	ck_assert_mem_eq(global_tls.interim_params->client_random,
			 &static_2nd_client_hello_tls13[2], 32);

	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_null(global_tls.pending);
	ck_assert_int_eq(global_tls.interim_params->seslen, 0);
	ck_assert_ptr_null(global_tls.interim_params->session);

	ck_assert_int_eq(global_tls.interim_params->cipher_suites->len, 3);
	ck_assert_int_eq(global_tls.interim_params->cipher_suites->list[0],
			 TLS_AES_128_GCM_SHA256);

	ck_assert_int_eq(global_tls.interim_params->cmplen, 1);
	ck_assert_ptr_nonnull(global_tls.interim_params->cmplist);
	ck_assert_int_eq(global_tls.interim_params->cmplist[0], TLS_COMPRESSION_NULL);

	ck_assert_ptr_null(global_tls.sighash_list);

	struct tls_extension *ext1, *ext2;
	ext1 = TAILQ_FIRST(&(global_tls.interim_params->head));
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SERVER_NAME);
	ck_assert_uint_eq(ext1->len, 11);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, 0xff01); // renegotiation_info
	ck_assert_uint_eq(ext1->len, 1);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SUPPORTED_GROUPS);
	ck_assert_uint_eq(ext1->len, 8);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_KEY_SHARE);
	ck_assert_uint_eq(ext1->len, 71);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SUPPORTED_VERSIONS);
	ck_assert_uint_eq(ext1->len, 3);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_SIGNATURE_ALGO);
	ck_assert_uint_eq(ext1->len, 32);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_COOKIE);
	ck_assert_uint_eq(ext1->len, 116);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_PSK_KEY_EXCHANGE_MODES);
	ck_assert_uint_eq(ext1->len, 2);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, 0x1c); // record_size_limit
	ck_assert_uint_eq(ext1->len, 2);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_nonnull(ext1);

	ck_assert_int_eq(ext1->type, TLS_EXT_PADDING);
	ck_assert_uint_eq(ext1->len, 175);

	ext2 = ext1;
	ext1 = TAILQ_NEXT(ext2, link);
	ck_assert_ptr_null(ext1);

	ck_assert_int_eq(global_tls.errnum, 0);
}
END_TEST

/* NULL check of pointer args */
/* note: exclude this test case from test suite because need to fix
 *       the target function. */
__attribute__((unused))
START_TEST (test_tls_hs_chello_parse_failure_args_null)
{
	TLS *tls = NULL;
	struct tls_hs_msg *msg = NULL;
	bool rc = tls_hs_chello_parse(tls, msg);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if ((randlen = read_client_random(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_chello_parse_failure_client_random)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_too_short;
	msg.len = sizeof(static_client_hello_too_short);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 5);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((seslen = read_session_id(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_chello_parse_failure_session_id)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12_session_error;
	msg.len = sizeof(static_client_hello_tls12_session_error);

	tls_session_new_failure = true;
	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 7);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((cslen = read_cipher_suite(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_chello_parse_failure_cipher_suite)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12_cipher_suite_error;
	msg.len = sizeof(static_client_hello_tls12_cipher_suite_error);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 10);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((cmplen = read_cmp_method(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_chello_parse_failure_cmp_method)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12_cmp_method_error;
	msg.len = sizeof(static_client_hello_tls12_cmp_method_error);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 13);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((extlen = tls_hs_extension_parse(tls, msg, offset)) < 0) {... */
START_TEST (test_tls_hs_chello_parse_failure_tls_hs_extension_parse)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12_ext_error;
	msg.len = sizeof(static_client_hello_tls12_ext_error);

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if (offset != msg->len) {... */
START_TEST (test_tls_hs_chello_parse_failure_length_mismatch)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12) + 1; /* wrong value */

	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_INVALID_RECORD_LENGTH);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 17);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_DECODE_ERROR);
}
END_TEST

/* if ((tls->sighash_list == NULL) && ... */
START_TEST (test_tls_hs_chello_parse_normal_sighash_list_nonnull)
{
	struct tls_hs_msg msg;
	struct tls_hs_sighash_list sighash;
	global_tls.sighash_list = &sighash; /* */

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12_no_extension;
	msg.len = sizeof(static_client_hello_tls12_no_extension);

	fprintf(stderr, "### from %s\n", __func__);
	bool rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_1)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_HELLO_REQUEST;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_2)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_SERVER_HELLO;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_3)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_CERTIFICATE;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_4)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_SERVER_KEY_EXCHANGE;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_5)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_CERTIFICATE_REQUEST;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_6)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_SERVER_HELLO_DONE;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_7)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_CERTIFICATE_VERIFY;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_8)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_CLIENT_KEY_EXCHANGE;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_failure_type_9)
{
	struct tls_hs_msg msg;
	bool rc;

	msg.type = TLS_HANDSHAKE_FINISHED;
	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_error, ERR_ST_TLS_UNEXPECTED_MSG);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 16);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
}
END_TEST

#if 0
START_TEST (test_tls_hs_chello_parse_type)
{
	struct tls_hs_msg msg;
	enum tls_alert_desc alert_desc;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;

	error == ERR_ST_TLS_UNEXPECTED_MSG;
	location = ERR_LC_TLS2;
	point = ERR_PT_TLS_HS_MSG_CHELLO + 16;
	alert = TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
	bool rc = tls_hs_chello_parse(&tls, &msg);
	ck_assert_int_eq(rc, false);
}
END_TEST

START_TEST (test_tls_hs_chello_parse_type)
{
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.len = offset + verlen;
	msg.msg[offset + 0] = ;
	msg.msg[offset + 1] = ;


	bool rc = tls_hs_chello_parse(&tls, &msg);
	ck_assert_int_eq(rc, false);
}
END_TEST
#endif

Suite *tls_hs_chello_parse_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits1;
	TCase *tc_limits2;

	s = suite_create("tls_hs_chello_parse()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_tls_hs_chello_parse_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_chello_parse_normal_tls13);
	tcase_add_test(tc_core, test_tls_hs_chello_parse_normal_tls13_2ndchello);
	tcase_add_test(tc_core, test_tls_hs_chello_parse_normal_sighash_list_nonnull);
	suite_add_tcase(s, tc_core);

	/* Limits test case */
	tc_limits1 = tcase_create("Limits1");
	tcase_add_checked_fixture(tc_limits1, setup_server, teardown_tls);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_1);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_2);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_3);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_4);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_5);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_6);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_7);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_8);
	tcase_add_test(tc_limits1, test_tls_hs_chello_parse_failure_type_9);
	suite_add_tcase(s, tc_limits1);

	tc_limits2 = tcase_create("Limits2");
	tcase_add_checked_fixture(tc_limits2, setup_server, teardown_tls);
	//tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_args_null);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_client_random);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_session_id);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_cipher_suite);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_cmp_method);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_tls_hs_extension_parse);
	tcase_add_test(tc_limits2, test_tls_hs_chello_parse_failure_length_mismatch);
	suite_add_tcase(s, tc_limits2);

	return s;
}

/*
 * unit testing for interpret_random()
 */
/* normal case */
START_TEST (test_interpret_random_normal)
{
	bool rc;

	static_msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	static_msg.msg = static_client_hello_common;
	static_msg.len = static_client_hello_common_length;

	rc = tls_hs_chello_parse(&global_tls, &static_msg);

	ck_assert_int_eq(rc, true);

	rc = interpret_random(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_mem_eq(&global_tls.client_random,
			 &static_client_hello_common[2], 32);
}
END_TEST

Suite *interpret_random_suite(void)
{
	Suite *s;
	TCase *tc_core;

	s = suite_create("interpret_random()");

	/* Core test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_interpret_random_normal);
	suite_add_tcase(s, tc_core);

	return s;
}

/*
 * unit testing for interpret_session_id()
 */
/* normal case */
START_TEST (test_interpret_session_id_normal_tls12_new_session)
{
	int32_t rc;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;

	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_ne(global_tls.resession, true);
	ck_assert_ptr_nonnull(global_tls.pending);
}
END_TEST

/* if (seslen > 0) {...} */
START_TEST (test_interpret_session_id_normal_tls12_resession)
{
	int32_t rc;
	const uint8_t session_id[] = "12345678123456781234567812345678";
	const int seslen = 32;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = seslen;
	global_tls.interim_params->session = (uint8_t *)session_id;

	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.resession, true);
	ck_assert_ptr_nonnull(global_tls.pending);

	global_tls.interim_params->session = NULL;
}
END_TEST

START_TEST (test_interpret_session_id_normal_tls13_new_session)
{
	int32_t rc;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;

	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
}
END_TEST

START_TEST (test_interpret_session_id_normal_tls13_compat)
{
	int32_t rc;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 32;
	global_tls.interim_params->session = static_compat_session_id;

	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
	ck_assert_uint_eq(global_tls.interim_params->seslen, 32);
	ck_assert_ptr_eq(global_tls.interim_params->session,
			 static_compat_session_id);

	global_tls.interim_params->session = NULL;
}
END_TEST

/* ...} else if ((tls->pending = tls_session_new()) == NULL) {... */
START_TEST (test_interpret_session_id_failure_tls12_get_session)
{
	int32_t rc;

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;

	tls_session_new_failure = true;
	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_SESSION);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 8);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_INTERNAL_ERROR);
}
END_TEST

/* if (tls->interim_params->seslen != 0 ... */
START_TEST (test_interpret_session_id_failure_tls13_seslen)
{
	int32_t rc;
	const uint8_t session_id[] = "abcdefghijklmnopqrstuvwxyz";

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 26;
	global_tls.interim_params->session = (uint8_t *)session_id;

	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_ILLEGAL_PARAMETER);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 18);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);

	global_tls.interim_params->session = NULL;
}
END_TEST

/* if ((tls->pending = tls_session_new()) == NULL) {... */
START_TEST (test_interpret_session_id_failure_tls13_get_session)
{
	int32_t rc;

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	global_tls.interim_params->seslen = 0;
	global_tls.interim_params->session = NULL;

	tls_session_new_failure = true;
	rc = interpret_session_id(&global_tls);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_GET_SESSION);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 19);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_INTERNAL_ERROR);
}
END_TEST

Suite *interpret_session_id_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_session_id()");

	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12_new_session);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls12_resession);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls13_new_session);
	tcase_add_test(tc_core, test_interpret_session_id_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_interpret_session_id_failure_tls12_get_session);
	tcase_add_test(tc_limits, test_interpret_session_id_failure_tls13_seslen);
	tcase_add_test(tc_limits, test_interpret_session_id_failure_tls13_get_session);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_cipher()
 */
/* normal case */
START_TEST (test_interpret_cipher_normal)
{
	bool rc;
	struct tls_hs_msg msg;
	struct tls_hs_interim_params *params;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_common;
	msg.len = static_client_hello_common_length;

	rc = tls_hs_chello_parse(&global_tls, &msg);

	params = global_tls.interim_params;

	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(params->cipher_suites->len, 1);

	rc = interpret_cipher(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_ptr_ne(global_tls.chello_cipher_suites, params->cipher_suites);
	ck_assert_mem_eq(global_tls.chello_cipher_suites->list,
			 params->cipher_suites->list,
			 params->cipher_suites->len * sizeof(enum tls_cipher_suite));
}
END_TEST

START_TEST (test_interpret_cipher_failure)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_common;
	msg.len = static_client_hello_common_length;

	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_cipher_list_alloc_failure = true;

	rc = interpret_cipher(&global_tls);

	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_INTERNAL_ERROR);
}
END_TEST

Suite *interpret_cipher_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_cipher()");

	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_interpret_cipher_normal);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_interpret_cipher_failure);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_cmp_method()
 */
/* normal case */
START_TEST (test_interpret_cmp_method_normal)
{
	int32_t rc;
	uint8_t cmp_methods[1];

	cmp_methods[0] = TLS_COMPRESSION_NULL;
	global_tls.interim_params->cmplen = 1;
	global_tls.interim_params->cmplist = cmp_methods;
	global_tls.pending = malloc(sizeof(struct tls_session_param));

	rc = interpret_cmp_method(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);

	global_tls.interim_params->cmplist = NULL;
}
END_TEST

/* switch (version) { ... */
START_TEST (test_interpret_cmp_method_failure_cmplen)
{
	int32_t rc;
	uint8_t cmp_methods[2];

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	cmp_methods[0] = TLS_COMPRESSION_NULL;
	cmp_methods[1] = 1; // unknown compression method
	global_tls.interim_params->cmplen = 2;
	global_tls.interim_params->cmplist = cmp_methods;
	global_tls.pending = malloc(sizeof(struct tls_session_param));

	rc = interpret_cmp_method(&global_tls);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_ILLEGAL_PARAMETER);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 20);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);

	global_tls.interim_params->cmplist = NULL;
}
END_TEST

/* ... */
START_TEST (test_interpret_cmp_method_failure_unknown_method)
{
	int32_t rc;
	uint8_t cmp_methods[1];

	cmp_methods[0] = 1; // unknown compression method
	global_tls.interim_params->cmplen = 1;
	global_tls.interim_params->cmplist = cmp_methods;
	global_tls.pending = malloc(sizeof(struct tls_session_param));

	rc = interpret_cmp_method(&global_tls);
	ck_assert_int_eq(rc, false);
	ck_assert_int_eq(global_tls.errnum, 0);

	ck_assert_int_eq(ut_error, ERR_ST_TLS_ILLEGAL_PARAMETER);
	ck_assert_int_eq(ut_location, ERR_LC_TLS2);
	ck_assert_int_eq(ut_point, ERR_PT_TLS_HS_MSG_CHELLO + 21);
	ck_assert_int_eq(ut_alert_desc, TLS_ALERT_DESC_ILLEGAL_PARAMETER);

	global_tls.interim_params->cmplist = NULL;
}
END_TEST

Suite *interpret_cmp_method_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_cmp_method()");

	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_interpret_cmp_method_normal);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_interpret_cmp_method_failure_cmplen);
	tcase_add_test(tc_limits, test_interpret_cmp_method_failure_unknown_method);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for interpret_ext_list()
 */
/* normal case */
START_TEST (test_interpret_ext_list_normal_tls12)
{
	bool rc;
	struct tls_hs_msg msg;
	bool *recv_flags = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	recv_flags = global_tls.interim_params->recv_ext_flags;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_EC_POINT_FORMATS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13)
{
	bool rc;
	struct tls_hs_msg msg;
	bool *recv_flags = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	recv_flags = global_tls.interim_params->recv_ext_flags;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13_2ndchello)
{
	bool rc;
	struct tls_hs_msg msg;
	bool *recv_flags = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_2nd_client_hello_tls13;
	msg.len = sizeof(static_2nd_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	recv_flags = global_tls.interim_params->recv_ext_flags;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_COOKIE], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);
}
END_TEST

START_TEST (test_interpret_ext_list_normal_tls13_compat)
{
	bool rc;
	struct tls_hs_msg msg;
	bool *recv_flags = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13_compat;
	msg.len = sizeof(static_client_hello_tls13_compat);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	recv_flags = global_tls.interim_params->recv_ext_flags;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls12_elliptic_curves)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	tls_hs_ecc_read_elliptic_curves_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls13_elliptic_curves)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_ecc_read_elliptic_curves_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls12_ec_point_formats)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	tls_hs_ecc_read_point_format_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls12_ext_signature_algo)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	tls_hs_sighash_read_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls13_ext_signature_algo)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_sighash_read_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls13_servername)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_servername_read_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! read_ext_list(tls, ext->type, &msg, 0)) {... */
START_TEST (test_interpret_ext_list_failure_tls13_keyshare)
{
	bool rc;
	struct tls_hs_msg msg;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	rc = tls_hs_chello_parse(&global_tls, &msg);
	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_keyshare_read_false = true;
	rc = interpret_ext_list(&global_tls);
	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *interpret_ext_list_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("interpret_ext_list()");

	/* read_client_version test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls12);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13_2ndchello);
	tcase_add_test(tc_core, test_interpret_ext_list_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls12_elliptic_curves);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_elliptic_curves);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls12_ec_point_formats);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls12_ext_signature_algo);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_ext_signature_algo);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_servername);
	tcase_add_test(tc_limits, test_interpret_ext_list_failure_tls13_keyshare);
	suite_add_tcase(s, tc_limits);

	return s;
}

/*
 * unit testing for tls_hs_chello_interpret()
 */
/* normal case */
START_TEST (test_tls_hs_chello_interpret_normal_tls12)
{
	bool rc;
	struct tls_hs_msg msg;
	struct tls_hs_interim_params *params;
	bool *recv_flags = NULL;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	fprintf(stderr, "### from %s\n", __func__);
	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	params = global_tls.interim_params;
	recv_flags = params->recv_ext_flags;

	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_mem_eq(global_tls.client_random,
			 &(static_client_hello_tls12[2]), 32);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
	ck_assert_ptr_ne(global_tls.chello_cipher_suites, params->cipher_suites);
	ck_assert_mem_eq(global_tls.chello_cipher_suites->list,
			 params->cipher_suites->list,
			 params->cipher_suites->len * sizeof(enum tls_cipher_suite));
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_EC_POINT_FORMATS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
}
END_TEST

START_TEST (test_tls_hs_chello_interpret_normal_tls13)
{
	bool rc;
	struct tls_hs_msg msg;
	struct tls_hs_interim_params *params;
	bool *recv_flags = NULL;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13;
	msg.len = sizeof(static_client_hello_tls13);

	fprintf(stderr, "### from %s\n", __func__);
	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_sighash_read_set_sighash_list = true;
	tls_hs_sighash_read_set_sighash_list_cert = true;
	params = global_tls.interim_params;
	recv_flags = params->recv_ext_flags;

	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_mem_eq(global_tls.client_random,
			 &(static_client_hello_tls13[2]), 32);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
	ck_assert_ptr_ne(global_tls.chello_cipher_suites, params->cipher_suites);
	ck_assert_mem_eq(global_tls.chello_cipher_suites->list,
			 params->cipher_suites->list,
			 params->cipher_suites->len * sizeof(enum tls_cipher_suite));
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);

	global_tls.sighash_list = NULL;
	global_tls.sighash_list_cert = NULL;
}
END_TEST

START_TEST (test_tls_hs_chello_interpret_normal_tls13_2ndchello)
{
	bool rc;
	struct tls_hs_msg msg;
	struct tls_hs_interim_params *params;
	bool *recv_flags = NULL;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_2nd_client_hello_tls13;
	msg.len = sizeof(static_2nd_client_hello_tls13);

	fprintf(stderr, "### from %s\n", __func__);
	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_sighash_read_set_sighash_list = true;
	tls_hs_sighash_read_set_sighash_list_cert = true;
	params = global_tls.interim_params;
	recv_flags = params->recv_ext_flags;

	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_mem_eq(global_tls.client_random,
			 &(static_2nd_client_hello_tls13[2]), 32);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
	ck_assert_ptr_ne(global_tls.chello_cipher_suites, params->cipher_suites);
	ck_assert_mem_eq(global_tls.chello_cipher_suites->list,
			 params->cipher_suites->list,
			 params->cipher_suites->len * sizeof(enum tls_cipher_suite));
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_COOKIE], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);

	global_tls.sighash_list = NULL;
	global_tls.sighash_list_cert = NULL;
}
END_TEST

START_TEST (test_tls_hs_chello_interpret_normal_tls13_compat)
{
	bool rc;
	struct tls_hs_msg msg;
	struct tls_hs_interim_params *params;
	bool *recv_flags = NULL;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls13_compat;
	msg.len = sizeof(static_client_hello_tls13_compat);

	fprintf(stderr, "### from %s\n", __func__);
	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS13,
					 &(global_tls.negotiated_version));
	tls_hs_sighash_read_set_sighash_list = true;
	tls_hs_sighash_read_set_sighash_list_cert = true;
	params = global_tls.interim_params;
	recv_flags = params->recv_ext_flags;

	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, true);
	ck_assert_mem_eq(global_tls.client_random,
			 &(static_client_hello_tls13_compat[2]), 32);
	ck_assert_int_eq(global_tls.resession, false);
	ck_assert_ptr_nonnull(global_tls.pending);
	ck_assert_ptr_ne(global_tls.chello_cipher_suites, params->cipher_suites);
	ck_assert_mem_eq(global_tls.chello_cipher_suites->list,
			 params->cipher_suites->list,
			 params->cipher_suites->len * sizeof(enum tls_cipher_suite));
	ck_assert_int_eq(global_tls.pending->compression_algorithm,
			 TLS_COMPRESSION_NULL);
	ck_assert_int_eq(recv_flags[TLS_EXT_SERVER_NAME], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_ELLIPTIC_CURVES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SIGNATURE_ALGO], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_SUPPORTED_VERSIONS], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_PSK_KEY_EXCHANGE_MODES], true);
	ck_assert_int_eq(recv_flags[TLS_EXT_KEY_SHARE], true);

	global_tls.sighash_list = NULL;
	global_tls.sighash_list_cert = NULL;
}
END_TEST

/* if (! interpret_session_id(tls)) {... */
START_TEST (test_tls_hs_chello_interpret_failure_interpret_session_id)
{
	bool rc;
	struct tls_hs_msg msg;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_session_new_failure = true;
	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! interpret_cmp_method(tls)) {... */
START_TEST (test_tls_hs_chello_interpret_failure_interpret_cmp_method)
{
	bool rc;
	struct tls_hs_msg msg;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_compress_set_failure = true;
	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/* if (! interpret_ext_list(tls)) {... */
START_TEST (test_tls_hs_chello_interpret_failure_interpret_ext_list)
{
	bool rc;
	struct tls_hs_msg msg;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	rc = tls_hs_chello_parse(&global_tls, &msg);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	tls_hs_ecc_read_point_format_false = true;
	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

/*
    if ((tls->sighash_list == NULL) &&
        (tls_hs_sighash_save_default(tls) == false)) {...
*/
START_TEST (test_tls_hs_chello_interpret_failure_sighash_save_default)
{
	bool rc;
	struct tls_hs_msg msg;

	global_tls.sighash_list = NULL;

	msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
	msg.msg = static_client_hello_tls12;
	msg.len = sizeof(static_client_hello_tls12);

	tls_hs_sighash_save_default_failure = true;
	fprintf(stderr, "### from %s\n", __func__);

	rc = tls_hs_chello_parse(&global_tls, &msg);

	ck_assert_int_eq(rc, true);

	tls_util_convert_ver_to_protover(TLS_VER_TLS12,
					 &(global_tls.negotiated_version));
	rc = tls_hs_chello_interpret(&global_tls);

	ck_assert_int_eq(rc, false);
}
END_TEST

Suite *tls_hs_chello_interpret_suite(void)
{
	Suite *s;
	TCase *tc_core;
	TCase *tc_limits;

	s = suite_create("tls_hs_chello_interpret()");

	/* read_client_version test case */
	tc_core = tcase_create("Core");
	tcase_add_checked_fixture(tc_core, setup_server, teardown_tls);
	tcase_add_test(tc_core, test_tls_hs_chello_interpret_normal_tls12);
	tcase_add_test(tc_core, test_tls_hs_chello_interpret_normal_tls13);
	tcase_add_test(tc_core, test_tls_hs_chello_interpret_normal_tls13_2ndchello);
	tcase_add_test(tc_core, test_tls_hs_chello_interpret_normal_tls13_compat);
	suite_add_tcase(s, tc_core);

	tc_limits = tcase_create("Limits");
	tcase_add_checked_fixture(tc_limits, setup_server, teardown_tls);
	tcase_add_test(tc_limits, test_tls_hs_chello_interpret_failure_interpret_session_id);
	tcase_add_test(tc_limits, test_tls_hs_chello_interpret_failure_interpret_cmp_method);
	tcase_add_test(tc_limits, test_tls_hs_chello_interpret_failure_interpret_ext_list);
	tcase_add_test(tc_limits, test_tls_hs_chello_interpret_failure_sighash_save_default);
	suite_add_tcase(s, tc_limits);

	return s;
}

/**
 * test name (src file) suite (dummy).
 */
Suite *chello_suite(void)
{
	Suite *s;

	return s = suite_create("chello_suites");
}

int main(void)
{
	int number_failed;
	SRunner *sr;

	sr = srunner_create(chello_suite());

	srunner_add_suite(sr, write_client_version_suite());
	srunner_add_suite(sr, write_client_random_suite());
	srunner_add_suite(sr, write_session_id_suite());
	srunner_add_suite(sr, write_cipher_suite_suite());
	srunner_add_suite(sr, write_cmp_method_suite());
	srunner_add_suite(sr, write_ext_sigalgo_suite());
	srunner_add_suite(sr, write_ext_sigalgo_cert_suite());
	srunner_add_suite(sr, write_ext_suite());
	srunner_add_suite(sr, tls_hs_chello_compose_suite());

	srunner_add_suite(sr, read_client_version_suite());
	srunner_add_suite(sr, read_client_random_suite());
	srunner_add_suite(sr, read_session_id_suite());
	srunner_add_suite(sr, read_cipher_suite_suite());
	srunner_add_suite(sr, read_cmp_method_suite());
	srunner_add_suite(sr, check_ext_availability_tls12_suite());
	srunner_add_suite(sr, check_ext_availability_tls13_suite());
	srunner_add_suite(sr, tls_hs_extension_parse_suite());
	srunner_add_suite(sr, read_ext_list_suite());
	srunner_add_suite(sr, tls_hs_chello_parse_suite());

	srunner_add_suite(sr, interpret_random_suite());
	srunner_add_suite(sr, interpret_session_id_suite());
	srunner_add_suite(sr, interpret_cipher_suite());
	srunner_add_suite(sr, interpret_cmp_method_suite());
	srunner_add_suite(sr, interpret_ext_list_suite());
	srunner_add_suite(sr, tls_hs_chello_interpret_suite());

	srunner_run_all(sr, CK_NORMAL);
	number_failed = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

/* ------------------------------------------------------------------------- */
#pragma GCC diagnostic ignored "-Wunused-parameter"
/* puts stub functions below this line. */

void OK_set_error(int error, int location, int point, CK_RV *info)
{
	ut_error = error;
	ut_location = location;
	ut_point = point;

	fprintf(stderr, "error:%0x location:%0x point:%0x\n",
		error, location, point);
	if (info != NULL) {
		fprintf(stderr, "*info=%zu\n", (size_t)(*info));
	} else {
		fprintf(stderr, "info=NULL\n");
	}
}

bool tls_alert_send(TLS *tls,
		    const enum tls_alert_level level,
		    const enum tls_alert_desc desc)
{
	ut_alert_desc = desc;
	return true;
}

uint64_t tls_util_get_epochtime(void)
{
	if (tls_util_get_epochtime_failure) {
		return 0;
	}

	return CH_EPOCH_TIME + tls_util_get_epochtime_increment;
}
bool tls_util_get_random(uint8_t *buf, uint32_t len)
{
	if (tls_util_get_random_failure) {
		if (tls_util_get_random_through_count == 0) {
			return false;
		}
		tls_util_get_random_through_count--;
	}

	memcpy(buf, static_session_id, len);

	if (tls_util_get_random_need_randomness) {
		long int ret;
		int count = len;
		srandom(CH_EPOCH_TIME);

		while (count > 0) {
			ret = random();
			if ((unsigned int)count >= sizeof(long int)) {
				memcpy(buf, &ret, sizeof(long int));
			} else {
				memcpy(buf, &ret, count);
			}
			count -= sizeof(long int);
		}
	}

	return true;
}
struct tls_session_param * tls_session_new(void)
{
	struct tls_session_param *param = NULL;

	if (!tls_session_new_failure) {
		param = calloc(1, sizeof(struct tls_session_param));
		if (param == NULL) {
			ck_abort_msg("cannot create tls_session_param");
		}
		/* session id setting is omitted. */
	}

	return param;
}
struct tls_session_param * tls_session_find_by_id(const uint8_t *session_id,
						  const uint32_t len)
{
	struct tls_session_param *param;

	if (session_id == NULL || len > 32) {
		ck_abort_msg("cannot create tls_session_param");
	}

	param = calloc(1, sizeof(struct tls_session_param));
	if (param == NULL) {
		ck_abort_msg("cannot create tls_session_param");
	}

	param->session_id_length = len;
	memcpy(param->session_id, session_id, len);

	return param;
}
void tls_session_refer(struct tls_session_param *session)
{
	;
}
struct tls_cipher_list *tls_cipher_supported_list(TLS* tls,
			const struct tls_cipher_list *preference)
{
	if (tls_cipher_supported_list_failure) {
		return NULL;
	}

	return &static_cipher_list;
}
void tls_cipher_list_free(struct tls_cipher_list *list)
{
	tls_cipher_list_alloc_count--;
}
int32_t tls_hs_sighash_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_sighash_write_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_sigalgo_extension,
	       sizeof(static_sigalgo_extension));
	static_msg.len += sizeof(static_sigalgo_extension);
	return sizeof(static_sigalgo_extension);
}
int32_t tls_hs_sighash_cert_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_sighash_cert_write_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_sigalgo_cert_extension,
	       sizeof(static_sigalgo_cert_extension));
	static_msg.len += sizeof(static_sigalgo_cert_extension);
	return sizeof(static_sigalgo_cert_extension);
}
int32_t tls_hs_ecc_write_elliptic_curves(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_ecc_write_elliptic_curves_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_elliptic_curves_extension,
	       sizeof(static_elliptic_curves_extension));
	static_msg.len += sizeof(static_elliptic_curves_extension);
	return sizeof(static_elliptic_curves_extension);
}
int32_t tls_hs_ecc_write_ec_point_formats(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_ecc_write_ec_point_formats_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_ec_point_formats_extension,
	       sizeof(static_ec_point_formats_extension));
	static_msg.len += sizeof(static_ec_point_formats_extension);
	return sizeof(static_ec_point_formats_extension);
}
int32_t tls_hs_supported_versions_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_supported_versions_write_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_supported_versions_extension,
	       sizeof(static_supported_versions_extension));
	static_msg.len += sizeof(static_supported_versions_extension);
	return sizeof(static_supported_versions_extension);
}
int32_t tls_hs_keyshare_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_keyshare_write_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_keyshare_extension,
	       sizeof(static_keyshare_extension));
	static_msg.len += sizeof(static_keyshare_extension);
	return sizeof(static_keyshare_extension);
}
int32_t tls_hs_servername_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_servername_write_failure) {
		return -1;
	}

	memcpy(&(msg->msg[msg->len]), static_servername_extension,
	       sizeof(static_servername_extension));
	static_msg.len += sizeof(static_servername_extension);
	return sizeof(static_servername_extension);
}
int32_t tls_hs_cookie_write(TLS *tls, struct tls_hs_msg *msg)
{
	if (tls_hs_cookie_write_failure) {
		return -1;
	}

	return 0;
}

static enum tls_cipher_suite list1[1024];
static struct tls_cipher_list cipher_list1 = {.len = 1024, .list = list1};

static enum tls_cipher_suite list2[1024];
static struct tls_cipher_list cipher_list2 = {.len = 1024, .list = list2};

struct tls_cipher_list *tls_cipher_list_alloc(const uint16_t len)
{
	struct tls_cipher_list *cipher_list;

	if (tls_cipher_list_alloc_failure) {
		return NULL;
	}

	switch(tls_cipher_list_alloc_count) {
	case 0:
		cipher_list = &cipher_list1;
		break;
	case 1:
		cipher_list = &cipher_list2;
		break;
	default:
		return NULL;
	}
	tls_cipher_list_alloc_count++;

	cipher_list->len = len;
	return cipher_list;
}
void tls_cipher_suite_dump(enum tls_cipher_suite suite)
{
	fprintf(stderr, "cipher_suite: %04x\n", suite);
}
bool tls_compress_set(TLS *tls, enum compression_algorithm algo)
{
	if (tls_compress_set_failure) {
		return false;
	}

	tls->pending->compression_algorithm = algo;
	return true;
}
int32_t tls_hs_ecc_read_elliptic_curves(TLS *tls,
					const struct tls_hs_msg *msg,
					const uint32_t offset)
{
	if (tls_hs_ecc_read_elliptic_curves_false) {
		return -1;
	}
	return 0;
}
int32_t tls_hs_ecc_read_point_format(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset)
{
	if (tls_hs_ecc_read_point_format_false) {
		return -1;
	}
	return 0;
}
int32_t tls_hs_sighash_read(TLS *tls,
			    const struct tls_hs_msg *msg,
			    const uint32_t offset)
{
	if (tls_hs_sighash_read_false) {
		return -1;
	}

	if (tls_hs_sighash_read_set_sighash_list) {
		tls->sighash_list = &tls_hs_sighash_list_tls13;
	}

	if (tls_hs_sighash_read_set_sighash_list_cert) {
		tls->sighash_list_cert = &tls_hs_sighash_list_cert_tls13;
	}

	return 0;
}
int32_t tls_hs_sighash_cert_read(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset) {
	if (tls_hs_sighash_read_false) {
		return -1;
	}

	if (tls_hs_sighash_read_set_sighash_list_cert) {
		tls->sighash_list_cert = &tls_hs_sighash_list_cert_tls13;
	}

	return 0;
}
int32_t tls_hs_keyshare_read(TLS *tls, const struct tls_hs_msg *msg,
			      const uint32_t offset)
{
	if (tls_hs_keyshare_read_false) {
		return -1;
	}

	return 0;
}
int32_t tls_hs_servername_read(TLS *tls, const struct tls_hs_msg *msg,
			       const uint32_t offset)
{
	if (tls_hs_servername_read_false) {
		return -1;
	}

	return 0;
}


static void _util_write_2(uint8_t *buf, int32_t val)
{
	buf[0] = ((val) >> 8) & 0xff;
	buf[1] = ((val)     ) & 0xff;
}

void tls_util_write_2(uint8_t *buf, int32_t val)
{
	_util_write_2(buf, val);
}

uint16_t tls_util_read_2(uint8_t *buf)
{
	return (((buf[0] << 8) & 0xff00) |
		((buf[1]     ) & 0x00ff));
}

void tls_util_convert_ver_to_protover(uint16_t version,
				      struct tls_protocol_version *version_st)
{
	version_st->major = (version >> 8) & 0xff;
	version_st->minor = (version     ) & 0xff;
}

uint16_t tls_util_convert_protover_to_ver(struct tls_protocol_version *version_st)
{
	return (version_st->major << 8) + ((version_st->minor) & 0x00ff);
}

bool tls_util_check_version_in_supported_version(
	struct tls_protocol_version_list *vlist,
	uint16_t version)
{
	for (int i = 0; i < vlist->len; i++) {
		if (vlist->list[i] == version) {
			return true;
		}
	}

	return false;
}

bool tls_hs_msg_write_1(struct tls_hs_msg *msg, const uint8_t dat)
{
	if (tls_hs_msg_write_1_failure) {
		if (tls_hs_msg_write_1_through_count == 0) {
			return false;
		}
		tls_hs_msg_write_1_through_count--;
	}

	msg->msg[msg->len] = dat;
	msg->len++;
	return true;
}

bool tls_hs_msg_write_2(struct tls_hs_msg *msg, const uint16_t dat)
{
	if (tls_hs_msg_write_2_failure) {
		if (tls_hs_msg_write_2_through_count == 0) {
			return false;
		}
		tls_hs_msg_write_2_through_count--;
	}

	tls_util_write_2(&msg->msg[msg->len], dat);
	msg->len +=2;

	return true;
}

bool tls_hs_msg_write_n(struct tls_hs_msg *msg,
			const uint8_t *dat, const uint32_t len)
{
	if (tls_hs_msg_write_n_failure) {
		return false;
	}

	memcpy(&(msg->msg[msg->len]), dat, len);
	msg->len += len;
	return true;
}

struct tls_hs_msg * tls_hs_msg_init()
{
	if (tls_hs_msg_init_failure) {
		return NULL;
	}
	return &static_msg;
}

void tls_hs_msg_free(struct tls_hs_msg *p)
{
	;
}

bool tls_hs_check_state(TLS *tls, enum tls_state state) {
	if (tls->state == state) {
		return true;
	}
	return false;
}

struct tls_extension *tls_extension_init(void)
{
	struct tls_extension *ext;

	if ((ext = malloc(1 * sizeof(struct tls_extension))) == NULL) {
		TLS_DPRINTF("extension: malloc: %s", strerror(errno));
		return NULL;
	}
	ext->opaque = NULL;

	return ext;
}

void tls_extension_free(struct tls_extension *ext)
{
	if (ext == NULL) {
		return;
	}

	free(ext->opaque);
	ext->opaque = NULL;

	free(ext);
}

struct tls_hs_interim_params *tls_hs_interim_params_init(void) {
	struct tls_hs_interim_params *params;

	if ((params = calloc(1, sizeof(struct tls_hs_interim_params))) == NULL) {
		TLS_DPRINTF("interim_params: malloc: %s", strerror(errno));
		return NULL;
	}
	TAILQ_INIT(&(params->head));

	return params;
}

void tls_hs_interim_params_free(struct tls_hs_interim_params *params) {
	struct tls_extension *ext;

	if (params == NULL) {
		return;
	}

	free(params->session);
	params->session = NULL;

	free(params->cmplist);
	params->cmplist = NULL;

	while (!TAILQ_EMPTY(&(params->head))) {
		ext = TAILQ_FIRST(&(params->head));
		TAILQ_REMOVE(&(params->head), ext, link);
		tls_extension_free(ext);
	}

	free(params);
}

int32_t tls_hs_extension_parse(TLS *tls,
			      const struct tls_hs_msg *msg,
			      const uint32_t offset) {
	uint32_t read_bytes = 0;

	/* no extesion provided. */
	if (msg->len == offset) {
		return 0;
	}

	/* extension provided. */
	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	const uint32_t list_length = tls_util_read_2(&(msg->msg[offset]));
	if (list_length > TLS_EXT_SIZE_MAX) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	if (msg->len < (offset + read_bytes + list_length)) {
		TLS_DPRINTF("extension: invalid record length");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* NOTE: in the case of tls->resession == true, do full check of
	 * extension. */
	for (uint32_t off = 0; off < list_length;) {
		uint32_t base = offset + read_bytes;

		struct tls_extension *ext;
		if ((ext = tls_extension_init()) == NULL) {
			TLS_DPRINTF("tls_extension_init");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		ext->type = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		ext->len = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		TLS_DPRINTF("extension: type = %02d, len = %d", ext->type, ext->len);

		if ((ext->opaque = malloc(1 * ext->len)) == NULL) {
			tls_extension_free(ext);
			TLS_DPRINTF("malloc: %s", strerror(errno));
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		memcpy(ext->opaque, &(msg->msg[base+off]), ext->len);
		TAILQ_INSERT_TAIL(&(tls->interim_params->head), ext, link);
		off += ext->len;
	}
	read_bytes += list_length;

	return read_bytes;
}

static struct tls_hs_sighash_algo static_algo_list = {
	.hash = TLS_HASH_ALGO_NONE, .sig = TLS_SIG_ALGO_ANON,
};
static struct tls_hs_sighash_list static_sighash_list = {
	.len = 1, .list = &static_algo_list,
};

bool tls_hs_sighash_save_default(TLS *tls)
{
	fprintf(stderr, "%s called.\n", __func__);
	if (tls_hs_sighash_save_default_failure) {
		return false;
	}

	tls->sighash_list = &static_sighash_list;
	return true;
}
