/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_cipher.h"
#include "tls_alert.h"

/**
 * write transcript hash to tls_hs_msg structure according to selected
 * hash algorithms that is associated with cipher suite.
 */
static int32_t write_message_hash(TLS *tls, struct tls_hs_msg *msg);

static int32_t write_message_hash(TLS *tls, struct tls_hs_msg *msg)
{
	enum tls_hs_sighash_hash_algo hash_algo;
	if ((hash_algo = tls_cipher_hashalgo(tls->pending->cipher_suite))
	    == TLS_HASH_ALGO_NONE) {
		TLS_DPRINTF("tls_cipher_hashalgo");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	uint8_t hash_len = tls_hs_sighash_get_hash_size(hash_algo);
	uint8_t hash[hash_len];
	tls_hs_hash_get_digest(hash_algo, tls, hash);

	if (tls_hs_msg_write_n(msg, hash, hash_len) == false) {
		TLS_DPRINTF("tls_hs_msg_write_n");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	return hash_len;
}

struct tls_hs_msg *tls_hs_msghash_make(TLS *tls)
{
	uint32_t offset = 0;

	/* compose message hash */
	struct tls_hs_msg *msg;
	if ((msg = tls_hs_msg_init()) == NULL) {
		TLS_DPRINTF("tls_hs_msg_init");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	/* MessageHash message has following structure.
	 *
	 * | type                      (1) |
	 * | length of message         (3) |
	 * | hash                      (n) | n == hash length
	 */
	msg->type = TLS_HANDSHAKE_MESSAGE_HASH;

	int32_t write_bytes;
	if ((write_bytes = write_message_hash(tls, msg)) < 0) {
		goto failed;
	}
	offset += write_bytes;

	msg->len = offset;

	return msg;

failed:
	tls_hs_msg_free(msg);
	return NULL;
}
