/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_cipher.h"
#include "tls_compress.h"
#include "tls_alert.h"

#include <string.h>

/**
 * check whether the received client hello extension is available in
 * this module.
 *
 * TODO: i think this function should be moved to other file.
 * (e.g. handshake/extension/extension.c?).
 */
static bool check_ext_availability_tls12(const enum tls_extension_type type);

static bool check_ext_availability_tls13(const enum tls_extension_type type);

static bool check_ext_availability(TLS *tls, const enum tls_extension_type type);

/**
 * write client version to the send data.
 *
 * default version is the supported max version.
 */
static int32_t write_client_version(TLS *tls, struct tls_hs_msg *msg);

/**
 * write client random to the send data.
 */
static int32_t write_client_random(TLS *tls, struct tls_hs_msg *msg);

/**
 * write session id to the send data.
 *
 * initialize session of tls structure too.
 */
static int32_t write_session_id(TLS *tls, struct tls_hs_msg *msg);

/**
 * write cipher suite list that want to use to the send data.
 */
static int32_t write_cipher_suite(TLS *tls, struct tls_hs_msg *msg);

/**
 * write compression algorithm list that want to use to the send data.
 */
static int32_t write_cmp_method(struct tls_hs_msg *msg);

/**
 * write signature algorithm extension as the extension data.
 */
static int32_t write_ext_sigalgo(TLS *tls, struct tls_hs_msg *msg);

/**
 * write signature algorithm cert extension as the extension data.
 */
static int32_t write_ext_sigalgo_cert(TLS *tls, struct tls_hs_msg *msg);

/**
 * write client hello extension that want to use to the send data.
 */
static int32_t write_ext(TLS *tls, struct tls_hs_msg *msg);

/**
 * read client version from received handshake.
 */
static int32_t read_client_version(TLS *tls,
				   const struct tls_hs_msg *msg,
				   const uint32_t offset);

/**
 * read client random from received handshake.
 */
static int32_t read_client_random(TLS *tls,
				  const struct tls_hs_msg *msg,
				  const uint32_t offset);

/**
 * read session id from received handshake.
 *
 * if session id is included in the handshake, server checks whether the
 * session id is included in session list that is managed by server. if
 * that is included, server will progress to resumption process.
 */
static int32_t read_session_id(TLS *tls,
			       const struct tls_hs_msg *msg,
			       const uint32_t offset);

/**
 * read list of cipher suites from received handshake.
 *
 * The entries of the list that was read is temporary stored in
 * tls->chello_cipher_suites.
 */
static int32_t read_cipher_suite(TLS *tls,
				 const struct tls_hs_msg *msg,
				 const uint32_t offset);

/**
 * read compression method from received handshake. and determine a
 * compression algorithm that is going to use in this connection
 * (session).
 *
 * only NULL compression method is accepted.
 */
static int32_t read_cmp_method(TLS *tls,
			       const struct tls_hs_msg *msg,
			       const uint32_t offset);

/**
 * read extension from client hello extension list. and handle it.
 */
static bool read_ext_list(TLS *tls,
			  const enum tls_extension_type type,
			  const struct tls_hs_msg *msg,
			  const uint32_t offset);

/**
 * interpret random stored in tls_interim_params.
 */
static bool interpret_random(TLS *tls);

/**
 * interpret session id stored in tls_interim_params.
 */
static bool interpret_session_id_up_to_tls12(TLS *tls);
static bool interpret_session_id_tls13(TLS *tls);
static bool interpret_session_id(TLS *tls);

/**
 * interpret cipher suites stored in tls_interim_params.
 */
static bool interpret_cipher(TLS *tls);

/**
 * interpret compression method stored in tls_interim_params.
 */
static bool interpret_cmp_method(TLS *tls);

/**
 * interpret extensions stored in tls_interim_params.
 */
static bool interpret_ext_list(TLS *tls);

static bool check_ext_availability_tls12(const enum tls_extension_type type) {
	switch(type) {
	case TLS_EXT_SERVER_NAME:
	case TLS_EXT_ELLIPTIC_CURVES:
	case TLS_EXT_EC_POINT_FORMATS:
	case TLS_EXT_SIGNATURE_ALGO:
		return true;

	default:
		break;
	}

	return false;
}

static bool check_ext_availability_tls13(const enum tls_extension_type type) {
	switch(type) {
	case TLS_EXT_SERVER_NAME:
	case TLS_EXT_MAX_FRAGMENT_LENGTH:
	case TLS_EXT_STATUS_REQUEST:
	case TLS_EXT_SUPPORTED_GROUPS:
	case TLS_EXT_SIGNATURE_ALGO:
	case TLS_EXT_USE_SRTP:
	case TLS_EXT_HEARTBEAT:
	case TLS_EXT_APP_LAYER_PROTO_NEGOTIATION:
	case TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP:
	case TLS_EXT_CLIENT_CERTIFICATE_TYPE:
	case TLS_EXT_SERVER_CERTIFICATE_TYPE:
	case TLS_EXT_PADDING:
	case TLS_EXT_PRE_SHARED_KEY:
	case TLS_EXT_EARLY_DATA:
	case TLS_EXT_SUPPORTED_VERSIONS:
	case TLS_EXT_COOKIE:
	case TLS_EXT_PSK_KEY_EXCHANGE_MODES:
	case TLS_EXT_CERTIFICATE_AUTHORITIES:
	case TLS_EXT_POST_HANDSHAKE_AUTH:
	case TLS_EXT_SIGNATURE_ALGO_CERT:
	case TLS_EXT_KEY_SHARE:
		return true;

	default:
		break;
	}

	return false;
}

static bool check_ext_availability(TLS *tls,
					 const enum tls_extension_type type) {
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));

	switch(version) {
	case TLS_VER_SSL30:
		/* Not supported */
		break;

	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
		/* Not implemented */
		break;

	case TLS_VER_TLS12:
		return check_ext_availability_tls12(type);

	case TLS_VER_TLS13:
		return check_ext_availability_tls13(type);

	default:
		/* Unknown version */
		break;
	}

	return false;
}

static int32_t write_client_version(TLS *tls, struct tls_hs_msg *msg) {
	const int32_t verlen = 2;

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 * legacy_version:  In previous versions of TLS, this field was used for
	 *    version negotiation and represented the highest version number
	 *    supported by the client.  Experience has shown that many servers
	 *    do not properly implement version negotiation, leading to "version
	 *    intolerance" in which the server rejects an otherwise acceptable
	 *    ClientHello with a version number higher than it supports.  In
	 *    TLS 1.3, the client indicates its version preferences in the
	 *    "supported_versions" extension (Section 4.2.1) and the
	 *    legacy_version field MUST be set to 0x0303, which is the version
	 *    number for TLS 1.2.  TLS 1.3 ClientHellos are identified as having
	 *    a legacy_version of 0x0303 and a supported_versions extension
	 *    present with 0x0304 as the highest version indicated therein.
	 *    (See Appendix D for details about backward compatibility.)
	 */
	/* TODO: add version check and process branch. prior to TLS 1.3,
	 * we have to write highest version number supported by client.
	 * so we should search highest vesion in supported versions.
	 * when supporetd versions includes TLS 1.2 or TLS 1.3, write version number
	 * of TLS 1.2. otherwise write highest supported version number.
	 */
	uint16_t client_version = TLS_VER_TLS12;
	uint16_t record_version = TLS_VER_TLS10;
	if (! tls_hs_msg_write_2(msg, client_version)) {
		return -1;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO) == true) {
		/* save client hello version to use client key exchange
		 * handshake protocol message. */
		tls_util_convert_ver_to_protover(client_version, &(tls->client_version));

		/* save record version for first client hello message. */
		tls_util_convert_ver_to_protover(record_version, &(tls->record_version));
	}

	return verlen;
}

static int32_t write_client_random(TLS *tls, struct tls_hs_msg *msg) {
	const int32_t randlen = 32;

	uint64_t sec;

	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO) == true) {
		if ((sec = tls_util_get_epochtime()) == 0) {
			OK_set_error(ERR_ST_TLS_GET_EPOCHTIME,
				     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 0, NULL);
			return -1;
		}

		memcpy(&(tls->client_random[0]), &sec, 4);

		uint8_t crandom[28];
		if (! tls_util_get_random(&(crandom[0]), 28)) {
			OK_set_error(ERR_ST_TLS_GET_RANDOM,
				     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 1, NULL);
			return -1;
		}

		memcpy(&(tls->client_random[4]), &crandom, 28);
	}

	if(! tls_hs_msg_write_n(msg, tls->client_random, randlen)) {
		return -1;
	}

	return randlen;
}

static int32_t write_session_id(TLS *tls, struct tls_hs_msg *msg) {
	const int32_t length_bytes = 1;

	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO) == true) {
		if ((tls->pending = tls_session_new()) == NULL) {
			OK_set_error(ERR_ST_TLS_GET_SESSION,
				     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 2, NULL);
			return -1;
		}

		/* increment reference count about the handshake. */
		tls_session_refer(tls->pending);
	}

	/* TODO: client can use compatibility mode by providing session id.
	 *
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    legacy_session_id:  Versions of TLS before TLS 1.3 supported a
	 *       "session resumption" feature which has been merged with pre-shared
	 *       keys in this version (see Section 2.2).  A client which has a
	 *       cached session ID set by a pre-TLS 1.3 server SHOULD set this
	 *       field to that value.  In compatibility mode (see Appendix D.4),
	 *       this field MUST be non-empty, so a client not offering a
	 *       pre-TLS 1.3 session MUST generate a new 32-byte value.  This value
	 *       need not be random but SHOULD be unpredictable to avoid
	 *       implementations fixating on a specific value (also known as
	 *       ossification).  Otherwise, it MUST be set as a zero-length vector
	 *       (i.e., a zero-valued single byte length field).
	 *
	 * RFC8446 D.4.  Middlebox Compatibility Mode
	 *
	 *    -  The client always provides a non-empty session ID in the
	 *       ClientHello, as described in the legacy_session_id section of
	 *       Section 4.1.2.
	 *
	 *    -  If not offering early data, the client sends a dummy
	 *       change_cipher_spec record (see the third paragraph of Section 5)
	 *       immediately before its second flight.  This may either be before
	 *       its second ClientHello or before its encrypted handshake flight.
	 *       If offering early data, the record is placed immediately after the
	 *       first ClientHello.
	 *
	 *    -  The server sends a dummy change_cipher_spec record immediately
	 *       after its first handshake message.  This may either be after a
	 *       ServerHello or a HelloRetryRequest.
	 *
	 *    When put together, these changes make the TLS 1.3 handshake resemble
	 *    TLS 1.2 session resumption, which improves the chance of successfully
	 *    connecting through middleboxes.  This "compatibility mode" is
	 *    partially negotiated: the client can opt to provide a session ID or
	 *    not, and the server has to echo it.  Either side can send
	 *    change_cipher_spec at any time during the handshake, as they must be
	 *    ignored by the peer, but if the client sends a non-empty session ID,
	 *    the server MUST send the change_cipher_spec as described in this
	 *    appendix.
	 */
	/* XXX: consider re-session */
	int32_t session_id_length = 0;

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           opaque legacy_session_id<0..32>;
	 */
	const uint8_t session_id_min = 0;
	const uint8_t session_id_max = 32;
	if (session_id_length < session_id_min ||
	    session_id_max < session_id_length) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 4, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	if (! tls_hs_msg_write_1(msg, session_id_length)) {
		return -1;
	}

	/* init for re-session (check by recv of server hello) */
	tls->pending->session_id_length = session_id_length;

	return length_bytes;
}

static int32_t write_cipher_suite(TLS *tls, struct tls_hs_msg *msg) {
	int32_t off = 0;

	struct tls_cipher_list *list = tls_cipher_supported_list(tls, NULL);
	if (list == NULL) {
		return -1;
	}

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           CipherSuite cipher_suites<2..2^16-2>;
	 */
	const uint16_t list_len_min = 2;
	const uint16_t list_len_max = (2 << (16 - 1)) - 2;
	if (list->len * 2 < list_len_min || list_len_max < list->len * 2) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 25, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	if (! tls_hs_msg_write_2(msg, list->len * 2)) {
		tls_cipher_list_free(list);
		return -1;
	}
	off += 2;

	for (int i = 0; i < list->len; ++i) {
		if (! tls_hs_msg_write_2(msg, list->list[i])) {
			tls_cipher_list_free(list);
			return -1;
		}
		off += 2;
	}

	tls_cipher_list_free(list);
	return off;
}

static int32_t write_cmp_method(struct tls_hs_msg *msg) {
	if (! tls_hs_msg_write_1(msg, 1)) {
		return -1;
	}

	if (! tls_hs_msg_write_1(msg, TLS_COMPRESSION_NULL)) {
		return -1;
	}

	return 2;
}

static int32_t write_ext_sigalgo(TLS *tls, struct tls_hs_msg *msg) {
	int32_t off = 0;

	/* RFC 5246 section 7.4.1.4.1 says.
	 *
	 *   Note: this extension is not meaningful for TLS versions
	 *   prior to 1.2.  Clients MUST NOT offer it if they are
	 *   offering prior versions.  However, even if clients do offer
	 *   it, the rules specified in [TLSEXT] require servers to
	 *   ignore extensions they do not understand.
	 */

	struct tls_protocol_version_list *vlist = &(tls->supported_versions);
	if (! tls_util_check_version_in_supported_version(vlist, TLS_VER_TLS12) &&
	    ! tls_util_check_version_in_supported_version(vlist, TLS_VER_TLS13)) {
		return 0;
	}

	const uint32_t type_bytes = 2;
	if (tls_hs_msg_write_2(msg, TLS_EXT_SIGNATURE_ALGO) == false) {
		return -1;
	}
	off += type_bytes;

	/* write dummy length bytes. */
	int32_t pos = msg->len;

	const uint32_t len_bytes = 2;
	if (tls_hs_msg_write_2(msg, 0) == false) {
		return -1;
	}
	off += len_bytes;

	int32_t sighash_len;
	if ((sighash_len = tls_hs_sighash_write(tls, msg)) < 0) {
		return -1;
	}
	off += sighash_len;

	const int32_t extlen_max = TLS_EXT_SIZE_MAX;
	if (sighash_len > extlen_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 26, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	tls_util_write_2(&(msg->msg[pos]), sighash_len);

	return off;
}

static int32_t write_ext_sigalgo_cert(TLS *tls, struct tls_hs_msg *msg) {
	int32_t off = 0;

	struct tls_protocol_version_list *vlist = &(tls->supported_versions);
	if (tls_util_check_version_in_supported_version(vlist, TLS_VER_TLS13)
	    == false) {
		return 0;
	}

	const uint32_t type_bytes = 2;
	if (tls_hs_msg_write_2(msg, TLS_EXT_SIGNATURE_ALGO_CERT) == false) {
		return -1;
	}
	off += type_bytes;

	/* write dummy length bytes. */
	int32_t pos = msg->len;

	const uint32_t len_bytes = 2;
	if (tls_hs_msg_write_2(msg, 0) == false) {
		return -1;
	}
	off += len_bytes;

	int32_t sighash_len;
	if ((sighash_len = tls_hs_sighash_cert_write(tls, msg)) < 0) {
		return -1;
	}
	off += sighash_len;

	const int32_t extlen_max = TLS_EXT_SIZE_MAX;
	if (sighash_len > extlen_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 27, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	tls_util_write_2(&(msg->msg[pos]), sighash_len);

	return off;
}

static int32_t write_ext(TLS *tls, struct tls_hs_msg *msg) {
	int32_t off = 0;

	/* write dummy length bytes */
	const int32_t pos = msg->len;
	const int32_t length_bytes = 2;
	if (tls_hs_msg_write_2(msg, 0) == false) {
		return -1;
	}
	off += length_bytes;

	/* RFC 5246 section 7.4.1.4 says.
	 *
	 *  The extension format is:
	 *
	 *     struct {
	 *         ExtensionType extension_type;
	 *         opaque extension_data<0..2^16-1>;
	 *     } Extension;
	 *
	 *     enum {
	 *         signature_algorithms(13), (65535)
	 *     } ExtensionType;
	 */

	int32_t extlen = 0;
	bool *sent_exts = tls->interim_params->sent_ext_flags;

	int32_t server_name_len = 0;
	if ((server_name_len = tls_hs_servername_write(tls, msg)) < 0) {
		return -1;
	} else if (server_name_len > 0) {
		sent_exts[TLS_EXT_SERVER_NAME] = true;
	}
	extlen += server_name_len;

	int32_t sigalgo_len = 0;
	if ((sigalgo_len = write_ext_sigalgo(tls, msg)) < 0) {
		return -1;
	} else if (sigalgo_len > 0) {
		sent_exts[TLS_EXT_SIGNATURE_ALGO] = true;
	}
	extlen += sigalgo_len;

	int32_t sigalgo_cert_len = 0;
	if ((sigalgo_cert_len = write_ext_sigalgo_cert(tls, msg)) < 0) {
		return -1;
	} else if (sigalgo_cert_len > 0) {
		sent_exts[TLS_EXT_SIGNATURE_ALGO_CERT] = true;
	}
	extlen += sigalgo_cert_len;

	int32_t elliptic_curves_len;
	elliptic_curves_len = tls_hs_ecc_write_elliptic_curves(tls, msg);
	if (elliptic_curves_len < 0) {
		return -1;
	} else if (elliptic_curves_len > 0) {
		sent_exts[TLS_EXT_ELLIPTIC_CURVES] = true;
	}
	extlen += elliptic_curves_len;

	int32_t ec_point_formats_len;
	ec_point_formats_len = tls_hs_ecc_write_ec_point_formats(tls, msg);
	if (ec_point_formats_len < 0) {
		return -1;
	} else if (ec_point_formats_len > 0) {
		sent_exts[TLS_EXT_EC_POINT_FORMATS] = true;
	}
	extlen += ec_point_formats_len;

	/*
	 * TODO: implement early_data extension after PSK is ready to use.
	 * early_data extension can be sent only in first client hello as
	 * described below.
	 *
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    -  Removing the "early_data" extension (Section 4.2.10) if one was
	 *       present.  Early data is not permitted after a HelloRetryRequest.
	 */

	int32_t supported_versions_len;
	supported_versions_len = tls_hs_supported_versions_write(tls, msg);
	if (supported_versions_len < 0) {
		return -1;
	} else if (supported_versions_len > 0) {
		sent_exts[TLS_EXT_SUPPORTED_VERSIONS] = true;
	}
	extlen += supported_versions_len;

	int32_t cookie_len;
	cookie_len = tls_hs_cookie_write(tls, msg);
	if (cookie_len < 0) {
		return -1;
	} else if (cookie_len > 0) {
		sent_exts[TLS_EXT_COOKIE] = true;
	}
	extlen += cookie_len;

	int32_t key_share_len;
	key_share_len = tls_hs_keyshare_write(tls, msg);
	if (key_share_len < 0) {
		return -1;
	} else if (key_share_len > 0) {
		sent_exts[TLS_EXT_KEY_SHARE] = true;
	}
	extlen += key_share_len;

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           Extension extensions<8..2^16-1>;
	 */
	const int32_t extlen_min = 8;
	if (tls_util_check_version_in_supported_version(
		&(tls->supported_versions), TLS_VER_TLS13) == true &&
	    extlen < extlen_min) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 28, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	const int32_t extensions_length_max = TLS_VECTOR_2_BYTE_SIZE_MAX;
	if (extlen > extensions_length_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 29, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	/* write extension length. */
	tls_util_write_2(&(msg->msg[pos]), extlen);

	off += extlen;

	/* if any extension do not be written, revert the dummy length
	 * bytes that was written in the first. */
	if (off == length_bytes) {
		off      -= length_bytes;
		msg->len -= length_bytes;
	}

	return off;
}

static int32_t read_client_version(TLS *tls,
				   const struct tls_hs_msg *msg,
				   const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t verlen = 2;
	if (msg->len < (offset + verlen)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 3, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += verlen;

	tls->interim_params->version.major = msg->msg[offset + 0];
	tls->interim_params->version.minor = msg->msg[offset + 1];

	return read_bytes;
}

static int32_t read_client_random(TLS *tls,
				  const struct tls_hs_msg *msg,
				  const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t randlen = 32;
	if (msg->len < (offset + randlen)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 5, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += randlen;

	memcpy(tls->interim_params->client_random, &(msg->msg[offset]), randlen);

	return read_bytes;
}

static int32_t read_session_id(TLS *tls,
			       const struct tls_hs_msg *msg,
			       const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t length_bytes = 1;
	if (msg->len < (offset + length_bytes)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 6, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	const uint8_t seslen = msg->msg[offset] & 0xff;
	if (msg->len < (offset + read_bytes + seslen)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           opaque legacy_session_id<0..32>;
	 */
	const uint8_t seslen_max = 32;
	if (seslen > seslen_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 30, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	tls->interim_params->seslen = seslen;

	if (seslen > 0) {
		/* check session id for re-session. */
		uint8_t *session = &(msg->msg[offset + read_bytes]);
		tls->interim_params->session = malloc(1 * seslen);
		if (tls->interim_params->session == NULL) {
			OK_set_error(ERR_ST_TLS_MALLOC,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CHELLO + 11, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		memcpy(tls->interim_params->session, session, seslen);
	}
	read_bytes += seslen;

	return read_bytes;
}

static int32_t read_cipher_suite(TLS *tls,
				 const struct tls_hs_msg *msg,
				 const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	uint16_t cslist_bytes = tls_util_read_2(&(msg->msg[offset]));
	read_bytes += length_bytes;

	TLS_DPRINTF("hs: chello: list of cipher suites (bytes) = %d",
		    cslist_bytes);
	if (msg->len < (offset + read_bytes + cslist_bytes)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 10, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           CipherSuite cipher_suites<2..2^16-2>;
	 */
	const uint16_t cslist_bytes_min = 2;
	const uint16_t cslist_bytes_max = (2 << (16 - 1)) - 2;
	if (cslist_bytes < cslist_bytes_min ||
	    cslist_bytes_max < cslist_bytes) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 31, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* check if length is even number */
	if ((cslist_bytes % 2) == 1) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CHELLO2 + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	uint16_t cslist_length = cslist_bytes / 2;

	struct tls_cipher_list *cipher;
	if ((cipher = tls_cipher_list_alloc(cslist_length)) == NULL) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	int32_t base = offset + read_bytes;
	for (uint16_t i = 0; i < cipher->len; i++) {
		cipher->list[i] = tls_util_read_2(&(msg->msg[base + i * 2]));
		tls_cipher_suite_dump(cipher->list[i]);
	}

	tls->interim_params->cipher_suites = cipher;

	read_bytes += cslist_bytes;
	return read_bytes;
}

static int32_t read_cmp_method(TLS *tls,
			       const struct tls_hs_msg *msg,
			       const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t length_bytes = 1;
	if (msg->len < (offset + length_bytes)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 12, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	const uint8_t list_length = msg->msg[offset];
	if (msg->len < (offset + read_bytes + list_length)) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 13, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *           opaque legacy_compression_methods<1..2^8-1>;
	 */
	const uint8_t list_length_min = 1;
	if(list_length < list_length_min) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CHELLO2 + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
	}

	read_bytes += list_length;
	tls->interim_params->cmplen = list_length;

	int32_t base = offset + length_bytes;
	tls->interim_params->cmplist = malloc(1 * list_length);
	if (tls->interim_params->cmplist == NULL) {
		OK_set_error(ERR_ST_TLS_MALLOC,
		     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 14, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}
	memcpy(tls->interim_params->cmplist, &(msg->msg[base]), list_length);

	return read_bytes;
}

static bool read_ext_list(TLS *tls,
			  const enum tls_extension_type type,
			  const struct tls_hs_msg *msg,
			  const uint32_t offset) {
	/*
	 * assume unknown extensions never come because check
	 * is performed before this function is called.
	 */
	/*
	 * RFC8446 4.2.  Extensions
	 *
	 *                        There MUST NOT be more than one extension of the
	 *    same type in a given extension block.
	 */
	bool *recv_exts = tls->interim_params->recv_ext_flags;
	if (recv_exts[type] == true) {
		TLS_DPRINTF("chello: extensions of same type come multiple times");
		OK_set_error(ERR_ST_TLS_SAME_TYPE_EXTENSION,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 15, NULL);
		return false;
	}
	recv_exts[type] = true;

	switch(type) {
	case TLS_EXT_SERVER_NAME:
		if (tls_hs_servername_read(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	case TLS_EXT_ELLIPTIC_CURVES:
		if (tls_hs_ecc_read_elliptic_curves(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	case TLS_EXT_EC_POINT_FORMATS:
		if (tls_hs_ecc_read_point_format(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	case TLS_EXT_SIGNATURE_ALGO:
		if (tls_hs_sighash_read(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	case TLS_EXT_SIGNATURE_ALGO_CERT:
		if (tls_hs_sighash_cert_read(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	case TLS_EXT_SUPPORTED_VERSIONS:
		return true;

	case TLS_EXT_KEY_SHARE:
		if (tls_hs_keyshare_read(tls, msg, offset) < 0) {
			return false;
		}
		return true;

	default:
		TLS_DPRINTF("ignore extension type = %d", type);
		return true;
	}

	return false;
}

static bool interpret_random(TLS *tls) {
	const uint32_t randlen = 32;
	memcpy(tls->client_random, tls->interim_params->client_random, randlen);

	return true;
}

static bool interpret_session_id_up_to_tls12(TLS *tls) {
	struct tls_session_param *param = NULL;

	if (tls->interim_params->seslen > 0) {
		uint8_t *session = tls->interim_params->session;
		param = tls_session_find_by_id(session, tls->interim_params->seslen);
	}

	/* the case of seslen == 0 or session is not found in the
	 * session list of server. */
	if (param != NULL) {
		tls->resession = true;
		tls->pending = param;
	} else if ((tls->pending = tls_session_new()) == NULL) {
		OK_set_error(ERR_ST_TLS_GET_SESSION,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	/* increment reference count about the handshake. */
	tls_session_refer(tls->pending);

	return true;
}

static bool interpret_session_id_tls13(TLS *tls) {
	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    legacy_session_id:  Versions of TLS before TLS 1.3 supported a
	 *       "session resumption" feature which has been merged with pre-shared
	 *       keys in this version (see Section 2.2).  A client which has a
	 *       cached session ID set by a pre-TLS 1.3 server SHOULD set this
	 *       field to that value.  In compatibility mode (see Appendix D.4),
	 *       this field MUST be non-empty, so a client not offering a
	 *       pre-TLS 1.3 session MUST generate a new 32-byte value.  This value
	 *       need not be random but SHOULD be unpredictable to avoid
	 *       implementations fixating on a specific value (also known as
	 *       ossification).  Otherwise, it MUST be set as a zero-length vector
	 *       (i.e., a zero-valued single byte length field).
	 */
	if (tls->interim_params->seslen != 0
	    && tls->interim_params->seslen != 32) {
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 18, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if ((tls->pending = tls_session_new()) == NULL) {
		OK_set_error(ERR_ST_TLS_GET_SESSION,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 19, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	/* increment reference count about the handshake. */
	tls_session_refer(tls->pending);

	return true;
}

static bool interpret_session_id(TLS *tls) {
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		return interpret_session_id_tls13(tls);

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		return interpret_session_id_up_to_tls12(tls);
	}
}

static bool interpret_cipher(TLS *tls) {
	uint16_t list_len = tls->interim_params->cipher_suites->len;
	if ((tls->chello_cipher_suites = tls_cipher_list_alloc(list_len))
	    == NULL) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	memcpy(tls->chello_cipher_suites->list,
	       tls->interim_params->cipher_suites->list,
	       sizeof(enum tls_cipher_suite) * list_len);

	return true;
}

static bool interpret_cmp_method(TLS *tls) {
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		/*
		 * RFC8446 4.1.2.  Client Hello
		 *
		 *    legacy_compression_methods:  Versions of TLS before 1.3 supported
		 *       compression with the list of supported compression methods being
		 *       sent in this field.  For every TLS 1.3 ClientHello, this vector
		 *       MUST contain exactly one byte, set to zero, which corresponds to
		 *       the "null" compression method in prior versions of TLS.  If a
		 *       TLS 1.3 ClientHello is received with any other value in this
		 *       field, the server MUST abort the handshake with an
		 *       "illegal_parameter" alert.
		 */
		if (tls->interim_params->cmplen != 1) {
			TLS_DPRINTF("contain method(s) except null");
			OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
				     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 20,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
			return false;
		}
		break;

	default:
		break;
	}

	for (uint8_t i = 0; i < tls->interim_params->cmplen; ++i) {
		uint8_t cmp_method = tls->interim_params->cmplist[i];

		/* NOTE: if server support the compression method except
		 * NULL, in resession case, check whether that special
		 * compression method is included. */

		/* server supports null compression method only. */
		if (cmp_method == TLS_COMPRESSION_NULL) {
			/* Normally, tls_compress_set does not fail in
			 * this context since cmp_method is
			 * TLS_COMPRESSION_NULL only. */
			if (! tls_compress_set(tls, cmp_method)) {
				TLS_DPRINTF("failed to set compression method %d", cmp_method);
				return false;
			}
			goto fin;
		}
	}

	/* there is not null compression method in the list of
	 * compression method that provided by client hello. */
	OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
		     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 21, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
	return false;

fin:
	return true;
}

static bool interpret_ext_list(TLS *tls) {
	struct tls_hs_interim_params *params = tls->interim_params;
	struct tls_extension *ext;
	struct tls_hs_msg msg;

	/*
	 * search an extension list for supported_groups and
	 * signature_algorithms_cert extension.
	 */
	struct tls_extension *ext_sg = NULL;
	struct tls_extension *ext_sig_cert = NULL;
	TAILQ_FOREACH(ext, &(params->head), link) {
		if (ext->type == TLS_EXT_SUPPORTED_GROUPS) {
			ext_sg = ext;
		}

		if (ext->type == TLS_EXT_SIGNATURE_ALGO_CERT) {
			ext_sig_cert = ext;
		}
	}

	/*
	 * supported_groups extension must be processed earlier than key_share
	 * extension in TLS 1.3. though there is no process order like that
	 * in TLS 1.0, 1.1 and 1.2, process it here too.
	 */
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
		/* Not supported */
		break;

	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	case TLS_VER_TLS13:
		if (ext_sg != NULL) {
			msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
			msg.len = ext_sg->len;
			msg.max = ext_sg->len;
			msg.msg = ext_sg->opaque;

			if (! read_ext_list(tls, ext_sg->type, &msg, 0)) {
				/* alerts is sent by internal of tls_hs_ext_read. */
				return false;
			}

			TAILQ_REMOVE(&(params->head), ext_sg, link);
		}
		break;

	default:
		break;
	}

	/*
	 * signature_algorithms_cert extension must be processed earlier than
	 * signature_algorithms extension in TLS 1.3.
	 */
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		/* Not supported */
		break;

	case TLS_VER_TLS13:
		if (ext_sig_cert != NULL) {
			msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
			msg.len = ext_sig_cert->len;
			msg.max = ext_sig_cert->len;
			msg.msg = ext_sig_cert->opaque;

			if (! read_ext_list(tls, ext_sig_cert->type, &msg, 0)) {
				/* alerts is sent by internal of read_ext_list. */
				return false;
			}

			TAILQ_REMOVE(&(params->head), ext_sig_cert, link);
		}
		break;

	default:
		break;
	}

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 * Servers MUST ignore unrecognized extensions.
	 */
	TAILQ_FOREACH(ext, &(params->head), link) {
		if (check_ext_availability(tls, ext->type) == false) {
			TLS_DPRINTF("skip unknown extension %d", ext->type);
		} else {
			msg.type = TLS_HANDSHAKE_CLIENT_HELLO;
			msg.len = ext->len;
			msg.max = ext->len;
			msg.msg = ext->opaque;

			if (! read_ext_list(tls, ext->type, &msg, 0)) {
				/* alerts is sent by internal of tls_hs_ext_read. */
				return false;
			}
		}
	}

	/*
	 * enqueue supported_groups and signature_algorithms_cert extension
	 * for later use.
	 */
	if (ext_sg != NULL) {
		TAILQ_INSERT_TAIL(&(params->head), ext_sg, link);
	}

	if (ext_sig_cert != NULL) {
		TAILQ_INSERT_TAIL(&(params->head), ext_sig_cert, link);
	}

	/*
	 * RFC8446 4.2.3.  Signature Algorithms
	 *
	 *                                      ...  Clients which desire the
	 *    server to authenticate itself via a certificate MUST send the
	 *    "signature_algorithms" extension.  If a server is authenticating via
	 *    a certificate and the client has not sent a "signature_algorithms"
	 *    extension, then the server MUST abort the handshake with a
	 *    "missing_extension" alert (see Section 9.2).
	 *
	 * 9.2.  Mandatory-to-Implement Extensions
	 *
	 *    A client is considered to be attempting to negotiate using this
	 *    specification if the ClientHello contains a "supported_versions"
	 *    extension with 0x0304 contained in its body.  Such a ClientHello
	 *    message MUST meet the following requirements:
	 *
	 *    -  If not containing a "pre_shared_key" extension, it MUST contain
	 *       both a "signature_algorithms" extension and a "supported_groups"
	 *       extension.
	 *
	 *    -  If containing a "supported_groups" extension, it MUST also contain
	 *       a "key_share" extension, and vice versa.  An empty
	 *       KeyShare.client_shares vector is permitted.
	 *
	 *    Servers receiving a ClientHello which does not conform to these
	 *    requirements MUST abort the handshake with a "missing_extension"
	 *    alert.
	 */
	switch (version) {
	case TLS_VER_TLS13:
		if (params->recv_ext_flags[TLS_EXT_PRE_SHARED_KEY] == false) {
			if (params->recv_ext_flags[TLS_EXT_SIGNATURE_ALGO] == false ||
			    params->recv_ext_flags[TLS_EXT_SUPPORTED_GROUPS] == false) {
				TLS_DPRINTF("missing extension");
				OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
					     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 22, NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_MISSING_EXTENSION);
				return false;
			}
		}

		if ((params->recv_ext_flags[TLS_EXT_SUPPORTED_GROUPS] == true &&
		     params->recv_ext_flags[TLS_EXT_KEY_SHARE] == false) ||
		    (params->recv_ext_flags[TLS_EXT_SUPPORTED_GROUPS] == false &&
		     params->recv_ext_flags[TLS_EXT_KEY_SHARE] == true)) {
				TLS_DPRINTF("missing extension");
				OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
					     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 23, NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_MISSING_EXTENSION);
				return false;
		}

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		break;
	}

	return true;
}

struct tls_hs_msg * tls_hs_chello_compose(TLS *tls) {
	uint32_t offset = 0;

	struct tls_hs_msg *msg;
	if ((msg = tls_hs_msg_init()) == NULL) {
		TLS_DPRINTF("tls_hs_msg_init");
		return NULL;
	}

	/* clientHello message has following structure.
	 *
	 * | type                      (1) |
	 * | length of message         (3) |
	 * | major version             (1) |
	 * | minor version             (1) |
	 * | gmt unix time             (4) |
	 * | random                   (28) |
	 * | session id length         (1) |
	 * | session id                (n) | n >= 0 || n <= 32
	 * | cipher suite length       (1) |
	 * | cipher suite              (n) | n > 0
	 * | compression method length (1) |
	 * | compression method        (n) | n > 0
	 * | extension length          (2) |
	 * | extension                 (n) |
	 */

	msg->type = TLS_HANDSHAKE_CLIENT_HELLO;

	/* write version number string */
	int32_t verlen;
	if ((verlen = write_client_version(tls, msg)) < 0) {
		TLS_DPRINTF("write_client_version");
		goto failed;
	}
	offset += verlen;

	/* write client random. */
	int32_t randlen;
	if ((randlen = write_client_random(tls, msg)) < 0) {
		TLS_DPRINTF("write_client_random");
		goto failed;
	}
	offset += randlen;

	/* write session id. */
	int32_t seslen;
	if ((seslen = write_session_id(tls, msg)) < 0) {
		TLS_DPRINTF("write_session_id");
		goto failed;
	}
	offset += seslen;

	/* write cipher suite. */
	int32_t cslen;
	if ((cslen = write_cipher_suite(tls, msg)) < 0) {
		TLS_DPRINTF("write_cipher_suite");
		goto failed;
	}
	offset += cslen;

	/* write compression method. */
	int32_t cmplen;
	if ((cmplen = write_cmp_method(msg)) < 0) {
		TLS_DPRINTF("write_cmp_method");
		goto failed;
	}
	offset += cmplen;

	/* Extension */
	int32_t extlen;
	if ((extlen = write_ext(tls, msg)) < 0) {
		TLS_DPRINTF("write_ext");
		goto failed;
	}
	offset += extlen;

	/* write length of handshake. */
	msg->len = offset;

	return msg;

failed:
	tls_hs_msg_free(msg);
	return NULL;
}

bool tls_hs_chello_parse(TLS *tls, struct tls_hs_msg *msg) {
	uint32_t offset = 0;

	if (msg->type != TLS_HANDSHAKE_CLIENT_HELLO) {
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 16, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return false;
	}

	int32_t verlen;
	if ((verlen = read_client_version(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_client_version");
		return false;
	}
	offset += verlen;

	int32_t randlen;
	if ((randlen = read_client_random(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_client_random");
		return false;
	}
	offset += randlen;

	int32_t seslen;
	if ((seslen = read_session_id(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_session_id");
		return false;
	}
	offset += seslen;

	int32_t cslen;
	if ((cslen = read_cipher_suite(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_cipher_suite");
		return false;
	}
	offset += cslen;

	int32_t cmplen;
	if ((cmplen = read_cmp_method(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_cmp_method");
		return false;
	}
	offset += cmplen;

	int32_t extlen;
	if ((extlen = tls_hs_extension_parse(tls, msg, offset)) < 0) {
		TLS_DPRINTF("tls_hs_extension_parse");
		return false;
	}
	offset += extlen;

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *                           If negotiating a version of TLS prior to 1.3,
	 *    a server MUST check that the message either contains no data after
	 *    legacy_compression_methods or that it contains a valid extensions
	 *    block with no data following.  If not, then it MUST abort the
	 *    handshake with a "decode_error" alert.
	 */
	if (offset != msg->len) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 17, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return false;
	}

	return true;
}

bool tls_hs_chello_interpret(TLS *tls) {
	if (! interpret_random(tls)) {
		return false;
	}

	if (! interpret_session_id(tls)) {
		return false;
	}

	if (! interpret_cipher(tls)) {
		return false;
	}

	if (! interpret_cmp_method(tls)) {
		return false;
	}

	if (! interpret_ext_list(tls)) {
		return false;
	}

	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		/*
		 * init sighash_list member of tls structure by default value.
		 *
		 * prior to TLS 1.2, signature and hash algorithm negotiation
		 * doesn't exist in specifications, so always use a default
		 * value.
		 *
		 * in TLS 1.2, client can provide available signature and hash
		 * algorithm by signature algorithm extension in client hello.
		 * if it is provided, use a value in the extension. otherwise,
		 * use a default value.
		 */
		if ((tls->sighash_list == NULL) &&
		    (tls_hs_sighash_save_default(tls) == false)) {
			return false;
		}
		break;

	case TLS_VER_TLS13:
		if (tls->sighash_list == NULL) {
			TLS_DPRINTF("chello: missing signature scheme");
			OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
				     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CHELLO + 24, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
			return false;
		}

	default:
		break;
	}

	return true;
}

bool tls_hs_chello_interpret_extensions(TLS *tls) {
	if (! interpret_ext_list(tls)) {
		return false;
	}

	struct tls_extension *ext;
	while (!TAILQ_EMPTY(&(tls->interim_params->head))) {
		ext = TAILQ_FIRST(&(tls->interim_params->head));
		TAILQ_REMOVE(&(tls->interim_params->head), ext, link);
		tls_extension_free(ext);
	}

	return true;
}
