/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_cert.h"
#include "tls_alert.h"

/* for ASN1_read_cert */
#include <aicrypto/ok_asn1.h>

/* for PKCS12. */
#include <aicrypto/ok_pkcs12.h>

#include <string.h>

/**
 * enum to distinguish server certificate and client certificate.
 *
 * since this file deal with server certificate and client certificate
 * lumped, this enum was prepared to distinguish them.
 */
enum tls_cert_t {
	/** server certificate */
	TLS_CERT_SERVER,

	/** client certificate */
	TLS_CERT_CLIENT
};

/**
 * get pkcs12 data from tls structure.
 */
static PKCS12 * get_pkcs12(const enum tls_cert_t type, const TLS *tls);

/**
 * extract a certificate from send certificate handshake protocol
 * message.
 */
static Cert* get_cert(TLS *tls, uint8_t *buf, const uint32_t len);

/**
 * check whether signature scheme passed by arguement is a valid rsa signature
 * scheme for certificate.
 */
static bool check_rsa_sigscheme_availability(uint16_t sigscheme);

/**
 * check whether signature scheme passed by arguement is a valid ecdsa signature
 * scheme for certificate.
 */
static bool check_ecdsa_sigscheme_availability(uint16_t sigscheme,
					       int curve_type);

/**
 * check whether signature scheme passed by argument is a valid rsassa_pss signature
 * scheme for certificate.
 */
static bool check_rsassa_pss_sigscheme_availability(uint16_t sigscheme);

/**
 * search sighash list for rsa valid signature scheme.
 */
static bool search_sighash_for_rsa(struct tls_hs_sighash_list *list);

/**
 * search sighash list for ecdsa valid signature scheme.
 */
static bool search_sighash_for_ecdsa(struct tls_hs_sighash_list *list,
				     int curve_type);

/**
 * search sighash list for rsassa_pss valid signature scheme.
 */
static bool search_sighash_for_rsassa_pss(struct tls_hs_sighash_list *list) UNUSED;

/**
 * check whether the certificate that send matches acceptable
 * signature/hash algorithm for peer.
 */
static bool check_cert_sighash(TLS *tls, Cert* cert,
			       struct tls_hs_sighash_list *sighash_list);

/**
 * check whether acceptable certificate chain is produced via provided
 * signature schemes.
 */
static int check_certchain_produced_by_sighash(
	PKCS12 *p12, struct tls_hs_sighash_list *sighash_list);

/**
 * check certificate of server.
 *
 * This function is used when the client has received the Server Certificate
 * message.
 */
static bool check_cert_server(TLS *tls, PKCS12 *p12);

/**
 * check certificate of client.
 */
static bool check_cert_client(TLS *tls, PKCS12 *p12);

/**
 * check certificate compatibility with signature/hash algorithms.
 */
static bool is_compatible_with_sighash(TLS *tls, PKCS12 *p12,
				       struct tls_hs_sighash_list *sighash_list);

/**
 * get list of signature schemes for fallback.
 */
struct tls_hs_sighash_list *get_fallback_sighash_list(struct tls *tls);

/**
 * Set private key from PKCS12 to TLS.
 */
static int32_t set_privkey_from_pkcs12(TLS *tls, PKCS12 *p12);

/**
 * Set public key from PKCS12 to TLS.
 */
static int32_t set_pubkey_from_pkcs12(TLS *tls, PKCS12 *p12);

/**
 * check whether the received extension is available in this module.
 */
static bool check_ext_availability_tls13(const enum tls_extension_type type);

static bool check_ext_availability(TLS *tls,
				   const enum tls_extension_type type);

/**
 * handle extensions stored in list.
 */
static bool read_ext_list(TLS *tls,
			  const enum tls_extension_type type,
			  const struct tls_hs_msg *msg UNUSED,
			  const uint32_t offset UNUSED);

/**
 * interpret extensions stored in tls_interim_params.
 */
static bool interpret_ext_list(TLS *tls);

/**
 * write the certificate chain as a send data.
 */
static int32_t write_certchain_up_to_tls12(const enum tls_cert_t type,
			       TLS *tls, struct tls_hs_msg *msg);

static int32_t write_certchain_tls13(const enum tls_cert_t type,
			       TLS *tls, struct tls_hs_msg *msg);

/**
 * read the certificate chain from the sent certificate handshake
 * protocol message.
 */
static int32_t read_certchain_up_to_tls12(const enum tls_cert_t type,
			      TLS *tls,
			      struct tls_hs_msg *msg,
			      const uint32_t offset);

static int32_t read_certchain_tls13(const enum tls_cert_t type,
			      TLS *tls,
			      struct tls_hs_msg *msg,
			      const uint32_t offset);

/**
 * write server certificate data to message structure.
 */
static int32_t write_scert(TLS *tls, struct tls_hs_msg *msg);

/**
 * read server certificate data from message structure.
 */
static int32_t read_scert(TLS *tls, struct tls_hs_msg *msg,
			  const uint32_t offset);

/**
 * write client certificate data to message structure.
 */
static int32_t write_ccert(TLS *tls, struct tls_hs_msg *msg);

/**
 * read client certificate data from message structure.
 */
static int32_t read_ccert(TLS *tls, struct tls_hs_msg *msg,
			  const uint32_t offset);

/**
 * byte of request context length part in the certificate handshake
 * protocol message.
 */
static const int32_t TLS_CERT_REQUEST_CONTEXT_LENGTH_BYTES = 1;

/**
 * bytes of certificate chain length part in the certificate handshake
 * protocol message.
 */
static const int32_t TLS_CERT_CHAIN_LENGTH_BYTES = 3;

/**
 * bytes of certificate length part in the certificate handshake
 * protocol message.
 */
static const int32_t TLS_CERT_LENGTH_BYTES = 3;

/**
 * bytes of extension length part in the certificate handshake
 * protocol message.
 */
static const int32_t TLS_CERT_EXT_LENGTH_BYTES = 2;

static PKCS12 * get_pkcs12(const enum tls_cert_t type, const TLS *tls) {
	switch (type) {
	case TLS_CERT_CLIENT:
		return tls->pkcs12_client;
		break;

	case TLS_CERT_SERVER:
		return tls->pkcs12_server;

	default:
		assert(!"unknown cert type.");
	}

	return NULL;
}

static Cert* get_cert(TLS *tls, uint8_t *buf, const uint32_t len) {
	uint8_t *der;

	if ((der = malloc (len * sizeof(uint8_t))) == NULL) {
		TLS_DPRINTF("hs: m: cert: malloc %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	memcpy(der, &(buf[0]), len);

	/* variable der is set by internal of ASN1_read_cert() to the
	 * cert structure member der (cert->der). To free both cert and
	 * der, use Cert_free function. */
	Cert *cert;
	if ((cert = ASN1_read_cert(der)) == NULL) {
		TLS_DPRINTF("hs: m: cert: ASN1_read_cert");
		OK_set_error(ERR_ST_TLS_ASN1_READ_CERT,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_BAD_CERTIFICATE);
		free(der);
		return NULL;
	}

	return cert;
}

static bool check_rsa_sigscheme_availability(uint16_t sigscheme) {
	switch (sigscheme) {
	case TLS_SS_RSA_PSS_RSAE_SHA256:
	case TLS_SS_RSA_PSS_RSAE_SHA384:
	case TLS_SS_RSA_PSS_RSAE_SHA512:
		return true;

	default:
		return false;
	}
}

static bool check_ecdsa_sigscheme_availability(uint16_t sigscheme,
					       int curve_type) {
	uint16_t expected_ss = TLS_SS_ANON_NONE;
	switch (tls_hs_ecdh_get_named_curve(curve_type)) {
	case TLS_ECC_CURVE_SECP256R1:
		expected_ss = TLS_SS_ECDSA_SECP256R1_SHA256;
		break;

	case TLS_ECC_CURVE_SECP384R1:
		expected_ss = TLS_SS_ECDSA_SECP384R1_SHA384;
		break;

	case TLS_ECC_CURVE_SECP521R1:
		expected_ss = TLS_SS_ECDSA_SECP521R1_SHA512;
		break;

	default:
		return false;
	}

	return expected_ss == sigscheme ? true : false;
}

static bool check_rsassa_pss_sigscheme_availability(uint16_t sigscheme) {
	switch (sigscheme) {
	case TLS_SS_RSA_PSS_PSS_SHA256:
	case TLS_SS_RSA_PSS_PSS_SHA384:
	case TLS_SS_RSA_PSS_PSS_SHA512:
		return true;

	default:
		return false;
	}
}

static bool search_sighash_for_rsa(struct tls_hs_sighash_list *list) {
	uint16_t ss;
	for (int i = 0; i < list->len; i++) {
		ss = tls_hs_sighash_convert_sighash_to_sigscheme(
		    &(list->list[i]));
		if (check_rsa_sigscheme_availability(ss) == true) {
			return true;
		}
	}

	return false;
}

static bool search_sighash_for_ecdsa(struct tls_hs_sighash_list *list,
				     int curve_type) {
	uint16_t ss;
	for (int i = 0; i < list->len; i++) {
		ss = tls_hs_sighash_convert_sighash_to_sigscheme(
		    &(list->list[i]));
		if (check_ecdsa_sigscheme_availability(ss, curve_type)
		    == true) {
			return true;
		}
	}

	return false;
}

UNUSED
static bool search_sighash_for_rsassa_pss(struct tls_hs_sighash_list *list) {
	uint16_t ss;
	for (int i = 0; i < list->len; i++) {
		ss = tls_hs_sighash_convert_sighash_to_sigscheme(
		    &(list->list[i]));
		if (check_rsassa_pss_sigscheme_availability(ss) == true) {
			return true;
		}
	}

	return false;
}

static bool check_cert_sighash(TLS *tls, Cert* cert,
			       struct tls_hs_sighash_list *sighash_list) {
	for (int i = 0; i < sighash_list->len; ++i) {
		struct tls_hs_sighash_algo sighash = sighash_list->list[i];

		int ai_sig;
		if ((ai_sig = tls_hs_sighash_get_ai_sig_type(sighash)) < 0) {
			TLS_DPRINTF("hs: m: cert: unsupported sig type: %d, %d",
				    sighash.sig, sighash.hash);
			OK_set_error(ERR_ST_TLS_AICRYPTO_SIG_TYPE,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 2, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return false;
		}

		if (cert->signature_algo == ai_sig) {
			return true;
		}
	}

	TLS_DPRINTF("hs: m: cert: unsupported cert");
	OK_set_error(ERR_ST_TLS_BAD_CERT,
		     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 3, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
	return false;
}

static int check_certchain_produced_by_sighash(
	PKCS12 *p12, struct tls_hs_sighash_list *sighash_list) {
	int32_t depth;
	if ((depth = P12_max_depth(p12, OBJ_P12v1Bag_CERT)) < 0) {
		TLS_DPRINTF("P12_max_depth");
		OK_set_error(ERR_ST_TLS_P12_MAX_DEPTH,
			     ERR_LC_TLS6,
			     ERR_PT_TLS_HS_MSG_CERT2 + 0, NULL);
		return -1;
	}

	for (int i = depth; i >= 0; --i) {
		P12_Baggage *bag;
		if ((bag = P12_find_bag(p12,
					OBJ_P12v1Bag_CERT, (char)i)) == NULL) {
			TLS_DPRINTF("P12_find_bag");
			OK_set_error(ERR_ST_TLS_P12_FIND_BAG,
				     ERR_LC_TLS6,
				     ERR_PT_TLS_HS_MSG_CERT2 + 1, NULL);
			return -1;
		}

		P12_CertBag *cbag = (P12_CertBag *)bag;
		Cert *cert = (cbag)->cert;

		for (int j = 0; j < sighash_list->len; ++j) {
			struct tls_hs_sighash_algo sighash = sighash_list->list[j];

			int ai_sig;
			if ((ai_sig = tls_hs_sighash_get_ai_sig_type(sighash)) < 0) {
				TLS_DPRINTF("tls_hs_sighash_get_ai_sig_type");
				return -1;
			}

			if (cert->signature_algo == ai_sig) {
				goto found;
			}
		}

		return 0;

	found:
		;
	}

	return 1;
}

static bool check_cert_server(TLS *tls, PKCS12 *p12)
{
	struct tls_cert_info cert_info;
	struct tls_hs_ecc_eclist eclist;
	struct tls_hs_ecc_pflist pflist;

	if (tls_cert_info_get(p12, &cert_info) != true) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	/*
	 * Whether the received certificate can use the ECC cipher.
	 * Compare with supported extension.
	 */
	tls_hs_ecc_get_supported_eclist(&eclist);
	tls_hs_ecc_get_supported_pflist(&pflist);
	tls->ecdh->eclist = &eclist;

	bool can_use_ecc;
	can_use_ecc = tls_cert_info_can_use_ecc_cipher_suite(&cert_info,
							     &eclist, &pflist);

TLS_DPRINTF("hs: m: cert: can_use_ecc=%d", can_use_ecc);
	if (tls_cert_info_available(tls->pending->cipher_suite, &cert_info,
				    can_use_ecc, true) == true) {
		switch (tls->keymethod) {
		case TLS_KXC_ECDHE_RSA:
			tls_hs_ecdh_set_curve(tls->ecdh);
			break;

		case TLS_KXC_ECDH_RSA:
			tls_hs_ecdh_set_curve_by_cert(tls->ecdh, &cert_info);
			if (tls->ecdh->namedcurve == 0) {
				OK_set_error(ERR_ST_TLS_UNSUPPORTED_CURVE,
					     ERR_LC_TLS2,
					     ERR_PT_TLS_HS_MSG_CERT + 26, NULL);
				tls->ecdh->eclist = NULL;
				return false;
			}
			break;

		default:
			/* Nothing to do */
			break;
		}
	} else {
		tls->ecdh->eclist = NULL;
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 2,
			     NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}
	tls->ecdh->eclist = NULL;

	return true;
}

static bool check_cert_client(TLS *tls, PKCS12 *p12) {
	/* get end entity certificate */
	Cert* usercert;
	if ((usercert = P12_get_usercert(p12)) == NULL) {
		TLS_DPRINTF("hs: m: cert: P12_get_usercert");
		OK_set_error(ERR_ST_TLS_P12_GET_USERCERT,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	/* get certificate list that sent in the certificate request.
	 * tls->certtype_list_server is not usable. that member is for
	 * client. */
	struct tls_cert_type_list *certtypes;
	if ((certtypes =  tls_cert_type_list(tls)) == NULL) {
		return false;
	}

	bool ret = false;
	for (int i = 0; i < certtypes->len; ++i) {
		enum tls_cert_type type = certtypes->list[i];
		switch(type) {
		case TLS_CERT_RSA_SIGN:
			/* RFC 5246 section 7.4.6 says
			 *
			 *  RSA public key; the certificate MUST allow the
			 *  key to be used for signing with the signature
			 *  scheme and hash algorithm that will be
			 *  employed in the certificate verify message. */
			if (usercert->pubkey_algo != KEY_RSA_PUB) {
				TLS_DPRINTF("hs: m: cert: pubkey algo");
				OK_set_error(ERR_ST_TLS_NOT_RSA_PUB,
					     ERR_LC_TLS2,
					     ERR_PT_TLS_HS_MSG_CERT + 9, NULL);
				goto done;
			}

			if (usercert->ext == NULL) {
				ret = true;
				goto done;
			}

			for (CertExt *ext = usercert->ext; ext != NULL;
			     ext = ext->next) {
				if (ext->extnID != OBJ_X509v3_KEY_Usage) {
					continue;
				}

				CE_KUsage *ext_keyusage = (CE_KUsage*)ext;
				/* digitalSignature only? */
				if ((ext_keyusage->flag & digitalSignature)
				    == 0) {
					TLS_DPRINTF("hs: m: cert: digitalSignature bit");
					OK_set_error(
						ERR_ST_TLS_CERT_FLAG_DIGITALSIGNATURE,
						ERR_LC_TLS2,
						ERR_PT_TLS_HS_MSG_CERT + 10,
						NULL);
					TLS_ALERT_FATAL(
						tls,
						TLS_ALERT_DESC_BAD_CERTIFICATE);
					goto done;
				}
			}
			ret = true;
			goto done;

		case TLS_CERT_DSS_SIGN:
		case TLS_CERT_RSA_FIXED_DH:
		case TLS_CERT_DSS_FIXED_DH:
			/* TODO: not implementation. */
			TLS_DPRINTF("hs: m: cert: "
				    "unsupported client certificate type");
			break;

		default:
			TLS_DPRINTF("hs: m: cert: "
				    "unknown client certificate type");
			break;
		}
	}

	TLS_DPRINTF("hs: m: cert: unsupported cert");
	OK_set_error(ERR_ST_TLS_UNSUPPORTED_CERT,
		     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 11, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);

done:
	tls_cert_type_free(certtypes);
	return ret;
}

static bool is_compatible_with_sighash(TLS *tls, PKCS12 *p12,
				       struct tls_hs_sighash_list *sighash_list) {
	struct tls_cert_info cert_info;

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *            ...   The sender's certificate MUST come in the first
	 *    CertificateEntry in the list.  Each following certificate SHOULD
	 *    directly certify the one immediately preceding it.  Because
	 *    certificate validation requires that trust anchors be distributed
	 *    independently, a certificate that specifies a trust anchor MAY be
	 *    omitted from the chain, provided that supported peers are known to
	 *    possess any omitted certificates.
	 *
	 *    Note: Prior to TLS 1.3, "certificate_list" ordering required each
	 *    certificate to certify the one immediately preceding it; however,
	 *    some implementations allowed some flexibility.  Servers sometimes
	 *    send both a current and deprecated intermediate for transitional
	 *    purposes, and others are simply configured incorrectly, but these
	 *    cases can nonetheless be validated properly.  For maximum
	 *    compatibility, all implementations SHOULD be prepared to handle
	 *    potentially extraneous certificates and arbitrary orderings from any
	 *    TLS version, with the exception of the end-entity certificate which
	 *    MUST be first.
	 */
	if (tls_cert_info_get(p12, &cert_info) == false) {
		TLS_DPRINTF("hs: m: cert: tls_cert_info_get");
		return false;
	}

	/*
	 * RFC8446 4.4.2.2.  Server Certificate Selection
	 *
	 *    -  The server's end-entity certificate's public key (and associated
	 *       restrictions) MUST be compatible with the selected authentication
	 *       algorithm from the client's "signature_algorithms" extension
	 *       (currently RSA, ECDSA, or EdDSA).
	 *
	 * RFC8446 4.4.2.3.  Client Certificate Selection
	 *
	 *    -  The certificates MUST be signed using an acceptable signature
	 *       algorithm, as described in Section 4.3.2.  Note that this relaxes
	 *       the constraints on certificate-signing algorithms found in prior
	 *       versions of TLS.
	 */
	switch (cert_info.pubkey_algo) {
	case KEY_RSA_PUB:
		if (search_sighash_for_rsa(sighash_list) == false) {
			TLS_DPRINTF("hs: m: cert: key doesn't match signature"
				    " algorighm");
			OK_set_error(ERR_ST_TLS_UNMATCH_SIGHASH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 3,
				     NULL);
			return false;
		}

		TLS_DPRINTF("hs: m: cert: RSA certificate");
		break;

	case KEY_ECDSA_PUB:
		if (search_sighash_for_ecdsa(sighash_list, cert_info.curve_type)
		    == false) {
			TLS_DPRINTF("hs: m: cert: key doesn't match signature"
				    " algorighm");
			OK_set_error(ERR_ST_TLS_UNMATCH_SIGHASH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 4,
				     NULL);
			return false;
		}

		TLS_DPRINTF("hs: m: cert: ECDSA certificate");
		break;

	default:
		/* TODO: ED25519, ED448 and RSASSA-PSS are not supported */
		TLS_DPRINTF("hs: m: cert: unknown certificate type");
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_CERT,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 5, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
		return false;
	}

	/*
	 * RFC8446 4.4.2.2.  Server Certificate Selection
	 *
	 *    -  The certificate MUST allow the key to be used for signing (i.e.,
	 *       the digitalSignature bit MUST be set if the Key Usage extension is
	 *       present) with a signature scheme indicated in the client's
	 *       "signature_algorithms"/"signature_algorithms_cert" extensions (see
	 *       Section 4.2.3).
	 */
	if ((cert_info.keyusage & digitalSignature) == 0) {
		TLS_DPRINTF("hs: m: cert: digitalSignature bit down");
		OK_set_error(ERR_ST_TLS_CERT_FLAG_DIGITALSIGNATURE,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 6, NULL);
		return false;
	}

	return true;
}

struct tls_hs_sighash_list *get_fallback_sighash_list(struct tls *tls) {
	struct tls_hs_sighash_list *fallback_list;

	if ((fallback_list = tls_hs_sighash_list_cert(tls)) == NULL) {
		TLS_DPRINTF("tls_hs_sighash_list_cert");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	struct tls_hs_sighash_algo list_tmp[fallback_list->len];
	uint16_t actlen = 0;

	uint16_t sigscheme;
	for (int i = 0; i < fallback_list->len; i++) {
		sigscheme = tls_hs_sighash_convert_sighash_to_sigscheme(
			&(fallback_list->list[i]));

		switch (sigscheme) {
		case TLS_SS_RSA_PCKS1_SHA1:
		case TLS_SS_ECDSA_SHA1:
			continue;

		default:
			break;
		}

		list_tmp[actlen] = fallback_list->list[i];
		actlen++;
	}

	fallback_list->len = actlen;
	size_t length = actlen * sizeof(struct tls_hs_sighash_algo);
	memcpy(fallback_list->list, list_tmp, length);

	return fallback_list;
}

static int32_t set_privkey_from_pkcs12(TLS *tls, PKCS12 *p12)
{
	switch (tls->keymethod) {
	case TLS_KXC_ECDH_ECDSA:
	case TLS_KXC_ECDH_RSA:
		if (!tls_hs_ecdh_set_privkey_from_pkcs12(tls, p12)) {
			TLS_DPRINTF("hs: m: cert: XXX");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
	default:
		/* Nothing to do. */
		;
	}
	return true;
}

static int32_t set_pubkey_from_pkcs12(TLS *tls, PKCS12 *p12)
{
	switch (tls->keymethod) {
	case TLS_KXC_ECDH_ECDSA:
	case TLS_KXC_ECDH_RSA:
		if (!tls_hs_ecdh_set_pubkey_from_pkcs12(tls, p12)) {
			TLS_DPRINTF("hs: m: cert: XXX");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return false;
		}
		;
	default:
		/* Nothing to do. */
		;
	}
	return true;
}

static bool check_ext_availability_tls13(const enum tls_extension_type type) {
	switch(type) {
	case TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP:
	case  TLS_EXT_STATUS_REQUEST:
		return true;

	default:
		break;
	}

	return false;
}

static bool check_ext_availability(TLS *tls,
				   const enum tls_extension_type type) {
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));

	switch(version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		/* Not supported */
		break;

	case TLS_VER_TLS13:
		return check_ext_availability_tls13(type);

	default:
		/* Unknown version */
		break;
	}

	return false;
}

static bool read_ext_list(TLS *tls,
			  const enum tls_extension_type type,
			  const struct tls_hs_msg *msg UNUSED,
			  const uint32_t offset UNUSED)
{
	/*
	 * assume unknown extensions never come because check
	 * is performed before this function is called.
	 */
	/*
	 * RFC8446 4.2.  Extensions
	 *
	 *                        There MUST NOT be more than one extension of the
	 *    same type in a given extension block.
	 */
	bool *recv_exts = tls->interim_params->recv_ext_flags;
	if (recv_exts[type] == true) {
		TLS_DPRINTF("cert: extensions of same type come multiple times");
		OK_set_error(ERR_ST_TLS_SAME_TYPE_EXTENSION,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 7, NULL);
		return false;
	}
	recv_exts[type] = true;

	switch(type) {
	case TLS_EXT_SIGNED_CERTIFICATE_TIMESTAMP:
	case TLS_EXT_STATUS_REQUEST:
		/* Not implemented */
		return false;

	default:
		assert(!"unknown extension type");
	}

	return false;
}

static bool interpret_ext_list(TLS *tls) {
	struct tls_hs_interim_params *params = tls->interim_params;
	struct tls_extension *ext;
	struct tls_hs_msg msg;

	TAILQ_FOREACH(ext, &(params->head), link) {
		if (check_ext_availability(tls, ext->type) == false) {
			OK_set_error(ERR_ST_TLS_UNSUPPORTED_EXTENSION,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 8,
				     NULL);
			TLS_ALERT_FATAL(tls,
				TLS_ALERT_DESC_UNSUPPORTED_EXTENSION);
			return false;
		}

		msg.type = TLS_HANDSHAKE_CERTIFICATE;
		msg.len = ext->len;
		msg.max = ext->len;
		msg.msg = ext->opaque;

		if (! read_ext_list(tls, ext->type, &msg, 0)) {
			/* alerts is sent by internal of tls_hs_ext_read. */
			return false;
		}
	}

	while (!TAILQ_EMPTY(&(params->head))) {
		ext = TAILQ_FIRST(&(params->head));
		TAILQ_REMOVE(&(params->head), ext, link);
		tls_extension_free(ext);
	}

	return true;
}

static int32_t write_certchain_up_to_tls12(const enum tls_cert_t type,
			       TLS *tls,
			       struct tls_hs_msg *msg) {
	int32_t offset = 0;

	PKCS12 *p12 = get_pkcs12(type, tls);
	if (type == TLS_CERT_SERVER) {
		if (p12 == NULL) {
			/* do not permit null certificate in the server
			 * certificate in this function.  null server
			 * certificate is possible in the case of key exchange
			 * method is anonymouse dh. but, since the check is over
			 * beforehand, do not care that. */
			TLS_DPRINTF("hs: m: cert: unsupported cert");
			OK_set_error(ERR_ST_TLS_PKCS12_SERVER_NULL,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 12, NULL);
			return -1;
		}

		/*
		 * NOTE: Previously, there was check of the server certificate
		 * (check_cert_server) here. That process has been changed to
		 * run when the ClientHello message received. Therefore, there
		 * is no need to check here.
		 */

		/*
		 * Get a private key associated with the certificate
		 * that to be sent. It use after parsing a
		 * ClientKeyExchange message.
		 */
		if (set_privkey_from_pkcs12(tls, p12) == false) {
			return -1;
		}
	}

	/* write dummy (zero) certificate chain length byte. */
	int32_t pos = msg->len;
	if (! tls_hs_msg_write_3(msg, 0)) {
		return -1;
	}
	offset += TLS_CERT_CHAIN_LENGTH_BYTES;

	/* if application try to send client certificate, and do not
	 * have client certificate, send null certificate. */
	if (type == TLS_CERT_CLIENT) {
		if (p12 == NULL) {
			TLS_DPRINTF("hs: m: cert: NULL");
			goto done;
		}
		if (check_cert_client(tls, p12) == false) {
			return -1;
		}
	}

	int32_t depth;
	if ((depth = P12_max_depth(p12, OBJ_P12v1Bag_CERT)) < 0) {
		TLS_DPRINTF("hs: m: cert: P12_max_depth");
		OK_set_error(ERR_ST_TLS_P12_MAX_DEPTH,
			     ERR_LC_TLS2,
			     ERR_PT_TLS_HS_MSG_CERT + 13, NULL);
		return -1;
	}

	for (int i = depth; i >= 0; --i) {
		P12_Baggage *bag;
		if ((bag = P12_find_bag(p12,
					OBJ_P12v1Bag_CERT, (char)i)) == NULL) {
			TLS_DPRINTF("hs: m: cert: P12_find_bag");
			OK_set_error(ERR_ST_TLS_P12_FIND_BAG,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 14, NULL);
			return -1;
		}

		P12_CertBag *cbag = (P12_CertBag *)bag;

		Cert *cert = (cbag)->cert;

		int32_t len = tls_util_asn1_length(&(cert->der[1]));
		if (! tls_hs_msg_write_3(msg, len)) {
			return -1;
		}
		offset += TLS_CERT_LENGTH_BYTES;

		TLS_DPRINTF("hs: m: cert: len     = %d", len);
		TLS_DPRINTF("hs: m: cert: version = %d (0x%x)",
			    cert->version + 1, cert->version);
		TLS_DPRINTF("hs: m: cert: sig     = %d", cert->signature_algo);
		TLS_DPRINTF("hs: m: cert: pubkey  = %d", cert->pubkey_algo);

		uint16_t version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_TLS12: {
			/* x509 certificate version representation is
			 * zero-origin. use the int instead of uint32_t
			 * because cert->version is int. */
			const int version_of_x509v3 = 3 - 1;
			if (cert->version != version_of_x509v3) {
				TLS_DPRINTF("hs: m: cert: x509 version != 3");
				OK_set_error(ERR_ST_TLS_X509_VERSION,
					     ERR_LC_TLS2,
					     ERR_PT_TLS_HS_MSG_CERT + 15, NULL);
				TLS_ALERT_FATAL(
					tls,
					TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
				return -1;
			}

			if ((tls->sighash_list != NULL) &&
			    (check_cert_sighash(tls, cert, tls->sighash_list)
				== false)) {
				return -1;
			}
		}
			break;

		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		default:
			break;
		}

		if (! tls_hs_msg_write_n(msg, cert->der, len)) {
			return -1;
		}
		offset += len;
	}

	/* write certificate chain length */
	uint32_t chainlen = offset - TLS_CERT_CHAIN_LENGTH_BYTES;

	/*
	 * RFC5246 7.4.2.  Server Certificate
	 *
	 *           ASN.1Cert certificate_list<0..2^24-1>;
	 */
	uint32_t list_length_max = TLS_VECTOR_3_BYTE_SIZE_MAX;
	if (chainlen > list_length_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 4, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	tls_util_write_3(&(msg->msg[pos]), chainlen);

done:
	TLS_DPRINTF("hs: m: cert: chain len = %d", offset);

	return offset;
}

static int32_t write_certchain_tls13(const enum tls_cert_t type,
			       TLS *tls,
			       struct tls_hs_msg *msg) {
	int32_t offset = 0;
	int32_t start = msg->len;

	/*
	 * certificate message has following structure.
	 *
	 * | certificate_request_context (n) | n >= 0 || n <= 2^8 -1
	 * | certificate_list            (n) | n >= 0 || n <= 2^24 -1
	 */

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *    certificate_request_context:  If this message is in response to a
	 *       CertificateRequest, the value of certificate_request_context in
	 *       that message.  Otherwise (in the case of server authentication),
	 *       this field SHALL be zero length.
	 */
	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *           opaque certificate_request_context<0..2^8-1>;
	 */
	uint8_t reqctxlen = 0;
	uint8_t *reqctx = NULL;
	if (type == TLS_CERT_CLIENT) {
		/*
		 * TODO: set length of certificate_request_context
		 * and the data.
		 */
	}

	if (tls_hs_msg_write_1(msg, reqctxlen) == false) {
		TLS_DPRINTF("hs: m: cert: tls_hs_msg_write_1");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}
	offset += TLS_CERT_REQUEST_CONTEXT_LENGTH_BYTES;

	if (reqctxlen > 0 && reqctx != NULL) {
		if (tls_hs_msg_write_n(msg, reqctx, reqctxlen) == false) {
			TLS_DPRINTF("hs: m: cert: tls_hs_msg_write_n");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		offset += reqctxlen;
	}

	/* write dummy (zero) certificate chain length byte. */
	int32_t pos = msg->len;
	if (! tls_hs_msg_write_3(msg, 0)) {
		return -1;
	}
	offset += TLS_CERT_CHAIN_LENGTH_BYTES;

	PKCS12 *p12 = get_pkcs12(type, tls);
	switch (type) {
	case TLS_CERT_SERVER:
		if (p12 == NULL) {
			/*
			 * RFC8446 4.4.2.  Certificate
			 *
			 *    The server MUST send a Certificate message whenever the agreed-upon
			 *    key exchange method uses certificates for authentication (this
			 *    includes all key exchange methods defined in this document
			 *    except PSK).
			 */
			TLS_DPRINTF("hs: m: cert: missing certificate");
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 9,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		break;

	case TLS_CERT_CLIENT:
		if (p12 == NULL) {
			/*
			 *  RFC8446 4.4.2.  Certificate
			 *
			 *                   .. If the server requests client authentication but no
			 *    suitable certificate is available, the client MUST send a Certificate
			 *    message containing no certificates (i.e., with the "certificate_list"
			 *    field having length 0).
			 */
			TLS_DPRINTF("hs: m: cert: NULL");
			goto done;
		}
		break;

	default:
		TLS_DPRINTF("hs: m: cert: unknown entity");
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 10, NULL);
		return -1;
	}

	/*
	 * TODO: consider rules below when selecting certificate to be sent.
	 *
	 * RFC8446 4.4.2.2.  Server Certificate Selection
	 *
	 *    -  The "server_name" [RFC6066] and "certificate_authorities"
	 *       extensions are used to guide certificate selection.  As servers
	 *       MAY require the presence of the "server_name" extension, clients
	 *       SHOULD send this extension, when applicable.
	 *
	 * RFC8446 4.4.2.3.  Client Certificate Selection
	 *
	 *    -  If the "certificate_authorities" extension in the
	 *       CertificateRequest message was present, at least one of the
	 *       certificates in the certificate chain SHOULD be issued by one of
	 *       the listed CAs.
	 *
	 *    -  The certificates MUST be signed using an acceptable signature
	 *       algorithm, as described in Section 4.3.2.  Note that this relaxes
	 *       the constraints on certificate-signing algorithms found in prior
	 *       versions of TLS.
	 *
	 *    -  If the CertificateRequest message contained a non-empty
	 *       "oid_filters" extension, the end-entity certificate MUST match the
	 *       extension OIDs that are recognized by the client, as described in
	 *       Section 4.2.5.
	 */

	/*
	 * check if public key in end-entity certificate is compatible with
	 * one of signature scheme in tls->sighash_list. this list is
	 * intersection of the schemes in "signature_algorithms"
	 * extension sent by peer and own supported signature schemes.
	 */
	bool fallback = false;
	if (is_compatible_with_sighash(tls, p12, tls->sighash_list)
	    == false) {
		switch (type) {
		case TLS_CERT_SERVER:
			fallback = true;
			break;

		case TLS_CERT_CLIENT:
		default:
			TLS_DPRINTF("hs: m: cert: the pubkey of cert is"
				    " uncomaptible with signature algorithms");
			TLS_ALERT_FATAL(
				tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
			return -1;
		}
	}

	/*
	 * check if trust chain can be produced via signature schemes provided
	 * by peer.
	 */
	int ret = check_certchain_produced_by_sighash(p12,
		tls->sighash_list_cert);
	if (ret == 0) {
		switch (type) {
		case TLS_CERT_SERVER:
			fallback = true;
			break;

		case TLS_CERT_CLIENT:
		default:
			TLS_DPRINTF("hs: m: cert: can't produce cert chain via"
				    " signature algorithms");
			TLS_ALERT_FATAL(
				tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
			return -1;
		}
	} else if (ret < 0) {
		TLS_DPRINTF("check_certchain_produced_by_sighash");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.4.2.2.  Server Certificate Selection
	 *
	 *    All certificates provided by the server MUST be signed by a signature
	 *    algorithm advertised by the client if it is able to provide such a
	 *    chain (see Section 4.2.3).  Certificates that are self-signed or
	 *    certificates that are expected to be trust anchors are not validated
	 *    as part of the chain and therefore MAY be signed with any algorithm.
	 *
	 *    If the server cannot produce a certificate chain that is signed only
	 *    via the indicated supported algorithms, then it SHOULD continue the
	 *    handshake by sending the client a certificate chain of its choice
	 *    that may include algorithms that are not known to be supported by the
	 *    client.  This fallback chain SHOULD NOT use the deprecated SHA-1 hash
	 *    algorithm in general, but MAY do so if the client's advertisement
	 *    permits it, and MUST NOT do so otherwise.
	 */
	if (type == TLS_CERT_SERVER && fallback == true) {
		/*
		 * fallback happenes. try one more check. if the check
		 * is not passed, abort handshake.
		 */
		struct tls_hs_sighash_list *fallback_list;
		if ((fallback_list = get_fallback_sighash_list(tls)) == NULL) {
			TLS_DPRINTF("get_fallback_sighash_list");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		ret = check_certchain_produced_by_sighash(p12, fallback_list);
		tls_hs_sighash_free(fallback_list);
		if (ret == 0) {
			TLS_DPRINTF("hs: m: cert: cert chain can't be"
				    " produced");
			TLS_ALERT_FATAL(
				tls, TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
			return -1;
		} else if (ret < 0) {
			TLS_ALERT_FATAL(
				tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
	}

	int32_t depth;
	if ((depth = P12_max_depth(p12, OBJ_P12v1Bag_CERT)) < 0) {
		TLS_DPRINTF("hs: m: cert: P12_max_depth");
		OK_set_error(ERR_ST_TLS_P12_MAX_DEPTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 11, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	for (int i = depth; i >= 0; --i) {
		P12_Baggage *bag;
		if ((bag = P12_find_bag(p12,
					OBJ_P12v1Bag_CERT, (char)i)) == NULL) {
			TLS_DPRINTF("hs: m: cert: P12_find_bag");
			OK_set_error(ERR_ST_TLS_P12_FIND_BAG,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 12,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		P12_CertBag *cbag = (P12_CertBag *)bag;

		Cert *cert = (cbag)->cert;

		int32_t len = tls_util_asn1_length(&(cert->der[1]));
		if (tls_hs_msg_write_3(msg, len) == false) {
			TLS_DPRINTF("hs: m: cert: tls_hs_msg_write_3");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		offset += TLS_CERT_LENGTH_BYTES;

		/*
		 * RFC8446 4.4.2.  Certificate
		 *
		 *                 opaque cert_data<1..2^24-1>;
		 */
		const int32_t cert_length_min = 1;
		const int32_t cert_length_max = TLS_VECTOR_3_BYTE_SIZE_MAX;
		if (len < cert_length_min || cert_length_max < len) {
			TLS_DPRINTF("hs: m: cert: invalid entry length");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 29, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		TLS_DPRINTF("hs: m: cert: len     = %d", len);
		TLS_DPRINTF("hs: m: cert: version = %d (0x%x)",
			    cert->version + 1, cert->version);
		TLS_DPRINTF("hs: m: cert: sig     = %d", cert->signature_algo);
		TLS_DPRINTF("hs: m: cert: pubkey  = %d", cert->pubkey_algo);

		/* x509 certificate version representation is
		 * zero-origin. use the int instead of uint32_t
		 * because cert->version is int. */
		const int version_of_x509v3 = 3 - 1;
		if (cert->version != version_of_x509v3) {
			TLS_DPRINTF("hs: m: cert: x509 version != 3");
			OK_set_error(ERR_ST_TLS_X509_VERSION,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 13,
				     NULL);
			TLS_ALERT_FATAL(tls,
				TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
			return -1;
		}

		if (tls_hs_msg_write_n(msg, cert->der, len) == false) {
			return -1;
		}
		offset += len;

		/*
		 * RFC8446 4.4.2.  Certificate
		 *
		 *              Extensions in the Certificate message from the server MUST
		 *       correspond to ones from the ClientHello message.  Extensions in
		 *       the Certificate message from the client MUST correspond to
		 *       extensions in the CertificateRequest message from the server.  If
		 *       an extension applies to the entire chain, it SHOULD be included in
		 *       the first CertificateEntry.
		 */
		/*
		 * RFC8446 4.4.2.  Certificate
		 *
		 *           Extension extensions<0..2^16-1>;
		 */
		/* write extension per certificate */
		uint16_t extlen = 0;
		if (tls_hs_msg_write_2(msg, extlen) == false) {
			TLS_DPRINTF("hs: m: cert: tls_hs_msg_write_2");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		offset += TLS_CERT_EXT_LENGTH_BYTES;

		TLS_DPRINTF("hs: m: cert: extention len = %d", extlen);

		if (extlen > 0) {
			/* TODO: implement extension related to certificate */
			uint8_t *ext = NULL;
			if (tls_hs_msg_write_n(msg, ext, extlen) == false) {
				TLS_DPRINTF("hs: m: cert: tls_hs_msg_write_n");
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_INTERNAL_ERROR);
				return -1;
			}
			offset += extlen;
		}
	}

	/* write certificate chain length */
	uint32_t chainlen = offset - (pos - start) - TLS_CERT_CHAIN_LENGTH_BYTES;

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *           CertificateEntry certificate_list<0..2^24-1>;
	 */
	uint32_t list_length_max = TLS_VECTOR_3_BYTE_SIZE_MAX;
	if (chainlen > list_length_max) {
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 30, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	tls_util_write_3(&(msg->msg[pos]), chainlen);

done:
	TLS_DPRINTF("hs: m: cert: chain len = %d", offset);

	return offset;
}

static int32_t read_certchain_up_to_tls12(const enum tls_cert_t type,
			      TLS *tls,
			      struct tls_hs_msg *msg,
			      const uint32_t offset) {
	uint32_t read_bytes = 0;

	if (msg->len < (offset + TLS_CERT_CHAIN_LENGTH_BYTES)) {
		TLS_DPRINTF("hs: m: cert: record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2,
			     ERR_PT_TLS_HS_MSG_CERT + 16, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC5246 7.4.2.  Server Certificate
	 *
	 *           ASN.1Cert certificate_list<0..2^24-1>;
	 */
	uint32_t chainlen = tls_util_read_3(&(msg->msg[offset]));
	read_bytes += TLS_CERT_CHAIN_LENGTH_BYTES;

	/* if server received null certificate in the client certificate
	 * handshake protocol message, return immediately. */
	if ((type == TLS_CERT_CLIENT) && (chainlen == 0)) {
		goto done;
	}

	PKCS12 *p12;
	if ((p12 = P12_new()) == NULL) {
		TLS_DPRINTF("hs: m: cert: P12_new");
		OK_set_error(ERR_ST_TLS_P12_NEW,
			     ERR_LC_TLS2,
			     ERR_PT_TLS_HS_MSG_CERT + 17, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	TLS_DPRINTF("hs: m: cert: chain len = %d", chainlen);
	while (read_bytes < (chainlen + TLS_CERT_CHAIN_LENGTH_BYTES)) {
		if (msg->len < (offset + read_bytes + TLS_CERT_LENGTH_BYTES)) {
			TLS_DPRINTF("hs: m: cert: record length");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 18, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			goto err;
		}

		const int32_t certlen = tls_util_read_3(
			&(msg->msg[offset + read_bytes]));
		read_bytes += TLS_CERT_LENGTH_BYTES;

		if (msg->len < (offset + read_bytes + certlen)) {
			TLS_DPRINTF("hs: m: cert: record length");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 19, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			goto err;
		}
		TLS_DPRINTF("hs: m: cert: len     = %d", certlen);

		Cert *cert;
		if ((cert = get_cert(tls,
				     &(msg->msg[offset + read_bytes]),
				     certlen)) == NULL) {
			goto err;
		}

		TLS_DPRINTF("hs: m: cert: version = %d (0x%x)",
			    cert->version + 1, cert->version);
		TLS_DPRINTF("hs: m: cert: sig     = %d", cert->signature_algo);
		TLS_DPRINTF("hs: m: cert: pubkey  = %d", cert->pubkey_algo);

		uint16_t version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_TLS12: {
			/* x509 certificate version representation is
			 * zero-origin. use the int instead of uint32_t
			 * because cert->version is int. */
			const int version_of_x509v3 = 3 - 1;
			if (cert->version != version_of_x509v3) {
				TLS_DPRINTF("hs: m: cert: x509 version != 3");
				OK_set_error(ERR_ST_TLS_X509_VERSION,
					     ERR_LC_TLS2,
					     ERR_PT_TLS_HS_MSG_CERT + 20, NULL);
				TLS_ALERT_FATAL(
					tls,
					TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
				goto err;
			}

#if 0
			/* Cannot verify using tls->sighash_list because
			 * tls->sighash_list saves the list of hash/signature
			 * algorithm pairs provided by peer.
			 *
			 * To verify NEED the new variable saving the list of
			 * hash/signature algorithm pairs for self.
			 */

			/* XXX: both server and client certificate should
			 * support. */
			if ((tls->sighash_list != NULL) &&
			    (check_cert_sighash(tls, cert)) == false) {
				goto err;
			}
#endif
		}
			break;

		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		default:
			break;
		}

		if (P12_add_cert(p12, cert, NULL, 0xff) < 0) {
			TLS_DPRINTF("hs: m: cert: P12_add_cert");
			OK_set_error(ERR_ST_TLS_P12_ADD_CERT,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_MSG_CERT + 21, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}

		read_bytes += certlen;
	}

	if (type == TLS_CERT_SERVER) {
		if (check_cert_server(tls, p12) == false) {
			goto err;
		}

		/*
		 * Get a public key from server's certificate.
		 * It use after constructing ClientKeyExchange message.
		 */
		if (set_pubkey_from_pkcs12(tls, p12) == false) {
			goto err;
		}
	}

	if ((type == TLS_CERT_CLIENT) &&
	    (check_cert_client(tls, p12) == false)) {
		goto err;
	}

	if (! TLS_cert_verify(tls, p12)) {
		goto err;
	}

	switch (type) {
	case TLS_CERT_CLIENT:
		tls->pkcs12_client = p12;
		break;

	case TLS_CERT_SERVER:
		tls->pkcs12_server = p12;
		break;

	default:
		assert(!"unknown cert type.");
	}

done:
	return read_bytes;

err:
	P12_free(p12);
	return -1;
}

static int32_t read_certchain_tls13(const enum tls_cert_t type,
			      TLS *tls,
			      struct tls_hs_msg *msg,
			      const uint32_t offset) {
	uint32_t read_bytes = 0;

	if (msg->len < (offset + TLS_CERT_REQUEST_CONTEXT_LENGTH_BYTES)) {
		TLS_DPRINTF("hs: m: cert: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 14, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *           opaque certificate_request_context<0..2^8-1>;
	 */
	uint8_t reqctxlen = msg->msg[offset];
	read_bytes += TLS_CERT_REQUEST_CONTEXT_LENGTH_BYTES;

	TLS_DPRINTF("hs: m: cert: request context len = %d", reqctxlen);

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *    certificate_request_context:  If this message is in response to a
	 *       CertificateRequest, the value of certificate_request_context in
	 *       that message.  Otherwise (in the case of server authentication),
	 *       this field SHALL be zero length.
	 */
	if (reqctxlen > 0) {
		if (type == TLS_CERT_CLIENT) {
			/* TODO: check request context */
			read_bytes += reqctxlen;
		} else {
			TLS_DPRINTF("hs: m: cert: request context must be zero");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 15,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			return -1;
		}
	}

	if (msg->len < (offset + read_bytes + TLS_CERT_CHAIN_LENGTH_BYTES)) {
		TLS_DPRINTF("hs: m: cert: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 16, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *           CertificateEntry certificate_list<0..2^24-1>;
	 */
	uint32_t chainlen = tls_util_read_3(&(msg->msg[offset + read_bytes]));
	read_bytes += TLS_CERT_CHAIN_LENGTH_BYTES;

	/*
	 *  RFC8446 4.4.2.  Certificate
	 *
	 *                   .. If the server requests client authentication but no
	 *    suitable certificate is available, the client MUST send a Certificate
	 *    message containing no certificates (i.e., with the "certificate_list"
	 *    field having length 0).
	 */
	if ((type == TLS_CERT_CLIENT) && (chainlen == 0)) {
		goto done;
	}

	PKCS12 *p12;
	if ((p12 = P12_new()) == NULL) {
		TLS_DPRINTF("hs: m: cert: P12_new");
		OK_set_error(ERR_ST_TLS_P12_NEW,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 17, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	/*
	 * XXX: use sighash_list_cert that was sent to peer. it is not
	 * implemented now, then use tls_hs_sighash_list_cert() as default
	 * value.
	 */
	struct tls_hs_sighash_list *sslist = NULL;
	struct tls_hs_sighash_list *sslist_cert = NULL;
	if ((sslist_cert = tls_hs_sighash_list_cert(tls)) == NULL) {
		TLS_DPRINTF("hs: m: cert: tls_hs_sighash_list_cert");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	/* process CertificateEntry */
	TLS_DPRINTF("hs: m: cert: chain len = %d", chainlen);
	const uint32_t certchainend = read_bytes + chainlen;
	uint32_t cert_count = 0;
	while (read_bytes < certchainend) {
		if (msg->len < (offset + read_bytes + TLS_CERT_LENGTH_BYTES)) {
			TLS_DPRINTF("hs: m: cert: record length");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 18,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			goto err;
		}

		/*
		 * RFC8446 4.4.2.  Certificate
		 *
		 *                 opaque cert_data<1..2^24-1>;
		 */
		const int32_t certlen = tls_util_read_3(
			&(msg->msg[offset + read_bytes]));
		const int32_t cert_length_min = 1;
		if (certlen < cert_length_min) {
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 31,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			goto err;
		}

		read_bytes += TLS_CERT_LENGTH_BYTES;

		if (msg->len < (offset + read_bytes + certlen)) {
			TLS_DPRINTF("hs: m: cert: record length");
			OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 19,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
			goto err;
		}
		TLS_DPRINTF("hs: m: cert: len     = %d", certlen);

		Cert *cert;
		if ((cert = get_cert(tls,
				     &(msg->msg[offset + read_bytes]),
				     certlen)) == NULL) {
			goto err;
		}

		TLS_DPRINTF("hs: m: cert: version = %d (0x%x)",
			    cert->version + 1, cert->version);
		TLS_DPRINTF("hs: m: cert: sig     = %d", cert->signature_algo);
		TLS_DPRINTF("hs: m: cert: pubkey  = %d", cert->pubkey_algo);

		/*
		 * x509 certificate version representation is zero-origin.
		 * use int instead of uint32_t because cert->version is int.
		 */
		const int version_of_x509v3 = 3 - 1;
		if (cert->version != version_of_x509v3) {
			TLS_DPRINTF("hs: m: cert: x509 version != 3");
			OK_set_error(ERR_ST_TLS_X509_VERSION,
				     ERR_LC_TLS6,
				     ERR_PT_TLS_HS_MSG_CERT2 + 20, NULL);
			TLS_ALERT_FATAL(tls,
				TLS_ALERT_DESC_UNSUPPORTED_CERTIFICATE);
			goto err;
		}

		/*
		 * RFC8446 4.4.2.4.  Receiving a Certificate Message
		 *
		 *    Any endpoint receiving any certificate which it would need to
		 *    validate using any signature algorithm using an MD5 hash MUST abort
		 *    the handshake with a "bad_certificate" alert.  SHA-1 is deprecated,
		 *    and it is RECOMMENDED that any endpoint receiving any certificate
		 *    which it would need to validate using any signature algorithm using a
		 *    SHA-1 hash abort the handshake with a "bad_certificate" alert.  For
		 *    clarity, this means that endpoints can accept these algorithms for
		 *    certificates that are self-signed or are trust anchors.
		 */
		uint16_t md5algo = TLS_SS_RSA_PKCS1_MD5;
		int ai_sig = tls_hs_sighash_get_ai_sig_type_by_ss(md5algo);
		if (ai_sig < 0) {
			TLS_DPRINTF("hs: m: cert: unsupported sig type: %4x", md5algo);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		} else if (cert->signature_algo == ai_sig) {
			TLS_DPRINTF("hs: m: cert: md5 algorithm is used");
			OK_set_error(ERR_ST_TLS_FORBIDDEN_SIGSCHEME,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 21,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_BAD_CERTIFICATE);
			goto err;
		}

		/*
		 * RFC8446 4.4.2.2.  Server Certificate Selection
		 *
		 *    If the client cannot construct an acceptable chain using the provided
		 *    certificates and decides to abort the handshake, then it MUST abort
		 *    the handshake with an appropriate certificate-related alert (by
		 *    default, "unsupported_certificate"; see Section 6.2 for more
		 *    information).
		 */

		/*
		 * check if signature algorithm of certificate matches one of
		 * signature scheme in sighash_list_cert sent to peer. this list
		 * contains the schemes in "signature_algorithms" or
		 * "signature_algorithms_cert" extension.
		 */
		if (check_cert_sighash(tls, cert, sslist_cert) == false) {
			goto err;
		}

		if (P12_add_cert(p12, cert, NULL, 0xff) < 0) {
			TLS_DPRINTF("hs: m: cert: P12_add_cert");
			OK_set_error(ERR_ST_TLS_P12_ADD_CERT,
				     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 22,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}
		read_bytes += certlen;

		/*
		 * RFC8446 4.4.2.  Certificate
		 *
		 *              Extensions in the Certificate message from the server MUST
		 *       correspond to ones from the ClientHello message.  Extensions in
		 *       the Certificate message from the client MUST correspond to
		 *       extensions in the CertificateRequest message from the server.  If
		 *       an extension applies to the entire chain, it SHOULD be included in
		 *       the first CertificateEntry.
		 */
		/* read extension per certificate */
		int32_t extlen;
		if ((extlen = tls_hs_extension_parse(tls, msg,
						     offset + read_bytes)) < 0) {
			TLS_DPRINTF("tls_hs_extension_parse");
			goto err;
		}

		TLS_DPRINTF("hs: m: cert: extention len = %d", extlen);

		if (extlen > 0 && (interpret_ext_list(tls) == false)) {
			TLS_DPRINTF("interpret_ext_list");
			goto err;
		}
		read_bytes += extlen;

		cert_count++;
	}

	/*
	 * RFC8446 4.4.2.4.  Receiving a Certificate Message
	 *
	 *    If the server supplies an empty Certificate message, the client MUST
	 *    abort the handshake with a "decode_error" alert.
	 */
	if (type == TLS_CERT_SERVER && cert_count == 0) {
		TLS_DPRINTF("hs: m: cert: server certificate == null.");
		OK_set_error(ERR_ST_TLS_EMPTY_CERT,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 23, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		goto err;
	}

	/*
	 * XXX: use sighash_list_cert that was sent to peer. it is not
	 * implemented now, then use tls_hs_sighash_list_cert() as default
	 * value.
	 */
	if ((sslist = tls_hs_sighash_list(tls)) == NULL) {
		TLS_DPRINTF("hs: m: cert: tls_hs_sighash_list");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	if (is_compatible_with_sighash(tls, p12, sslist) == false) {
		goto err;
	}

	if (TLS_cert_verify(tls, p12) == false) {
		TLS_DPRINTF("certificate verification failed");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_BAD_CERTIFICATE);
		goto err;
	}

	switch (type) {
	case TLS_CERT_SERVER:
		tls->pkcs12_server = p12;
		break;

	case TLS_CERT_CLIENT:
		tls->pkcs12_client = p12;
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 24, NULL);
		goto err;
	}

	tls_hs_sighash_free(sslist);
	tls_hs_sighash_free(sslist_cert);

done:
	return read_bytes;

err:
	P12_free(p12);
	tls_hs_sighash_free(sslist);
	tls_hs_sighash_free(sslist_cert);

	return -1;
}

static int32_t write_scert(TLS *tls, struct tls_hs_msg *msg) {
	enum tls_cert_t type = TLS_CERT_SERVER;

	int32_t certlen = 0;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if ((certlen = write_certchain_up_to_tls12(type, tls, msg)) < 0) {
			TLS_DPRINTF("hs: m: cert: write_certificate_chain");
			return -1;
		}
		break;

	case TLS_VER_TLS13:
		if ((certlen = write_certchain_tls13(type, tls, msg)) < 0) {
			TLS_DPRINTF("hs: m: cert: write_certificate_chain");
			return -1;
		}
		break;

	default:
		TLS_DPRINTF("unknown version");
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 25, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	return certlen;
}

static int32_t read_scert(TLS *tls, struct tls_hs_msg *msg,
			  const uint32_t offset) {
	enum tls_cert_t type = TLS_CERT_SERVER;

	int32_t certlen;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if ((certlen = read_certchain_up_to_tls12(type, tls, msg, offset)) < 0) {
			TLS_DPRINTF("hs: m: cert: read_certificate_chain");
			return -1;
		}
		break;

	case TLS_VER_TLS13:
		if ((certlen = read_certchain_tls13(type, tls, msg, offset)) < 0) {
			TLS_DPRINTF("hs: m: cert: read_certificate_chain");
			return -1;
		}
		break;

	default:
		TLS_DPRINTF("unknown version");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	return certlen;
}

static int32_t write_ccert(TLS *tls, struct tls_hs_msg *msg) {
	enum tls_cert_t type = TLS_CERT_CLIENT;

	int32_t certlen = 0;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if ((certlen = write_certchain_up_to_tls12(type, tls, msg)) < 0) {
			TLS_DPRINTF("hs: m: cert: write_certificate_chain");
			return -1;
		}

		/* in the case of body of certificate message is null. */
		if (tls->pkcs12_client == NULL) {
			tls->ccert_null = true;
		}
		break;

	case TLS_VER_TLS13:
		if ((certlen = write_certchain_tls13(type, tls, msg)) < 0) {
			TLS_DPRINTF("hs: m: cert: write_certificate_chain");
			return -1;
		}

		/* in the case of body of certificate message is null. */
		if (tls->pkcs12_client == NULL) {
			tls->ccert_null = true;
		}
		break;

	default:
		TLS_DPRINTF("unknown version");
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS6, ERR_PT_TLS_HS_MSG_CERT2 + 26, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	return certlen;
}

static int32_t read_ccert(TLS *tls, struct tls_hs_msg *msg,
			  const uint32_t offset) {
	enum tls_cert_t type = TLS_CERT_CLIENT;

	int32_t certlen = 0;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if ((certlen = read_certchain_up_to_tls12(type, tls, msg, offset)) < 0) {
			TLS_DPRINTF("hs: m: cert: read_certificate_chain");
			return -1;
		}

		/* in the case of body of certificate message is null. */
		if (tls->pkcs12_client == NULL) {
			TLS_DPRINTF("hs: m: cert: client certificate == null.");
			if (TLS_opt_get(tls, TLS_OPT_MAY_USE_CLIENT_CERT_AUTH) < 0) {
				OK_set_error(ERR_ST_TLS_PKCS12_CLIENT_NULL,
					     ERR_LC_TLS6,
					     ERR_PT_TLS_HS_MSG_CERT2 + 27, NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
				return -1;
			}

			tls->ccert_null = true;
		}
		break;

	case TLS_VER_TLS13:
		if ((certlen = read_certchain_tls13(type, tls, msg, offset)) < 0) {
			TLS_DPRINTF("hs: m: cert: read_certificate_chain");
			return -1;
		}

		/* in the case of body of certificate message is null. */
		if (tls->pkcs12_client == NULL) {
			TLS_DPRINTF("hs: m: cert: client certificate == null.");
			if (TLS_opt_get(tls, TLS_OPT_MAY_USE_CLIENT_CERT_AUTH) < 0) {
				/*
				 * RFC8446 4.4.2.4.  Receiving a Certificate Message
				 *
				 *    If the client does not send any certificates (i.e., it sends an empty
				 *    Certificate message), the server MAY at its discretion either
				 *    continue the handshake without client authentication or abort the
				 *    handshake with a "certificate_required" alert.  Also, if some aspect
				 *    of the certificate chain was unacceptable (e.g., it was not signed by
				 *    a known, trusted CA), the server MAY at its discretion either
				 *    continue the handshake (considering the client unauthenticated) or
				 *    abort the handshake.
				 */
				OK_set_error(ERR_ST_TLS_PKCS12_CLIENT_NULL,
					     ERR_LC_TLS6,
					     ERR_PT_TLS_HS_MSG_CERT2 + 28, NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_CERTIFICATE_REQUIRED);
				return -1;
			}

			tls->ccert_null = true;
		}
		break;

	default:
		TLS_DPRINTF("unknown version");
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	return certlen;
}

struct tls_hs_msg * tls_hs_scert_compose(TLS *tls) {
	uint32_t offset = 0;

	struct tls_hs_msg *msg;

	if ((msg = tls_hs_msg_init()) == NULL) {
		TLS_DPRINTF("hs: m: cert: tls_hs_msg_init");
		return NULL;
	}

	msg->type = TLS_HANDSHAKE_CERTIFICATE;

	int32_t write_bytes = 0;
	if ((write_bytes = write_scert(tls, msg)) < 0) {
		TLS_DPRINTF("write_scert");
		goto failed;
	}
	offset += write_bytes;

	msg->len = offset;

	return msg;

failed:
	tls_hs_msg_free(msg);
	return NULL;
}

bool tls_hs_scert_parse(TLS *tls, struct tls_hs_msg *msg) {
	uint32_t offset = 0;

	if (msg->type != TLS_HANDSHAKE_CERTIFICATE) {
		TLS_DPRINTF("hs: m: cert: TLS_HANDSHAKE_CERTIFICATE");
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 22, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return false;
	}

	int32_t read_bytes = 0;
	if ((read_bytes = read_scert(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_scert");
		return false;
	}
	offset += read_bytes;

	if (offset != msg->len) {
		TLS_DPRINTF("hs: m: cert: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 23, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return false;
	}

	return true;
}

struct tls_hs_msg * tls_hs_ccert_compose(TLS *tls) {
	uint32_t offset = 0;

	struct tls_hs_msg *msg;

	if ((msg = tls_hs_msg_init()) == NULL) {
		TLS_DPRINTF("hs: m: cert: tls_hs_msg_init");
		return NULL;
	}

	msg->type = TLS_HANDSHAKE_CERTIFICATE;

	int32_t write_bytes;
	if ((write_bytes = write_ccert(tls, msg)) < 0) {
		TLS_DPRINTF("write_ccert");
		goto failed;
	}
	offset += write_bytes;

	msg->len = offset;

	return msg;

failed:
	tls_hs_msg_free(msg);
	return NULL;
}

bool tls_hs_ccert_parse(TLS *tls, struct tls_hs_msg *msg) {
	uint32_t offset = 0;

	if (msg->type != TLS_HANDSHAKE_CERTIFICATE) {
		TLS_DPRINTF("hs: m: cert: TLS_HANDSHAKE_CERTIFICATE");
		OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 24, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
		return false;
	}

	int32_t read_bytes = 0;
	if ((read_bytes = read_ccert(tls, msg, offset)) < 0) {
		TLS_DPRINTF("read_ccert");
		return false;
	}
	offset += read_bytes;

	if (offset != msg->len) {
		TLS_DPRINTF("hs: m: cert: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS2, ERR_PT_TLS_HS_MSG_CERT + 25, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return false;
	}

	return true;
}
