/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "extension/tls_sighash.h"
#include "tls_alert.h"

/* for OBJ_SIG_MD5RSA, OBJ_SIG_SHA1RSA, OBJ_SIG_SHA224RSA,
 * OBJ_SIG_SHA256RSA, OBJ_SIG_SHA384RSA, OBJ_SIG_SHA512RSA,
 * OBJ_SIG_SHA1DSA, OBJ_SIG_SHA224DSA, OBJ_SIG_SHA256DSA,
 * OBJ_SIG_SHA1ECDSA, OBJ_SIG_SHA224ECDSA, OBJ_SIG_SHA256ECDSA,
 * OBJ_SIG_SHA384ECDSA and OBJ_SIG_SHA512ECDSA, OBJ_HASH_MD5,
 * OBJ_HASH_SHA1, OBJ_HASH_SHA224, OBJ_HASH_SHA256, OBJ_HASH_SHA384 and
 * OBJ_HASH_SHA512. */
#include <aicrypto/ok_asn1.h>

/* for hash_size(). */
#include <aicrypto/ok_tool.h>

/**
 * compose signature/hash algorithm pairs list from a list passed by argument.
 *
 * if the returned list becomes unnecessary, you must free by tls_hs_sighash_free
 * function as the list was allocated by malloc.
 */
static struct tls_hs_sighash_list *compose_sighash(
	TLS *tls, struct tls_hs_sighash_list src);

/**
 * set signature scheme value to tls_hs_sighash_algo structure.
 */
static void convert_sigscheme_to_sigalgo(uint16_t sigscheme,
				  struct tls_hs_sighash_algo *sigalgo);

/**
 * merge signature schemes into signature scheme list uniquely.
 */
static uint16_t merge_signature_scheme_list(uint16_t *schemes,
					    uint16_t schemes_len,
					    uint16_t offset,
					    uint16_t *version_schemes,
					    uint16_t version_schemes_len);

/**
 * check availability of specified signature/hash algorithm pair by signature
 * scheme.
 */
static bool sighash_availablep_by_ss(const TLS *tls, uint16_t sigscheme);

/**
 * check availability of specified signature/hash algorithm pair in
 * signature algorithm cert extension by signature scheme.
 */
static bool sighash_cert_availablep_by_ss(const TLS *tls, uint16_t sigscheme);

/**
 * default set of signature/hash algorithm that is used by TLS 1.2 in
 * this module.
 */
UNUSED
static struct tls_hs_sighash_algo tls_hs_sighash_list_tls12[] = {
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA512 },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA384 },
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA256 },
#ifdef TLS_OBSOLETE_ALGO
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA224 },
#endif
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA1   },
#ifdef TLS_OBSOLETE_ALGO
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_MD5    },
#endif
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA512 },
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA384 },
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA256 },
#ifdef TLS_OBSOLETE_ALGO
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA224 },
#endif
	{ .sig = TLS_SIG_ALGO_ECDSA, .hash = TLS_HASH_ALGO_SHA1   },
#ifdef TLS_OBSOLETE_ALGO
	{ .sig = TLS_SIG_ALGO_DSA,   .hash = TLS_HASH_ALGO_SHA256 },
	{ .sig = TLS_SIG_ALGO_DSA,   .hash = TLS_HASH_ALGO_SHA224 },
	{ .sig = TLS_SIG_ALGO_DSA,   .hash = TLS_HASH_ALGO_SHA1   }
#endif
};

/**
 * default set of signature scheme that is used by TLS 1.2 in this module.
 */
static uint16_t tls_hs_sigscheme_list_tls12[] = {
	TLS_SS_RSA_PKCS1_SHA512,
	TLS_SS_RSA_PKCS1_SHA384,
	TLS_SS_RSA_PKCS1_SHA256,
#ifdef TLS_OBSOLETE_ALGO
	TLS_SS_RSA_PKCS1_SHA224,
#endif
	TLS_SS_RSA_PCKS1_SHA1,
#ifdef TLS_OBSOLETE_ALGO
	TLS_SS_RSA_PKCS1_MD5,
#endif
	TLS_SS_ECDSA_SECP521R1_SHA512,
	TLS_SS_ECDSA_SECP384R1_SHA384,
	TLS_SS_ECDSA_SECP256R1_SHA256,
#ifdef TLS_OBSOLETE_ALGO
	TLS_SS_ECDSA_SHA224,
#endif
	TLS_SS_ECDSA_SHA1,
#ifdef TLS_OBSOLETE_ALGO
	TLS_SS_DSA_SHA256,
	TLS_SS_DSA_SHA224,
	TLS_SS_DSA_SHA1,
#endif
};

/**
 * default set of signature scheme that is used by TLS 1.3 in this module.
 */
static uint16_t tls_hs_sigscheme_list_tls13[] = {
	TLS_SS_RSA_PSS_RSAE_SHA512,
	TLS_SS_RSA_PSS_RSAE_SHA384,
	TLS_SS_RSA_PSS_RSAE_SHA256,
	TLS_SS_ECDSA_SECP521R1_SHA512,
	TLS_SS_ECDSA_SECP384R1_SHA384,
	TLS_SS_ECDSA_SECP256R1_SHA256,
};

static uint16_t tls_hs_sigscheme_cert_list_tls13[] = {
	TLS_SS_RSA_PKCS1_SHA512,
	TLS_SS_RSA_PKCS1_SHA384,
	TLS_SS_RSA_PKCS1_SHA256,
	TLS_SS_ECDSA_SECP521R1_SHA512,
	TLS_SS_ECDSA_SECP384R1_SHA384,
	TLS_SS_ECDSA_SECP256R1_SHA256,
	TLS_SS_RSA_PCKS1_SHA1,
	TLS_SS_ECDSA_SHA1
};

/**
 * default set of signature/hash algorithm that is used when client did
 * not send signature algorithm extension and negotiated key exchange
 * algorithm was one of rsa, dhe_rsa, dh_rsa, rsa_psk, ecdh_rsa and
 * ecdhe_rsa.
 */
static struct tls_hs_sighash_algo tls_hs_sighash_list_default_rss[] = {
	{ .sig = TLS_SIG_ALGO_RSA,   .hash = TLS_HASH_ALGO_SHA1 },
};

/**
 * default set of signature/hash algorithm that is used when client did
 * not send signature algorithm extension and negotiated key exchange
 * algorithm was one of dhe_dss and dh_dss.
 */
static struct tls_hs_sighash_algo tls_hs_sighash_list_default_dsa[] = {
	{ .sig = TLS_SIG_ALGO_DSA,   .hash = TLS_HASH_ALGO_SHA1 },
};

static struct tls_hs_sighash_list *compose_sighash(
	TLS *tls, struct tls_hs_sighash_list src) {
	struct tls_hs_sighash_list *sighash;

	if ((sighash = malloc(
		     1 * sizeof (struct tls_hs_sighash_list))) == NULL) {
		TLS_DPRINTF("sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}

	sighash->len = src.len;
	uint32_t sighash_list_size =
		sighash->len * sizeof (struct tls_hs_sighash_algo);

	if ((sighash->list = malloc(sighash_list_size)) == NULL) {
		TLS_DPRINTF("sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		free(sighash);
		return NULL;
	}

	memcpy(&(sighash->list[0]), &(src.list[0]), sighash_list_size);

	return sighash;
}

static void convert_sigscheme_to_sigalgo(uint16_t sigscheme,
				  struct tls_hs_sighash_algo *sigalgo)
{
	sigalgo->hash = (sigscheme >> 8) & 0xff;
	sigalgo->sig  = (sigscheme     ) & 0xff;
}

uint16_t tls_hs_sighash_convert_sighash_to_sigscheme(
    struct tls_hs_sighash_algo *sigalgo)
{
	return (sigalgo->hash << 8) + ((sigalgo->sig) & 0x00ff);
}

static uint16_t merge_signature_scheme_list(uint16_t *schemes,
					    uint16_t schemes_len,
					    uint16_t offset,
					    uint16_t *version_schemes,
					    uint16_t version_schemes_len)
{
	bool duplicated = false;
	uint16_t actlen = 0;

	for (int i = 0; i < version_schemes_len; i++, duplicated = false) {
		if (schemes_len == (offset + actlen)) {
			break;
		}

		for (int j = 0; j < schemes_len - offset; j++) {
			if (schemes[offset+j] == version_schemes[i]) {
				duplicated = true;
				break;
			}
		}

		if (duplicated == true) {
			continue;
		}

		schemes[offset+actlen] = version_schemes[i];;
		actlen++;
	}

	return actlen;
}

static bool sighash_availablep_by_ss(const TLS *tls, uint16_t sigscheme) {
	uint16_t *list;
	int len;

	uint16_t version = tls_util_convert_protover_to_ver(
		&(((TLS *) tls)->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
		/* TODO: not implementation. */
		return false;

	case TLS_VER_TLS12:
		len = sizeof (tls_hs_sigscheme_list_tls12) / sizeof (uint16_t);
		list = tls_hs_sigscheme_list_tls12;
		break;

	case TLS_VER_TLS13:
		len = sizeof(tls_hs_sigscheme_list_tls13) / sizeof (uint16_t);
		list = tls_hs_sigscheme_list_tls13;
		break;

	default:
		return false;
	}

	for (int i = 0; i < len; ++i) {
		if (list[i] == sigscheme) {
			return true;
		}
	}

	return false;
}

static bool sighash_cert_availablep_by_ss(const TLS *tls, uint16_t sigscheme) {
	uint16_t *list;
	int len;

	uint16_t version = tls_util_convert_protover_to_ver(
		&(((TLS *) tls)->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		/* Not supported */
		return false;

	case TLS_VER_TLS13:
		len = sizeof(tls_hs_sigscheme_cert_list_tls13)
			/ sizeof (uint16_t);
		list = tls_hs_sigscheme_cert_list_tls13;
		break;

	default:
		return false;
	}

	for (int i = 0; i < len; ++i) {
		if (list[i] == sigscheme) {
			return true;
		}
	}

	return false;
}

struct tls_hs_sighash_list * tls_hs_sighash_list(TLS *tls) {
	struct tls_hs_sighash_list *list = NULL;
	uint16_t buf_len = TLS_EXT_SIG_ALGO_SIZE_MAX / sizeof(uint16_t);
	uint16_t *buf = NULL;
	int32_t sslist_len = 0;

	if ((buf = malloc(TLS_EXT_SIG_ALGO_SIZE_MAX)) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}
	memset(buf, 0, TLS_EXT_SIG_ALGO_SIZE_MAX);

	uint16_t version;
	uint16_t *ver_sslist;
	uint16_t ver_sslist_len;
	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_RECV_CHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_2NDCHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_RECV_2NDCHELLO)) {

		for (int i = 0; i < tls->supported_versions.len; i++) {
			ver_sslist = NULL;
			version = tls->supported_versions.list[i];

			switch (version) {
			case TLS_VER_SSL30:
			case TLS_VER_TLS10:
			case TLS_VER_TLS11:
				/* Not supported */
				break;

			case TLS_VER_TLS12:
				ver_sslist_len = sizeof(tls_hs_sigscheme_list_tls12)
					/ sizeof(uint16_t);
				ver_sslist = tls_hs_sigscheme_list_tls12;
				break;

			case TLS_VER_TLS13:
				ver_sslist_len = sizeof(tls_hs_sigscheme_list_tls13)
					/ sizeof(uint16_t);
				ver_sslist = tls_hs_sigscheme_list_tls13;
				break;

			default:
				/* Ignore unknown version */
				break;
			}

			if (ver_sslist != NULL) {
				sslist_len += merge_signature_scheme_list(buf,
						  buf_len,
						  sslist_len,
						  ver_sslist,
						  ver_sslist_len);
			}

			/* check buffer is full */
			if (buf_len == sslist_len) {
				break;
			}
		}
	} else {
		version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
			/* Not supported */
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5,
				     ERR_PT_TLS_HS_EXT_SIGHASH2 + 1,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;

		case TLS_VER_TLS12:
			sslist_len  = sizeof (tls_hs_sigscheme_list_tls12) /
				sizeof (uint16_t);
			ver_sslist = tls_hs_sigscheme_list_tls12;
			break;

		case TLS_VER_TLS13:
			sslist_len = sizeof(tls_hs_sigscheme_list_tls13) /
				sizeof(uint16_t);
			ver_sslist = tls_hs_sigscheme_list_tls13;
			break;

		default:
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5,
				     ERR_PT_TLS_HS_EXT_SIGHASH2 + 2,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}

		memcpy(buf, ver_sslist, sslist_len * sizeof(uint16_t));
	}

	if ((list = malloc (1 * sizeof (struct tls_hs_sighash_list))) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 3, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	list->len = sslist_len;

	int32_t list_size = list->len * sizeof (struct tls_hs_sighash_algo);
	if ((list->list = malloc (list_size)) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 4, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	for (uint16_t i = 0; i < list->len; ++i) {
		convert_sigscheme_to_sigalgo(buf[i], &(list->list[i]));
	}

	free(buf);

	return list;

err:
	free(list);
	free(buf);

	return NULL;
}

struct tls_hs_sighash_list * tls_hs_sighash_list_cert(TLS *tls) {
	struct tls_hs_sighash_list *list = NULL;
	uint16_t buf_len = TLS_EXT_SIG_ALGO_SIZE_MAX / sizeof(uint16_t);
	uint16_t *buf = NULL;
	int32_t sslist_len = 0;

	if ((buf = malloc(TLS_EXT_SIG_ALGO_SIZE_MAX)) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 5, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return NULL;
	}
	memset(buf, 0, TLS_EXT_SIG_ALGO_SIZE_MAX);

	uint16_t version;
	uint16_t *ver_sslist;
	uint16_t ver_sslist_len;
	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_RECV_CHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_2NDCHELLO) ||
	    tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_RECV_2NDCHELLO)) {


		for (int i = 0; i < tls->supported_versions.len; i++) {
			ver_sslist = NULL;
			version = tls->supported_versions.list[i];

			switch (version) {
			case TLS_VER_SSL30:
			case TLS_VER_TLS10:
			case TLS_VER_TLS11:
			case TLS_VER_TLS12:
				/* Not supported */
				break;

			case TLS_VER_TLS13:
				ver_sslist_len = sizeof(tls_hs_sigscheme_cert_list_tls13)
					/ sizeof(uint16_t);
				ver_sslist = tls_hs_sigscheme_cert_list_tls13;
				break;

			default:
				/* Ignore unknown version */
				break;
			}

			if (ver_sslist != NULL) {
				sslist_len += merge_signature_scheme_list(buf,
						  buf_len,
						  sslist_len,
						  ver_sslist,
						  ver_sslist_len);
			}

			/* check buffer is full */
			if (buf_len == sslist_len) {
				break;
			}
		}
	} else {
		version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
			/* Not supported */
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 6,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;

		case TLS_VER_TLS13:
			sslist_len = sizeof(tls_hs_sigscheme_cert_list_tls13) /
				sizeof(uint16_t);
			ver_sslist = tls_hs_sigscheme_cert_list_tls13;
			break;

		default:
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 7,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}

		memcpy(buf, ver_sslist, sslist_len * sizeof(uint16_t));
	}

	if ((list = malloc (1 * sizeof (struct tls_hs_sighash_list))) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	list->len = sslist_len;

	int32_t list_size = list->len * sizeof (struct tls_hs_sighash_algo);
	if ((list->list = malloc (list_size)) == NULL) {
		TLS_DPRINTF("hs: ext: sighash: malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	for (uint16_t i = 0; i < list->len; ++i) {
		convert_sigscheme_to_sigalgo(buf[i], &(list->list[i]));
	}

	free(buf);

	return list;

err:
	free(list);
	free(buf);

	return NULL;
}

void tls_hs_sighash_free(struct tls_hs_sighash_list *sighash) {

	if (sighash == NULL) {
		return;
	}

	free(sighash->list);
	sighash->list = NULL;
	sighash->len  = 0;

	free(sighash);
	sighash = NULL;
}

bool tls_hs_sighash_availablep(const TLS *tls,
			       const struct tls_hs_sighash_algo sighash) {
	struct tls_hs_sighash_algo sh = sighash;
	uint16_t ss = tls_hs_sighash_convert_sighash_to_sigscheme(&sh);
	return sighash_availablep_by_ss(tls, ss);
}

bool tls_hs_sighash_availablep_cert(const TLS *tls,
				    const struct tls_hs_sighash_algo sighash) {
	struct tls_hs_sighash_algo sh = sighash;
	uint16_t ss = tls_hs_sighash_convert_sighash_to_sigscheme(&sh);
	return sighash_cert_availablep_by_ss(tls, ss);
}

int32_t tls_hs_sighash_get_hash_size_by_ss(const uint16_t sigscheme) {
	int aihash_algo = tls_hs_sighash_get_ai_hash_type_by_ss(sigscheme);
	if (aihash_algo < 0) {
		return -1;
	}

	return hash_size(aihash_algo);
}

int32_t tls_hs_sighash_get_hash_size(const enum tls_hs_sighash_hash_algo hash) {
	if (hash == TLS_HASH_ALGO_NONE) {
		return 0;
	}

	int aihash_algo = tls_hs_sighash_get_ai_hash_type(hash);
	if (aihash_algo < 0) {
		return -1;
	}

	return hash_size(aihash_algo);
}

int32_t tls_hs_sighash_get_ai_hash_type_by_ss(uint16_t sigscheme) {
	switch (sigscheme) {
	case TLS_SS_RSA_PKCS1_MD5:
		return OBJ_HASH_MD5;

	case TLS_SS_RSA_PCKS1_SHA1:
	case TLS_SS_ECDSA_SHA1:
	case TLS_SS_DSA_SHA1:
		return OBJ_HASH_SHA1;

	case TLS_SS_RSA_PKCS1_SHA224:
	case TLS_SS_DSA_SHA224:
	case TLS_SS_ECDSA_SHA224:
		return OBJ_HASH_SHA224;

	case TLS_SS_RSA_PKCS1_SHA256:
	case TLS_SS_ECDSA_SECP256R1_SHA256:
	case TLS_SS_DSA_SHA256:
	case TLS_SS_RSA_PSS_RSAE_SHA256:
	case TLS_SS_RSA_PSS_PSS_SHA256:
		return OBJ_HASH_SHA256;

	case TLS_SS_RSA_PKCS1_SHA384:
	case TLS_SS_ECDSA_SECP384R1_SHA384:
	case TLS_SS_RSA_PSS_RSAE_SHA384:
	case TLS_SS_RSA_PSS_PSS_SHA384:
		return OBJ_HASH_SHA384;

	case TLS_SS_RSA_PKCS1_SHA512:
	case TLS_SS_ECDSA_SECP521R1_SHA512:
	case TLS_SS_RSA_PSS_RSAE_SHA512:
	case TLS_SS_RSA_PSS_PSS_SHA512:
		return OBJ_HASH_SHA512;

	case TLS_SS_ED25519:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 10,
			     NULL);
		return -1;

	case TLS_SS_ED448:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 11,
			     NULL);
		return -1;

	default:
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 12,
			     NULL);
		return -1;
	}
}

int32_t tls_hs_sighash_get_ai_hash_type(enum tls_hs_sighash_hash_algo hash) {
	switch(hash) {
	case TLS_HASH_ALGO_MD5:
		return OBJ_HASH_MD5;

	case TLS_HASH_ALGO_SHA1:
		return OBJ_HASH_SHA1;

	case TLS_HASH_ALGO_SHA224:
		return OBJ_HASH_SHA224;

	case TLS_HASH_ALGO_SHA256:
		return OBJ_HASH_SHA256;

	case TLS_HASH_ALGO_SHA384:
		return OBJ_HASH_SHA384;

	case TLS_HASH_ALGO_SHA512:
		return OBJ_HASH_SHA512;

	default:
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_HASH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH2 + 13,
			     NULL);
		return -1;
	}
}

int32_t tls_hs_sighash_get_ai_sig_type_by_ss(uint16_t sigscheme) {
	switch (sigscheme) {
	case TLS_SS_RSA_PKCS1_MD5:
		return OBJ_SIG_MD5RSA;

	case TLS_SS_RSA_PCKS1_SHA1:
		return OBJ_SIG_SHA1RSA;

	case TLS_SS_RSA_PKCS1_SHA224:
		return OBJ_SIG_SHA224RSA;

	case TLS_SS_RSA_PKCS1_SHA256:
		return OBJ_SIG_SHA256RSA;

	case TLS_SS_RSA_PKCS1_SHA384:
		return OBJ_SIG_SHA384RSA;

	case TLS_SS_RSA_PKCS1_SHA512:
		return OBJ_SIG_SHA512RSA;

	case TLS_SS_DSA_SHA1:
		return OBJ_SIG_SHA1DSA;

	case TLS_SS_DSA_SHA224:
		return OBJ_SIG_SHA224DSA;

	case TLS_SS_DSA_SHA256:
		return OBJ_SIG_SHA256DSA;

	case TLS_SS_ECDSA_SHA1:
		return OBJ_SIG_SHA1ECDSA;

	case TLS_SS_ECDSA_SHA224:
		return OBJ_SIG_SHA224ECDSA;

	case TLS_SS_ECDSA_SECP256R1_SHA256:
		return OBJ_SIG_SHA256ECDSA;

	case TLS_SS_ECDSA_SECP384R1_SHA384:
		return OBJ_SIG_SHA384ECDSA;

	case TLS_SS_ECDSA_SECP521R1_SHA512:
		return OBJ_SIG_SHA512ECDSA;

	case TLS_SS_RSA_PSS_RSAE_SHA256:
	case TLS_SS_RSA_PSS_PSS_SHA256:
		return OBJ_SIG_SHA256RSA_PSS;

	case TLS_SS_RSA_PSS_RSAE_SHA384:
	case TLS_SS_RSA_PSS_PSS_SHA384:
		return OBJ_SIG_SHA384RSA_PSS;

	case TLS_SS_RSA_PSS_RSAE_SHA512:
	case TLS_SS_RSA_PSS_PSS_SHA512:
		return OBJ_SIG_SHA512RSA_PSS;

	case TLS_SS_ED25519:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 14,
			     NULL);
		return -1;

	case TLS_SS_ED448:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 15,
			     NULL);
		return -1;

	default:
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 16,
			     NULL);
		return -1;
	}
}

int32_t tls_hs_sighash_get_ai_sig_type(struct tls_hs_sighash_algo sighash) {
	uint16_t sigscheme = tls_hs_sighash_convert_sighash_to_sigscheme(&sighash);
	return tls_hs_sighash_get_ai_sig_type_by_ss(sigscheme);
}

int32_t tls_hs_sighash_get_hash_type(uint16_t sigscheme,
				    enum tls_hs_sighash_hash_algo *hash_algo) {
	switch (sigscheme) {
	case TLS_SS_RSA_PKCS1_MD5:
		*hash_algo = TLS_HASH_ALGO_MD5;
		break;

	case TLS_SS_RSA_PCKS1_SHA1:
	case TLS_SS_ECDSA_SHA1:
	case TLS_SS_DSA_SHA1:
		*hash_algo = TLS_HASH_ALGO_SHA1;
		break;

	case TLS_SS_RSA_PKCS1_SHA224:
	case TLS_SS_DSA_SHA224:
	case TLS_SS_ECDSA_SHA224:
		*hash_algo = TLS_HASH_ALGO_SHA224;
		break;

	case TLS_SS_RSA_PKCS1_SHA256:
	case TLS_SS_ECDSA_SECP256R1_SHA256:
	case TLS_SS_DSA_SHA256:
	case TLS_SS_RSA_PSS_RSAE_SHA256:
	case TLS_SS_RSA_PSS_PSS_SHA256:
		*hash_algo = TLS_HASH_ALGO_SHA256;
		break;

	case TLS_SS_RSA_PKCS1_SHA384:
	case TLS_SS_ECDSA_SECP384R1_SHA384:
	case TLS_SS_RSA_PSS_RSAE_SHA384:
	case TLS_SS_RSA_PSS_PSS_SHA384:
		*hash_algo = TLS_HASH_ALGO_SHA384;
		break;

	case TLS_SS_RSA_PKCS1_SHA512:
	case TLS_SS_ECDSA_SECP521R1_SHA512:
	case TLS_SS_RSA_PSS_RSAE_SHA512:
	case TLS_SS_RSA_PSS_PSS_SHA512:
		*hash_algo = TLS_HASH_ALGO_SHA512;
		break;

	case TLS_SS_ED25519:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 17,
			     NULL);
		return -1;

	case TLS_SS_ED448:
		/* XXX: Not implemented */
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 18,
			     NULL);
		return -1;

	default:
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_SIGSCHEME,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 19,
			     NULL);
		return -1;
	}

	return 0;
}

int32_t tls_hs_sighash_write(TLS *tls, struct tls_hs_msg *msg) {
	/* write acceptable signathre hash pair list in the client
	 * (server). it is not necessary to save this data (it's
	 * clear). */
	int32_t offset = 0;

	struct tls_hs_sighash_list *sighash;
	if ((sighash = tls_hs_sighash_list(tls)) == NULL) {
		return -1;
	}

	/*
	 * RFC8446 4.2.3.  Signature Algorithms
	 *
	 *           SignatureScheme supported_signature_algorithms<2..2^16-2>;
	 *
	 * RFC5246 7.4.1.4.1.  Signature Algorithms
	 *
	 *       SignatureAndHashAlgorithm
	 *         supported_signature_algorithms<2..2^16-2>;
	 */
	const uint16_t list_length_min = 2;
	const uint16_t list_length_max = (2 << (16 - 1)) - 2;
	if (sighash->len * 2 < list_length_min ||
	    list_length_max < sighash->len * 2) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 4, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	if (! tls_hs_msg_write_2(msg, sighash->len * 2)) {
		tls_hs_sighash_free(sighash);
		return -1;
	}
	offset += 2;

	for (int i = 0; i < sighash->len; ++i) {
		if (! tls_hs_msg_write_1(msg, sighash->list[i].hash)) {
			tls_hs_sighash_free(sighash);
			return -1;
		}
		offset++;

		if (! tls_hs_msg_write_1(msg, sighash->list[i].sig)) {
			tls_hs_sighash_free(sighash);
			return -1;
		}
		offset++;
	}

	/* I think it should remember what the client (server) sent as
	 * client (server) acceptable signature/hash algorithm pair
	 * list. but, it is clear here. so I skip that routine. */
	tls_hs_sighash_free(sighash);

	return offset;
}

int32_t tls_hs_sighash_cert_write(TLS *tls, struct tls_hs_msg *msg) {
	int32_t offset = 0;

	struct tls_hs_sighash_list *sighash;
	if ((sighash = tls_hs_sighash_list_cert(tls)) == NULL) {
		return -1;
	}

	/*
	 * RFC8446 4.2.3.  Signature Algorithms
	 *
	 *           SignatureScheme supported_signature_algorithms<2..2^16-2>;
	 *
	 * RFC5246 7.4.1.4.1.  Signature Algorithms
	 *
	 *       SignatureAndHashAlgorithm
	 *         supported_signature_algorithms<2..2^16-2>;
	 */
	const uint16_t list_length_min = 2;
	const uint16_t list_length_max = (2 << (16 - 1)) - 2;
	if (sighash->len * 2 < list_length_min ||
	    list_length_max < sighash->len * 2) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 5, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	if (! tls_hs_msg_write_2(msg, sighash->len * 2)) {
		tls_hs_sighash_free(sighash);
		return -1;
	}
	offset += 2;

	for (int i = 0; i < sighash->len; ++i) {
		uint16_t ss = tls_hs_sighash_convert_sighash_to_sigscheme(
			&(sighash->list[i]));

		if (! tls_hs_msg_write_2(msg, ss)) {
			tls_hs_sighash_free(sighash);
			return -1;
		}
		offset += 2;
	}

	tls_hs_sighash_free(sighash);

	return offset;
}

int32_t tls_hs_sighash_read(TLS *tls,
			    const struct tls_hs_msg *msg,
			    const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 3, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	uint16_t list_length = tls_util_read_2(&(msg->msg[offset]));
	if (msg->len < (offset + read_bytes + list_length)) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 2, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.2.3.  Signature Algorithms
	 *
	 *           SignatureScheme supported_signature_algorithms<2..2^16-2>;
	 *
	 * RFC5246 7.4.1.4.1.  Signature Algorithms
	 *
	 *       SignatureAndHashAlgorithm
	 *         supported_signature_algorithms<2..2^16-2>;
	 */
	const uint16_t list_length_min = 2;
	const uint16_t list_length_max = (2 << (16 - 1)) - 2;
	if (list_length < list_length_min || list_length_max < list_length) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 6, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* check if list_length is odd number */
	if ((list_length % 2) == 1) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	bool need_list_cert = false;
	uint16_t version = tls_util_convert_protover_to_ver(
		&(((TLS *) tls)->negotiated_version));
	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		break;

	case TLS_VER_TLS13:
		need_list_cert = tls->sighash_list_cert == NULL ? true : false;
		break;

	default:
		break;
	}

	const uint32_t base = offset + read_bytes;
	struct tls_hs_sighash_algo algo_list[list_length];
	struct tls_hs_sighash_algo algo_list_cert[list_length];
	int32_t len  = 0;
	int32_t len_cert  = 0;
	for (uint16_t i = 0; i < list_length; i = i + 2) {
		enum tls_hs_sighash_hash_algo hash = msg->msg[base + i + 0];
		enum tls_hs_sighash_sig_algo  algo = msg->msg[base + i + 1];

		struct tls_hs_sighash_algo pair = { .hash = hash, .sig = algo };
		if (tls_hs_sighash_availablep(tls, pair) == true) {
			algo_list[len] = pair;
			len++;
		}

		if (need_list_cert == true &&
		    tls_hs_sighash_availablep_cert(tls, pair) == true) {
			algo_list_cert[len_cert] = pair;
			len_cert++;
		}
	}
	read_bytes += list_length;

	if (len == 0 || (need_list_cert == true && len_cert == 0)) {
		TLS_DPRINTF("sighash: negotiation failure of parameters");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			      ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 20,
			      NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return -1;
	}

	struct tls_hs_sighash_list sighash = {
		.len  = len,
		.list = algo_list
	};

	if (tls->sighash_list != NULL ||
	    (tls->sighash_list = compose_sighash(tls, sighash)) == NULL) {
		return -1;
	}

	struct tls_hs_sighash_list sighash_cert = {
		.len  = len_cert,
		.list = algo_list_cert
	};

	if (need_list_cert == true &&
	    (tls->sighash_list_cert = compose_sighash(tls, sighash_cert))
	    == NULL) {
		return -1;
	}

	return read_bytes;
}

int32_t tls_hs_sighash_cert_read(TLS *tls,
				     const struct tls_hs_msg *msg,
				     const uint32_t offset) {
	uint32_t read_bytes = 0;

	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 21,
			     NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	uint16_t list_length = tls_util_read_2(&(msg->msg[offset]));
	if (msg->len < (offset + read_bytes + list_length)) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 22,
			     NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/*
	 * RFC8446 4.2.3.  Signature Algorithms
	 *
	 *           SignatureScheme supported_signature_algorithms<2..2^16-2>;
	 *
	 * RFC5246 7.4.1.4.1.  Signature Algorithms
	 *
	 *       SignatureAndHashAlgorithm
	 *         supported_signature_algorithms<2..2^16-2>;
	 */
	const uint16_t list_length_min = 2;
	const uint16_t list_length_max = (2 << (16 - 1)) - 2;
	if (list_length < list_length_min || list_length_max < list_length) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* check if list_length is odd number */
	if ((list_length % 2) == 1) {
		TLS_DPRINTF("sighash: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS3, ERR_PT_TLS_HS_EXT_SIGHASH + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	const uint32_t base = offset + read_bytes;
	struct tls_hs_sighash_algo algo_list[list_length];
	int32_t len  = 0;
	for (uint16_t i = 0; i < list_length; i = i + 2) {
		enum tls_hs_sighash_hash_algo hash = msg->msg[base + i + 0];
		enum tls_hs_sighash_sig_algo  algo = msg->msg[base + i + 1];

		struct tls_hs_sighash_algo pair = { .hash = hash, .sig = algo };
		if (tls_hs_sighash_availablep_cert(tls, pair) == true) {
			algo_list[len] = pair;
			len++;
		}
	}
	read_bytes += list_length;

	if (len == 0) {
		TLS_DPRINTF("sighash: negotiation failure of parameters");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 23,
			     NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return -1;
	}

	struct tls_hs_sighash_list sighash = {
		.len  = len,
		.list = algo_list
	};

	if (tls->sighash_list_cert != NULL ||
	    (tls->sighash_list_cert = compose_sighash(tls, sighash)) == NULL) {
		return -1;
	}

	return read_bytes;
}

bool tls_hs_sighash_save_default(TLS *tls) {
	struct tls_hs_sighash_list sighash;

	/*
	 * RFC5246 7.4.1.4.1.  Signature Algorithms
	 *
	 *    If the client does not send the signature_algorithms extension, the
	 *    server MUST do the following:
	 *
	 *    -  If the negotiated key exchange algorithm is one of (RSA, DHE_RSA,
	 *       DH_RSA, RSA_PSK, ECDH_RSA, ECDHE_RSA), behave as if client had
	 *       sent the value {sha1,rsa}.
	 *
	 *    -  If the negotiated key exchange algorithm is one of (DHE_DSS,
	 *       DH_DSS), behave as if the client had sent the value {sha1,dsa}.
	 *
	 *    -  If the negotiated key exchange algorithm is one of (ECDH_ECDSA,
	 *       ECDHE_ECDSA), behave as if the client had sent value {sha1,ecdsa}.
	 */
	switch(tls->keymethod) {
	case TLS_KXC_RSA:
	case TLS_KXC_DHE_RSA:
	case TLS_KXC_DH_RSA:
	case TLS_KXC_ECDH_RSA:
	case TLS_KXC_ECDHE_RSA:
		sighash.list = tls_hs_sighash_list_default_rss;
		sighash.len  = (sizeof (tls_hs_sighash_list_default_rss) /
				sizeof (struct tls_hs_sighash_algo));
		break;

	case TLS_KXC_DHE_DSS:
	case TLS_KXC_DH_DSS:
		sighash.list = tls_hs_sighash_list_default_dsa;
		sighash.len  = (sizeof (tls_hs_sighash_list_default_dsa) /
				sizeof (struct tls_hs_sighash_algo));
		break;

	case TLS_KXC_ECDH_ECDSA:
	case TLS_KXC_ECDHE_ECDSA:
		/* TODO: do implemetation.
		 * Return false because not implemented yet.
		 */

	case TLS_KXC_DH_anon:
	case TLS_KXC_ECDH_anon:
	default:
		OK_set_error(ERR_ST_TLS_UNSUPPORTED_KEY_EXCHANGE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_SIGHASH2 + 24,
			     NULL);
		return false;
	}

	if (tls->sighash_list != NULL ||
	    (tls->sighash_list = compose_sighash(tls, sighash)) == NULL) {
		return false;
	}

	return true;
}
