/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_alert.h"

int32_t tls_hs_extension_parse(TLS *tls,
			      const struct tls_hs_msg *msg,
			      const uint32_t offset) {
	uint32_t read_bytes = 0;

	/* no extesion provided. */
	if (msg->len == offset) {
		return 0;
	}

	/* extension provided. */
	const int32_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("extension: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_PARSE + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}
	read_bytes += length_bytes;

	const uint32_t list_length = tls_util_read_2(&(msg->msg[offset]));
	if (list_length > TLS_EXT_SIZE_MAX) {
		TLS_DPRINTF("extension: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_PARSE + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	if (msg->len < (offset + read_bytes + list_length)) {
		TLS_DPRINTF("extension: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_PARSE + 2, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	/* NOTE: in the case of tls->resession == true, do full check of
	 * extension. */
	for (uint32_t off = 0; off < list_length;) {
		uint32_t base = offset + read_bytes;

		struct tls_extension *ext;
		if ((ext = tls_extension_init()) == NULL) {
			TLS_DPRINTF("tls_extension_init");
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}

		ext->type = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		ext->len = tls_util_read_2(&(msg->msg[base + off]));
		off += 2;

		TLS_DPRINTF("extension: type = %02d, len = %d", ext->type, ext->len);

		if ((ext->opaque = malloc(1 * ext->len)) == NULL) {
			tls_extension_free(ext);
			TLS_DPRINTF("malloc: %s", strerror(errno));
			OK_set_error(ERR_ST_TLS_MALLOC,
				     ERR_LC_TLS5,
				     ERR_PT_TLS_HS_EXT_PARSE + 3,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			return -1;
		}
		memcpy(ext->opaque, &(msg->msg[base+off]), ext->len);
		TAILQ_INSERT_TAIL(&(tls->interim_params->head), ext, link);
		off += ext->len;
	}
	read_bytes += list_length;

	return read_bytes;
}
