/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_alert.h"

/**
 * allocate tls_hs_cookie structure.
 */
struct tls_hs_cookie *alloc_cookie(uint32_t length);

/**
 * save cookie to tls_hs_cookie structure.
 */
struct tls_hs_cookie *save_cookie(uint8_t *cookie, uint32_t length);

/**
 * Write the struct Cookie and its length.
 *
 * RFC8446 4.2.2.  Cookie
 *
 *       struct {
 *           opaque cookie<1..2^16-1>;
 *       } Cookie;
 */
static int32_t write_cookie_for_2ndchello(TLS *tls, struct tls_hs_msg *msg);

/**
 * Read the struct Cookie and its length.
 *
 * RFC8446 4.2.2.  Cookie
 *
 *       struct {
 *           opaque cookie<1..2^16-1>;
 *       } Cookie;
 */
static int32_t read_cookie_in_hrr(TLS *tls, const struct tls_hs_msg *msg,
			   const uint32_t offset);

struct tls_hs_cookie *alloc_cookie(uint32_t length)
{
	struct tls_hs_cookie *cookie;
	if ((cookie = malloc(sizeof(struct tls_hs_cookie))) == NULL) {
		TLS_DPRINTF("malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE, NULL);
		return NULL;
	}

	cookie->len = length;
	if ((cookie->content = malloc(length)) == NULL) {
		TLS_DPRINTF("malloc: %s", strerror(errno));
		OK_set_error(ERR_ST_TLS_MALLOC,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 1, NULL);
		free(cookie);
		return NULL;
	}

	return cookie;
}

struct tls_hs_cookie *save_cookie(uint8_t *cookie, uint32_t length)
{
	struct tls_hs_cookie *cookie_st;
	if ((cookie_st = alloc_cookie(length)) == NULL) {
		return NULL;
	}

	memcpy(cookie_st->content, cookie, length);

	return cookie_st;
}

static int32_t write_cookie_for_2ndchello(TLS *tls, struct tls_hs_msg *msg)
{
	int32_t offset = 0;

	const int32_t length_bytes = 2;
	if (tls_hs_msg_write_2(msg, tls->cookie->len) == false) {
		TLS_DPRINTF("cookie: tls_hs_msg_write_2");
		return -1;
	}
	offset += length_bytes;

	/*
	 * RFC8446 4.2.2.  Cookie
	 *
	 *           opaque cookie<1..2^16-1>;
	 */
	const uint16_t cookie_len_min = 1;
	if (tls->cookie->len < cookie_len_min) {
		TLS_DPRINTF("cookie: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	if (tls_hs_msg_write_n(msg, tls->cookie->content, tls->cookie->len)
	    == false) {
		TLS_DPRINTF("cookie: tls_hs_msg_write_n");
		return -1;
	}
	offset += tls->cookie->len;

	return offset;
}

static int32_t read_cookie_in_hrr(TLS *tls, const struct tls_hs_msg *msg,
			   const uint32_t offset)
{
	uint32_t read_bytes = 0;

	const uint16_t length_bytes = 2;
	if (msg->len < (offset + length_bytes)) {
		TLS_DPRINTF("cookie: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 2, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	uint16_t length = tls_util_read_2(&(msg->msg[offset]));
	read_bytes += length_bytes;

	/*
	 * RFC8446 4.2.2.  Cookie
	 *
	 *           opaque cookie<1..2^16-1>;
	 */
	const uint16_t cookie_len_min = 1;
	if (length < cookie_len_min) {
		TLS_DPRINTF("cookie: invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_DECODE_ERROR);
		return -1;
	}

	if ((tls->cookie = save_cookie(&(msg->msg[offset+read_bytes]), length))
	    == NULL) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}
	read_bytes += length;

	return read_bytes;
}

int32_t tls_hs_cookie_write(TLS *tls, struct tls_hs_msg *msg)
{
	int32_t offset = 0;
	uint16_t version;

	switch (msg->type) {
	case TLS_HANDSHAKE_CLIENT_HELLO:
		/* check if state is for 2nd client hello */
		if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_2NDCHELLO)
		    == false ||
		    tls->cookie == NULL) {
			return 0;
		}
		break;

	case TLS_HANDSHAKE_SERVER_HELLO:
		version = tls_util_convert_protover_to_ver(
			&(tls->negotiated_version));
		switch (version) {
		case TLS_VER_TLS13:
			/*
			 * TODO: implement process of sending cookie to client
			 * in hello retry request.
			 */
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
		default:
			return 0;
		}
		break;

	default:
		return 0;
	}

	const uint32_t type_bytes = 2;
	if (tls_hs_msg_write_2(msg, TLS_EXT_COOKIE) == false) {
		return -1;
	}
	offset += type_bytes;

	/* write dummy length bytes. */
	int32_t pos = msg->len;

	const uint32_t len_bytes = 2;
	if (tls_hs_msg_write_2(msg, 0) == false) {
		return -1;
	}
	offset += len_bytes;

	int32_t act_len;
	switch (msg->type) {
	case TLS_HANDSHAKE_SERVER_HELLO:
		/*
		 * TODO: implement process of sending cookie to client in hello
		 * retry request.
		 *
		 * RFC8446 4.2.2.  Cookie
		 *
		 *    When sending a HelloRetryRequest, the server MAY provide a "cookie"
		 *    extension to the client (this is an exception to the usual rule that
		 *    the only extensions that may be sent are those that appear in the
		 *    ClientHello).
		 */
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 3, NULL);
		return -1;

	case TLS_HANDSHAKE_CLIENT_HELLO:
		/*
		 * RFC8446 4.2.2.  Cookie
		 *
		 *                   When sending the new ClientHello, the client MUST copy
		 *    the contents of the extension received in the HelloRetryRequest into
		 *    a "cookie" extension in the new ClientHello.  Clients MUST NOT use
		 *    cookies in their initial ClientHello in subsequent connections.
		 */
		if ((act_len = write_cookie_for_2ndchello(tls, msg)) < 0) {
			TLS_DPRINTF("write_cookie_for_2ndchello");
			return -1;
		}
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 4, NULL);
		return -1;
	}
	offset += act_len;

	const int32_t extlen_max = TLS_EXT_SIZE_MAX;
	if (act_len > extlen_max) {
		TLS_DPRINTF("invalid record length");
		OK_set_error(ERR_ST_TLS_INVALID_RECORD_LENGTH,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return -1;
	}

	tls_util_write_2(&(msg->msg[pos]), act_len);

	return offset;
}

int32_t tls_hs_cookie_read(TLS *tls, const struct tls_hs_msg *msg,
			   const uint32_t offset)
{
	uint32_t read_bytes = 0;

	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		break;

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		return 0;
	}

	int32_t len = 0;
	switch (msg->type) {
	case TLS_HANDSHAKE_SERVER_HELLO:
		/*
		 * RFC8446 4.2.2.  Cookie
		 *
		 *                   When sending the new ClientHello, the client MUST copy
		 *    the contents of the extension received in the HelloRetryRequest into
		 *    a "cookie" extension in the new ClientHello.  Clients MUST NOT use
		 *    cookies in their initial ClientHello in subsequent connections.
		 */
		if ((len = read_cookie_in_hrr(tls, msg, offset)) < 0) {
			TLS_DPRINTF("cookie: read_cookie_in_hrr");
			return -1;
		}
		break;

	case TLS_HANDSHAKE_CLIENT_HELLO:
		/*
		 * TODO: implement process of receiving cookie from client
		 * in 2nd client hello.
		 *
		 * RFC8446 4.2.2.  Cookie
		 *
		 *    When sending a HelloRetryRequest, the server MAY provide a "cookie"
		 *    extension to the client (this is an exception to the usual rule that
		 *    the only extensions that may be sent are those that appear in the
		 *    ClientHello).
		 */
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 5, NULL);
		return -1;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_EXT_COOKIE + 6, NULL);
		return -1;
	}
	read_bytes += len;

	return read_bytes;
}

void tls_hs_cookie_free(struct tls_hs_cookie *cookie)
{
	if (cookie == NULL) {
		return;
	}

	free(cookie->content);
	free(cookie);
}
