/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_record.h"
#include "tls_handshake.h"
#include "tls_cipher.h"
#include "tls_alert.h"
#include "tls_ccs.h"

/** @see tls_key.c */
extern bool tls_key_make_master_secret(TLS *tls);

/** @see tls_key.c */
extern bool tls_key_make_key_block(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_early_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_handshake_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_handshake_traffic_secrets(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_master_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_application_traffic_secret(TLS *tls,
					      char *label,
					      struct tls_connection *connection);

/** @see tls_key.c */
bool tls_key_make_traffic_key(TLS *tls, struct tls_connection *connection);

/** @see finale.c */
extern enum hs_phase tls_hs_finale_write_first(TLS *tls);

/** @see finale.c */
extern enum hs_phase tls_hs_finale_read_first(TLS *tls);

/**
 * confirm whether the server sends (server) certificate handshake
 * protocol message.
 */
static bool check_auth_by_cert(const TLS *tls);

/**
 * confirm whether the server sends (server) certificate request
 * handshake protocol message.
 */
static bool check_send_certreq(const TLS *tls);

/**
 * make master secret and keyblock.
 */
static bool make_key_block(TLS *tls);

/**
 * derive handshake secrets.
 */
static bool derive_handshake_secrets(TLS *tls);

/**
 * select key exchange method.
 */
static bool select_key_exchange_method(TLS *tls);

/**
 * establish protocol version based on client hello message.
 */
static bool establish_protocol_version(TLS *tls);

/**
 * search tls_hs_interim_params structure for an extension that have
 * specified type.
 */
static struct tls_extension *search_extension(
	struct tls_hs_interim_params *params, enum tls_extension_type type);

/**
 * compare client hello messages to check if there is no difference except
 * a few extensions.
 */
static bool compare_client_hellos(TLS *tls);

/**
 * interpret client hello values according to protocol version.
 */
static bool interpret_client_hello(TLS *tls);

/**
 * interpret second client hello(received after hello retry request) values
 * according to protocol version.
 */
static bool interpret_2ndclient_hello(TLS *tls);

/**
 * replace client hello with message hash about hash of handshake message.
 */
static bool replace_transcript_with_msghash(TLS *tls);

/**
 * make server hello data and send it to the client.
 */
static bool write_server_hello(TLS *tls, enum tls_state state);

/**
 * make encrypted extensions data and send it to the client.
 */
static bool write_encrypted_extensions_tls13(TLS *tls);

/**
 * make (server) certificate data and send it to the client.
 */
static bool write_certificate_up_to_tls12(TLS *tls);
static bool write_certificate_tls13(TLS *tls);

/**
 * make (server) certificate verify data and send it to the client.
 */
static bool write_certificate_verify_tls13(TLS *tls);

/**
 * make server key exchange data and send it to the client.
 */
static bool write_server_key_exchange_up_to_tls12(TLS *tls);

/**
 * make certificate request data and send it to the client.
 */
static bool write_certificate_request_up_to_tls12(TLS *tls);
static bool write_certificate_request_tls13(TLS *tls);

/**
 * make sever hello done data and send it to the client.
 */
static bool write_server_hello_done_up_to_tls12(TLS *tls);

/**
 * read handshake data from client and parse it as considering it client
 * hello data.
 */
static bool read_client_hello(TLS *tls, enum tls_state state);

/**
 * read handshake data from client and parse it as considering it
 * (client) certificate data.
 */
static bool read_certificate_up_to_tls12(TLS *tls);
static bool read_certificate_tls13(TLS *tls);

/**
 * read handshake data from client and parse it as considering it client
 * key exchnage data.
 */
static bool read_client_key_exchange_up_to_tls12(TLS *tls);

/**
 * read handshake data from client and parse it as considering it
 * certificate verify data.
 */
static bool read_certificate_verify_up_to_tls12(TLS *tls);
static bool read_certificate_verify_tls13(TLS *tls);

/**
 * do TLS_HS_PHASE_HELLO phase.
 *
 * if resumption is determined in the negotiation between the client
 * hello and server hello, make keyblock and go to final phase.
 */
static enum hs_phase do_phase_hello(TLS *tls);

/**
 * do TLS_HS_PHASE_CERTIFICATE_SENDING phase.
 */
static bool do_phase_send_cert_up_to_tls12(TLS *tls);
static bool do_phase_send_cert_tls13(TLS *tls UNUSED);
static enum hs_phase do_phase_send_cert(TLS *tls);

/**
 * do TLS_HS_PHASE_CERTIFICATE_RECVING phase.
 */
static bool do_phase_recv_cert_up_to_tls12(TLS *tls);
static bool do_phase_recv_cert_tls13(TLS *tls UNUSED);
static enum hs_phase do_phase_recv_cert(TLS *tls);

/**
 * do TLS_HS_PHASE_FINAL phase.
 */
static enum hs_phase do_phase_final_up_to_tls12(TLS *tls);
static enum hs_phase do_phase_final_tls13(TLS *tls);
static enum hs_phase do_phase_final(TLS *tls);

/**
 * clean up handshake specific data.
 */
static void cleanup(TLS *tls);

static bool check_auth_by_cert(const TLS *tls) {
	/* TODO: consider better function name. */
	switch(tls->keymethod) {
	case TLS_KXC_RSA:
		break;

	case TLS_KXC_ECDH_RSA:
	case TLS_KXC_ECDHE_RSA:
		/* TODO: XXX: Fix later. */
		break;

	case TLS_KXC_DHE:
	case TLS_KXC_ECDHE:
		break;

	default:
		return false;
	}

	return true;
}

static bool check_send_certreq(const TLS *tls) {
	/* *non-anonymous* server can optionally request a certificate
	    from the client (by RFC 5246 section 7.4.4). */
	if (tls->opt.use_certreq == true && check_auth_by_cert(tls) == true) {
		return true;
	}

	return false;
}

static bool make_key_block(TLS *tls) {
	/* generate master secret. */
	if (! tls_key_make_master_secret(tls)) {
		TLS_DPRINTF("hs: server: tls_key_make_master_secret");
		return false;
	}

	/* generate key block. */
	if (! tls_key_make_key_block(tls)) {
		TLS_DPRINTF("hs: server: tls_key_keyblock_gen");
		return false;
	}

	return true;
}

static bool derive_handshake_secrets(TLS *tls) {
	if (tls_key_derive_handshake_secret(tls) == false) {
		TLS_DPRINTF("hs: server: tls_key_derive_handshake_secret");
		return false;
	}

	tls_hs_update_hash(tls);

	if (tls_key_derive_handshake_traffic_secrets(tls) == false) {
		TLS_DPRINTF("hs: server: tls_key_derive_handshake_traffic_secrets");
		return false;
	}

	return true;
}

static bool select_key_exchange_method(TLS *tls) {
	/*
	 * RFC8446 2.2.  Resumption and Pre-Shared Key (PSK)
	 *
	 *    As the server is authenticating via a PSK, it does not send a
	 *    Certificate or a CertificateVerify message.  When a client offers
	 *    resumption via a PSK, it SHOULD also supply a "key_share" extension
	 *    to the server to allow the server to decline resumption and fall back
	 *    to a full handshake, if needed.  The server responds with a
	 *    "pre_shared_key" extension to negotiate the use of PSK key
	 *    establishment and can (as shown here) respond with a "key_share"
	 *    extension to do (EC)DHE key establishment, thus providing forward
	 *    secrecy.
	 */

	/*
	 * TODO: check if resumption is available at first, then fall back to
	 * full handshake. currently, resumption is not implemented, so skip
	 * this check.
	*/

	/*
	 * RFC8446 4.1.1.  Cryptographic Negotiation
	 *
	 *    -  A "supported_groups" (Section 4.2.7) extension which indicates the
	 *       (EC)DHE groups which the client supports and a "key_share"
	 *       (Section 4.2.8) extension which contains (EC)DHE shares for some
	 *       or all of these groups.
	 *
	 *    -  A "signature_algorithms" (Section 4.2.3) extension which indicates
	 *       the signature algorithms which the client can accept.  A
	 *       "signature_algorithms_cert" extension (Section 4.2.3) may also be
	 *       added to indicate certificate-specific signature algorithms.
	 *
	 *    -  A "pre_shared_key" (Section 4.2.11) extension which contains a
	 *       list of symmetric key identities known to the client and a
	 *       "psk_key_exchange_modes" (Section 4.2.9) extension which indicates
	 *       the key exchange modes that may be used with PSKs.
	 *
	 *    If the server does not select a PSK, then the first three of these
	 *    options are entirely orthogonal: the server independently selects a
	 *    cipher suite, an (EC)DHE group and key share for key establishment,
	 *    and a signature algorithm/certificate pair to authenticate itself to
	 *    the client.  If there is no overlap between the received
	 *    "supported_groups" and the groups supported by the server, then the
	 *    server MUST abort the handshake with a "handshake_failure" or an
	 *    "insufficient_security" alert.
	 *
	 *    If the server selects a PSK, then it MUST also select a key
	 *    establishment mode from the set indicated by the client's
	 *    "psk_key_exchange_modes" extension (at present, PSK alone or with
	 *    (EC)DHE).  Note that if the PSK can be used without (EC)DHE, then
	 *    non-overlap in the "supported_groups" parameters need not be fatal,
	 *    as it is in the non-PSK case discussed in the previous paragraph.
	 */
	struct tls_hs_interim_params *params = tls->interim_params;
	if (params->recv_ext_flags[TLS_EXT_SUPPORTED_GROUPS] == false) {
		TLS_DPRINTF("server: missing supported_groups extension");
		OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 0, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_MISSING_EXTENSION);
		return false;
	}

	if (params->recv_ext_flags[TLS_EXT_KEY_SHARE] == false) {
		TLS_DPRINTF("server: missing key_share extension");
		OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 1, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_MISSING_EXTENSION);
		return false;
	}

	if (params->recv_ext_flags[TLS_EXT_SIGNATURE_ALGO] == false) {
		TLS_DPRINTF("server: missing signature_algorithms extension");
		OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 2, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_MISSING_EXTENSION);
		return false;
	}

	struct tls_hs_key_share *share;
	TAILQ_FOREACH(share, &(params->share_head), link) {
		/* TODO: add procedure for Finite-Field-Group */
		switch (share->group) {
		case TLS_NAMED_GROUP_SECP256R1:
		case TLS_NAMED_GROUP_X25519:
		case TLS_NAMED_GROUP_X448:
			tls->ecdh->namedcurve = share->group;
			tls->keymethod = TLS_KXC_ECDHE;

			if (tls_hs_ecdhkey_set_peer_pubkey(
				tls->ecdh, &(share->key)) != true) {
				TLS_DPRINTF("server: tls_hs_ecdhkey_set_peer_pubkey");
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
				return false;
			}

			TLS_DPRINTF("server: Decided group: %d", share->group);
			goto selected;

		case TLS_NAMED_GROUP_SECP384R1:
		case TLS_NAMED_GROUP_SECP521R1:
			/* Not implemented */
			break;

		default:
			break;
		}
	}

	/*
	 * RFC8446 4.1.1.  Cryptographic Negotiation
	 *
	 *                 If there is no overlap between the received
	 *    "supported_groups" and the groups supported by the server, then the
	 *    server MUST abort the handshake with a "handshake_failure" or an
	 *    "insufficient_security" alert.
	 */
	struct tls_hs_ecc_eclist *eclist = tls->ecdh->eclist;
	if (eclist->len == 0) {
		TLS_DPRINTF("server: no available group");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 3, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	/*
	 * TODO: select key exchange method by namedcurve or group when
	 * implementing Finite-Field-Group.
	 */
	tls->ecdh->namedcurve = eclist->list[0];
	tls->keymethod = TLS_KXC_ECDHE;

	/* store first client hello information for later use. */
	tls->first_hello_params = tls->interim_params;

	/* try to send hello retry request */
	tls_hs_change_state(tls, TLS_STATE_HS_BEFORE_SEND_HRREQ);

selected:
	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_HRREQ) == false) {
		struct tls_extension *ext;
		while (!TAILQ_EMPTY(&(tls->interim_params->head))) {
			ext = TAILQ_FIRST(&(tls->interim_params->head));
			TAILQ_REMOVE(&(tls->interim_params->head), ext, link);
			tls_extension_free(ext);
		}
	}

	return true;
}

static bool establish_protocol_version(TLS *tls) {
	struct tls_extension *ext, *ext_sv = NULL;
	uint16_t version;
	uint16_t client_version;
	uint16_t record_version;

	/*
	 * TODO:
	 *
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    If a server established a TLS connection with a previous version of
	 *    TLS and receives a TLS 1.3 ClientHello in a renegotiation, it MUST
	 *    retain the previous protocol version.  In particular, it MUST NOT
	 *    negotiate TLS 1.3.
	 */

	TAILQ_FOREACH(ext, &(tls->interim_params->head), link) {
		if (ext->type == TLS_EXT_SUPPORTED_VERSIONS) {
			if (ext_sv == NULL) {
				ext_sv = ext;
			} else {
				/*
				 * supported versions extensions come, so handle this
				 * as an error. see RFC8446 p.37.
				 */
				OK_set_error(ERR_ST_TLS_UNSUPPORTED_EXTENSION,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_SERVER2 + 4,
					     NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNSUPPORTED_EXTENSION);
				return false;
			}
		}
	}

	/* version negotiation by supported versions extension */
	if (ext_sv != NULL) {
		/*
		 * RFC8446 4.2.1.  Supported Versions
		 *
		 *    If this extension is present in the ClientHello, servers MUST NOT use
		 *    the ClientHello.legacy_version value for version negotiation and MUST
		 *    use only the "supported_versions" extension to determine client
		 *    preferences.  Servers MUST only select a version of TLS present in
		 *    that extension and MUST ignore any unknown versions that are present
		 *    in that extension.  Note that this mechanism makes it possible to
		 *    negotiate a version prior to TLS 1.2 if one side supports a sparse
		 *    range.  Implementations of TLS 1.3 which choose to support prior
		 *    versions of TLS SHOULD support TLS 1.2.  Servers MUST be prepared to
		 *    receive ClientHellos that include this extension but do not include
		 *    0x0304 in the list of versions.
		 *
		 *    A server which negotiates a version of TLS prior to TLS 1.3 MUST set
		 *    ServerHello.version and MUST NOT send the "supported_versions"
		 *    extension.  A server which negotiates TLS 1.3 MUST respond by sending
		 *    a "supported_versions" extension containing the selected version
		 *    value (0x0304).  It MUST set the ServerHello.legacy_version field to
		 *    0x0303 (TLS 1.2).  Clients MUST check for this extension prior to
		 *    processing the rest of the ServerHello (although they will have to
		 *    parse the ServerHello in order to read the extension).  If this
		 *    extension is present, clients MUST ignore the
		 *    ServerHello.legacy_version value and MUST use only the
		 *    "supported_versions" extension to determine the selected version.  If
		 *    the "supported_versions" extension in the ServerHello contains a
		 *    version not offered by the client or contains a version prior to
		 *    TLS 1.3, the client MUST abort the handshake with an
		 *    "illegal_parameter" alert.
		 */

		struct tls_hs_msg msg = {
			.type = TLS_HANDSHAKE_CLIENT_HELLO,
			.len = ext_sv->len,
			.max = ext_sv->len,
			.msg = ext_sv->opaque
		};

		if (tls_hs_supported_versions_read(tls, &msg, 0) < 0) {
			goto err;
		}

		uint16_t list_len = tls->peer_supported_versions.len;
		uint16_t *list = tls->peer_supported_versions.list;
		uint16_t msg_version;

		for (int i = 0; i < list_len; i++) {
			version = list[i];

			/*
			 * check whether protocol version is acceptable.
			 * supported_versions in TLS structure may be controlled
			 * by command line option in future.
			 */
			if (tls_util_check_version_in_supported_version(
				&(tls->supported_versions), version) == false) {
				continue;
			}

			switch (version) {
			case TLS_VER_SSL30:
			case TLS_VER_TLS10:
			case TLS_VER_TLS11:
				/* Not implemented */
				continue;

			case TLS_VER_TLS12:
				client_version = version;
				record_version = version;
				goto found_supported_version;

			case TLS_VER_TLS13:
				msg_version = tls_util_convert_protover_to_ver(
					&(tls->interim_params->version));
				if (msg_version != TLS_VER_TLS12) {
					OK_set_error(
						ERR_ST_TLS_ILLEGAL_PARAMETER,
						ERR_LC_TLS5,
						ERR_PT_TLS_HS_CS_SERVER2 + 5,
						NULL);
					TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_ILLEGAL_PARAMETER);
					return false;
				}
				/*
				 * RFC8446 5.1.  Record Layer
				 *
				 *                                     In order to maximize backward
				 *    compatibility, a record containing an initial ClientHello SHOULD have
				 *    version 0x0301 (reflecting TLS 1.0) and a record containing a second
				 *    ClientHello or a ServerHello MUST have version 0x0303 (reflecting
				 *    TLS 1.2).
				 */
				client_version = TLS_VER_TLS12;
				record_version = TLS_VER_TLS12;
				goto found_supported_version;

			default:
				/* Ignore unknown version */
				continue;
			}
		}

		goto err;
	} else {
		version = tls_util_convert_protover_to_ver(
			&(tls->interim_params->version));

		/*
		 * RFC8446 D.5.  Security Restrictions Related to Backward Compatibility
		 *
		 *    The security of SSL 3.0 [RFC6101] is considered insufficient for the
		 *    reasons enumerated in [RFC7568], and it MUST NOT be negotiated for
		 *    any reason.
		 */
		/*
		 * RFC8446 D.5.  Security Restrictions Related to Backward Compatibility
		 *
		 *   Implementations MUST NOT send a ClientHello.legacy_version or
		 *   ServerHello.legacy_version set to 0x0300 or less.  Any endpoint
		 *   receiving a Hello message with ClientHello.legacy_version or
		 *   ServerHello.legacy_version set to 0x0300 MUST abort the handshake
		 *   with a "protocol_version" alert.
		 */
		if (version < TLS_VER_SSL30) {
			goto err;
		}

		/*
		 * RFC8446 D.2.  Negotiating with an Older Client
		 *
		 *                                                         If the
		 *    "supported_versions" extension is not present, the server MUST
		 *    negotiate the minimum of ClientHello.legacy_version and TLS 1.2.  For
		 *    example, if the server supports TLS 1.0, 1.1, and 1.2, and
		 *    legacy_version is TLS 1.0, the server will proceed with a TLS 1.0
		 *    ServerHello.  If the "supported_versions" extension is absent and the
		 *    server only supports versions greater than
		 *    ClientHello.legacy_version, the server MUST abort the handshake with
		 *    a "protocol_version" alert.
		 */
		switch(version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
			/* TODO: implementation */
			goto err;

		case TLS_VER_TLS12:
			break;

		default:
			/*
			 * RFC8446 4.2.1.  Supported Versions
			 *
			 * If this extension is not present, servers which are compliant with
			 * this specification and which also support TLS 1.2 MUST negotiate
			 * TLS 1.2 or prior as specified in [RFC5246], even if
			 * ClientHello.legacy_version is 0x0304 or later.  Servers MAY abort the
			 * handshake upon receiving a ClientHello with legacy_version 0x0304 or
			 * later.
			 */
			version = TLS_VER_TLS12;
			break;
		}

		client_version = version;
		record_version = version;
	}

found_supported_version:
	/* save client version for future handshake. this value is used
	 * by premaster_serect. */
	tls_util_convert_ver_to_protover(client_version, &(tls->client_version));
	TLS_DPRINTF("server: client_version: %04x", client_version);

	/* select negotiated version. if you want to select version
	 * other than version that is provided by client, modify
	 * here. */
	tls_util_convert_ver_to_protover(version, &(tls->negotiated_version));
	TLS_DPRINTF("server: negotiated version: %04x", version);

	/*
	 * set appropriate version used for tls record according to
	 * specifications. */
	tls_util_convert_ver_to_protover(record_version, &(tls->record_version));
	TLS_DPRINTF("server: record_version: %04x", record_version);

	return true;

err:
	tls->errnum = TLS_ERR_PROTOCOL_VERSION;
	OK_set_error(ERR_ST_TLS_PROTOCOL_VERSION,
		     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 6, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_PROTOCOL_VERSION);
	return false;
}

static struct tls_extension *search_extension(
	struct tls_hs_interim_params *params, enum tls_extension_type type) {

	struct tls_extension *ext;
	TAILQ_FOREACH(ext, &(params->head), link) {
		if (ext->type == type) {
			return ext;
		}
	}

	return NULL;
}

static bool compare_client_hellos(TLS *tls) {
	struct tls_hs_interim_params *first_params = tls->first_hello_params;
	struct tls_hs_interim_params *second_params = tls->interim_params;

	/*
	 * RFC8446 4.1.2.  Client Hello
	 *
	 *    ClientHello as its first TLS message.  The client will also send a
	 *    ClientHello when the server has responded to its ClientHello with a
	 *    HelloRetryRequest.  In that case, the client MUST send the same
	 *    ClientHello without modification, except as follows:
	 *
	 *    -  If a "key_share" extension was supplied in the HelloRetryRequest,
	 *       replacing the list of shares with a list containing a single
	 *       KeyShareEntry from the indicated group.
	 *
	 *    -  Removing the "early_data" extension (Section 4.2.10) if one was
	 *       present.  Early data is not permitted after a HelloRetryRequest.
	 *
	 *    -  Including a "cookie" extension if one was provided in the
	 *       HelloRetryRequest.
	 *
	 *    -  Updating the "pre_shared_key" extension if present by recomputing
	 *       the "obfuscated_ticket_age" and binder values and (optionally)
	 *       removing any PSKs which are incompatible with the server's
	 *       indicated cipher suite.
	 *
	 *    -  Optionally adding, removing, or changing the length of the
	 *       "padding" extension [RFC7685].
	 *
	 *    -  Other modifications that may be allowed by an extension defined in
	 *       the future and present in the HelloRetryRequest.
	 */

	/*
	 * check if second client hello message has the same content as first client
	 * hello except extensions.
	 */
	if (memcmp(&(first_params->version), &(second_params->version),
		   sizeof(struct tls_protocol_version)) != 0) {
		TLS_DPRINTF("protocol version is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (first_params->seslen != second_params->seslen) {
		TLS_DPRINTF("session parameter is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (memcmp(first_params->session, second_params->session,
		   first_params->seslen) != 0) {
		TLS_DPRINTF("session parameter is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 9, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (first_params->cipher_suites->len !=
	    second_params->cipher_suites->len) {
		TLS_DPRINTF("cipher suite list is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 10, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (memcmp(first_params->cipher_suites->list,
		   second_params->cipher_suites->list,
		   first_params->cipher_suites->len) != 0) {
		TLS_DPRINTF("cipher suite list is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 11, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (first_params->cmplen != second_params->cmplen) {
		TLS_DPRINTF("compression method is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 12, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	if (memcmp(first_params->cmplist, second_params->cmplist,
		   first_params->cmplen) != 0) {
		TLS_DPRINTF("compression method is different from first one");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 13, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	/*
	 * compare extensions between first client hello and second client hello
	 * considering exceptions above.
	 */
	struct tls_extension *first_ext;
	struct tls_extension *second_ext;
	struct tls_extension *temp_ext;
	TAILQ_FOREACH_SAFE(first_ext, &(first_params->head), link, temp_ext) {
		switch (first_ext->type) {
		case TLS_EXT_PADDING:
		case TLS_EXT_PRE_SHARED_KEY:
		case TLS_EXT_EARLY_DATA:
		case TLS_EXT_KEY_SHARE:
			continue;

		default:
			if ((second_ext = search_extension(second_params,
							   first_ext->type)) == NULL) {
				TLS_DPRINTF("server: missing  extension");
				OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_SERVER2 + 14,
					     NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_MISSING_EXTENSION);
				return false;
			}

			if (first_ext->len != second_ext->len) {
				TLS_DPRINTF("server: extension contains"
					    " different value");
				OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_SERVER2 + 15,
					     NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_ILLEGAL_PARAMETER);
				return false;
			}

			if (memcmp(first_ext->opaque, second_ext->opaque,
				   first_ext->len) != 0) {
				TLS_DPRINTF("server: extension contains"
					    " different value");
				OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_SERVER2 + 16,
					     NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_ILLEGAL_PARAMETER);
				return false;
			}

			second_params->recv_ext_flags[first_ext->type] = true;
			TAILQ_REMOVE(&(second_params->head), second_ext, link);
			TAILQ_REMOVE(&(first_params->head), first_ext, link);
			tls_extension_free(first_ext);
			tls_extension_free(second_ext);
			break;
		}
	}

	/*
	 * check if remaining extensions are permitted to be in second client
	 * hello.
	 */
	struct tls_extension *ext;
	TAILQ_FOREACH(ext, &(second_params->head), link) {
		switch (ext->type) {
		case TLS_EXT_PADDING:
		case TLS_EXT_PRE_SHARED_KEY:
		case TLS_EXT_KEY_SHARE:
		case TLS_EXT_COOKIE:
			TLS_DPRINTF("remaining extension type: %u", ext->type);
			break;

		default:
			OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 17,
				     NULL);
			return false;
		}
	}

	return true;
}

static bool interpret_client_hello(TLS *tls) {
	if (! tls_hs_chello_interpret(tls)) {
		return false;
	}

	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		if (select_key_exchange_method(tls) == false) {
			return false;
		}
		break;

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		break;
	}

	if (! tls_cipher_select(tls, tls->chello_cipher_suites)) {
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_HRREQ) == true) {
		return true;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_AFTER_RECV_CHELLO);

	return true;
}

static bool interpret_2ndclient_hello(TLS *tls) {
	/*
	 * RFC8446 4.1.4.  Hello Retry Request
	 *
	 *                          Servers MUST ensure that they negotiate the
	 *    same cipher suite when receiving a conformant updated ClientHello (if
	 *    the server selects the cipher suite as the first step in the
	 *    negotiation, then this will happen automatically).
	 */
	/*
	 * compare 1st client hello with 2nd client hello except extensions.
	 * if they have same content, there is no need to interpret 2nd
	 * client hello again. otherwise, 2nd client hello is illegal.
	 */
	if (compare_client_hellos(tls) == false) {
		return false;
	}

	if (tls_hs_chello_interpret_extensions(tls) == false) {
		return false;
	}

	/* check if public key is not empty */
	if (TAILQ_EMPTY(&(tls->interim_params->share_head))) {
		TLS_DPRINTF("public key is empty");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 18, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	struct tls_hs_key_share *share;
	share = TAILQ_FIRST(&(tls->interim_params->share_head));
	if (tls_hs_ecdhkey_set_peer_pubkey(tls->ecdh, &(share->key))
	    != true) {
		TLS_DPRINTF("server: tls_hs_ecdhkey_set_peer_pubkey");
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 19, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		return false;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_AFTER_RECV_2NDCHELLO);

	return true;
}

static bool replace_transcript_with_msghash(TLS *tls) {
	tls_hs_update_hash(tls);

	struct tls_hs_msg *msg;
	if ((msg = tls_hs_msghash_make(tls)) == NULL) {
		return false;
	}

	/* init handshake message again and enqueue message hash */
	tls_hs_hash_init(tls);
	TAILQ_INSERT_TAIL(tls->queue_handshake, msg, link);

	return true;
}

static bool write_server_hello(TLS *tls, enum tls_state state) {
	struct tls_hs_msg *msg;

	if ((msg = tls_hs_shello_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: server: tls_hs_shello_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: server: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, state);

	return true;
}

static bool write_encrypted_extensions_tls13(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((msg = tls_hs_encext_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: server: tls_hs_encext_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: server: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_SEND_ENCEXT);

	return true;
}

static bool write_certificate_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if (check_auth_by_cert(tls)) {
		if ((msg = tls_hs_scert_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: server: tls_hs_scert_compose");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: server: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_SCERT);
	}

	return true;
}

static bool write_certificate_tls13(TLS *tls) {
	/*
	 * RFC8446 4.1.1.  Cryptographic Negotiation
	 *
	 *   If the server successfully selects parameters and does not require a
	 *   HelloRetryRequest, it indicates the selected parameters in the
	 *   ServerHello as follows:
	 *
	 *   -  If PSK is being used, then the server will send a "pre_shared_key"
	 *      extension indicating the selected key.
	 *
	 *   -  When (EC)DHE is in use, the server will also provide a "key_share"
	 *      extension.  If PSK is not being used, then (EC)DHE and
	 *      certificate-based authentication are always used.
	 *
	 *   -  When authenticating via a certificate, the server will send the
	 *      Certificate (Section 4.4.2) and CertificateVerify (Section 4.4.3)
	 *      messages.  In TLS 1.3 as defined by this document, either a PSK or
	 *      a certificate is always used, but not both.  Future documents may
	 *      define how to use them together.
	 */
	return write_certificate_up_to_tls12(tls);
}

static bool write_certificate_verify_tls13(TLS *tls) {
	struct tls_hs_msg *msg;

	/*
	 * RFC8446 4.4.3.  Certificate Verify
	 *
	 *                       Servers MUST send this message when authenticating
	 *    via a certificate.
	 */
	if (tls_hs_check_state(tls, TLS_STATE_HS_SEND_SCERT)) {
		if ((msg = tls_hs_certvfy_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: server: tls_hs_certvfy_compose");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: server: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_SCERTVFY);
	}

	return true;
}

static bool write_server_key_exchange_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if (tls_hs_skeyexc_need_to_send(tls)) {
		if ((msg = tls_hs_skeyexc_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: server: tls_hs_skeyexc_parse");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: server: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_SKEYEXC);
	}

	return true;
}

static bool write_certificate_request_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if (check_send_certreq(tls)) {
		if ((msg = tls_hs_certreq_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: server: tls_hs_certreq_compose");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: server: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_CERTREQ);
	}

	return true;
}

static bool write_certificate_request_tls13(TLS *tls) {
	return write_certificate_request_up_to_tls12(tls);
}

static bool write_server_hello_done_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((msg = tls_hs_shellodone_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: server: tls_hs_shellodone_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: server: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_SEND_SHELLODONE);

	return true;
}

static bool write_finished_tls13(TLS *tls) {
	struct tls_hs_msg *msg;

	/* before writing finished message, re-calculate hash (make
	 * queue empty). */
	tls_hs_update_hash(tls);

	/* send finished handshake protocol message. */
	if ((msg = tls_hs_finished_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: finale: tls_hs_finished_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: finale: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}

	enum tls_hs_sighash_hash_algo hash = tls_cipher_hashalgo(
		tls->pending->cipher_suite);
	tls_hs_hash_get_digest(hash, tls, tls->application_secret_context);

	tls_hs_change_state(tls, TLS_STATE_HS_SEND_FINISH);

	return true;
}

static bool read_client_hello(TLS *tls, enum tls_state state) {
	struct tls_hs_msg *msg;

	tls_hs_change_state(tls, state);
	if ((msg = tls_handshake_read(tls)) == NULL) {
		TLS_DPRINTF("hs: server: tls_handshake_read");
		return false;
	}

	if (! tls_hs_chello_parse(tls, msg)) {
		TLS_DPRINTF("hs: server: tls_hs_chello_parse");
		return false;
	}

	return true;
}

static bool read_certificate_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if (tls->certreq_used == true) {
		if ((msg = tls_handshake_read(tls)) == NULL) {
			return false;
		}

		if (! tls_hs_ccert_parse(tls, msg)) {
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_RECV_CCERT);
	}

	return true;
}

static bool read_certificate_tls13(TLS *tls) {
	return read_certificate_up_to_tls12(tls);
}

static bool read_client_key_exchange_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((msg = tls_handshake_read(tls)) == NULL) {
		return false;
	}

	if (! tls_hs_ckeyexc_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_CKEYEXC);

	return true;
}

static bool read_certificate_verify_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((tls->certreq_used == true) &&
	    (tls->ccert_null   == false)) {
		if ((msg = tls_handshake_read(tls)) == NULL) {
			return false;
		}

		if (! tls_hs_certvfy_parse(tls, msg)) {
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_RECV_CCERTVFY);
	}

	return true;
}

static bool read_certificate_verify_tls13(TLS *tls) {
	return read_certificate_verify_up_to_tls12(tls);
}

static bool read_finished_tls13(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((msg = tls_handshake_read(tls)) == NULL) {
		return false;
	}

	if (! tls_hs_finished_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_FINISH);

	return true;
}

static enum hs_phase do_phase_hello(TLS *tls) {
	TLS_DPRINTF("handshake: server: phase: hello start");

	if (read_client_hello(tls, TLS_STATE_HS_BEFORE_RECV_CHELLO) == false) {
		goto err;
	}

	if (establish_protocol_version(tls) == false) {
		goto err;
	}

	if (interpret_client_hello(tls) == false) {
		goto err;
	}

	/* init hash for finished (algorithm that used in finished are
	 * determined by receiving and parsing client hello.). */
	tls_hs_hash_init(tls);

	bool send_ccs = false;
	const uint8_t ses_len_max = 32;
	if (tls_hs_check_state(tls, TLS_STATE_HS_BEFORE_SEND_HRREQ) == true) {
		if (replace_transcript_with_msghash(tls) == false) {
			goto err;
		}

		if (write_server_hello(tls, TLS_STATE_HS_BEFORE_RECV_2NDCHELLO)
		    == false) {
			goto err;
		}

		/* compatibility mode when sending hello retry request. */
		if (tls->interim_params->seslen == ses_len_max &&
		    send_ccs == false) {
			if (tls_ccs_send(tls) == false) {
				goto err;
			}
			send_ccs = true;
		}

		tls->first_hello_params = tls->interim_params;
		if ((tls->interim_params = tls_hs_interim_params_init())
		    == NULL) {
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 20,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}

		if (read_client_hello(tls, TLS_STATE_HS_BEFORE_RECV_2NDCHELLO)
		    == false) {
			goto err;
		}

		if (interpret_2ndclient_hello(tls) == false) {
			goto err;
		}
	}

	if (write_server_hello(tls, TLS_STATE_HS_SEND_SHELLO) == false) {
		goto err;
	}

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		/*
		 * RFC8446 D.4.  Middlebox Compatibility Mode
		 *
		 *    -  The server sends a dummy change_cipher_spec record immediately
		 *       after its first handshake message.  This may either be after a
		 *       ServerHello or a HelloRetryRequest.
		 *
		 *    When put together, these changes make the TLS 1.3 handshake resemble
		 *    TLS 1.2 session resumption, which improves the chance of successfully
		 *    connecting through middleboxes.  This "compatibility mode" is
		 *    partially negotiated: the client can opt to provide a session ID or
		 *    not, and the server has to echo it.  Either side can send
		 *    change_cipher_spec at any time during the handshake, as they must be
		 *    ignored by the peer, but if the client sends a non-empty session ID,
		 *    the server MUST send the change_cipher_spec as described in this
		 *    appendix.
		 */
		if (tls->interim_params->seslen == ses_len_max &&
		    send_ccs == false) {
			if (tls_ccs_send(tls) == false) {
				goto err;
			}
			send_ccs = true;
		}

		/*
		 * TODO: call tls_key_derive_early_secret() early on
		 * when implementing PSK. it may be appropriate to call
		 * the function after receiving client hello.
		 */
		if (tls_key_derive_early_secret(tls) == false) {
			TLS_DPRINTF("hs: server: tls_key_derive_early_secret");
			goto err;
		}

		if (! derive_handshake_secrets(tls)) {
			TLS_DPRINTF("hs: server: derive_handshake_secrets");
			goto err;
		}

		/* set parameter of selected cipher suite. */
		if (tls_cipher_param_set(tls->pending->cipher_suite,
			&(tls->active_read.cipher)) == false) {
			TLS_DPRINTF("server: tls_cipher_param_set");
			goto err;
		}

		if (tls_key_make_traffic_key(tls,
			&(tls->active_read)) == false) {
			TLS_DPRINTF("server: tls_key_make_traffic_key");
			goto err;
		}

		if (tls_cipher_param_set(tls->pending->cipher_suite,
			&(tls->active_write.cipher)) == false) {
			TLS_DPRINTF("server: tls_cipher_param_set");
			goto err;
		}

		if (tls_key_make_traffic_key(tls,
			&(tls->active_write)) == false) {
			TLS_DPRINTF("server: tls_key_make_traffic_key");
			goto err;
		}

		/*
		 * RFC8446 4.3.1.  Encrypted Extensions
		 *
		 *    In all handshakes, the server MUST send the EncryptedExtensions
		 *    message immediately after the ServerHello message.  This is the first
		 *    message that is encrypted under keys derived from the
		 *    server_handshake_traffic_secret.
		 */
		if (write_encrypted_extensions_tls13(tls) == false) {
			goto err;
		}
		break;

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		/* if resession phase, next is final phase. */
		if (tls->resession == true) {
			/* it is not necessary to generate master secret in the
			 * resession. */
			if (! tls_key_make_key_block(tls)) {
				TLS_DPRINTF("hs: server: tls_key_keyblock_gen");
				goto err;
			}

			tls_hs_change_state(tls, TLS_STATE_HS_BEFORE_FINISH);

			return TLS_HS_PHASE_FINAL;
		}
		break;
	}

	TLS_DPRINTF("hs: server: phase: hello finish (ok)");
	return TLS_HS_PHASE_CERTIFICATE_SENDING;

err:
	TLS_DPRINTF("hs: server: phase: hello finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static bool do_phase_send_cert_up_to_tls12(TLS *tls) {
	if (write_certificate_up_to_tls12(tls) == false) {
		goto err;
	}

	if (write_server_key_exchange_up_to_tls12(tls) == false) {
		goto err;
	}

	if (write_certificate_request_up_to_tls12(tls) == false) {
		goto err;
	}

	if (write_server_hello_done_up_to_tls12(tls) == false) {
		goto err;
	}

	return true;

err:
	return false;
}

static bool do_phase_send_cert_tls13(TLS *tls UNUSED) {
	/*
	 * RFC8446 4.3.2.  Certificate Request
	 *
	 *    A server which is authenticating with a certificate MAY optionally
	 *    request a certificate from the client.  This message, if sent, MUST
	 *    follow EncryptedExtensions.
	 */
	if (write_certificate_request_tls13(tls) == false) {
		goto err;
	}

	if (write_certificate_tls13(tls) == false) {
		goto err;
	}

	/*
	 * RFC8446 4.4.3.  Certificate Verify
	 *
	 *                    When sent, this message MUST appear immediately after
	 *    the Certificate message and immediately prior to the Finished
	 *    message.
	 */
	if (write_certificate_verify_tls13(tls) == false) {
		goto err;
	}

	if (write_finished_tls13(tls) == false) {
		goto err;
	}

	/*
	 * RFC8446 4.4.4.  Finished
	 *
	 *    Any records following a Finished message MUST be encrypted under the
	 *    appropriate application traffic key as described in Section 7.2.  In
	 *    particular, this includes any alerts sent by the server in response
	 *    to client Certificate and CertificateVerify messages.
	 */
	if (tls_key_derive_master_secret(tls) == false) {
		TLS_DPRINTF("tls_key_derive_master_secret");
		goto err;
	}

	if (tls_key_derive_application_traffic_secret(tls,
						      "s ap traffic",
						       &(tls->active_write)) == false) {
		TLS_DPRINTF("tls_key_derive_application_traffic_secret");
		goto err;
	}

	if (tls_key_make_traffic_key(tls, &(tls->active_write)) == false) {
		TLS_DPRINTF("tls_key_make_traffic_key");
		goto err;
	}

	return true;

err:
	return false;
}

static enum hs_phase do_phase_send_cert(TLS *tls) {
	TLS_DPRINTF("hs: server: phase: send_cert start");

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if (do_phase_send_cert_up_to_tls12(tls) == false) {
			goto err;
		}
		break;

	case TLS_VER_TLS13:
		if (do_phase_send_cert_tls13(tls) == false) {
			goto err;
		}
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 21, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	TLS_DPRINTF("hs: server: phase: send_cert finish (ok)");
	return TLS_HS_PHASE_CERTIFICATE_RECVING;
err:
	TLS_DPRINTF("hs: server: phase: send_cert finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static bool do_phase_recv_cert_up_to_tls12(TLS *tls) {
	if (read_certificate_up_to_tls12(tls) == false) {
		goto err;
	}

	if (read_client_key_exchange_up_to_tls12(tls) == false) {
		goto err;
	}

	if (read_certificate_verify_up_to_tls12(tls) == false) {
		goto err;
	}

	if (make_key_block(tls) == false) {
		goto err;
	}

	/* before client finished reception. */
	tls_hs_change_state(tls,  TLS_STATE_HS_BEFORE_FINISH);

	return true;

err:
	return false;
}

static bool do_phase_recv_cert_tls13(TLS *tls UNUSED) {
	if (read_certificate_tls13(tls) == false) {
		goto err;
	}

	if (read_certificate_verify_tls13(tls) == false) {
		goto err;
	}

	/* before client finished reception. */
	tls_hs_change_state(tls,  TLS_STATE_HS_BEFORE_FINISH);

	return true;

err:
	return false;
}

static enum hs_phase do_phase_recv_cert(TLS *tls) {
	TLS_DPRINTF("hs: server: phase: recv_cert start");

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if (do_phase_recv_cert_up_to_tls12(tls) == false) {
			goto err;
		}
		break;

	case TLS_VER_TLS13:
		if (do_phase_recv_cert_tls13(tls) == false) {
			goto err;
		}
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_SERVER2 + 22, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	TLS_DPRINTF("hs: server: phase: recv_cert finish (ok)");
	return TLS_HS_PHASE_FINAL;

err:
	TLS_DPRINTF("hs: server: phase: recv_cert finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static enum hs_phase do_phase_final_up_to_tls12(TLS *tls) {
	if (tls->resession == true) {
		return tls_hs_finale_write_first(tls);
	} else {
		return tls_hs_finale_read_first(tls);
	}
}

static enum hs_phase do_phase_final_tls13(TLS *tls) {
	if (read_finished_tls13(tls) == false) {
		return TLS_HS_PHASE_FAILD;
	}

	if (tls_key_derive_application_traffic_secret(tls,
						      "c ap traffic",
						       &(tls->active_read)) == false) {
		TLS_DPRINTF("tls_key_derive_application_traffic_secret");
		return TLS_HS_PHASE_FAILD;
	}

	if (tls_key_make_traffic_key(tls, &(tls->active_read)) == false) {
		TLS_DPRINTF("tls_key_make_traffic_key");
		return TLS_HS_PHASE_FAILD;
	}

	return TLS_HS_PHASE_DONE;
}

static enum hs_phase do_phase_final(TLS *tls) {
	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		return do_phase_final_up_to_tls12(tls);

	case TLS_VER_TLS13:
		return do_phase_final_tls13(tls);

	default:
		break;
	}

	return TLS_HS_PHASE_FAILD;
}

static void cleanup(TLS *tls) {
	if (tls->chello_cipher_suites != NULL) {
		tls_cipher_list_free(tls->chello_cipher_suites);
		tls->chello_cipher_suites = NULL;
	}

	tls_hs_sighash_free(tls->sighash_list);
	tls->sighash_list = NULL;

	if (tls->sighash_list_cert != NULL) {
		tls_hs_sighash_free(tls->sighash_list_cert);
		tls->sighash_list_cert = NULL;
	}

	if (tls->ecdh != NULL) {
		tls_hs_ecdh_free(tls->ecdh);
		tls->ecdh = NULL;
	}
}

bool tls_hs_cs_server_handshake(TLS *tls) {
	enum hs_phase phase = TLS_HS_PHASE_HELLO;

	while (true) {
		switch (phase) {
		case TLS_HS_PHASE_HELLO:
			phase = do_phase_hello(tls);
			break;

		case TLS_HS_PHASE_CERTIFICATE_SENDING:
			phase = do_phase_send_cert(tls);
			break;

		case TLS_HS_PHASE_CERTIFICATE_RECVING:
			phase = do_phase_recv_cert(tls);
			break;

		case TLS_HS_PHASE_FINAL:
			phase = do_phase_final(tls);
			break;

		case TLS_HS_PHASE_DONE:
			goto fin;

		case TLS_HS_PHASE_FAILD:
			goto err;
		}
	}

fin:
	cleanup(tls);
	return true;

err:
	cleanup(tls);
	return false;
}
