/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_handshake.h"
#include "tls_ccs.h"

/**
 * write change cipher spec protocol to the peer.
 */
static bool write_ccs(TLS *tls);

/**
 * make finished data and send it to the peer.
 */
static bool write_finished(TLS *tls);

/**
 * read handshake data and parse it as considering it finished data.
 */
static bool read_finished(TLS *tls);

static bool write_ccs(TLS *tls) {
	if (! tls_ccs_send(tls)) {
		TLS_DPRINTF("hs: finale: tls_ccs_send");
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_CCS_SEND);

	return true;
}

static bool write_finished(TLS *tls) {
	struct tls_hs_msg *msg;

	/* before writing finished message, re-calculate hash (make
	 * queue empty). */
	tls_hs_update_hash(tls);

	/* send finished handshake protocol message. */
	if ((msg = tls_hs_finished_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: finale: tls_hs_finished_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: finale: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_SEND_FINISH);

	return true;
}

static bool read_finished(TLS *tls) {
	struct tls_hs_msg *msg;

	/* read finish handshake protocol message. and, if tls->state is
	 * right state, this tls_handshake_read process CCS
	 * internally. */
	if ((msg = tls_handshake_read(tls)) == NULL) {
		TLS_DPRINTF("hs: finale: tls_handshake_read");
		return false;
	}

	/* confrim that CCS was read by the tls_handshake_read. */
	if (! tls_hs_check_state(tls, TLS_STATE_CCS_RECV)) {
		TLS_DPRINTF("hs: finale: tls_hs_check_state: not recv ccs");
		return false;
	}

	if (! tls_hs_finished_parse(tls, msg)) {
		TLS_DPRINTF("hs: finale: tls_hs_finished_parse");
		return false;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_RECV_FINISH);

	return true;
}

enum hs_phase tls_hs_finale_write_first(TLS *tls) {
	TLS_DPRINTF("hs: finale(w): phase: final start");

	if (write_ccs(tls) == false) {
		goto err;
	}

	if (write_finished(tls) == false) {
		goto err;
	}

	if (read_finished(tls) == false) {
		goto err;
	}

	TLS_DPRINTF("hs: finale(w): phase: final finish (ok)");
	return TLS_HS_PHASE_DONE;

err:
	TLS_DPRINTF("hs: finale(w): phase: final finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

enum hs_phase tls_hs_finale_read_first(TLS *tls) {
	TLS_DPRINTF("hs: finale(r): phase: final start");

	if (read_finished(tls) == false) {
		goto err;
	}

	if (write_ccs(tls) == false) {
		goto err;
	}

	if (write_finished(tls) == false) {
		goto err;
	}

	TLS_DPRINTF("hs: finale(r): phase: final finish (ok)");
	return TLS_HS_PHASE_DONE;
err:
	TLS_DPRINTF("hs: finale(r): phase: final finish (ng)");
	return TLS_HS_PHASE_FAILD;
}
