/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "tls_record.h"
#include "tls_handshake.h"
#include "tls_ccs.h"
#include "tls_cert.h"
#include "extension/tls_sighash.h"
#include "tls_alert.h"
#include "tls_cipher.h"

/** @see tls_key.c */
extern bool tls_key_make_master_secret(TLS *tls);

/** @see tls_key.c */
extern bool tls_key_make_key_block(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_early_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_handshake_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_handshake_traffic_secrets(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_master_secret(TLS *tls);

/** @see tls_key.c */
bool tls_key_derive_application_traffic_secret(TLS *tls,
					      char *label,
					      struct tls_connection *connection);

/** @see tls_key.c */
bool tls_key_make_traffic_key(TLS *tls, struct tls_connection *connection);

/* @see finale.c */
extern enum hs_phase tls_hs_finale_write_first(TLS *tls);

/* @see finale.c */
extern enum hs_phase tls_hs_finale_read_first(TLS *tls);

/**
 * establish protocol version based on server hello message.
 */
static bool establish_protocol_version(TLS *tls);

/**
 * make master secret and keyblock.
 */
static bool make_key_block(TLS *tls);

/**
 * derive handshake secrets.
 */
static bool derive_handshake_secrets(TLS *tls);

/**
 * select key exchange method.
 */
static bool select_key_exchange_method(TLS *tls);

/**
 * search tls_hs_interim_params structure for an extension that have
 * specified type.
 */
static struct tls_extension *search_extension(
	struct tls_hs_interim_params *params, enum tls_extension_type type);

/**
 * compare hello retry request and server hello messages to check if
 * there is no difference except a few extensions.
 */
static bool compare_server_hellos(TLS *tls);

/**
 * interpret server hello values according to protocol version.
 */
static bool interpret_server_hello(TLS *tls);

/**
 * interpret second server hello(received after hello retry request) values.
 */
static bool interpret_2ndserver_hello(TLS *tls);

/**
 * replace client hello with message hash about hash of handshake message.
 */
static bool replace_transcript_with_msghash(TLS *tls);

/**
 * make client hello data and send it to the server.
 */
static bool write_client_hello(TLS *tls, enum tls_state before,
			       enum tls_state after);

/**
 * make (client) certificate data and send it to the server.
 */
static bool write_certificate_up_to_tls12(TLS *tls);
static bool write_certificate_tls13(TLS *tls);

/**
 * make certificate verify data and send it to the server.
 */
static bool write_certificate_verify_up_to_tls12(TLS *tls);
static bool write_certificate_verify_tls13(TLS *tls);

/**
 * make client key exchange data and send it to the server.
 */
static bool write_client_key_exchange_up_to_tls12(TLS *tls);

/**
 * make finished data and send it to the server.
 */
static bool write_finished_tls13(TLS *tls);

/**
 * read handshake data and parse it as considering it hello request
 * data.
 */
static bool read_hello_request(TLS *tls);

/**
 * read handshake data and parse it as considering it server hello data.
 */
static bool read_server_hello(TLS *tls, enum tls_state state);

/**
 * read handshake data and parse it as considering it encrypted
 * extensions data.
 */
static bool read_encrypted_extensions_tls13(TLS *tls);

/**
 * read handshake data and parse it as considering it (server)
 * certificate data.
 */
static bool read_certificate_up_to_tls12(TLS *tls, struct tls_hs_msg *msg);
static bool read_certificate_tls13(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake data and parse it as considering it server key
 * exchange data.
 */
static bool read_server_key_exchange_up_to_tls12(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake data and parse it as considering it certificate
 * request data.
 */
static bool read_certificate_request_up_to_tls12(TLS *tls, struct tls_hs_msg *msg);
static bool read_certificate_request_tls13(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake data and parse it as considering it certificate
 * verify data.
 */
static bool read_certificate_verify_tls13(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake data and parse it as considering it server hello done
 * data.
 */
static bool read_server_hello_done_up_to_tls12(TLS *tls, struct tls_hs_msg *msg);

/**
 * read handshake data and parse it as considering it (server) finished
 * data.
 */
static bool read_finished_tls13(TLS *tls, struct tls_hs_msg *msg);

/**
 * do TLS_HS_PHASE_HELLO phase.
 *
 * if resumption is determined in the negotiation between the client
 * hello and server hello, make keyblock and go to final phase.
 */
static enum hs_phase do_phase_hello(TLS *tls);

/**
 * do TLS_HS_PHASE_CERTIFICATE_RECVING phase.
 */
static bool do_phase_recv_cert_up_to_tls12(TLS *tls);
static bool do_phase_recv_cert_tls13(TLS *tls);
static enum hs_phase do_phase_recv_cert(TLS *tls);

/**
 * do TLS_HS_PHASE_CERTIFICATE_SENDING phase.
 */
static bool do_phase_send_cert_up_to_tls12(TLS *tls);
static bool do_phase_send_cert_tls13(TLS *tls);
static enum hs_phase do_phase_send_cert(TLS *tls);

/**
 * do TLS_HS_PHASE_FINAL phase.
 */
static enum hs_phase do_phase_final_up_to_tls12(TLS *tls);
static enum hs_phase do_phase_final_tls13(TLS *tls);
static enum hs_phase do_phase_final(TLS *tls);

/**
 * clean up handshake specific data.
 */
static void cleanup(TLS *tls);

static bool establish_protocol_version(TLS *tls) {
	struct tls_extension *ext, *ext_sv = NULL;
	uint16_t version;
	uint16_t client_version;
	uint16_t record_version;

	TAILQ_FOREACH(ext, &(tls->interim_params->head), link) {
		if (ext->type == TLS_EXT_SUPPORTED_VERSIONS) {
			if (ext_sv == NULL) {
				ext_sv = ext;
			} else {
				/*
				 * supported versions extensions come, so handle this
				 * as an error. see RFC8446 p.37.
				 */
				OK_set_error(ERR_ST_TLS_UNSUPPORTED_EXTENSION,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_CLIENT2 + 0,
					     NULL);
				TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNSUPPORTED_EXTENSION);
				return false;
			}
		}
	}

	/* version negotiation by supported versions extension */
	if (ext_sv != NULL) {
		/*
		 * RFC8446 4.2.1.  Supported Versions
		 *
		 * A server which negotiates a version of TLS prior to TLS 1.3 MUST set
		 * ServerHello.version and MUST NOT send the "supported_versions"
		 * extension.  A server which negotiates TLS 1.3 MUST respond by sending
		 * a "supported_versions" extension containing the selected version
		 * value (0x0304).  It MUST set the ServerHello.legacy_version field to
		 * 0x0303 (TLS 1.2).  Clients MUST check for this extension prior to
		 * processing the rest of the ServerHello (although they will have to
		 * parse the ServerHello in order to read the extension).  If this
		 * extension is present, clients MUST ignore the
		 * ServerHello.legacy_version value and MUST use only the
		 * "supported_versions" extension to determine the selected version.  If
		 * the "supported_versions" extension in the ServerHello contains a
		 * version not offered by the client or contains a version prior to
		 * TLS 1.3, the client MUST abort the handshake with an
		 * "illegal_parameter" alert.
		 */

		struct tls_hs_msg msg = {
			.type = TLS_HANDSHAKE_SERVER_HELLO,
			.len = ext_sv->len,
			.max = ext_sv->len,
			.msg = ext_sv->opaque
		};

		if (tls_hs_supported_versions_read(tls, &msg, 0) < 0) {
			goto err;
		}

		version = tls->peer_supported_versions.list[0];

		/*
		 * check whether the version is included in supported
		 * versions extension client sent.
		 */
		if (! tls_util_check_version_in_supported_version(
				&(tls->supported_versions), version)) {
			OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 1,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
			return false;
		}

		/*
		 * check whether version included in supported version extension is
		 * equal to TLS 1.3.
		 */
		uint16_t msg_version;
		switch (version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
		case TLS_VER_TLS12:
			OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 2,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
			return false;

		case TLS_VER_TLS13:
			msg_version = tls_util_convert_protover_to_ver(
				&(tls->interim_params->version));
			TLS_DPRINTF("client: msg_version: %04x", msg_version);
			if (msg_version != TLS_VER_TLS12) {
				OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_CLIENT2 + 3, NULL);
				TLS_ALERT_FATAL(tls,
					TLS_ALERT_DESC_ILLEGAL_PARAMETER);
				return false;
			}
			/*
			 * RFC8446 5.1.  Record Layer
			 *
			 *                                     In order to maximize backward
			 *    compatibility, a record containing an initial ClientHello SHOULD have
			 *    version 0x0301 (reflecting TLS 1.0) and a record containing a second
			 *    ClientHello or a ServerHello MUST have version 0x0303 (reflecting
			 *    TLS 1.2).
			 */
			record_version = TLS_VER_TLS12;
			goto found_supported_version;

		default:
			/* Unknown version */
			goto err;
		}
	} else {
		version = tls_util_convert_protover_to_ver(
			&(tls->interim_params->version));

		if (version < TLS_VER_SSL30) {
			goto err;
		}

		/*
		 * check whether server responded greater version than
		 * client version that client sent.
		 */
		client_version = tls_util_convert_protover_to_ver(
			&(tls->client_version));
		if (version > client_version) {
			goto err;
		}

		/*
		 * RFC8446 D.1.  Negotiating with an Older Server
		 *
		 *    If the version chosen by the server is not supported by the client
		 *    (or is not acceptable), the client MUST abort the handshake with a
		 *    "protocol_version" alert.
		 */
		switch(version) {
		case TLS_VER_SSL30:
		case TLS_VER_TLS10:
		case TLS_VER_TLS11:
			/* TODO: implementation */
			goto err;

		case TLS_VER_TLS12:
			break;

		default:
			/* Unknown version */
			goto err;
		}

		record_version = version;
	}

found_supported_version:
	tls_util_convert_ver_to_protover(version, &(tls->negotiated_version));
	TLS_DPRINTF("client: negotiated version: %04x", version);

	/*
	 * set appropriate version used for tls record according to
	 * specifications. */
	tls_util_convert_ver_to_protover(record_version, &(tls->record_version));
	TLS_DPRINTF("client: record_version: %04x", record_version);

	return true;

err:
	tls->errnum = TLS_ERR_PROTOCOL_VERSION;
	OK_set_error(ERR_ST_TLS_PROTOCOL_VERSION,
		     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 4, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_PROTOCOL_VERSION);
	return false;
}

static bool make_key_block(TLS *tls) {
	/* generate master secret. */
	if (! tls_key_make_master_secret(tls)) {
		TLS_DPRINTF("hs: client: tls_key_master_secret_gen");
		return false;
	}

	/* generate key block. */
	if (! tls_key_make_key_block(tls)) {
		TLS_DPRINTF("hs: client: tls_key_keyblock_gen");
		return false;
	}

	return true;
}

static bool derive_handshake_secrets(TLS *tls) {
	if (tls_key_derive_handshake_secret(tls) == false) {
		TLS_DPRINTF("hs: client: tls_key_derive_handshake_secret");
		return false;
	}

	tls_hs_update_hash(tls);

	if (tls_key_derive_handshake_traffic_secrets(tls) == false) {
		TLS_DPRINTF("hs: client: tls_key_derive_handshake_traffic_secrets");
		return false;
	}

	return true;
}

static bool select_key_exchange_method(TLS *tls) {
	/*
	 * RFC8446 2.2.  Resumption and Pre-Shared Key (PSK)
	 *
	 *    As the server is authenticating via a PSK, it does not send a
	 *    Certificate or a CertificateVerify message.  When a client offers
	 *    resumption via a PSK, it SHOULD also supply a "key_share" extension
	 *    to the server to allow the server to decline resumption and fall back
	 *    to a full handshake, if needed.  The server responds with a
	 *    "pre_shared_key" extension to negotiate the use of PSK key
	 *    establishment and can (as shown here) respond with a "key_share"
	 *    extension to do (EC)DHE key establishment, thus providing forward
	 *    secrecy.
	 */

	/*
	 * TODO: check if resumption is available at first, then fall back to
	 * full handshake. currently, resumption is not implemented, so skip
	 * this check.
	*/

	struct tls_hs_interim_params *params = tls->interim_params;
	if (params->sent_ext_flags[TLS_EXT_SUPPORTED_GROUPS] == false) {
		TLS_DPRINTF("server: missing supported_groups extension");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 5, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	if (params->sent_ext_flags[TLS_EXT_KEY_SHARE] == false) {
		TLS_DPRINTF("server: missing key_share extension");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 6, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	if (params->sent_ext_flags[TLS_EXT_SIGNATURE_ALGO] == false) {
		TLS_DPRINTF("server: missing signature_algorithms extension");
		OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 7, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);
		return false;
	}

	switch (tls->ecdh->namedcurve) {
	case TLS_NAMED_GROUP_SECP256R1:
	case TLS_NAMED_GROUP_X25519:
	case TLS_NAMED_GROUP_X448:
		tls->keymethod = TLS_KXC_ECDHE;
		return true;

	case TLS_NAMED_GROUP_SECP384R1:
	case TLS_NAMED_GROUP_SECP521R1:
		/* Not implemented */
		break;

	default:
		break;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_AFTER_RECV_HRREQ) == true) {
		/*
		 * RFC8446 4.1.4.  Hello Retry Request
		 *
		 *                           Clients MUST abort the handshake with an
		 *    "illegal_parameter" alert if the HelloRetryRequest would not result
		 *    in any change in the ClientHello.
		 */
		TLS_DPRINTF("client: hrr makes no change in selection of group ");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 8, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	TLS_DPRINTF("client: no valid key exchange method is found");
	OK_set_error(ERR_ST_TLS_HANDSHAKE_FAILURE,
		     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 9, NULL);
	TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_HANDSHAKE_FAILURE);

	return false;
}

static struct tls_extension *search_extension(
	struct tls_hs_interim_params *params, enum tls_extension_type type) {

	struct tls_extension *ext;
	TAILQ_FOREACH(ext, &(params->head), link) {
		if (ext->type == type) {
			return ext;
		}
	}

	return NULL;
}

static bool compare_server_hellos(TLS *tls) {
	struct tls_hs_interim_params *first_params = tls->first_hello_params;
	struct tls_hs_interim_params *second_params = tls->interim_params;

	if (memcmp(&(first_params->version), &(second_params->version),
		   sizeof(struct tls_protocol_version)) != 0) {
		TLS_DPRINTF("client: protocol_version is different from hrr");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 10, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	/*
	 * RFC8446 4.1.4.  Hello Retry Request
	 *
	 *                                                        Upon receiving
	 *    the ServerHello, clients MUST check that the cipher suite supplied in
	 *    the ServerHello is the same as that in the HelloRetryRequest and
	 *    otherwise abort the handshake with an "illegal_parameter" alert.
	 */
	if (first_params->cipher_suite != second_params->cipher_suite) {
		TLS_DPRINTF("client: cipher suite is different from hrr");
		OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 11, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_ILLEGAL_PARAMETER);
		return false;
	}

	/*
	 * RFC8446 4.1.4.  Hello Retry Request
	 *
	 *    The value of selected_version in the HelloRetryRequest
	 *    "supported_versions" extension MUST be retained in the ServerHello,
	 *    and a client MUST abort the handshake with an "illegal_parameter"
	 *    alert if the value changes.
	 */
	struct tls_extension *first_ext;
	struct tls_extension *second_ext;
	struct tls_extension *temp_ext;
	TAILQ_FOREACH_SAFE(first_ext, &(first_params->head), link, temp_ext) {
		switch (first_ext->type) {
		case TLS_EXT_SUPPORTED_VERSIONS:
			if ((second_ext = search_extension(second_params,
							   first_ext->type)) == NULL) {
				TLS_DPRINTF("client: missing supported_versions"
					    " extension");
				OK_set_error(ERR_ST_TLS_MISSING_EXTENSION,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_CLIENT2 + 12, NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_MISSING_EXTENSION);
				return false;
			}

			if (first_ext->len != second_ext->len) {
				TLS_DPRINTF("client: supported_versions contains"
					    " different values from hrr");
				OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_CLIENT2 + 13, NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_ILLEGAL_PARAMETER);
				return false;
			}

			if (memcmp(first_ext->opaque, second_ext->opaque,
				   first_ext->len) != 0) {
				TLS_DPRINTF("client: supported_versions contains"
					    " different values from hrr");
				OK_set_error(ERR_ST_TLS_ILLEGAL_PARAMETER,
					     ERR_LC_TLS5,
					     ERR_PT_TLS_HS_CS_CLIENT2 + 14, NULL);
				TLS_ALERT_FATAL(tls,
						TLS_ALERT_DESC_ILLEGAL_PARAMETER);
				return false;
			}

			second_params->recv_ext_flags[first_ext->type] = true;
			TAILQ_REMOVE(&(second_params->head), second_ext, link);
			tls_extension_free(second_ext);
			break;

		default:
			continue;
		}
	}

	return true;
}

static bool interpret_server_hello(TLS *tls) {
	if (! tls_hs_shello_interpret(tls)) {
		return false;
	}

	uint16_t version = tls_util_convert_protover_to_ver(
		&(tls->negotiated_version));
	switch (version) {
	case TLS_VER_TLS13:
		if (select_key_exchange_method(tls) == false) {
			return false;
		}
		break;

	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
	default:
		break;
	}

	if (tls_hs_check_state(tls, TLS_STATE_HS_AFTER_RECV_HRREQ) == true) {
		return true;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_AFTER_RECV_SHELLO);

	return true;
}

static bool interpret_2ndserver_hello(TLS *tls) {
	if (compare_server_hellos(tls) == false) {
		return false;
	}

	if (tls_hs_shello_interpret(tls) == false) {
		return false;
	}

	struct tls_extension *ext;
	while (!TAILQ_EMPTY(&(tls->interim_params->head))) {
		ext = TAILQ_FIRST(&(tls->interim_params->head));
		TAILQ_REMOVE(&(tls->interim_params->head), ext, link);
		tls_extension_free(ext);
	}

	tls_hs_change_state(tls, TLS_STATE_HS_AFTER_RECV_2NDSHELLO);

	return true;
}

static bool replace_transcript_with_msghash(TLS *tls) {
	struct tls_hs_msg *msg_hrr;
	msg_hrr = TAILQ_LAST(tls->queue_handshake, tls_handshake_queue);
	TAILQ_REMOVE(tls->queue_handshake, msg_hrr, link);

	tls_hs_update_hash(tls);

	struct tls_hs_msg *msg;
	if ((msg = tls_hs_msghash_make(tls)) == NULL) {
		return false;
	}

	/*
	 * init handshake message again and enqueue message hash and hello
	 * retry request.
	 */
	tls_hs_hash_init(tls);
	TAILQ_INSERT_TAIL(tls->queue_handshake, msg, link);
	TAILQ_INSERT_TAIL(tls->queue_handshake, msg_hrr, link);

	return true;
}

static bool write_client_hello(TLS *tls, enum tls_state before,
			       enum tls_state after) {
	struct tls_hs_msg *msg;

	tls_hs_change_state(tls, before);
	if ((msg = tls_hs_chello_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: client: tls_hs_chello_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: client: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, after);

	return true;
}

static bool write_certificate_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	/* send client certificate handshake protocol message. client
	 * certificate is sent when server send certificate request
	 * handshake protocol message. */
	if (tls->certreq_used == true) {
		if ((msg = tls_hs_ccert_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: client: tls_hs_ccert_compose");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: client: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_CCERT);
	}

	return true;
}

static bool write_certificate_tls13(TLS *tls) {
	return write_certificate_up_to_tls12(tls);
}

static bool write_certificate_verify_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	/* send certificate verify handshake protocol message.
	 * certificate verify is sent when client sends client
	 * certificate and certificate that sent hasthe signature
	 * aibility.  */
	if ((tls->certreq_used == true) &&
	    (tls->ccert_null   == false)) {
		/* before writing certificate verify handshake protocol
		 * messaage, re-calculate hash (make queue empty). */
		tls_hs_update_hash(tls);

		if ((msg = tls_hs_certvfy_compose(tls)) == NULL) {
			TLS_DPRINTF("hs: client: tls_hs_certvfy_compose");
			return false;
		}

		if (! tls_handshake_write(tls, msg)) {
			TLS_DPRINTF("hs: client: tls_handshake_write");
			tls_hs_msg_free(msg);
			return false;
		}
		tls_hs_change_state(tls, TLS_STATE_HS_SEND_CCERTVFY);
	}

	return true;
}

static bool write_certificate_verify_tls13(TLS *tls) {
	/*
	 * RFC8446 4.4.3.  Certificate Verify
	 *
	 *                        Clients MUST send this message whenever
	 *    authenticating via a certificate (i.e., when the Certificate message
	 *    is non-empty).
	 */
	return write_certificate_verify_up_to_tls12(tls);
}

static bool write_client_key_exchange_up_to_tls12(TLS *tls) {
	struct tls_hs_msg *msg;

	/* send client key exchange handshake protocol message.  client
	 * key exchange is always sent by client.  */
	if ((msg = tls_hs_ckeyexc_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: client: tls_hs_ckeyexc_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: client: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_SEND_CKEYEXC);

	return true;
}

static bool write_finished_tls13(TLS *tls) {
	/*
	 * RFC8446 4.4.2.  Certificate
	 *
	 *                             A Finished message MUST be sent regardless
	 *    of whether the Certificate message is empty.
	 */
	struct tls_hs_msg *msg;

	/* before writing finished message, re-calculate hash (make
	 * queue empty). */
	tls_hs_update_hash(tls);

	/* send finished handshake protocol message. */
	if ((msg = tls_hs_finished_compose(tls)) == NULL) {
		TLS_DPRINTF("hs: finale: tls_hs_finished_compose");
		return false;
	}

	if (! tls_handshake_write(tls, msg)) {
		TLS_DPRINTF("hs: finale: tls_handshake_write");
		tls_hs_msg_free(msg);
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_SEND_FINISH);

	return true;
}

static bool read_hello_request(TLS *tls) {
	struct tls_hs_msg *msg;

	if (tls_hs_check_state(tls, TLS_STATE_HS_REINIT) == true) {
		if ((msg = tls_handshake_read(tls)) == NULL) {
			TLS_DPRINTF("hs: client: tls_handshake_read");
			OK_set_error(ERR_ST_TLS_TLS_RECORD_READ,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_CS_CLIENT + 0, NULL);
			return false;
		}

		if (! tls_hs_helloreq_parse(tls, msg)) {
			TLS_DPRINTF("hs: client: tls_hs_helloreq_parse");
			tls_hs_msg_free(msg);
			return false;
		}

		/* Hello request do not be added to the queue (because
		 * Hello request do not be used in the hash
		 * calculation.) So, free allocated memory here
		 * explicitly. */
		tls_hs_msg_free(msg);
		tls_hs_change_state(tls, TLS_STATE_HS_RECV_HELLOREQ);
	}

	return true;
}

static bool read_server_hello(TLS *tls, enum tls_state state) {
	struct tls_hs_msg *msg;

	tls_hs_change_state(tls, state);
	if ((msg = tls_handshake_read(tls)) == NULL) {
		TLS_DPRINTF("hs: client: tls_handshake_read");
		return false;
	}

	if (! tls_hs_shello_parse(tls, msg)) {
		TLS_DPRINTF("hs: client: tls_hs_shello_parse");
		return false;
	}

	return true;
}

static bool read_encrypted_extensions_tls13(TLS *tls) {
	struct tls_hs_msg *msg;

	if ((msg = tls_handshake_read(tls)) == NULL) {
		TLS_DPRINTF("hs: client: tls_handshake_read");
		return false;
	}

	if (! tls_hs_encext_parse(tls, msg)) {
		TLS_DPRINTF("hs: client: tls_hs_encext_parse");
		return false;
	}

	if (! tls_hs_encext_interpret(tls)) {
		TLS_DPRINTF("hs: client: tls_hs_encext_interpret");
		return false;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_RECV_ENCEXT);

	return true;
}

static bool read_certificate_up_to_tls12(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_AFTER_RECV_SHELLO)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 15,
			     NULL);
		return false;
	}

	if (! tls_hs_scert_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_SCERT);

	return true;
}

static bool read_certificate_tls13(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_RECV_ENCEXT) &&
	    ! tls_hs_check_state(tls, TLS_STATE_HS_RECV_CERTREQ)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 16,
			     NULL);
		return false;
	}

	if (! tls_hs_scert_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_SCERT);

	return true;
}

static bool read_server_key_exchange_up_to_tls12(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_AFTER_RECV_SHELLO) &&
	    ! tls_hs_check_state(tls, TLS_STATE_HS_RECV_SCERT)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 17,
			     NULL);
		return false;
	}

	if (! tls_hs_skeyexc_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_SKEYEXC);

	return true;
}

static bool read_certificate_request_up_to_tls12(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_RECV_SCERT) &&
	    ! tls_hs_check_state(tls, TLS_STATE_HS_RECV_SKEYEXC)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 18,
			     NULL);
		return false;
	}

	if (! tls_hs_certreq_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_CERTREQ);

	return true;
}

static bool read_certificate_request_tls13(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_RECV_ENCEXT)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 19,
			     NULL);
		return false;
	}

	if (! tls_hs_certreq_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_CERTREQ);

	return true;
}

static bool read_certificate_verify_tls13(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_RECV_SCERT)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 20, NULL);
		return false;
	}

	if (! tls_hs_certvfy_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_SCERTVFY);

	return true;
}

static bool read_server_hello_done_up_to_tls12(TLS *tls, struct tls_hs_msg *msg) {
	/* TODO: only server hello done is possible? (DH_anon?) */
	if (! tls_hs_shellodone_parse(tls, msg)) {
		return false;
	}
	tls_hs_change_state(tls, TLS_STATE_HS_RECV_SHELLODONE);

	return true;
}

static bool read_finished_tls13(TLS *tls, struct tls_hs_msg *msg) {
	if (! tls_hs_check_state(tls, TLS_STATE_HS_RECV_ENCEXT) &&
	    ! tls_hs_check_state(tls, TLS_STATE_HS_RECV_SCERTVFY) &&
	    ! tls_hs_check_state(tls, TLS_STATE_HS_RECV_CERTREQ)) {
		OK_set_error(ERR_ST_TLS_INVALID_STATUS,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 21,
			     NULL);
		return false;
	}

	if (! tls_hs_finished_parse(tls, msg)) {
		return false;
	}

	tls_hs_update_hash(tls);

	enum tls_hs_sighash_hash_algo hash = tls_cipher_hashalgo(
		tls->pending->cipher_suite);
	tls_hs_hash_get_digest(hash, tls, tls->application_secret_context);

	tls_hs_change_state(tls, TLS_STATE_HS_RECV_FINISH);

	return true;
}

static enum hs_phase do_phase_hello(TLS *tls) {
	TLS_DPRINTF("hs: client: phase: hello start");

	if (read_hello_request(tls) == false) {
		goto err;
	}

	if (write_client_hello(tls, TLS_STATE_HS_BEFORE_SEND_CHELLO,
			       TLS_STATE_HS_AFTER_SEND_CHELLO) == false) {
		goto err;
	}

	if (read_server_hello(tls, TLS_STATE_HS_BEFORE_RECV_SHELLO) == false) {
		goto err;
	}

	if (establish_protocol_version(tls) == false) {
		goto err;
	}

	if (interpret_server_hello(tls) == false) {
		goto err;
	}

	/* init hash for finished. */
	tls_hs_hash_init(tls);

	/* send second client hello as received hello retry request. */
	if (tls_hs_check_state(tls, TLS_STATE_HS_AFTER_RECV_HRREQ) == true) {
		tls->first_hello_params = tls->interim_params;
		if ((tls->interim_params = tls_hs_interim_params_init())
		    == NULL) {
			OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
				     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 22,
				     NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
			goto err;
		}

		if (replace_transcript_with_msghash(tls) == false) {
			goto err;
		}

		if (write_client_hello(tls, TLS_STATE_HS_BEFORE_SEND_2NDCHELLO,
				       TLS_STATE_HS_AFTER_SEND_2NDCHELLO)
		    == false) {
			goto err;
		}

		if (read_server_hello(tls, TLS_STATE_HS_BEFORE_RECV_2NDSHELLO)
		    == false) {
			goto err;
		}

		if (interpret_2ndserver_hello(tls) == false) {
			goto err;
		}
	}

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));
	if (version == TLS_VER_TLS13) {
		/*
		 * TODO: call tls_key_derive_early_secret() early on when
		 * implementing PSK. it may be appropriate to call the function
		 * after sending client hello.
		 */
		if (tls_key_derive_early_secret(tls) == false) {
			TLS_DPRINTF("hs: client: tls_key_derive_early_secret");
			goto err;
		}

		if (derive_handshake_secrets(tls) == false) {
			goto err;
		}

		if (tls_cipher_param_set(tls->pending->cipher_suite,
			&(tls->active_read.cipher)) == false) {
			TLS_DPRINTF("client: tls_cipher_param_set");
			goto err;
		}

		if (tls_key_make_traffic_key(tls,
			&(tls->active_read)) == false) {
			TLS_DPRINTF("client: tls_key_make_traffic_key");
			goto err;
		}

		if (tls_cipher_param_set(tls->pending->cipher_suite,
			&(tls->active_write.cipher)) == false) {
			TLS_DPRINTF("client: tls_cipher_param_set");
			goto err;
		}

		if (tls_key_make_traffic_key(tls,
			&(tls->active_write)) == false) {
			TLS_DPRINTF("client: tls_key_make_traffic_key");
			goto err;
		}

		if (read_encrypted_extensions_tls13(tls) == false) {
			goto err;
		}
	} else {
		/* if ression phase, next is final phase. */
		if (tls->resession == true) {
			/* it is not necessary to generate master secret in the
			 * resession. */
			if (! tls_key_make_key_block(tls)) {
				TLS_DPRINTF("hs: client: tls_key_keyblock_gen");
				goto err;
			}

			tls_hs_change_state(tls, TLS_STATE_HS_BEFORE_FINISH);

			return TLS_HS_PHASE_FINAL;
		}
	}

	TLS_DPRINTF("hs: client: phase: hello finish (ok)");
	return TLS_HS_PHASE_CERTIFICATE_RECVING;

err:
	TLS_DPRINTF("hs: client: phase: hello finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static bool do_phase_recv_cert_up_to_tls12(TLS *tls) {
	while (true) {
		struct tls_hs_msg *msg;
		if ((msg = tls_handshake_read(tls)) == NULL) {
			goto err;
		}

		/* if received the server hello, next handshake message
		 * is one of server certificate or server key
		 * exchange. */
		switch (msg->type) {
		case TLS_HANDSHAKE_CERTIFICATE:
			if (read_certificate_up_to_tls12(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_SERVER_KEY_EXCHANGE:
			if (read_server_key_exchange_up_to_tls12(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_CERTIFICATE_REQUEST:
			if (read_certificate_request_up_to_tls12(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_SERVER_HELLO_DONE:
			if (read_server_hello_done_up_to_tls12(tls, msg) == false) {
				goto err;
			}
			goto fin;

		default:
			TLS_DPRINTF("hs: client: unexpected hs msg");
			OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
				     ERR_LC_TLS2,
				     ERR_PT_TLS_HS_CS_CLIENT + 1, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			goto err;
		}
	}

fin:
	return true;

err:
	return false;
}

static bool do_phase_recv_cert_tls13(TLS *tls) {
	while (true) {
		struct tls_hs_msg *msg;
		if ((msg = tls_handshake_read(tls)) == NULL) {
			goto err;
		}

		/* if received the server hello, next handshake message
		 * is one of server certificate or finished. */
		switch (msg->type) {
		case TLS_HANDSHAKE_CERTIFICATE_REQUEST:
			if (read_certificate_request_tls13(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_CERTIFICATE:
			if (read_certificate_tls13(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_CERTIFICATE_VERIFY:
			if (read_certificate_verify_tls13(tls, msg) == false) {
				goto err;
			}
			break;

		case TLS_HANDSHAKE_FINISHED:
			if (read_finished_tls13(tls, msg) == false) {
				goto err;
			}

			/*
			 * RFC8446 4.4.4.  Finished
			 *
			 *    Any records following a Finished message MUST be encrypted under the
			 *    appropriate application traffic key as described in Section 7.2.  In
			 *    particular, this includes any alerts sent by the server in response
			 *    to client Certificate and CertificateVerify messages.
			 */
			if (tls_key_derive_master_secret(tls) == false) {
				TLS_DPRINTF("tls_key_derive_master_secret");
				goto err;
			}

			if (tls_key_derive_application_traffic_secret(
				tls, "s ap traffic", &(tls->active_read))
			    == false) {
				TLS_DPRINTF("tls_key_derive_application_traffic_secret");
				goto err;
			}

			if (tls_key_make_traffic_key(tls, &(tls->active_read))
			    == false) {
				TLS_DPRINTF("tls_key_make_traffic_key");
				goto err;
			}
			goto fin;

		default:
			TLS_DPRINTF("hs: client: unexpected hs msg");
			OK_set_error(ERR_ST_TLS_UNEXPECTED_MSG,
				     ERR_LC_TLS5,
				     ERR_PT_TLS_HS_CS_CLIENT2 + 23, NULL);
			TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_UNEXPECTED_MESSAGE);
			goto err;
		}
	}

fin:
	return true;

err:
	return false;
}

static enum hs_phase do_phase_recv_cert(TLS *tls) {
	TLS_DPRINTF("hs: client: phase: recv_cert start");

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if (do_phase_recv_cert_up_to_tls12(tls) == false) {
			goto err;
		}
		break;

	case TLS_VER_TLS13:
		if (do_phase_recv_cert_tls13(tls) == false) {
			goto err;
		}
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 24, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	TLS_DPRINTF("hs: client: phase: recv_cert finish (ok)");
	return TLS_HS_PHASE_CERTIFICATE_SENDING;

err:
	TLS_DPRINTF("hs: client: phase: recv_cert finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static bool do_phase_send_cert_up_to_tls12(TLS *tls) {
	if (write_certificate_up_to_tls12(tls) == false) {
		goto err;
	}

	if (write_client_key_exchange_up_to_tls12(tls) == false) {
		goto err;
	}

	if (write_certificate_verify_up_to_tls12(tls) == false) {
		goto err;
	}

	if (make_key_block(tls) == false) {
		goto err;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_BEFORE_FINISH);

	return true;

err:
	return false;
}

static bool do_phase_send_cert_tls13(TLS *tls) {
	if (write_certificate_tls13(tls) == false) {
		goto err;
	}

	/*
	 * RFC8446 4.4.3.  Certificate Verify
	 *
	 *                    When sent, this message MUST appear immediately after
	 *    the Certificate message and immediately prior to the Finished
	 *    message.
	 */
	if (write_certificate_verify_tls13(tls) == false) {
		goto err;
	}

	tls_hs_change_state(tls, TLS_STATE_HS_BEFORE_FINISH);

	return true;

err:
	return false;
}

static enum hs_phase do_phase_send_cert(TLS *tls) {
	TLS_DPRINTF("hs: client: phase: send_cert start");

	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		if (do_phase_send_cert_up_to_tls12(tls) == false) {
			goto err;
		}
		break;

	case TLS_VER_TLS13:
		if (do_phase_send_cert_tls13(tls) == false) {
			goto err;
		}
		break;

	default:
		OK_set_error(ERR_ST_TLS_INTERNAL_ERROR,
			     ERR_LC_TLS5, ERR_PT_TLS_HS_CS_CLIENT2 + 25, NULL);
		TLS_ALERT_FATAL(tls, TLS_ALERT_DESC_INTERNAL_ERROR);
		goto err;
	}

	TLS_DPRINTF("hs: client: phase: send_cert finish (ok)");
	return TLS_HS_PHASE_FINAL;

err:
	TLS_DPRINTF("hs: client: phase: send_cert finish (ng)");
	return TLS_HS_PHASE_FAILD;
}

static enum hs_phase do_phase_final_up_to_tls12(TLS *tls) {
	if (tls->resession == true) {
		return tls_hs_finale_read_first(tls);
	} else {
		return tls_hs_finale_write_first(tls);
	}
}

static enum hs_phase do_phase_final_tls13(TLS *tls) {
	if (write_finished_tls13(tls) == false) {
		return TLS_HS_PHASE_FAILD;
	}

	if (tls_key_derive_application_traffic_secret(tls,
						      "c ap traffic",
						       &(tls->active_write)) == false) {
		TLS_DPRINTF("tls_key_derive_application_traffic_secret");
		return TLS_HS_PHASE_FAILD;
	}

	if (tls_key_make_traffic_key(tls, &(tls->active_write)) == false) {
		TLS_DPRINTF("tls_key_make_traffic_key");
		return TLS_HS_PHASE_FAILD;
	}

	return TLS_HS_PHASE_DONE;
}

static enum hs_phase do_phase_final(TLS *tls) {
	uint16_t version = tls_util_convert_protover_to_ver(&(tls->negotiated_version));

	switch (version) {
	case TLS_VER_SSL30:
	case TLS_VER_TLS10:
	case TLS_VER_TLS11:
	case TLS_VER_TLS12:
		return do_phase_final_up_to_tls12(tls);

	case TLS_VER_TLS13:
		return do_phase_final_tls13(tls);

	default:
		break;
	}

	return TLS_HS_PHASE_FAILD;
}

static void cleanup(TLS *tls) {
	if (tls->certtype_list_server != NULL) {
		tls_cert_type_free(tls->certtype_list_server);
		tls->certtype_list_server = NULL;
	}

	if (tls->sighash_list != NULL) {
		tls_hs_sighash_free(tls->sighash_list);
		tls->sighash_list = NULL;
	}

	if (tls->sighash_list_cert != NULL) {
		tls_hs_sighash_free(tls->sighash_list_cert);
		tls->sighash_list_cert = NULL;
	}

	if (tls->ecdh != NULL) {
		tls_hs_ecdh_free(tls->ecdh);
		tls->ecdh = NULL;
	}
}

bool tls_hs_cs_client_handshake(TLS *tls) {
	enum hs_phase phase = TLS_HS_PHASE_HELLO;

	while (true) {
		switch (phase) {
		case TLS_HS_PHASE_HELLO:
			phase = do_phase_hello(tls);
			break;

		case TLS_HS_PHASE_CERTIFICATE_RECVING:
			phase = do_phase_recv_cert(tls);
			break;

		case TLS_HS_PHASE_CERTIFICATE_SENDING:
			phase = do_phase_send_cert(tls);
			break;

		case TLS_HS_PHASE_FINAL:
			phase = do_phase_final(tls);
			break;

		case TLS_HS_PHASE_DONE:
			goto fin;

		case TLS_HS_PHASE_FAILD:
			goto err;
		}
	}

fin:
	cleanup(tls);
	return true;

err:
	cleanup(tls);
	return false;
}
