/* storetestfc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_store.h>
#include <aicrypto/ok_tool.h>

#define CACERT		"ca.cer"
#define CAKEY		"ca.key"
#define CACRL		"test.crl"
#define USERCERT	"test.cer"
#define USERREQ		"test.p10"
#define USERKEY		"test.key"
#define PKIXCERT1	"pkix-ex1.ber"
#define PKIXCERT2	"pkix-ex2.ber"
#define CROSSPAIR	"cross.der"
#define DSACERT		"dsacert.der"
#define DSAKEY		"dsakey.der"
#define ECCACERT	"ecca.cer"
#define ECCAKEY		"ecca.key"
#define ECUSERCERT	"ecusr.cer"
#define ECUSERKEY	"ecusr.key"
#define STORE_PATH	"."

#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);


void get_store_fname(char *buf,char *path, char *name, int ctx_type);

int test_store_new(){
	CStore *cs;
	Cert *ct1,*ct2,*ct3;
	CRL *crl;
	Req *req;
	Key *ky1,*ky2,*ky3;
	unsigned char *der;
	char buf[256];
	int i;
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_ECUSERCERT = get_fpath(PATH, ECUSERCERT);
	char *fp_DSACERT = get_fpath(PATH, DSACERT);
	char *fp_CACRL = get_fpath(PATH, CACRL);
	char *fp_USERREQ = get_fpath(PATH, USERREQ);
	char *fp_USERKEY = get_fpath(PATH, USERKEY);
	char *fp_DSAKEY = get_fpath(PATH, DSAKEY);
	char *fp_ECUSERKEY = get_fpath(PATH, ECUSERKEY);

	/* check certificate store */
	for(i=0;i<11;i++){
		if ((ct1 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if ((ct2 = Cert_read_file(fp_ECUSERCERT)) == NULL){
			printf("test store cert read file (2) -- error!\n");
			goto error;
		}
		if ((ct3 = Cert_read_file(fp_DSACERT)) == NULL){
			printf("test store cert read file (3) -- error!\n");
			goto error;
		}
		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_CERT);
		remove(buf);

		/* get new store */
		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_CERT,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		if(CStore_add_cert(cs,ct1,"hoge",AIST_MY)){
			printf("cannot add a cert to the MY store (1) -- error!\n");
			goto error;
		}
		if(CStore_add_cert(cs,ct2,"hoge2",AIST_MY)){
			printf("cannot add a cert to the MY store (2) -- error!\n");
			goto error;
		}
		if(CStore_add_cert(cs,ct3,"hoge3",AIST_MY)){
			printf("cannot add a cert to the MY store (3) -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		/* reload test */
		if(CStore_add_cert(cs,ct3,"hoge4",AIST_MY)){
			printf("cannot add a cert to the MY store (4) -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Cert_free(ct1); Cert_free(ct2); Cert_free(ct3);

		if((i%5)==0) printf("test new & add Cert store -- ok : %d\n",i);
	}

	/* check CRL store */
	for(i=0;i<11;i++){
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test store CRL read file -- error!\n");
			goto error;
		}
		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_CRL);
		remove(buf);

		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_CRL,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		if(CStore_add_crl(cs,crl,"hoge",AIST_MY)){
			printf("cannot add a CRL to the MY store -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		/* reload test */
		if(CStore_add_crl(cs,crl,"hoge2",AIST_MY)){
			printf("cannot add a CRL to the MY store -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		CRL_free(crl);

		if((i%5)==0) printf("test new & add CRL store -- ok : %d\n",i);
	}

	/* check Req store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store Req read file -- error!\n");
			goto error;
		}
		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_CSR);
		remove(buf);

		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_CSR,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		if(CStore_add_req(cs,req,"hoge",AIST_MY)){
			printf("cannot add a Req to the MY store -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test new & add Req store -- ok : %d\n",i);
	}

	/* check Key store */
	for(i=0;i<11;i++){
		OK_set_passwd("abcde");
		if ((ky1 = (Key*)PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			goto error;
		}
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL){
			printf("test read DSA key file -- error!\n");
			goto error;
		}
		if((ky2=(Key*)ASN1_read_dsaprv(der))==NULL){
			printf("test decode DSA key -- error!\n");
			goto error;
		}
		if ((der = ASN1_read_der(fp_ECUSERKEY)) == NULL){
			printf("test read ECDSA key file -- error!\n");
			goto error;
		}
		if((ky3=(Key*)ASN1_read_ecdsaprv(der))==NULL){
			printf("test decode ECDSA key -- error!\n");
			goto error;
		}

		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_KEY);
		remove(buf);

		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_KEY,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if(CStore_add_key(cs,ky1,"hoge",AIST_MY)){
			printf("cannot add a Key to the MY store (1) -- error!\n");
			goto error;
		}
		if(CStore_add_key(cs,ky2,"hoge3",AIST_MY)){
			printf("cannot add a Key to the MY store (2) -- error!\n");
			goto error;
		}
		if(CStore_add_key(cs,ky3,"hoge2",AIST_MY)){
			printf("cannot add a Key to the MY store (3) -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Key_free(ky1); Key_free(ky2); Key_free(ky3);

		if((i%5)==0) printf("test new & add KEY store -- ok : %d\n",i);
	}

	/* check DER store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store DER read file -- error!\n");
			goto error;
		}
		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_DER);
		remove(buf);

		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_DER,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		if(CStore_add_der(cs,req->der,"hoge",AIST_MY)){
			printf("cannot add a DER to the MY store -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test new & add DER store -- ok : %d\n",i);
	}

	/* check ENCDER store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store ENCDER read file -- error!\n");
			goto error;
		}
		/* remove file first*/
		get_store_fname(buf,STORE_PATH,STORE_MY,CSTORE_CTX_ENCDER);
		remove(buf);

		if((cs=CStore_new_file(STORE_PATH,STORE_MY,CSTORE_CTX_ENCDER,CSMODE_NULL))==NULL){
			printf("test store get new store -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if(CStore_add_encder(cs,req->der,"hoge",AIST_MY)){
			printf("cannot add a DER to the MY store -- error!\n");
			goto error;
		}
		if(CStore_update(cs)){
			printf("cannot update the MY store -- error!\n");
			goto error;
		}
		if(CStore_reload(cs)){
			printf("cannot reload the MY store -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test new & add ENCDER store -- ok : %d\n",i);
	}
	return 0;
error:
	if (fp_USERCERT) free(fp_USERCERT);
	if (fp_ECUSERCERT) free(fp_ECUSERCERT);
	if (fp_DSACERT) free(fp_DSACERT);
	if (fp_CACRL) free(fp_CACRL);
	if (fp_USERREQ) free(fp_USERREQ);
	if (fp_USERKEY) free(fp_USERKEY);
	if (fp_DSAKEY) free(fp_DSAKEY);
	if (fp_ECUSERKEY) free(fp_ECUSERKEY);
	return -1;
}


int test_store_search(){
	CStore *cs;
	CSBag *bg;
	Cert *ct1,*ct2,*ct3;
	CertList *ctl;
	CRL *crl;
	Req *req;
	unsigned char hash[32];
	int i,j;
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_CACRL = get_fpath(PATH, CACRL);
	char *fp_USERREQ = get_fpath(PATH, USERREQ);
	char *fp_ECUSERCERT = get_fpath(PATH, ECUSERCERT);
	char *fp_DSACERT = get_fpath(PATH, DSACERT);

	/* search check certificate store */
	for(i=0;i<11;i++){
		if ((ct1 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_CERT,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n%s\n",OK_get_errstr());
			goto error;
		}
		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a cert bag by ID -- error!\n");
			goto error;
		}
		if(strcmp(ct1->subject,((Cert*)bg->cache)->subject)) goto error;
		/* search by serialnumber */
		if((bg=CStore_find_bySNum(CStore_get_firstBag(cs),ct1->serialNumber))==NULL){
			printf("cannot get a cert bag by serialNumber -- error!\n");
			goto error;
		}
		if(strcmp(ct1->subject,((Cert*)bg->cache)->subject)) goto error;
		/* search by subject */
		if((bg=CStore_find_bySubject(CStore_get_firstBag(cs),ct1->subject))==NULL){
			printf("cannot get a cert bag by subject -- error!\n");
			goto error;
		}
		if(strcmp(ct1->subject,((Cert*)bg->cache)->subject)) goto error;
		/* search by issuer */
		if((bg=CStore_find_byIssuer(CStore_get_firstBag(cs),ct1->issuer))==NULL){
			printf("cannot get a cert bag by issuer -- error!\n");
			goto error;
		}
		if(strcmp(ct1->subject,((Cert*)bg->cache)->subject)) goto error;
		/* search by key_hash */
		if(cs_get_keyhash(ct1->pubkey,hash,&j)) goto error;
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a cert bag by KeyHash -- error!\n");
			goto error;
		}
		if(strcmp(ct1->subject,((Cert*)bg->cache)->subject)) goto error;

		/* error check */
		if(CStore_find_byID(CStore_get_firstBag(cs),"hoge4")){
			printf("bad cert bag byID (should be NULL) -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Cert_free(ct1);

		if((i%5)==0) printf("test open & search Cert store -- ok : %d\n",i);
	}

	/* search check CRL store */
	for(i=0;i<11;i++){
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test store CRL read file -- error!\n");
			goto error;
		}
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_CRL,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a CRL bag by ID -- error!\n");
			goto error;
		}
		if(strcmp(crl->issuer,((CRL*)bg->cache)->issuer)) goto error;
		/* search by Issuer */
		if((bg=CStore_find_byIssuer(CStore_get_firstBag(cs),crl->issuer))==NULL){
			printf("cannot get a CRL bag by Issuer -- error!\n");
			goto error;
		}
		if(strcmp(crl->issuer,((CRL*)bg->cache)->issuer)) goto error;

		/* error check */
		if(CStore_find_bySubject(CStore_get_firstBag(cs),crl->issuer)){
			printf("bad cert bag bySubject (should be NULL) -- error!\n");
			goto error;
		}

		CStore_close(cs);
		CRL_free(crl);

		if((i%5)==0) printf("test open & search CRL store -- ok : %d\n",i);
	}

	/* check Req store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store Req read file -- error!\n");
			goto error;
		}
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_CSR,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a Req bag by ID -- error!\n");
			goto error;
		}
		if(strcmp(req->subject,((Req*)bg->cache)->subject)) goto error;
		/* search by Issuer */
		if((bg=CStore_find_bySubject(CStore_get_firstBag(cs),req->subject))==NULL){
			printf("cannot get a Req bag by Subject -- error!\n");
			goto error;
		}
		if(strcmp(req->subject,((Req*)bg->cache)->subject)) goto error;
		/* search by KeyHash */
		if(cs_get_keyhash(req->pubkey,hash,&j)) goto error;
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a Req bag by KeyHash -- error!\n");
			goto error;
		}
		if(strcmp(req->subject,((Req*)bg->cache)->subject)) goto error;

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test open & search Req store -- ok : %d\n",i);
	}

	/* check private key store */
	for(i=0;i<11;i++){
		if ((ct1 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if ((ct2 = Cert_read_file(fp_ECUSERCERT)) == NULL){
			printf("test store cert read file (2) -- error!\n");
			goto error;
		}
		if ((ct3 = Cert_read_file(fp_DSACERT)) == NULL){
			printf("test store cert read file (3) -- error!\n");
			goto error;
		}
		
		OK_set_passwd("abcde");
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_KEY,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}

		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a Req bag by ID -- error!\n");
			goto error;
		}
		if((ct1->pubkey->key_type+1) != ((Key*)bg->cache)->key_type) goto error;

		/* search by KeyHash -- RSA key */
		if(cs_get_keyhash(ct1->pubkey,hash,&j)) goto error;
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a Key bag by KeyHash (RSA) -- error!\n");
			goto error;
		}
		/* search by KeyHash -- ECDSA key */
		if(cs_get_keyhash(ct2->pubkey,hash,&j)) goto error;
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a Key bag by KeyHash (ECDSA) -- error!\n");
			goto error;
		}
		/* search by KeyHash -- DSA key */
		if(cs_get_keyhash(ct3->pubkey,hash,&j)) goto error;
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a Key bag by KeyHash (DSA) -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Cert_free(ct1); Cert_free(ct2); Cert_free(ct3);

		if((i%5)==0) printf("test open & search KEY store -- ok : %d\n",i);
	}

	/* check DER store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store DER read file -- error!\n");
			goto error;
		}
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_DER,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a DER bag by ID -- error!\n");
			goto error;
		}
		/* search by KeyHash */
		ASN1_skip_(req->der,&j);
		OK_SHA1(j,req->der,hash);
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a DER bag by KeyHash -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test open & search DER store -- ok : %d\n",i);
	}

	/* check ENCDER store */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store ENCDER read file -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_ENCDER,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		/* search by ID */
		if((bg=CStore_find_byID(CStore_get_firstBag(cs),"hoge"))==NULL){
			printf("cannot get a ENCDER bag by ID -- error!\n");
			goto error;
		}
		/* search by KeyHash */
		ASN1_skip_(req->der,&j);
		OK_SHA1(j,req->der,hash);
		if((bg=CStore_find_byKeyHash(CStore_get_firstBag(cs),hash))==NULL){
			printf("cannot get a ENCDER bag by KeyHash -- error!\n");
			goto error;
		}
		if(memcmp(bg->cache,req->der,j)){
			printf("cannot decrypto a ENCDER bag -- error!\n");
			goto error;
		}

		CStore_close(cs);
		Req_free(req);

		if((i%5)==0) printf("test open & search ENCDER store -- ok : %d\n",i);
	}

	/* test store to list */
	for(i=0;i<11;i++){
		/* cache mode (default) */
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_CERT,CSMODE_CACHE))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		if((ctl=CStore_2certlist(cs))==NULL){
			printf("test store to certlist -- error!\n%s\n",OK_get_errstr());
			goto error;
		}
		CStore_close(cs);
		Certlist_free_all(ctl);
		/* non cache mode */
		if((cs=CStore_open_file(STORE_PATH,STORE_MY,CSTORE_CTX_CERT,CSMODE_NULL))==NULL){
			printf("test store get open store -- error!\n");
			goto error;
		}
		if((ctl=CStore_2certlist(cs))==NULL){
			printf("test store to certlist -- error!\n");
			goto error;
		}
		CStore_close(cs);
		Certlist_free_all(ctl);
		if((i%5)==0) printf("test store to certlist -- ok : %d\n",i);
	}
	return 0;
error:
	if (fp_USERCERT) free(fp_USERCERT);
	if (fp_CACRL) free(fp_CACRL);
	if (fp_USERREQ) free(fp_USERREQ);
	if (fp_ECUSERCERT) free(fp_ECUSERCERT);
	if (fp_DSACERT) free(fp_DSACERT);
	printf("test search Store -- error!! : %d\n",i);
	return -1;
}


int test_manager_new(){
	Cert *ct1,*ct2;
	CRL *crl;
	Key *key;
	Req *req;
	STManager *stm;
	int i;
	char *fp_CACERT = get_fpath(PATH, CACERT);
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_CACRL = get_fpath(PATH, CACRL);
	char *fp_USERKEY = get_fpath(PATH, USERKEY);
	char *fp_USERREQ = get_fpath(PATH, USERREQ);
	char *rmfile = get_fpath(STORE_PATH, "*.acs");
	size_t command_len = strlen("rm ") + strlen(rmfile) + 1;
	char *command;
	if ((command = (char *)malloc(command_len)) == NULL) goto error;
	strcpy(command, "rm ");
	strcat(command, rmfile);

	/* clean manager directory and create new one */
	for(i=0;i<11;i++){
		/* system("rm ./\*.acs"); */
		system(command);

		if ((stm = STM_system_new(STORE_PATH)) == NULL){
			printf("cannot get new store manager -- error!\n");
			goto error;
		}
		STM_close(stm);

		if ((stm = STM_open(STORE_PATH)) == NULL){
			printf("cannot get open store manager -- error!\n");
			goto error;
		}
		STM_close(stm);

		if((i%5)==0) printf("test new & open store manager -- ok : %d\n",i);
	}

	/* store certificate & CRL */
	for(i=0;i<11;i++){
		if ((stm = STM_open(STORE_PATH)) == NULL){
			printf("cannot get open store manager -- error!\n");
			goto error;
		}
		if ((ct1 = Cert_read_file(fp_CACERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if ((ct2 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test store CRL read file -- error!\n");
			goto error;
		}
		if(STM_import_cert(stm, ct1, "hoge-test-id-00")){
			printf("test store import cert -- error!\n");
			goto error;
		}
		/* replace test */
		if(STM_import_cert(stm, ct1, "hoge-test-id-01")){
			printf("test store import cert -- error!\n");
			goto error;
		}
		if(STM_find_byID(stm,STORE_ROOT,CSTORE_ON_STORAGE,CSTORE_CTX_CERT,"hoge-test-id-00")){
			printf("test store cert replace -- error!\n");
			goto error;
		}
		/* should be error -- this prints out [aicrypto error],
		 * so this test was removed. (because it's pretty confusing. )
		if(!STM_import_cert(stm, ct1, "hoge-test-id-01")){
			printf("test store import cert (2) -- error!\n");
			goto error;
		} */
		if(STM_import_cert(stm, ct2, NULL)){
			printf("test store import cert (3) -- error!\n");
			goto error;
		}
		/* crl test */
		if(STM_import_crl(stm, crl, "hoge-test-id-00")){
			printf("test store import CRL -- error!\n");
			goto error;
		}
		/* replace test */
		if(STM_import_crl(stm, crl, "hoge-test-id-01")){
			printf("test store import CRL -- error!\n");
			goto error;
		}
		if(STM_find_byID(stm,STORE_ROOT,CSTORE_ON_STORAGE,CSTORE_CTX_CRL,"hoge-test-id-00")){
			printf("test store CRL replace -- error!\n");
			goto error;
		}
		/* delete test */
		if(STM_del_byID(stm,STORE_ROOT,CSTORE_ON_STORAGE,CSTORE_CTX_CRL,"hoge-test-id-01")){
			printf("test store CRL delete -- error!\n");
			goto error;
		}
		if(STM_del_byID(stm,STORE_ROOT,CSTORE_ON_STORAGE,CSTORE_CTX_CERT,"hoge-test-id-01")){
			printf("test store cert delete -- error!\n");
			goto error;
		}

		STM_close(stm);
		Cert_free(ct1); Cert_free(ct2);
		CRL_free(crl);
		if((i%5)==0) printf("test import (delete) cert & CRL -- ok : %d\n",i);
	}

	/* store certificate & private key */
	for(i=0;i<11;i++){
		if ((stm = STM_open(STORE_PATH)) == NULL){
			printf("cannot get open store manager -- error!\n");
			goto error;
		}
		if ((ct1 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if ((key = (Key*)PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			goto error;
		}
		if(STM_import_certkey(stm, ct1, key, NULL)){
			printf("test store import cert & key -- error!\n");
			goto error;
		}
		/* replace test */
		if(STM_import_certkey(stm, ct1, key, "hoge-test-id-00")){
			printf("test store import cert & key -- error!\n");
			goto error;
		}
		/* delete test */
		if(STM_del_byID(stm,STORE_MY,CSTORE_ON_STORAGE,CSTORE_CTX_CERT,"hoge-test-id-00")){
			printf("test store cert delete -- error!\n");
			goto error;
		}
		if(STM_del_byID(stm,STORE_MY,CSTORE_ON_STORAGE,CSTORE_CTX_KEY,"hoge-test-id-00")){
			printf("test store CRL delete -- error!\n");
			goto error;
		}

		STM_close(stm);
		Cert_free(ct1); Key_free(key);
		if((i%5)==0) printf("test import (delete) cert & key -- ok : %d\n",i);
	}

	/* store CSR & private key & certificate */
	for(i=0;i<11;i++){
		if ((stm = STM_open(STORE_PATH)) == NULL){
			printf("cannot get open store manager -- error!\n");
			goto error;
		}
		if ((ct1 = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test store cert read file -- error!\n");
			goto error;
		}
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test store req read file -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if ((key = (Key*)PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			goto error;
		}
		if(STM_import_reqkey(stm, req, key, NULL)){
			printf("test store import req & key -- error!\n");
			goto error;
		}
		/* certificate should be stored in "MY" store. */
		if(STM_import_cert(stm, ct1, "hoge-test-id-00")){
			printf("test store import a cert -- error!\n");
			goto error;
		}
		/* delete test */
		if(STM_del_byID(stm,STORE_MY,CSTORE_ON_STORAGE,CSTORE_CTX_CERT,"hoge-test-id-00")){
			printf("test store cert delete -- error!\n");
			goto error;
		}
		STM_close(stm);
		Cert_free(ct1); Req_free(req); Key_free(key);
		if((i%5)==0) printf("test import req & key -- ok : %d\n",i);
	}

	/* cert verify test */
	if ((stm = STM_open(STORE_PATH)) == NULL){
		printf("cannot get open store manager -- error!\n");
		goto error;
	}
	if ((ct1 = Cert_read_file(fp_CACERT)) == NULL){
		printf("test store cert read file -- error!\n");
		goto error;
	}
	if ((ct2 = Cert_read_file(fp_USERCERT)) == NULL){
		printf("test store cert read file -- error!\n");
		goto error;
	}
	if ((crl = CRL_read_file(fp_CACRL)) == NULL){
		printf("test store CRL read file -- error!\n");
		goto error;
	}
	if(STM_import_cert(stm, ct1, "hoge-test-id-00")){
		printf("test store import a cert -- error!\n");
		goto error;
	}
	if(STM_import_cert(stm, ct2, "hoge-test-id-01")){
		printf("test store import a cert -- error!\n");
		goto error;
	}
	if(STM_import_crl(stm, crl, "hoge-test-id-00")){
		printf("test store import a crl -- error!\n");
		goto error;
	}
	if(STM_update(stm)){
		printf("test store update -- error!\n");
		goto error;
	}
	/* reload store test */
	if(STM_import_cert(stm, ct2, "hoge-test-id-02")){
		printf("test store import a cert -- error!\n");
		goto error;
	}
	if(STM_reload(stm)){
		printf("test store reload -- error!\n");
		goto error;
	}
	STM_close(stm);
	Cert_free(ct1); CRL_free(crl);

	for(i=0;i<11;i++){
		if ((stm = STM_open(STORE_PATH)) == NULL){
			printf("cannot get open store manager -- error!\n");
			goto error;
		}
		if(STM_verify_cert(stm,ct2,0)){
			printf("test store cert verify -- error!\n");
			goto error;
		}
		STM_close(stm);

		if((i%5)==0) printf("test manager certificate verify -- ok : %d\n",i);
	}
	Cert_free(ct2); 
	if (command) free(command);
	return 0;
error:
	if (fp_CACERT) free(fp_CACERT);
	if (fp_USERCERT) free(fp_USERCERT);
	if (fp_CACRL) free(fp_CACRL);
	if (fp_USERKEY) free(fp_USERKEY);
	if (fp_USERREQ) free(fp_USERREQ);
	if (command) free(command);
	printf("test store manager -- error!! : %d\n",i);
	return -1;
}
