/* testserver2.c */
/* SSL test server (parent&child process version) */
/*
 * Copyright (c) 2012-2015 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>  /* EINTR */

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include <sys/types.h>

#ifdef __WINDOWS__
#undef ULONG
#include <winsock2.h>
#include <io.h>
#else
#include <sys/socket.h>
#include <netinet/in.h>

#include <signal.h>
#endif

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_ssl.h>
#include <aicrypto/ok_x509.h>
#include "ssl.h"

char httpstxt[]="HTTP/1.1 200 OK\r\nDate: Wed, 25 Oct 2000 11:11:11 GMT\r\nServer: AiSSL Test server (UNIX)\r\nLast-Modified: Thu, 12 Oct 2000 08:23:54 GMT\r\nAccept-Ranges: bytes\r\nContent-Length: 73\r\nKeep-Alive: timeout=15, max=100\r\nConnection: Keep-Alive\r\nContent-Type: text/html\r\n\r\n<html><title>SSL Connection!</title>\r\n<body>SSL Connection!</body>\r\n</html>\r\n";

#define SSL_PORT	"11111"

void test_do(SSL *ssl);
void check_opt(int argc, char **argv);

int no_scert=0; /* don't set server certificate */
int req_flag=0;	/* set certificate request flag */
int no_recon=0; /* no SSL reconnection flag */
int vfy_cert=0; /* verify client certificate flag */
int af=AF_UNSPEC; /* address family */

int main(int argc, char **argv){
	SSL *ssls[FD_SETSIZE];
	SSL **s = ssls;
	int ssls_len = 0;
	Cert *ct;
	char buf[64];
	struct addrinfo hints;
	struct addrinfo *ai, *aitop;
	fd_set fds, readfds;
	int i, j, smax, error;


#ifdef __WINDOWS__
	WSADATA	wsaData;

	if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
		printf("error : WSAStartup\n");
		return(-1);}
#endif

	check_opt(argc,argv);

#if defined(linux)
	/* SYSV can use this function to clean zombie process :-) */
	signal(SIGCLD, SIG_IGN);
#endif /* FIXME: add some code for BSD variants. */

	/* get addrinfo list */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = af;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_PASSIVE;
	if((error = getaddrinfo(NULL, SSL_PORT, &hints, &aitop)) != 0) {
		printf("error : getaddrinfo(%s)\n", gai_strerror(error));
		return -1;
	}

	for(ai = aitop; ai; ai = ai->ai_next) {
		/* 1. get server socket */
		if (NULL == (*s = SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))){
			printf("error : SSL_socket\n");
			return(-1);}
		if(ai->ai_family == AF_INET6) {
			/* disable IPv4 mapped address */
			int on = 1;
			if(SSL_setsockopt(*s, IPPROTO_IPV6, IPV6_V6ONLY, &on, sizeof(on)) == -1) {
				printf("error : SSL_setsockopt\n");
				return -1;
			}
		}

		/* 2. setup server information */
		/* 2.1 read server certificate & private key */
		if(!no_scert){
			if(SSL_set_server_p12(*s,"./00001.p12","abcde")){
				printf("error : SSL_set_server_p12\n");
				return(-1);}

			printf("read server certificate:\n");
			ct = SSL_get_server_cert(*s);	/* server PKCS#12 file */
			printf("subject: %s\n",ct->subject);
			printf("issuer:  %s\n",ct->issuer);
		}

		/* 2.2 read verify.idx for verifying client certificate */
		if((vfy_cert)&&(req_flag)){
			if(SSL_set_store(*s,NULL)){
				printf("error : SSL_set_store : %s\n",OK_get_errstr());
				return(-1);
			}
		}else{
			SSL_set_vfytype(*s,SSL_DONT_VERIFY);
		}

		/* 2.3 certificate request will be sent */
		if(req_flag){
			printf("set certificate request option.\n");
			SSL_setopt(*s,SSL_OPT_CERTREQ);
		}

		/* 2.4 set ssl reconnection cache buffer */
		if(no_recon){
			SSL_set_list_max(*s,0);
		}

		/* 3. bind & listen socket */
		if(-1 == SSL_bind(*s,ai->ai_addr,ai->ai_addrlen)){
			printf("error : SSL_bind\n");
			return(-1);}
		if(SSL_listen(*s,5)){
			printf("error : SSL_listen\n");
			return(-1);}

		++s;
		++ssls_len;
		if(ssls_len >= FD_SETSIZE) {
			break;
		}
	}
	freeaddrinfo(aitop);

	FD_ZERO(&fds);
	smax = -1;
	for(i = 0; i < ssls_len; ++i) {
		FD_SET(ssls[i]->sock, &fds);
		if(ssls[i]->sock > smax) {
			smax = ssls[i]->sock;
		}
	}
	for(i = 0; i < 1000; ++i) {
		memcpy(&readfds, &fds, sizeof(fd_set));
		if(select(smax + 1, &readfds, NULL, NULL, NULL) < 0) {
			if(errno == EINTR)
				continue;
			printf("error : select\n");
			return -1;
		}
		for(j = 0; j < ssls_len; ++j) {
			if(FD_ISSET(ssls[j]->sock, &readfds)) {
				/* 4. accept the connection */
				struct sockaddr_storage sa;
				socklen_t sa_len = sizeof(sa);
				SSL *ssl;
				if(NULL == (ssl = SSL_accept(ssls[j],(struct sockaddr *)&sa,&sa_len))){
					printf("error: SSL_accept\n");
					SSL_close(ssls[j]);
					return(-1);}

				/* 5. do SSL handshake :-) */
				if(SSL_handshake(ssl)){
					printf("error: SSL_handshake\n");
					SSL_close(ssl);
					SSL_close(ssls[j]);
					return(-1);}

				/* the server might get SSL reconnection. and its handshake is
				 * faster than full handshake. so, new "ssl" should be listed in
				 * the list of the listening SSL socket.
				 *
				 * !caution! this function must be after SSL_handshake() done.
				 */
				if(SSL_add_connect_list(ssls[j],ssl)){
					/* it may occur when "ssls[j]" doesn't have ssls[j]->ctx. */
					printf("error: SSL_add_connect_list\n");
				}

				printf("SSL connection was established!\n");
				SSL_cspec_str(ssl->ctx,buf);
				printf("using cipher : %s\n",buf);

				if(ct = SSL_get_client_cert(ssl)){
					/* client PKCS#12 file */
					printf("read client certificate:\n");
					printf("subject: %s\n",ct->subject);
					printf("issuer:  %s\n",ct->issuer);
				}
#ifdef __WINDOWS__
				/* 6. and now, send messages with SSL_read & SSL_write !! */
				test_do(ssl);
#else
				if(fork()==0){
					/* this one is child process */
					/* 6. and now, send messages with SSL_read & SSL_write !! */
					test_do(ssl);
					SSL_free(ssl);
					SSL_free(ssls[j]);
					exit(0);
				}

				/* this one is parent process */
				SSL_close(ssl);
#endif

				/* at first time, SSL_free(ssl) will not work, because its "ssl" is
				 * listed in the connection list of the listen SSL socket "ssls[j]" ...
				 * so, it will be FREEd as same timing as SSL_free(ssls[j]) .
				 */
				SSL_free(ssl);
			}
		}
	}

	for(i = 0; i < ssls_len; ++i) {
		SSL_close(ssls[i]);
		SSL_free(ssls[i]);
	}
#ifdef __WINDOWS__
	WSACleanup();
#endif
	exit(0);
}

void test_do(SSL *ssl){
	unsigned char buf[1024];
	int i;

	memset(buf,0,1024);
	/* read message first */
	if((i=SSL_read(ssl,buf,1024))<0){
		printf("error : SSL_read()\n");
		return;}
	printf("now getting one message : %s\n",buf);

	printf("now writing one message : %s\n",httpstxt);
	if(SSL_write(ssl,httpstxt,strlen(httpstxt))<0){
		printf("error : SSL_write()\n");
		return;}

	/* server doesn't need to get close_notify.
	 * this one is just for test.
	 */
	SSL_read(ssl,buf,256);

	SSL_close(ssl);
}

void check_opt(int argc, char **argv){
    int   i;

    for(i=1;i<argc;i++){
        if(!strcmp(argv[i],"-cr")){
	    req_flag=1;
        }else if(!strcmp(argv[i],"-nr")){
	    no_recon=1;
        }else if(!strcmp(argv[i],"-ns")){
	    no_scert=1;
        }else if(!strcmp(argv[i],"-v")){
	    vfy_cert=1;
        }else if(!strcmp(argv[i],"-4")){
	    af=AF_INET;
        }else if(!strcmp(argv[i],"-6")){
	    af=AF_INET6;
        }else{
	    printf("usage:\n");
	    printf("tserv2 [option]\n");
	    printf("\t-cr  ... send certificate request in that SSL handsake.\n");
	    printf("\t-nr  ... don't use SSL reconnection.\n");
	    printf("\t-ns  ... don't use server certificate.\n");
	    printf("\t-v   ... verify client certificate (with -cr)\n");
	    printf("\t-4   ... use IPv4 only\n");
	    printf("\t-6   ... use IPv6 only\n");
	    exit(0);
	}
    }
}

