/* testclient.c - SSL test client (simple version) */
/*
 * Copyright (c) 2012-2015 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include <sys/types.h>
#ifdef __WINDOWS__
#undef ULONG
#include <winsock2.h>
#include <io.h>
#else
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_ssl.h>
#include <aicrypto/ok_x509.h>
#include "ssl.h"

#define SSL_SERV		"localhost"
#define SSL_SERV4		"127.0.0.1"
#define SSL_SERV6		"::1"
#define SERVER_PORT		"11112"

#ifdef __WINDOWS__
#define P12FNAME		".\\00001.p12"
#else
#define P12FNAME		"00001.p12"
#endif
#ifndef PATH
# define PATH	"."	/* path to the directory that includes P12FNAME */
#endif
#ifndef STORE_PATH
# define STORE_PATH	"../test"	/* FIXME: should be more appropriate */
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);

static int myread( int sock,char *buf,int len);
static int mywrite(int sock,char *buf,int len);
static int read_debug(SSL *ssl, int dat);
static int write_debug(SSL *ssl, int dat);
static int vfy_func(SSL *ssl, Cert *ct);

int test_do(SSL *ssl);

int main(int argc, char **argv){
	SSL *s;
	Cert *ct;
	char buf[64];
	struct addrinfo hints;
	struct addrinfo *aihead, *ai;
	int af = AF_UNSPEC;
	char *ssl_serv = SSL_SERV;
	int error;
	char *fp_P12FNAME = get_fpath(PATH, P12FNAME);

	/* parse command line options like check_opt() in testclient2.c */
	if(argc == 1+1) {
		if(strcmp(argv[1], "-4") == 0) {
			/* use IPv4 only */
			af = AF_INET;
			ssl_serv = SSL_SERV4;
		} else if(strcmp(argv[1], "-6") == 0) {
			/* use IPv6 only */
			af = AF_INET6;
			ssl_serv = SSL_SERV6;
		} else {
			printf("error : unknown option\n");
			return -1;
		}
	}

#ifdef __WINDOWS__
	WSADATA	wsaData;

	if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
		printf("error : WSAStartup\n");
		return(-1);}
#endif

	/* get addrinfo list */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = af;
	hints.ai_socktype = SOCK_STREAM;
	if((error = getaddrinfo(ssl_serv, SERVER_PORT, &hints, &aihead)) != 0) {
		printf("error : getaddrinfo(%s)\n", gai_strerror(error));
		return -1;
	}

	for (ai = aihead; ai != NULL; ai = ai->ai_next) {
		/* 1. get client socket */
		if (NULL == (s = SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))){
			printf("warning : SSL_socket\n");
			goto loop_warn;
		}

		/* 2. setup client information */
		/* 2.1 read client certificate & private key */
#if 1
		if(SSL_set_client_p12(s,fp_P12FNAME,"abcde")){
			printf("warning : SSL_set_client_p12\n");
			goto loop_warn;
		}

		printf("read client certificate:\n");
		ct = SSL_get_client_cert(s);	/* server PKCS#12 file */
		printf("subject: %s\n",ct->subject);
		printf("issuer:  %s\n",ct->issuer);
#endif
		/* 2.2 read verify.idx for verifying server certificate */
		/* if certlist path is set NULL, this means that SSL uses
		 * the default path. (it's depend on the installation.
		 * if you install library at /usr/local/aica , 
		 * the default path will be /usr/local/aica/certs ...)
		 */
		if(SSL_set_store(s,STORE_PATH)){ /* store manager path is "../test" */
			/* cannot get certificate list.
			 * so don't verify any server certificates >:(
			 */
			SSL_set_vfytype(s,SSL_DONT_VERIFY);
		}

		/* 2.3 set debug callback functions */
		SSL_set_read_cb(s,myread);
		SSL_set_write_cb(s,mywrite);
		SSL_set_readdebug_cb(s,read_debug);
		SSL_set_writedebug_cb(s,write_debug);
		SSL_set_vfy_cb(s,vfy_func);


		/* 3. connect to the server */
		if (SSL_connect(s,ai->ai_addr,ai->ai_addrlen)) {
			printf("warning : SSL_connect\n");
			goto loop_warn;
		}

		break;

loop_warn:
		if (s != NULL) {
			SSL_close(s);
			SSL_free(s);
			s = NULL;
		}
	}

	if (s == NULL) {
		printf("error : no socket connection to %s\n", SSL_SERV);
		return(-1);
	}

	/* 4. do SSL handshake :-) */
	if(SSL_handshake(s)){
		printf("error : SSL_handshake\n");
		return(-1);
	}

	printf("SSL connection was established!\n");
	SSL_cspec_str(s->ctx,buf);
	printf("using cipher : %s\n",buf);

	if(s->ctx->sp12){
		printf("read server certificate:\n");
		ct = SSL_get_server_cert(s);	/* client PKCS#12 file */
		printf("subject: %s\n",ct->subject);
		printf("issuer:  %s\n",ct->issuer);
	}

	/* 5. and now, send messages with SSL_read & SSL_write !! */
	if(test_do(s))
	    return -1;

	freeaddrinfo(aihead);

#ifdef __WINDOWS__
	WSACleanup();
#endif
	return 0;
}

int test_do(SSL *ssl){
	unsigned char buf[256];
	int	i;

	memset(buf,0,256);
	/* read message first */
	strcpy(buf,"Hello World!");
	printf("now write message!\n");
	printf ("Send %zd chars:'%s'\n",strlen(buf),buf);
	if(SSL_write(ssl,buf,strlen(buf))<0){
		printf("error : SSL_write()\n");
		return -1;}

	printf("now waiting message!\n");
	if((i=SSL_read(ssl,buf,256))<0){
		printf("error : SSL_write()\n");
		return -1;}

	buf[i] = '\0';
	printf ("Recv %d chars:'%s'\n",i,buf);

	SSL_shutdown(ssl,2);  /* send SSL/TLS close_notify */

	SSL_close(ssl);
	SSL_free(ssl);
	return 0;
}

static int read_debug(SSL *ssl, int dat){
    printf("c:r_debug  : md=%d, op=%.2x, st=%d, %d\n",
            ssl->mode,ssl->opt,ssl->ctx->state,dat);
    return 0;
}

static int write_debug(SSL *ssl, int dat){
    printf("c:w_debug : md=%d, op=%.2x, st=%d, %d\n",
            ssl->mode,ssl->opt,ssl->ctx->state,dat);
    return 0;
}

static int vfy_func(SSL *ssl, Cert *ct){
    printf("c:vfy_func: %s\n",ct->subject);
    return 0;
}

static int myread(int sock,char *buf,int len){
    int i;
#ifdef __WINDOWS__
	i=recv(sock,buf,len,0);
#else
	i=read(sock,buf,len);
#endif
    printf("&%d&",i);
    return i;
}

static int mywrite(int sock,char *buf,int len){
    int i;
#ifdef __WINDOWS__
	i=send(sock,buf,len,0);
#else
	i=write(sock,buf,len);
#endif
    printf("#%d#",i);
    return i;
}
