/* ssl_hskey.c - Handshake procedures */
/*
 * Modified by National Insitute of Informatics in Japan, 2011-2015.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_des.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_rc2.h>
#ifdef HAVE_ARC4
#include <aicrypto/ok_rc4.h>
#endif
#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_ssl.h>
#include "ssl.h"

/*-----------------------------------------
  SSL Handshake (Generate Master Secret)
-----------------------------------------*/
void SSL_gen_mastersecret(SSLCTX *ctx){
	MD5_CTX		mctx;
	SHA1_CTX	sctx;
	int	i;
	unsigned char tmp[32];
	unsigned char hstr[4][4] = {"A","BB","CCC"};

	for(i=0;i<3;i++){
		SHA1init(&sctx);
		SHA1update(&sctx,hstr[i],i+1);
		SHA1update(&sctx,ctx->premaster,48);
		SHA1update(&sctx,ctx->chello->random,32);
		SHA1update(&sctx,ctx->shello->random,32);
		SHA1final(tmp,&sctx);

		MD5Init(&mctx);
		MD5Update(&mctx,ctx->premaster,48);
		MD5Update(&mctx,tmp,20);
		MD5Final(&(ctx->master_secret[i<<4]),&mctx);	/* [i*16] */
	}
}

static Key *
NULLkey_new()
{
	Key *ret;

	ret = malloc(sizeof(Key));
	if (ret == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_SSLHS,
		    ERR_PT_SSLHS_KEY + 3, NULL);
		return NULL;
	}

	ret->key_type = KEY_NULL;
	ret->size = 0;
	return ret;
}


/*-----------------------------------------
  SSL Handshake (Generate Master Secret)
-----------------------------------------*/
int SSL_gen_writekey(SSLCTX *ctx){
	MD5_CTX		mctx;
	SHA1_CTX	sctx;
	int i, p;
	int hash_size;
	int key_size;
	int iv_size;
	int block_size;
	int key_block_size;
	unsigned char tmp[32];
	unsigned char skey[32];
	unsigned char ckey[32];
	unsigned char siv[32];
	unsigned char civ[32];
#define KEY_BLOCK_MAX 10
	unsigned char keyblock[MD5_DIGESTSIZE * KEY_BLOCK_MAX];
	unsigned char *hstr[KEY_BLOCK_MAX] = {
		"A", "BB", "CCC", "DDDD", "EEEEE", "FFFFFF", "GGGGGGG",
		"HHHHHHHH", "IIIIIIIII", "JJJJJJJJJJ"};

	hash_size = ctx->cspec->hash_size;
	key_size = ctx->cspec->key_material;
	iv_size = ctx->cspec->IV_size;
	block_size = ctx->cspec->block_size;

	key_block_size = ((hash_size + key_size + iv_size) * 2
		 + MD5_DIGESTSIZE - 1) / MD5_DIGESTSIZE;
	if (key_block_size > KEY_BLOCK_MAX) {
		printf("key_block_size = %d\n", key_block_size); /* debug */
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,
			     ERR_LC_SSLHS, ERR_PT_SSLHS_KEY + 2, NULL);
		ctx->errnum = SSL_AD_ILLEGAL_PARAMETER | (SSL_AL_FATAL << 8);
		return -1;
	}

	for (i = 0; i < key_block_size; i++) {
		SHA1init(&sctx);
		SHA1update(&sctx,hstr[i],i+1);
		SHA1update(&sctx,ctx->master_secret,48);
		SHA1update(&sctx,ctx->shello->random,32);
		SHA1update(&sctx,ctx->chello->random,32);
		SHA1final(tmp,&sctx);

		MD5Init(&mctx);
		MD5Update(&mctx,ctx->master_secret,48);
		MD5Update(&mctx,tmp,20);
		MD5Final(&(keyblock[i * MD5_DIGESTSIZE]), &mctx);
	}

	/* set write MAC */
	if (hash_size) {
		memcpy(ctx->client_write_MAC_secret, keyblock, hash_size);
		p = hash_size;
		memcpy(ctx->server_write_MAC_secret, &(keyblock[p]),
		       hash_size);
		p += hash_size;
	}
	/* set key */
	if (key_size) {
		memcpy(ckey, &(keyblock[p]), key_size);
		p += key_size;
		memcpy(skey, &(keyblock[p]), key_size);
		p += key_size;
		if(ctx->cspec->is_exportable){
			/* export crient key */
			MD5Init(&mctx);
			MD5Update(&mctx, ckey, key_size);
			MD5Update(&mctx,ctx->chello->random,32);
			MD5Update(&mctx,ctx->shello->random,32);
			MD5Final(ckey,&mctx);
			/* export crient key */
			MD5Init(&mctx);
			MD5Update(&mctx, skey, key_size);
			MD5Update(&mctx,ctx->shello->random,32);
			MD5Update(&mctx,ctx->chello->random,32);
			MD5Final(skey,&mctx);
			key_size = MD5_DIGESTSIZE;
		}
	}
	/* set iv */
	if (iv_size) {
		if(ctx->cspec->is_exportable){
			/* export crient IV */
			MD5Init(&mctx);
			MD5Update(&mctx,ctx->chello->random,32);
			MD5Update(&mctx,ctx->shello->random,32);
			MD5Final(civ,&mctx);
			/* export server IV */
			MD5Init(&mctx);
			MD5Update(&mctx,ctx->shello->random,32);
			MD5Update(&mctx,ctx->chello->random,32);
			MD5Final(siv, &mctx);

		}else{
			memcpy(civ, &(keyblock[p]), iv_size);
			p += iv_size;
			memcpy(siv, &(keyblock[p]), iv_size);
			p += iv_size;
		}
	}
#if 0 /* debug */
	{
		int n;

		printf("client_write_MAC_secret: ");
		for (n = 0; n < hash_size; n++) {
			printf("%02x ", ctx->client_write_MAC_secret[n]);
		}
		printf("\n");
		printf("server_write_MAC_secret: ");
		for (n = 0; n < hash_size; n++) {
			printf("%02x ", ctx->server_write_MAC_secret[n]);
		}
		printf("\n");
		printf("ckey: ");
		for (n = 0; n < key_size; n++) {
			printf("%02x ", ckey[n]);
		}
		printf("\n");
		printf("skey: ");
		for (n = 0; n < key_size; n++) {
			printf("%02x ", skey[n]);
		}
		printf("\n");
		printf("civ: ");
		for (n = 0; n < iv_size; n++) {
			printf("%02x ", civ[n]);
		}
		printf("\n");
		printf("siv: ");
		for (n = 0; n < iv_size; n++) {
			printf("%02x ", siv[n]);
		}
		printf("\n");
	}
#endif /* debug */
	switch(ctx->cspec->bulk_cipher_algorithm){
	case OBJ_CRYALGO_AES256CBC:
	case OBJ_CRYALGO_AES128CBC:
		ctx->ckey = (Key *)AESkey_new(key_size, ckey, block_size);
		ctx->skey = (Key *)AESkey_new(key_size, skey, block_size);
		AES_set_iv((Key_AES *)ctx->ckey, civ);
		AES_set_iv((Key_AES *)ctx->skey, siv);
		break;
	case OBJ_CRYALGO_3DESCBC:
		ctx->ckey = (Key *)DES3key_new_c(key_size, ckey);
		ctx->skey = (Key *)DES3key_new_c(key_size, skey);
		DES3_set_iv((Key_3DES *)ctx->ckey, civ);
		DES3_set_iv((Key_3DES *)ctx->skey, siv);
		break;
	case OBJ_CRYALGO_DESCBC:
		ctx->ckey = (Key *)DESkey_new(key_size, ckey);
		ctx->skey = (Key *)DESkey_new(key_size, skey);
		DES_set_iv((Key_DES *)ctx->ckey, civ);
		DES_set_iv((Key_DES *)ctx->skey, siv);
		break;
	case OBJ_CRYALGO_RC2CBC:
		ctx->ckey = (Key *)RC2key_new(key_size, ckey);
		ctx->skey = (Key *)RC2key_new(key_size, skey);
		RC2_set_iv((Key_RC2 *)(ctx->ckey), civ);
		RC2_set_iv((Key_RC2 *)(ctx->skey), siv);
		break;
#ifdef HAVE_ARC4
	case OBJ_CRYALGO_RC4:
		ctx->ckey = (Key *)RC4key_new(key_size, ckey);
		ctx->skey = (Key *)RC4key_new(key_size, skey);
		break;
#endif
	case 0: /* NULL Cipher */
		ctx->ckey = NULLkey_new();
		ctx->skey = NULLkey_new();
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_SSLHS,ERR_PT_SSLHS_KEY,NULL);
		ctx->errnum = SSL_AD_ILLEGAL_PARAMETER | (SSL_AL_FATAL<<8);
		return -1;
	}
	if((ctx->ckey==NULL)||(ctx->skey==NULL)){
		ctx->errnum = SSL_AD_CLOSE_NOTIFY | (SSL_AL_FATAL<<8);
		return -1;
	}

	return 0;
}


/*-----------------------------------------
  SSL Handshake (Calculate Hash)
-----------------------------------------*/
void SSL_hs_hashinit(SSLCTX *ctx){
	MD5Init(ctx->hsmsg_md5);
	SHA1init(ctx->hsmsg_sha1);
}

void SSL_hs_hashupdate(SSLCTX *ctx,unsigned char *in,int len){
	if(len<0) return;
	MD5Update(ctx->hsmsg_md5,in,len);
	SHA1update(ctx->hsmsg_sha1,in,len);
}

void SSL_hs_hashfinal(SSLCTX *ctx,unsigned char *md5,unsigned char *sha1){
	MD5Final(md5,ctx->hsmsg_md5);
	SHA1final(sha1,ctx->hsmsg_sha1);
}

