/* spkac_asn1.c - make struct SPKAC to DER */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 1998-2005
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_spkac.h>

/* cert_tool.c */
int set_digalgo_from_sigalgo(int algo);

/*-----------------------------------------
  Get certificate DER from SPKAC
-----------------------------------------*/
unsigned char *SPKAC_toDER(SPKAC *spk,Key *prv,unsigned char *buf,int *ret_len){
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=SPKAC_estimate_der_size(spk))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_SPKAC,ERR_PT_SPKACASN1,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}
	if(SPKAC_set_sigalgo(spk,prv)) goto error;

	if(SPKAC_DER_pkac(spk,ret,&i)) goto error;

	if(x509_set_signature(ret,prv,&spk->signature,&spk->siglen)) goto error;
	cp = ret+i;

	if(x509_DER_algoid(spk->signature_algo,NULL,cp,&j)) goto error;
	cp+=j; i+=j;

	ASN1_set_bitstring(0,spk->siglen,spk->signature,cp,&j);
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);
	return ret;
error:
	if(ret!=buf) free(ret);
	return NULL;
}

/*-----------------------------------------
  Get certificate data DER from SPKAC
-----------------------------------------*/
int SPKAC_set_sigalgo(SPKAC *spk,Key *prv){
	int kt;

	if((kt=set_digalgo_from_sigalgo(default_cert_sig_algo))<0)
		return -1;

	if((kt!=prv->key_type)&&(prv->key_type<=KEY_ECDSA_PRV)){
		OK_set_error(ERR_ST_UNMATCHEDPARAM,ERR_LC_SPKAC,ERR_PT_SPKACASN1+1,NULL);
		return -1;
	}
	spk->signature_algo = default_cert_sig_algo;
	return 0;
}

/*-----------------------------------------
  Get certificate data DER from SPKAC
-----------------------------------------*/
int SPKAC_DER_pkac(SPKAC *spk,unsigned char *ret,int *ret_len){
	unsigned char *cp;
	int i,j,err=-1;

	/* PublicKeyAndChallenge */
	if(x509_DER_pubkey(spk->pubkey,ret,&i)) goto done;
	cp = ret+i;

	if(ASN1_set_ia5(spk->challenge,cp,&j)) goto done;
	cp+=j; i+=j;

	ASN1_set_sequence(i,ret,ret_len);

	err=0;
done:
	return err;
}

/*-----------------------------------------
  estimate certificate DER size from SPKAC
-----------------------------------------*/
int SPKAC_estimate_der_size(SPKAC *spk){
	int ret;

	if(spk==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_SPKAC,ERR_PT_SPKACASN1+3,NULL);
		return -1;
	}

	/* PublicKeyAndChallenge */
	ret = 32 + strlen(spk->challenge);

	/* public key */
	switch(spk->pubkey_algo){
	case KEY_RSA_PUB:
		ret+=spk->pubkey->size+32;
		break;
	case KEY_DSA_PUB:
		ret+=DSApub_estimate_der_size((Pubkey_DSA*)spk->pubkey);
		break;
	case KEY_ECDSA_PUB:
		ret+=ECDSApub_estimate_der_size((Pubkey_ECDSA*)spk->pubkey);
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_SPKAC,ERR_PT_SPKACASN1+3,NULL);
		return -1;
	}

	/* signature len */
	/* actually, signature will be set in SPKAC_toDER.
	 * therefore, signature length might be depened on Private key length.
	 * if current certificate doesn't have signature information, just set
	 * enough big size of signature. (currently it's 2048bit)
	 */
	if((spk->signature==NULL)||(spk->siglen<=0))
		ret+=256+24;
	else
		ret+=spk->siglen+24;

	return ret;
}

