/* spkac.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 1998-2005
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_spkac.h>

/*-----------------------------------------
  make new struct SPKAC
-----------------------------------------*/
SPKAC *SPKAC_new(void){
	SPKAC	*ret;

	if((ret=(SPKAC*)malloc(sizeof(SPKAC)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_SPKAC,ERR_PT_SPKAC,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(SPKAC));
	ret->signature_algo = OBJ_SIG_NULL;
	ret->pubkey_algo = KEY_NULL;
	return ret;
}

/*-----------------------------------------
  free struct cert
-----------------------------------------*/
void SPKAC_free(SPKAC *spk){
	if(spk==NULL) return;
	if(spk->pubkey)	 Key_free(spk->pubkey);
	if(spk->challenge)	 free(spk->challenge);
	if(spk->signature)	 free(spk->signature);
	if(spk->der)		 free(spk->der);
	free(spk);
}

/*-----------------------------------------
  Duplicate struct Cert
-----------------------------------------*/
SPKAC *SPKAC_dup(SPKAC *src){
	SPKAC *ret=NULL;

	if(src==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_SPKAC,ERR_PT_SPKAC+2,NULL);
		return NULL;
	}

	if((ret=SPKAC_new())==NULL) goto error;
	ret->pubkey_algo    = src->pubkey_algo;
	ret->siglen         = src->siglen;
	ret->signature_algo = src->signature_algo;

	if((ret->pubkey=Key_dup(src->pubkey))==NULL) goto error;
	if ((ret->challenge = strdup(src->challenge)) == NULL) goto error;

	if((ret->signature=(unsigned char*)malloc(src->siglen))==NULL) goto error;
	memcpy(ret->signature,src->signature,src->siglen);

	if(src->der){
		if((ret->der=ASN1_dup(src->der))==NULL)
			goto error;
	}

	return ret;
error:
	OK_set_error(ERR_ST_MEMALLOC,ERR_LC_SPKAC,ERR_PT_SPKAC+2,NULL);
	SPKAC_free(ret);
	return NULL;
}


/*-----------------------------------------------
  SPKAC signature verify
  return 0  ... verify OK
  return err  ... verify Failed(err=number)
-----------------------------------------------*/
int SPKAC_verify(SPKAC *spk){
	int i,ret = 0;

	i = ASN1_vfy_sig(spk->pubkey,ASN1_next(spk->der),spk->signature,spk->signature_algo);
	if(i > 0) ret = X509_VFY_ERR_SIGNATURE;
	if(i==-2) ret = X509_VFY_ERR_UNKOWN_SIG_ALGO;
	if(i < 0) ret = X509_VFY_ERR_SYSTEMERR;

	return ret;
}

