/* sha3testnist.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <aicrypto/nrg_sha3.h>
#include "sha3test.h"
#include "sha3testnist.h"

/** input message */
struct tv nist_tv[6];

/*
 * Test vector for SHA3-224 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-224_1630.pdf
 */
static uint8_t sha3_224_expected[6][SHA3_224_DIGESTSIZE] = {
	/* 0-bit */
	{0x6B,0x4E,0x03,0x42,0x36,0x67,0xDB,0xB7,
	 0x3B,0x6E,0x15,0x45,0x4F,0x0E,0xB1,0xAB,
	 0xD4,0x59,0x7F,0x9A,0x1B,0x07,0x8E,0x3F,
	 0x5B,0x5A,0x6B,0xC7},
	/* 5-bit */
	{0xFF,0xBA,0xD5,0xDA,0x96,0xBA,0xD7,0x17,
	 0x89,0x33,0x02,0x06,0xDC,0x67,0x68,0xEC,
	 0xAE,0xB1,0xB3,0x2D,0xCA,0x6B,0x33,0x01,
	 0x48,0x96,0x74,0xAB},
	/* 30-bit */
	{0xD6,0x66,0xA5,0x14,0xCC,0x9D,0xBA,0x25,
	 0xAC,0x1B,0xA6,0x9E,0xD3,0x93,0x04,0x60,
	 0xDE,0xAA,0xC9,0x85,0x1B,0x5F,0x0B,0xAA,
	 0xB0,0x07,0xDF,0x3B},
	/* 1600-bit */
	{0x93,0x76,0x81,0x6A,0xBA,0x50,0x3F,0x72,
	 0xF9,0x6C,0xE7,0xEB,0x65,0xAC,0x09,0x5D,
	 0xEE,0xE3,0xBE,0x4B,0xF9,0xBB,0xC2,0xA1,
	 0xCB,0x7E,0x11,0xE0},
	/* 1605-bit */
	{0x22,0xD2,0xF7,0xBB,0x0B,0x17,0x3F,0xD8,
	 0xC1,0x96,0x86,0xF9,0x17,0x31,0x66,0xE3,
	 0xEE,0x62,0x73,0x80,0x47,0xD7,0xEA,0xDD,
	 0x69,0xEF,0xB2,0x28},
	/* 1630-bit */
	{0x4E,0x90,0x7B,0xB1,0x05,0x78,0x61,0xF2,
	 0x00,0xA5,0x99,0xE9,0xD4,0xF8,0x5B,0x02,
	 0xD8,0x84,0x53,0xBF,0x5B,0x8A,0xCE,0x9A,
	 0xC5,0x89,0x13,0x4C},
};

/*
 * Test vector for SHA3-256 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-256_1630.pdf
 */
static uint8_t sha3_256_expected[6][SHA3_256_DIGESTSIZE] = {
	/* 0-bit */
	{0xA7,0xFF,0xC6,0xF8,0xBF,0x1E,0xD7,0x66,
	 0x51,0xC1,0x47,0x56,0xA0,0x61,0xD6,0x62,
	 0xF5,0x80,0xFF,0x4D,0xE4,0x3B,0x49,0xFA,
	 0x82,0xD8,0x0A,0x4B,0x80,0xF8,0x43,0x4A},
	/* 5-bit */
	{0x7B,0x00,0x47,0xCF,0x5A,0x45,0x68,0x82,
	 0x36,0x3C,0xBF,0x0F,0xB0,0x53,0x22,0xCF,
	 0x65,0xF4,0xB7,0x05,0x9A,0x46,0x36,0x5E,
	 0x83,0x01,0x32,0xE3,0xB5,0xD9,0x57,0xAF},
	/* 30-bit */
	{0xC8,0x24,0x2F,0xEF,0x40,0x9E,0x5A,0xE9,
	 0xD1,0xF1,0xC8,0x57,0xAE,0x4D,0xC6,0x24,
	 0xB9,0x2B,0x19,0x80,0x9F,0x62,0xAA,0x8C,
	 0x07,0x41,0x1C,0x54,0xA0,0x78,0xB1,0xD0},
	/* 1600-bit */
	{0x79,0xF3,0x8A,0xDE,0xC5,0xC2,0x03,0x07,
	 0xA9,0x8E,0xF7,0x6E,0x83,0x24,0xAF,0xBF,
	 0xD4,0x6C,0xFD,0x81,0xB2,0x2E,0x39,0x73,
	 0xC6,0x5F,0xA1,0xBD,0x9D,0xE3,0x17,0x87},
	/* 1605-bit */
	{0x81,0xEE,0x76,0x9B,0xED,0x09,0x50,0x86,
	 0x2B,0x1D,0xDD,0xED,0x2E,0x84,0xAA,0xA6,
	 0xAB,0x7B,0xFD,0xD3,0xCE,0xAA,0x47,0x1B,
	 0xE3,0x11,0x63,0xD4,0x03,0x36,0x36,0x3C},
	/* 1630-bit */
	{0x52,0x86,0x0A,0xA3,0x01,0x21,0x4C,0x61,
	 0x0D,0x92,0x2A,0x6B,0x6C,0xAB,0x98,0x1C,
	 0xCD,0x06,0x01,0x2E,0x54,0xEF,0x68,0x9D,
	 0x74,0x40,0x21,0xE7,0x38,0xB9,0xED,0x20},
};

/**
 * Testcase for SHA3-384 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-384_1630.pdf
 */
static uint8_t sha3_384_expected[6][SHA3_384_DIGESTSIZE] = {
	/* 0-bit */
	{0x0C,0x63,0xA7,0x5B,0x84,0x5E,0x4F,0x7D,
	 0x01,0x10,0x7D,0x85,0x2E,0x4C,0x24,0x85,
	 0xC5,0x1A,0x50,0xAA,0xAA,0x94,0xFC,0x61,
	 0x99,0x5E,0x71,0xBB,0xEE,0x98,0x3A,0x2A,
	 0xC3,0x71,0x38,0x31,0x26,0x4A,0xDB,0x47,
	 0xFB,0x6B,0xD1,0xE0,0x58,0xD5,0xF0,0x04},
	/* 5-bit */
	{0x73,0x7C,0x9B,0x49,0x18,0x85,0xE9,0xBF,
	 0x74,0x28,0xE7,0x92,0x74,0x1A,0x7B,0xF8,
	 0xDC,0xA9,0x65,0x34,0x71,0xC3,0xE1,0x48,
	 0x47,0x3F,0x2C,0x23,0x6B,0x6A,0x0A,0x64,
	 0x55,0xEB,0x1D,0xCE,0x9F,0x77,0x9B,0x4B,
	 0x6B,0x23,0x7F,0xEF,0x17,0x1B,0x1C,0x64},
	/* 30-bit */
	{0x95,0x5B,0x4D,0xD1,0xBE,0x03,0x26,0x1B,
	 0xD7,0x6F,0x80,0x7A,0x7E,0xFD,0x43,0x24,
	 0x35,0xC4,0x17,0x36,0x28,0x11,0xB8,0xA5,
	 0x0C,0x56,0x4E,0x7E,0xE9,0x58,0x5E,0x1A,
	 0xC7,0x62,0x6D,0xDE,0x2F,0xDC,0x03,0x0F,
	 0x87,0x61,0x96,0xEA,0x26,0x7F,0x08,0xC3},
	/* 1600-bit */
	{0x18,0x81,0xDE,0x2C,0xA7,0xE4,0x1E,0xF9,
	 0x5D,0xC4,0x73,0x2B,0x8F,0x5F,0x00,0x2B,
	 0x18,0x9C,0xC1,0xE4,0x2B,0x74,0x16,0x8E,
	 0xD1,0x73,0x26,0x49,0xCE,0x1D,0xBC,0xDD,
	 0x76,0x19,0x7A,0x31,0xFD,0x55,0xEE,0x98,
	 0x9F,0x2D,0x70,0x50,0xDD,0x47,0x3E,0x8F},
	/* 1605-bit */
	{0xA3,0x1F,0xDB,0xD8,0xD5,0x76,0x55,0x1C,
	 0x21,0xFB,0x11,0x91,0xB5,0x4B,0xDA,0x65,
	 0xB6,0xC5,0xFE,0x97,0xF0,0xF4,0xA6,0x91,
	 0x03,0x42,0x4B,0x43,0xF7,0xFD,0xB8,0x35,
	 0x97,0x9F,0xDB,0xEA,0xE8,0xB3,0xFE,0x16,
	 0xCB,0x82,0xE5,0x87,0x38,0x1E,0xB6,0x24},
	/* 1630-bit */
	{0x34,0x85,0xD3,0xB2,0x80,0xBD,0x38,0x4C,
	 0xF4,0xA7,0x77,0x84,0x4E,0x94,0x67,0x81,
	 0x73,0x05,0x5D,0x1C,0xBC,0x40,0xC7,0xC2,
	 0xC3,0x83,0x3D,0x9E,0xF1,0x23,0x45,0x17,
	 0x2D,0x6F,0xCD,0x31,0x92,0x3B,0xB8,0x79,
	 0x5A,0xC8,0x18,0x47,0xD3,0xD8,0x85,0x5C},
};

/**
 * Testcase for SHA3-512 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA3-512_1630.pdf
 */
static uint8_t sha3_512_expected[6][SHA3_512_DIGESTSIZE]={
	/* 0-bit */
	{0xA6,0x9F,0x73,0xCC,0xA2,0x3A,0x9A,0xC5,
	 0xC8,0xB5,0x67,0xDC,0x18,0x5A,0x75,0x6E,
	 0x97,0xC9,0x82,0x16,0x4F,0xE2,0x58,0x59,
	 0xE0,0xD1,0xDC,0xC1,0x47,0x5C,0x80,0xA6,
	 0x15,0xB2,0x12,0x3A,0xF1,0xF5,0xF9,0x4C,
	 0x11,0xE3,0xE9,0x40,0x2C,0x3A,0xC5,0x58,
	 0xF5,0x00,0x19,0x9D,0x95,0xB6,0xD3,0xE3,
	 0x01,0x75,0x85,0x86,0x28,0x1D,0xCD,0x26},
	/* 5-bit */
	{0xA1,0x3E,0x01,0x49,0x41,0x14,0xC0,0x98,
	 0x00,0x62,0x2A,0x70,0x28,0x8C,0x43,0x21,
	 0x21,0xCE,0x70,0x03,0x9D,0x75,0x3C,0xAD,
	 0xD2,0xE0,0x06,0xE4,0xD9,0x61,0xCB,0x27,
	 0x54,0x4C,0x14,0x81,0xE5,0x81,0x4B,0xDC,
	 0xEB,0x53,0xBE,0x67,0x33,0xD5,0xE0,0x99,
	 0x79,0x5E,0x5E,0x81,0x91,0x8A,0xDD,0xB0,
	 0x58,0xE2,0x2A,0x9F,0x24,0x88,0x3F,0x37},
	/* 30-bit */
	{0x98,0x34,0xC0,0x5A,0x11,0xE1,0xC5,0xD3,
	 0xDA,0x9C,0x74,0x0E,0x1C,0x10,0x6D,0x9E,
	 0x59,0x0A,0x0E,0x53,0x0B,0x6F,0x6A,0xAA,
	 0x78,0x30,0x52,0x5D,0x07,0x5C,0xA5,0xDB,
	 0x1B,0xD8,0xA6,0xAA,0x98,0x1A,0x28,0x61,
	 0x3A,0xC3,0x34,0x93,0x4A,0x01,0x82,0x3C,
	 0xD4,0x5F,0x45,0xE4,0x9B,0x6D,0x7E,0x69,
	 0x17,0xF2,0xF1,0x67,0x78,0x06,0x7B,0xAB},
	/* 1600-bit */
	{0xE7,0x6D,0xFA,0xD2,0x20,0x84,0xA8,0xB1,
	 0x46,0x7F,0xCF,0x2F,0xFA,0x58,0x36,0x1B,
	 0xEC,0x76,0x28,0xED,0xF5,0xF3,0xFD,0xC0,
	 0xE4,0x80,0x5D,0xC4,0x8C,0xAE,0xEC,0xA8,
	 0x1B,0x7C,0x13,0xC3,0x0A,0xDF,0x52,0xA3,
	 0x65,0x95,0x84,0x73,0x9A,0x2D,0xF4,0x6B,
	 0xE5,0x89,0xC5,0x1C,0xA1,0xA4,0xA8,0x41,
	 0x6D,0xF6,0x54,0x5A,0x1C,0xE8,0xBA,0x00},
	/* 1605-bit */
	{0xFC,0x4A,0x16,0x7C,0xCB,0x31,0xA9,0x37,
	 0xD6,0x98,0xFD,0xE8,0x2B,0x04,0x34,0x8C,
	 0x95,0x39,0xB2,0x8F,0x0C,0x9D,0x3B,0x45,
	 0x05,0x70,0x9C,0x03,0x81,0x23,0x50,0xE4,
	 0x99,0x0E,0x96,0x22,0x97,0x4F,0x6E,0x57,
	 0x5C,0x47,0x86,0x1C,0x0D,0x2E,0x63,0x8C,
	 0xCF,0xC2,0x02,0x3C,0x36,0x5B,0xB6,0x0A,
	 0x93,0xF5,0x28,0x55,0x06,0x98,0x78,0x6B},
	/* 1630-bit */
	{0xCF,0x9A,0x30,0xAC,0x1F,0x1F,0x6A,0xC0,
	 0x91,0x6F,0x9F,0xEF,0x19,0x19,0xC5,0x95,
	 0xDE,0xBE,0x2E,0xE8,0x0C,0x85,0x42,0x12,
	 0x10,0xFD,0xF0,0x5F,0x1C,0x6A,0xF7,0x3A,
	 0xA9,0xCA,0xC8,0x81,0xD0,0xF9,0x1D,0xB6,
	 0xD0,0x34,0xA2,0xBB,0xAD,0xC1,0xCF,0x7F,
	 0xBC,0xB2,0xEC,0xFA,0x9D,0x19,0x1D,0x3A,
	 0x50,0x16,0xFB,0x3F,0xAD,0x87,0x09,0xC9},
};


/**
 * Testcase for SHAKE128 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE128_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE128_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE128_Msg1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE128_Msg1630.pdf
 */
static uint8_t shake128_expected[6][DEFAULT_SHAKE128_DIGESTSIZE]={
	/* 0-bit */
	{
0x7F,0x9C,0x2B,0xA4,0xE8,0x8F,0x82,0x7D,0x61,0x60,0x45,0x50,0x76,0x05,0x85,0x3E,
0xD7,0x3B,0x80,0x93,0xF6,0xEF,0xBC,0x88,0xEB,0x1A,0x6E,0xAC,0xFA,0x66,0xEF,0x26,
0x3C,0xB1,0xEE,0xA9,0x88,0x00,0x4B,0x93,0x10,0x3C,0xFB,0x0A,0xEE,0xFD,0x2A,0x68,
0x6E,0x01,0xFA,0x4A,0x58,0xE8,0xA3,0x63,0x9C,0xA8,0xA1,0xE3,0xF9,0xAE,0x57,0xE2,
0x35,0xB8,0xCC,0x87,0x3C,0x23,0xDC,0x62,0xB8,0xD2,0x60,0x16,0x9A,0xFA,0x2F,0x75,
0xAB,0x91,0x6A,0x58,0xD9,0x74,0x91,0x88,0x35,0xD2,0x5E,0x6A,0x43,0x50,0x85,0xB2,
0xBA,0xDF,0xD6,0xDF,0xAA,0xC3,0x59,0xA5,0xEF,0xBB,0x7B,0xCC,0x4B,0x59,0xD5,0x38,
0xDF,0x9A,0x04,0x30,0x2E,0x10,0xC8,0xBC,0x1C,0xBF,0x1A,0x0B,0x3A,0x51,0x20,0xEA,
0x17,0xCD,0xA7,0xCF,0xAD,0x76,0x5F,0x56,0x23,0x47,0x4D,0x36,0x8C,0xCC,0xA8,0xAF,
0x00,0x07,0xCD,0x9F,0x5E,0x4C,0x84,0x9F,0x16,0x7A,0x58,0x0B,0x14,0xAA,0xBD,0xEF,
0xAE,0xE7,0xEE,0xF4,0x7C,0xB0,0xFC,0xA9,0x76,0x7B,0xE1,0xFD,0xA6,0x94,0x19,0xDF,
0xB9,0x27,0xE9,0xDF,0x07,0x34,0x8B,0x19,0x66,0x91,0xAB,0xAE,0xB5,0x80,0xB3,0x2D,
0xEF,0x58,0x53,0x8B,0x8D,0x23,0xF8,0x77,0x32,0xEA,0x63,0xB0,0x2B,0x4F,0xA0,0xF4,
0x87,0x33,0x60,0xE2,0x84,0x19,0x28,0xCD,0x60,0xDD,0x4C,0xEE,0x8C,0xC0,0xD4,0xC9,
0x22,0xA9,0x61,0x88,0xD0,0x32,0x67,0x5C,0x8A,0xC8,0x50,0x93,0x3C,0x7A,0xFF,0x15,
0x33,0xB9,0x4C,0x83,0x4A,0xDB,0xB6,0x9C,0x61,0x15,0xBA,0xD4,0x69,0x2D,0x86,0x19,
0xF9,0x0B,0x0C,0xDF,0x8A,0x7B,0x9C,0x26,0x40,0x29,0xAC,0x18,0x5B,0x70,0xB8,0x3F,
0x28,0x01,0xF2,0xF4,0xB3,0xF7,0x0C,0x59,0x3E,0xA3,0xAE,0xEB,0x61,0x3A,0x7F,0x1B,
0x1D,0xE3,0x3F,0xD7,0x50,0x81,0xF5,0x92,0x30,0x5F,0x2E,0x45,0x26,0xED,0xC0,0x96,
0x31,0xB1,0x09,0x58,0xF4,0x64,0xD8,0x89,0xF3,0x1B,0xA0,0x10,0x25,0x0F,0xDA,0x7F,
0x13,0x68,0xEC,0x29,0x67,0xFC,0x84,0xEF,0x2A,0xE9,0xAF,0xF2,0x68,0xE0,0xB1,0x70,
0x0A,0xFF,0xC6,0x82,0x0B,0x52,0x3A,0x3D,0x91,0x71,0x35,0xF2,0xDF,0xF2,0xEE,0x06,
0xBF,0xE7,0x2B,0x31,0x24,0x72,0x1D,0x4A,0x26,0xC0,0x4E,0x53,0xA7,0x5E,0x30,0xE7,
0x3A,0x7A,0x9C,0x4A,0x95,0xD9,0x1C,0x55,0xD4,0x95,0xE9,0xF5,0x1D,0xD0,0xB5,0xE9,
0xD8,0x3C,0x6D,0x5E,0x8C,0xE8,0x03,0xAA,0x62,0xB8,0xD6,0x54,0xDB,0x53,0xD0,0x9B,
0x8D,0xCF,0xF2,0x73,0xCD,0xFE,0xB5,0x73,0xFA,0xD8,0xBC,0xD4,0x55,0x78,0xBE,0xC2,
0xE7,0x70,0xD0,0x1E,0xFD,0xE8,0x6E,0x72,0x1A,0x3F,0x7C,0x6C,0xCE,0x27,0x5D,0xAB,
0xE6,0xE2,0x14,0x3F,0x1A,0xF1,0x8D,0xA7,0xEF,0xDD,0xC4,0xC7,0xB7,0x0B,0x5E,0x34,
0x5D,0xB9,0x3C,0xC9,0x36,0xBE,0xA3,0x23,0x49,0x1C,0xCB,0x38,0xA3,0x88,0xF5,0x46,
0xA9,0xFF,0x00,0xDD,0x4E,0x13,0x00,0xB9,0xB2,0x15,0x3D,0x20,0x41,0xD2,0x05,0xB4,
0x43,0xE4,0x1B,0x45,0xA6,0x53,0xF2,0xA5,0xC4,0x49,0x2C,0x1A,0xDD,0x54,0x45,0x12,
0xDD,0xA2,0x52,0x98,0x33,0x46,0x2B,0x71,0xA4,0x1A,0x45,0xBE,0x97,0x29,0x0B,0x6F
	},
	/* 5-bit */
	{
0x2E,0x0A,0xBF,0xBA,0x83,0xE6,0x72,0x0B,0xFB,0xC2,0x25,0xFF,0x6B,0x7A,0xB9,0xFF,
0xCE,0x58,0xBA,0x02,0x7E,0xE3,0xD8,0x98,0x76,0x4F,0xEF,0x28,0x7D,0xDE,0xCC,0xCA,
0x3E,0x6E,0x59,0x98,0x41,0x1E,0x7D,0xDB,0x32,0xF6,0x75,0x38,0xF5,0x00,0xB1,0x8C,
0x8C,0x97,0xC4,0x52,0xC3,0x70,0xEA,0x2C,0xF0,0xAF,0xCA,0x3E,0x05,0xDE,0x7E,0x4D,
0xE2,0x7F,0xA4,0x41,0xA9,0xCB,0x34,0xFD,0x17,0xC9,0x78,0xB4,0x2D,0x5B,0x7E,0x7F,
0x9A,0xB1,0x8F,0xFE,0xFF,0xC3,0xC5,0xAC,0x2F,0x3A,0x45,0x5E,0xEB,0xFD,0xC7,0x6C,
0xEA,0xEB,0x0A,0x2C,0xCA,0x22,0xEE,0xF6,0xE6,0x37,0xF4,0xCA,0xBE,0x5C,0x51,0xDE,
0xD2,0xE3,0xFA,0xD8,0xB9,0x52,0x70,0xA3,0x21,0x84,0x56,0x64,0xF1,0x07,0xD1,0x64,
0x96,0xBB,0x7A,0xBF,0xBE,0x75,0x04,0xB6,0xED,0xE2,0xE8,0x9E,0x4B,0x99,0x6F,0xB5,
0x8E,0xFD,0xC4,0x18,0x1F,0x91,0x63,0x38,0x1C,0xBE,0x7B,0xC0,0x06,0xA7,0xA2,0x05,
0x98,0x9C,0x52,0x6C,0xD1,0xBD,0x68,0x98,0x36,0x93,0xB4,0xBD,0xC5,0x37,0x28,0xB2,
0x41,0xC1,0xCF,0xF4,0x2B,0xB6,0x11,0x50,0x2C,0x35,0x20,0x5C,0xAB,0xB2,0x88,0x75,
0x56,0x55,0xD6,0x20,0xC6,0x79,0x94,0xF0,0x64,0x51,0x18,0x7F,0x6F,0xD1,0x7E,0x04,
0x66,0x82,0xBA,0x12,0x86,0x06,0x3F,0xF8,0x8F,0xE2,0x50,0x8D,0x1F,0xCA,0xF9,0x03,
0x5A,0x12,0x31,0xAD,0x41,0x50,0xA9,0xC9,0xB2,0x4C,0x9B,0x2D,0x66,0xB2,0xAD,0x1B,
0xDE,0x0B,0xD0,0xBB,0xCB,0x8B,0xE0,0x5B,0x83,0x52,0x29,0xEF,0x79,0x19,0x73,0x73,
0x23,0x42,0x44,0x01,0xE1,0xD8,0x37,0xB6,0x6E,0xB4,0xE6,0x30,0xFF,0x1D,0xE7,0x0C,
0xB3,0x17,0xC2,0xBA,0xCB,0x08,0x00,0x1D,0x34,0x77,0xB7,0xA7,0x0A,0x57,0x6D,0x20,
0x86,0x90,0x33,0x58,0x9D,0x85,0xA0,0x1D,0xDB,0x2B,0x66,0x46,0xC0,0x43,0xB5,0x9F,
0xC0,0x11,0x31,0x1D,0xA6,0x66,0xFA,0x5A,0xD1,0xD6,0x38,0x7F,0xA9,0xBC,0x40,0x15,
0xA3,0x8A,0x51,0xD1,0xDA,0x1E,0xA6,0x1D,0x64,0x8D,0xC8,0xE3,0x9A,0x88,0xB9,0xD6,
0x22,0xBD,0xE2,0x07,0xFD,0xAB,0xC6,0xF2,0x82,0x7A,0x88,0x0C,0x33,0x0B,0xBF,0x6D,
0xF7,0x33,0x77,0x4B,0x65,0x3E,0x57,0x30,0x5D,0x78,0xDC,0xE1,0x12,0xF1,0x0A,0x2C,
0x71,0xF4,0xCD,0xAD,0x92,0xED,0x11,0x3E,0x1C,0xEA,0x63,0xB9,0x19,0x25,0xED,0x28,
0x19,0x1E,0x6D,0xBB,0xB5,0xAA,0x5A,0x2A,0xFD,0xA5,0x1F,0xC0,0x5A,0x3A,0xF5,0x25,
0x8B,0x87,0x66,0x52,0x43,0x55,0x0F,0x28,0x94,0x8A,0xE2,0xB8,0xBE,0xB6,0xBC,0x9C,
0x77,0x0B,0x35,0xF0,0x67,0xEA,0xA6,0x41,0xEF,0xE6,0x5B,0x1A,0x44,0x90,0x9D,0x1B,
0x14,0x9F,0x97,0xEE,0xA6,0x01,0x39,0x1C,0x60,0x9E,0xC8,0x1D,0x19,0x30,0xF5,0x7C,
0x18,0xA4,0xE0,0xFA,0xB4,0x91,0xD1,0xCA,0xDF,0xD5,0x04,0x83,0x44,0x9E,0xDC,0x0F,
0x07,0xFF,0xB2,0x4D,0x2C,0x6F,0x9A,0x9A,0x3B,0xFF,0x39,0xAE,0x3D,0x57,0xF5,0x60,
0x65,0x4D,0x7D,0x75,0xC9,0x08,0xAB,0xE6,0x25,0x64,0x75,0x3E,0xAC,0x39,0xD7,0x50,
0x3D,0xA6,0xD3,0x7C,0x2E,0x32,0xE1,0xAF,0x3B,0x8A,0xEC,0x8A,0xE3,0x06,0x9C,0xD9
	},
	/* 30-bit */
	{
0x6D,0x5D,0x39,0xC5,0x5F,0x3C,0xCA,0x56,0x7F,0xEA,0xF4,0x22,0xDC,0x64,0xBA,0x17,
0x40,0x1D,0x07,0x75,0x6D,0x78,0xB0,0xFA,0x3D,0x54,0x6D,0x66,0xAF,0xC2,0x76,0x71,
0xE0,0x01,0x06,0x85,0xFC,0x69,0xA7,0xEC,0x3C,0x53,0x67,0xB8,0xFA,0x5F,0xDA,0x39,
0xD5,0x7C,0xE5,0x3F,0x15,0x3F,0xA4,0x03,0x1D,0x27,0x72,0x06,0x77,0x0A,0xEC,0x6B,
0x2D,0xDF,0x16,0xAE,0xFA,0xB6,0x69,0x11,0x0D,0x6E,0x4A,0x29,0x6A,0x14,0xFB,0x14,
0x86,0xB0,0x84,0x6B,0x69,0x05,0x43,0xE4,0x05,0x7F,0x7F,0x42,0xAA,0x8C,0x0E,0x6A,
0x5A,0x56,0xB6,0x0B,0x68,0x8D,0x55,0xA1,0x96,0xDF,0x6F,0x39,0x76,0xE3,0x06,0x88,
0xCB,0xB6,0xAF,0xD4,0x85,0x25,0xD7,0x64,0x90,0x35,0x7F,0x3F,0xD8,0x97,0xBA,0xFC,
0x87,0x36,0xD9,0x07,0xB9,0xBA,0xC8,0x16,0x59,0x1F,0xC2,0x4E,0x79,0x36,0x0B,0xE3,
0xA7,0xFF,0xA6,0x29,0x82,0xC4,0x5A,0xBB,0x0E,0x58,0x4C,0x07,0xEC,0x93,0xA1,0x95,
0x30,0x50,0x9D,0x9F,0x81,0x62,0x15,0xD7,0x27,0x7B,0xB9,0x99,0x43,0x7C,0x82,0x14,
0x50,0xF0,0x75,0x92,0x81,0xCD,0x8E,0x16,0xA3,0x48,0x3E,0x3C,0xC7,0x52,0x09,0x1B,
0x7A,0xAE,0x92,0x90,0x9D,0x2F,0x50,0x1E,0xF7,0xDC,0xE9,0x89,0x75,0x98,0x91,0xB3,
0x37,0x7C,0xEA,0xB4,0x93,0xFF,0xE4,0x96,0x01,0x0A,0x0C,0x7E,0x51,0x95,0x99,0x94,
0xF5,0x6F,0x56,0x5E,0x63,0x3A,0xF6,0x09,0x3A,0xC6,0xE1,0xE0,0xF0,0x04,0x88,0x71,
0xEC,0x47,0x78,0xF4,0x8E,0xF8,0xBD,0x5B,0xCB,0x80,0xEA,0x7D,0xF9,0xFF,0x47,0x11,
0xC8,0x1E,0x24,0xC0,0x22,0x1C,0x2A,0xD9,0x74,0x4F,0xBA,0x79,0x35,0xEA,0xEC,0xA1,
0x14,0x22,0x4F,0xD1,0x08,0xEF,0xC5,0xAC,0x74,0xC6,0x62,0x52,0x08,0x92,0x75,0xB4,
0x27,0x76,0x73,0x70,0x8C,0x4A,0xF9,0x2F,0x88,0x13,0xB1,0x93,0x59,0x9F,0xD6,0x4B,
0xD7,0x48,0x4F,0x2E,0x5E,0xC3,0x69,0xE3,0x64,0x64,0x99,0x76,0x8E,0x58,0x1D,0xD0,
0x53,0xAA,0x48,0x14,0xD8,0xBF,0x1A,0xCF,0xF5,0xFD,0x77,0x45,0x19,0xA7,0x49,0xBE,
0x66,0x75,0x47,0x41,0xEB,0xC5,0x36,0x22,0x12,0xA9,0xFE,0xA8,0xA8,0x14,0xE9,0xE0,
0x10,0xBC,0x27,0x20,0xB3,0xB7,0xD9,0x4F,0xAB,0x74,0xBC,0x7F,0x92,0x3E,0x10,0x72,
0xB8,0xA5,0xDD,0xDD,0xA8,0x3B,0xA0,0x15,0x7D,0x8C,0xBA,0x55,0xC1,0x92,0xDF,0x69,
0x65,0xCB,0x7D,0xBA,0x46,0xA3,0x34,0x0D,0xF8,0xC3,0xFA,0x89,0xC7,0xC4,0xDB,0x53,
0x9D,0x38,0xDC,0x40,0x6F,0x1D,0x2C,0xF5,0x4E,0x59,0x05,0x58,0x0B,0x44,0x04,0xBF,
0xD7,0xB3,0x71,0x95,0x61,0xC5,0xA5,0x9D,0x5D,0xFD,0xB1,0xBF,0x93,0xDF,0x13,0x82,
0x52,0x25,0xED,0xCC,0xE0,0xFA,0x7D,0x87,0xEF,0xCD,0x23,0x9F,0xEB,0x49,0xFC,0x9E,
0x2D,0xE9,0xD8,0x28,0xFE,0xEB,0x1F,0x2C,0xF5,0x79,0xB9,0x5D,0xD0,0x50,0xAB,0x2C,
0xA4,0x71,0x05,0xA8,0xD3,0x0F,0x3F,0xD2,0xA1,0x15,0x4C,0x15,0xF8,0x7F,0xB3,0x7B,
0x2C,0x71,0x56,0xBD,0x7F,0x3C,0xF2,0xB7,0x45,0xC9,0x12,0xA4,0x0B,0xC1,0xB5,0x59,
0xB6,0x56,0xE3,0xE9,0x03,0xCC,0x57,0x33,0xE8,0x6B,0xA1,0x5D,0xFE,0xF7,0x06,0x78
	},
	/* 1600-bit */
	{
0x13,0x1A,0xB8,0xD2,0xB5,0x94,0x94,0x6B,0x9C,0x81,0x33,0x3F,0x9B,0xB6,0xE0,0xCE,
0x75,0xC3,0xB9,0x31,0x04,0xFA,0x34,0x69,0xD3,0x91,0x74,0x57,0x38,0x5D,0xA0,0x37,
0xCF,0x23,0x2E,0xF7,0x16,0x4A,0x6D,0x1E,0xB4,0x48,0xC8,0x90,0x81,0x86,0xAD,0x85,
0x2D,0x3F,0x85,0xA5,0xCF,0x28,0xDA,0x1A,0xB6,0xFE,0x34,0x38,0x17,0x19,0x78,0x46,
0x7F,0x1C,0x05,0xD5,0x8C,0x7E,0xF3,0x8C,0x28,0x4C,0x41,0xF6,0xC2,0x22,0x1A,0x76,
0xF1,0x2A,0xB1,0xC0,0x40,0x82,0x66,0x02,0x50,0x80,0x22,0x94,0xFB,0x87,0x18,0x02,
0x13,0xFD,0xEF,0x5B,0x0E,0xCB,0x7D,0xF5,0x0C,0xA1,0xF8,0x55,0x5B,0xE1,0x4D,0x32,
0xE1,0x0F,0x6E,0xDC,0xDE,0x89,0x2C,0x09,0x42,0x4B,0x29,0xF5,0x97,0xAF,0xC2,0x70,
0xC9,0x04,0x55,0x6B,0xFC,0xB4,0x7A,0x7D,0x40,0x77,0x8D,0x39,0x09,0x23,0x64,0x2B,
0x3C,0xBD,0x05,0x79,0xE6,0x09,0x08,0xD5,0xA0,0x00,0xC1,0xD0,0x8B,0x98,0xEF,0x93,
0x3F,0x80,0x64,0x45,0xBF,0x87,0xF8,0xB0,0x09,0xBA,0x9E,0x94,0xF7,0x26,0x61,0x22,
0xED,0x7A,0xC2,0x4E,0x5E,0x26,0x6C,0x42,0xA8,0x2F,0xA1,0xBB,0xEF,0xB7,0xB8,0xDB,
0x00,0x66,0xE1,0x6A,0x85,0xE0,0x49,0x3F,0x07,0xDF,0x48,0x09,0xAE,0xC0,0x84,0xA5,
0x93,0x74,0x8A,0xC3,0xDD,0xE5,0xA6,0xD7,0xAA,0xE1,0xE8,0xB6,0xE5,0x35,0x2B,0x2D,
0x71,0xEF,0xBB,0x47,0xD4,0xCA,0xEE,0xD5,0xE6,0xD6,0x33,0x80,0x5D,0x2D,0x32,0x3E,
0x6F,0xD8,0x1B,0x46,0x84,0xB9,0x3A,0x26,0x77,0xD4,0x5E,0x74,0x21,0xC2,0xC6,0xAE,
0xA2,0x59,0xB8,0x55,0xA6,0x98,0xFD,0x7D,0x13,0x47,0x7A,0x1F,0xE5,0x3E,0x5A,0x4A,
0x61,0x97,0xDB,0xEC,0x5C,0xE9,0x5F,0x50,0x5B,0x52,0x0B,0xCD,0x95,0x70,0xC4,0xA8,
0x26,0x5A,0x7E,0x01,0xF8,0x9C,0x0C,0x00,0x2C,0x59,0xBF,0xEC,0x6C,0xD4,0xA5,0xC1,
0x09,0x25,0x89,0x53,0xEE,0x5E,0xE7,0x0C,0xD5,0x77,0xEE,0x21,0x7A,0xF2,0x1F,0xA7,
0x01,0x78,0xF0,0x94,0x6C,0x9B,0xF6,0xCA,0x87,0x51,0x79,0x34,0x79,0xF6,0xB5,0x37,
0x73,0x7E,0x40,0xB6,0xED,0x28,0x51,0x1D,0x8A,0x2D,0x7E,0x73,0xEB,0x75,0xF8,0xDA,
0xAC,0x91,0x2F,0xF9,0x06,0xE0,0xAB,0x95,0x5B,0x08,0x3B,0xAC,0x45,0xA8,0xE5,0xE9,
0xB7,0x44,0xC8,0x50,0x6F,0x37,0xE9,0xB4,0xE7,0x49,0xA1,0x84,0xB3,0x0F,0x43,0xEB,
0x18,0x8D,0x85,0x5F,0x1B,0x70,0xD7,0x1F,0xF3,0xE5,0x0C,0x53,0x7A,0xC1,0xB0,0xF8,
0x97,0x4F,0x0F,0xE1,0xA6,0xAD,0x29,0x5B,0xA4,0x2F,0x6A,0xEC,0x74,0xD1,0x23,0xA7,
0xAB,0xED,0xDE,0x6E,0x2C,0x07,0x11,0xCA,0xB3,0x6B,0xE5,0xAC,0xB1,0xA5,0xA1,0x1A,
0x4B,0x1D,0xB0,0x8B,0xA6,0x98,0x2E,0xFC,0xCD,0x71,0x69,0x29,0xA7,0x74,0x1C,0xFC,
0x63,0xAA,0x44,0x35,0xE0,0xB6,0x9A,0x90,0x63,0xE8,0x80,0x79,0x5C,0x3D,0xC5,0xEF,
0x32,0x72,0xE1,0x1C,0x49,0x7A,0x91,0xAC,0xF6,0x99,0xFE,0xFE,0xE2,0x06,0x22,0x7A,
0x44,0xC9,0xFB,0x35,0x9F,0xD5,0x6A,0xC0,0xA9,0xA7,0x5A,0x74,0x3C,0xFF,0x68,0x62,
0xF1,0x7D,0x72,0x59,0xAB,0x07,0x52,0x16,0xC0,0x69,0x95,0x11,0x64,0x3B,0x64,0x39
	},
	/* 1605-bit */
	{
0x4A,0xC3,0x8E,0xBD,0x16,0x78,0xB4,0xA4,0x52,0x79,0x2C,0x56,0x73,0xF9,0x77,0x7D,
0x36,0xB5,0x54,0x51,0xAA,0xAE,0x24,0x24,0x92,0x49,0x42,0xD3,0x18,0xA2,0xF6,0xF5,
0x1B,0xBC,0x83,0x7D,0xCC,0x70,0x22,0xC5,0x40,0x3B,0x69,0xD2,0x9A,0xC9,0x9A,0x74,
0x5F,0x06,0xD0,0x6F,0x2A,0x41,0xB0,0xCC,0x24,0x3C,0xD2,0x70,0xFA,0x44,0xD4,0x30,
0x65,0xAF,0x00,0xD2,0xAD,0x35,0x8B,0xD5,0xA5,0xD0,0x6D,0x33,0x1B,0xC2,0x30,0xCD,
0x8D,0xDA,0x46,0x55,0x62,0x8F,0x91,0x02,0x71,0x1A,0xDA,0xFB,0x76,0x36,0xC1,0x60,
0xB2,0xD2,0x5E,0xC6,0x23,0x5A,0x2F,0xE0,0xF3,0x73,0x94,0xD8,0x7F,0xC5,0xFF,0xD7,
0xDB,0xF1,0x99,0x3E,0x55,0x8A,0xEB,0xEA,0x6C,0x61,0xE9,0x07,0x18,0x8C,0x61,0xF5,
0xFC,0xDE,0x27,0x8E,0x26,0x4F,0x95,0x8F,0xFD,0x7B,0x33,0x82,0xDC,0x10,0x13,0x9B,
0x62,0x5E,0x12,0x41,0xAB,0x5B,0xBC,0x2A,0x1F,0xBC,0xAC,0x31,0xA3,0x35,0xCF,0xC7,
0xB2,0x0E,0x42,0x77,0x12,0x24,0x6C,0xBB,0x55,0x23,0x22,0x59,0xA7,0xEF,0x16,0x02,
0xBD,0x56,0xF6,0x56,0x7D,0x66,0x94,0x2D,0x4A,0x71,0x49,0xF4,0x22,0x22,0x10,0xB0,
0x74,0xEA,0x54,0x15,0x4B,0x38,0xE8,0xFD,0xFA,0x0D,0xCF,0x4F,0xA3,0xEC,0xD2,0x15,
0x4E,0x83,0x18,0xA6,0x57,0x8B,0x53,0x5D,0xBC,0xFC,0x21,0x7A,0x3C,0xAB,0x52,0x53,
0x29,0x65,0x84,0x6F,0x89,0x78,0x14,0x57,0x02,0x55,0x63,0xE2,0xDC,0x15,0xCC,0x3A,
0xF9,0x02,0xBA,0x2A,0xD2,0x80,0xFF,0xBB,0xBF,0xA4,0xC5,0x2B,0x60,0xFA,0x41,0xBA,
0xC2,0x1F,0x4A,0xB2,0x35,0x36,0x26,0x81,0x19,0xFC,0x98,0xCD,0x98,0x2D,0xA5,0xCD,
0x5D,0xA2,0x1E,0x1B,0x56,0x92,0xD4,0x71,0x05,0xDE,0x9F,0x1E,0x01,0x32,0xC6,0xFE,
0x31,0x5D,0x67,0xFA,0x46,0x49,0x97,0xC2,0xAB,0x55,0x33,0xC7,0x9F,0x98,0xE6,0xE6,
0x4F,0xF8,0x08,0x02,0xA7,0xFE,0x96,0xCA,0x04,0xA8,0x1F,0x88,0x55,0x27,0x37,0x0A,
0x22,0x06,0xB1,0x0B,0x39,0x36,0xDD,0x81,0xB8,0x24,0x63,0x53,0xF4,0xCD,0x90,0x51,
0x10,0x89,0x26,0x8D,0x74,0x4F,0x21,0x0A,0xC6,0x89,0xD4,0x9D,0x28,0x75,0x05,0x4A,
0x72,0x7B,0x60,0x4D,0x13,0xD2,0x69,0xB3,0x71,0x90,0xD4,0x27,0xC7,0xD1,0x5C,0xCC,
0xDC,0xD7,0x87,0x0E,0x0B,0x8A,0xDB,0xEB,0x97,0x71,0x11,0xA9,0xBC,0xF7,0x78,0x1A,
0x16,0x13,0x56,0xA5,0x94,0x1C,0x79,0x99,0x07,0xEF,0x9D,0x3B,0x1A,0x44,0x1F,0x09,
0x51,0x5F,0x28,0x31,0xC4,0xFA,0xFD,0xE3,0xDC,0x7C,0x1E,0x9B,0x5A,0xA5,0x7D,0x3E,
0x83,0xCD,0x67,0x34,0xDA,0x3D,0x8B,0x9E,0xF3,0xFC,0x44,0x88,0x05,0xEA,0x29,0xC9,
0x9C,0xBA,0x6B,0x35,0x2B,0xCA,0xBE,0x2F,0xD9,0x70,0xAE,0x95,0x80,0xD2,0xBF,0x25,
0x15,0x2B,0x96,0x0E,0x6B,0x80,0x6D,0x87,0xD7,0xD0,0x60,0x8B,0x24,0x7F,0x61,0x08,
0x9E,0x29,0x86,0x92,0xC2,0x7F,0x19,0xC5,0x2D,0x03,0xEB,0xE3,0x95,0xA3,0x68,0x06,
0xAD,0x54,0x0B,0xEC,0x2D,0x04,0x6C,0x18,0xE3,0x55,0xFA,0xF8,0x31,0x3D,0x2E,0xF8,
0x99,0x5E,0xE6,0xAA,0xE4,0x25,0x68,0xF3,0x14,0x93,0x3E,0x3A,0x21,0xE5,0xBE,0x40
	},
	/* 1630-bit */
	{
0x89,0x84,0x6D,0xC7,0x76,0xAC,0x0F,0x01,0x45,0x72,0xEA,0x79,0xF5,0x60,0x77,0x34,
0x51,0x00,0x29,0x38,0x24,0x8E,0x68,0x82,0x56,0x9A,0xC3,0x2A,0xEA,0xB1,0x91,0xFC,
0xAC,0xDE,0x68,0xEB,0x07,0x55,0x75,0x39,0xC4,0x84,0x5F,0xB4,0x44,0x10,0x8E,0x6E,
0x05,0x45,0xE7,0x31,0xFC,0xCA,0x2D,0x4F,0x67,0xA3,0xBF,0xD4,0x1C,0xFF,0x3E,0xAF,
0x35,0xEE,0xFB,0x53,0x44,0x11,0x77,0x96,0x5B,0xB5,0x16,0x95,0x0C,0xF5,0xDC,0xB2,
0xAA,0xFC,0xBB,0xC6,0x30,0x0E,0x8E,0xEF,0xD9,0xBC,0xD0,0xE5,0xF3,0x2D,0x1A,0x4E,
0x87,0x2E,0x0F,0x1D,0xBD,0x8F,0x8E,0x00,0xCB,0xB8,0x78,0x69,0x8C,0x58,0x83,0xE3,
0xCA,0x18,0x4B,0x94,0x90,0x38,0x9E,0x46,0x00,0x2C,0x08,0xA0,0xB1,0x6B,0x05,0xA3,
0x6B,0x2C,0xB5,0xA1,0xCA,0xE0,0x8E,0x11,0xAD,0x97,0x2F,0xD2,0x4A,0xF7,0x01,0x01,
0xCE,0x47,0x46,0xC8,0x4F,0x16,0x71,0x87,0x7F,0x0D,0xF6,0xC4,0x15,0xD1,0x67,0x0F,
0xF4,0x0B,0x8D,0xDE,0xDD,0x89,0xCC,0x3E,0x65,0x6D,0xB9,0x05,0x80,0x49,0xD6,0x09,
0xB6,0x78,0x4C,0xC9,0xD0,0x5E,0x60,0xCC,0x6A,0xC9,0xC8,0x19,0x49,0x93,0xBA,0x29,
0x15,0x8F,0xD4,0xDB,0x8C,0xF2,0x25,0xE9,0x57,0x4F,0x18,0xA7,0x7F,0x66,0xEC,0x10,
0x52,0xBF,0x17,0x99,0x3B,0xDA,0x20,0x6A,0x17,0x73,0x7D,0x78,0x5B,0xD4,0xC1,0x8C,
0xEE,0x4C,0x76,0xAA,0x57,0x35,0xA5,0x22,0x3F,0x3C,0x55,0xE7,0x9D,0xAE,0xC1,0x3D,
0x4B,0xF6,0x0F,0x15,0x62,0xE0,0xAD,0x0F,0xA3,0xB5,0x58,0xEC,0xCF,0xA8,0xAB,0x3E,
0xEF,0x61,0x47,0x4D,0x57,0x6E,0x8C,0xAF,0x4C,0x11,0xE4,0xDE,0x5C,0xCB,0x36,0xD7,
0xDF,0x7D,0x89,0x2C,0x1F,0xCA,0x20,0x17,0xBE,0x8B,0xBD,0xA5,0xA4,0x71,0x95,0x44,
0x8C,0xC6,0x7A,0x07,0x8E,0x62,0x8A,0x2E,0xF7,0x63,0xFF,0xE1,0xDC,0x9D,0x9D,0x6F,
0xF7,0x8E,0x68,0x96,0x1C,0x33,0xFF,0xD9,0x00,0x0C,0x11,0xDE,0xE7,0xF7,0x40,0x8D,
0x8D,0xA5,0xC6,0x05,0xB0,0xB4,0xD5,0x6B,0xB5,0x5E,0x93,0x64,0xC7,0x7B,0xFA,0xD9,
0xC8,0x19,0x1E,0xD6,0xE1,0xFE,0x7B,0x7A,0x93,0x7C,0x6D,0x07,0x09,0x5F,0xE5,0xEA,
0x91,0xA7,0x00,0xB4,0xBD,0xFC,0x17,0xB4,0x28,0xD0,0x36,0x92,0x2A,0xA8,0xAB,0x5E,
0x2C,0xD5,0x85,0x84,0x6F,0xB8,0x1F,0xC6,0x93,0xB8,0xD5,0x9B,0xF8,0x5C,0x74,0xBC,
0x70,0x0C,0xD2,0xBC,0x3E,0x6A,0xAB,0x43,0x7D,0x93,0xD8,0xA3,0x0F,0x1C,0xF6,0x92,
0xEF,0xEF,0x43,0x60,0x20,0x28,0xE0,0xCE,0x57,0x42,0xEB,0x3F,0x4F,0x4D,0x5B,0x02,
0x91,0x58,0xDD,0x68,0x96,0xAC,0xB5,0xE3,0xA7,0xF6,0x84,0xD9,0xAA,0x89,0x14,0xE7,
0x09,0x74,0xB2,0x23,0xA6,0xFE,0xC3,0x8D,0x76,0xC7,0x47,0x3E,0x86,0xE4,0xB9,0xB3,
0x2C,0x62,0x1E,0x20,0x15,0xC5,0x5E,0x94,0x7D,0xD0,0x16,0xC6,0x75,0xC8,0x23,0x68,
0xCE,0x26,0xFB,0x45,0x6A,0x5B,0x65,0x88,0x1A,0xF5,0x13,0xBF,0xDC,0x88,0x68,0x7C,
0x63,0x81,0x67,0x6A,0xBB,0xD2,0xD9,0x10,0x4E,0xD2,0x3A,0x9E,0x89,0x31,0x02,0x46,
0xB0,0x26,0xCE,0xDD,0x57,0x59,0x5B,0x1A,0xB6,0xFE,0x88,0xA7,0x84,0xBE,0x0C,0x06
	},
};

/**
 * Testcase for SHAKE256 that exists at NIST's web page.
 *
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg0.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg5.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg30.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg1600.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg1605.pdf
 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHAKE256_Msg1630.pdf
 */
static uint8_t shake256_expected[6][DEFAULT_SHAKE256_DIGESTSIZE]={
	/* 0-bit */
	{
0x46,0xB9,0xDD,0x2B,0x0B,0xA8,0x8D,0x13,0x23,0x3B,0x3F,0xEB,0x74,0x3E,0xEB,0x24,
0x3F,0xCD,0x52,0xEA,0x62,0xB8,0x1B,0x82,0xB5,0x0C,0x27,0x64,0x6E,0xD5,0x76,0x2F,
0xD7,0x5D,0xC4,0xDD,0xD8,0xC0,0xF2,0x00,0xCB,0x05,0x01,0x9D,0x67,0xB5,0x92,0xF6,
0xFC,0x82,0x1C,0x49,0x47,0x9A,0xB4,0x86,0x40,0x29,0x2E,0xAC,0xB3,0xB7,0xC4,0xBE,
0x14,0x1E,0x96,0x61,0x6F,0xB1,0x39,0x57,0x69,0x2C,0xC7,0xED,0xD0,0xB4,0x5A,0xE3,
0xDC,0x07,0x22,0x3C,0x8E,0x92,0x93,0x7B,0xEF,0x84,0xBC,0x0E,0xAB,0x86,0x28,0x53,
0x34,0x9E,0xC7,0x55,0x46,0xF5,0x8F,0xB7,0xC2,0x77,0x5C,0x38,0x46,0x2C,0x50,0x10,
0xD8,0x46,0xC1,0x85,0xC1,0x51,0x11,0xE5,0x95,0x52,0x2A,0x6B,0xCD,0x16,0xCF,0x86,
0xF3,0xD1,0x22,0x10,0x9E,0x3B,0x1F,0xDD,0x94,0x3B,0x6A,0xEC,0x46,0x8A,0x2D,0x62,
0x1A,0x7C,0x06,0xC6,0xA9,0x57,0xC6,0x2B,0x54,0xDA,0xFC,0x3B,0xE8,0x75,0x67,0xD6,
0x77,0x23,0x13,0x95,0xF6,0x14,0x72,0x93,0xB6,0x8C,0xEA,0xB7,0xA9,0xE0,0xC5,0x8D,
0x86,0x4E,0x8E,0xFD,0xE4,0xE1,0xB9,0xA4,0x6C,0xBE,0x85,0x47,0x13,0x67,0x2F,0x5C,
0xAA,0xAE,0x31,0x4E,0xD9,0x08,0x3D,0xAB,0x4B,0x09,0x9F,0x8E,0x30,0x0F,0x01,0xB8,
0x65,0x0F,0x1F,0x4B,0x1D,0x8F,0xCF,0x3F,0x3C,0xB5,0x3F,0xB8,0xE9,0xEB,0x2E,0xA2,
0x03,0xBD,0xC9,0x70,0xF5,0x0A,0xE5,0x54,0x28,0xA9,0x1F,0x7F,0x53,0xAC,0x26,0x6B,
0x28,0x41,0x9C,0x37,0x78,0xA1,0x5F,0xD2,0x48,0xD3,0x39,0xED,0xE7,0x85,0xFB,0x7F,
0x5A,0x1A,0xAA,0x96,0xD3,0x13,0xEA,0xCC,0x89,0x09,0x36,0xC1,0x73,0xCD,0xCD,0x0F,
0xAB,0x88,0x2C,0x45,0x75,0x5F,0xEB,0x3A,0xED,0x96,0xD4,0x77,0xFF,0x96,0x39,0x0B,
0xF9,0xA6,0x6D,0x13,0x68,0xB2,0x08,0xE2,0x1F,0x7C,0x10,0xD0,0x4A,0x3D,0xBD,0x4E,
0x36,0x06,0x33,0xE5,0xDB,0x4B,0x60,0x26,0x01,0xC1,0x4C,0xEA,0x73,0x7D,0xB3,0xDC,
0xF7,0x22,0x63,0x2C,0xC7,0x78,0x51,0xCB,0xDD,0xE2,0xAA,0xF0,0xA3,0x3A,0x07,0xB3,
0x73,0x44,0x5D,0xF4,0x90,0xCC,0x8F,0xC1,0xE4,0x16,0x0F,0xF1,0x18,0x37,0x8F,0x11,
0xF0,0x47,0x7D,0xE0,0x55,0xA8,0x1A,0x9E,0xDA,0x57,0xA4,0xA2,0xCF,0xB0,0xC8,0x39,
0x29,0xD3,0x10,0x91,0x2F,0x72,0x9E,0xC6,0xCF,0xA3,0x6C,0x6A,0xC6,0xA7,0x58,0x37,
0x14,0x30,0x45,0xD7,0x91,0xCC,0x85,0xEF,0xF5,0xB2,0x19,0x32,0xF2,0x38,0x61,0xBC,
0xF2,0x3A,0x52,0xB5,0xDA,0x67,0xEA,0xF7,0xBA,0xAE,0x0F,0x5F,0xB1,0x36,0x9D,0xB7,
0x8F,0x3A,0xC4,0x5F,0x8C,0x4A,0xC5,0x67,0x1D,0x85,0x73,0x5C,0xDD,0xDB,0x09,0xD2,
0xB1,0xE3,0x4A,0x1F,0xC0,0x66,0xFF,0x4A,0x16,0x2C,0xB2,0x63,0xD6,0x54,0x12,0x74,
0xAE,0x2F,0xCC,0x86,0x5F,0x61,0x8A,0xBE,0x27,0xC1,0x24,0xCD,0x8B,0x07,0x4C,0xCD,
0x51,0x63,0x01,0xB9,0x18,0x75,0x82,0x4D,0x09,0x95,0x8F,0x34,0x1E,0xF2,0x74,0xBD,
0xAB,0x0B,0xAE,0x31,0x63,0x39,0x89,0x43,0x04,0xE3,0x58,0x77,0xB0,0xC2,0x8A,0x9B,
0x1F,0xD1,0x66,0xC7,0x96,0xB9,0xCC,0x25,0x8A,0x06,0x4A,0x8F,0x57,0xE2,0x7F,0x2A,
	},
	/* 5-bit */
	{
0x48,0xA5,0xC1,0x1A,0xBA,0xEE,0xFF,0x09,0x2F,0x36,0x46,0xEF,0x0D,0x6B,0x3D,0x3F,
0xF7,0x6C,0x2F,0x55,0xF9,0xC7,0x32,0xAC,0x64,0x70,0xC0,0x37,0x64,0x00,0x82,0x12,
0xE2,0x1B,0x14,0x67,0x77,0x8B,0x18,0x19,0x89,0xF8,0x88,0x58,0x21,0x1B,0x45,0xDF,
0x87,0x99,0xCF,0x96,0x1F,0x80,0x0D,0xFA,0xC9,0x9E,0x64,0x40,0x39,0xE2,0x97,0x9A,
0x40,0x16,0xF5,0x45,0x6F,0xF4,0x21,0xC5,0xB3,0x85,0xDA,0x2B,0x85,0x5D,0xA7,0xE3,
0x1C,0x8C,0x2E,0x8E,0x4B,0xA4,0x1E,0xB4,0x09,0x5C,0xB9,0x99,0xD9,0x75,0x9C,0xB4,
0x03,0x58,0xDA,0x85,0x62,0xA2,0xE6,0x13,0x49,0xE0,0x5A,0x2E,0x13,0xF1,0xB7,0x4E,
0xC9,0xE6,0x9F,0x5B,0x42,0x6D,0xC7,0x41,0x38,0xFF,0xCD,0xC5,0x71,0xC3,0x2B,0x39,
0xB9,0xF5,0x55,0x63,0xE1,0xA9,0x9D,0xC4,0x22,0xC3,0x06,0x02,0x6D,0x6A,0x0F,0x9D,
0xE8,0x51,0x62,0xB3,0x86,0x79,0x4C,0xA0,0x68,0x8B,0x76,0x4B,0x3D,0x32,0x20,0x0C,
0xC4,0x59,0x74,0x97,0x32,0xA0,0xF3,0xA3,0x41,0xC0,0xEF,0xC9,0x6A,0x22,0xC6,0x3B,
0xAD,0x7D,0x96,0xCC,0x9B,0xA4,0x76,0x8C,0x6F,0xCF,0xA1,0xF2,0x00,0x10,0x7C,0xF9,
0xFA,0xE5,0xC0,0xD7,0x54,0x95,0x8C,0x5A,0x75,0x6B,0x37,0x6A,0x3B,0xE6,0x9F,0x88,
0x07,0x4F,0x20,0x0E,0x9E,0x95,0xA8,0xCA,0x5B,0xCF,0x96,0x99,0x98,0xDB,0x1D,0xC3,
0x7D,0x0D,0x3D,0x91,0x6F,0x6C,0xAA,0xB3,0xF0,0x37,0x82,0xC9,0xC4,0x4A,0x2E,0x14,
0xE8,0x07,0x86,0xBE,0xCE,0x45,0x87,0xB9,0xEF,0x82,0xCB,0xF4,0x54,0xE0,0xE3,0x4B,
0xD1,0x75,0xAE,0x57,0xD3,0x6A,0xF4,0xE7,0x26,0xB2,0x21,0x33,0x2C,0xED,0x36,0xC8,
0xCE,0x2E,0x06,0x20,0x3C,0x65,0x6A,0xE8,0xDA,0x03,0x7D,0x08,0xE7,0x16,0x0B,0x48,
0x0C,0x1A,0x85,0x16,0xBF,0x06,0xDD,0x97,0xBF,0x4A,0xA4,0xC0,0x24,0x93,0x10,0xDC,
0x0B,0x06,0x5D,0xC6,0x39,0x57,0x63,0x55,0x38,0x4D,0x16,0x5C,0x6A,0x50,0x9B,0x12,
0xF7,0xBB,0xD1,0xE1,0x5B,0x22,0xBC,0xE0,0x2F,0xA0,0x48,0xDD,0xFA,0xAC,0xF7,0x41,
0x5F,0x49,0xB6,0x32,0x4C,0x1D,0x06,0x7B,0x52,0x64,0xE1,0x12,0x5F,0x7F,0x75,0x42,
0x7F,0x31,0x2B,0xD9,0x34,0x6E,0xB4,0xE4,0x00,0xB1,0xF7,0xCB,0x31,0x28,0x8C,0x9E,
0x3F,0x73,0x5E,0xCA,0x9C,0xED,0x0D,0xB8,0x88,0xE2,0xE2,0xF4,0x02,0x24,0x3B,0xD6,
0x46,0x18,0xA2,0x3E,0x10,0xF9,0xC2,0x29,0x39,0x74,0x40,0x54,0x2D,0x0A,0xB1,0xB2,
0xE1,0x0D,0xAC,0xC5,0xC9,0x5E,0x59,0x7F,0x2C,0x7E,0xA3,0x84,0x38,0x10,0x5F,0x97,
0x80,0x3D,0xBB,0x03,0xFC,0xC0,0xFD,0x41,0x6B,0x09,0x05,0xA4,0x1D,0x18,0x4D,0xEB,
0x23,0x89,0x05,0x77,0x58,0x91,0xF9,0x35,0x01,0xFB,0x41,0x76,0xA3,0xBD,0x6C,0x46,
0x44,0x61,0xD3,0x6E,0xE8,0xB0,0x08,0xAA,0xBD,0x9E,0x26,0xA3,0x40,0x55,0xE8,0x0C,
0x8C,0x81,0x3E,0xEB,0xA0,0x7F,0x72,0x8A,0xB3,0x2B,0x15,0x60,0x5A,0xD1,0x61,0xA0,
0x66,0x9F,0x6F,0xCE,0x5C,0x55,0x09,0xFB,0xB6,0xAF,0xD2,0x4A,0xEA,0xCC,0x5F,0xA4,
0xA5,0x15,0x23,0xE6,0xB1,0x73,0x24,0x6E,0xD4,0xBF,0xA5,0x21,0xD7,0x4F,0xC6,0xBB,
	},
	/* 30-bit */
	{
0x46,0x5D,0x08,0x1D,0xFF,0x87,0x5E,0x39,0x62,0x00,0xE4,0x48,0x1A,0x3E,0x9D,0xCD,
0x88,0xD0,0x79,0xAA,0x6D,0x66,0x22,0x6C,0xB6,0xBA,0x45,0x41,0x07,0xCB,0x81,0xA7,
0x84,0x1A,0xB0,0x29,0x60,0xDE,0x27,0x9C,0xCB,0xE3,0x4B,0x42,0xC3,0x65,0x85,0xAD,
0x86,0x96,0x4D,0xB0,0xDB,0x52,0xB6,0xE7,0xB4,0x36,0x9E,0xCE,0x8F,0x72,0x48,0x58,
0x9B,0xA7,0x8A,0xB1,0x82,0x8F,0xFC,0x33,0x5C,0xB1,0x23,0x97,0x11,0x9B,0xFD,0x2B,
0x87,0xEB,0x78,0x98,0xAE,0xB9,0x56,0xB6,0xF2,0x3D,0xDF,0x0B,0xD4,0x00,0x43,0x86,
0xA8,0xE5,0x26,0x55,0x4E,0xF4,0xE4,0x83,0xFA,0xCE,0xE3,0x0D,0xD3,0x2E,0x20,0x4F,
0xFF,0x8C,0x36,0xBB,0xD6,0x02,0xA5,0x76,0xD1,0x39,0x08,0x9C,0x75,0xA8,0x05,0x02,
0x66,0xFC,0xBF,0x72,0x1E,0x44,0x43,0xDE,0x46,0x45,0x83,0x29,0x22,0xEB,0x8A,0xAE,
0x39,0xD1,0xF5,0x72,0x84,0x53,0x64,0x81,0x7B,0x00,0x33,0x54,0x38,0x99,0x94,0x00,
0x23,0xF2,0xE9,0x65,0xA6,0x0A,0x80,0xEB,0x22,0x1E,0xB1,0x9D,0xC5,0x7B,0x12,0x12,
0x91,0x56,0x4C,0x6F,0x69,0x35,0x83,0xB3,0xAC,0x7C,0x6F,0x27,0x2F,0x4F,0x67,0xA1,
0x9A,0x76,0x78,0xD4,0x23,0x4B,0x0B,0xF4,0xA2,0xEB,0xC0,0x8A,0xA2,0x35,0xB9,0x78,
0x8D,0xB7,0x87,0x16,0x1F,0x66,0x17,0x02,0x28,0x65,0xC0,0xEF,0x9A,0xA5,0x33,0x80,
0x2D,0x13,0x6C,0xDB,0xC7,0xAE,0xBA,0x53,0x2A,0xCF,0x1B,0xE1,0x83,0xB0,0x29,0x5A,
0xB0,0xE3,0x3A,0x2E,0xF6,0x9B,0xE3,0x56,0xDA,0xAF,0x30,0x96,0x87,0x15,0x3E,0x2F,
0x99,0xA1,0x24,0x36,0x09,0xD6,0x03,0x12,0x6A,0x8C,0x82,0x3E,0x88,0x43,0xE4,0x59,
0xBF,0xC7,0x2B,0x30,0x69,0x1C,0xDC,0xC3,0xDD,0xB2,0x7C,0xF0,0x28,0xAF,0xD5,0x1E,
0x44,0x37,0xEE,0x3B,0x71,0xC0,0xC1,0xEC,0x87,0xA9,0x34,0x36,0xF0,0xC2,0x47,0xB7,
0xE8,0xC5,0x0C,0xE9,0x68,0x25,0xC9,0x70,0x29,0x99,0x7A,0x74,0xC3,0x18,0xAF,0xAC,
0xAA,0x18,0xA0,0x18,0x0B,0xC7,0xF2,0xF0,0xF1,0xC5,0xE7,0xEF,0x1A,0x2D,0x18,0x3A,
0xC7,0xEE,0x7E,0x49,0x15,0xC3,0xB6,0x8C,0x30,0x97,0x8A,0xB6,0xC4,0x28,0x19,0x34,
0x41,0xDF,0x47,0x05,0xB7,0x22,0xCE,0x25,0xA0,0x8A,0x1F,0xAD,0xCA,0x0E,0xEF,0x1F,
0xAF,0xE8,0x3A,0xDF,0x13,0x02,0x1D,0x52,0x0D,0xE5,0xC8,0x27,0xFF,0x9A,0x97,0xB7,
0x55,0x46,0x19,0x3A,0x9B,0x92,0x3F,0x05,0x90,0x38,0x5D,0xC4,0xBF,0xF7,0xC4,0x9D,
0x49,0x15,0xB5,0xA3,0x65,0xDB,0x4C,0x84,0xDD,0xCB,0x18,0x5D,0xE8,0xF9,0xEE,0xB3,
0x34,0x96,0x5A,0x42,0xF1,0x38,0x1C,0x8B,0xAD,0xC2,0x2B,0xA1,0xF8,0xEE,0x4C,0x0E,
0x4D,0xAA,0xF7,0xA8,0x8E,0x7F,0x42,0xDD,0xB8,0x14,0x8F,0x3B,0xF8,0xD3,0xB8,0xD7,
0x4F,0x09,0x81,0x55,0xA3,0x7C,0xB4,0xCB,0x27,0x87,0x6B,0x85,0xDA,0x60,0x2E,0x5C,
0x78,0x9C,0x10,0xE0,0x3B,0xE7,0x34,0x07,0xBA,0xB8,0xC4,0x92,0x13,0xF8,0xC7,0x4E,
0x12,0x66,0xCE,0x9B,0x11,0x28,0x6E,0x67,0x4C,0xA9,0xC1,0x0C,0x9C,0x99,0x55,0x04,
0x9A,0x66,0xE9,0x05,0x1D,0x9A,0x2B,0x1F,0xC9,0xAF,0xE2,0x67,0x98,0xE9,0xCE,0xC6,
	},
	/* 1600-bit */
	{
0xCD,0x8A,0x92,0x0E,0xD1,0x41,0xAA,0x04,0x07,0xA2,0x2D,0x59,0x28,0x86,0x52,0xE9,
0xD9,0xF1,0xA7,0xEE,0x0C,0x1E,0x7C,0x1C,0xA6,0x99,0x42,0x4D,0xA8,0x4A,0x90,0x4D,
0x2D,0x70,0x0C,0xAA,0xE7,0x39,0x6E,0xCE,0x96,0x60,0x44,0x40,0x57,0x7D,0xA4,0xF3,
0xAA,0x22,0xAE,0xB8,0x85,0x7F,0x96,0x1C,0x4C,0xD8,0xE0,0x6F,0x0A,0xE6,0x61,0x0B,
0x10,0x48,0xA7,0xF6,0x4E,0x10,0x74,0xCD,0x62,0x9E,0x85,0xAD,0x75,0x66,0x04,0x8E,
0xFC,0x4F,0xB5,0x00,0xB4,0x86,0xA3,0x30,0x9A,0x8F,0x26,0x72,0x4C,0x0E,0xD6,0x28,
0x00,0x1A,0x10,0x99,0x42,0x24,0x68,0xDE,0x72,0x6F,0x10,0x61,0xD9,0x9E,0xB9,0xE9,
0x36,0x04,0xD5,0xAA,0x74,0x67,0xD4,0xB1,0xBD,0x64,0x84,0x58,0x2A,0x38,0x43,0x17,
0xD7,0xF4,0x7D,0x75,0x0B,0x8F,0x54,0x99,0x51,0x2B,0xB8,0x5A,0x22,0x6C,0x42,0x43,
0x55,0x6E,0x69,0x6F,0x6B,0xD0,0x72,0xC5,0xAA,0x2D,0x9B,0x69,0x73,0x02,0x44,0xB5,
0x68,0x53,0xD1,0x69,0x70,0xAD,0x81,0x7E,0x21,0x3E,0x47,0x06,0x18,0x17,0x80,0x01,
0xC9,0xFB,0x56,0xC5,0x4F,0xEF,0xA5,0xFE,0xE6,0x7D,0x2D,0xA5,0x24,0xBB,0x3B,0x0B,
0x61,0xEF,0x0E,0x91,0x14,0xA9,0x2C,0xDB,0xB6,0xCC,0xCB,0x98,0x61,0x5C,0xFE,0x76,
0xE3,0x51,0x0D,0xD8,0x8D,0x1C,0xC2,0x8F,0xF9,0x92,0x87,0x51,0x2F,0x24,0xBF,0xAF,
0xA1,0xA7,0x68,0x77,0xB6,0xF3,0x71,0x98,0xE3,0xA6,0x41,0xC6,0x8A,0x7C,0x42,0xD4,
0x5F,0xA7,0xAC,0xC1,0x0D,0xAE,0x5F,0x3C,0xEF,0xB7,0xB7,0x35,0xF1,0x2D,0x4E,0x58,
0x9F,0x7A,0x45,0x6E,0x78,0xC0,0xF5,0xE4,0xC4,0x47,0x1F,0xFF,0xA5,0xE4,0xFA,0x05,
0x14,0xAE,0x97,0x4D,0x8C,0x26,0x48,0x51,0x3B,0x5D,0xB4,0x94,0xCE,0xA8,0x47,0x15,
0x6D,0x27,0x7A,0xD0,0xE1,0x41,0xC2,0x4C,0x78,0x39,0x06,0x4C,0xD0,0x88,0x51,0xBC,
0x2E,0x7C,0xA1,0x09,0xFD,0x4E,0x25,0x1C,0x35,0xBB,0x0A,0x04,0xFB,0x05,0xB3,0x64,
0xFF,0x8C,0x4D,0x8B,0x59,0xBC,0x30,0x3E,0x25,0x32,0x8C,0x09,0xA8,0x82,0xE9,0x52,
0x51,0x8E,0x1A,0x8A,0xE0,0xFF,0x26,0x5D,0x61,0xC4,0x65,0x89,0x69,0x73,0xD7,0x49,
0x04,0x99,0xDC,0x63,0x9F,0xB8,0x50,0x2B,0x39,0x45,0x67,0x91,0xB1,0xB6,0xEC,0x5B,
0xCC,0x5D,0x9A,0xC3,0x6A,0x6D,0xF6,0x22,0xA0,0x70,0xD4,0x3F,0xED,0x78,0x1F,0x5F,
0x14,0x9F,0x7B,0x62,0x67,0x5E,0x7D,0x1A,0x4D,0x6D,0xEC,0x48,0xC1,0xC7,0x16,0x45,
0x86,0xEA,0xE0,0x6A,0x51,0x20,0x8C,0x0B,0x79,0x12,0x44,0xD3,0x07,0x72,0x65,0x05,
0xC3,0xAD,0x4B,0x26,0xB6,0x82,0x23,0x77,0x25,0x7A,0xA1,0x52,0x03,0x75,0x60,0xA7,
0x39,0x71,0x4A,0x3C,0xA7,0x9B,0xD6,0x05,0x54,0x7C,0x9B,0x78,0xDD,0x1F,0x59,0x6F,
0x2D,0x4F,0x17,0x91,0xBC,0x68,0x9A,0x0E,0x9B,0x79,0x9A,0x37,0x33,0x9C,0x04,0x27,
0x57,0x33,0x74,0x01,0x43,0xEF,0x5D,0x2B,0x58,0xB9,0x6A,0x36,0x3D,0x4E,0x08,0x07,
0x6A,0x1A,0x9D,0x78,0x46,0x43,0x6E,0x4D,0xCA,0x57,0x28,0xB6,0xF7,0x60,0xEE,0xF0,
0xCA,0x92,0xBF,0x0B,0xE5,0x61,0x5E,0x96,0x95,0x9D,0x76,0x71,0x97,0xA0,0xBE,0xEB,
	},
	/* 1605-bit */
	{
0x98,0xD0,0x93,0xB0,0x67,0x47,0x57,0x60,0x12,0x4F,0xFB,0x92,0x04,0xA5,0xB3,0x27,
0xC6,0xBB,0x05,0xC5,0x4F,0xF2,0x34,0xF0,0xB4,0x3F,0xAC,0x72,0x40,0x41,0x51,0x66,
0xA8,0xC7,0x05,0xEA,0x0D,0x73,0x9F,0x08,0x08,0xB0,0x65,0x76,0xD9,0x96,0x66,0x2C,
0x1F,0x37,0x66,0x94,0xD9,0x8F,0x51,0x57,0x19,0xB6,0x64,0x07,0x72,0x0D,0xCF,0x78,
0x1C,0x51,0xCD,0x56,0xEF,0x8B,0x61,0x0C,0x66,0x8D,0xDC,0x1A,0xC1,0xC2,0xC4,0x29,
0xEA,0x4D,0x6F,0x27,0x4A,0xA7,0xA7,0x73,0xBF,0x8B,0x0C,0xAB,0x30,0x6F,0x1E,0xEE,
0x2A,0x17,0x1B,0x91,0x33,0x4E,0xA0,0xFA,0xCD,0x2A,0xAC,0x1F,0x51,0xD4,0xD5,0xEB,
0x0E,0x63,0xA4,0xE6,0x75,0x4E,0xCA,0xFE,0xEC,0x24,0x6B,0x7A,0xAF,0x58,0xD0,0xE0,
0xA9,0x74,0xC7,0xFF,0x40,0x58,0xBD,0xBD,0xED,0xB3,0x3E,0xD0,0x4B,0x0F,0xA4,0x5D,
0x70,0xC7,0xC8,0x4F,0x3D,0xA1,0x3E,0x4F,0x7D,0x1B,0xED,0xDB,0x53,0x4D,0x37,0xE5,
0xAB,0xDF,0xB2,0x9F,0x2B,0x44,0xC4,0xFB,0x0D,0x6C,0xCA,0xB8,0x31,0xD9,0x0B,0xA4,
0x6A,0x00,0x53,0x06,0x62,0xF9,0x07,0xDE,0xDD,0x47,0x9E,0x9B,0x54,0x28,0xE5,0xE2,
0xDB,0x80,0x40,0xB0,0xE2,0xB1,0xF1,0x74,0xCE,0x34,0x7F,0x32,0xA0,0x6A,0x5A,0xC2,
0x2B,0x19,0xAA,0xFE,0x92,0x7B,0x88,0x78,0xD0,0xC8,0x10,0x3A,0x4D,0x2F,0x19,0xE3,
0x23,0x36,0xC6,0x4C,0xFA,0xDC,0x1B,0x9A,0xCB,0x39,0x78,0xA8,0x29,0x85,0x71,0xDC,
0xD8,0x9C,0x36,0xA6,0x56,0x92,0x81,0x6D,0x0C,0x61,0xCE,0x0E,0xD1,0x79,0x42,0x36,
0x70,0x17,0xBD,0x40,0xF5,0x9D,0xFB,0xAE,0x34,0x63,0x58,0x27,0x92,0x0A,0xFE,0x7A,
0x27,0xBF,0x56,0x70,0x09,0xA1,0x38,0x40,0x3F,0x06,0xB6,0xE4,0xDE,0x94,0xDA,0x07,
0x7D,0xB4,0x97,0x73,0xC2,0x35,0x46,0x61,0x19,0x42,0x6F,0x79,0x88,0x8D,0x3A,0x81,
0xB4,0x07,0xDF,0xEB,0xA8,0x7E,0x01,0xCD,0x48,0xF9,0x0E,0x01,0xB6,0xF9,0x02,0x43,
0xC4,0x01,0x25,0xDE,0x47,0xE8,0xC8,0xF3,0xE6,0xEA,0x33,0x88,0xCB,0xFE,0xEB,0x36,
0x54,0x1E,0xF2,0x3D,0x2C,0x83,0x48,0x45,0x8E,0xA2,0x8C,0xAA,0x50,0x66,0xF4,0x98,
0x37,0x76,0xF0,0xCB,0x2F,0xDC,0x66,0x04,0x9C,0xF8,0x8A,0xC8,0xEA,0xE5,0x12,0x12,
0xAA,0xCE,0x86,0x7B,0xEA,0x4C,0x3C,0xAE,0xE4,0x4F,0x14,0x7A,0x9B,0xF9,0x9D,0x04,
0x87,0x4E,0x87,0x22,0xD0,0x3D,0x3F,0x5F,0xF6,0xEF,0x3B,0xEB,0xE7,0x64,0x2F,0xE4,
0x91,0x6C,0x5F,0x10,0xFF,0x3F,0xD6,0x13,0x87,0xD5,0xD9,0x1B,0xCD,0x32,0xF9,0xE8,
0xE4,0x59,0x3D,0xCA,0xAD,0x23,0xEC,0xCC,0x05,0xD2,0xFC,0x9B,0xE2,0xC1,0xCD,0x63,
0x0E,0xA1,0x23,0xDC,0xA9,0xCB,0x69,0x38,0xD6,0x0C,0xDD,0xED,0xC1,0x1E,0x1E,0x9B,
0xC9,0xD2,0x68,0xA5,0x45,0x6B,0xA9,0xCC,0xFF,0x18,0x59,0x7C,0x5F,0xF9,0x73,0x57,
0x08,0x41,0x3B,0x9D,0x84,0xB9,0xF4,0x72,0x19,0x37,0xCC,0x65,0x95,0x71,0x27,0x97,
0x53,0x2B,0x48,0xD6,0xF1,0xA2,0xD1,0x72,0x3B,0x07,0xD5,0x46,0x0B,0xC1,0x39,0x16,
0xD9,0x6E,0x88,0x18,0x07,0x13,0xAC,0x33,0xD2,0xC2,0x32,0xE3,0x5E,0x76,0x4E,0x04,
	},
	/* 1630-bit */
	{
0x8A,0x83,0x25,0x07,0x9B,0x0F,0xC3,0x26,0x5D,0x52,0xF5,0x98,0x55,0xCA,0xFE,0x65,
0x5D,0xF4,0x38,0xAA,0x63,0x9F,0x6F,0xEC,0x99,0x1F,0x24,0x94,0x33,0x0C,0xE3,0x2F,
0xA3,0x7F,0x7D,0xB9,0x0F,0x69,0x66,0xD8,0xE4,0xA4,0x6E,0x50,0xC5,0xED,0xE5,0x7B,
0x9B,0x8F,0x08,0x2A,0x96,0x62,0x7F,0x73,0x04,0x75,0x02,0x9A,0x61,0x92,0x29,0xD8,
0x4F,0x43,0x2E,0xD6,0x9F,0xD0,0x59,0x23,0x4D,0x4D,0x7D,0xD3,0x58,0xE8,0x39,0x3F,
0x6A,0x36,0xA4,0x5C,0xCF,0x04,0x1F,0x90,0xFC,0x0A,0x4E,0x58,0x02,0xD7,0x30,0x63,
0xD3,0x65,0x31,0x33,0x6A,0x00,0x90,0xEC,0xFE,0x1A,0x4D,0x4D,0x29,0xAA,0x82,0x4B,
0xA4,0x2B,0x49,0x37,0xB4,0xBB,0x98,0xF4,0xF3,0x3A,0x0E,0x3B,0xD8,0xB5,0x11,0xE6,
0x95,0x28,0xD5,0x95,0x37,0x11,0x0D,0x75,0x21,0xFB,0x78,0xAC,0xA0,0x18,0xDF,0x76,
0x16,0x0F,0x54,0xA3,0x42,0x1B,0x84,0x14,0x92,0x64,0xED,0x03,0x2F,0x6D,0xCE,0x46,
0x7A,0x73,0x1A,0x8E,0x34,0x04,0x8E,0x3A,0x46,0xE9,0x80,0x39,0xDF,0x3C,0x32,0x8D,
0xEB,0xFB,0xE5,0xD1,0xBC,0x8B,0xE7,0xFF,0x4E,0xF8,0x91,0x7B,0x01,0xF0,0xB7,0x89,
0x36,0x72,0x49,0x2D,0x6E,0xE5,0xC7,0x1D,0xF2,0xD0,0x53,0x1F,0x8B,0x68,0x47,0x64,
0xBA,0x0A,0x2B,0x57,0xEC,0x6A,0x4F,0x60,0xBA,0x4F,0x36,0xFE,0x2D,0xB0,0xE6,0x5A,
0xD7,0xAA,0x5F,0x14,0xF3,0xEF,0x9F,0x34,0xA0,0xAB,0x5B,0xC3,0x3D,0x48,0x87,0x33,
0xBA,0x36,0xBF,0x4B,0x2B,0x4F,0xCE,0x02,0x8E,0xFF,0x8C,0x6C,0xE0,0x3B,0x19,0x2C,
0xF0,0x75,0xCC,0x9F,0x00,0xD2,0x9C,0x0E,0x06,0xC3,0x5C,0x44,0x89,0xD2,0x7F,0x07,
0xFA,0x49,0xA9,0x1C,0xA9,0x24,0x71,0xE3,0x4D,0xAB,0x77,0x87,0xAE,0x24,0xA6,0xE0,
0xF3,0x09,0xEF,0x0B,0xA5,0x3F,0x7C,0x8B,0x29,0x92,0x52,0x0A,0x07,0xBE,0xDD,0x50,
0x9A,0x0B,0x6D,0xBE,0xA5,0x70,0xA5,0x96,0x0E,0xD6,0x24,0x82,0x6D,0xD8,0xEC,0xD1,
0x91,0x5C,0x87,0x32,0x7E,0x74,0x49,0x1C,0x40,0x5A,0x74,0x11,0xC1,0x2C,0x0D,0x44,
0x97,0x51,0x26,0x89,0xBD,0x7F,0x5A,0xDB,0xED,0xB0,0x2C,0x6D,0x2E,0x68,0x47,0x4E,
0x8B,0xF3,0x1B,0x88,0x40,0x40,0x81,0x8F,0x4B,0xCA,0x03,0xA4,0x52,0x17,0xEA,0xC7,
0x08,0x3A,0xD3,0xA3,0x3C,0xB8,0x47,0x7A,0x04,0xC9,0xE3,0x26,0x6A,0x13,0x34,0x77,
0xDE,0x45,0xE7,0x18,0x30,0xA4,0x0E,0xB0,0xD0,0x75,0xAF,0xCC,0xFC,0xD9,0xDC,0x54,
0x8D,0x0D,0x52,0x94,0x60,0xEA,0x7A,0xC2,0xAD,0xAC,0x72,0x2E,0x76,0x78,0xEF,0x59,
0x7D,0xD3,0xB4,0x95,0xBD,0x7D,0x1A,0x8F,0xF3,0x94,0x48,0xBB,0xAB,0x1D,0xC6,0xA8,
0x84,0x81,0x80,0x1C,0xF5,0xA8,0x01,0x0E,0x87,0x3C,0x31,0xE4,0x79,0xA5,0xE3,0xDB,
0x3D,0x4E,0x67,0xD1,0xD9,0x48,0xE6,0x7C,0xC6,0x6F,0xD7,0x5A,0x4A,0x19,0xC1,0x20,
0x66,0x2E,0xF5,0x59,0x77,0xBD,0xDB,0xAC,0x07,0x21,0xC8,0x0D,0x69,0x90,0x26,0x93,
0xC8,0x3D,0x5E,0xF7,0xBC,0x27,0xEF,0xA3,0x93,0xAF,0x4C,0x43,0x9F,0xC3,0x99,0x58,
0xE0,0xE7,0x55,0x37,0x35,0x88,0x02,0xEF,0x08,0x53,0xB7,0x47,0x0B,0x0F,0x19,0xAC,
	},
};

void init_nist_examples()
{
	int i, j, n;

	/*
	 * Msg as bit string (0-bit)
	 *
	 * (empty message)
	 */
	nist_tv[0].databitlen = 0;
	nist_tv[0].data[0] = 0x00;

	/*
	 * Msg as bit string (5-bit)
	 *
	 * 1 1 0 0 1
	 *
	 * 5 bit
	 * 1100 1000 => 0001 0011
	 * 00     07    07     00
	 */
	nist_tv[1].databitlen = 5;
	nist_tv[1].data[0] = 0x13; /* 0001 0011 */

	/*
	 * Msg as bit string (30-bit)
	 *
	 * 1 1 0 0 1 0 1 0 0 0 0 1 1 0 1 0 1 1 0 1 1 1 1 0 1 0 0 1 1 0
	 *
	 * 30 bit
	 * 1100 1010 0001 1010 => 0101 0011 0101 1000
	 * 00     07 10     17    07     01 17     10
	 * 1101 1110 1001 1000 => 0111 1011 0001 1001
	 * 00     07 10     17    07     01 17     10
	 */
	nist_tv[2].databitlen = 30;
	nist_tv[2].data[0] = 0x53; /* 0101 0011 */
	nist_tv[2].data[1] = 0x58; /* 0101 1000 */
	nist_tv[2].data[2] = 0x7b; /* 0111 1011 */
	nist_tv[2].data[3] = 0x19; /* 0001 1001 */

	/*
	 * Msg as bit string (1600-bit)
	 *
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 * (repeate 48 times)
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 *
	 * 11000101 x 8 x 25
	 *
	 * 11000101 => 10100011
	 * 00    07    07    00
	 */
	nist_tv[3].databitlen = 1600;
	for (i = 0; i < 200; i++) {
		nist_tv[3].data[i] = 0xa3; /* 1010 0011 */
	}

	/*
	 * Msg as bit string (1605-bit)
	 *
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 * (repeate 48 times)
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 * 1 1 0 0 0
	 *
	 * 11000101 x 4 x 50 + 11000
	 *
	 * 1100 0000 => 0000 0011
	 * 00     07    07     00
	 */
	nist_tv[4].databitlen = 1605;
	for (i = 0; i < 200; i++) {
		nist_tv[4].data[i] = 0xa3; /* 1010 0011 */
	}
	nist_tv[4].data[i] = 0x03; /* 0000 0011 */

	/*
	 * Msg as bit string (1630-bit)
	 *
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 * (repeate 48 times)
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1
	 * 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1 0 1 1 1 0 0 0 1
	 *
	 * 11000101 x 4 x 50 + 11000101 x 3 + 110001
	 *
	 * 1100 0100 => 0010 0011
	 * 00     07    07     00
	 */
	nist_tv[5].databitlen = 1630;
	for (i = 0; i < 203; i++) {
		nist_tv[5].data[i] = 0xa3; /* 1010 0011 */
	}
	nist_tv[5].data[i] = 0x23; /* 0010 0011 */
};

static int bitlen_to_index(size_t databitlen)
{
	int index;

	switch (databitlen) {
	case 0:
		index = 0;
		break;

	case 5:
		index = 1;
		break;

	case 30:
		index = 2;
		break;

	case 1600:
		index = 3;
		break;

	case 1605:
		index = 4;
		break;

	case 1630:
		index = 5;
		break;

	default:
		index = -1;
	}

	return index;
}

uint8_t *get_sha3_224_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return sha3_224_expected[index];
}

uint8_t *get_sha3_256_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return sha3_256_expected[index];
}

uint8_t *get_sha3_384_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return sha3_384_expected[index];
}

uint8_t *get_sha3_512_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return sha3_512_expected[index];
}

uint8_t *get_shake128_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return shake128_expected[index];
}

uint8_t *get_shake256_expected(size_t databitlen)
{
	int index;

	if ((index = bitlen_to_index(databitlen)) < 0) {
		printf("invalid message length: %d\n", index);
		return NULL;
	}

	return shake256_expected[index];
}
