/* sha3.c */
/*
 * Copyright (c) 2014-2017 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */

#include <stdio.h>
#include <string.h>
#include <aicrypto/nrg_sha3.h>

#include "KeccakP-1600-SnP.h"
#include "Modes/KeccakHash.h"

/**
 * SHA-3 internal context object
 *
 * This structure has the internal context information of SHA-3
 * and the size of that MUST be lower equal than SHA3_CTX.
 */
typedef struct _sha3_contxt {
	enum { FIXED, VARIABLE } lentype;
	unsigned int		capacity;
	unsigned int		hashlen;
	unsigned char		delimitedSuffix;
	Keccak_HashInstance	hash;
} _SHA3_CTX;

#define SHA3_224_CAPACITY	(224 * 2)
#define SHA3_256_CAPACITY	(256 * 2)
#define SHA3_384_CAPACITY	(384 * 2)
#define SHA3_512_CAPACITY	(512 * 2)
#define SHAKE128_CAPACITY	(128 * 2)
#define SHAKE256_CAPACITY	(256 * 2)


/**
 * Initialize a SHA-3 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
static void SHA3init(_SHA3_CTX *ctx)
{
	unsigned int rate = 1600 - ctx->capacity;

	Keccak_HashInitialize(&ctx->hash, rate, ctx->capacity,
			      ctx->hashlen * 8, ctx->delimitedSuffix);
}

/**
 * Update a SHA-3 context.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
static void SHA3update(_SHA3_CTX *ctx, unsigned char *in, int len)
{
	if ((in == NULL) || (len < 0))
		return;

	Keccak_HashUpdate(&ctx->hash, in, len * 8);
}

/**
 * Finalize a SHA-3 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 28bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have enough space for return.
 */
static void SHA3final(unsigned char *ret, _SHA3_CTX *ctx)
{
	Keccak_HashFinal(&ctx->hash, ret);
	if (ctx->lentype == VARIABLE) {
		Keccak_HashSqueeze(&ctx->hash, ret, ctx->hashlen * 8);
	}
}


/**
 * Compute SHA3-224.
 *
 */
void NRG_SHA3_224(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHA3_224init(&ctx);
	SHA3_224update(&ctx, in, len);
	SHA3_224final(ret, &ctx);
}

/**
 * Initialize a SHA3-224 context.
 *
 */
void SHA3_224init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHA3_224_CAPACITY;
	ctx->hashlen = SHA3_224_DIGESTSIZE;
	ctx->delimitedSuffix = 0x06;
	ctx->lentype = FIXED;

	SHA3init(ctx);
}

/**
 * Update a SHA3-224 context.
 *
 */
void SHA3_224update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHA3-224 context.
 *
 */
void SHA3_224final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3final(ret, ctx);
}


/**
 * Compute SHA3-256.
 *
 */
void NRG_SHA3_256(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHA3_256init(&ctx);
	SHA3_256update(&ctx, in, len);
	SHA3_256final(ret, &ctx);
}

/**
 * Initialize a SHA3-256 context.
 *
 */
void SHA3_256init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHA3_256_CAPACITY;
	ctx->hashlen = SHA3_256_DIGESTSIZE;
	ctx->delimitedSuffix = 0x06;
	ctx->lentype = FIXED;

	SHA3init(ctx);
}

/**
 * Update a SHA3-256 context.
 *
 */
void SHA3_256update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHA3-256 context.
 *
 */
void SHA3_256final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3final(ret, ctx);
}


/**
 * Compute SHA3-384.
 *
 */
void NRG_SHA3_384(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHA3_384init(&ctx);
	SHA3_384update(&ctx, in, len);
	SHA3_384final(ret, &ctx);
}

/**
 * Initialize a SHA3-384 context.
 *
 */
void SHA3_384init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHA3_384_CAPACITY;
	ctx->hashlen = SHA3_384_DIGESTSIZE;
	ctx->delimitedSuffix = 0x06;
	ctx->lentype = FIXED;

	SHA3init(ctx);
}

/**
 * Update a SHA3-384 context.
 *
 */
void SHA3_384update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHA3-384 context.
 *
 */
void SHA3_384final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3final(ret, ctx);
}


/**
 * Compute SHA3-512.
 *
 */
void NRG_SHA3_512(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHA3_512init(&ctx);
	SHA3_512update(&ctx, in, len);
	SHA3_512final(ret, &ctx);
}

/**
 * Initialize a SHA3-512 context.
 *
 */
void SHA3_512init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHA3_512_CAPACITY;
	ctx->hashlen = SHA3_512_DIGESTSIZE;
	ctx->delimitedSuffix = 0x06;
	ctx->lentype = FIXED;

	SHA3init(ctx);
}

/**
 * Update a SHA3-512 context.
 *
 */
void SHA3_512update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHA3-512 context.
 *
 */
void SHA3_512final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3final(ret, ctx);
}


/**
 * Compute SHAKE128.
 *
 */
void NRG_SHAKE128(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHAKE128init(&ctx);
	SHAKE128update(&ctx, in, len);
	SHAKE128final(ret, &ctx);
}

/**
 * Initialize a SHAKE128 context.
 *
 */
void SHAKE128init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHAKE128_CAPACITY;
	ctx->hashlen = SHAKE128_DIGESTSIZE;
	ctx->delimitedSuffix = 0x1f;
	ctx->lentype = VARIABLE;

	SHA3init(ctx);
}

/**
 * Update a SHAKE128 context.
 *
 */
void SHAKE128update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHAKE128 context.
 *
 */
void SHAKE128final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->hashlen = DEFAULT_SHAKE128_DIGESTSIZE;

	SHA3final(ret, ctx);
}

/**
 * Finalize a SHAKE128 context.
 *
 */
void SHAKE128final2(unsigned char *ret, int len, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->hashlen = len;

	SHA3final(ret, ctx);
}


/**
 * Compute SHAKE256.
 *
 */
void NRG_SHAKE256(int len, unsigned char *in, unsigned char *ret)
{
	SHA3_CTX ctx;

	if (len < 0) {
		return;
	}
	SHAKE256init(&ctx);
	SHAKE256update(&ctx, in, len);
	SHAKE256final(ret, &ctx);
}

/**
 * Initialize a SHAKE256 context.
 *
 */
void SHAKE256init(SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->capacity = SHAKE256_CAPACITY;
	ctx->hashlen = SHAKE256_DIGESTSIZE;
	ctx->delimitedSuffix = 0x1f;
	ctx->lentype = VARIABLE;

	SHA3init(ctx);
}

/**
 * Update a SHAKE256 context.
 *
 */
void SHAKE256update(SHA3_CTX *sha3ctx, unsigned char *in, int len)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	SHA3update(ctx, in, len);
}

/**
 * Finalize a SHAKE256 context.
 *
 */
void SHAKE256final(unsigned char *ret, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->hashlen = DEFAULT_SHAKE256_DIGESTSIZE;

	SHA3final(ret, ctx);
}

/**
 * Finalize a SHAKE256 context.
 *
 */
void SHAKE256final2(unsigned char *ret, int len, SHA3_CTX *sha3ctx)
{
	_SHA3_CTX *ctx = (_SHA3_CTX *)sha3ctx;

	ctx->hashlen = len;

	SHA3final(ret, ctx);
}

