/* sha2test.c */
/*
 * Copyright (c) 2012-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_sha2.h>

static void print_charhex(const char *name, const unsigned char *cs, int len)
{
	int i;

	printf("%s: ", name);
	for (i = 0; i < len; i++) {
		printf("%02x", cs[i]);
	}
	puts("");
}

int test_sha2_all(char *in)
{
	unsigned char ret224[SHA224_DIGESTSIZE];
	unsigned char ret256[SHA256_DIGESTSIZE];
	unsigned char ret384[SHA384_DIGESTSIZE];
	unsigned char ret512[SHA512_DIGESTSIZE];
	unsigned char ret512224[SHA512224_DIGESTSIZE];
	unsigned char ret512256[SHA512256_DIGESTSIZE];

	OK_SHA224(strlen(in), (unsigned char *)in, ret224);
	print_charhex("SHA-224", ret224, sizeof(ret224));

	OK_SHA256(strlen(in), (unsigned char *)in, ret256);
	print_charhex("SHA-256", ret256, sizeof(ret256));

	OK_SHA384(strlen(in), (unsigned char *)in, ret384);
	print_charhex("SHA-384", ret384, sizeof(ret384));

	OK_SHA512(strlen(in), (unsigned char *)in, ret512);
	print_charhex("SHA-512", ret512, sizeof(ret512));

	OK_SHA512224(strlen(in), (unsigned char *)in, ret512224);
	print_charhex("SHA-512/224", ret512224, sizeof(ret512224));

	OK_SHA512256(strlen(in), (unsigned char *)in, ret512256);
	print_charhex("SHA-512/256", ret512256, sizeof(ret512256));

	return EXIT_SUCCESS;
}

int test_sha224()
{
	SHA256_CTX ctx;
	unsigned char ret[SHA224_DIGESTSIZE], in2[1000];

	/*
	 * RFC3874
	 * 3.2.  Test Vector #2
	 * 75388b16 512776cc 5dba5da1 fd890150 b0c6455c b4f58b19 52522525
	 */
	char in[] = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
	unsigned char ans1[SHA224_DIGESTSIZE] = {
		0x75, 0x38, 0x8b, 0x16, 0x51, 0x27, 0x76, 0xcc,
		0x5d, 0xba, 0x5d, 0xa1, 0xfd, 0x89, 0x01, 0x50,
		0xb0, 0xc6, 0x45, 0x5c, 0xb4, 0xf5, 0x8b, 0x19,
		0x52, 0x52, 0x25, 0x25
	};

	/*
	 * RFC3874
	 * 3.3.  Test Vector #3
	 * 20794655 980c91d8 bbb4c1ea 97618a4b f03f4258 1948b2ee 4ee7ad67
	 */
	unsigned char ans2[SHA224_DIGESTSIZE] = {
		0x20, 0x79, 0x46, 0x55, 0x98, 0x0c, 0x91, 0xd8,
		0xbb, 0xb4, 0xc1, 0xea, 0x97, 0x61, 0x8a, 0x4b,
		0xf0, 0x3f, 0x42, 0x58, 0x19, 0x48, 0xb2, 0xee,
		0x4e, 0xe7, 0xad, 0x67
	};

	/* size 0 */
	unsigned char ans3[SHA224_DIGESTSIZE] = {
		0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9,
		0x47, 0x61, 0x02, 0xbb, 0x28, 0x82, 0x34, 0xc4,
		0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a,
		0xc5, 0xb3, 0xe4, 0x2f
	};
	int i;

	/*
	 * RFC3874
	 * 3.2.  Test Vector #2
	 */
	OK_SHA224(strlen(in), (unsigned char *)in, ret);
	if (memcmp(ret, ans1, SHA224_DIGESTSIZE)) {
		printf("error : test sha224 1\n");
		return -1;
	} else
		printf("test sha224 ok -- 1\n");

	/*
	 * RFC3874
	 * 3.3.  Test Vector #3
	 * 
	 * Let the message M be the binary-coded form of the ASCII string
	 * which consists of 1,000,000 repetitions of the character "a".
	 */
	memset(in2, 'a', 1000);
	SHA224init(&ctx);
	for (i = 0; i < 1000; i++)
		SHA224update(&ctx, in2, 1000);
	SHA224final(ret, &ctx);

	if (memcmp(ret, ans2, SHA224_DIGESTSIZE)) {
		printf("error : test sha224 2\n");
		return -1;
	} else
		printf("test sha224 ok -- 2\n");

	/* Test size 0 */
	OK_SHA224(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans3, SHA224_DIGESTSIZE)) {
		printf("error : test sha224 3\n");
		return -1;
	} else {
		printf("test sha224 ok -- 3\n");
	}

	return 0;
}

int test_sha256()
{
	SHA256_CTX ctx;
	unsigned char ret[SHA256_DIGESTSIZE], in2[1000];

	/*
	 * FIPS PUB 180-2
	 * B.2 SHA-256 Example (Multi-Block Message) 
	 */
	char in[] = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
	unsigned char ans1[SHA256_DIGESTSIZE] = {
		0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
		0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
		0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
		0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1
	};

	/*
	 * FIPS PUB 180-2
	 * B.3 SHA-256 Example (Long Message)
	 * 
	 * cdc76e5c 9914fb92 81a1c7e2 84d73e67
	 * f1809a48 a497200e 046d39cc c7112cd0
	 */
	unsigned char ans2[SHA256_DIGESTSIZE] = {
		0xcd, 0xc7, 0x6e, 0x5c, 0x99, 0x14, 0xfb, 0x92,
		0x81, 0xa1, 0xc7, 0xe2, 0x84, 0xd7, 0x3e, 0x67,
		0xf1, 0x80, 0x9a, 0x48, 0xa4, 0x97, 0x20, 0x0e,
		0x04, 0x6d, 0x39, 0xcc, 0xc7, 0x11, 0x2c, 0xd0
	};

	/* size 0 */
	unsigned char ans3[SHA256_DIGESTSIZE] = {
		0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
		0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
		0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
		0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55
	};
	int i;

	/*
	 * FIPS PUB 180-2
	 * B.2 SHA-256 Example (Multi-Block Message) 
	 */
	OK_SHA256(strlen(in), (unsigned char *)in, ret);
	if (memcmp(ret, ans1, SHA256_DIGESTSIZE)) {
		printf("error : test sha256 1\n");
		return -1;
	} else
		printf("test sha256 ok -- 1\n");

	/*
	 * FIPS PUB 180-2
	 * B.3 SHA-256 Example (Long Message) 
	 * 
	 * Let the message M be the binary-coded form of the ASCII string
	 * which consists of 1,000,000 repetitions of the character "a".
	 */
	memset(in2, 'a', 1000);
	SHA256init(&ctx);
	for (i = 0; i < 1000; i++)
		SHA256update(&ctx, in2, 1000);
	SHA256final(ret, &ctx);

	if (memcmp(ret, ans2, SHA256_DIGESTSIZE)) {
		printf("error : test sha256 2\n");
		return -1;
	} else
		printf("test sha256 ok -- 2\n");

	/* Test size 0 */
	OK_SHA256(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans3, SHA256_DIGESTSIZE)) {
		printf("error : test sha256 3\n");
		return -1;
	} else {
		printf("test sha256 ok -- 3\n");
	}

	return 0;
}

int test_sha384()
{
	SHA512_CTX ctx;
	unsigned char ret[SHA384_DIGESTSIZE], in2[1000];

	/*
	 * FIPS PUB 180-2
	 * D.2 SHA-384 Example (Multi-Block Message) 
	 * 
	 * 09330c33f71147e8 3d192fc782cd1b47 53111b173b3b05d2
	 * 2fa08086e3b0f712 fcc7c71a557e2db9 66c3e9fa91746039
	 */
	char in[] =
	    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
	    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
	unsigned char ans1[SHA384_DIGESTSIZE] = {
		0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
		0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
		0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
		0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
		0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
		0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39
	};

	/*
	 * FIPS PUB 180-2
	 * D.3 SHA-384 Example (Long Message) 
	 * 
	 * 9d0e1809716474cb 086e834e310a4a1c ed149e9c00f24852
	 * 7972cec5704c2a5b 07b8b3dc38ecc4eb ae97ddd87f3d8985
	 */
	unsigned char ans2[SHA384_DIGESTSIZE] = {
		0x9d, 0x0e, 0x18, 0x09, 0x71, 0x64, 0x74, 0xcb,
		0x08, 0x6e, 0x83, 0x4e, 0x31, 0x0a, 0x4a, 0x1c,
		0xed, 0x14, 0x9e, 0x9c, 0x00, 0xf2, 0x48, 0x52,
		0x79, 0x72, 0xce, 0xc5, 0x70, 0x4c, 0x2a, 0x5b,
		0x07, 0xb8, 0xb3, 0xdc, 0x38, 0xec, 0xc4, 0xeb,
		0xae, 0x97, 0xdd, 0xd8, 0x7f, 0x3d, 0x89, 0x85
	};

	/* size 0 */
	unsigned char ans3[SHA384_DIGESTSIZE] = {
		0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
		0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
		0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
		0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
		0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
		0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b
	};
	int i;

	/*
	 * FIPS PUB 180-2
	 * D.2 SHA-384 Example (Multi-Block Message) 
	 * 
	 * Let the message, M, be the 896-bit ( l = 896) ASCII string 
	 * "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn 
	 * hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu". 
	 */
	OK_SHA384(strlen(in), (unsigned char *)in, ret);
	if (memcmp(ret, ans1, SHA384_DIGESTSIZE)) {
		printf("error : test sha384 1\n");
		return -1;
	} else
		printf("test sha384 ok -- 1\n");

	/*
	 * FIPS PUB 180-2
	 * D.3 SHA-384 Example (Long Message) 
	 * 
	 * Let the message M be the binary-coded form of the ASCII string
	 * which consists of 1,000,000 repetitions of the character "a".
	 */
	memset(in2, 'a', 1000);
	SHA384init(&ctx);
	for (i = 0; i < 1000; i++)
		SHA384update(&ctx, in2, 1000);
	SHA384final(ret, &ctx);

	if (memcmp(ret, ans2, SHA384_DIGESTSIZE)) {
		printf("error : test sha384 2\n");
		return -1;
	} else
		printf("test sha384 ok -- 2\n");

	/* Test size 0 */
	OK_SHA384(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans3, SHA384_DIGESTSIZE)) {
		printf("error : test sha384 3\n");
		return -1;
	} else {
		printf("test sha384 ok -- 3\n");
	}

	return 0;
}

int test_sha512()
{
	SHA512_CTX ctx;
	unsigned char ret[SHA512_DIGESTSIZE], in2[1000];

	/*
	 * FIPS PUB 180-2
	 * C.2 SHA-512 Example (Multi-Block Message) 
	 * 
	 * 8e959b75dae313da 8cf4f72814fc143f 8f7779c6eb9f7fa1
	 * 7299aeadb6889018 501d289e4900f7e4 331b99dec4b5433a
	 * c7d329eeb6dd2654 5e96e55b874be909
	 */
	char in[] =
	    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
	    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
	unsigned char ans1[SHA512_DIGESTSIZE] = {
		0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
		0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
		0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
		0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
		0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
		0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
		0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
		0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09
	};

	/*
	 * FIPS PUB 180-2
	 * C.3 SHA-512 Example (Long Message) 
	 * 
	 * e718483d0ce76964 4e2e42c7bc15b463 8e1f98b13b204428
	 * 5632a803afa973eb de0ff244877ea60a 4cb0432ce577c31b
	 * eb009c5c2c49aa2e 4eadb217ad8cc09b
	 */
	unsigned char ans2[SHA512_DIGESTSIZE] = {
		0xe7, 0x18, 0x48, 0x3d, 0x0c, 0xe7, 0x69, 0x64,
		0x4e, 0x2e, 0x42, 0xc7, 0xbc, 0x15, 0xb4, 0x63,
		0x8e, 0x1f, 0x98, 0xb1, 0x3b, 0x20, 0x44, 0x28,
		0x56, 0x32, 0xa8, 0x03, 0xaf, 0xa9, 0x73, 0xeb,
		0xde, 0x0f, 0xf2, 0x44, 0x87, 0x7e, 0xa6, 0x0a,
		0x4c, 0xb0, 0x43, 0x2c, 0xe5, 0x77, 0xc3, 0x1b,
		0xeb, 0x00, 0x9c, 0x5c, 0x2c, 0x49, 0xaa, 0x2e,
		0x4e, 0xad, 0xb2, 0x17, 0xad, 0x8c, 0xc0, 0x9b
	};

	/* size 0 */
	unsigned char ans3[SHA512_DIGESTSIZE] = {
		0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
		0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
		0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
		0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
		0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
		0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
		0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
		0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e
	};
	int i;

	/*
	 * FIPS PUB 180-2
	 * C.2 SHA-512 Example (Multi-Block Message) 
	 * 
	 * Let the message, M, be the 896-bit (l = 896) ASCII string 
	 * "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn 
	 * hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu". 
	 */
	OK_SHA512(strlen(in), (unsigned char *)in, ret);

	if (memcmp(ret, ans1, SHA512_DIGESTSIZE)) {
		printf("error : test sha512 1\n");
		return -1;
	} else
		printf("test sha512 ok -- 1\n");

	/*
	 * FIPS PUB 180-2
	 * C.3 SHA-512 Example (Long Message) 
	 * 
	 * Let the message M be the binary-coded form of the ASCII string
	 * which consists of 1,000,000 repetitions of the character "a".
	 */
	memset(in2, 'a', 1000);
	SHA512init(&ctx);
	for (i = 0; i < 1000; i++)
		SHA512update(&ctx, in2, 1000);
	SHA512final(ret, &ctx);

	if (memcmp(ret, ans2, SHA512_DIGESTSIZE)) {
		printf("error : test sha512 2\n");
		return -1;
	} else
		printf("test sha512 ok -- 2\n");

	/* Test size 0 */
	OK_SHA512(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans3, SHA512_DIGESTSIZE)) {
		printf("error : test sha512 3\n");
		return -1;
	} else {
		printf("test sha512 ok -- 3\n");
	}

	return 0;
}

int test_sha512224()
{
	unsigned char ret[SHA512224_DIGESTSIZE];

	/*
	 * FIPS PUB 180-4
	 * A.2 Implementation Notes
	 * http://csrc.nist.gov/groups/ST/toolkit/examples.html
	 *
	 * SHA-512/224: Two Block Message Sample
	 * 23FEC5BB 94D60B23 30819264 0B0C4533
	 * 35D66473 4FE40E72 68674AF9
	 */
	char in[] =
	    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
	    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
	unsigned char ans1[SHA512224_DIGESTSIZE] = {
		0x23, 0xfe, 0xc5, 0xbb, 0x94, 0xd6, 0x0b, 0x23,
		0x30, 0x81, 0x92, 0x64, 0x0b, 0x0c, 0x45, 0x33,
		0x35, 0xd6, 0x64, 0x73, 0x4f, 0xe4, 0x0e, 0x72,
		0x68, 0x67, 0x4a, 0xf9
	};
	/* size 0 */
	unsigned char ans2[SHA512224_DIGESTSIZE] = {
		0x6e, 0xd0, 0xdd, 0x02, 0x80, 0x6f, 0xa8, 0x9e,
		0x25, 0xde, 0x06, 0x0c, 0x19, 0xd3, 0xac, 0x86,
		0xca, 0xbb, 0x87, 0xd6, 0xa0, 0xdd, 0xd0, 0x5c,
		0x33, 0x3b, 0x84, 0xf4
	};

	OK_SHA512224(strlen(in), (unsigned char *)in, ret);
	if (memcmp(ret, ans1, SHA512224_DIGESTSIZE)) {
		printf("error : test sha512/224 1\n");
		return -1;
	} else
		printf("test sha512/224 ok -- 1\n");

	/* Test size 0 */
	OK_SHA512224(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans2, SHA512224_DIGESTSIZE)) {
		printf("error : test sha512/224 2\n");
		return -1;
	} else {
		printf("test sha512/224 ok -- 2\n");
	}

	return 0;
}

int test_sha512256()
{
	unsigned char ret[SHA512256_DIGESTSIZE];

	/*
	 * FIPS PUB 180-4
	 * A.2 Implementation Notes
	 * http://csrc.nist.gov/groups/ST/toolkit/examples.html
	 *
	 * SHA-512/256: Two Block Message Sample
	 * 3928E184 FB8690F8 40DA3988 121D31BE 65CB9D3E
	 * F83EE614 6FEAC861 E19B563A
	 */
	char in[] =
	    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
	    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
	unsigned char ans1[SHA512256_DIGESTSIZE] = {
		0x39, 0x28, 0xe1, 0x84, 0xfb, 0x86, 0x90, 0xf8,
		0x40, 0xda, 0x39, 0x88, 0x12, 0x1d, 0x31, 0xbe,
		0x65, 0xcb, 0x9d, 0x3e, 0xf8, 0x3e, 0xe6, 0x14,
		0x6f, 0xea, 0xc8, 0x61, 0xe1, 0x9b, 0x56, 0x3a
	};
	/* size 0 */
	unsigned char ans2[SHA512256_DIGESTSIZE] = {
		0xc6, 0x72, 0xb8, 0xd1, 0xef, 0x56, 0xed, 0x28,
		0xab, 0x87, 0xc3, 0x62, 0x2c, 0x51, 0x14, 0x06,
		0x9b, 0xdd, 0x3a, 0xd7, 0xb8, 0xf9, 0x73, 0x74,
		0x98, 0xd0, 0xc0, 0x1e, 0xce, 0xf0, 0x96, 0x7a
	};

	OK_SHA512256(strlen(in), (unsigned char *)in, ret);
	if (memcmp(ret, ans1, SHA512256_DIGESTSIZE)) {
		printf("error : test sha512/256 1\n");
		return -1;
	} else
		printf("test sha512/256 ok -- 1\n");

	/* Test size 0 */
	OK_SHA512256(0, (unsigned char *)in, ret);
	if (memcmp(ret, ans2, SHA512256_DIGESTSIZE)) {
		printf("error : test sha512/256 2\n");
		return -1;
	} else {
		printf("test sha512/256 ok -- 2\n");
	}

	return 0;
}
