/* sha1.c */
/*
 * Modified by National Institute of Informatics in Japan, 2011-2015.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <aicrypto/ok_sha1.h>

/**
 * Logical functions and constants defined in RFC 3174, Sec.5.
 * For section number, the same applies hereafter.
 */
#define f1(B,C,D)	(((B)&(C))|((~(B))&(D)))	/* 0  <= t <= 19 */
#define f2(B,C,D)	((B)^(C)^(D))			/* 20 <= t <= 39 */
#define f3(B,C,D)	(((B)&(C))|((B)&(D))|((C)&(D)))	/* 40 <= t <= 59 */
#define f4(B,C,D) 	((B)^(C)^(D))			/* 60 <= t <= 79 */

#define K1	0x5A827999	/* 0  <= t <= 19 */
#define K2	0x6ED9EBA1	/* 20 <= t <= 39 */
#define K3	0x8F1BBCDC	/* 40 <= t <= 59 */
#define K4	0xCA62C1D6	/* 60 <= t <= 79 */

/**
 * The circular left shift operation defined in Sec.3, (c).
 */
#define S(X,n)	(((X)<<n)|((X)>>(32-n)))

/**
 * === DRAFT ===
 * Initial values of H's defined in Sec.6.1.
 */
static const uint32_t initH[] = {
	0x67452301, 0xEFCDAB89, 0x98BADCFE, 0x10325476, 0xC3D2E1F0
};

static void sha1_trans(uint32_t *w, uint32_t *H);
static void set_w(unsigned char *in, uint32_t *w, size_t n);
static void calc_w(uint32_t *w);
static void set_length(uint32_t len[2], uint32_t *w);

void uc2ul(unsigned char *in, uint32_t *w, size_t n);
void ul2uc(uint32_t *w, unsigned char *ret);

/**
 * === DRAFT ===
 * Compute the message digest of a message.
 * The message digest is a 160-bit string and stored in 'ret'.
 * Note that the buffer 'in' must contain all of the message.
 */
void OK_SHA1(unsigned int len, unsigned char *in, unsigned char *ret)
{
	SHA1_CTX ctx;

	if (in == NULL)
		return;

	SHA1init(&ctx);
	SHA1update(&ctx, in, len);
	SHA1final(ret, &ctx);
}

/**
 * === DRAFT ===
 * Initialize SHA1_CTX.
 */
void SHA1init(SHA1_CTX *ctx)
{
	int i;

	for (i = 0; i < 5; i++)
		ctx->H[i] = initH[i];
	ctx->len[0] = ctx->len[1] = 0;
	ctx->mod = 0;
}

/**
 * === DRAFT ===
 * Compute the intermediate hash (five 32-bit words) H[].
 *
 * @param[in] len length of the input buffer in bytes.
 */
void SHA1update(SHA1_CTX *ctx, unsigned char *in, unsigned int len)
{
	uint32_t w[80];
	uint32_t *H;
	unsigned char *dat;
	int i, mod, tmp;

	if (in == NULL)
		return;

	H = ctx->H;

	/* update the length in bits. */
	if ((ctx->len[0] += ((uint32_t)len << 3)) < ((uint32_t)len << 3))
		ctx->len[1]++;
	ctx->len[1] += ((uint32_t)len >> 29);

	dat = ctx->dat;
	mod = ctx->mod;

	if (len + mod <= SHA1_BLOCKSIZE) {
		ctx->mod = len + mod;
		memcpy(&dat[mod], in, len);
	} else {
		memcpy(&dat[mod], in, SHA1_BLOCKSIZE - mod);
		set_w(dat, w, SHA1_BLOCKSIZE);
		calc_w(w);
		sha1_trans(w, H);

		tmp = len - SHA1_BLOCKSIZE;
		for (i = SHA1_BLOCKSIZE - mod; i < tmp; i += SHA1_BLOCKSIZE) {
			set_w(&in[i], w, SHA1_BLOCKSIZE);
			calc_w(w);
			sha1_trans(w, H);
		}
		ctx->mod = len - i;
		memcpy(dat, &in[i], ctx->mod);
	}
}

/**
 * === DRAFT ===
 * Compute the message digest by using the last message block M(n)
 * and the result of SHA1update().
 */
void SHA1final(unsigned char *ret, SHA1_CTX *ctx)
{
	uint32_t w[80];
	uint32_t *H;
	int mod;

	H = ctx->H;
	mod = ctx->mod;

	/* M(n-1) or M(n)? */
	set_w(ctx->dat, w, mod);
	/*
	 * check whether 2-word representation of length fits in a block.
	 *   mod + 8 < SHA1_BLOCKSIZE (in bytes)
	 */
	if (mod + 8 >= SHA1_BLOCKSIZE) {
		calc_w(w);
		sha1_trans(w, H);

		/* clear w[] */
		memset(w, 0, SHA1_BLOCKSIZE);
		if (mod == SHA1_BLOCKSIZE)
			w[0] = (uint32_t) 0x80000000;
	}
	/* M(n) */	
	set_length(ctx->len, w);
	calc_w(w);
	sha1_trans(w, H);
	ul2uc(ctx->H, ret);
}

/*
 * Type conversion between uint8 and uint32 strings
 */
/**
 * === DRAFT ===
 * Convert uint8 to uint32.
 *
 * This function converts an input unsigned 8-bit strings
 * to unsigned 32-bit words. Here, the word "long" is merely used
 * as 32-bit integer.
 *
 * @param[in] in input 8-bit strings.
 * @param[out] w output 32-bit words.
 * @param[in] n length of input string in bytes. this must be a multiple of 4.
 */
void uc2ul(unsigned char *in, uint32_t *w, size_t n)
{
	int i, j;
	for (i = 0, j = 0; j < n; i++, j += 4)
		w[i] = ((uint32_t)in[j  ] << 24) |
		       ((uint32_t)in[j+1] << 16) |
		       ((uint32_t)in[j+2] <<  8) |
		       ((uint32_t)in[j+3]);
}
/**
 * === DRAFT ===
 * Convert uint32 to uint8.
 *
 * This function converts five input unsigned 32-bit words
 * to unsigned 8-bit strings.
 *
 * @param[in] H pointer to five 32-bit words.
 * @param[out] ret returns the 160-bit string.
 */
void ul2uc(uint32_t *H, unsigned char *ret)
{
	int i, j;
	for (i = j = 0; i < 5; i++, j += 4) {
		ret[j  ] = (unsigned char)(H[i] >> 24);
		ret[j+1] = (unsigned char)(H[i] >> 16);
		ret[j+2] = (unsigned char)(H[i] >>  8);
		ret[j+3] = (unsigned char) H[i];
	}
}

/**
 * === DRAFT ===
 * Divide M(i) into 16 words W(0), ..., W(15).
 *
 * Append "1" to the message block if the size of the message block
 * is less than 64 bytes. (Sec.4, (a))
 *
 * @param[in] in input 8-bit strings.
 * @param[out] w (eighty) output 32-bit words.
 * @param[in] n size of input buffer in bytes.
 */
void set_w(unsigned char *in, uint32_t *w, size_t n)
{
	int quo, mod;

	/* clear w */
	memset(w, 0, SHA1_BLOCKSIZE);	/* sixteen 32-bit words (64-bytes) */

	/* set w[0], ..., w[15] */
	if (n == SHA1_BLOCKSIZE) {
		uc2ul(in, w, n);
	} else {
		quo = n / 4;	/* quotient in words */
		mod = n % 4;	/* in bytes */
		uc2ul(in, w, n - mod);

		/* Append "1" */
		switch (mod) {
		case 0:
			w[quo] = (uint32_t) 0x80000000;
			break;
		case 1:
			w[quo] = ((uint32_t)in[n-1] << 24) |
				  (uint32_t) 0x800000;
			break;
		case 2:
			w[quo] = ((uint32_t)in[n-2] << 24) |
				 ((uint32_t)in[n-1] << 16) |
				  (uint32_t) 0x8000;
			break;
		case 3:
			w[quo] = ((uint32_t)in[n-3] << 24) |
				 ((uint32_t)in[n-2] << 16) |
				 ((uint32_t)in[n-1] <<  8) |
				  (uint32_t) 0x80;
			break;
		}
	}
}

/**
 * Perform step (b) in Sec.6.1.
 */
static void calc_w(uint32_t *w)
{
	int t;
	for (t = 16; t < 80; t++)
		w[t] = S(w[t-3] ^ w[t-8] ^ w[t-14] ^ w[t-16], 1);
}
/**
 * Set the length of the original massage.
 */
static void set_length(uint32_t len[2], uint32_t *w)
{
	w[14] = len[1];
	w[15] = len[0];
}

/**
 * Perform steps (c), (d) and (e) in Sec.6.1.
 */
static void sha1_trans(uint32_t *w, uint32_t *H)
{
	uint32_t A, B, C, D, E, TEMP;
	int t;

	A = H[0];
	B = H[1];
	C = H[2];
	D = H[3];
	E = H[4];

	for (t = 0; t < 20; t++) {
		TEMP = S(A, 5) + f1(B, C, D) + E + w[t] + K1;
		E = D;
		D = C;
		C = S(B, 30);
		B = A;
		A = TEMP;
#ifdef DEBUG
		/*
		 * We can easily compare the following with
		 * http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA1.pdf
		 */
		printf("t=%.2d: %.8x  %.8x  %.8x  %.8x  %.8x\n",t,A,B,C,D,E);
#endif
	}
	for (; t < 40; t++) {
		TEMP = S(A, 5) + f2(B, C, D) + E + w[t] + K2;
		E = D;
		D = C;
		C = S(B, 30);
		B = A;
		A = TEMP;
	}
	for (; t < 60; t++) {
		TEMP = S(A, 5) + f3(B, C, D) + E + w[t] + K3;
		E = D;
		D = C;
		C = S(B, 30);
		B = A;
		A = TEMP;
	}
	for (; t < 80; t++) {
		TEMP = S(A, 5) + f4(B, C, D) + E + w[t] + K4;
		E = D;
		D = C;
		C = S(B, 30);
		B = A;
		A = TEMP;
	}

	H[0] += A;
	H[1] += B;
	H[2] += C;
	H[3] += D;
	H[4] += E;
}
