/* siggenPSS_3_TruncatedSHAs_2048_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-3.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 * SigGenPSS_186-3_TruncatedSHAs.txt
 */ 
#include "siggenPSStest.h"

static const unsigned short sn1[] = {
	0xabf4, 0x4a77, 0x0836, 0xfee4, 0x34fc, 0x7bf1, 0x7b06, 0x428a,
	0x7cb5, 0x0d27, 0x213f, 0xacac, 0x88fb, 0x38d3, 0xa29e, 0xe816,
	0x69d1, 0x8e71, 0xdaa3, 0xc78f, 0xd897, 0x74a2, 0x23e7, 0xa15d,
	0xf421, 0x6290, 0xc4fd, 0xa8e1, 0x5a00, 0x088d, 0x4071, 0xb682,
	0xf312, 0x1c45, 0xe25d, 0x93b0, 0x3b23, 0x32a9, 0xf1a0, 0x7720,
	0x1854, 0x8302, 0xf60b, 0x2d36, 0x6528, 0xb236, 0x4a89, 0x62c8,
	0xbf0a, 0xb64e, 0x87e1, 0xd6ae, 0x1343, 0x2038, 0xe3f9, 0xe9fd,
	0x8eb3, 0x7aef, 0x736a, 0x1998, 0x7c25, 0x9106, 0xf4cb, 0xbcc4,
	0xed6a, 0x4d67, 0x9fa9, 0x4224, 0x8909, 0x2cc4, 0xf12b, 0x17d9,
	0x74fe, 0x7994, 0x7799, 0x5b8c, 0xfa34, 0x94bc, 0x5b42, 0x6ec3,
	0xb2d8, 0xc023, 0x7396, 0x9630, 0xc003, 0xe08a, 0x42e5, 0xe620,
	0x72df, 0xcb11, 0x83ea, 0x5364, 0x513e, 0x6e35, 0x7bd6, 0xc47d,
	0x69c4, 0x1767, 0xa304, 0xb7e7, 0xba11, 0x5f34, 0x8963, 0xb49b,
	0x02f3, 0x4c87, 0xe17f, 0x5fdd, 0x2967, 0x4dc1, 0x25d3, 0x33a4,
	0xc0c6, 0xba96, 0xa5b8, 0xafb4, 0xe4f7, 0x85d2, 0xe9ee, 0xcb01,
	0x65e7, 0xf2bb, 0xb464, 0xe2a6, 0x5f03, 0xbaec, 0x25d7, 0xf519,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x009b, 0x34af,
};

static const unsigned short sd1[] = {
	0x1fd0, 0x4b94, 0xcdfa, 0x3cdb, 0x2250, 0x82a2, 0x61b1, 0x2b73,
	0x9ae7, 0xc359, 0x148a, 0xcd5a, 0xc089, 0x9b54, 0x347e, 0xea93,
	0xe137, 0x5aac, 0x688e, 0x3a2e, 0x6a40, 0xa4f4, 0x0101, 0x26ed,
	0xf569, 0xa039, 0x566b, 0x3157, 0x84bb, 0x6a3b, 0x3e37, 0x0e89,
	0x694f, 0x4e6d, 0x9460, 0x8df9, 0xe1fc, 0x086e, 0xb9b3, 0xc283,
	0xfcb7, 0xaaff, 0x7011, 0xe2d5, 0xa6f0, 0xaa56, 0xf2fd, 0x64d7,
	0xdd75, 0xabce, 0x5e37, 0x247c, 0x6ccc, 0xec7c, 0xb04e, 0x0b5e,
	0x5970, 0x20c7, 0xf63c, 0xce7d, 0xd7f8, 0x4564, 0xa6c0, 0xfa8d,
	0x7381, 0x3d18, 0x7878, 0x271d, 0x74cf, 0x8b69, 0x6ff6, 0x7620,
	0x4847, 0xcffa, 0x599f, 0x61cc, 0x3ae3, 0x9fc1, 0x805b, 0xe01f,
	0xa230, 0x8011, 0xb7b1, 0xf1c3, 0x9be8, 0x7869, 0xb7fa, 0xea7b,
	0xf22d, 0x7fb2, 0xadff, 0xee26, 0xb364, 0x0145, 0x54f6, 0xcf31,
	0xe98d, 0x3fb6, 0x8175, 0x731a, 0xf233, 0x5504, 0x8a22, 0xe9a3,
	0xde43, 0x2fe4, 0x34f8, 0x8cf0, 0x689a, 0xaba5, 0xaefb, 0xff08,
	0xb9d9, 0x96d9, 0x2152, 0xa5ba, 0xb5a6, 0x50c7, 0x3a1d, 0xb065,
	0xb8fc, 0xd4a9, 0xe260, 0xc69b, 0x7c80, 0x6cad, 0xda8b, 0x0c4b,
};

#define HAS_SHA512224_TESTVECTOR

#define HAS_SHA512256_TESTVECTOR

/* 10 elements */
static mod2048_t SHA512224[] = {
	/* SHA512224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x87, 0x7b, 0x48, 0x92, 0x8e, 0x3f, 0xae, 0x87,
			0x7f, 0x43, 0x61, 0x79, 0xc7, 0x4f, 0xd2, 0xa7,
			0xfb, 0xb3, 0xe4, 0xf7, 0xbe, 0x96, 0xf2, 0xe5,
			0xe6, 0x8b, 0x14, 0xbf, 0xda, 0x65, 0x9e, 0x14,
			0x3e, 0x4d, 0xea, 0x6a, 0xd3, 0x11, 0x58, 0x2f,
			0x98, 0xbf, 0x92, 0xf2, 0x36, 0x73, 0x15, 0xd3,
			0x9a, 0x6d, 0xed, 0x38, 0xe4, 0x66, 0xbf, 0xc2,
			0x80, 0x24, 0x0d, 0x80, 0x92, 0x4d, 0xb6, 0x62,
			0x98, 0x79, 0xbc, 0xf0, 0x4b, 0x8a, 0x4a, 0x3b,
			0xc5, 0x40, 0xf6, 0x14, 0xe9, 0x08, 0x43, 0x7d,
			0x00, 0x15, 0x6a, 0x5d, 0xa0, 0xb2, 0x5a, 0x78,
			0xd2, 0xd7, 0xfd, 0x9d, 0x73, 0xfa, 0xbd, 0x21,
			0xdb, 0x27, 0x47, 0x8d, 0xbd, 0x53, 0x8e, 0x94,
			0x4e, 0xee, 0x13, 0x48, 0xaa, 0xb7, 0x65, 0x8d,
			0x73, 0xd4, 0xe3, 0xcd, 0xbb, 0x22, 0xc9, 0xed,
			0x24, 0xca, 0x6b, 0x80, 0x45, 0x15, 0xbb, 0x3b,
		},
		/* S0 : 256 byte */
		{
			0xa0, 0x85, 0x96, 0xd4, 0x3d, 0x48, 0x3c, 0x01,
			0xd9, 0x59, 0x69, 0x1c, 0x1b, 0x8d, 0x5b, 0xb2,
			0x2b, 0xb6, 0xe1, 0xbd, 0xb2, 0x50, 0xa8, 0xa4,
			0xd7, 0xd4, 0xc1, 0x92, 0xe3, 0x33, 0x89, 0xba,
			0xf5, 0x32, 0x8e, 0x84, 0xd8, 0x26, 0x21, 0x15,
			0xd5, 0xed, 0x0e, 0xb1, 0xbf, 0xaa, 0x50, 0x92,
			0x85, 0xdc, 0xd1, 0x7b, 0xb4, 0xda, 0x68, 0x4e,
			0x3f, 0xab, 0x33, 0x9b, 0xf3, 0x56, 0xd7, 0x4a,
			0x26, 0xe7, 0x85, 0xe1, 0xc4, 0xe6, 0xb2, 0xb1,
			0x75, 0x18, 0xed, 0x0f, 0xe7, 0x96, 0xc8, 0x6e,
			0xd0, 0xb5, 0xbe, 0x64, 0xb5, 0x71, 0xf0, 0xab,
			0xf1, 0x96, 0x67, 0xe9, 0x86, 0x56, 0x05, 0x4e,
			0xdd, 0x3b, 0x1b, 0xd2, 0x0c, 0x6a, 0xed, 0xb1,
			0x7b, 0x39, 0xaa, 0xbf, 0xc5, 0x54, 0x26, 0x73,
			0xfb, 0x73, 0xbd, 0x96, 0x14, 0xea, 0x37, 0xac,
			0x4d, 0xfd, 0x22, 0xdb, 0xdb, 0x1a, 0x55, 0xd3,
			0x02, 0x89, 0x84, 0xe6, 0x54, 0x75, 0xd1, 0x4c,
			0xaa, 0xf7, 0x08, 0x80, 0x04, 0x55, 0x5a, 0xdc,
			0x39, 0xb3, 0xca, 0xd9, 0x15, 0x26, 0xb3, 0x81,
			0x2c, 0x94, 0x3c, 0x25, 0x04, 0xbc, 0x91, 0x45,
			0xb7, 0x84, 0xf8, 0xae, 0x53, 0x4b, 0x6a, 0xc6,
			0xd1, 0x39, 0x5b, 0xea, 0xaa, 0x3d, 0x80, 0x46,
			0x7b, 0xf9, 0xb8, 0x3e, 0xc0, 0xcd, 0x2e, 0xdf,
			0x31, 0xe5, 0x0c, 0x58, 0x2c, 0xd0, 0x35, 0xb4,
			0x42, 0x15, 0x77, 0x93, 0x93, 0x54, 0x3e, 0x25,
			0x0d, 0xbf, 0x1c, 0x78, 0x9e, 0x24, 0x8b, 0xf9,
			0x80, 0xfe, 0x4e, 0xcf, 0xe2, 0x3c, 0x84, 0xc7,
			0x82, 0x30, 0xdb, 0x16, 0x9a, 0x3b, 0x1f, 0x29,
			0x59, 0xf6, 0x3c, 0xc6, 0xa6, 0x95, 0x0c, 0xbb,
			0x30, 0xc9, 0x1f, 0x7b, 0xd6, 0x8f, 0xe8, 0x29,
			0x8c, 0x3c, 0x16, 0x19, 0xaf, 0x9e, 0xe4, 0x13,
			0x16, 0x9e, 0x59, 0x42, 0x62, 0x6b, 0xfa, 0x92,
		},
		0, /* salt length */
		/* V0 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6c, 0x2c, 0xdf, 0x5f, 0xcb, 0xb2, 0xa7, 0x35,
			0xf7, 0x60, 0x45, 0xa5, 0x0c, 0x35, 0xcc, 0x43,
			0xdb, 0x0c, 0xb8, 0x6b, 0xe0, 0xc4, 0x80, 0x85,
			0x75, 0x9d, 0x6f, 0x88, 0x44, 0xe5, 0xaf, 0xbd,
			0x02, 0xd5, 0xa7, 0xc6, 0x5b, 0xb6, 0x87, 0xde,
			0x28, 0x47, 0x20, 0x62, 0xe8, 0x63, 0xfa, 0x91,
			0x6e, 0xdf, 0xa0, 0x90, 0xd7, 0xb2, 0x3f, 0xb0,
			0xbf, 0x58, 0x6f, 0xf4, 0xb5, 0x78, 0x51, 0xce,
			0xeb, 0xc4, 0xda, 0x2a, 0x32, 0x20, 0x6e, 0xad,
			0x26, 0x07, 0xb1, 0x3f, 0xa6, 0x2b, 0x9a, 0xc7,
			0x88, 0x5c, 0xc9, 0x35, 0x8a, 0xab, 0x1a, 0xc6,
			0x8b, 0xcf, 0x7a, 0x83, 0xfb, 0xc8, 0xf9, 0x0c,
			0x90, 0xc2, 0x21, 0x93, 0x3a, 0x3b, 0x86, 0x2b,
			0x5a, 0x68, 0x9f, 0x43, 0x2f, 0xd5, 0xd4, 0x6d,
			0x90, 0x4d, 0xd8, 0x65, 0xe1, 0xf9, 0x39, 0x4e,
			0x40, 0xcb, 0x36, 0x3e, 0xff, 0x1e, 0xd0, 0xfd,
		},
		/* S1 : 256 byte */
		{
			0x2c, 0x97, 0x96, 0xe8, 0x96, 0xb6, 0xa0, 0x34,
			0xad, 0x2b, 0xd3, 0x9f, 0xf8, 0x5f, 0xbd, 0x27,
			0x64, 0x28, 0x0f, 0x38, 0xe4, 0xa4, 0xe7, 0xea,
			0x7e, 0x36, 0x9e, 0xf6, 0x5b, 0x5c, 0xc4, 0x32,
			0x26, 0xec, 0x96, 0x15, 0xbe, 0x32, 0x83, 0x30,
			0xb2, 0xe5, 0x98, 0xab, 0x6e, 0x40, 0x78, 0x08,
			0xc4, 0x4c, 0xce, 0x49, 0xc8, 0x9d, 0x02, 0xad,
			0xef, 0x34, 0x9a, 0xf2, 0x4a, 0xc5, 0xc2, 0x79,
			0xe1, 0xea, 0x6e, 0xb5, 0xd7, 0x93, 0x0a, 0x61,
			0x25, 0x59, 0xec, 0x86, 0x07, 0xe1, 0x59, 0xbe,
			0x5d, 0xf9, 0x5f, 0x6d, 0xf6, 0x22, 0x7a, 0x4d,
			0x55, 0xa6, 0xb4, 0xe7, 0x5b, 0xe5, 0xb7, 0xca,
			0x59, 0xf9, 0x3c, 0x7c, 0x93, 0xd2, 0xc1, 0xfa,
			0x1f, 0x25, 0xc4, 0xac, 0xf8, 0xe5, 0x5f, 0xc9,
			0x20, 0xd4, 0xe2, 0x1b, 0x84, 0xad, 0x71, 0xd4,
			0x60, 0x1e, 0x3c, 0xf6, 0x8c, 0xbc, 0xa1, 0xb4,
			0x1e, 0xcd, 0xd2, 0xb7, 0x70, 0x13, 0x4d, 0x33,
			0x6e, 0x6a, 0x9c, 0x02, 0xe4, 0x53, 0x1f, 0xc4,
			0xbf, 0x8b, 0xbf, 0xd1, 0x57, 0xe1, 0xd5, 0x0a,
			0x63, 0x3e, 0x0b, 0x22, 0xc4, 0x9e, 0xba, 0x6d,
			0xbf, 0xad, 0x27, 0x43, 0xd0, 0xf5, 0x86, 0xd4,
			0x9c, 0x9c, 0xd7, 0x4b, 0xbe, 0x72, 0x5f, 0x66,
			0x33, 0xfd, 0xdf, 0xc7, 0x3b, 0x06, 0xfd, 0xf4,
			0xc8, 0x67, 0x3d, 0x4c, 0x72, 0x3e, 0x0d, 0x81,
			0xe5, 0x57, 0x8e, 0x18, 0x6a, 0xf0, 0x59, 0xe1,
			0xe6, 0x21, 0x97, 0xe2, 0x70, 0xf7, 0xfc, 0xcd,
			0x97, 0x50, 0x6b, 0x5d, 0x9b, 0xcf, 0x9e, 0x17,
			0x46, 0x3e, 0xfe, 0x6a, 0xc1, 0x5e, 0x3f, 0x1b,
			0x7f, 0xe2, 0xad, 0xdb, 0x1d, 0xd4, 0x97, 0x32,
			0x57, 0x56, 0x76, 0xbb, 0xee, 0xf8, 0x33, 0x9c,
			0xe9, 0x6a, 0x88, 0x00, 0x65, 0x7d, 0x1d, 0x7b,
			0xf8, 0xad, 0xb9, 0xce, 0x5f, 0xf3, 0x80, 0xaf,
		},
		0, /* salt length */
		/* V1 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x64, 0xbe, 0x9e, 0x23, 0x54, 0xf1, 0x3b, 0xec,
			0x61, 0xd7, 0xa5, 0x9a, 0xc9, 0x83, 0x31, 0x88,
			0x90, 0xbe, 0x15, 0x5d, 0xcf, 0x07, 0x64, 0x31,
			0x4f, 0xd8, 0x40, 0xf1, 0x56, 0x2b, 0x4b, 0x04,
			0xc1, 0x3d, 0x5e, 0x2e, 0x53, 0x92, 0x7e, 0xb5,
			0x4d, 0x39, 0x0b, 0xc3, 0x3f, 0x51, 0x3b, 0x0c,
			0x1d, 0xcc, 0xc7, 0xb0, 0x95, 0x73, 0xe4, 0x63,
			0xa0, 0x2a, 0x75, 0xcb, 0x58, 0x76, 0x7f, 0x29,
			0x09, 0x3c, 0xdd, 0x3c, 0x8a, 0xcc, 0x35, 0x63,
			0xce, 0x1d, 0x2a, 0x01, 0xb6, 0x5a, 0xd3, 0x25,
			0x29, 0x9e, 0x6c, 0x6b, 0x67, 0xfd, 0x4c, 0x1d,
			0x9a, 0xc3, 0x17, 0x9e, 0xaf, 0x00, 0x35, 0x09,
			0x63, 0x44, 0x83, 0x15, 0xa1, 0xa7, 0x48, 0x3c,
			0x0b, 0x0c, 0x6a, 0x1c, 0xd7, 0xa7, 0xe2, 0x1d,
			0xdc, 0xbd, 0x6d, 0x54, 0xec, 0xab, 0x84, 0xa7,
			0x65, 0x2a, 0x8f, 0x32, 0x03, 0xd1, 0x56, 0xea,
		},
		/* S2 : 256 byte */
		{
			0x57, 0x45, 0x7c, 0x84, 0x41, 0x56, 0xd5, 0xe6,
			0x08, 0x57, 0x9f, 0x11, 0x7e, 0x87, 0xf9, 0x72,
			0x1d, 0xfb, 0x86, 0xc7, 0x2d, 0xe9, 0xb4, 0x64,
			0xdf, 0xf6, 0x02, 0x01, 0xe8, 0x1a, 0x50, 0x95,
			0xb8, 0x91, 0xe5, 0xc2, 0xd5, 0xcd, 0x1d, 0x90,
			0x2b, 0xb3, 0xb7, 0x58, 0x81, 0xf9, 0x30, 0xed,
			0xbb, 0x3a, 0x95, 0x5b, 0xc3, 0x5a, 0x10, 0x22,
			0x02, 0xb6, 0x67, 0x66, 0x37, 0x79, 0x2d, 0x22,
			0x13, 0x5c, 0xc7, 0xbc, 0x67, 0x41, 0x12, 0x58,
			0xbc, 0x3a, 0x5a, 0x4d, 0x0b, 0xdd, 0x5b, 0xc2,
			0xe2, 0x36, 0x2e, 0x6b, 0xbe, 0xf3, 0x0c, 0xb4,
			0x05, 0x73, 0xe4, 0xcd, 0x40, 0x4f, 0x37, 0x40,
			0x90, 0xa1, 0xb0, 0x41, 0x7c, 0x96, 0x59, 0x24,
			0x4e, 0x33, 0x87, 0x8f, 0x29, 0x70, 0x06, 0xb3,
			0x11, 0x36, 0x35, 0xaf, 0x57, 0x13, 0xf2, 0x4d,
			0x97, 0x1f, 0x32, 0xfa, 0x2b, 0xe6, 0x33, 0xee,
			0x7b, 0xf1, 0x2c, 0x45, 0x39, 0xff, 0xed, 0xe0,
			0x02, 0x6c, 0xbb, 0x9e, 0xee, 0xcc, 0x83, 0x12,
			0xf3, 0x75, 0xd1, 0xa4, 0xc8, 0x67, 0x6c, 0xc5,
			0x1a, 0x9f, 0x26, 0xfb, 0xf6, 0x49, 0x26, 0xad,
			0xea, 0x4a, 0x1a, 0x37, 0x46, 0xa4, 0x27, 0x6e,
			0x5d, 0x4d, 0x2f, 0x28, 0x2e, 0x81, 0x00, 0x7e,
			0xc0, 0xe4, 0x98, 0xe5, 0x4f, 0xfb, 0x2e, 0x7a,
			0x31, 0x8a, 0xb2, 0x1f, 0x5a, 0x10, 0xd6, 0xc7,
			0xf4, 0xd5, 0x2c, 0x47, 0x23, 0xf2, 0x2f, 0xc5,
			0x1e, 0xe4, 0x89, 0x92, 0x21, 0xf7, 0xca, 0xb1,
			0x54, 0x77, 0x32, 0x6b, 0xeb, 0xfd, 0x03, 0x1a,
			0x77, 0x1e, 0x93, 0x7e, 0x37, 0x54, 0xbe, 0x36,
			0x95, 0x22, 0x7f, 0xfe, 0xd0, 0xf8, 0x76, 0x77,
			0xc0, 0xfc, 0xe2, 0xfb, 0xa4, 0x03, 0x45, 0xb7,
			0xf7, 0x7c, 0x86, 0xae, 0x04, 0xec, 0x5c, 0xec,
			0xb3, 0x4b, 0xbc, 0x62, 0x11, 0x2f, 0xd9, 0x57,
		},
		0, /* salt length */
		/* V2 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x40, 0xc6, 0x22, 0x73, 0xc4, 0xb1, 0x39, 0xe7,
			0x80, 0x7d, 0x1f, 0x98, 0x7b, 0x9d, 0xc8, 0xc0,
			0xda, 0x35, 0x1b, 0xb9, 0x62, 0x8a, 0x97, 0x10,
			0x39, 0xb8, 0x3c, 0xda, 0xec, 0xa3, 0xa9, 0x4b,
			0xb6, 0x2f, 0x82, 0x84, 0x00, 0xb4, 0x14, 0xad,
			0x24, 0xac, 0x3c, 0x44, 0x76, 0xaf, 0x84, 0xb4,
			0x85, 0xa2, 0x90, 0x2a, 0x9e, 0x2b, 0xb9, 0xa4,
			0x92, 0x67, 0xf7, 0x4b, 0xb9, 0xe0, 0x04, 0x02,
			0x37, 0xb9, 0xbe, 0x3f, 0x7e, 0x2c, 0xce, 0x19,
			0x15, 0x37, 0x87, 0xa3, 0x97, 0x91, 0x10, 0x79,
			0xfe, 0xe3, 0xc6, 0x98, 0x21, 0x35, 0xdc, 0x73,
			0x7c, 0xa6, 0x44, 0x43, 0x30, 0x61, 0xd3, 0x9e,
			0x4a, 0xcc, 0xa0, 0x4b, 0x48, 0x03, 0xad, 0x55,
			0xda, 0x84, 0xc9, 0x5e, 0xe5, 0x2c, 0xb4, 0x36,
			0xcd, 0x62, 0x85, 0xac, 0xf4, 0x92, 0x49, 0xa4,
			0x7e, 0xdd, 0xed, 0x6a, 0x58, 0x0e, 0x5e, 0x46,
		},
		/* S3 : 256 byte */
		{
			0x3e, 0x6a, 0x06, 0x21, 0xcf, 0xaa, 0x5f, 0xf9,
			0xc8, 0x96, 0xde, 0xd6, 0x5f, 0xa0, 0x48, 0x0a,
			0x86, 0xcf, 0xd2, 0x2c, 0x73, 0x67, 0xbb, 0x11,
			0x72, 0x1b, 0x91, 0x24, 0x41, 0xee, 0xfc, 0xc4,
			0xf4, 0x3f, 0xd6, 0xc6, 0x0b, 0xc5, 0x19, 0xc7,
			0x2d, 0x2a, 0xe3, 0xec, 0x7c, 0x94, 0x5b, 0x15,
			0xd0, 0xa6, 0x64, 0xb6, 0x18, 0x32, 0x24, 0xcf,
			0x08, 0xd9, 0xa1, 0x4b, 0x3b, 0x30, 0xfa, 0xfb,
			0xf7, 0xfa, 0x90, 0xa5, 0x82, 0xf6, 0x8f, 0xa4,
			0x28, 0xfd, 0x1c, 0xd6, 0xc3, 0xc7, 0x64, 0x00,
			0xa4, 0xc0, 0xa1, 0xa2, 0x56, 0x61, 0x6b, 0xcc,
			0x34, 0x94, 0xb2, 0xd9, 0x63, 0x54, 0xce, 0x0e,
			0x31, 0xd6, 0x42, 0x12, 0x99, 0xcf, 0xe5, 0x6a,
			0x87, 0x64, 0xf0, 0xba, 0x6f, 0x97, 0xf7, 0x7f,
			0x06, 0x3c, 0x3f, 0xeb, 0x66, 0xe0, 0xd3, 0xbc,
			0x89, 0xd1, 0x74, 0xbd, 0x03, 0x9b, 0x23, 0x9e,
			0xdd, 0x55, 0x3b, 0xe0, 0xc6, 0x77, 0xd7, 0xcd,
			0xd1, 0xb1, 0x24, 0x55, 0x63, 0x45, 0x3e, 0x61,
			0x9c, 0x46, 0x73, 0xf7, 0x48, 0x0b, 0xdc, 0xe0,
			0x96, 0xc4, 0xa7, 0x34, 0x4e, 0x7e, 0xd3, 0x35,
			0xd2, 0x84, 0xce, 0xe2, 0x91, 0x79, 0x8f, 0xba,
			0x60, 0x9e, 0x16, 0xa0, 0x61, 0x33, 0xb6, 0x38,
			0x91, 0xca, 0x9c, 0x29, 0x3d, 0xce, 0xe0, 0x39,
			0xaf, 0x1f, 0xaf, 0xda, 0x36, 0xfd, 0xce, 0x52,
			0xf1, 0x0d, 0x24, 0xbb, 0xca, 0x87, 0x56, 0x65,
			0x9b, 0xdb, 0x1b, 0x09, 0x29, 0x7c, 0x95, 0x56,
			0x1b, 0x94, 0x36, 0x2f, 0x85, 0x89, 0x8f, 0xcb,
			0x9b, 0xd9, 0x04, 0x96, 0xce, 0x1e, 0xa4, 0xb0,
			0xdb, 0x31, 0x6d, 0x00, 0x57, 0xea, 0x12, 0xf7,
			0x70, 0xd2, 0xd6, 0x1f, 0xa5, 0xa1, 0xed, 0x01,
			0xf5, 0x17, 0xcb, 0x43, 0x9d, 0x7a, 0x3c, 0xc5,
			0xa7, 0x0f, 0xf9, 0x61, 0xe9, 0xd5, 0x12, 0x31,
		},
		0, /* salt length */
		/* V3 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xca, 0x89, 0xb4, 0x2f, 0x3a, 0x53, 0x4f, 0xb6,
			0xdd, 0x07, 0x9b, 0xd8, 0xa4, 0x63, 0xb1, 0x58,
			0x1c, 0x51, 0x21, 0x97, 0xf1, 0x5f, 0x94, 0xab,
			0xb2, 0x6f, 0xb0, 0xb0, 0xd3, 0xaf, 0x3c, 0x63,
			0xf8, 0xcc, 0x85, 0xed, 0x71, 0x63, 0x00, 0xfc,
			0x00, 0xb9, 0x30, 0x18, 0xe3, 0xab, 0xca, 0x35,
			0x0e, 0xe8, 0x18, 0x84, 0xb9, 0x5d, 0x9b, 0x8f,
			0x1a, 0xf1, 0xf4, 0x85, 0x74, 0x73, 0x05, 0xd2,
			0x50, 0xd0, 0x53, 0xf3, 0x0a, 0x8a, 0xae, 0x89,
			0x90, 0x65, 0x15, 0xac, 0x67, 0x64, 0xd4, 0x81,
			0xef, 0xf3, 0xef, 0x0f, 0xe7, 0x66, 0xc3, 0x4e,
			0x33, 0xa2, 0xeb, 0x81, 0xdd, 0x1a, 0x61, 0xc6,
			0x5b, 0xfe, 0xe5, 0xec, 0x10, 0xae, 0xa3, 0x84,
			0x15, 0x75, 0x10, 0x3b, 0x38, 0x74, 0x15, 0x67,
			0x45, 0xdc, 0x6f, 0xdf, 0x85, 0x61, 0x56, 0x10,
			0x85, 0xeb, 0x5c, 0x4d, 0x18, 0x8e, 0x99, 0x67,
		},
		/* S4 : 256 byte */
		{
			0x6c, 0x6b, 0x9d, 0x1a, 0x5a, 0xa4, 0x86, 0xe6,
			0xf9, 0xb9, 0xcc, 0x83, 0xed, 0xf8, 0x39, 0x0b,
			0xdb, 0x1e, 0x90, 0x86, 0xd3, 0x60, 0x06, 0x98,
			0xad, 0x2c, 0x56, 0x98, 0x0f, 0x24, 0x98, 0x60,
			0xe7, 0x39, 0x75, 0xb5, 0x1f, 0xad, 0xfe, 0x64,
			0xf4, 0x38, 0x3f, 0x99, 0xb5, 0x45, 0x71, 0x4d,
			0xa2, 0x5b, 0x24, 0x6b, 0x8f, 0xee, 0xe2, 0x34,
			0xaf, 0x16, 0xf4, 0x3f, 0x78, 0x22, 0xfe, 0xf5,
			0x69, 0x85, 0x18, 0x17, 0xad, 0x66, 0x9a, 0xc7,
			0x5d, 0x31, 0x1e, 0xd8, 0x2a, 0xaf, 0x0d, 0x31,
			0x98, 0x7b, 0x88, 0x5b, 0xac, 0xab, 0x88, 0x86,
			0xe4, 0xf5, 0x3d, 0xa9, 0x3f, 0xb2, 0x08, 0xd9,
			0x4d, 0xd7, 0x76, 0x71, 0x72, 0xe2, 0x2c, 0xa9,
			0xd1, 0xc9, 0x67, 0x7f, 0x14, 0x84, 0xe2, 0x1b,
			0xb7, 0x1b, 0xc6, 0xa0, 0x1c, 0x1d, 0xb8, 0xeb,
			0x95, 0xf5, 0xb2, 0xec, 0xf0, 0x62, 0x4b, 0xbf,
			0xa0, 0x48, 0x14, 0x10, 0x22, 0xa7, 0x81, 0xe3,
			0x95, 0x99, 0x35, 0x29, 0x0f, 0xd4, 0x77, 0xd9,
			0x5c, 0x29, 0x99, 0xbd, 0x0d, 0xc0, 0x8a, 0x92,
			0x61, 0x4f, 0x7e, 0xff, 0x0d, 0x11, 0xcf, 0x56,
			0xaf, 0xd3, 0xff, 0x05, 0x64, 0x6d, 0x47, 0xa0,
			0x05, 0x59, 0x23, 0x81, 0x31, 0x09, 0x95, 0x32,
			0xb4, 0x25, 0xfd, 0x44, 0x83, 0x0e, 0xd2, 0xaf,
			0x86, 0x77, 0xe5, 0x39, 0xa1, 0x35, 0xdf, 0xaf,
			0x8a, 0x0f, 0x86, 0xe9, 0xda, 0x0b, 0xd6, 0x14,
			0xa5, 0x9a, 0x22, 0xa1, 0xc0, 0x09, 0xd3, 0x03,
			0xd6, 0x23, 0xfd, 0x17, 0x8d, 0x93, 0x5d, 0x93,
			0x22, 0xf9, 0x84, 0x0c, 0x9b, 0x58, 0x49, 0x4c,
			0x1e, 0x68, 0x92, 0x37, 0xf8, 0x1c, 0xec, 0xa7,
			0x27, 0xb0, 0x92, 0x03, 0xb7, 0x2f, 0x75, 0x0d,
			0xa3, 0x14, 0xc9, 0xbc, 0x58, 0x39, 0x78, 0xc3,
			0x40, 0x10, 0xe9, 0xaf, 0x7c, 0xd0, 0xaa, 0x5f,
		},
		0, /* salt length */
		/* V4 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x84, 0x7c, 0x8d, 0x5f, 0x08, 0x21, 0x49, 0x67,
			0x7d, 0x5c, 0x08, 0x16, 0xe9, 0xec, 0x20, 0xbc,
			0x6e, 0x39, 0xe5, 0xad, 0x1f, 0xdb, 0x1c, 0xe7,
			0x12, 0x60, 0xb3, 0xf4, 0xb6, 0x4a, 0xaf, 0x12,
			0x67, 0x58, 0x65, 0x92, 0x40, 0x07, 0xed, 0x7e,
			0x31, 0x57, 0xd4, 0x7e, 0x51, 0xd2, 0x11, 0xfc,
			0x75, 0xcb, 0x3a, 0x87, 0x01, 0xe4, 0xf8, 0x75,
			0xfe, 0xd3, 0xb9, 0xcd, 0x6d, 0x65, 0xa4, 0x64,
			0xdb, 0xc2, 0xd2, 0x45, 0x36, 0xf4, 0xbb, 0x56,
			0xdf, 0xa3, 0xee, 0x39, 0x00, 0x22, 0x06, 0x71,
			0x20, 0xb0, 0xd9, 0xc6, 0xf5, 0xf7, 0xf2, 0xea,
			0x79, 0xd9, 0x42, 0x5c, 0x40, 0x6c, 0x12, 0x8d,
			0x08, 0x43, 0x3c, 0x42, 0x92, 0xf0, 0x9b, 0x39,
			0xb0, 0xc6, 0xc0, 0x0f, 0xa0, 0xe4, 0xe6, 0x61,
			0x98, 0x6f, 0x28, 0x32, 0xa2, 0x1e, 0xf2, 0x8c,
			0xaa, 0xfc, 0xb6, 0x6a, 0xd5, 0x69, 0xe0, 0xd6,
		},
		/* S5 : 256 byte */
		{
			0x6e, 0x33, 0x2a, 0xad, 0xa4, 0x53, 0x68, 0x14,
			0x9e, 0x43, 0x4a, 0x5f, 0xd5, 0x6a, 0xab, 0xd1,
			0x03, 0x4c, 0x23, 0x87, 0xd6, 0x62, 0xb3, 0x23,
			0xc6, 0xd0, 0x09, 0x19, 0xf9, 0x08, 0x08, 0xb7,
			0x5d, 0x01, 0x1a, 0x94, 0x18, 0xf8, 0xe8, 0x76,
			0xe1, 0x14, 0x20, 0x08, 0xb2, 0xc6, 0x37, 0xc9,
			0x48, 0x87, 0xcd, 0xd6, 0xd9, 0x34, 0x13, 0x75,
			0x93, 0x4e, 0xfa, 0x02, 0x2c, 0xbd, 0x37, 0x2a,
			0x85, 0x73, 0xaf, 0xbc, 0x2f, 0x22, 0x1c, 0x2b,
			0x25, 0x81, 0x4d, 0xe4, 0x37, 0x08, 0xb9, 0xe3,
			0x06, 0x8f, 0x68, 0x5f, 0x7a, 0x08, 0xc4, 0x22,
			0xbd, 0x65, 0x91, 0x1f, 0x46, 0x09, 0xee, 0x05,
			0x9f, 0x9e, 0x1e, 0x4a, 0x66, 0x58, 0x83, 0x81,
			0x92, 0xff, 0xeb, 0x22, 0x1d, 0x16, 0xa5, 0xbc,
			0xb0, 0x99, 0xd3, 0x7d, 0x00, 0x32, 0xb0, 0x4f,
			0xaf, 0xe8, 0x4f, 0xd4, 0xe5, 0x35, 0x15, 0x3f,
			0xcc, 0x98, 0x0b, 0x4c, 0x2e, 0x8c, 0xe8, 0x47,
			0x3a, 0xd1, 0x84, 0x19, 0x1c, 0xde, 0xce, 0x6d,
			0x8b, 0x3d, 0x06, 0xc8, 0x8a, 0xf2, 0x4e, 0x80,
			0xea, 0xbc, 0xac, 0x5c, 0x38, 0xed, 0x36, 0xbd,
			0x79, 0x0d, 0x5c, 0xac, 0x56, 0x2b, 0xb6, 0x2b,
			0x92, 0xf2, 0x70, 0xd9, 0xc2, 0x69, 0xc7, 0xf4,
			0xa3, 0x8b, 0x4d, 0xb9, 0x6b, 0x66, 0x17, 0x33,
			0xe5, 0xbc, 0xc0, 0x92, 0x81, 0x84, 0xb6, 0xbe,
			0xf1, 0xb9, 0xd9, 0x6d, 0xe1, 0xda, 0xfb, 0xde,
			0x41, 0xd4, 0x81, 0xcf, 0x68, 0x3f, 0xdf, 0x43,
			0xf4, 0x17, 0x93, 0xa5, 0x4a, 0x82, 0xcf, 0xbf,
			0xc9, 0xe1, 0x7c, 0xf6, 0x0e, 0x7d, 0x46, 0xd8,
			0xc7, 0x32, 0x3a, 0x13, 0x08, 0xaa, 0x4c, 0x1e,
			0xfd, 0x53, 0xe7, 0xca, 0x9b, 0x6a, 0xdc, 0xe1,
			0x76, 0x12, 0xce, 0x87, 0x87, 0x2b, 0x11, 0x16,
			0x51, 0x22, 0xdf, 0x1a, 0x18, 0xa9, 0x2c, 0x5e,
		},
		0, /* salt length */
		/* V5 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x32, 0xc9, 0x80, 0x58, 0x89, 0xb6, 0x20, 0x81,
			0x34, 0x89, 0x6c, 0x8e, 0x74, 0xcd, 0xd0, 0x0d,
			0x35, 0x11, 0xb4, 0x95, 0x40, 0x46, 0x51, 0x4d,
			0xb2, 0x68, 0xc3, 0xad, 0xad, 0x61, 0x5f, 0x89,
			0x4d, 0x2a, 0x46, 0x4b, 0xde, 0x33, 0x3a, 0x80,
			0x4c, 0x05, 0xb1, 0x96, 0xda, 0x26, 0x28, 0xe3,
			0x17, 0x3c, 0xba, 0xea, 0x0f, 0x76, 0xf1, 0xda,
			0xbe, 0x28, 0xdc, 0x58, 0xca, 0xb5, 0x62, 0x7e,
			0x71, 0xb2, 0xd5, 0x24, 0xbf, 0x48, 0xcb, 0x5e,
			0x05, 0xda, 0x29, 0x45, 0x88, 0xe8, 0x80, 0xfb,
			0x76, 0x34, 0x9d, 0xe9, 0x1e, 0x23, 0x5b, 0x4b,
			0x5f, 0x65, 0xbc, 0xef, 0x61, 0xd8, 0x38, 0x39,
			0x84, 0xaa, 0x55, 0x6b, 0x96, 0xbf, 0x78, 0x23,
			0x49, 0x52, 0xfb, 0x26, 0xe4, 0xde, 0x7f, 0x5b,
			0x38, 0x3f, 0x84, 0x1b, 0xd6, 0x14, 0x37, 0xa8,
			0x7f, 0x69, 0x8a, 0xfa, 0xdc, 0x93, 0x8a, 0xc2,
		},
		/* S6 : 256 byte */
		{
			0x28, 0xf1, 0x0b, 0xb4, 0xf6, 0xf9, 0xbf, 0x3c,
			0x5e, 0x33, 0xbd, 0xb9, 0xc0, 0xf4, 0x8c, 0xf5,
			0x83, 0xbd, 0xa5, 0x8b, 0x2c, 0xec, 0xf1, 0x91,
			0x7e, 0xa4, 0x50, 0xf5, 0x99, 0x51, 0xda, 0x19,
			0x1e, 0x8f, 0x77, 0xa0, 0x59, 0x29, 0xef, 0xa6,
			0xc8, 0xeb, 0x32, 0x46, 0x29, 0xa2, 0x1e, 0x9e,
			0x89, 0x4a, 0xd9, 0xa4, 0x5b, 0x48, 0xe9, 0x67,
			0xc7, 0xf2, 0xae, 0xd3, 0xa4, 0x4b, 0xc6, 0x82,
			0x00, 0x53, 0x59, 0x01, 0xce, 0x34, 0x2d, 0x36,
			0xa2, 0xe1, 0xa8, 0x78, 0x5c, 0x35, 0x31, 0x19,
			0x97, 0x5e, 0xf3, 0x8a, 0x00, 0x64, 0x87, 0x7d,
			0x3b, 0x67, 0xa2, 0x20, 0xac, 0x6c, 0xed, 0x59,
			0x5e, 0x1f, 0xf3, 0x51, 0x90, 0x2e, 0x06, 0x5c,
			0x65, 0xec, 0x94, 0x23, 0x8b, 0x4d, 0x5d, 0x18,
			0xb9, 0xb4, 0xc1, 0x46, 0x48, 0x9b, 0x1a, 0xc2,
			0xc2, 0xcc, 0x70, 0xc1, 0x59, 0xdf, 0xff, 0x13,
			0xd1, 0x3b, 0x05, 0x25, 0x43, 0x55, 0xaf, 0x33,
			0x81, 0x49, 0x9c, 0x2b, 0x20, 0x72, 0xe2, 0x0e,
			0x6d, 0xfc, 0xaf, 0x3b, 0x89, 0xf4, 0x6d, 0x01,
			0x54, 0xcc, 0xe3, 0x53, 0xf4, 0xb5, 0xb1, 0xa5,
			0x1b, 0x05, 0x86, 0x41, 0x7e, 0xf8, 0x24, 0xd0,
			0x00, 0x90, 0xf9, 0xe7, 0x1e, 0xa2, 0x74, 0xa7,
			0xe1, 0x20, 0x73, 0x48, 0x2f, 0x46, 0x7c, 0x6e,
			0x7d, 0x38, 0x3d, 0x9c, 0xa0, 0x27, 0xc0, 0xfb,
			0xb5, 0x1d, 0xb4, 0xe0, 0x13, 0x92, 0x27, 0xc2,
			0xf6, 0x77, 0x51, 0x32, 0xc4, 0xc7, 0x5c, 0x95,
			0x48, 0xf0, 0xf1, 0x39, 0xc3, 0x47, 0x3b, 0xe0,
			0x3d, 0x35, 0xc8, 0xf3, 0x05, 0x4b, 0x59, 0x20,
			0xae, 0xdf, 0x7e, 0x9e, 0x0b, 0xd3, 0xb0, 0xd2,
			0x1d, 0x27, 0x1a, 0xb7, 0x74, 0x14, 0xb7, 0x5f,
			0x3a, 0xd1, 0x12, 0x83, 0xa4, 0x7a, 0x35, 0xa5,
			0x43, 0x92, 0x46, 0x4e, 0x50, 0x05, 0x3b, 0xc8,
		},
		0, /* salt length */
		/* V6 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xe6, 0x31, 0xde, 0x1c, 0xbe, 0x35, 0x6a, 0xf1,
			0x18, 0xde, 0x8a, 0x78, 0x08, 0xbe, 0x1c, 0x6c,
			0x6f, 0x01, 0x2b, 0xfc, 0xcf, 0xf3, 0x62, 0x71,
			0x5d, 0xc2, 0xc7, 0x10, 0x58, 0x84, 0xa6, 0x42,
			0x9a, 0x7b, 0xca, 0xf2, 0x6d, 0x98, 0xf3, 0x37,
			0x9f, 0x3f, 0x85, 0x33, 0xd9, 0x4a, 0x7e, 0x3b,
			0x2c, 0x50, 0xaa, 0x37, 0xed, 0x27, 0x1a, 0x4c,
			0xc3, 0xa7, 0xd2, 0xb9, 0xfe, 0xb1, 0xc2, 0xff,
			0x2a, 0x70, 0xf8, 0x45, 0xe4, 0xdf, 0x13, 0xfa,
			0xed, 0x8c, 0xfb, 0xbc, 0x7d, 0x9b, 0xcf, 0x79,
			0xda, 0x0d, 0x1f, 0xd4, 0x28, 0x43, 0x5c, 0xca,
			0x83, 0xbd, 0x8c, 0x95, 0x4b, 0x28, 0x42, 0xd0,
			0xbf, 0x98, 0xce, 0xdd, 0xce, 0x12, 0xb1, 0x84,
			0x29, 0x4d, 0x54, 0xdc, 0x9d, 0xb9, 0xf9, 0x6f,
			0xa0, 0xc1, 0xf1, 0x99, 0x2a, 0x4d, 0x0f, 0x31,
			0xc6, 0x8c, 0x5e, 0x6b, 0xe8, 0xf0, 0x0f, 0xfd,
		},
		/* S7 : 256 byte */
		{
			0x1d, 0x2d, 0xa7, 0x02, 0x0c, 0x7f, 0x30, 0x7d,
			0xfc, 0x97, 0xd7, 0x75, 0xad, 0x8d, 0x1a, 0x21,
			0x87, 0x61, 0xfb, 0x9d, 0x55, 0xc6, 0x32, 0x39,
			0x8a, 0x63, 0xec, 0xb2, 0xea, 0xc7, 0x71, 0x28,
			0x99, 0x36, 0x2d, 0xd8, 0xe3, 0x73, 0x22, 0x6a,
			0xe0, 0x95, 0x5c, 0x8d, 0x7a, 0x37, 0xae, 0x75,
			0xfe, 0x74, 0xbf, 0x42, 0xd8, 0xaf, 0x0e, 0x6c,
			0x4f, 0x61, 0xa5, 0x32, 0xc1, 0xfb, 0xb9, 0x22,
			0x3b, 0x22, 0xcf, 0x2d, 0xe1, 0x28, 0x3a, 0x7b,
			0x4f, 0x00, 0xcf, 0x2e, 0xe4, 0x73, 0xf7, 0x9d,
			0xe7, 0xb3, 0x64, 0x32, 0x7e, 0x34, 0x43, 0xdb,
			0xbf, 0x29, 0x83, 0x31, 0x2b, 0x5d, 0xf9, 0xa2,
			0xb6, 0x4b, 0x9c, 0x8c, 0x1b, 0xa6, 0x7b, 0x56,
			0x8c, 0xa2, 0x4b, 0x37, 0xeb, 0xf6, 0x95, 0xa1,
			0x08, 0x0b, 0x2f, 0x1c, 0x47, 0x2b, 0xb9, 0x65,
			0xc3, 0xb4, 0x9f, 0xae, 0x48, 0x04, 0x86, 0xd5,
			0xad, 0x08, 0x2b, 0x6c, 0x64, 0x92, 0x41, 0xb9,
			0xfb, 0xd0, 0x31, 0xcc, 0x67, 0x9e, 0xa6, 0x83,
			0xc4, 0x46, 0xbd, 0x13, 0x1c, 0x70, 0x60, 0xee,
			0x77, 0x14, 0x0d, 0xf1, 0xeb, 0x4f, 0xbc, 0x28,
			0x94, 0x8a, 0xc4, 0x7f, 0x6f, 0x50, 0x0f, 0x3d,
			0x27, 0xc4, 0xaa, 0x76, 0xae, 0x50, 0xc3, 0xea,
			0x8a, 0xe1, 0xf1, 0xdb, 0xfe, 0x8d, 0xe2, 0xdf,
			0x42, 0xa9, 0x3c, 0x11, 0x29, 0x94, 0x26, 0x4b,
			0xa5, 0xe6, 0xa5, 0xc6, 0x7a, 0xca, 0x13, 0xc4,
			0x85, 0xc3, 0x26, 0xfa, 0xdb, 0x4a, 0x44, 0xd3,
			0x53, 0x37, 0x91, 0x65, 0x88, 0xbc, 0x57, 0xec,
			0x45, 0x90, 0x8b, 0x5f, 0x27, 0xa0, 0xd0, 0x5d,
			0x8f, 0xe5, 0xb2, 0xfc, 0xe1, 0x09, 0xfc, 0x7e,
			0x0e, 0x2b, 0x2e, 0xdd, 0xd9, 0x36, 0xbb, 0x4a,
			0x11, 0x6a, 0xa2, 0xee, 0x6f, 0xb5, 0xec, 0x2f,
			0x39, 0xd4, 0x76, 0xa0, 0x97, 0x3e, 0x9a, 0xdd,
		},
		0, /* salt length */
		/* V7 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x60, 0x78, 0x89, 0xb7, 0xba, 0x00, 0x71, 0xc5,
			0x84, 0x09, 0x60, 0x14, 0x7d, 0xa4, 0x38, 0xd3,
			0xad, 0x0e, 0x51, 0xc7, 0x54, 0xd4, 0x11, 0x90,
			0x20, 0x3a, 0xbd, 0x2e, 0xa2, 0xde, 0x8e, 0xc3,
			0x03, 0x99, 0x08, 0x9a, 0xf3, 0xa6, 0x87, 0x69,
			0xe3, 0x06, 0x1b, 0xd3, 0x74, 0x22, 0x95, 0x31,
			0x2f, 0xb3, 0xe2, 0x92, 0x76, 0xdf, 0x98, 0x00,
			0xba, 0xa4, 0xcf, 0x06, 0xef, 0x7a, 0x32, 0x6f,
			0x3d, 0x34, 0x56, 0x4f, 0xf9, 0x28, 0x4a, 0x48,
			0xd3, 0x8f, 0x6f, 0x06, 0x51, 0x2f, 0x89, 0x0c,
			0x7c, 0x37, 0x17, 0xb3, 0xd9, 0x6a, 0xe0, 0x6f,
			0x9a, 0x7f, 0xcf, 0x55, 0x16, 0xc8, 0x00, 0xda,
			0x77, 0x13, 0x1b, 0xde, 0xb2, 0xce, 0xe1, 0xe9,
			0xbd, 0x6e, 0x04, 0xb4, 0xfc, 0x11, 0x3c, 0xeb,
			0xfc, 0x61, 0x97, 0x31, 0x63, 0xbc, 0x11, 0x65,
			0x27, 0x93, 0x64, 0x23, 0xff, 0x10, 0x21, 0x8e,
		},
		/* S8 : 256 byte */
		{
			0x5c, 0x31, 0x8b, 0x9e, 0x83, 0xc7, 0x8b, 0xb1,
			0xa3, 0xa2, 0x6a, 0x3e, 0x6b, 0xbf, 0x61, 0x9c,
			0xa5, 0x66, 0xca, 0x0f, 0xbc, 0xff, 0xd4, 0xb3,
			0x82, 0x16, 0x3c, 0x7d, 0xea, 0xda, 0x6c, 0x09,
			0x5f, 0xbe, 0x3b, 0xe8, 0xf7, 0x50, 0x1e, 0x5b,
			0x09, 0xb8, 0xea, 0x4c, 0xd7, 0x0b, 0x50, 0xd5,
			0x19, 0xb7, 0xd8, 0x57, 0x2c, 0xe6, 0x0e, 0x35,
			0x4d, 0x19, 0x3e, 0xd5, 0x84, 0xe4, 0x99, 0xe4,
			0xfd, 0xab, 0x6b, 0xc6, 0x6e, 0x7c, 0x10, 0xfa,
			0xa8, 0xeb, 0x09, 0x7a, 0x11, 0xc7, 0xc0, 0x21,
			0x80, 0xaf, 0xec, 0x54, 0x5a, 0x69, 0xe7, 0x8b,
			0x12, 0x84, 0x41, 0xff, 0x1f, 0xb3, 0xca, 0x59,
			0x25, 0x8b, 0xd7, 0x02, 0xb3, 0x4a, 0x30, 0xe3,
			0x9a, 0x31, 0xbd, 0xe6, 0x9b, 0x50, 0xeb, 0x31,
			0x2f, 0x9a, 0xac, 0x82, 0xc8, 0xb5, 0x01, 0x3a,
			0x57, 0xed, 0xea, 0xd1, 0xc4, 0xa8, 0x25, 0x01,
			0x58, 0x63, 0x9d, 0x63, 0x7b, 0x3e, 0xcd, 0xe9,
			0xc1, 0x76, 0x4b, 0xae, 0x81, 0x77, 0xe5, 0x96,
			0xe8, 0xda, 0x98, 0xba, 0x9b, 0xad, 0xa5, 0x50,
			0xc7, 0xa5, 0x9e, 0xa6, 0x62, 0xf1, 0x2d, 0xe6,
			0x0d, 0xc9, 0x65, 0xc9, 0xc0, 0x8f, 0x85, 0xbf,
			0xd1, 0x79, 0x56, 0x0e, 0x65, 0xc5, 0x63, 0x94,
			0xc9, 0x8a, 0x0e, 0xb4, 0x76, 0xe4, 0x05, 0x53,
			0x97, 0x17, 0xec, 0x66, 0x7a, 0x0f, 0x29, 0xf0,
			0xa6, 0x3b, 0x57, 0xdf, 0x17, 0x1a, 0x60, 0xe7,
			0x6f, 0xfc, 0xe8, 0x33, 0x38, 0xd3, 0x7b, 0x12,
			0x34, 0xaf, 0x79, 0x62, 0x36, 0x54, 0xf9, 0x11,
			0xbf, 0x38, 0xdf, 0x14, 0x6b, 0x8e, 0xfd, 0x8e,
			0xf3, 0xe4, 0xd9, 0xe0, 0x30, 0x36, 0x74, 0xcd,
			0xde, 0x2b, 0xe7, 0xf7, 0xa0, 0x5a, 0xb6, 0x7c,
			0x4f, 0xf0, 0x14, 0x92, 0xec, 0x61, 0xc9, 0xca,
			0xbf, 0x60, 0x80, 0x70, 0xfd, 0x4b, 0x55, 0x42,
		},
		0, /* salt length */
		/* V8 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x8d, 0xc8, 0xf2, 0xd4, 0x3a, 0x54, 0xd5, 0xda,
			0x60, 0x80, 0xbb, 0x26, 0xc0, 0xd5, 0x9b, 0x26,
			0x21, 0xcf, 0x91, 0xd4, 0xa3, 0xe2, 0xd4, 0xde,
			0x50, 0xbb, 0xe8, 0x04, 0xf4, 0xc8, 0x15, 0xc2,
			0x2e, 0xfa, 0x17, 0x30, 0xc8, 0xca, 0x17, 0x26,
			0x44, 0x7a, 0xda, 0xa3, 0xb7, 0x9d, 0x39, 0x70,
			0xdb, 0xd9, 0xd1, 0x00, 0x5f, 0xec, 0xfb, 0x9b,
			0x81, 0xed, 0xff, 0xbc, 0xdf, 0xd4, 0x84, 0xb7,
			0x8a, 0x3d, 0x4b, 0x9e, 0x5d, 0x69, 0x1d, 0x66,
			0x8d, 0x86, 0x02, 0x46, 0x80, 0x30, 0xb4, 0x60,
			0xe3, 0x37, 0x53, 0xa3, 0xf7, 0xa3, 0x5a, 0xf0,
			0x2b, 0xf5, 0xd2, 0x7b, 0xf0, 0xb0, 0xc6, 0x75,
			0xc9, 0x18, 0xf6, 0xe8, 0xa1, 0x3a, 0xcf, 0xe2,
			0x62, 0x2c, 0x9b, 0xd5, 0xc3, 0x96, 0xf6, 0x3e,
			0x62, 0x71, 0x81, 0x85, 0x12, 0x0f, 0xda, 0x24,
			0x76, 0x5c, 0xcb, 0x0c, 0xcf, 0x63, 0xc1, 0x44,
		},
		/* S9 : 256 byte */
		{
			0x42, 0xb5, 0x30, 0xb3, 0x15, 0xfb, 0x55, 0x07,
			0xcf, 0x9a, 0x36, 0x1b, 0x45, 0xeb, 0x17, 0x94,
			0x42, 0x99, 0x2c, 0x1e, 0x6b, 0x3d, 0xf9, 0xc4,
			0x05, 0xc4, 0x4f, 0xa8, 0xf0, 0x7f, 0x6b, 0xdb,
			0x12, 0xc7, 0x92, 0x9f, 0x08, 0x30, 0x8e, 0x65,
			0xa1, 0xaa, 0x4a, 0x8c, 0xe6, 0x9c, 0x72, 0x00,
			0x21, 0x44, 0x80, 0x1c, 0x1c, 0xc5, 0x0f, 0xf1,
			0xb5, 0x54, 0x18, 0xad, 0x24, 0xa4, 0xc8, 0x53,
			0xfb, 0xc3, 0xfa, 0xf0, 0x25, 0xf6, 0xfb, 0x59,
			0x2c, 0x76, 0x8e, 0xae, 0x48, 0xdd, 0xbf, 0xe1,
			0x48, 0x59, 0x04, 0xc8, 0x7c, 0xdc, 0x5a, 0x0b,
			0xc6, 0xe2, 0x7c, 0xfd, 0x21, 0xa5, 0xbc, 0x48,
			0x2c, 0x24, 0x55, 0xfb, 0x32, 0xba, 0xc1, 0x68,
			0xde, 0x11, 0x04, 0x93, 0x9c, 0xe0, 0x06, 0xb3,
			0x00, 0x5d, 0xc4, 0xc9, 0x9d, 0xc6, 0xe1, 0x3a,
			0x03, 0xb3, 0xee, 0xc8, 0x38, 0xf3, 0xa7, 0xf4,
			0x15, 0x1b, 0x47, 0x02, 0x12, 0xb4, 0x5c, 0xf9,
			0x46, 0x33, 0x87, 0x50, 0x55, 0x4a, 0xc3, 0xac,
			0x55, 0x3e, 0x3c, 0x0a, 0x74, 0x6e, 0x3c, 0x43,
			0x5a, 0x05, 0xf4, 0x46, 0x39, 0xf9, 0x12, 0x17,
			0x71, 0x87, 0xfb, 0xbb, 0xab, 0x58, 0xec, 0x8a,
			0xe1, 0xf5, 0xd1, 0x48, 0xd1, 0xa3, 0xb0, 0xf7,
			0x94, 0x68, 0x76, 0xf9, 0x9b, 0x84, 0x4e, 0x17,
			0x37, 0x7d, 0x62, 0xcd, 0x4c, 0x38, 0xe8, 0x8b,
			0xa3, 0x87, 0xbc, 0xc9, 0x2a, 0x72, 0xc8, 0x55,
			0xc5, 0xb8, 0x27, 0xdf, 0x85, 0x97, 0xc3, 0x8e,
			0x01, 0xd3, 0xec, 0xf0, 0xac, 0x4b, 0x51, 0xd5,
			0xc1, 0xf5, 0x4b, 0xc0, 0xa6, 0xe5, 0xa8, 0xf5,
			0x76, 0x5b, 0x67, 0x88, 0xbd, 0xf8, 0x66, 0xe0,
			0x90, 0x51, 0x2a, 0x16, 0x28, 0x3d, 0xcb, 0x68,
			0x91, 0x69, 0x85, 0x1d, 0x04, 0xfe, 0xc3, 0x3f,
			0x89, 0x0d, 0xf7, 0xda, 0xe8, 0x2d, 0xda, 0xa2,
		},
		0, /* salt length */
		/* V9 : 0 byte */
		{
			0x00, 
		},
	},
};

/* 10 elements */
static mod2048_t SHA512256[] = {
	/* SHA512256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xb7, 0x5c, 0x51, 0x76, 0xeb, 0x13, 0x77, 0x61,
			0x50, 0xb2, 0x6d, 0x43, 0xa1, 0x1b, 0x61, 0x20,
			0x21, 0x91, 0x54, 0x82, 0x83, 0x12, 0x21, 0x6f,
			0x07, 0xe7, 0xbb, 0x98, 0xe7, 0x79, 0xb6, 0xd3,
			0xf1, 0x66, 0x09, 0xcb, 0x75, 0xd4, 0x12, 0xf6,
			0x61, 0x06, 0xda, 0xc8, 0xab, 0x2d, 0x9b, 0x27,
			0xef, 0x80, 0x7f, 0x1e, 0x5a, 0x8e, 0xab, 0xd5,
			0xc6, 0xd3, 0x7c, 0xc1, 0x26, 0x8a, 0x10, 0x18,
			0x25, 0xc3, 0x09, 0x88, 0x0a, 0xb5, 0x8d, 0x28,
			0xdf, 0x55, 0x91, 0xaf, 0xd2, 0x28, 0x58, 0xe4,
			0xdb, 0x55, 0x36, 0x55, 0x9f, 0xfe, 0x9a, 0x4c,
			0x95, 0xd0, 0x4c, 0xff, 0x67, 0xd9, 0xf2, 0x07,
			0x9b, 0x9a, 0xd9, 0x13, 0x72, 0x5e, 0xf2, 0xde,
			0x10, 0xca, 0x1b, 0x5e, 0xde, 0xb4, 0xa0, 0x3f,
			0x2d, 0x39, 0x02, 0x8f, 0x1c, 0x09, 0xf7, 0xbb,
			0xbb, 0x06, 0xb1, 0xba, 0x70, 0x0e, 0x63, 0x88,
		},
		/* S0 : 256 byte */
		{
			0x83, 0x85, 0x57, 0x73, 0xc5, 0xc8, 0xd6, 0x4c,
			0x4d, 0x84, 0xa9, 0x2b, 0x82, 0x25, 0x9d, 0xa5,
			0xbc, 0x6a, 0xd0, 0xa6, 0xa1, 0xcd, 0xc0, 0x01,
			0xbc, 0xda, 0x47, 0x2b, 0x41, 0xa9, 0x88, 0x41,
			0xbc, 0x85, 0xff, 0x46, 0x69, 0x2c, 0x95, 0x56,
			0xe5, 0x10, 0x01, 0xa0, 0xd6, 0x56, 0x6d, 0x7e,
			0x48, 0x77, 0x61, 0x66, 0xe2, 0xed, 0x69, 0xce,
			0x38, 0x6d, 0x96, 0xe5, 0x44, 0xbd, 0x49, 0x8d,
			0x0f, 0x58, 0xec, 0x68, 0xf1, 0xa5, 0x73, 0x9d,
			0xb4, 0x24, 0xb4, 0x3e, 0x1a, 0xac, 0x84, 0xbf,
			0xf4, 0x59, 0x52, 0x8c, 0x09, 0x28, 0xa0, 0x4e,
			0xec, 0xd6, 0xe1, 0x99, 0xbc, 0x56, 0x8c, 0xf3,
			0x76, 0xd6, 0xab, 0xfa, 0xc1, 0xef, 0xf0, 0x99,
			0x34, 0x04, 0xba, 0x4a, 0x40, 0xc8, 0xa8, 0x0c,
			0xc5, 0x5d, 0x06, 0x8b, 0x6f, 0xe5, 0xa7, 0x70,
			0x9a, 0x9b, 0x39, 0xe6, 0x06, 0x86, 0x09, 0x8f,
			0x32, 0x89, 0x62, 0x3f, 0xef, 0xf4, 0x11, 0xfd,
			0xf1, 0xb3, 0x33, 0xef, 0x66, 0xd1, 0xb4, 0x0e,
			0x44, 0xf1, 0x04, 0x1a, 0x05, 0xa7, 0xc5, 0x3a,
			0xb8, 0x8e, 0x4a, 0x95, 0xa7, 0x8b, 0x24, 0xed,
			0x82, 0x1f, 0x77, 0x4f, 0x74, 0x8b, 0x80, 0xcb,
			0x9f, 0x78, 0x2c, 0x8c, 0xfd, 0x33, 0x88, 0x23,
			0x47, 0x36, 0x18, 0x43, 0x72, 0x3a, 0xcd, 0x1b,
			0x23, 0x66, 0x48, 0x04, 0x23, 0xdc, 0xb9, 0x53,
			0xd2, 0x2a, 0xd1, 0xa3, 0x03, 0x29, 0xe0, 0x42,
			0xae, 0x0e, 0x92, 0x64, 0x46, 0x4a, 0x28, 0x1a,
			0x51, 0xfa, 0xd7, 0xb5, 0x46, 0x6e, 0x94, 0xc3,
			0x43, 0x8c, 0xb6, 0x75, 0xbb, 0xc5, 0xbe, 0x78,
			0x15, 0x11, 0xdc, 0x51, 0x2a, 0x8c, 0x15, 0x5e,
			0xd0, 0x9b, 0xb8, 0xb6, 0xef, 0x66, 0xdb, 0x60,
			0x46, 0x18, 0xe8, 0x3e, 0x71, 0x8b, 0xfa, 0xfc,
			0xd5, 0xd8, 0x82, 0x67, 0x6a, 0xfc, 0x27, 0x3b,
		},
		0, /* salt length */
		/* V0 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x63, 0x15, 0xb1, 0x75, 0x54, 0x61, 0xac, 0x27,
			0x90, 0xfc, 0x33, 0xcc, 0x32, 0xa1, 0xe4, 0x1a,
			0x6b, 0xcc, 0x1b, 0x87, 0x27, 0xcd, 0xbb, 0x01,
			0xed, 0x1b, 0x2b, 0xe0, 0x07, 0xec, 0x9e, 0x19,
			0xe4, 0x02, 0x98, 0xb4, 0xba, 0xc8, 0xae, 0x80,
			0x6f, 0x9d, 0x8d, 0x05, 0xba, 0x70, 0x3d, 0xba,
			0x86, 0x8d, 0xa7, 0x78, 0x97, 0xb6, 0x55, 0x2f,
			0x6f, 0x76, 0x7a, 0xd8, 0x73, 0xb2, 0x32, 0xaa,
			0x4a, 0x81, 0x0a, 0x91, 0x86, 0x3e, 0xc3, 0xdc,
			0x86, 0xdb, 0x53, 0x35, 0x9a, 0x77, 0x2d, 0xd7,
			0x69, 0x33, 0xc2, 0xfb, 0x90, 0x42, 0x48, 0x93,
			0x8c, 0x40, 0xba, 0x3b, 0xda, 0xc5, 0x20, 0x6d,
			0x5c, 0x39, 0x10, 0xf4, 0xff, 0xea, 0x75, 0xa3,
			0x9b, 0x5b, 0x8f, 0x46, 0x1b, 0xe0, 0x3b, 0xd9,
			0xdd, 0x67, 0x75, 0xf8, 0x5c, 0x99, 0x1b, 0x97,
			0x0c, 0x22, 0xf3, 0xf3, 0x85, 0x4c, 0xf0, 0xc8,
		},
		/* S1 : 256 byte */
		{
			0x40, 0x62, 0x31, 0x68, 0x1c, 0x7a, 0xd1, 0xcf,
			0x28, 0x7f, 0x97, 0xda, 0xa6, 0x02, 0xec, 0xa3,
			0x54, 0x7e, 0xb0, 0xe1, 0xad, 0x19, 0x6a, 0xa9,
			0x4c, 0x83, 0x3f, 0xba, 0x93, 0xe9, 0x5d, 0xcb,
			0xe7, 0xf6, 0xfd, 0x71, 0x15, 0x89, 0x40, 0xd4,
			0xc1, 0xdf, 0x03, 0xd8, 0x6c, 0xb4, 0x4c, 0xe4,
			0x74, 0x6c, 0x33, 0x34, 0x44, 0xc3, 0x85, 0x11,
			0x0d, 0xc4, 0x31, 0xc9, 0x00, 0x61, 0x40, 0x27,
			0x4a, 0xb4, 0x9d, 0x66, 0x78, 0x9a, 0x87, 0x50,
			0x7b, 0x02, 0x55, 0x11, 0xd1, 0x66, 0xbd, 0x9c,
			0x42, 0xf2, 0x0f, 0x62, 0xa4, 0x07, 0xcb, 0xe4,
			0x73, 0xc7, 0xda, 0x81, 0x5f, 0xf2, 0x82, 0xd5,
			0xd7, 0x27, 0x89, 0x8d, 0xac, 0x2e, 0x8e, 0xf7,
			0x35, 0xe1, 0x72, 0x0d, 0xbd, 0xff, 0xac, 0x02,
			0xe7, 0x95, 0x6d, 0xdd, 0xe1, 0x3e, 0x35, 0x52,
			0x11, 0xc5, 0x92, 0x2d, 0xaa, 0xff, 0x52, 0xab,
			0x95, 0xc4, 0x0c, 0xec, 0x6a, 0x4b, 0x3d, 0xe4,
			0x60, 0x87, 0x31, 0x9d, 0xc6, 0x23, 0x54, 0xe1,
			0x56, 0x83, 0x4d, 0x0b, 0x84, 0x31, 0x02, 0x6b,
			0x86, 0x07, 0xe0, 0x79, 0x71, 0x4f, 0xfe, 0xb9,
			0x39, 0x77, 0x06, 0x44, 0x99, 0x00, 0x90, 0x8a,
			0xdb, 0x26, 0xde, 0x94, 0x8a, 0xd1, 0x96, 0x09,
			0x15, 0xb2, 0xfe, 0x26, 0xb4, 0x77, 0x47, 0xa7,
			0xbf, 0x03, 0x4b, 0xed, 0xfd, 0xcb, 0x37, 0xfc,
			0x57, 0x93, 0x8c, 0x2f, 0x40, 0xa9, 0xde, 0x96,
			0x83, 0xe7, 0x89, 0x06, 0xda, 0x00, 0x5c, 0x30,
			0x5a, 0xce, 0xa3, 0xf6, 0xc1, 0xa1, 0x86, 0xfb,
			0xa5, 0xbd, 0x36, 0x50, 0x8c, 0xf0, 0x97, 0x6d,
			0x45, 0x20, 0x4c, 0x3a, 0x2d, 0x90, 0xbb, 0x2f,
			0x14, 0xf7, 0xb5, 0xd4, 0x41, 0xd3, 0x72, 0xb2,
			0xe1, 0xc6, 0x5f, 0xf9, 0xe7, 0x74, 0xf0, 0x1a,
			0xdc, 0x72, 0xf3, 0x92, 0xc9, 0x89, 0xd4, 0x0d,
		},
		0, /* salt length */
		/* V1 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x40, 0x88, 0x57, 0x24, 0x85, 0xb7, 0x0f, 0x74,
			0x6f, 0xaf, 0x49, 0x98, 0xc4, 0xa7, 0xb7, 0x90,
			0x57, 0xbd, 0x49, 0x99, 0x41, 0x31, 0x96, 0xfe,
			0x6f, 0xd1, 0x7d, 0x31, 0x5d, 0xb4, 0x0c, 0x1a,
			0x20, 0x2b, 0xa0, 0xaf, 0xaa, 0xf5, 0x4e, 0xc5,
			0x4a, 0xb2, 0x82, 0x25, 0x72, 0x9a, 0xe6, 0xe3,
			0xbd, 0x6f, 0x86, 0xeb, 0x30, 0xb0, 0x31, 0xdd,
			0xc1, 0x10, 0x22, 0x1b, 0x15, 0x27, 0xe0, 0x20,
			0x8e, 0x66, 0x62, 0x13, 0x85, 0x50, 0x77, 0x9b,
			0xd4, 0xe7, 0x65, 0xf6, 0x98, 0x65, 0xd0, 0x45,
			0x35, 0x2d, 0xde, 0x2a, 0x9a, 0x01, 0x99, 0x95,
			0xc6, 0x75, 0x95, 0xdb, 0x8a, 0xa4, 0xd2, 0x09,
			0x91, 0x21, 0xe3, 0x77, 0x9f, 0x31, 0x50, 0xa0,
			0x16, 0x37, 0x3a, 0x30, 0xbe, 0x6f, 0x3d, 0x5f,
			0xc5, 0xdf, 0x9b, 0x7e, 0x05, 0x8b, 0x96, 0xcf,
			0x86, 0xe9, 0x1d, 0x9a, 0x44, 0x94, 0xab, 0x7d,
		},
		/* S2 : 256 byte */
		{
			0x27, 0x57, 0x2f, 0x4d, 0xc4, 0x78, 0x5c, 0x62,
			0x6f, 0xa3, 0x0a, 0xdf, 0x0f, 0xc6, 0x72, 0x02,
			0xba, 0xbb, 0x69, 0xfa, 0x73, 0x13, 0x78, 0xe9,
			0xaa, 0x30, 0x07, 0x2f, 0xa7, 0x0c, 0x9d, 0x5c,
			0xc5, 0x44, 0x94, 0x13, 0x6d, 0x39, 0x4e, 0x57,
			0xbd, 0x38, 0x0c, 0x55, 0x03, 0xde, 0xbd, 0x0d,
			0x04, 0x10, 0x30, 0x61, 0x7c, 0x9f, 0x3c, 0xb4,
			0x59, 0x15, 0xe6, 0x03, 0x7c, 0x18, 0x36, 0x99,
			0xc7, 0x08, 0xf2, 0x4d, 0xc1, 0xc3, 0x4d, 0xe0,
			0xb0, 0xd3, 0x39, 0x5e, 0xc4, 0xf7, 0xf2, 0x3d,
			0x95, 0x9e, 0x2a, 0x65, 0x36, 0x82, 0x4b, 0xb6,
			0x45, 0x7e, 0x7b, 0xfa, 0xf9, 0x00, 0x6a, 0xe2,
			0x39, 0x70, 0xb0, 0xb5, 0xb3, 0x56, 0x20, 0x4f,
			0xde, 0xc8, 0xa1, 0x34, 0xc1, 0x5d, 0xab, 0x60,
			0x4e, 0xdb, 0x79, 0xf9, 0x1c, 0x9c, 0x67, 0x65,
			0x00, 0xa1, 0x1c, 0x62, 0xde, 0xab, 0x67, 0x95,
			0x3d, 0x4f, 0xc0, 0x1f, 0x63, 0x3d, 0x7c, 0x15,
			0x99, 0x27, 0xca, 0x44, 0xda, 0x0a, 0x41, 0xc7,
			0x48, 0xcd, 0x78, 0x71, 0x39, 0x5e, 0x60, 0x02,
			0x83, 0x66, 0xe6, 0x92, 0x57, 0xad, 0xfe, 0xb4,
			0xf0, 0x79, 0xa4, 0xf1, 0x19, 0x03, 0xd1, 0xc4,
			0x6a, 0x43, 0x87, 0xbd, 0x29, 0xf3, 0xc4, 0xf2,
			0xb1, 0x35, 0xbf, 0x1f, 0x68, 0x3c, 0x2d, 0x1d,
			0xc6, 0x27, 0x61, 0xd6, 0x5c, 0x84, 0xa0, 0xd9,
			0x14, 0xbf, 0x22, 0xa0, 0xaf, 0x02, 0x58, 0x10,
			0xea, 0x9f, 0x75, 0xe0, 0xa5, 0x0c, 0x6c, 0x1d,
			0xf1, 0xcc, 0x07, 0x21, 0x2d, 0xf6, 0x25, 0xbc,
			0x6f, 0x4a, 0xe7, 0x07, 0x54, 0xdb, 0xaf, 0x7a,
			0x6a, 0x41, 0xa4, 0x6b, 0x8c, 0x23, 0x18, 0x26,
			0x0e, 0xf0, 0x98, 0x22, 0x05, 0x17, 0xb9, 0xcb,
			0xad, 0xba, 0x89, 0xf9, 0x18, 0xc8, 0x95, 0xf5,
			0x2f, 0x48, 0xbf, 0x14, 0x6a, 0xa3, 0xe1, 0x77,
		},
		0, /* salt length */
		/* V2 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x29, 0xbc, 0xd9, 0x72, 0x86, 0xa1, 0x7d, 0xce,
			0xd9, 0xb3, 0x67, 0xf4, 0x03, 0x9f, 0x9f, 0x97,
			0x7d, 0x50, 0x77, 0x90, 0x47, 0x0f, 0x1f, 0x25,
			0x21, 0xef, 0x74, 0x8c, 0xfe, 0x68, 0x19, 0xab,
			0xb6, 0x42, 0xc9, 0x22, 0xfb, 0xb4, 0xeb, 0x8b,
			0x8e, 0xce, 0x67, 0x00, 0xb2, 0x14, 0xb9, 0xde,
			0xe2, 0x6c, 0x44, 0xc9, 0xbf, 0x3a, 0xe8, 0xf1,
			0x4c, 0xc9, 0xd6, 0x93, 0x5d, 0xed, 0xa3, 0xc2,
			0x4d, 0xe6, 0x9c, 0x67, 0xf0, 0x88, 0x5a, 0x87,
			0xc8, 0x99, 0x96, 0xc4, 0x7c, 0x7b, 0x3e, 0x27,
			0x85, 0x0a, 0xc7, 0x1c, 0x2b, 0xc8, 0xc6, 0xbe,
			0xb0, 0x38, 0xba, 0x55, 0xcb, 0x87, 0x2c, 0x1d,
			0x58, 0x71, 0xfb, 0x4a, 0x4d, 0x63, 0xf1, 0x48,
			0xf0, 0xdd, 0x99, 0x47, 0x47, 0x1b, 0x55, 0xf7,
			0xd0, 0xf4, 0xab, 0x90, 0x73, 0x02, 0xe0, 0x16,
			0xb5, 0x03, 0xc8, 0xdb, 0x2e, 0x7f, 0xdc, 0x45,
		},
		/* S3 : 256 byte */
		{
			0x84, 0x18, 0x3b, 0x2f, 0x03, 0xa2, 0x79, 0x54,
			0x83, 0x3e, 0x90, 0xcc, 0xd8, 0xce, 0x5d, 0xac,
			0xf6, 0xd1, 0xb6, 0x8f, 0xbc, 0x87, 0x86, 0x2f,
			0x48, 0xb1, 0x13, 0xdd, 0x1e, 0x50, 0xc3, 0x95,
			0xac, 0x3c, 0x86, 0xee, 0xaa, 0x0a, 0x8b, 0x0c,
			0x69, 0x09, 0xf4, 0x7e, 0xb7, 0x98, 0x98, 0x7b,
			0x24, 0xa6, 0x9b, 0x3c, 0x1b, 0x11, 0x46, 0x52,
			0xc5, 0xfe, 0xc2, 0x35, 0x84, 0xdb, 0x9d, 0xd5,
			0xea, 0xae, 0x0e, 0xa8, 0xff, 0x94, 0x98, 0xc6,
			0x94, 0x2b, 0x7e, 0xcb, 0x51, 0xbb, 0x09, 0x69,
			0x71, 0x0b, 0x3a, 0x47, 0xb9, 0xe4, 0x13, 0x80,
			0x82, 0x6d, 0x54, 0x66, 0x8b, 0xc8, 0xea, 0xf5,
			0xda, 0x5c, 0x46, 0xc8, 0x36, 0x68, 0x9a, 0x95,
			0x06, 0x30, 0xdf, 0x22, 0xad, 0xc5, 0xeb, 0x6f,
			0x6a, 0x3a, 0xcb, 0xa7, 0x9e, 0x5c, 0x11, 0xbe,
			0xf0, 0x42, 0x07, 0x0c, 0xe6, 0x95, 0xad, 0x92,
			0xe9, 0xa4, 0x5f, 0x9c, 0x83, 0x1d, 0x23, 0x7e,
			0x43, 0x9d, 0x28, 0x4f, 0x7f, 0x62, 0xd5, 0x29,
			0x02, 0xa9, 0x95, 0x65, 0x1c, 0x3c, 0x0f, 0xa8,
			0x72, 0x9d, 0x79, 0x54, 0xc9, 0x37, 0xef, 0xf7,
			0xc7, 0x7d, 0x10, 0x65, 0x36, 0xb2, 0x04, 0x2e,
			0x2d, 0x6c, 0xe7, 0x84, 0x0b, 0x1d, 0x1e, 0x07,
			0x72, 0xb9, 0xfa, 0x5f, 0x23, 0xd1, 0x45, 0x01,
			0x33, 0xb1, 0x17, 0x29, 0xe1, 0x47, 0x8d, 0x20,
			0xe5, 0xfb, 0xf1, 0xf3, 0x8a, 0x3c, 0x95, 0xd9,
			0x38, 0xbd, 0xca, 0x36, 0x2c, 0xe9, 0x47, 0xd2,
			0xaa, 0xb1, 0xb9, 0xec, 0xfc, 0xd4, 0xfc, 0x88,
			0x76, 0x43, 0x4c, 0x1b, 0xf8, 0x45, 0xe8, 0x10,
			0xae, 0xb0, 0xb1, 0xd9, 0x11, 0x73, 0x76, 0xf0,
			0xb9, 0x3d, 0x35, 0x08, 0x4b, 0xd4, 0x35, 0x70,
			0x1b, 0x70, 0xfd, 0xfc, 0x69, 0x30, 0xd7, 0x54,
			0xa1, 0xf7, 0xb1, 0xe1, 0x20, 0x75, 0x0e, 0x56,
		},
		0, /* salt length */
		/* V3 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x62, 0x59, 0x62, 0x2d, 0x0a, 0xea, 0x50, 0xb8,
			0xa6, 0x9e, 0xe3, 0x3a, 0xae, 0x19, 0xf7, 0xad,
			0x84, 0x46, 0x9d, 0x72, 0x58, 0xa9, 0x1b, 0x42,
			0x49, 0xf1, 0x34, 0x67, 0xcc, 0xcd, 0xa9, 0x2c,
			0x52, 0x01, 0xff, 0x92, 0x25, 0xb9, 0x83, 0x44,
			0x93, 0x1d, 0x8c, 0xcd, 0x4f, 0x25, 0x00, 0x09,
			0x14, 0xaf, 0x19, 0xd9, 0x8b, 0xcf, 0x69, 0x1b,
			0x6f, 0x66, 0x1c, 0xda, 0xef, 0xf6, 0x7e, 0xc1,
			0x54, 0xb4, 0xbd, 0xd6, 0xe3, 0xdf, 0xef, 0x65,
			0x05, 0x51, 0x56, 0x39, 0xa5, 0x54, 0xe3, 0x12,
			0xda, 0xb6, 0xfa, 0x54, 0xc6, 0x20, 0x75, 0xfd,
			0xbe, 0xa3, 0xeb, 0xd9, 0xaa, 0x94, 0x32, 0xf9,
			0xaf, 0x9a, 0x22, 0x61, 0x0c, 0x05, 0x9c, 0x01,
			0x42, 0x61, 0xcb, 0xe7, 0x78, 0x4b, 0xaa, 0xb2,
			0x1d, 0x84, 0xc7, 0x4a, 0xcf, 0x44, 0x46, 0xce,
			0x7c, 0xd1, 0x28, 0xce, 0xc7, 0x4f, 0xce, 0xbb,
		},
		/* S4 : 256 byte */
		{
			0x0b, 0xb0, 0xc8, 0xcf, 0x7a, 0xa6, 0x8a, 0xbe,
			0x7d, 0x1f, 0x3f, 0x65, 0x16, 0x6b, 0xbd, 0x11,
			0x0b, 0x8d, 0x37, 0x1f, 0x85, 0x93, 0x12, 0xaf,
			0xed, 0x10, 0x24, 0x1f, 0xe0, 0x07, 0xa7, 0xcb,
			0x97, 0x7d, 0x01, 0x27, 0x5d, 0x51, 0x2a, 0x97,
			0xd7, 0x0d, 0x9b, 0x11, 0x4b, 0x08, 0x2e, 0x61,
			0x51, 0x60, 0x43, 0xb1, 0x95, 0x8a, 0xb7, 0x59,
			0xbd, 0x86, 0xe0, 0xef, 0xc5, 0x2b, 0x13, 0xb6,
			0xc4, 0x49, 0xc3, 0xfd, 0x4f, 0xd1, 0x22, 0x73,
			0x7a, 0xd0, 0xd5, 0x12, 0x44, 0xa8, 0x3d, 0xdc,
			0x7d, 0x4d, 0xa6, 0xf6, 0x6a, 0x6c, 0x56, 0xfc,
			0x5e, 0x72, 0x4c, 0x26, 0x4e, 0xf6, 0x27, 0x71,
			0x7c, 0xa2, 0x84, 0x39, 0xea, 0x8f, 0xe9, 0xb4,
			0x44, 0xd5, 0x6e, 0xf0, 0x96, 0x01, 0x01, 0x42,
			0xe2, 0x1b, 0x91, 0x0c, 0x10, 0xd0, 0xad, 0x66,
			0x2c, 0xbe, 0x45, 0xb7, 0x1c, 0x55, 0x07, 0x83,
			0xd0, 0x21, 0x6d, 0xed, 0x08, 0xbb, 0x6d, 0xc5,
			0xaa, 0x23, 0x80, 0x23, 0xd9, 0x53, 0xd7, 0xb6,
			0xea, 0xa5, 0x94, 0x34, 0x52, 0x56, 0x1e, 0x48,
			0xdc, 0xd3, 0x40, 0xff, 0x41, 0xe9, 0x89, 0xf1,
			0x9a, 0x76, 0xc9, 0x9f, 0x0f, 0x3e, 0xe2, 0xbd,
			0x19, 0x51, 0xe5, 0x62, 0x3c, 0x9f, 0xdd, 0x32,
			0x37, 0x39, 0x82, 0xc1, 0xda, 0x91, 0xba, 0xb7,
			0x38, 0x33, 0x51, 0x6b, 0x51, 0x10, 0x7a, 0x13,
			0xb2, 0xe8, 0x90, 0x84, 0x65, 0xd9, 0xc1, 0xa9,
			0x1d, 0x3c, 0xdd, 0xa4, 0xf9, 0xcc, 0xce, 0x1d,
			0xb3, 0xfa, 0x6b, 0x6a, 0x7a, 0xc2, 0x3e, 0xea,
			0xa4, 0xc9, 0x5f, 0x77, 0xcc, 0xab, 0x55, 0x83,
			0x77, 0x33, 0x14, 0x20, 0x07, 0x5b, 0xf1, 0xd1,
			0x94, 0x6a, 0xbe, 0x13, 0x93, 0x31, 0xa0, 0x81,
			0x8a, 0xec, 0xb3, 0x2a, 0x6c, 0xf9, 0xe6, 0x5b,
			0xa8, 0x97, 0x9b, 0xb5, 0x62, 0x7d, 0x22, 0xc5,
		},
		0, /* salt length */
		/* V4 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x35, 0x6a, 0x94, 0x6a, 0xd7, 0xf0, 0xf8, 0x99,
			0x7e, 0xa5, 0x18, 0xb8, 0xe3, 0x6d, 0x19, 0xe7,
			0xf9, 0x7f, 0x25, 0x3c, 0xd1, 0xbc, 0x6c, 0xb5,
			0xba, 0x3e, 0x69, 0x71, 0xac, 0x33, 0x67, 0x20,
			0xe7, 0xec, 0x43, 0xb9, 0xd7, 0x31, 0x73, 0x6b,
			0x3b, 0xd8, 0x47, 0xd5, 0x86, 0x96, 0x6f, 0x79,
			0x85, 0x14, 0xbe, 0x6e, 0x43, 0x45, 0x2e, 0x56,
			0x2a, 0xf8, 0x23, 0x90, 0x6a, 0xa9, 0xf4, 0xaf,
			0x9f, 0x9d, 0xa7, 0x2a, 0xae, 0x2e, 0xfc, 0x54,
			0xef, 0x63, 0xae, 0x8f, 0x92, 0xc3, 0xf5, 0xbd,
			0x3b, 0x6b, 0x4f, 0xf8, 0xf0, 0x5f, 0x80, 0x83,
			0x35, 0x1e, 0xd7, 0x00, 0x75, 0x79, 0xb4, 0xe3,
			0xa4, 0xa4, 0x47, 0x26, 0x23, 0x8e, 0x9a, 0x3f,
			0xe4, 0x6e, 0xd2, 0x4f, 0xce, 0x9e, 0xb2, 0x9c,
			0x63, 0xad, 0x60, 0x41, 0xa0, 0xb0, 0x6c, 0x24,
			0x24, 0xd8, 0x0e, 0x29, 0x95, 0x4a, 0xaf, 0xa5,
		},
		/* S5 : 256 byte */
		{
			0xa9, 0x95, 0x98, 0x1f, 0xc9, 0xe3, 0x02, 0x2e,
			0x31, 0xe8, 0xf1, 0xe9, 0x51, 0xff, 0x4e, 0x15,
			0x97, 0xbb, 0x61, 0x47, 0x89, 0xa4, 0x0d, 0xec,
			0x03, 0x47, 0xf1, 0x7c, 0xe3, 0x51, 0x1d, 0xf0,
			0x88, 0x2b, 0x60, 0xbe, 0xca, 0xdb, 0xb3, 0x4d,
			0x7a, 0xb3, 0xed, 0x74, 0xf5, 0x8d, 0xc5, 0x39,
			0xbb, 0xe5, 0x91, 0x0b, 0x90, 0x60, 0xfa, 0xcb,
			0xdc, 0x81, 0x58, 0x59, 0xb2, 0xf6, 0xfe, 0x80,
			0x69, 0xb7, 0x10, 0xb1, 0x91, 0x99, 0x13, 0xbc,
			0x14, 0x30, 0x81, 0xff, 0xeb, 0x7a, 0xf3, 0x3a,
			0x50, 0x85, 0x8b, 0xcb, 0x0e, 0x98, 0x56, 0x60,
			0xf5, 0xe9, 0xfa, 0xc4, 0xfa, 0x9c, 0x39, 0xaf,
			0xa7, 0x6e, 0x7a, 0x57, 0xa1, 0x60, 0xf1, 0xcf,
			0x81, 0x43, 0xcc, 0x09, 0xd8, 0x75, 0x81, 0x09,
			0x63, 0x55, 0x88, 0x46, 0x9a, 0xaf, 0x0e, 0xf3,
			0xcf, 0xab, 0xa3, 0xe8, 0x15, 0xb2, 0x71, 0x2e,
			0x9e, 0xc1, 0xb6, 0xac, 0xc0, 0xe0, 0xc8, 0x87,
			0x18, 0x74, 0x14, 0xb7, 0x0f, 0xa5, 0x46, 0xdf,
			0x49, 0xeb, 0x45, 0xf4, 0xae, 0x9c, 0xa1, 0x74,
			0x33, 0x11, 0xb4, 0xd4, 0xf1, 0x1d, 0x22, 0xd3,
			0x93, 0x4f, 0x1a, 0x3a, 0x11, 0xaa, 0x70, 0x0f,
			0x7b, 0x99, 0xb5, 0x17, 0x58, 0x6c, 0xa1, 0xd7,
			0xc1, 0xf4, 0xe2, 0xe4, 0x4a, 0x75, 0xa4, 0xc1,
			0xb9, 0xc8, 0x64, 0xd9, 0xb0, 0xa1, 0x76, 0xc7,
			0xef, 0x08, 0xff, 0xd1, 0xf4, 0xe1, 0x0c, 0x9f,
			0x9c, 0x8e, 0xfa, 0x92, 0x66, 0x05, 0xb1, 0x5d,
			0xd1, 0x10, 0xa6, 0xd6, 0xc8, 0x3b, 0x75, 0xea,
			0x44, 0x5c, 0x07, 0xb5, 0x75, 0x62, 0x89, 0xe6,
			0x9d, 0x0d, 0xaf, 0x94, 0x78, 0xf4, 0x68, 0xab,
			0x3d, 0xe4, 0x5b, 0x1e, 0x51, 0x51, 0xad, 0x7c,
			0x0a, 0xfd, 0xda, 0x46, 0xbd, 0x55, 0xe8, 0xcd,
			0x92, 0xc2, 0x5c, 0x8e, 0x9e, 0x43, 0x69, 0x1a,
		},
		0, /* salt length */
		/* V5 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xab, 0xd3, 0x5f, 0x9e, 0x6b, 0xd9, 0xbc, 0x82,
			0x15, 0x1b, 0x77, 0x0a, 0x8d, 0xbb, 0xbf, 0xfb,
			0x9a, 0x37, 0xa6, 0x72, 0xf4, 0x33, 0xd4, 0xc7,
			0x9e, 0x79, 0x78, 0xb6, 0x75, 0x3e, 0x85, 0xc1,
			0x0a, 0xa7, 0x1a, 0xd0, 0xad, 0x8b, 0x41, 0x1b,
			0xca, 0x7b, 0xdd, 0x49, 0x79, 0xff, 0xde, 0xd9,
			0x50, 0x23, 0xb7, 0xc2, 0x7b, 0x32, 0x12, 0x56,
			0x0f, 0xc2, 0xc2, 0x6c, 0x9c, 0xf0, 0xf9, 0xbe,
			0x02, 0xed, 0xbb, 0x90, 0x85, 0xba, 0x92, 0x93,
			0x23, 0x2f, 0x84, 0x08, 0x24, 0x42, 0x42, 0x28,
			0xf4, 0x86, 0xa2, 0xc4, 0x2e, 0xf2, 0x4f, 0x29,
			0xf9, 0xe4, 0xbf, 0x46, 0x97, 0x65, 0x66, 0x11,
			0x3b, 0xaf, 0x2a, 0x26, 0x1b, 0xed, 0x16, 0xb1,
			0xc7, 0x41, 0x52, 0x8f, 0xb9, 0x0b, 0x53, 0xb0,
			0x2d, 0x69, 0x4f, 0xc0, 0x33, 0xfa, 0xf4, 0x06,
			0x02, 0xe8, 0x59, 0x8c, 0x8f, 0x25, 0x35, 0x80,
		},
		/* S6 : 256 byte */
		{
			0x63, 0x27, 0x14, 0x99, 0xbf, 0xb0, 0x66, 0x1d,
			0xc5, 0xb2, 0x8e, 0x39, 0xe1, 0x9a, 0x7c, 0x1d,
			0xd0, 0x3f, 0x74, 0x0e, 0xc9, 0xaf, 0x3b, 0xab,
			0xaf, 0x73, 0x9b, 0x26, 0x50, 0xaf, 0x5e, 0xb8,
			0x76, 0x66, 0x5c, 0xf7, 0xfa, 0x2c, 0x68, 0x4e,
			0xff, 0x47, 0xdc, 0xc1, 0x53, 0x6d, 0x4f, 0x74,
			0x04, 0x0a, 0xbd, 0x27, 0x38, 0x04, 0x62, 0x88,
			0x6f, 0x1d, 0x55, 0x8e, 0x92, 0xfc, 0x87, 0x60,
			0xae, 0x05, 0xf9, 0x9d, 0xb3, 0x57, 0x05, 0x05,
			0x1f, 0x9e, 0xc7, 0x65, 0x44, 0x0a, 0x43, 0xc4,
			0x29, 0xd2, 0x06, 0x4c, 0x8e, 0x51, 0xe2, 0x3f,
			0xc3, 0x54, 0x32, 0xde, 0x3f, 0x0b, 0x5f, 0x4b,
			0x71, 0x2d, 0x2d, 0x6f, 0x06, 0xe0, 0xf7, 0x3e,
			0x51, 0x65, 0xdf, 0x0e, 0xf2, 0x7f, 0x53, 0x26,
			0xf8, 0x87, 0x32, 0xed, 0x21, 0x5e, 0x29, 0x1d,
			0xac, 0x2e, 0xb5, 0x11, 0x19, 0x0a, 0x8d, 0x14,
			0xa8, 0xff, 0xf9, 0xba, 0x10, 0xe6, 0x9e, 0x4d,
			0x2d, 0x30, 0x64, 0x74, 0x6b, 0x8b, 0x86, 0x74,
			0xb7, 0x50, 0x48, 0xf5, 0x48, 0x59, 0xec, 0xa4,
			0x3a, 0x62, 0x98, 0x8f, 0x00, 0x3f, 0x49, 0xae,
			0x87, 0xd3, 0x6b, 0xf4, 0x40, 0x4f, 0xa7, 0xce,
			0xf5, 0xd3, 0x04, 0x16, 0xf2, 0x54, 0xfb, 0x04,
			0x4b, 0x42, 0x0f, 0x94, 0x9f, 0xb1, 0x89, 0xdc,
			0x43, 0x1f, 0x3e, 0x43, 0x88, 0x7f, 0xad, 0x48,
			0x96, 0x95, 0x6f, 0x5e, 0x16, 0xfa, 0xeb, 0x67,
			0x26, 0x4c, 0xb0, 0x2e, 0x29, 0xb3, 0x31, 0x7a,
			0xbb, 0xe9, 0x31, 0x2e, 0xc2, 0x5b, 0xd2, 0xfa,
			0x06, 0xe8, 0x41, 0x78, 0xa3, 0x9f, 0xa5, 0x41,
			0xe3, 0x51, 0x0d, 0x28, 0xf3, 0x5f, 0x90, 0xa2,
			0x06, 0xba, 0x47, 0x34, 0xaf, 0xa3, 0x49, 0x52,
			0x8f, 0x2e, 0xb2, 0x4c, 0x21, 0x36, 0x99, 0xff,
			0x99, 0x53, 0xff, 0x0b, 0x5c, 0xc7, 0xad, 0x6f,
		},
		0, /* salt length */
		/* V6 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x1f, 0xba, 0x00, 0x5c, 0x70, 0xaa, 0xce, 0x58,
			0x87, 0x86, 0x15, 0x35, 0x1e, 0x74, 0x72, 0xe6,
			0xf0, 0x93, 0x9b, 0x1f, 0x6d, 0xde, 0xaa, 0x5d,
			0xb3, 0x54, 0xb8, 0x26, 0xba, 0x56, 0xae, 0x92,
			0xce, 0x50, 0x58, 0x0a, 0x43, 0x9b, 0xbb, 0x50,
			0x64, 0xa2, 0xa9, 0xbf, 0xac, 0x44, 0x92, 0xd5,
			0x39, 0x7a, 0x9d, 0xee, 0x7d, 0x3a, 0xf7, 0x52,
			0x37, 0x9b, 0x6b, 0x72, 0xa1, 0x39, 0x34, 0x6f,
			0xeb, 0xdb, 0x0f, 0xdc, 0xe9, 0x53, 0x94, 0xc5,
			0x09, 0xa6, 0xc5, 0xf0, 0x87, 0x6d, 0xe8, 0x62,
			0xe4, 0x7b, 0x92, 0x25, 0x94, 0xc0, 0x05, 0x49,
			0xf7, 0x6d, 0xbb, 0x29, 0x8a, 0x59, 0x43, 0xf0,
			0x5f, 0xa4, 0x4c, 0x5b, 0xca, 0x9a, 0x00, 0xc0,
			0x5e, 0xda, 0x93, 0x4f, 0x17, 0xb7, 0x1b, 0x98,
			0xd9, 0xde, 0xa2, 0x4d, 0x19, 0x39, 0x79, 0x49,
			0xda, 0x14, 0xd0, 0xd2, 0xdc, 0x7f, 0x84, 0x1b,
		},
		/* S7 : 256 byte */
		{
			0x24, 0x8b, 0xea, 0x97, 0x7c, 0x17, 0xb3, 0x28,
			0x76, 0x45, 0x2b, 0x64, 0x53, 0x09, 0xdc, 0x7b,
			0xf9, 0x89, 0xcf, 0x60, 0x09, 0x0d, 0xe1, 0x45,
			0x1e, 0x97, 0x76, 0xd0, 0x91, 0x07, 0xea, 0xa4,
			0xc2, 0x98, 0x97, 0xed, 0x6c, 0x45, 0xe7, 0x4a,
			0x00, 0x42, 0xe8, 0xd8, 0x28, 0x46, 0xe3, 0xea,
			0x15, 0xa3, 0xc0, 0x9c, 0x1b, 0xd4, 0xb9, 0xa7,
			0xc0, 0xa4, 0x82, 0xcc, 0xfa, 0xc3, 0x24, 0xae,
			0xab, 0x4e, 0x20, 0x9a, 0xc3, 0x73, 0x88, 0x01,
			0x08, 0x4b, 0x87, 0x06, 0xf6, 0x03, 0x41, 0x3b,
			0xbf, 0x5b, 0xf7, 0xfc, 0x10, 0xae, 0xaf, 0x2f,
			0x69, 0x07, 0x92, 0xa2, 0x7e, 0x47, 0x46, 0x2f,
			0x84, 0x2a, 0x22, 0x13, 0xbd, 0xd4, 0x6e, 0x2f,
			0x00, 0x64, 0x89, 0xd6, 0xf7, 0xf0, 0x30, 0xda,
			0xf3, 0x2b, 0x0b, 0xab, 0x6f, 0x7c, 0x15, 0x85,
			0x4a, 0xb9, 0x58, 0x70, 0x62, 0x0b, 0x28, 0x9a,
			0x17, 0x0a, 0xd4, 0xd8, 0x92, 0x2b, 0x26, 0x3f,
			0x82, 0xe4, 0x7e, 0x4f, 0x06, 0xc5, 0x1a, 0xf2,
			0xb0, 0x21, 0xea, 0xf0, 0x44, 0x8a, 0x7b, 0x3e,
			0xcb, 0x73, 0x1e, 0xa9, 0x4a, 0x08, 0x22, 0x94,
			0x58, 0x68, 0x22, 0x9b, 0x8e, 0x2a, 0x19, 0x81,
			0xd8, 0x1a, 0xac, 0x63, 0x15, 0xb2, 0xde, 0x19,
			0x7e, 0x29, 0x63, 0xf3, 0x49, 0xba, 0x5c, 0xea,
			0x98, 0x14, 0x7e, 0xec, 0xb3, 0xa8, 0x67, 0x32,
			0x50, 0x36, 0x60, 0x38, 0xb5, 0xb7, 0x3b, 0x2c,
			0x43, 0xf5, 0x89, 0xe0, 0x8b, 0xad, 0x52, 0xea,
			0x81, 0x7a, 0xb7, 0xe5, 0xc8, 0x88, 0xe4, 0xdc,
			0x41, 0xdd, 0x48, 0x4b, 0x3d, 0x74, 0x39, 0xf5,
			0x36, 0x57, 0x74, 0x9c, 0x5f, 0x12, 0xfd, 0x44,
			0x3f, 0xf8, 0x63, 0xc1, 0x1d, 0xbb, 0x9a, 0xc1,
			0x1a, 0xa0, 0xf4, 0x75, 0xda, 0x16, 0xd7, 0x69,
			0x18, 0xaf, 0xfa, 0x68, 0x45, 0x2a, 0x07, 0x32,
		},
		0, /* salt length */
		/* V7 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x73, 0x22, 0x82, 0x09, 0xad, 0x87, 0x5c, 0xa1,
			0xb9, 0x2e, 0x45, 0x8b, 0x47, 0xd6, 0x8f, 0x24,
			0x59, 0x4f, 0x5e, 0x4e, 0x52, 0x39, 0x5d, 0x5b,
			0x50, 0x0e, 0xaf, 0x2e, 0xbc, 0x63, 0x29, 0x9c,
			0x20, 0x6a, 0xbe, 0x18, 0x38, 0xbc, 0x3a, 0x90,
			0x4b, 0x55, 0x34, 0x92, 0xbb, 0xe0, 0x3d, 0xb3,
			0xdc, 0x36, 0x5e, 0x31, 0x02, 0x21, 0xc2, 0xd7,
			0xde, 0x65, 0xaf, 0x21, 0x0a, 0x01, 0xbb, 0xa1,
			0xe2, 0xb2, 0x3c, 0xa1, 0x96, 0xb9, 0x00, 0x0a,
			0x7d, 0xde, 0x3e, 0x85, 0xac, 0xb5, 0xcd, 0xdf,
			0xc8, 0x20, 0x94, 0xb2, 0x18, 0x15, 0x42, 0x42,
			0xde, 0xa1, 0x3f, 0xa5, 0xe8, 0x2d, 0xe2, 0x27,
			0x62, 0x01, 0xe4, 0xec, 0xee, 0x06, 0x14, 0xd7,
			0xf7, 0xd0, 0x4d, 0xab, 0x41, 0xdb, 0xa5, 0x70,
			0xc5, 0x15, 0xe8, 0x19, 0xbd, 0x9d, 0x45, 0x63,
			0xfb, 0xcb, 0x46, 0x00, 0x7e, 0x70, 0x06, 0xf5,
		},
		/* S8 : 256 byte */
		{
			0x96, 0xdb, 0x58, 0x37, 0xe5, 0x64, 0x4d, 0x02,
			0xaf, 0x09, 0xf7, 0x4a, 0x91, 0xe1, 0x2d, 0x9a,
			0xec, 0x84, 0xf6, 0x8e, 0x7c, 0x2f, 0x5b, 0x32,
			0x28, 0xec, 0x17, 0xe0, 0x0e, 0xe2, 0x58, 0x06,
			0xcc, 0xf7, 0xf7, 0x05, 0xb5, 0xc8, 0x21, 0x66,
			0xd8, 0xa0, 0x43, 0xc8, 0xa4, 0x11, 0x0f, 0xd2,
			0x89, 0x21, 0x0a, 0x55, 0x9e, 0x93, 0xb8, 0xad,
			0x98, 0x39, 0xbc, 0xbf, 0x18, 0xa3, 0xa7, 0x97,
			0x67, 0xd3, 0x61, 0x77, 0x29, 0xd8, 0xc3, 0x18,
			0xd7, 0xe2, 0x01, 0x9a, 0x33, 0x0d, 0x78, 0x16,
			0xa8, 0xe6, 0xcd, 0x63, 0x78, 0x19, 0xaa, 0x7a,
			0xce, 0x36, 0xdf, 0x61, 0x58, 0x53, 0xff, 0x90,
			0x25, 0x3c, 0xad, 0x34, 0xd7, 0xb8, 0xae, 0xaf,
			0x86, 0x20, 0x8e, 0x61, 0x54, 0xc8, 0xc6, 0x91,
			0x63, 0x29, 0x82, 0x09, 0x6d, 0x38, 0x06, 0x31,
			0x69, 0x3d, 0xca, 0xe3, 0x49, 0x2e, 0xaa, 0x6f,
			0xf8, 0xff, 0xc3, 0x45, 0x4f, 0x36, 0x02, 0xbc,
			0xa6, 0xec, 0x93, 0xcf, 0x3d, 0x56, 0xd3, 0x38,
			0xb7, 0xf1, 0xae, 0x45, 0xd9, 0xaf, 0xd1, 0x2d,
			0x1c, 0x03, 0x38, 0xfa, 0x73, 0x69, 0x4a, 0x61,
			0xc3, 0xc2, 0xb0, 0x19, 0xbf, 0x44, 0x4c, 0xde,
			0xa8, 0xea, 0xc9, 0x42, 0x45, 0xb0, 0x89, 0xcf,
			0x9c, 0xaa, 0x8a, 0xa6, 0xfa, 0xe7, 0x76, 0xd4,
			0x39, 0x41, 0xa1, 0x43, 0x53, 0x2f, 0xde, 0x33,
			0x34, 0x88, 0x18, 0xb6, 0x20, 0xbc, 0x1b, 0xea,
			0x4e, 0x79, 0x19, 0x09, 0xd7, 0xb6, 0x4f, 0x44,
			0x47, 0x4a, 0x0c, 0x1c, 0x20, 0xc9, 0xef, 0x4d,
			0x0c, 0x46, 0x7d, 0x4f, 0x93, 0x71, 0x92, 0x7e,
			0x2c, 0x8d, 0x2b, 0xba, 0x4c, 0xd8, 0x61, 0x8f,
			0xcc, 0xf5, 0xf9, 0x1b, 0xbb, 0xa1, 0x33, 0x6c,
			0xe4, 0x9b, 0x2f, 0x5d, 0xe1, 0x3b, 0x8f, 0x4d,
			0x79, 0x2d, 0x70, 0xa0, 0x70, 0xb3, 0x06, 0xff,
		},
		0, /* salt length */
		/* V8 : 0 byte */
		{
			0x00, 
		},
	},
	/* SHA512256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x09, 0xd1, 0x93, 0x2d, 0x2a, 0x83, 0x76, 0xd4,
			0x87, 0xb8, 0x33, 0xca, 0x5f, 0x43, 0xa0, 0x9e,
			0xd7, 0x41, 0xb3, 0x79, 0x41, 0x4e, 0xeb, 0xf7,
			0x9d, 0xf9, 0xeb, 0x35, 0x60, 0x75, 0x69, 0x2c,
			0x75, 0x91, 0x46, 0x3d, 0x82, 0xe9, 0xfe, 0x0c,
			0x6c, 0x5e, 0x43, 0x92, 0x1d, 0xc5, 0xe5, 0x6e,
			0x85, 0xdc, 0xf5, 0x7c, 0xde, 0x82, 0x2a, 0xa9,
			0x35, 0x27, 0x83, 0x06, 0x26, 0x81, 0x2e, 0x1b,
			0xe7, 0x80, 0xa9, 0x7d, 0x88, 0x85, 0x27, 0xa3,
			0x78, 0xcf, 0x35, 0xf5, 0x4c, 0x35, 0x9d, 0x26,
			0xbc, 0x1b, 0x8b, 0xc4, 0xee, 0x9f, 0xee, 0x0d,
			0xf8, 0x13, 0x21, 0x05, 0xa0, 0xb0, 0xcb, 0xc5,
			0x65, 0x88, 0x04, 0x16, 0xc4, 0xbd, 0xb9, 0x3e,
			0xb4, 0xf1, 0x51, 0x1a, 0x73, 0xfd, 0x90, 0x6d,
			0x84, 0xe6, 0xbc, 0x78, 0xf9, 0x0d, 0xff, 0xcb,
			0xee, 0xa5, 0x0d, 0x5a, 0x56, 0x6a, 0x06, 0xd3,
		},
		/* S9 : 256 byte */
		{
			0x78, 0x24, 0x8f, 0x02, 0xa7, 0x77, 0x28, 0x3a,
			0x46, 0x2d, 0xc6, 0xfc, 0xf9, 0x4d, 0x7f, 0x0b,
			0x0c, 0x4e, 0x6f, 0x10, 0x62, 0x2a, 0xf0, 0x34,
			0xf3, 0x8f, 0x32, 0x64, 0x69, 0xe0, 0xad, 0x1b,
			0xd8, 0xef, 0xc2, 0x2b, 0xb4, 0x41, 0xf9, 0xe7,
			0xa8, 0xcd, 0xa2, 0x5c, 0x85, 0x1a, 0x7a, 0xa6,
			0xd6, 0x90, 0x81, 0x9c, 0xc6, 0xff, 0x16, 0x3f,
			0xa3, 0xc0, 0xeb, 0xd5, 0x83, 0x39, 0x22, 0x56,
			0x62, 0x1a, 0x21, 0xdc, 0x8b, 0xbc, 0x19, 0x30,
			0x60, 0xb5, 0x1e, 0xb1, 0x94, 0xf8, 0x29, 0x54,
			0x38, 0xb1, 0x0f, 0x0b, 0xf7, 0x74, 0xf2, 0xeb,
			0x48, 0xd0, 0x30, 0xb6, 0xb6, 0x20, 0x6c, 0xbe,
			0x79, 0x16, 0x98, 0xe8, 0x9c, 0x71, 0xda, 0x2b,
			0xa3, 0xc0, 0x5b, 0x78, 0x99, 0x67, 0x09, 0x90,
			0x4b, 0x32, 0x88, 0xfc, 0x6e, 0x2d, 0x5b, 0xb5,
			0x92, 0x9e, 0xab, 0x74, 0xe9, 0xb0, 0x28, 0x09,
			0x2b, 0x45, 0xdd, 0x4a, 0xcf, 0x11, 0xed, 0xfd,
			0xf4, 0x9a, 0xa8, 0x25, 0x9c, 0x9a, 0x63, 0x98,
			0x68, 0xb7, 0x0f, 0x02, 0x73, 0xc6, 0x99, 0x91,
			0xca, 0x08, 0x88, 0x55, 0xb3, 0xd1, 0xc7, 0xfb,
			0x53, 0x4d, 0x54, 0xbf, 0x8e, 0x50, 0x7a, 0x33,
			0x96, 0x6b, 0x01, 0x85, 0xb0, 0xbd, 0x52, 0x04,
			0x59, 0x25, 0x15, 0xfd, 0x40, 0x2d, 0xa9, 0x22,
			0xba, 0x68, 0xa6, 0x83, 0x4e, 0x4b, 0x71, 0x76,
			0xdf, 0x5e, 0xb3, 0x11, 0x4e, 0xfa, 0x32, 0xdc,
			0x5d, 0x32, 0x84, 0x80, 0xb2, 0xcd, 0x0c, 0x01,
			0x8c, 0x78, 0x3b, 0xe8, 0x7b, 0x26, 0x7f, 0xf2,
			0xc5, 0x95, 0xa0, 0x39, 0x2a, 0x6d, 0x92, 0x0b,
			0x02, 0x3c, 0x21, 0x62, 0x0a, 0xa5, 0x8e, 0xba,
			0xe4, 0xb0, 0xcf, 0xf3, 0x47, 0xdd, 0xf0, 0x84,
			0x18, 0x18, 0x9c, 0xd1, 0x60, 0x47, 0xa8, 0x0d,
			0xca, 0x08, 0x7c, 0x8e, 0x0a, 0xbe, 0x29, 0x07,
		},
		0, /* salt length */
		/* V9 : 0 byte */
		{
			0x00, 
		},
	},
};

typedef mod2048_t tv_t;


#include "siggenPSStest.c"
